#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

""" Common code shared among TrafficPolicy CliPlugins """

from __future__ import absolute_import, division, print_function
import CliCommand
import CliMatcher
import CliPlugin.TapAggPmapCliLib as TapAggPmapCliLib
from CliPlugin import MirroringPolicyMapClassModeTapAgg
from CliPlugin import PolicyMapCliLib
from CliPlugin.ClassificationCliContextLib import MatchRuleBaseContext
from CliPlugin.ClassificationCliLib import ( ProtocolMixin,
                                   generateTcpFlagExpression )
from CliMode.TrafficPolicy import ( MatchRuleIpv4ConfigMode,
                                    MatchRuleIpv6ConfigMode,
                                    MatchRuleMacConfigMode,
                                    MatchRuleDefaultConfigMode,
                                    EncapInnerMatchIpv4ConfigMode,
                                    EncapInnerMatchIpv6ConfigMode,
                                    EncapInnerMatchMacConfigMode,
                                    EncapInnerVxlanMatchIpv4ConfigMode,
                                    ActionsConfigMode,
                                    MacActionsConfigMode )
from ClassificationLib import getIdFromDesc

from Toggles.TrafficPolicyToggleLib import ( toggleTrafficPolicyPpsPolicingEnabled )


from Arnet import IpGenAddr
import Tac
import BasicCliUtil
import six

ActionType = Tac.Type( "PolicyMap::ActionType" )
UniqueId = Tac.Type( "Ark::UniqueId" )
ClassPriorityConstant = Tac.Type( 'TrafficPolicy::ClassPriorityConstant' )
tacMatchOption = Tac.Type( 'PolicyMap::ClassMapMatchOption' )
matchIpAccessGroup = tacMatchOption.matchIpAccessGroup
matchIpv6AccessGroup = tacMatchOption.matchIpv6AccessGroup
matchMacAccessGroup = tacMatchOption.matchMacAccessGroup
ReservedClassMapNames = Tac.Type( 'TrafficPolicy::ReservedClassMapNames' )
RateUnit = Tac.Type( "PolicyMap::RateUnit" )
BurstUnit = Tac.Type( "PolicyMap::BurstUnit" )
DropWithMessageType4 = Tac.Type( 'PolicyMap::DropWithMessageType4' )
DropWithMessageType6 = Tac.Type( 'PolicyMap::DropWithMessageType6' )
DropWithCliMsgType = Tac.Type( 'TrafficPolicy::DropWithCliMsgType' )
neighborsConfigConflictMsg = (
   "The 'protocol neighbors' subcommand is not supported"
   " when any other match subcommands are configured" )
matchL4ProtocolConflictMsg = (
   "The 'protocol bgp' subcommand is not supported"
   " when any other match subcommands are configured" )
invalidPortConflictMsg = (
      "The '%s' subcommand is not supported if protocols other than "
      "'{tcp|udp|tcp udp}' are configured"
      )
invalidProtocolConflictMsg = (
      "The 'protocol' subcommand only supports 'tcp' or 'udp' if"
      " '{source|destination} port' is configured"
      )

# Rate and burst unit tokens are used to shape user input.
# We select what we allow based on toggles here, then the police command
# handler doesn't have to deal with unsupported values and should never see them.
rateUnitTokens = {
   RateUnit.bps : 'Rate in bps',
   RateUnit.kbps : 'Rate in kbps (default unit)',
   RateUnit.mbps : 'Rate in mbps',
   RateUnit.gbps : 'Rate in gbps',
}
if toggleTrafficPolicyPpsPolicingEnabled():
   rateUnitTokens[ RateUnit.pps ] = 'Rate in packets per second'

burstUnitTokens = {
   BurstUnit.bytes : 'Burst size in bytes (default unit)',
   BurstUnit.kbytes : 'Burst size in kbytes',
   BurstUnit.mbytes : 'Burst size in mbytes',
}
if toggleTrafficPolicyPpsPolicingEnabled():
   burstUnitTokens[ BurstUnit.packets ] = 'Burst size in packets'

icmpMsgTypeTokenToDropWithEnum = {
   'icmp' : {
      DropWithCliMsgType.icmpNetUnreachable.token :
         DropWithMessageType4.icmpNetUnreachable,
      DropWithCliMsgType.icmpHostUnreachable.token :
         DropWithMessageType4.icmpHostUnreachable,
      DropWithCliMsgType.icmpProtoUnreachable.token :
         DropWithMessageType4.icmpProtoUnreachable,
      DropWithCliMsgType.icmpPortUnreachable.token :
         DropWithMessageType4.icmpPortUnreachable,
      DropWithCliMsgType.icmpNetProhibited.token :
         DropWithMessageType4.icmpNetProhibited,
      DropWithCliMsgType.icmpHostProhibited.token :
         DropWithMessageType4.icmpHostProhibited,
      DropWithCliMsgType.icmpAdminProhibited.token :
         DropWithMessageType4.icmpAdminProhibited
   },
   'icmpv6' : {
      DropWithCliMsgType.icmp6NoRoute.token :
         DropWithMessageType6.icmp6NoRoute,
      DropWithCliMsgType.icmp6AdminProhibited.token :
         DropWithMessageType6.icmp6AdminProhibited,
      DropWithCliMsgType.icmp6AddrUnreachable.token :
         DropWithMessageType6.icmp6AddrUnreachable,
      DropWithCliMsgType.icmp6PortUnreachable.token :
         DropWithMessageType6.icmp6PortUnreachable
   }
}

icmpMsgTypeTokens = { msgType.token : msgType.helpMsg for msgType in
      [ DropWithCliMsgType.icmpNetUnreachable,
        DropWithCliMsgType.icmpHostUnreachable,
        DropWithCliMsgType.icmpProtoUnreachable,
        DropWithCliMsgType.icmpPortUnreachable,
        DropWithCliMsgType.icmpNetProhibited,
        DropWithCliMsgType.icmpHostProhibited,
        DropWithCliMsgType.icmpAdminProhibited,
      ] }

icmp6MsgTypeTokens = { msgType.token : msgType.helpMsg for msgType in
      [ DropWithCliMsgType.icmp6NoRoute,
        DropWithCliMsgType.icmp6AdminProhibited,
        DropWithCliMsgType.icmp6AddrUnreachable,
        DropWithCliMsgType.icmp6PortUnreachable,
      ] }

# packet limits will be in packets and should not
# require conversion
factors = {
      'bps' : 1e3,
      'kbps' : 1,
      'mbps' : 1e-3,
      'gbps' : 1e-6,
      'bytes' : 1,
      'kbytes' : 1e-3,
      'mbytes' : 1e-6
}

def policyHasAction( action, config, status, policyName ):
   if policyName in config.pmapType.pmap:
      currCfg = config.pmapType.pmap[ policyName ].currCfg
      for classAction in currCfg.classAction.values():
         for actionType in classAction.policyAction:
            if actionType == action:
               return True
   return False

def protectedTrafficPolicyNamesRegex():
   excludeKeywords = [ 'cpu', 'interface', 'field-set', 'vlan', 'fragment',
                       'enforcement' ]
   excludePattern = ''.join( BasicCliUtil.notAPrefixOf( k )
         for k in excludeKeywords )
   pattern = excludePattern + r'[A-Za-z0-9_:{}\[\]-]+'
   return pattern

def innerSfAddressFamily( matchType ):
   if matchType == 'mac':
      return 'ipunknown'
   return matchType

class TrafficPolicyMatchRuleAction( PolicyMapCliLib.PolicyRawRuleActionBase ):
   keyTag_ = 'matchRuleFilter'

   def __init__( self, trafficPolicyContext, ruleName, matchOption, sfilter ):
      super( TrafficPolicyMatchRuleAction, self ).__init__( trafficPolicyContext,
                                                            ruleName,
                                                            None,
                                                            matchOption )
      self.filter = sfilter

   def key( self ):
      return self.cmapName

   def addToPmap( self, pmap, seqnum ):
      super( TrafficPolicyMatchRuleAction, self ).addToPmap( pmap, seqnum )
      if self.cmapName in pmap.rawClassMap:
         cmap = pmap.rawClassMap[ self.cmapName ]
      else:
         cmap = pmap.rawClassMap.newMember( self.cmapName, UniqueId() )
      cmap.match.clear()
      self.cmapMatch = cmap.match.newMember( self.matchOption )
      self.cmapMatch.structuredFilter = ( "", )
      # When addRuleCommon is called, the match rule will be removed and re-added
      # this is needed in order to save structured filter
      self.cmapMatch.structuredFilter.copy( self.filter )
      # save the match description
      if self.filter.ruleDesc:
         cmap.matchDesc = self.filter.ruleDesc

   def actionCombinationError( self ):
      actions = self.actions()
      if 'log' in actions:
         if 'deny' not in actions:
            return "The 'log' action cannot be used without the 'drop' action"
      return None

#---------------------------------------------------------------
# The base config command for entering policy configuration
# i.e [ no|default ] keyword <policy-name>
#---------------------------------------------------------------
class PolicyConfigCmdBase( CliCommand.CliCommandClass ):
   @staticmethod
   def _feature():
      raise NotImplementedError

   @classmethod
   def _context( cls, name, mode ):
      raise NotImplementedError

   @classmethod
   def _removePolicy( cls, mode, name ):
      context = cls._context( name, mode )
      context.delPmap( mode, name )

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'POLICY_NAME' ]
      context = cls._context( name, mode )

      if context.hasPolicy( name ):
         context.copyEditPmap()
      else:
         context.newEditPmap()
      childMode = mode.childMode( context.childMode(),
                                  context=context,
                                  feature=cls._feature() )
      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      name = args[ 'POLICY_NAME' ]
      context = cls._context( name, mode )
      if context.hasPolicy( name ):
         cls._removePolicy( mode, name ) # pylint: disable=protected-access

#------------------------------------------------------------------------------------
# The "protocol tcp flags TCP_FLAGS"
#------------------------------------------------------------------------------------
class ProtocolTcpFlagsOnlyCmd( ProtocolMixin ):
   syntax = ( 'protocol FLAGS_EXPR' )
   noOrDefaultSyntax = syntax

   data = {
      'FLAGS_EXPR' : generateTcpFlagExpression( tcpFlagsSupported=True,
                                                notAllowed=True ),
      'protocol' : 'Protocol',
   }

   @classmethod
   def handler( cls, mode, args ):
      proto = args.get( cls._tcpFlagArgsListName )
      if not proto and not args:
         return
      cls._updateProtoAndPort( mode, args, proto, flags=True, add=True )
      cls._maybeHandleErrors( mode, args, proto )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      proto = args.get( cls._tcpFlagArgsListName )
      cls._updateProtoAndPort( mode, args, proto, flags=True, add=False )

class DefaultActionsBaseContext( object ):
   def __init__( self, trafficPolicyContext, matchOption ):
      self.trafficPolicyContext = trafficPolicyContext
      self.trafficPolicy = self.trafficPolicyContext.currentPolicy()
      self.trafficPolicyName = self.trafficPolicyContext.pmapName_
      self.matchOption = matchOption
      # Scratchpad that holds all changes within the childMode until
      # the policy is committed.
      self.editDefaultActions = self.initEditDefaultActions()

   @property
   def matchType( self ):
      if self.matchOption == matchMacAccessGroup:
         return 'mac'
      elif self.matchOption == matchIpAccessGroup:
         return 'ipv4'
      else:
         assert self.matchOption == matchIpv6AccessGroup
         return 'ipv6'

   def initEditDefaultActions( self ):
      editDefaultActions = Tac.newInstance( "PolicyMap::DefaultActions",
                                            self.matchOption )
      defaultActions = self.trafficPolicy.defaultAction.get( self.matchOption )
      if defaultActions is None:
         return editDefaultActions
      for action in defaultActions.action.values():
         editDefaultActions.action.addMember( action )
      return editDefaultActions

   def delAction( self, actionType ):
      '''
      Remove action from scratchpad.
      '''
      del self.editDefaultActions.action[ actionType ]

   def addAction( self, actionType, action ):
      '''
      Add action to scratchpad
      '''
      self.editDefaultActions.action.addMember( action )

   def childMode( self ):
      raise NotImplementedError

   def setAction( self, actionType, actionValue=None, no=False, clearActions=None ):
      raise NotImplementedError

   def commit( self ):
      # Remove stale actions
      prevDefaultActions = self.trafficPolicy.defaultAction.get( self.matchOption )
      if prevDefaultActions is not None:
         for actionType, prevAction in prevDefaultActions.action.items():
            currAction = self.editDefaultActions.action.get( actionType )
            if currAction is None or not currAction.equalTo( prevAction ):
               # Remove action from instantiating collection/container.
               self.trafficPolicyContext.removeAction( prevAction )
               del prevDefaultActions.action[ actionType ]
      elif self.editDefaultActions.action:
         self.trafficPolicy.defaultAction.newMember( self.matchOption )

      if not self.editDefaultActions.action:
         del self.trafficPolicy.defaultAction[ self.matchOption ]
         return

      # Copy all the current actions
      currDefaultAction = (
         self.trafficPolicy.defaultAction[ self.matchOption ] )
      for currAction in self.editDefaultActions.action.values():
         currDefaultAction.action.addMember( currAction )

class MatchRuleContext( MatchRuleBaseContext ):
   def __init__( self, trafficPolicyContext, ruleName, matchOption ):
      super( MatchRuleContext, self ).__init__( ruleName, matchOption )
      self.trafficPolicyContext = trafficPolicyContext
      self.trafficPolicy = self.trafficPolicyContext.currentPolicy()
      self.trafficPolicyName = self.trafficPolicyContext.pmapName_
      self.matchRuleAction = None
      # If this match rule existed previously, it will have a
      # structuredFilter containing the previous configuration.  Check
      # for this and copy any existing config to our new structuredFilter.
      if self.trafficPolicy:
         matchRule = self.trafficPolicy.rawClassMap.get( self.ruleName )
         if matchRule:
            if matchRule.match.newMember( self.matchOption ).structuredFilter \
               is not None:
               self.filter.copy(
                  matchRule.match[ self.matchOption ].structuredFilter )
      self.seqnum = 0

   @property
   def matchType( self ):
      if self.matchOption == matchMacAccessGroup:
         return 'mac'
      elif self.matchOption == matchIpAccessGroup:
         return 'ipv4'
      else:
         assert self.matchOption == matchIpv6AccessGroup
         return 'ipv6'

   def childMode( self, matchRuleName, matchOption ):
      if matchRuleName in self.trafficPolicyContext.reservedClassMapNames():
         # The default rules have their own mode
         configMode = MatchRuleDefaultConfigMode
      elif matchOption == matchIpAccessGroup:
         configMode = MatchRuleIpv4ConfigMode
      elif matchOption == matchMacAccessGroup:
         configMode = MatchRuleMacConfigMode
      else:
         assert matchOption == matchIpv6AccessGroup
         configMode = MatchRuleIpv6ConfigMode
      return configMode

   def updateMatchDesc( self, matchDesc, mode, add=True ):
      ruleId = 0
      if add:
         mId = getIdFromDesc( desc=matchDesc, descTypePolicy=False )
         maxRuleId = Tac.Value( 'Classification::Constants' ).maxRuleId
         if mId and mId not in range( 1, maxRuleId ):
            errStr = f'Rule id range should be from 1 to {maxRuleId}'
            return errStr
         ruleId = mId
      matchRule = self.trafficPolicy.rawClassMap.get( self.ruleName )
      # match rule exist
      if matchRule:
         matchRule.matchDesc = matchDesc
      sf = mode.trafficPolicyContext.matchRuleContext.filter
      sf.ruleId = ruleId
      sf.ruleDesc = matchDesc
      return ''

   def actionMode( self ):
      if self.matchOption == matchMacAccessGroup:
         return MacActionsConfigMode
      return ActionsConfigMode

   def setAggGroupActionBase( self, action, actionValue=None, no=False ):
      cmdDict = actionValue
      groupKey = 'groups'
      intfKey = 'intfs'
      # Either 'groups' or 'intfs' has to be in actionValue, but not both
      assert ( groupKey in cmdDict ) ^ ( intfKey in cmdDict )

      # Get correct intf list from actionValue
      if intfKey in cmdDict:
         ( _, intfList ) = MirroringPolicyMapClassModeTapAgg.getIntfOpAndList(
               ( '', cmdDict[ intfKey ] ) )
         cmdDict[ intfKey ] = intfList

      newDict = {}
      newDict[ groupKey ] = newDict[ intfKey ] = []
      curDict = {}
      if action:
         curDict[ groupKey ] = list( action.aggGroup )
         curDict[ intfKey ] = list( action.aggIntf )
      else:
         curDict[ groupKey ] = []
         curDict[ intfKey ] = []

      changedKey = groupKey if groupKey in cmdDict else intfKey
      unchangedKey = intfKey if groupKey in cmdDict else groupKey
      newDict[ unchangedKey ] = curDict[ unchangedKey ]

      if no and not cmdDict[ changedKey ]:
         newDict[ changedKey ] = []
      else:
         if no:
            newDict[ changedKey ] = [ element for element in curDict[ changedKey ]
                                      if element not in cmdDict[ changedKey ] ]
         else:
            newDict[ changedKey ] = curDict[ changedKey ] + cmdDict[ changedKey ]

      return newDict

   def setAggGroupAction( self, actionType, actionValue=None, no=False ):
      assert actionType == ActionType.setAggregationGroup
      groupKey = 'groups'
      intfKey = 'intfs'

      action = self.matchRuleAction.policyActions.get( actionType )
      newDict = self.setAggGroupActionBase( action, actionValue, no )

      self.matchRuleAction.delAction( actionType )
      if not newDict[ intfKey ] and not newDict[ groupKey ]:
         return
      actionsConfig = self.trafficPolicyContext.config().actions
      action = actionsConfig.setAggGroupAction.newMember(
            self.trafficPolicyName, UniqueId() )
      for group in newDict[ groupKey ]:
         action.aggGroup[ group ] = True
      for intf in newDict[ intfKey ]:
         action.aggIntf[ intf ] = True
      self.matchRuleAction.addAction( actionType, action )

   def setNexthopGroupAction( self, actionType, actionValue=None, no=False ):
      assert actionType == ActionType.setNexthopGroup
      action = self.matchRuleAction.policyActions.get( actionType )

      if action:
         curList = list( action.nexthopGroup )
      else:
         curList = []
      cmdList, ttlValue = actionValue
      newList = []

      # check for TTL
      ttlAction = self.matchRuleAction.policyActions.get( ActionType.setTtl )
      if ttlAction:
         curTtl = ttlAction.ttl
      else:
         curTtl = None

      if no and not cmdList:
         newList = []
      elif no:
         newList = [ group for group in curList if group not in cmdList ]
      else:
         newList = curList + cmdList

      self.matchRuleAction.delAction( actionType )
      self.matchRuleAction.delAction( ActionType.setTtl )
      if not newList:
         return
      actionsConfig = self.trafficPolicyContext.config().actions
      action = actionsConfig.setNexthopGroupAction.newMember(
            self.trafficPolicyName, UniqueId() )
      for group in newList:
         action.nexthopGroup[ group ] = True
      self.matchRuleAction.addAction( actionType, action )
      if no and ttlValue != 0:
         return
      elif ttlValue == 0 and curTtl is not None: # preserve the previous behavior
         ttlValue = curTtl
      if ttlValue != 0:
         newAction = actionsConfig.setTtlAction.newMember(
                        self.trafficPolicyName, UniqueId(), ttlValue )
         self.matchRuleAction.addAction( ActionType.setTtl, newAction )

   def setRedirectTunnelAction( self, actionType, actionValue=None, no=False ):
      assert actionType == ActionType.redirectTunnel

      if no:
         self.matchRuleAction.delAction( actionType )
         return

      curIntf = None
      curAction = self.matchRuleAction.policyActions.get( actionType )
      if curAction:
         curIntf = curAction.tunnelIntf
      newIntf = actionValue

      if newIntf == curIntf: # No change
         return

      # Program the action
      self.matchRuleAction.delAction( actionType )
      actionsConfig = self.trafficPolicyContext.config().actions
      action = actionsConfig.redirectTunnelAction.newMember(
            self.trafficPolicyName, UniqueId() )
      action.tunnelIntf = newIntf
      self.matchRuleAction.addAction( actionType, action )

   def delAllAction( self ):
      policyActions = list( self.matchRuleAction.actions() )
      for action in policyActions:
         self.matchRuleAction.delAction( action )

   def setAction( self, actionType, actionValue=None, no=False, clearActions=None ):
      if actionType == ActionType.setAggregationGroup:
         self.setAggGroupAction( actionType, actionValue, no )
      elif actionType == ActionType.setNexthopGroup:
         self.setNexthopGroupAction( actionType, actionValue, no )
      elif actionType == ActionType.redirectTunnel:
         self.setRedirectTunnelAction( actionType, actionValue, no )
      elif no:
         if actionType == ActionType.count and actionValue != "":
            action = self.matchRuleAction.policyActions.get( actionType )
            if action and action.counterName != actionValue:
               # Counter being deleted is not present in match rule, ignore request
               return
         if actionType == ActionType.setNexthop:
            action = self.matchRuleAction.policyActions.get( ActionType.setTtl )
            if action:
               self.matchRuleAction.delAction( ActionType.setTtl )
            self.matchRuleAction.delAction( actionType )
         elif actionType == ActionType.actionSet:
            actionsConfig = self.trafficPolicyContext.config().actions
            origAction = self.matchRuleAction.policyActions.get( actionType )
            name = self._getGrasetName( actionValue )
            if origAction and name in origAction.namedActionSet:
               self.matchRuleAction.delAction( actionType )

               action = actionsConfig.replicateAction.newMember(
                  self.trafficPolicyName, UniqueId() )
               action.copyAction( origAction )

               del action.namedActionSet[ name ]
               if action.namedActionSet:
                  self.matchRuleAction.addAction( actionType, action )
               else:
                  self.matchRuleAction.delAction( actionType )
         else:
            self.matchRuleAction.delAction( actionType )
      else:
         if clearActions:
            for action in clearActions:
               self.matchRuleAction.delAction( action )
         actionsConfig = self.trafficPolicyContext.config().actions
         if actionType == ActionType.deny:
            action = actionsConfig.dropAction.newMember( self.trafficPolicyName,
                                                         UniqueId() )
            action.msgType = actionValue
         elif actionType == ActionType.count:
            action = actionsConfig.countAction.newMember( self.trafficPolicyName,
                                                          UniqueId(),
                                                          actionValue )
         elif actionType == ActionType.log:
            action = actionsConfig.logAction.newMember( self.trafficPolicyName,
                                                        UniqueId() )
         elif actionType == ActionType.actionGoto:
            action = actionsConfig.gotoAction.newMember( self.trafficPolicyName,
                                                         UniqueId(),
                                                         actionValue )
         elif actionType == ActionType.police:
            action = actionsConfig.policeAction.newMember( self.trafficPolicyName,
                                                           UniqueId(),
                                                           actionValue[ 0 ],
                                                           actionValue[ 1 ] )
         elif actionType == ActionType.sample:
            action = actionsConfig.sampleAction.newMember( self.trafficPolicyName,
                                                           UniqueId() )
         elif actionType == ActionType.sampleAll:
            action = actionsConfig.sampleAllAction.newMember( self.trafficPolicyName,
                                                              UniqueId() )
         elif actionType == ActionType.useVrfSecondary:
            action = (
               actionsConfig.useVrfSecondaryAction.newMember( self.trafficPolicyName,
                                                              UniqueId() ) )
         elif actionType == ActionType.setDscp:
            action = actionsConfig.setDscpAction.newMember( self.trafficPolicyName,
                                                            UniqueId(),
                                                            actionValue )
         elif actionType == ActionType.setTc:
            action = actionsConfig.setTcAction.newMember( self.trafficPolicyName,
                                                          UniqueId(),
                                                          actionValue )
         elif actionType == ActionType.setIdentityTag:
            action = actionsConfig.setIdTagAction.newMember( self.trafficPolicyName,
                                                             UniqueId(),
                                                             actionValue )
         elif actionType == ActionType.stripHeaderBytes:
            action = actionsConfig.stripHdrBytesAction.newMember(
                  self.trafficPolicyName, UniqueId() )
            hdrInfo = actionValue
            hdrType = TapAggPmapCliLib.stripHdrDot1q
            stripHdr = TapAggPmapCliLib.stripHdrAction( hdrType, hdrInfo )
            action.stripHdrBytes = stripHdr
         elif actionType == ActionType.setNexthop:
            action = actionsConfig.setNexthopAction.newMember(
               self.trafficPolicyName, UniqueId() )
            ( ipAddrList, recursive, vrfName, ttlValue ) = actionValue

            for ipAddr in ipAddrList:
               action.nexthop[ IpGenAddr( str( ipAddr ) ) ] = True
            if recursive:
               action.recursive = True
            if vrfName:
               action.vrfName = vrfName
            # delete previous ttl action if it exists
            self.matchRuleAction.delAction( ActionType.setTtl )
            if ttlValue != 0:
               newAction = actionsConfig.setTtlAction.newMember(
                           self.trafficPolicyName, UniqueId(), ttlValue )
               self.matchRuleAction.addAction( ActionType.setTtl, newAction )
         elif actionType == ActionType.setTimestampHeader:
            action = actionsConfig.setTimestampHeaderAction.newMember(
                  self.trafficPolicyName, UniqueId() )
         elif actionType == ActionType.mirror:
            action = actionsConfig.mirrorAction.newMember( self.trafficPolicyName,
                                                           UniqueId() )
            action.mirrorSession = actionValue
         elif actionType == ActionType.setMacAddress:
            action = actionsConfig.setMacAddressAction.newMember(
                  self.trafficPolicyName, UniqueId(), actionValue )
         elif actionType == ActionType.sflow:
            action = actionsConfig.sflowAction.newMember( self.trafficPolicyName,
                                                          UniqueId() )
         elif actionType == ActionType.setHeaderRemove:
            action = actionsConfig.setHeaderRemoveAction.newMember(
               self.trafficPolicyName, UniqueId(), actionValue )
         elif actionType == ActionType.setDecapVrf:
            action = actionsConfig.setdDecapVrfAction.newMember(
               self.trafficPolicyName, UniqueId(),
               actionValue[ 0 ],
               actionValue[ 1 ],
               actionValue[ 2 ] )
         elif actionType == ActionType.actionSet:
            origAction = self.matchRuleAction.policyActions.get( actionType )
            self.matchRuleAction.delAction( actionType )

            action = actionsConfig.replicateAction.newMember(
               self.trafficPolicyName, UniqueId() )
            if origAction:
               action.copyAction( origAction )

            name = self._getGrasetName( actionValue )
            if name not in action.namedActionSet:
               action.namedActionSet.newMember( name )

            for actionSet in action.namedActionSet:
               action.updatedActionSet[ actionSet ] = True
            self.matchRuleAction.addAction( actionType, action )
         else:
            assert False, 'unknown action ' + actionType
         self.matchRuleAction.addAction( actionType, action )

   def _getGrasetName( self, shortName ):
      return f'grasetTP%{self.trafficPolicyName}%{self.ruleName}%{shortName}'

   def copyEditMatchRule( self, ruleName, seqnum ):
      self.seqnum = seqnum
      self.matchRuleAction = TrafficPolicyMatchRuleAction( self.trafficPolicyContext,
                                                           ruleName,
                                                           self.matchOption,
                                                           self.filter )
      classAction = self.trafficPolicy.classAction[ ruleName ]
      for actType, action in six.iteritems( classAction.policyAction ):
         self.matchRuleAction.originalActions.add( action )
         self.matchRuleAction.addAction( actType, action )

      # set the cmapMatch
      matchRule = self.trafficPolicy.rawClassMap.get( ruleName )
      self.matchRuleAction.cmapMatch = matchRule.match[ self.matchOption ]

   def newEditMatchRule( self, ruleName, seqnum ):
      if seqnum is None:
         seqnum = 0
      self.seqnum = seqnum
      self.matchRuleAction = TrafficPolicyMatchRuleAction( self.trafficPolicyContext,
                                                           ruleName,
                                                           self.matchOption,
                                                           self.filter )
   def commit( self ):
      # commit all pending rule changes
      self.trafficPolicyContext.addRuleCommon( self.seqnum, self.matchRuleAction )

class ReplicateContext( MatchRuleContext ):
   def __init__( self, matchRuleContext, mode, name ):
      self.__dict__.update( matchRuleContext.__dict__ )
      self.mode = mode
      self.actionsSetName = self._getGrasetName( name )
      self.actionsSetShortName = name
      self.trafficPolicy = matchRuleContext.trafficPolicy

   def setAction( self, actionType, actionValue=None, no=False, clearActions=None ):
      replicateAction = self.matchRuleAction.policyActions.get(
            ActionType.actionSet )
      actionSet = replicateAction.namedActionSet[ self.actionsSetName ]
      identity = UniqueId()
      pname = self.trafficPolicyName
      rname = self.ruleName

      if actionType == ActionType.setAggregationGroup:
         actionSet.internalGroupName = self.actionsSetName
         groupKey = 'groups'
         intfKey = 'intfs'

         oldAction = actionSet.setAggGroup
         newDict = self.setAggGroupActionBase( oldAction, actionValue, no )
         actionSet.setAggGroup = ( pname, identity, )
         if not newDict[ intfKey ] and not newDict[ groupKey ]:
            actionSet.setAggGroup = None
            return
         action = actionSet.setAggGroup
         for group in newDict[ groupKey ]:
            action.aggGroup[ group ] = True
         for intf in newDict[ intfKey ]:
            action.aggIntf[ intf ] = True
      elif actionType == ActionType.setIdentityTag:
         if no:
            actionSet.setIdentityTag = None
         else:
            actionSet.setIdentityTag = ( pname, identity, actionValue )
      elif actionType == ActionType.setMacAddress:
         if no:
            actionSet.setMacAddress = None
         else:
            actionSet.setMacAddress = ( pname, identity, actionValue )
      elif actionType == ActionType.stripHeaderBytes:
         if no:
            actionSet.stripHeaderBytes = None
         else:
            actionSet.stripHeaderBytes = ( pname, identity )
            hdrInfo = actionValue
            hdrType = TapAggPmapCliLib.stripHdrDot1q
            stripHdr = TapAggPmapCliLib.stripHdrAction( hdrType, hdrInfo )
            actionSet.stripHeaderBytes.stripHdrBytes = stripHdr
      else:
         assert False, f'action { actionType } not supported in action set'

   def commit( self ):
      # Action sets are rejected if they don't have their own redirect action.
      # Check here and log a warning if the set is to be rejected.
      replicateAction = self.matchRuleAction.policyActions.get(
                  ActionType.actionSet )
      actionSet = replicateAction.namedActionSet[ self.actionsSetName ]
      if not actionSet.setAggGroup:
         self.mode.addWarning( "Replication action sets are not deployed unless "
                               "they contain the redirect action" )

class PacketTypeContext( MatchRuleContext ):
   def __init__( self, trafficPolicyContext, outerMatchRuleContext ):
      self.trafficPolicyContext = trafficPolicyContext
      self.matchRuleContext = outerMatchRuleContext
      self.filter = Tac.newInstance( "Classification::StructuredFilter", "" )
      if self.matchRuleContext.filter:
         self.filter.copy( self.matchRuleContext.filter )

   def updatePacketTypeSet( self, cliPktType, exclude=None, no=False ):
      # need to improve this method once packet type reqs are fleshed out
      # to make it more generic for all packet types
      acceptedPacketTypes = [ 'vxlanDecap', 'multicast' ]
      if cliPktType not in acceptedPacketTypes:
         return
      if cliPktType == 'multicast' and exclude:
         # Multicast packet type should not set exclude option
         return

      tacPacketType = Tac.Type( "Classification::PacketType" )
      packetFlag = Tac.Value( "Classification::PacketFlag" )
      packetFlagMask = Tac.Value( "Classification::PacketFlag" )

      setattr( packetFlagMask, cliPktType, True )
      setattr( packetFlag, cliPktType, not exclude )
      currPktType = tacPacketType( packetFlag, packetFlagMask )

      # Check if pkt type contained within packet set, if so remove and then re-add
      # (if adding) with appropriate mask - needed for exclude case.
      for pktType in self.filter.packetTypeSet:
         if pktType.packetFlagMask and hasattr( pktType.packetFlagMask, cliPktType ):
            self.filter.packetTypeSet.remove( currPktType )
            break
      if not no:
         self.filter.packetTypeSet.add( currPktType )

   def delAllPacketType( self ):
      if self.filter and self.filter.packetTypeSet:
         self.filter.packetTypeSet.clear()

   def commit( self ):
      self.matchRuleContext.filter.copy( self.filter )

class InnerMatchRuleContext( MatchRuleContext ):
   def __init__( self, trafficPolicyContext, outerMatchRuleContext,
                 encapType, matchOption ):
      self.trafficPolicyContext = trafficPolicyContext
      self.matchRuleContext = outerMatchRuleContext
      self.encapType = encapType
      self.matchOption = matchOption
      self.filter = Tac.newInstance( "Classification::StructuredFilterInner", "",
            innerSfAddressFamily( self.matchType ) )
      # If inner match existed previously, it will have an innerMatch as part of
      # the matchRuleContext.filter.innerMatch. Copy existing innerMatch to
      # local filter. We will commit changes to matchRuleContext on commit
      if self.matchRuleContext.filter.innerMatch:
         self.filter.copy( self.matchRuleContext.filter.innerMatch )

   def copyEditMatchRule( self, ruleName, seqnum ):
      pass

   def newEditMatchRule( self, ruleName, seqnum ):
      pass

   def encapMode( self ):
      if self.matchType == 'ipv4':
         if self.encapType == 'vxlan':
            return EncapInnerVxlanMatchIpv4ConfigMode
         return EncapInnerMatchIpv4ConfigMode
      elif self.matchType == 'ipv6':
         return EncapInnerMatchIpv6ConfigMode
      assert self.matchType == 'mac'
      return EncapInnerMatchMacConfigMode

   def commit( self ):
      if not self.matchRuleContext.filter.innerMatch:
         self.matchRuleContext.filter.innerMatch = ( "",
               innerSfAddressFamily( self.matchType ) )
      self.matchRuleContext.filter.innerMatch.copy( self.filter )
      # Today, we only support a single encap type so it is safe to always override
      self.matchRuleContext.filter.encapField = ( self.encapType, )

#--------------------------------------------------------------------------------
# The "encapsulation ENCAP_TYPE ( ipv4|ipv6|mac )" command
#--------------------------------------------------------------------------------
encapTypeMatcher = CliMatcher.EnumMatcher( { 'gtpv1' : 'GTPv1 Encapsulation' } )

encapTypeVxlanMatcher = CliMatcher.EnumMatcher(
      { 'vxlan' : 'VXLAN Encapsulation' } )

class InnerMatchCmdBase( CliCommand.CliCommandClass ):
   @staticmethod
   def _feature():
      raise NotImplementedError

   @staticmethod
   def _context( trafficPolicyContext, matchRuleContext, encapType, matchOption ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      encapType = args[ 'ENCAP_TYPE' ]
      af = args[ 'AF' ]
      if encapType == 'vxlan' and af not in [ 'mac', 'ipv4' ]:
         return
      if encapType == 'gtpv1' and af not in [ 'ipv4', 'ipv6' ]:
         return
      if af == 'ipv4':
         matchOption = matchIpAccessGroup
      elif af == 'ipv6':
         matchOption = matchIpv6AccessGroup
      else:
         assert af == 'mac'
         matchOption = matchMacAccessGroup
      context = mode.getContext() # outer match context
      if context.hasOtherEncapType( encapType ):
         otherEncapType = 'gtpv1' if encapType == 'vxlan' else 'vxlan'
         cmd = "encapsulation {} <match type>".format( otherEncapType )
         warning = "Trying to edit encapsulation match with incorrect encap type. " \
                   "Did you mean '%s'?" % ( cmd )
         mode.addWarning( warning )
         return
      # Check if inner match has already been created for the opposing match type.
      # If so, print a warning to let the user known and return
      newMatchType = 'ipv4' if matchOption == matchIpAccessGroup else 'ipv6'
      if encapType == 'gtpv1' and context.hasOtherInnerMatch( newMatchType ):
         otherMatchType = ( 'ipv4' if matchOption == matchIpv6AccessGroup else
                            'ipv6' )
         cmd = "encapsulation %s %s" % ( encapType, otherMatchType )
         warning = "Trying to edit encapsulation match with incorrect match type. " \
                   "Did you mean '%s'?" % ( cmd )
         mode.addWarning( warning )
         return

      encapInnerContext = cls._context( mode.trafficPolicyContext,
                                        mode.trafficPolicyContext.matchRuleContext,
                                        encapType,
                                        matchOption )
      childMode = mode.childMode( encapInnerContext.encapMode(),
                                  trafficPolicyContext=mode.trafficPolicyContext,
                         matchRuleContext=mode.trafficPolicyContext.matchRuleContext,
                         innerRuleContext=encapInnerContext,
                         feature=cls._feature() )

      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      sf = mode.trafficPolicyContext.matchRuleContext.filter
      sf.encapField = None
      sf.innerMatch = None
