#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import threading
import sys
import BasicCli
import Cell
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
from CliCommon import AlreadyHandledError
import CliPlugin.VrfCli as VrfCli
from CliPlugin.ClassificationCliLib import protectedFieldSetNamesRegex
from CliPlugin.Ip6AddrMatcher import Ip6PrefixMatcher
from CliPlugin.IpAddrMatcher import IpPrefixMatcher, PREFIX_OVERLAP_REJECT
from CliPlugin.MacAddr import MacAddrMatcher
from CliPlugin.TrafficPolicyCliLib import protectedTrafficPolicyNamesRegex
from CliPlugin.TrafficPolicyCliModel import ( TrafficPolicyModel,
                                              TrafficPolicySummaryModel,
                                              TrafficPolicyVrfIntfOverrideModel,
                                              ProtocolNeighborsModel,
                                              ProtocolNeighbor,
                                              VrfRules, NeighborRule,
                                              TrafficPolicyErrorsModel,
                                              TrafficPolicyCountersModel,
                                              TrafficPolicyPrefixFieldSetModel,
                                              TrafficPolicyL4PortFieldSetModel,
                                              TrafficPolicyServiceFieldSetModel,
                                              TrafficPolicyVlanFieldSetModel,
                                              TrafficPolicyIntegerFieldSetModel,
                                              MacTrafficPolicyMacFieldSetModel,
                                              TrafficPolicyMacFieldSetModel )
from CliPlugin.PolicyMapModelImpl import (
      waitForPmapCounters,
      waitForAllPmapCounters,
      kickPmapCounterPoll,
)
from MultiRangeRule import MultiRangeMatcher
from Ark import synchronized
from Arnet import IpGenPrefix, EthAddr
import LazyMount
import SharkLazyMount
import ShowCommand
import Tac
import Tracing
from TypeFuture import TacLazyType
import six

t0 = Tracing.trace0

mergeSmLock = threading.Lock()

gv = CliGlobal.CliGlobal( dict( fieldSetConfig=None,
                                appRecConfig=None,
                                fieldSetConfigDir=None,
                                fieldSetMergeSm=None,
                                l3IntfConfig=None,
                                routingHwRouteStatus=None ) )
intfPolicyGV = CliGlobal.CliGlobal( dict( config=None, # Merged
                                          ingressIntfConfig=None, # Merged
                                          ingressVrfConfig=None,
                                          configMergeSm=None,
                                          entityManager=None,
                                          configCli=None,
                                          configDir=None,
                                          ingressIntfConfigDir=None,
                                          ingressIntfConfigCli=None,
                                          vtiStatusDir=None,
                                          dynVlanIntfDir=None ) )
config = None
entityManager = None
vrfConfig = None
cpuIntfConfig = None
enforcementConfig = None
statusDir = None
l3IntfStatus = None
cpuCounter = None
intfInputCounter = None
intfOutputCounter = None
vaclCounter = None
aegisOutputIntfConfig = None
aegisBridgedIntfConfig = None
intfParamStatus = None
interfaceTrafficPolicyHwStatus = None
counterStatusInput = None
counterStatusOutput = None
hwStatus = None
cpuHwStatus = None
tapAggCliConfig = None
switchIntfConfigDir = None


appConstants = TacLazyType( 'Classification::Constants' )
PortRange = TacLazyType( 'Classification::PortRange' )
VlanRange = TacLazyType( 'Classification::VlanRange' )
NumericalRange = TacLazyType( 'Classification::NumericalRange' )
ProtocolRange = TacLazyType( "Classification::ProtocolRange" )
PolicyMapCountersKey = TacLazyType( "PolicyMap::Counters::PolicyMapKey" )
ProtocolRange = TacLazyType( "Classification::ProtocolRange" )
NeighborProtocol = TacLazyType( "Classification::NeighborProtocol" )
protocolBgpTtlSec = NeighborProtocol.protocolBgpTtlSec

def interfaceGuard( mode, token ):
   # We support ingress traffic-policy on Strata through
   # DMF but not through the CLI. So add a guard here for
   # Strata
   if not hwStatus.ingressTrafficPolicyCliSupported:
      return CliParser.guardNotThisPlatform

   if ( hwStatus.ingressTrafficPolicySupported or
         hwStatus.egressTrafficPolicySupported ):
      return None
   return CliParser.guardNotThisPlatform

def cpuGuard( mode, token ):
   if cpuHwStatus.cpuTrafficPolicySupported:
      return None
   return CliParser.guardNotThisPlatform

def vaclGuard( mode, token ):
   if hwStatus.bridgedTrafficPolicySupported:
      return None
   return CliParser.guardNotThisPlatform

def ingressGuard( mode, token ):
   # We support ingress traffic-policy on Strata through
   # DMF but not through the CLI. So add a guard here for
   # Strata
   if not hwStatus.ingressTrafficPolicyCliSupported:
      return CliParser.guardNotThisPlatform

   if hwStatus.ingressTrafficPolicySupported:
      return None
   return CliParser.guardNotThisPlatform

def egressGuard( mode, token ):
   if hwStatus.egressTrafficPolicySupported:
      return None
   return CliParser.guardNotThisPlatform

def aegisIntfOverrideGuard( mode, token ):
   if hwStatus.intfAppAlwaysOverride:
      return None
   return CliParser.guardNotThisPlatform

def showTrafficPolicy( mode, policyName, cpu, interface, vlan,
                       summary, errors, detail, counters,
                       inputPolicies, outputPolicies, excludedVrfIntfShow ):
   t0( "showTrafficPolicy", policyName, cpu, interface, summary, errors )
   if cpu:
      statusInput = ( statusDir.get( 'cpu-vrf-hardware' ) or
                      statusDir.get( 'cpu-vrf-software' ) )
      statusOutput = None
      if statusInput is not None:
         LazyMount.force( config )
         LazyMount.force( vrfConfig )
         LazyMount.force( cpuIntfConfig )
         LazyMount.force( l3IntfStatus )
         LazyMount.force( cpuHwStatus )
         mostRecentCounter = countersSnapshot( mode, "cpu" )
         byteCountersSupported = False
         filterParams = Tac.Value( 'TrafficPolicyCli::FilterParams',
                                   policyName, summary, errors, detail, counters,
                                   byteCountersSupported=byteCountersSupported )
         helper = Tac.newInstance( 'TrafficPolicyCli::ShowTrafficPolicyCpu',
                                   filterParams, config, statusDir,
                                   mostRecentCounter,
                                   gv.fieldSetConfig,
                                   cpuHwStatus,
                                   enforcementConfig,
                                   None, # Shark counter status
                                   vrfConfig, cpuIntfConfig,
                                   VrfCli.allVrfConfig, l3IntfStatus )
      else:
         t0( "Feature not present to report TrafficPolicy cpu status" )
      sharkCountersSupported = False
   elif interface:
      statusInput = statusDir.get( 'interfaceInput' )
      statusOutput = statusDir.get( 'interfaceOutput' )
      sharkCountersSupported = (
            interfaceTrafficPolicyHwStatus.sharkCountersSupported )

      if statusInput is not None and statusOutput is not None:
         LazyMount.force( aegisOutputIntfConfig )
         LazyMount.force( intfParamStatus )
         LazyMount.force( tapAggCliConfig )
         LazyMount.force( switchIntfConfigDir )
         LazyMount.force( intfPolicyGV.ingressVrfConfig )
         LazyMount.force( gv.l3IntfConfig )
         LazyMount.force( gv.routingHwRouteStatus )
         counterStatusInputEntity = SharkLazyMount.force( counterStatusInput )
         counterStatusOutputEntity = SharkLazyMount.force( counterStatusOutput )
         mostRecentInputCounter = countersSnapshot( mode, "interfaceInput" )
         mostRecentOutputCounter = countersSnapshot( mode, "interfaceOutput" )
         byteCountersSupported = True
         # We want to start the interface config mergeSm only when we know that
         # we are on a platform that supports interface traffic-policy. It is
         # possible for platforms to include TrafficPolicy rpms but not support
         # interface traffic-policy.
         if intfPolicyGV.configMergeSm is None:
            startIntfConfigMergeSm()
         filterParams = Tac.Value( 'TrafficPolicyCli::FilterParams',
                                   policyName, summary, errors, detail, counters,
                                   byteCountersSupported=byteCountersSupported,
                                   excludedVrfIntfShow=excludedVrfIntfShow )
         helper = Tac.newInstance( 'TrafficPolicyCli::ShowTrafficPolicyInterface',
                                   filterParams, intfPolicyGV.config, statusDir,
                                   mostRecentInputCounter,
                                   gv.fieldSetConfig,
                                   gv.appRecConfig,
                                   interfaceTrafficPolicyHwStatus,
                                   counterStatusInputEntity,
                                   mostRecentOutputCounter,
                                   intfPolicyGV.ingressIntfConfig,
                                   aegisOutputIntfConfig,
                                   intfParamStatus,
                                   counterStatusOutputEntity,
                                   inputPolicies,    # filterInputPolicies
                                   outputPolicies,   # filterOutputPolicies
                                   tapAggCliConfig,
                                   switchIntfConfigDir,
                                   gv.routingHwRouteStatus,
                                   intfPolicyGV.ingressVrfConfig.vrfConfig,
                                   intfPolicyGV.vtiStatusDir,
                                   intfPolicyGV.dynVlanIntfDir,
                                   gv.l3IntfConfig )
      else:
         t0( "Feature not present to report TrafficPolicy interface status" )
   elif vlan:
      statusInput = statusDir.get( 'strataAegisBridged' )
      statusOutput = None
      if statusInput is not None:
         LazyMount.force( aegisBridgedIntfConfig )
         mostRecentCounter = countersSnapshot( mode, "vlan" )
         byteCountersSupported = False
         filterParams = Tac.Value( 'TrafficPolicyCli::FilterParams',
                                   policyName, summary, errors, detail, counters,
                                   byteCountersSupported=byteCountersSupported )
         counterStatusVaclEntity = SharkLazyMount.force( counterStatusInput )
         helper = Tac.newInstance( 'TrafficPolicyCli::ShowTrafficPolicyVAcl',
                                   filterParams, config,
                                   statusDir, mostRecentCounter,
                                   gv.fieldSetConfig,
                                   interfaceTrafficPolicyHwStatus,
                                   counterStatusVaclEntity, # Shark counter status
                                   aegisBridgedIntfConfig )
      else:
         t0( "Feature not present to report TrafficPolicy vlan status" )
      # Due to BUG1004403, we are incorrectly using ingress counter mount paths
      # for vacl, since ingress counters use shark we want to set shark to True
      # too.
      sharkCountersSupported = True
   else:
      assert False, "invalid command"

   if statusInput or statusOutput:
      fd = sys.stdout.fileno()
      fmt = mode.session_.outputFormat()
      revision = mode.session_.requestedModelRevision()

      if not summary:
         configPath = config
         if intfPolicyGV.configMergeSm is None and interface:
            startIntfConfigMergeSm()
            configPath = intfPolicyGV.config
         # TODO: Sand does not publish counters for interface traffic-policy
         # if the feature is not supported. To avoid major changes for now,
         # just putting hard-coded check for interface here.
         if statusInput:
            handleCounters( configPath, statusInput, counterStatusInput,
                            sharkCountersSupported, allowNotPublished=interface,
                            policyName=policyName )
         if statusOutput:
            handleCounters( configPath, statusOutput, counterStatusOutput,
                            sharkCountersSupported, allowNotPublished=interface,
                            policyName=policyName )

      helper.render( fd, fmt, revision )

   if summary:
      return TrafficPolicySummaryModel
   elif errors:
      return TrafficPolicyErrorsModel
   elif counters:
      return TrafficPolicyCountersModel
   elif excludedVrfIntfShow:
      return TrafficPolicyVrfIntfOverrideModel
   else:
      return TrafficPolicyModel

def handleCounters( policyConfig, policyStatusDir, counterStatusDir,
                    sharkCountersSupported, allowNotPublished,
                    policyName ):

   def expectNotPublished():
      return allowNotPublished and ( not policyStatusDir.countersSupported )

   if expectNotPublished():
      return

   def pollCounters( localPmapName ):
      if sharkCountersSupported:
         pmapKey = PolicyMapCountersKey( localPmapName, '' )
         # This waitFor is here in case the feature agent is not the Shark
         # writer. If PolicyMap is writing to Shark, the lambda will return
         # true right away.
         Tac.waitFor( lambda key=pmapKey: (
                         counterStatusDir.policyMapCounters.get( key ) or
                         expectNotPublished() ),
                      description="counter status entity to be created",
                      sleep=True )
         # In case policyStatusDir.countersSupported gets set after the initial
         # check.
         # TODO: This could also happen during waitForPmapCounters, but not
         # updating that for now
         if expectNotPublished():
            return False
         policyMapCounters = counterStatusDir.policyMapCounters[ pmapKey ]
      kickPmapCounterPoll( localPmapName, policyConfig.pmapType )
      return True

   # First poll all necessary counters, then wait for all counter updates to be
   # completed.
   if policyName:
      pmapStatus = policyStatusDir.status.get( policyName )
      if not pmapStatus or not pollCounters( policyName ):
         return
      counterStatus = pmapStatus
      if sharkCountersSupported:
         pmapKey = PolicyMapCountersKey( policyName, '' )
         counterStatus = counterStatusDir.policyMapCounters[ pmapKey ]
      waitForPmapCounters( policyName, policyConfig.pmapType,
                          [ counterStatus ],
                          sharkEnabled=sharkCountersSupported, kickPoll=False )
   else:
      pmapNamePmapCounterStatus = []
      waitForFn = None
      for policy, pStatus in policyStatusDir.status.items():
         if not pollCounters( policy ):
            return
         if sharkCountersSupported:
            pmapKey = PolicyMapCountersKey( policy, '' )
            policyMapCounters = counterStatusDir.policyMapCounters[ pmapKey ]
            counterStatus = [ policyMapCounters ]
         else:
            counterStatus = [ pStatus ]
         pmapNamePmapCounterStatus.append( ( policy, counterStatus ) )
      if pmapNamePmapCounterStatus:
         waitForAllPmapCounters( pmapNamePmapCounterStatus, policyConfig.pmapType,
                                 sharkEnabled=sharkCountersSupported )

   if sharkCountersSupported:
      Tac.waitFor( lambda: counterStatusDir.sharkControl.quiesced,
                   description="Shark writer to quiesce",
                   sleep=True )

# Returns the latest counter snapshot by looking at global snapshot and
# session snapshot timestamps.
def countersSnapshot( mode, counterType ):
   # get global snapshot
   counter = None
   if counterType == "cpu":
      counter = cpuCounter
      sessionSnapshot = mode.session.sessionData( 'cpuTrafficPolicySessionCounter',
                                                  None )
   elif counterType == "interfaceInput":
      counter = intfInputCounter
      sessionSnapshot = (
         mode.session.sessionData( 'intfInputTrafficPolicySessionCounter', None )
      )
   elif counterType == "interfaceOutput":
      counter = intfOutputCounter
      sessionSnapshot = (
         mode.session.sessionData( 'intfOutputTrafficPolicySessionCounter', None )
      )
   elif counterType == "vlan":
      counter = intfInputCounter
      sessionSnapshot = mode.session.sessionData( 'vaclTrafficPolicySessionCounter',
                                                  None )
   assert counter is not None
   LazyMount.force( counter )
   globalSnapshot = counter

   # compare session and global snapshots, return newest
   return globalSnapshot if sessionSnapshot is None or \
      globalSnapshot.timestamp > sessionSnapshot.timestamp else sessionSnapshot

# NOTE: DynamicNameMatcher's low priority comes handy when we have
# both policyNameMatcher and summaryMatcher take a hit on keyword
# summary. For eg: show traffic-policy summary. At this point, higher
# priority summary keyword wins and we display summary for all the
# policies. In the event we do have a policy-name called _SUMMARY_ (
# aka summary ), then the user has to type out 'show traffic-policy
# summary summary' to view its summary in isolation or 'show
# traffic-policy summary detail'. The base command for 'show
# traffic-policy summary' for the policy called _SUMMARY_ will not
# take effect.
trafficPolicyMatcher = CliMatcher.KeywordMatcher(
   'traffic-policy', helpdesc='Show traffic-policy rules' )
summaryMatcher = CliMatcher.KeywordMatcher(
   'summary', helpdesc='Traffic policy summary' )
detailMatcher = CliMatcher.KeywordMatcher(
   'detail', helpdesc='Traffic policy detail' )
errorsMatcher = CliMatcher.KeywordMatcher(
   'errors', helpdesc='Traffic policy errors' )
countersMatcher = CliMatcher.KeywordMatcher(
   'counters', helpdesc='Traffic policy counters' )
fieldSetMatcher = CliMatcher.KeywordMatcher(
   'field-set', helpdesc='Traffic policy field set' )

policyNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: sorted( config.pmapType.pmap ),
   "Traffic policy name", pattern=protectedTrafficPolicyNamesRegex(),
   priority=CliParser.PRIO_LOW )
interfaceMatcher = CliCommand.guardedKeyword( 'interface',
                                              helpdesc="Policy applied on interface",
                                              guard=interfaceGuard )
cpuMatcher = CliCommand.guardedKeyword( 'cpu',
                                        helpdesc="Policy applied on VRF(s)",
                                        guard=cpuGuard )
vaclMatcher = CliCommand.guardedKeyword( 'vlan',
                                        helpdesc="Policy applied on VLAN",
                                        guard=vaclGuard )
vrfMatcher = CliCommand.guardedKeyword( 'vrf',
                                        helpdesc="Policy applied on VRF(s)",
                                        guard=aegisIntfOverrideGuard )

ingressMatcher = CliCommand.guardedKeyword( 'input',
      helpdesc='Policy applied on interface input',
      guard=ingressGuard )

egressMatcher = CliCommand.guardedKeyword( 'output',
      helpdesc='Policy applied on interface output',
      guard=egressGuard )

def showTrafficPolicyHandler( mode, args ):
   """Digest the args dict and pass it down to the helper to render"""
   policyName = args.get( 'POLICY_NAME', '' )
   cpu = 'cpu' in args
   interface = 'interface' in args
   vlan = 'vlan' in args
   summary = 'summary' in args
   detail = 'detail' in args
   errors = 'errors' in args
   counters = 'counters' in args
   excludedVrfIntfShow = 'vrf' in args

   inputPolicies = 'input' in args
   outputPolicies = 'output' in args
   t0( "showTrafficPolicyHandler args", args )

   # If no direction specified, we want to fetch both directions
   if not inputPolicies and not outputPolicies:
      inputPolicies = True
      outputPolicies = True

   # Disallow dumping of policies where direction is not supported
   if not interfaceTrafficPolicyHwStatus.ingressTrafficPolicySupported:
      inputPolicies = False

   if not interfaceTrafficPolicyHwStatus.egressTrafficPolicySupported:
      outputPolicies = False

   return showTrafficPolicy( mode, policyName, cpu, interface, vlan,
                             summary, errors, detail, counters,
                             inputPolicies, outputPolicies, excludedVrfIntfShow )
# show traffic-policy [ <name> ] ( cpu | interface | vlan )

# NOTE: There are 4 different classes when we could have had just
# one ShowTrafficPolicy. This is because the CLI model ( see cliModel attribute )
# returned for each of these variants is different.
# ------------------------------------------------------------------------
# ------------------------------------------------------------------------
class ShowTrafficPolicy( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-policy [ POLICY_NAME ]
               ( cpu | vlan | ( interface [ input | output ] ) ) [ detail ]'''
   data = {
      'traffic-policy': trafficPolicyMatcher,
      'POLICY_NAME' : policyNameMatcher,
      'cpu' : cpuMatcher,
      'interface' : interfaceMatcher,
      'input' : ingressMatcher,
      'output' : egressMatcher,
      'vlan' : vaclMatcher,
      'detail' : detailMatcher,
   }
   cliModel = TrafficPolicyModel
   handler = showTrafficPolicyHandler

BasicCli.addShowCommandClass( ShowTrafficPolicy )
#------------------------------------------------------------------------
# show traffic-policy [ <name> ] ( cpu | interface | vlan ) summary
#------------------------------------------------------------------------
class ShowTrafficPolicySummary( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show traffic-policy [ POLICY_NAME ] '
              '( cpu | vlan | ( interface [ input | output ] ) ) summary' )
   data = {
      'traffic-policy': trafficPolicyMatcher,
      'POLICY_NAME' : policyNameMatcher,
      'summary': summaryMatcher,
      'cpu' : cpuMatcher,
      'interface' : interfaceMatcher,
      'input' : ingressMatcher,
      'output' : egressMatcher,
      'vlan' : vaclMatcher,
   }
   cliModel = TrafficPolicySummaryModel
   handler = showTrafficPolicyHandler

BasicCli.addShowCommandClass( ShowTrafficPolicySummary )
#------------------------------------------------------------------------
# show traffic-policy [ <name> ] ( cpu | interface | vlan ) errors
#------------------------------------------------------------------------
class ShowTrafficPolicyErrors( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show traffic-policy [ POLICY_NAME ] '
              '( cpu | vlan | ( interface [ input | output ] ) ) errors' )
   data = {
      'traffic-policy': trafficPolicyMatcher,
      'POLICY_NAME' : policyNameMatcher,
      'errors': errorsMatcher,
      'cpu' : cpuMatcher,
      'interface' : interfaceMatcher,
      'input' : ingressMatcher,
      'output' : egressMatcher,
      'vlan' : vaclMatcher,
   }
   cliModel = TrafficPolicyErrorsModel
   handler = showTrafficPolicyHandler

BasicCli.addShowCommandClass( ShowTrafficPolicyErrors )

#------------------------------------------------------------------------
# show traffic-policy protocol neighbors bgp
#------------------------------------------------------------------------
class ProtocolNeighborBgp( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-policy protocol neighbors bgp'''
   data = {
      'traffic-policy': 'Show traffic-policy rules',
      'protocol': 'protocol',
      'neighbors': 'neighbors',
      'bgp': 'BGP',
   }
   cliModel = ProtocolNeighborsModel

   @staticmethod
   def handler( mode, args ):
      def getRules( structuredFilterColl ):
         rules = []
         for sf in six.itervalues( structuredFilterColl ):
            assert len( sf.proto ) <= 1

            if not sf.proto:
               continue
            protoField = list( sf.proto.values() )[ 0 ]
            assert len( protoField.port ) <= 1
            if protoField.port:
               portField = list( protoField.port.values() ).pop()
               assert len( portField.sport ) <= 1
               assert len( portField.dport ) <= 1
            for source in sf.source:
               rule = NeighborRule()
               rule.sourcePrefix = source
               if protoField.port:
                  if list( portField.sport ):
                     rule.sourcePort = list( portField.sport )[ 0 ].rangeStart
                  if list( portField.dport ):
                     rule.destinationPort = list( portField.dport )[ 0 ].rangeStart
               if sf.neighborProtocol == protocolBgpTtlSec:
                  if sf.ttl:
                     assert len( sf.ttl ) == 1
                     ttlRange = list( sf.ttl ).pop()
                     rule.maximumHops = 255 - ttlRange.rangeStart
               rules.append( rule )
         return rules

      def getVrfRules( trafficPolicyClassMapVrfStatus ):
         vrfRules = {}
         for vrfName, vrfStatus in six.iteritems( trafficPolicyClassMapVrfStatus ):
            rules = getRules( vrfStatus.structuredFilter )
            vrfRules[ vrfName ] = VrfRules( rules=rules )
         return vrfRules

      status = statusDir.get( 'cpu-vrf-hardware' )
      if not enforcementConfig.shouldEnforceInHardware(
            cpuHwStatus.cpuTrafficPolicyEnforceInHardwareDefault ):
         status = statusDir.get( 'cpu-vrf-software' )

      if status is None:
         t0( "Feature not present to report TrafficPolicy cpu status" )
         return ProtocolNeighborsModel()
      assert status is not None

      policies = {}
      for policyName, policyStatus in six.iteritems( status.status ):
         vrfRules = {}
         for classStatus in six.itervalues( status.trafficPolicyClassMapStatus ):
            className = classStatus.className
            if className not in policyStatus.cmap:
               continue
            classVrfRules = \
                  getVrfRules( classStatus.trafficPolicyClassMapVrfStatus )
            for vrf, rules in six.iteritems( classVrfRules ):
               if vrf in vrfRules:
                  vrfRules[ vrf ].rules.extend( rules.rules )
               else:
                  vrfRules[ vrf ] = rules

         assert policyName not in policies
         policies[ policyName ] = ProtocolNeighbor( vrfs=vrfRules )
      return ProtocolNeighborsModel( policies=policies )

BasicCli.addShowCommandClass( ProtocolNeighborBgp )

# ------------------------------------------------------------------------
# show traffic-policy [ <name> ] ( cpu | interface ) counters
# ------------------------------------------------------------------------
class ShowTrafficPolicyCounters( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-policy [ POLICY_NAME ]
               ( cpu | vlan | ( interface [ input | output ] ) ) counters'''
   data = {
      'traffic-policy' : trafficPolicyMatcher,
      'POLICY_NAME' : policyNameMatcher,
      'cpu' : cpuMatcher,
      'interface' : interfaceMatcher,
      'input' : ingressMatcher,
      'output' : egressMatcher,
      'counters' : countersMatcher,
      'vlan' : vaclMatcher,
   }
   cliModel = TrafficPolicyCountersModel
   handler = showTrafficPolicyHandler

BasicCli.addShowCommandClass( ShowTrafficPolicyCounters )

# ------------------------------------------------------------------------
# show traffic-policy [ <name> ] interface vrf detail
# ------------------------------------------------------------------------
class ShowTrafficPolicyVrfIntfOverride( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-policy [ POLICY_NAME ] interface vrf detail'''
   data = {
      'traffic-policy' : trafficPolicyMatcher,
      'POLICY_NAME' : policyNameMatcher,
      'interface' : interfaceMatcher,
      'vrf' : vrfMatcher,
      'detail' : detailMatcher,
   }
   cliModel = TrafficPolicyVrfIntfOverrideModel
   handler = showTrafficPolicyHandler

BasicCli.addShowCommandClass( ShowTrafficPolicyVrfIntfOverride )

def showTrafficPolicyFieldSet( mode, fieldSetType, fieldSetName, args,
                               fsetDir=None ):
   # statusInput will try to grab some status that has field-set to policy mapping.
   # Platform agents only fill in one status.
   statusInput = None
   for statusV in statusDir.values():
      # XXX: to preserve old behavior, assign statusInput even without mapping
      # (printing "used in policies:" even when field-set is not used)
      statusInput = statusV
      if statusV.fieldSetToPolicy:
         break

   if fieldSetType == 'prefix':
      header = "Traffic policy"
      allowExceptPrefixes = True
      ipv4 = 'ipv4' in args
      filterOutput = 'PREFIX' in args
      exact = 'exact' in args
      prefix = IpGenPrefix( '0.0.0.0/0' )
      if filterOutput:
         if args.get( 'PREFIX' ) is None:
            mode.addError( 'Host part of prefix must be zero' )
            raise AlreadyHandledError()
         prefix = IpGenPrefix( str( args.get( 'PREFIX' ) ) )
      helper = Tac.newInstance( 'TrafficPolicyCli::ShowTrafficPolicyPrefixFieldSet',
                                gv.fieldSetConfig, fieldSetName, ipv4,
                                filterOutput, prefix, exact, statusInput,
                                header, allowExceptPrefixes )
      model = TrafficPolicyPrefixFieldSetModel
   elif fieldSetType == 'l4-port':
      filterOutput = 'PORTS' in args
      ports = args.get( 'PORTS', PortRange( 0, 0 ) )
      helper = Tac.newInstance( 'TrafficPolicyCli::ShowTrafficPolicyL4PortFieldSet',
                                gv.fieldSetConfig, fieldSetName, filterOutput,
                                ports, statusInput )
      model = TrafficPolicyL4PortFieldSetModel
   elif fieldSetType == 'vlan':
      filterOutput = 'VLANS' in args
      vlans = args.get( 'VLANS', VlanRange( 0, 0 ) )
      helper = Tac.newInstance( 'TrafficPolicyCli::ShowTrafficPolicyVlanFieldSet',
                                gv.fieldSetConfig, fieldSetName, filterOutput,
                                vlans, statusInput )
      model = TrafficPolicyVlanFieldSetModel
   elif fieldSetType == 'service':
      protos = args.get( 'PROTOS', ProtocolRange( 0, 0 ) )
      helper = Tac.newInstance( 'TrafficPolicyCli::ShowTrafficPolicyServiceFieldSet',
                                gv.fieldSetConfig, fieldSetName,
                                protos, statusInput )
      model = TrafficPolicyServiceFieldSetModel
   elif fieldSetType == 'integer':
      filterOutput = 'INTEGERS' in args
      integers = args.get( 'INTEGERS', NumericalRange( 0, 0 ) )
      helper = Tac.newInstance( 'TrafficPolicyCli::ShowTrafficPolicyIntegerFieldSet',
                                gv.fieldSetConfig, fieldSetName, filterOutput,
                                integers, statusInput )
      model = TrafficPolicyIntegerFieldSetModel
   elif fieldSetType == 'mac':
      if fsetDir:
         helper = Tac.newInstance(
            'TrafficPolicyCli::ShowMacTrafficPolicyMacFieldSet',
            fieldSetName, fsetDir )
         model = MacTrafficPolicyMacFieldSetModel
      else:
         filterOutput = 'MACS' in args
         macAddrs = EthAddr( str( args.get( 'MACS', '0000.0000.0000' ) ) )
         helper = Tac.newInstance(
            'TrafficPolicyCli::ShowTrafficPolicyMacFieldSet',
            fieldSetName, gv.fieldSetConfig, filterOutput, macAddrs, statusInput )
         model = TrafficPolicyMacFieldSetModel
   else:
      assert False, "invalid command"

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   revision = mode.session_.requestedModelRevision()
   helper.render( fd, fmt, revision )

   return model

# ------------------------------------------------------------------------
# show traffic-policy field-set ( ipv4 | ipv6 ) prefix [ <name> ]
#   [ <prefix> [ exact ] ]
# ------------------------------------------------------------------------
def showTrafficPolicyPrefixFieldSetHandler( mode, args ):
   """Digest the args dict and pass it down to the helper to render"""
   fieldSetType = 'prefix'
   fieldSetName = args.get( 'FIELD_SET_NAME', '' )
   t0( "showTrafficPolicyPrefixFieldSetHandler args", args )
   return showTrafficPolicyFieldSet( mode, fieldSetType, fieldSetName, args )

class ShowTrafficPolicyIpPrefixFieldSetBase( ShowCommand.ShowCliCommandClass ):
   _baseData = {
      'traffic-policy' : trafficPolicyMatcher,
      'field-set' : fieldSetMatcher,
      'ipv4' : 'IPv4 field sets',
      'ipv6' : 'IPv6 field sets',
      'prefix' : 'Prefix field set',
      'exact' : 'exact match'
   }
   cliModel = TrafficPolicyPrefixFieldSetModel
   handler = showTrafficPolicyPrefixFieldSetHandler

def getIpPrefixFieldSetNames( mode ):
   return gv.fieldSetConfig.fieldSetIpPrefix

ipPrefixFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpPrefixFieldSetNames,
   'IP prefix field-set name',
   # Note: pattern may not be appropriate when SDK-defined field sets are supported
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class ShowTrafficPolicyIpv4PrefixFieldSet( ShowTrafficPolicyIpPrefixFieldSetBase ):
   syntax = 'show traffic-policy field-set ipv4 prefix [ FIELD_SET_NAME ] '\
            '[ PREFIX [ exact ] ]'
   data = {
      'FIELD_SET_NAME' : ipPrefixFieldSetNameMatcher,
      "PREFIX" : IpPrefixMatcher( 'IPv4 prefix', overlap=PREFIX_OVERLAP_REJECT ),
   }
   data.update( ShowTrafficPolicyIpPrefixFieldSetBase._baseData )

BasicCli.addShowCommandClass( ShowTrafficPolicyIpv4PrefixFieldSet )

def getIpv6PrefixFieldSetNames( mode ):
   return gv.fieldSetConfig.fieldSetIpv6Prefix

ipv6PrefixFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpv6PrefixFieldSetNames,
   'IPv6 prefix field-set name',
   # Note: pattern may not be appropriate when SDK-defined field sets are supported
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class ShowTrafficPolicyIpv6PrefixFieldSet( ShowTrafficPolicyIpPrefixFieldSetBase ):
   syntax = 'show traffic-policy field-set ipv6 prefix [ FIELD_SET_NAME ]'\
            '[ PREFIX [ exact ] ]'
   data = {
      'FIELD_SET_NAME' : ipv6PrefixFieldSetNameMatcher,
      "PREFIX" : Ip6PrefixMatcher( 'IPv6 prefix', overlap=PREFIX_OVERLAP_REJECT ),
   }
   data.update( ShowTrafficPolicyIpPrefixFieldSetBase._baseData )

BasicCli.addShowCommandClass( ShowTrafficPolicyIpv6PrefixFieldSet )

# ------------------------------------------------------------------------
# show traffic-policy field-set l4-port [ <name> ] [ PORTS ]
# ------------------------------------------------------------------------
def showTrafficPolicyL4PortFieldSetHandler( mode, args ):
   """Digest the args dict and pass it down to the helper to render"""
   fieldSetType = 'l4-port'
   fieldSetName = args.get( 'FIELD_SET_NAME', '' )
   t0( "showTrafficPolicyL4PortFieldSetHandler args", args )
   return showTrafficPolicyFieldSet( mode, fieldSetType, fieldSetName, args )

def getL4PortFieldSetNames( mode ):
   return gv.fieldSetConfig.fieldSetL4Port

l4PortFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getL4PortFieldSetNames,
   "Layer 4 port field set name",
   pattern=protectedFieldSetNamesRegex( 'port' ),
   priority=CliParser.PRIO_LOW )

class ShowTrafficPolicyL4PortFieldSet( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-policy field-set l4-port [ FIELD_SET_NAME ] [ PORTS ]'
   data = {
      'traffic-policy' : trafficPolicyMatcher,
      'field-set' : fieldSetMatcher,
      'l4-port' : 'Layer 4 port',
      'FIELD_SET_NAME' : l4PortFieldSetNameMatcher,
      'PORTS' : MultiRangeMatcher( rangeFn=lambda : ( 0, appConstants.maxL4Port ),
                                   noSingletons=False,
                                   maxRanges=1,
                                   helpdesc='Port value or range of port values',
                                   value=lambda mode, grList :
                                   PortRange( grList.ranges()[ 0 ][ 0 ],
                                              grList.ranges()[ 0 ][ 1 ] ) ),
   }
   cliModel = TrafficPolicyL4PortFieldSetModel
   handler = showTrafficPolicyL4PortFieldSetHandler

BasicCli.addShowCommandClass( ShowTrafficPolicyL4PortFieldSet )

# ------------------------------------------------------------------------
# show traffic-policy field-set vlan [ <name> ] [ VLANS ]
# ------------------------------------------------------------------------
def showTrafficPolicyVlanFieldSetHandler( mode, args ):
   """Digest the args dict and pass it down to the helper to render"""
   fieldSetType = 'vlan'
   fieldSetName = args.get( 'FIELD_SET_NAME', '' )
   t0( "showTrafficPolicyVlanFieldSetHandler args", args )
   return showTrafficPolicyFieldSet( mode, fieldSetType, fieldSetName, args )

def getVlanFieldSetNames( mode ):
   return gv.fieldSetConfig.fieldSetVlan

vlanFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getVlanFieldSetNames,
   "VLAN field set name",
   pattern=protectedFieldSetNamesRegex( 'vlan' ),
   priority=CliParser.PRIO_LOW )

class ShowTrafficPolicyVlanFieldSet( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-policy field-set vlan [ FIELD_SET_NAME ] [ VLANS ]'
   data = {
      'traffic-policy' : trafficPolicyMatcher,
      'field-set' : fieldSetMatcher,
      'vlan' : 'VLAN field set',
      'FIELD_SET_NAME' : vlanFieldSetNameMatcher,
      'VLANS' : MultiRangeMatcher( rangeFn=lambda : ( 1, appConstants.maxVlan - 1 ),
                                   noSingletons=False,
                                   maxRanges=1,
                                   helpdesc='VLAN value or range of VLAN values',
                                   value=lambda mode, grList :
                                   VlanRange( grList.ranges()[ 0 ][ 0 ],
                                              grList.ranges()[ 0 ][ 1 ] ) ),
   }
   cliModel = TrafficPolicyVlanFieldSetModel
   handler = showTrafficPolicyVlanFieldSetHandler

BasicCli.addShowCommandClass( ShowTrafficPolicyVlanFieldSet )

# ------------------------------------------------------------------------
# show traffic-policy field-set SERVICE [ <name> ] [ PROTOS ]
# ------------------------------------------------------------------------
def showTrafficPolicyServiceFieldSetHandler( mode, args ):
   """Digest the args dict and pass it down to the helper to render"""
   fieldSetType = 'service'
   fieldSetName = args.get( 'FIELD_SET_NAME', '' )
   t0( "showTrafficPolicyServiceFieldSetHandler args", args )
   return showTrafficPolicyFieldSet( mode, fieldSetType, fieldSetName, args )

def getServiceFieldSetNames( mode ):
   return gv.fieldSetConfig.fieldSetService

serviceFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getVlanFieldSetNames,
   "service field-set name",
   pattern=protectedFieldSetNamesRegex( 'service' ),
   priority=CliParser.PRIO_LOW )

class ShowTrafficPolicyServiceFieldSet( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-policy field-set service [ FIELD_SET_NAME ] [ PROTOS ]'
   data = {
      'traffic-policy' : trafficPolicyMatcher,
      'field-set' : fieldSetMatcher,
      'service' : 'Networking Service',
      'FIELD_SET_NAME' : serviceFieldSetNameMatcher,
      'PROTOS' : MultiRangeMatcher( rangeFn=lambda : ( 1,
                                                   appConstants.maxProto - 1 ),
                                   noSingletons=False,
                                   maxRanges=1,
                                   helpdesc='SERVICE Proto range',
                                   value=lambda mode, grList :
                                   ProtocolRange( grList.ranges()[ 0 ][ 0 ],
                                              grList.ranges()[ 0 ][ 1 ] ) )
   }
   cliModel = TrafficPolicyServiceFieldSetModel
   handler = showTrafficPolicyServiceFieldSetHandler

BasicCli.addShowCommandClass( ShowTrafficPolicyServiceFieldSet )

# ------------------------------------------------------------------------
# show traffic-policy field-set integer [ <name> ] [ INTEGERS ]
# ------------------------------------------------------------------------
def showTrafficPolicyIntegerFieldSetHandler( mode, args ):
   """Digest the args dict and pass it down to the helper to render"""
   fieldSetType = 'integer'
   fieldSetName = args.get( 'FIELD_SET_NAME', '' )
   t0( "showTrafficPolicyIntegerFieldSetHandler args", args )
   return showTrafficPolicyFieldSet( mode, fieldSetType, fieldSetName, args )

def getIntegerFieldSetNames( mode ):
   return gv.fieldSetConfig.fieldSetInteger

integerFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIntegerFieldSetNames,
   "Integer field set name",
   pattern=protectedFieldSetNamesRegex( 'integer' ),
   priority=CliParser.PRIO_LOW )

class ShowTrafficPolicyIntegerFieldSet( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-policy field-set integer [ FIELD_SET_NAME ] [ INTEGERS ]'
   data = {
      'traffic-policy' : trafficPolicyMatcher,
      'field-set' : fieldSetMatcher,
      'integer' : 'Integer field set',
      'FIELD_SET_NAME' : integerFieldSetNameMatcher,
      'INTEGERS' : MultiRangeMatcher(
         rangeFn=lambda : ( 0, appConstants.maxInteger ), noSingletons=False,
         maxRanges=1, helpdesc='Integer value or range of integer values',
         value=lambda mode, grList : NumericalRange( grList.ranges()[ 0 ][ 0 ],
                                                     grList.ranges()[ 0 ][ 1 ] ) ),
   }
   cliModel = TrafficPolicyIntegerFieldSetModel
   handler = showTrafficPolicyIntegerFieldSetHandler

BasicCli.addShowCommandClass( ShowTrafficPolicyIntegerFieldSet )

# ------------------------------------------------------------------------
# show traffic-policy field-set mac [ <name> ] [ MACS ]
# ------------------------------------------------------------------------
def showTrafficPolicyMacAddrFieldSetHandler( mode, args ):
   """Digest the args dict and pass it down to the helper to render"""
   fieldSetType = 'mac'
   fieldSetName = args.get( 'FIELD_SET_NAME', '' )
   t0( "showTrafficPolicyMacAddrFieldSetHandler args", args )
   return showTrafficPolicyFieldSet( mode, fieldSetType, fieldSetName, args )

def getMacAddrFieldSetNames( mode ):
   return gv.fieldSetConfig.fieldSetMacAddr

macAddrFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getMacAddrFieldSetNames,
   "MAC address field set name",
   pattern=protectedFieldSetNamesRegex( 'mac' ),
   priority=CliParser.PRIO_LOW )

class ShowTrafficPolicyMacAddrFieldSet( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-policy field-set mac [ FIELD_SET_NAME ] [ MACS ]'
   data = {
      'traffic-policy' : trafficPolicyMatcher,
      'field-set' : fieldSetMatcher,
      'mac' : 'MAC address field set',
      'FIELD_SET_NAME' : macAddrFieldSetNameMatcher,
      'MACS' : MacAddrMatcher()
   }
   cliModel = TrafficPolicyMacFieldSetModel
   handler = showTrafficPolicyMacAddrFieldSetHandler

BasicCli.addShowCommandClass( ShowTrafficPolicyMacAddrFieldSet )

@synchronized( mergeSmLock )
def startFieldSetConfigMergeSm():
   mergedFieldSetConfig = Tac.newInstance( 'Classification::FieldSetConfig',
                                           'mergedConfig' )
   gv.fieldSetMergeSm = Tac.newInstance( 'Classification::FieldSetConfigMergeSm',
                                         gv.fieldSetConfigDir, mergedFieldSetConfig )
   gv.fieldSetConfig = mergedFieldSetConfig

@synchronized( mergeSmLock )
def startIntfConfigMergeSm():
   if intfPolicyGV.configMergeSm is not None:
      # BUG957056: Avoid creating the SM multiple times as CliGlobal attributes can
      # only be written once.
      return

   LazyMount.force( intfPolicyGV.configDir )
   LazyMount.force( intfPolicyGV.configCli )
   LazyMount.force( intfPolicyGV.ingressIntfConfigDir )
   LazyMount.force( intfPolicyGV.ingressIntfConfigCli )
   LazyMount.force( intfPolicyGV.vtiStatusDir )
   LazyMount.force( intfPolicyGV.dynVlanIntfDir )
   mergedPolicyConfig = Tac.newInstance( 'TrafficPolicy::TrafficPolicyConfig',
                                         'MergedPolicyConfig' )
   mergedIngressIntfConfig = Tac.newInstance( 'PolicyMap::IntfConfig',
                                              'MergedIngressIntfConfig' )
   mergedIngressIntfConfig.type = 'mapTrafficPolicy'

   intfPolicyGV.configMergeSm = Tac.newInstance( 'PolicyMap::ConfigMergeSm',
                                         intfPolicyGV.entityManager.cEntityManager(),
                                         intfPolicyGV.configDir,
                                         intfPolicyGV.ingressIntfConfigDir,
                                         intfPolicyGV.ingressIntfConfigCli,
                                         intfPolicyGV.configCli,
                                         intfPolicyGV.vtiStatusDir,
                                         intfPolicyGV.dynVlanIntfDir,
                                         mergedIngressIntfConfig,
                                         mergedPolicyConfig )
   intfPolicyGV.configMergeSm.initialized = True

   intfPolicyGV.ingressIntfConfig = mergedIngressIntfConfig
   intfPolicyGV.config = mergedPolicyConfig

def Plugin( em ):
   global config
   global entityManager
   global vrfConfig
   global cpuIntfConfig # intf-based CPU TP
   global enforcementConfig
   global statusDir
   global l3IntfStatus
   global cpuCounter
   global intfInputCounter
   global intfOutputCounter
   global vaclCounter
   global aegisOutputIntfConfig
   global aegisBridgedIntfConfig
   global intfParamStatus
   global interfaceTrafficPolicyHwStatus
   global counterStatusInput
   global counterStatusOutput
   global hwStatus
   global cpuHwStatus
   global tapAggCliConfig
   global switchIntfConfigDir


   entityManager = em
   intfPolicyGV.entityManager = entityManager
   # Cell specific mount path which holds status entities per linecard
   # in a Tac::Dir. At the moment, we can simply take one instance
   # under the Tac::Dir which belongs to Sand platform. On other
   # platforms ( like Strata ), we need a unified view of the status
   # reporting model or per slice view ( TBD ).
   statusDirPath = 'cell/%d/trafficPolicies/status' % Cell.cellId()
   mountGroup = entityManager.mountGroup()
   statusDir = mountGroup.mount( statusDirPath, 'Tac::Dir', 'ri' )
   callback = None
   gv.fieldSetConfigDir = mountGroup.mount( "trafficPolicies/fieldset/input/",
                                            "Tac::Dir", "ri" )
   callback = startFieldSetConfigMergeSm
   mountGroup.close( callback=callback )

   # ConfigMergeSm logic
   # SDK
   intfPolicyGV.configDir = LazyMount.mount( entityManager,
                                             "trafficPolicies/input/config",
                                             "Tac::Dir", "ri" )
   # CLI
   config = LazyMount.mount( entityManager,
                             "trafficPolicies/input/cli",
                             "TrafficPolicy::TrafficPolicyConfig",
                             "w" )
   intfPolicyGV.configCli = LazyMount.mount( entityManager,
                                             "trafficPolicies/input/cli",
                                             "TrafficPolicy::TrafficPolicyConfig",
                                             "w" )
   # SDK
   intfPolicyGV.ingressIntfConfigDir = LazyMount.mount( entityManager,
         "trafficPolicies/intf/input/config", "Tac::Dir", "ri" )
   # CLI
   intfPolicyGV.ingressIntfConfigCli = LazyMount.mount( entityManager,
                                           "trafficPolicies/intf/input/aegis",
                                           "PolicyMap::IntfConfig", "r" )

   gv.appRecConfig = LazyMount.mount( entityManager,
                                          'classification/app-recognition/config',
                                          'Classification::AppRecognitionConfig',
                                          'r' )

   intfPolicyGV.vtiStatusDir = LazyMount.mount( entityManager,
                                                'interface/status/eth/vxlan',
                                                'Vxlan::VtiStatusDir', 'r' )

   intfPolicyGV.dynVlanIntfDir = LazyMount.mount( entityManager,
                                                  'interface/input/dynvlanintf',
                                                  'Tac::Dir', 'ri' )
   vrfConfig = LazyMount.mount( entityManager, "trafficPolicies/cpu/vrf",
                                "PolicyMap::VrfConfig", "r" )
   cpuIntfConfig = LazyMount.mount( entityManager,
                                    "trafficPolicies/cpu/intf",
                                    "PolicyMap::IntfConfig", "r" )
   intfPolicyGV.ingressVrfConfig = LazyMount.mount( entityManager,
         "trafficPolicies/vrf/input/cli", "TrafficPolicy::TrafficPolicyVrfConfig",
         "r" )
   l3IntfStatus = LazyMount.mount( entityManager, "l3/intf/status",
                                   "L3::Intf::StatusDir", "r" )
   gv.l3IntfConfig = LazyMount.mount( entityManager, "l3/intf/config",
                                      "L3::Intf::ConfigDir", "r" )
   gv.routingHwRouteStatus = LazyMount.mount( entityManager,
                                              "routing/hardware/route/status",
                                              "Routing::Hardware::RouteStatus",
                                              "r" )
   aegisOutputIntfConfig = LazyMount.mount( entityManager,
                                            "trafficPolicies/intf/output/aegis",
                                            "PolicyMap::IntfConfig", "r" )
   aegisBridgedIntfConfig = LazyMount.mount( entityManager,
                                           "trafficPolicies/intf/bridged/aegis",
                                           "PolicyMap::IntfConfig", "r" )
   intfParamStatus = LazyMount.mount( entityManager,
                                      "trafficPolicies/param/status/interface",
                                      "TrafficPolicy::TrafficPolicyIntfParamStatus",
                                      "r" )
   cpuCounter = LazyMount.mount( entityManager, "trafficPolicies/counter/cpu",
                                 "TrafficPolicy::Counter", "r" )
   interfaceTrafficPolicyHwStatus = (
         LazyMount.mount( entityManager,
                          "trafficPolicies/hardware/status/interface",
                          "TrafficPolicy::HwStatus", "r" )
   )
   intfInputCounter = (
         LazyMount.mount( entityManager,
                          "trafficPolicies/counter/interface/input",
                          "TrafficPolicy::Counter", "r" )
   )
   intfOutputCounter = (
         LazyMount.mount( entityManager,
                          "trafficPolicies/counter/interface/output",
                          "TrafficPolicy::Counter", "r" )
   )
   vaclCounter = (
         LazyMount.mount( entityManager,
                          "trafficPolicies/counter/interface/bridged",
                          "TrafficPolicy::Counter", "r" )
   )
   hwStatus = LazyMount.mount( entityManager,
                               'trafficPolicies/hardware/status/interface',
                               'TrafficPolicy::HwStatus', 'r' )
   cpuHwStatus = LazyMount.mount( entityManager,
                               'trafficPolicies/hardware/status/cpu',
                               'TrafficPolicy::HwStatus', 'r' )
   tapAggCliConfig = LazyMount.mount( entityManager,
                                      'tapagg/cliconfig',
                                      'TapAgg::CliConfig', 'r' )
   switchIntfConfigDir = LazyMount.mount( entityManager,
                                          'bridging/switchIntfConfig',
                                          'Bridging::SwitchIntfConfigDir', 'r' )
   enforcementConfig = LazyMount.mount( entityManager,
                                'trafficPolicies/cpu/enforcementConfig',
                                'TrafficPolicy::EnforcementConfig',
                                'r' )
   autoUnmount = True
   sharkMountInfo = SharkLazyMount.mountInfo( 'shadow' )
   counterStatusInput = SharkLazyMount.mount(
      entityManager,
      "policyMap/counters/trafficPolicyInterface/input",
      "PolicyMap::Counters::PolicyMapTypeCounters",
      sharkMountInfo,
      autoUnmount )
   counterStatusOutput = SharkLazyMount.mount(
      entityManager,
      "policyMap/counters/trafficPolicyInterface/output",
      "PolicyMap::Counters::PolicyMapTypeCounters",
      sharkMountInfo,
      autoUnmount )
