#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import CliGlobal
import CliCommand
import CliMatcher
import CliParser
from CliPlugin import VxlanCli
import LazyMount
import Tac
from Toggles.TrafficPolicyToggleLib import (
   toggleTrafficPoliciesOnVniEnabled,
)
VxlanTunnelIntfType = Tac.Type( "Arnet::VxlanTunnelIntfId" ).vxlanTunnelIntfIdType
TrafficPolicyHwIntfIdPair = Tac.Type( "TrafficPolicy::HwStatus::IntfIdTypeIdPair" )

gv = CliGlobal.CliGlobal(
   vxlanIntf=TrafficPolicyHwIntfIdPair( VxlanTunnelIntfType, 0 ),
   trafficPolicyHwStatus=None,
   trafficPolicyCliConfig=None,
)

# -----------------------------------------------------------------------------------
# vxlan vni <VNI> traffic-policy input <PMAP_NAME>
# -----------------------------------------------------------------------------------
trafficPolicyNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: gv.trafficPolicyCliConfig.pmapType.pmap,
   'TrafficPolicy Policy Map Name' )

def guardTrafficPolicy( mode, token ):
   if gv.vxlanIntf in \
         gv.trafficPolicyHwStatus.ingressTrafficPolicySupportedForIntfType and \
         mode.intf.name == 'Vxlan1':
      return None
   return CliParser.guardNotThisPlatform

class TrafficPolicyOnVniCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vni VNI traffic-policy input PMAP_NAME'
   noOrDefaultSyntax = syntax.replace( 'PMAP_NAME', '[ PMAP_NAME ]' )
   data = {
      'vxlan' : VxlanCli.vxlanNode,
      'vni' : VxlanCli.vniMatcherForConfig,
      'VNI' : VxlanCli.vniMatcher,
      'traffic-policy' : CliCommand.guardedKeyword( 'traffic-policy',
         helpdesc='Traffic policy configuration on VNI',
         guard=guardTrafficPolicy ),
      'input' : 'Apply the traffic policy to overlay ingress packets',
      'PMAP_NAME' : trafficPolicyNameMatcher,
   }
   handler = "TrafficPolicyCliHandler.setTrafficPolicyOnVni"
   noOrDefaultHandler = "TrafficPolicyCliHandler.noTrafficPolicyOnVni"

# Note 'traffic-policy' is guarded by TrafficPolicy::HwStatus which is only set
# if the DFW or PolicyBasedVpn capability is set. The capabilities are set under
# toggle.  As requested adding a new toggle here in case DFW should ship without
# TrafficPoliciesOnVni supported
if toggleTrafficPoliciesOnVniEnabled():
   VxlanCli.VxlanIntfModelet.addCommandClass( TrafficPolicyOnVniCmd )

def Plugin( entityManager ):
   gv.trafficPolicyHwStatus = (
         LazyMount.mount( entityManager,
                          "trafficPolicies/hardware/status/interface",
                          "TrafficPolicy::HwStatus", "r" )
   )
   gv.trafficPolicyCliConfig = (
         LazyMount.mount( entityManager,
                             "trafficPolicies/input/cli",
                             "TrafficPolicy::TrafficPolicyConfig",
                             "r" )
   )
