#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericPrefix
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from TableOutput import createTable, Format

class TunnelCountersEntry( Model ):
   tunnelIndex = Int( help="Index of the tunnel entry in the source tunnel table" )
   endpoint = IpGenericPrefix( help="Endpoint of the tunnel", optional=True )
   txPackets = Int( help="Total number of transmitted packets", optional=True )
   txBytes = Int( help="Total number of transmitted bytes", optional=True )
   tunnelName = Str( help="Name for the tunnel entry", optional=True )

   def renderTunnelCountersTableRow( self, table ):
      tunnelEndpointStr = '-'
      txPacketsStr = self.txPackets if self.txPackets is not None else '-'
      txBytesStr = self.txBytes if self.txBytes is not None else '-'
      tunnelNameStr = self.tunnelName if self.tunnelName is not None else ''

      if self.endpoint:
         tunnelEndpointStr = str( self.endpoint )

      table.newRow( self.tunnelIndex, tunnelEndpointStr,
                    txPacketsStr, txBytesStr, tunnelNameStr )

class TunnelCounters( Model ):
   entries = List( valueType=TunnelCountersEntry,
                   help="tunnel hardware counters" )

   def render( self ):
      headings = ( "Index", "Endpoint", "Packets", "Bytes",
                   "Tunnel Name" )
      fl = Format( justify='left' )
      fr = Format( justify='right' )
      table = createTable( headings, tableWidth=79 )
      table.formatColumns( fr, fl, fr, fr, fl )

      for tunnelCountersEntry in self.entries:
         tunnelCountersEntry.renderTunnelCountersTableRow( table )

      print( table.output() )
