#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.DecapGroupCli as DecapGroupCli
from CliPlugin.IntfCli import Intf
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.IpAddrMatcher import IpPrefixMatcher, PREFIX_OVERLAP_AUTOZERO
from CliPlugin.Ip6AddrMatcher import Ip6PrefixMatcher
from CliPlugin.DecapGroupCli import (
      DecapGroupMode,
      tunnelTypeGuard,
      decapPayloadTypeGuard,
)
from CliPlugin.TunnelGlobalConfigMode import (
      matcherTunnel,
      matcherUdp,
      matcherDestination,
)
from CliPlugin.VrfCli import VrfExprFactory
from Intf.IntfRange import intfRangeMatcher

matcherDecapInterface = CliCommand.guardedKeyword( 'decap-interface',
      helpdesc='Configure the decap interface',
      guard=DecapGroupCli.decapGroupIntfGuard )

matcherIntfAll = CliMatcher.KeywordMatcher( 'all',
      helpdesc='Use all interfaces for decap group (except ones in exclude option)' )

matcherAddrFamily = CliMatcher.KeywordMatcher( 'address-family',
      helpdesc='Configure address family' )

matcherIpv4 = CliMatcher.KeywordMatcher( 'ipv4',
      helpdesc='Configure IPv4 address family' )

matcherIpv6 = CliMatcher.KeywordMatcher( 'ipv6',
      helpdesc='Configure IPv6 address family' )

matcherAddress = CliMatcher.KeywordMatcher( 'address',
      helpdesc='Configure address on decap interface' )

matcherAll = CliMatcher.KeywordMatcher( 'all',
      helpdesc='Configure all IP addresses' )

matcherPrimary = CliMatcher.KeywordMatcher( 'primary',
      helpdesc='Configure primary IP address' )

matcherExclude = CliMatcher.KeywordMatcher( 'exclude',
      helpdesc='Exclude specific interfaces from decap group' )

matcherTunnelType = CliCommand.Node(
      matcher=CliMatcher.EnumMatcher( {
         'ipip' : 'Set the tunnel type as IpIp',
         'gre' : 'Set the tunnel type as GRE',
         'udp' : 'Set the tunnel type as UDP',
      } ), guard=tunnelTypeGuard )

matcherGreType = CliCommand.guardedKeyword( 'gre',
      helpdesc='Configure tunnel type GRE parameters',
      guard=DecapGroupCli.greGuard )

matcherGreKeyToken = CliCommand.guardedKeyword( 'key',
      helpdesc='Configure GRE key parameters',
      guard=DecapGroupCli.greKeyToForwardingVrfMappingGuard )

matcherOverlayMpls = CliCommand.guardedKeyword( 'mpls',
      helpdesc='MPLS specific configuration',
      guard=DecapGroupCli.qosTcToMplsTcDecapGuard )

matcherGreKey = CliMatcher.IntegerMatcher( 1, 4294967294,
      helpdesc='GRE key' )

matcherPayloadType = CliCommand.Node(
      matcher=CliMatcher.EnumMatcher( {
         'mpls' : 'Set the payload type as mpls',
      } ), guard=decapPayloadTypeGuard )

matcherForwarding = CliCommand.guardedKeyword( 'forwarding',
      helpdesc='Configure forwarding settings for decap group',
      guard=DecapGroupCli.forwardingVrfSupportedGuard )

matcherGreDecapForwarding = CliMatcher.KeywordMatcher( 'forwarding',
      helpdesc='Configure forwarding settings for GRE decap group' )

def tunnelDecapIntfHandler( mode, args ):
   intfOption = args.get( 'INTF', 'all' )
   addressOption = args.get( 'primary' ) or args.get( 'all' )
   excludeIntfOption = args.get( 'INTFS', () )

   if 'ipv6' in args:
      mode.handleTunnelDecapIntfV6( intfOption=intfOption,
                                    excludeIntfOption=excludeIntfOption )
   else:
      mode.handleTunnelDecapIntf( intfOption=intfOption,
                                  addressOption=addressOption,
                                  excludeIntfOption=excludeIntfOption )

def delTunnelDecapIntfHandler( mode, args ):
   intfOption = args.get( 'INTF', 'all' )
   if 'ipv6' in args:
      mode.delTunnelDecapIntfV6( intfOption=intfOption )
   else:
      mode.delTunnelDecapIntf( intfOption=intfOption )

def tunnelDecapIpHandler( mode, args ):
   decapIps = args.get( 'DECAP_IP' ) or \
              args.get( 'DECAP_PREFIX' ) or \
              args.get( 'DECAP6_PREFIX' )

   if DecapGroupCli.maxIterForDecapIp() == 1 and len( decapIps ) > 1:
      mode.addError( 'Multiple decap IPs not supported' )
      return

   mode.handleTunnelDecapIp( decapIps=decapIps )

def delTunnelDecapIpHandler( mode, args ):
   decapIps = args.get( 'DECAP_IP' ) or \
              args.get( 'DECAP_PREFIX' ) or \
              args.get( 'DECAP6_PREFIX' )

   mode.delTunnelDecapIp( decapIps=decapIps )

def tunnelTypeHandler( mode, args ):
   tunnelType = args.get( 'TUNNEL_TYPE' )
   mode.handleTunnelType( tunnelType )

def tunnelDestinationPortHandler( mode, args ):
   payloadType = args.get( 'PAYLOAD_TYPE' )
   mode.handleTunnelDestinationPort( payloadType=payloadType )

def delTunnelDestinationPortHandler( mode, args ):
   mode.delTunnelDestinationPort()

def forwardingVrfHandler( mode, args ):
   vrfName = args.get( 'VRF_NAME' )
   mode.handleForwardingVrf( vrfName )

def delForwardingVrfHandler( mode, args ):
   mode.delForwardingVrf()

def greKeyToForwardingVrfMappingHandler( mode, args ):
   greKey = args.get( 'GRE_KEY' )
   vrfName = args.get( 'VRF_NAME' )
   mode.handleGreKeyToForwardingVrfMapping( greKey, vrfName )

def delGreKeyToForwardingVrfMappingHandler( mode, args ):
   greKey = args.get( 'GRE_KEY' )
   vrfName = args.get( 'VRF_NAME' )
   mode.delGreKeyToForwardingVrfMapping( greKey, vrfName )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-interface all [ exclude { INTFS } ]
#--------------------------------------------------------------------------------
class TunnelDecapAllIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-interface all-intf [ exclude { INTFS } ]'
   noOrDefaultSyntax = 'tunnel decap-interface all-intf'
   
   data = {
      'tunnel' : matcherTunnel,
      'decap-interface' : matcherDecapInterface,
      'all-intf' : matcherIntfAll,
      'exclude' : matcherExclude,
      'INTFS' : intfRangeMatcher,
   }

   handler = tunnelDecapIntfHandler
   noOrDefaultHandler = delTunnelDecapIntfHandler

DecapGroupMode.addCommandClass( TunnelDecapAllIntfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-interface INTF
#--------------------------------------------------------------------------------
class TunnelDecapIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-interface INTF'
   noOrDefaultSyntax = syntax
   
   data = {
      'tunnel' : matcherTunnel,
      'decap-interface' : matcherDecapInterface,
      'INTF' : Intf.matcher,
   }

   handler = tunnelDecapIntfHandler
   noOrDefaultHandler = delTunnelDecapIntfHandler

DecapGroupMode.addCommandClass( TunnelDecapIntfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-interface all
# address-family ipv4 address ( primary | all ) [ exclude { INTFS } ]
#--------------------------------------------------------------------------------
class TunnelDecapAllIntfIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-interface all-intf ' \
            'address-family ipv4 address ( primary | all ) [ exclude { INTFS } ]'
   noOrDefaultSyntax = 'tunnel decap-interface all-intf address-family ipv4 ...'
   
   data = {
      'tunnel' : matcherTunnel,
      'decap-interface' : matcherDecapInterface,
      'all-intf' : matcherIntfAll,
      'address-family' : matcherAddrFamily,
      'ipv4' : matcherIpv4,
      'address' : matcherAddress,
      'primary' : matcherPrimary,
      'all' : matcherAll,
      'exclude' : matcherExclude,
      'INTFS' : intfRangeMatcher,
   }

   handler = tunnelDecapIntfHandler
   noOrDefaultHandler = delTunnelDecapIntfHandler

DecapGroupMode.addCommandClass( TunnelDecapAllIntfIpv4Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-interface INTF
# address-family ipv4 address ( primary | all )
#--------------------------------------------------------------------------------
class TunnelDecapIntfIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-interface INTF ' \
            'address-family ipv4 address ( primary | all )'
   noOrDefaultSyntax = 'tunnel decap-interface INTF address-family ipv4 ...'
   
   data = {
      'tunnel' : matcherTunnel,
      'decap-interface' : matcherDecapInterface,
      'INTF' : Intf.matcher,
      'address-family' : matcherAddrFamily,
      'ipv4' : matcherIpv4,
      'address' : matcherAddress,
      'primary' : matcherPrimary,
      'all' : matcherAll,
   }

   handler = tunnelDecapIntfHandler
   noOrDefaultHandler = delTunnelDecapIntfHandler

DecapGroupMode.addCommandClass( TunnelDecapIntfIpv4Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-interface all address-family ipv6 address all 
# [ exclude { INTFS } ]
#--------------------------------------------------------------------------------
class TunnelDecapAllIntfIpv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-interface all-intf address-family ipv6 address all ' \
            '[ exclude { INTFS } ]'
   noOrDefaultSyntax = 'tunnel decap-interface all-intf address-family ipv6 ...'
   
   data = {
      'tunnel' : matcherTunnel,
      'decap-interface' : matcherDecapInterface,
      'all-intf' : matcherIntfAll,
      'address-family' : matcherAddrFamily,
      'ipv6' : matcherIpv6,
      'address' : matcherAddress,
      'all' : matcherAll,
      'exclude' : matcherExclude,
      'INTFS' : intfRangeMatcher,
   }

   handler = tunnelDecapIntfHandler
   noOrDefaultHandler = delTunnelDecapIntfHandler

DecapGroupMode.addCommandClass( TunnelDecapAllIntfIpv6Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-interface INTF address-family ipv6 address all
#--------------------------------------------------------------------------------
class TunnelDecapIntfIpv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-interface INTF address-family ipv6 address all'
   noOrDefaultSyntax = 'tunnel decap-interface INTF address-family ipv6 ...'
   
   data = {
      'tunnel' : matcherTunnel,
      'decap-interface' : matcherDecapInterface,
      'INTF' : Intf.matcher,
      'address-family' : matcherAddrFamily,
      'ipv6' : matcherIpv6,
      'address' : matcherAddress,
      'all' : matcherAll,
   }

   handler = tunnelDecapIntfHandler
   noOrDefaultHandler = delTunnelDecapIntfHandler

DecapGroupMode.addCommandClass( TunnelDecapIntfIpv6Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-ip { DECAP_IP }
#--------------------------------------------------------------------------------
class TunnelDecapIpCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-ip { { DECAP_IP } | { DECAP_PREFIX } | { DECAP6_PREFIX } }'
   noOrDefaultSyntax = syntax

   data = {
      'tunnel' : matcherTunnel,
      'decap-ip' : CliCommand.guardedKeyword( 'decap-ip',
                     helpdesc='Configure the decap IP address',
                     guard=DecapGroupCli.decapGroupIpGuard ),
      'DECAP_IP' : IpGenAddrMatcher( 'Decap IP address' ),
      'DECAP_PREFIX' : CliCommand.Node(
                       matcher=IpPrefixMatcher( 'Decap IP prefix',
                                                overlap=PREFIX_OVERLAP_AUTOZERO,
                                                prefixLenRange=( 1, 32 ) ),
                       guard=DecapGroupCli.decapPrefixInTcamSupportedGuard ),
      'DECAP6_PREFIX' : CliCommand.Node(
                       matcher=Ip6PrefixMatcher( 'Decap IPv6 prefix',
                                                overlap=PREFIX_OVERLAP_AUTOZERO,
                                                prefixLenRange=( 1, 128 ) ),
                       guard=DecapGroupCli.decapIp6SupportedGuard )
   }

   handler = tunnelDecapIpHandler
   noOrDefaultHandler = delTunnelDecapIpHandler

DecapGroupMode.addCommandClass( TunnelDecapIpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel overlay mpls qos map mpls-traffic-class to traffic-class
#--------------------------------------------------------------------------------
class TunnelGreOrUdpQosTcFromMplsTcCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel overlay mpls qos map mpls-traffic-class to traffic-class'
   noOrDefaultSyntax = syntax

   data = {
      'tunnel' : matcherTunnel,
      'overlay': 'Tunnel overlay',
      'mpls' : matcherOverlayMpls,
      'qos' : 'QoS configuration',
      'map' : 'Mapping configuration of different QoS parameters',
      'mpls-traffic-class' : 'MPLS traffic-class value',
      'to' : 'Configure QoS mapping',
      'traffic-class' : 'Set traffic-class value'
   }
   handler = DecapGroupMode.handleTunnelGreOrUdpQosTcFromMplsTc
   noOrDefaultHandler = DecapGroupMode.noTunnelGreOrUdpQosTcFromMplsTc

DecapGroupMode.addCommandClass( TunnelGreOrUdpQosTcFromMplsTcCmd )

#--------------------------------------------------------------------------------
# tunnel type TUNNEL_TYPE
#--------------------------------------------------------------------------------
class TunnelDecapTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel type TUNNEL_TYPE'

   data = {
      'tunnel' : matcherTunnel,
      'type' : 'Configure tunnel type',
      'TUNNEL_TYPE' : matcherTunnelType,
   }

   handler = tunnelTypeHandler

DecapGroupMode.addCommandClass( TunnelDecapTypeCmd )

#--------------------------------------------------------------------------------
# tunnel udp destination port 6635 payload PAYLOAD_TYPE
#--------------------------------------------------------------------------------
class TunnelUdpDestinationPortCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel udp destination port 6635 payload PAYLOAD_TYPE'
   noOrDefaultSyntax = 'tunnel udp destination port ...'

   data = {
      'tunnel' : matcherTunnel,
      'udp' : matcherUdp,
      'destination' : matcherDestination,
      'port' : 'Configure port for the tunnel',
      '6635' : 'Destination port',
      'payload' : 'Configure payload for the tunnel',
      'PAYLOAD_TYPE' : matcherPayloadType,
   }

   handler = tunnelDestinationPortHandler
   noOrDefaultHandler = delTunnelDestinationPortHandler

DecapGroupMode.addCommandClass( TunnelUdpDestinationPortCmd )

#--------------------------------------------------------------------------------
# forwarding vrf VRF_NAME
#--------------------------------------------------------------------------------
class ForwardingVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'forwarding VRF_NAME'
   noOrDefaultSyntax = 'forwarding vrf ...'

   data = {
      'forwarding' : matcherForwarding,
      'VRF_NAME' : VrfExprFactory( helpdesc='Configure forwarding vrf',
                                   inclDefaultVrf=True ),
      'vrf' : 'Configure forwarding vrf',
   }
   
   handler = forwardingVrfHandler
   noOrDefaultHandler = delForwardingVrfHandler

DecapGroupMode.addCommandClass( ForwardingVrfCmd )

#--------------------------------------------------------------------------------
# tunnel gre key GRE_KEY forwarding vrf VRF_NAME
#--------------------------------------------------------------------------------
class GreKeyToForwardingVrfMappingCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel gre key GRE_KEY forwarding VRF_NAME'
   noOrDefaultSyntax = syntax.replace( 'VRF_NAME', '[ VRF_NAME ]' )

   data = {
      'tunnel' : matcherTunnel,
      'gre' : matcherGreType,
      'key' : matcherGreKeyToken,
      'GRE_KEY' : matcherGreKey,
      'forwarding' : 'Configure forwarding settings for GRE decap group', 
      'VRF_NAME' : VrfExprFactory( helpdesc='Configure Forwarding VRF',
                                   inclDefaultVrf=True ),
   }

   handler = greKeyToForwardingVrfMappingHandler
   noOrDefaultHandler = delGreKeyToForwardingVrfMappingHandler

DecapGroupMode.addCommandClass( GreKeyToForwardingVrfMappingCmd )
