#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from TypeFuture import TacLazyType
import BasicCli
import BasicCliModes
import Toggles.TunnelToggleLib
import CliCommand
import CliMode.TunnelProfile
import CliMatcher
import ConfigMount
from CliToken.Ip import ipMatcherForConfig
from CliPlugin.IpAddrMatcher import ipPrefixMatcher
from CliPlugin.IpAddrMatcher import ipAddrMatcher
from CliPlugin.TunnelGlobalConfigMode import (
      matcherTunnel,
      matcherType,
      matcherUdp,
      matcherDestination,
)

IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
PolicyConfigEntry = TacLazyType( 'Tunnel::TunnelProfile::PolicyConfigEntry' )
TosValue = TacLazyType( 'Tunnel::TosValue' )
TristateU8 = TacLazyType( 'Ark::TristateU8' )

tunnelProfilesConfig = None

matcherProfiles = CliMatcher.KeywordMatcher( 'profiles',
      helpdesc='Configure tunnel profiles' )

matcherEncapsulation = CliMatcher.KeywordMatcher( 'encapsulation',
      helpdesc='Configure tunnel profiles encapsulation parameters' )

matcherPolicy = CliMatcher.KeywordMatcher( 'policy',
      helpdesc='Configure tunnel profiles policy' )

matcherPolicyName = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
      helpdesc='Name of the tunnel profile policy', helpname='WORD' )

matcherPolicySeq = CliMatcher.KeywordMatcher( 'seq',
      helpdesc='Configure tunnel profiles policy sequence' )

matcherPolicySeqMatch = CliMatcher.KeywordMatcher( 'match',
      helpdesc='Configure tunnel profiles policy sequence match' )

matcherTemplate = CliMatcher.KeywordMatcher( 'template',
      helpdesc='Configure a tunnel profiles template' )

matcherTemplateRef = CliMatcher.KeywordMatcher( 'template',
      helpdesc='Assign a tunnel profiles template' )

def getTemplateNames( mode ):
   return list( mode.encapTypeConfig.templateConfig ) if \
      mode.encapTypeConfig else []

matcherTemplateName = CliMatcher.DynamicNameMatcher(
   getTemplateNames,
   helpdesc='Name of the tunnel profile policy',
   pattern='[a-zA-Z0-9_-]+' )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel profiles
#--------------------------------------------------------------------------------
class TunnelProfilesMode( CliMode.TunnelProfile.TunnelProfilesBaseMode,
                          BasicCli.ConfigModeBase ):
   def __init__( self, parent, session ):
      CliMode.TunnelProfile.TunnelProfilesBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoTunnelProfilesMode( mode, args ):
   childMode = mode.childMode( TunnelProfilesMode )
   mode.session_.gotoChildMode( childMode )

def deleteTunnelProfiles( mode, args ):
   tunnelProfilesConfig.encapTypeConfig.clear()

class TunnelProfilesCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel profiles'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel': matcherTunnel,
      'profiles': matcherProfiles,
   }

   handler = gotoTunnelProfilesMode
   noOrDefaultHandler = deleteTunnelProfiles

if Toggles.TunnelToggleLib.toggleDynamicGueTunnelsEnabled():
   BasicCliModes.GlobalConfigMode.addCommandClass( TunnelProfilesCmd )

#--------------------------------------------------------------------------------
# [ no | default ] encapsulation type udp
#--------------------------------------------------------------------------------
class TunnelProfilesEncapTypeUdpMode( CliMode.TunnelProfile.\
                                   TunnelProfilesEncapTypeBaseMode, 
                                   BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, encapType ):
      CliMode.TunnelProfile.\
            TunnelProfilesEncapTypeBaseMode.__init__( self, encapType )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      encapTypeConfig = tunnelProfilesConfig.encapTypeConfig.get( 'udp' )
      if not encapTypeConfig:
         encapTypeConfig = tunnelProfilesConfig.encapTypeConfig.newMember( 'udp' )
      self.encapTypeConfig = encapTypeConfig

def gotoTunnelProfilesEncapTypeUdpMode( mode, args ):
   childMode = mode.childMode( TunnelProfilesEncapTypeUdpMode, encapType='udp' )
   mode.session_.gotoChildMode( childMode )

def deleteTunnelProfilesEncapTypeUdp( mode, args ):
   del tunnelProfilesConfig.encapTypeConfig[ 'udp' ]

class TunnelProfilesEncapTypeUdpCmd( CliCommand.CliCommandClass ):
   syntax = 'encapsulation type udp'
   noOrDefaultSyntax = syntax
   data = {
      'encapsulation': matcherEncapsulation,
      'type': matcherType,
      'udp': matcherUdp.matcher_,
   }
   handler = gotoTunnelProfilesEncapTypeUdpMode
   noOrDefaultHandler = deleteTunnelProfilesEncapTypeUdp

TunnelProfilesMode.addCommandClass( TunnelProfilesEncapTypeUdpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] template NAME
#--------------------------------------------------------------------------------
class TunnelProfilesTemplateMode( CliMode.TunnelProfile.\
                                  TunnelProfilesTemplateBaseMode,
                                  BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, templateName ):
      encapType = parent.encapType
      CliMode.TunnelProfile.\
         TunnelProfilesTemplateBaseMode.__init__( self, ( encapType,
                                                          templateName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.encapTypeConfig = parent.encapTypeConfig
      templateConfig = self.encapTypeConfig.templateConfig.get( templateName )
      if not templateConfig:
         templateConfig = \
            self.encapTypeConfig.templateConfig.newMember( templateName )
      self.templateConfig = templateConfig

def gotoTunnelProfilesTemplateMode( mode, args ):
   templateName = args.get( 'TEMPLATENAME' )
   childMode = mode.childMode( TunnelProfilesTemplateMode,
                               templateName=templateName )
   mode.session_.gotoChildMode( childMode )

def deleteTunnelProfilesTemplate( mode, args ):
   templateName = args.get( 'TEMPLATENAME' )
   del mode.encapTypeConfig.templateConfig[ templateName ]

class TunnelProfilesTemplateCmd( CliCommand.CliCommandClass ):
   syntax = 'template TEMPLATENAME'
   noOrDefaultSyntax = syntax
   data = {
      'template': matcherTemplate,
      'TEMPLATENAME': matcherTemplateName,
   }
   handler = gotoTunnelProfilesTemplateMode
   noOrDefaultHandler = deleteTunnelProfilesTemplate

if Toggles.TunnelToggleLib.toggleDynamicGueTunnelsTemplatesEnabled():
   TunnelProfilesEncapTypeUdpMode.addCommandClass( TunnelProfilesTemplateCmd )

#--------------------------------------------------------------------------------
# [ no | default ] source V4_ADDR
# [ no | default ] tos VALUE
# [ no | default ] ttl VALUE
#   in template sub mode
#--------------------------------------------------------------------------------
class TunnelProfilesTemplateSourceIpCmd( CliCommand.CliCommandClass ):
   syntax = 'source V4_ADDR'
   noOrDefaultSyntax = 'source ...'
   data = {
      'source': 'Source Address',
      'V4_ADDR': ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      addr = args[ 'V4_ADDR' ]
      mode.templateConfig.sourceIp = IpGenAddr( addr )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.templateConfig.sourceIp = IpGenAddr()

TunnelProfilesTemplateMode.addCommandClass( TunnelProfilesTemplateSourceIpCmd )

class TunnelProfilesTemplateTosCmd( CliCommand.CliCommandClass ):
   syntax = 'tos TOS'
   noOrDefaultSyntax = 'tos ...'
   data = {
      'tos': 'Type of Service',
      'TOS': CliMatcher.IntegerMatcher( 0, 255,
                                        helpdesc='Type of Service' )
   }

   @staticmethod
   def handler( mode, args ):
      tos = args[ 'TOS' ]
      mode.templateConfig.tos = TosValue( tos )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.templateConfig.tos = TosValue()

TunnelProfilesTemplateMode.addCommandClass( TunnelProfilesTemplateTosCmd )

class TunnelProfilesTemplateTtlCmd( CliCommand.CliCommandClass ):
   syntax = 'ttl TTL'
   noOrDefaultSyntax = 'ttl ...'
   data = {
      'ttl': 'Tunnel encapsulation TTL value',
      'TTL': CliMatcher.IntegerMatcher( 1, 64,
                                        helpdesc='TTL value' )
   }

   @staticmethod
   def handler( mode, args ):
      ttl = args[ 'TTL' ]
      mode.templateConfig.ttl = TristateU8( ttl, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.templateConfig.ttl = TristateU8()

TunnelProfilesTemplateMode.addCommandClass( TunnelProfilesTemplateTtlCmd )

#--------------------------------------------------------------------------------
# [ no | default ] policy POLICYNAME
#--------------------------------------------------------------------------------
class TunnelProfilesPolicyMode( CliMode.TunnelProfile.\
                                TunnelProfilePolicyBaseMode, 
                                BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, policyName ):
      encapType = parent.encapType
      CliMode.TunnelProfile.\
            TunnelProfilePolicyBaseMode.__init__( self, ( encapType, policyName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.encapTypeConfig = parent.encapTypeConfig
      policyConfig = self.encapTypeConfig.policyConfig.get( policyName )
      if not policyConfig:
         policyConfig = self.encapTypeConfig.policyConfig.newMember( policyName )
      self.policyConfig = policyConfig

def gotoTunnelProfilesPolicyMode( mode, args ):
   policyName = args.get( 'POLICYNAME' )
   childMode = mode.childMode( TunnelProfilesPolicyMode, policyName=policyName )
   mode.session_.gotoChildMode( childMode )

def deleteTunnelProfilesPolicy( mode, args ):
   policyName = args.get( 'POLICYNAME' )
   del mode.encapTypeConfig.policyConfig[ policyName ]

class TunnelProfilesPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'policy POLICYNAME'
   noOrDefaultSyntax = syntax
   data = {
      'policy': matcherPolicy,
      'POLICYNAME': matcherPolicyName,
   }
   handler = gotoTunnelProfilesPolicyMode
   noOrDefaultHandler = deleteTunnelProfilesPolicy

if Toggles.TunnelToggleLib.toggleDynamicGueTunnelsEnabled():
   TunnelProfilesEncapTypeUdpMode.addCommandClass( TunnelProfilesPolicyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] seq SEQNO
#--------------------------------------------------------------------------------
class TunnelProfilesPolicySeqMode( CliMode.TunnelProfile.\
                                   TunnelProfilePolicySeqBaseMode, 
                                   BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, seqno ):
      policyName = parent.policyName
      CliMode.TunnelProfile.\
            TunnelProfilePolicySeqBaseMode.__init__( self, ( policyName, seqno ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.encapTypeConfig = parent.encapTypeConfig
      self.policyConfig = parent.policyConfig
      policyConfigEntry = self.policyConfig.policyConfigEntry.get( seqno )
      if not policyConfigEntry:
         policyConfigEntry = self.policyConfig.policyConfigEntry.newMember( seqno )
      self.policyConfigEntry = policyConfigEntry

def gotoTunnelProfilesPolicySeqMode( mode, args ):
   seqno = args.get( 'SEQNO' )
   childMode = mode.childMode( TunnelProfilesPolicySeqMode, seqno=seqno )
   mode.session_.gotoChildMode( childMode )

def deleteTunnelProfilesPolicySeq( mode, args ):
   seqno = args.get( 'SEQNO' )
   del mode.policyConfig.policyConfigEntry[ seqno ]

class TunnelProfilesPolicySeqCmd( CliCommand.CliCommandClass ):
   syntax = 'seq SEQNO'
   noOrDefaultSyntax = syntax
   data = {
      'seq': matcherPolicySeq,
      'SEQNO': CliMatcher.IntegerMatcher( 1, 0x00FFFFFF, 
                                          helpdesc='Index in the sequence' ),
   }
   handler = gotoTunnelProfilesPolicySeqMode
   noOrDefaultHandler = deleteTunnelProfilesPolicySeq

if Toggles.TunnelToggleLib.toggleDynamicGueTunnelsEnabled():
   TunnelProfilesPolicyMode.addCommandClass( TunnelProfilesPolicySeqCmd )

#--------------------------------------------------------------------------------
# [ no | default ] match destination ip ( PREFIX | ( prefix-list PREFIXLIST ) )
#--------------------------------------------------------------------------------
class TunnelProfilesPolicyMatchDestinationIpCmd( CliCommand.CliCommandClass ):
   syntax = 'match destination ip ( PREFIX | ( prefix-list PREFIXLIST ) )'
   noOrDefaultSyntax = 'match destination ip ...'
   data = {
      'match': matcherPolicySeqMatch,
      'destination': matcherDestination.matcher_,
      'ip': ipMatcherForConfig,
      'PREFIX': ipPrefixMatcher,
      'prefix-list' : 'Prefix list',
      'PREFIXLIST': CliMatcher.PatternMatcher( 
                           pattern='[a-zA-Z0-9_-]+', 
                           helpdesc='Prefix list', 
                           helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      prefix = args.get( 'PREFIX' )
      prefixList = args.get( 'PREFIXLIST' )
      if prefix:
         mode.policyConfigEntry.matchDestinationIpPrefix = IpGenPrefix( prefix )
         mode.policyConfigEntry.matchDestinationIpPrefixList = ""
      elif prefixList:
         mode.policyConfigEntry.matchDestinationIpPrefix = IpGenPrefix( "" )
         mode.policyConfigEntry.matchDestinationIpPrefixList = prefixList

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.policyConfigEntry.matchDestinationIpPrefix = IpGenPrefix( "" )
      mode.policyConfigEntry.matchDestinationIpPrefixList = ""

TunnelProfilesPolicySeqMode.addCommandClass(
      TunnelProfilesPolicyMatchDestinationIpCmd )

class TunnelProfilesPolicyTemplateCmd( CliCommand.CliCommandClass ):
   syntax = 'template TEMPLATENAME'
   noOrDefaultSyntax = 'template ...'
   data = {
      'template': matcherTemplateRef,
      'TEMPLATENAME': matcherTemplateName,
   }

   @staticmethod
   def handler( mode, args ):
      templateName = args[ 'TEMPLATENAME' ]
      mode.policyConfigEntry.templateName = templateName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.policyConfigEntry.templateName = ''

if Toggles.TunnelToggleLib.toggleDynamicGueTunnelsTemplatesEnabled():
   TunnelProfilesPolicySeqMode.addCommandClass( TunnelProfilesPolicyTemplateCmd )

class TunnelProfilesPolicySourceIpCmd( CliCommand.CliCommandClass ):
   syntax = 'source V4_ADDR'
   noOrDefaultSyntax = 'source ...'
   data = {
      'source': 'Source Address',
      'V4_ADDR': ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      addr = args[ 'V4_ADDR' ]
      mode.policyConfigEntry.sourceIp = IpGenAddr( addr )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.policyConfigEntry.sourceIp = IpGenAddr()

TunnelProfilesPolicySeqMode.addCommandClass( TunnelProfilesPolicySourceIpCmd )

class TunnelProfilesPolicyTosCmd( CliCommand.CliCommandClass ):
   syntax = 'tos TOS'
   noOrDefaultSyntax = 'tos ...'
   data = {
      'tos': 'Type of Service',
      'TOS': CliMatcher.IntegerMatcher( 0, 255,
                                        helpdesc='Type of Service' )
   }

   @staticmethod
   def handler( mode, args ):
      tos = args[ 'TOS' ]
      mode.policyConfigEntry.tos = TosValue( tos )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.policyConfigEntry.tos = TosValue()

TunnelProfilesPolicySeqMode.addCommandClass( TunnelProfilesPolicyTosCmd )

class TunnelProfilesPolicyTtlCmd( CliCommand.CliCommandClass ):
   syntax = 'ttl TTL'
   noOrDefaultSyntax = 'ttl ...'
   data = {
      'ttl': 'Tunnel encapsulation TTL value',
      'TTL': CliMatcher.IntegerMatcher( 1, 64,
                                        helpdesc='TTL value' )
   }

   @staticmethod
   def handler( mode, args ):
      ttl = args[ 'TTL' ]
      mode.policyConfigEntry.ttl = TristateU8( ttl, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.policyConfigEntry.ttl = TristateU8( 64, False )

TunnelProfilesPolicySeqMode.addCommandClass( TunnelProfilesPolicyTtlCmd )

def Plugin( entityManager ):
   global tunnelProfilesConfig
   tunnelProfilesConfig = ConfigMount.mount( entityManager,
         "tunnel/tunnelProfiles/config",
         "Tunnel::TunnelProfile::TunnelProfilesConfig", "wi" )
