# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from ArnetModel import IpGenericPrefix
from Arnet import IpGenPrefix
from CliModel import Dict
from CliModel import Enum
from CliModel import List
from CliModel import Int
from CliModel import Str
from CliModel import Model
from TableOutput import createTable, Format
from CliPlugin.TunnelModels import tableTypeAttrs
import Tac
import six

# pylint: disable-next=undefined-variable,bad-option-value
largeInt = int if six.PY3 else long  # pylint:disable=long-builtin

class TunnelRibEntry( Model ):
   __revision__ = 2
   tunnelType = Enum( values=tableTypeAttrs, help="Tunnel type" )
   tunnelIndexes = List( valueType=int, help="List of tunnel indexes ending at "
                                              "this tunnel endpoint" )
   metric = Int( help="IGP metric of tunnel endpoint entry" )
   metricType = Enum( help="Metric type of tunnel endpoint entry",
                      values=( 'metric', 'MED', 'AIGP' ), optional=True )
   metric2 = Int( help="IGP metric 2 of tunnel endpoint entry" )
   pref = Int( help="Underlying IGP route's preference" )
   pref2 = Int( help="Preference 2 of tunnel endpoint entry" )
   tunnelPreference = Int( help="Tunnel preference used for tunnel RIB election" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         if dictRepr[ 'tunnelType' ] == "Nexthop Group":
            dictRepr[ 'tunnelType' ] = "Nexthop-Group"
      return dictRepr

   def renderTunnelRibEntry( self, table, prefix ):
      table.newRow( prefix, self.tunnelType,
            ', '.join( str( index ) for index in self.tunnelIndexes ),
            self.tunnelPreference, self.pref, self.metric, self.metricType )

   def renderColoredTunnelRibEntry( self, table, prefix, color ):
      table.newRow( prefix, color, self.tunnelType,
            ', '.join( str( index ) for index in self.tunnelIndexes ),
            self.tunnelPreference, self.pref, self.metric, self.metricType )

class TunnelRibEntriesByTunnelType( Model ):
   entries = List( valueType=TunnelRibEntry,
                   help="List of tunnel entries of different tunnel "
                   "types towards the same endpoint" )

   def renderTunnelRibEntry( self, table, prefix ):
      for entry in self.entries:
         entry.renderTunnelRibEntry( table, prefix )

   def renderColoredTunnelRibEntry( self, table, prefix, color ):
      for entry in self.entries:
         entry.renderColoredTunnelRibEntry( table, prefix, color )

class TunnelRibSummary( Model ):
   name = Str( help='Tunnel RIB name' )
   tunnels = Dict( keyType=str, valueType=int,
                   help='Tunnel count keyed by tunnel type' )

   def render( self ):
      print( "Tunnel RIB: %s" % self.name ) # pylint: disable=consider-using-f-string
      headings = ( 'Tunnel Type', 'Tunnel Count' )
      table = createTable( ( ( h, 'l' ) for h in headings ) )
      tunnelTypeFormat = Format( justify='left' )
      countFormat = Format( justify='right' )
      table.formatColumns( tunnelTypeFormat, countFormat )
      totalCount = 0
      for tunnelType, count in sorted( self.tunnels.items() ):
         table.newRow( tunnelType, count )
         totalCount = totalCount + count
      table.newRow()
      table.newRow( 'Total tunnels', totalCount )
      print( table.output() )

class TunnelRibAllSummary( Model ):
   tunnels = Dict( keyType=str, valueType=int,
                   help='Tunnel count keyed by tunnel RIB name' )

   def render( self ):
      headings = ( 'Tunnel RIB', 'Tunnel Count' )
      table = createTable( ( ( h, 'l' ) for h in headings ) )
      tunnelRibFormat = Format( justify='left' )
      tunnelRibFormat.noPadLeftIs( True )
      countFormat = Format( justify='right' )
      table.formatColumns( tunnelRibFormat, countFormat )
      for tunnelRibName, count in sorted( self.tunnels.items() ):
         table.newRow( tunnelRibName, count )
      print( table.output() )

class TunnelRib( Model ):
   __revision__ = 3
   # _isBrief = Bool( help="Only show prefix to tunnel index mappings" )
   name = Str( help='Tunnel RIB name' )
   entries = Dict( keyType=IpGenericPrefix, valueType=TunnelRibEntriesByTunnelType,
                   help="Tunnel entries keyed by prefix" )

   def degrade( self, dictRepr, revision ):
      if revision < 3:
         entriesDict = {}
         for ip, en in dictRepr[ 'entries' ].items():
            # Just pick the first entry from the list
            # There can be two entries if SR and flexAlgo tunnels form ecmp
            if en[ 'entries' ]:
               entriesDict[ ip ] = en[ 'entries' ][ 0 ]
            else:
               # Return an empty model
               entriesDict[ ip ] = TunnelRibEntry().toDict()
         dictRepr[ 'entries' ] = entriesDict
      return dictRepr

   def render( self ):
      print( "Tunnel RIB: %s" % self.name ) # pylint: disable=consider-using-f-string
      headings = ( "Endpoint", "Tunnel Type", "Index(es)",
                   "Tunnel Preference", "IGP Preference", "IGP Metric",
                   "Metric Type" )
      table = createTable( ( ( h, 'l' ) for h in headings ) )

      def _orderByPrefix( entriesItem ):
         pfx = IpGenPrefix( entriesItem[ 0 ] )
         # Need address-family first as sortKey order puts v6 prefixes first
         return ( pfx.af, pfx.sortKey )

      for prefix, tunnelRibEntry in sorted( self.entries.items(),
                                            key=_orderByPrefix ):
         tunnelRibEntry.renderTunnelRibEntry( table, prefix )

      print( table.output() )

class TunnelRibColors( Model ):
   colors = Dict( keyType=int, valueType=TunnelRibEntry,
                   help="Colored tunnel entries keyed by color" )

class ColoredTunnelRib( Model ):
   name = Str( help='System colored tunnel RIB name' )
   endpoints = Dict( keyType=IpGenericPrefix, valueType=TunnelRibColors,
                     help="Colored tunnel entries keyed by endpoint" )

   def addColoredRibEntry( self, prefix, color, ribEntry ):
      prefixColors = TunnelRibColors()
      if prefix in self.endpoints:
         prefixColors = self.endpoints[ prefix ]
      prefixColors.colors[ color ] = ribEntry
      self.endpoints[ prefix ] = prefixColors

   def render( self ):
      print( "Tunnel RIB: %s" % self.name ) # pylint: disable=consider-using-f-string
      headings = ( "Endpoint", "Color", "Tunnel Type", "Index(es)",
                   "Tunnel Preference", "IGP Preference", "IGP Metric",
                   "Metric Type" )
      table = createTable( ( ( h, 'l' ) for h in headings ), tableWidth=120 )

      def _orderByPrefix( entriesItem ):
         pfx = IpGenPrefix( entriesItem[ 0 ] )
         # Need address-family first as sortKey order puts v6 prefixes first
         return ( pfx.af, pfx.sortKey )

      for prefix, colors in sorted( self.endpoints.items(), key=_orderByPrefix ):
         for color, tunnelRibEntry in sorted( colors.colors.items() ):
            tunnelRibEntry.renderColoredTunnelRibEntry( table, prefix, color )

      print( table.output() )

class InternalTunnelRibCandidateEntry( Model ):
   tunnelType = Enum( values=tableTypeAttrs, help="Tunnel type" )
   tunnelIndexes = List( valueType=largeInt, help="Indexes of tunnels ending at "
                                              "this tunnel endpoint" )
   tunnelPreference = Int( help="Tunnel preference used for tunnel RIB election" )
   tunnelMetric = Int( help="Tunnel metric used for tunnel RIB election" )
   
   def renderInternalTunnelRibCandidateEntry( self, table, prefix, showTep ):
      optionalPrefix = prefix if showTep else ""
      table.newRow( optionalPrefix, self.tunnelType,
            ', '.join( str( index ) for index in self.tunnelIndexes ),
            self.tunnelPreference, self.tunnelMetric )


class InternalTunnelRibCandidates( Model ):
   candidates = List( valueType=InternalTunnelRibCandidateEntry,
         help="Candidate entries for tunnel RIB election," 
                " sorted by preference order")

   def renderInternalTunnelRibCandidates( self, table, prefix ):
      if self.candidates:
         self.candidates[ 0 ].renderInternalTunnelRibCandidateEntry( table, prefix, 
                                                                     showTep=True )
         for candidate in self.candidates[ 1: ]:
            candidate.renderInternalTunnelRibCandidateEntry( table, prefix, 
                                                             showTep=False )

class InternalTunnelRib( Model ):
   name = Str( help='Tunnel RIB name' )
   endpoints = Dict( keyType=IpGenericPrefix, 
                           valueType=InternalTunnelRibCandidates, 
                           help="Candidate tunnel entries, keyed by prefix" )

   def render( self ):
      print( "Tunnel RIB: %s" % self.name ) # pylint: disable=consider-using-f-string
      headings = ( "Endpoint", "Tunnel Type", "Index(es)",
                   "Tunnel Preference", "Tunnel Metric" )
      table = createTable( ( ( h, 'l' ) for h in headings ) )

      def _orderByPrefix( endpointsItem ):
         pfx = IpGenPrefix( endpointsItem[ 0 ] )
         # Need address-family first as sortKey order puts v6 prefixes first
         return ( pfx.af, pfx.sortKey )

      for prefix, tunnelRibTepCandidates in sorted( self.endpoints.items(),
                                            key=_orderByPrefix ):
         tunnelRibTepCandidates.renderInternalTunnelRibCandidates( table, prefix )

      print( table.output() )

class MulticastTunnelRibEntry( Model ):
   tunnelType = Enum( values=tableTypeAttrs, help="Tunnel type" )
   tunnelIndexes = List( valueType=int, help="List of tunnel indexes ending at "
                                              "this tunnel endpoint" )

   def renderTunnelMribEntry( self, table, fecElement ):
      table.newRow( fecElement, self.tunnelType,
            ', '.join( str( index ) for index in self.tunnelIndexes ) )

class MulticastTunnelRib( Model ):
   name = Str( help='Tunnel RIB name' )
   tunnels = Dict( keyType=str, valueType=MulticastTunnelRibEntry,
                help="Tunnel entries keyed by FEC Element" )

   def render( self ):
      headings = ( "Tunnel", "Tunnel Type", "Index(es)" )
      tunnelKeyFormat = Format( justify='left' )
      tunnelKeyFormat.padLimitIs( True )
      tunnelTypeFormat = Format( justify='left' )
      tunnelTypeFormat.padLimitIs( True )
      tunnelIndexFormat = Format( justify='right' )
      tunnelIndexFormat.padLimitIs( True )
      table = createTable( headings )
      table.formatColumns( tunnelKeyFormat, tunnelTypeFormat, tunnelIndexFormat )

      for fecElement, tunnelMribEntry in sorted( self.tunnels.items() ):
         tunnelMribEntry.renderTunnelMribEntry( table, fecElement )

      print( table.output() )
