# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
import CliToken.Monitor
import CliToken.Clear
import ConfigMount
import LazyMount
import ShowCommand
from TypeFuture import TacLazyType

twampLightHwStatus = None
twampLightConfig = None
twampLightStatus = None
ClearRequest = TacLazyType( "Twamp::Light::ClearRequest" )
ClearResultRequest = TacLazyType( "Twamp::Light::ClearResultRequest" )

def twampLightSupportedGuard( mode, token ):
   if twampLightHwStatus.twampLightSupported:
      return None
   return CliParser.guardNotThisPlatform

def getIpAddresses( mode ):
   ipAddresses = set()
   for intfStatus in twampLightStatus.intfStatus.values():
      for senderIpAddress in intfStatus.senderIpAddress.keys():
         ipAddresses.add( str( senderIpAddress ) )
      for reflectorIpAddress in intfStatus.reflectorIpAddress.keys():
         ipAddresses.add( str( reflectorIpAddress ) )

   return ipAddresses

ipMatcher = CliMatcher.DynamicNameMatcher(
   getIpAddresses,
   helpdesc='IP matcher',
   helpname="IP",
   pattern=r"[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+" )

# -----------------------------------------------------------------------------------
# clear monitor twamp-light counters [ reflector | sender ] [ ip IP ]
# -----------------------------------------------------------------------------------
class TwampLightCountersClearCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'clear monitor twamp-light counters [ reflector | sender ] [ ip IP ]'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'monitor': CliToken.Monitor.monitorMatcherForClear,
      'twamp-light': CliCommand.guardedKeyword( 'twamp-light',
                                                 helpdesc='Clear TWAMP light',
                                                 guard=twampLightSupportedGuard ),
      'counters': 'Clear TWAMP light counters',
      'reflector': 'Clear TWAMP light reflector counters',
      'sender': 'Clear TWAMP light sender counters',
      'ip': 'Clear results for specified IP Address',
      'IP': ipMatcher,
   }

   handler = "TwampLightCliHandler.handlerTwampLightCountersClearCmd"

BasicCli.addShowCommandClass( TwampLightCountersClearCmd )

# -----------------------------------------------------------------------------------
# clear monitor twamp-light [ ip IP ]
# -----------------------------------------------------------------------------------
class TwampLightResultsClearCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'clear monitor twamp-light [ ip IP ]'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'monitor': CliToken.Monitor.monitorMatcherForClear,
      'twamp-light': CliCommand.guardedKeyword( 'twamp-light',
                                                 helpdesc='Clear TWAMP light',
                                                 guard=twampLightSupportedGuard ),
      'ip': 'Clear results for specified IP Address',
      'IP': IpGenAddrMatcher(
                     helpdesc='Clear results for specified IP Address' ),
   }

   handler = "TwampLightCliHandler.handlerTwampLightResultsClearCmd"

BasicCli.addShowCommandClass( TwampLightResultsClearCmd )

def Plugin( entityManager ):
   global twampLightConfig
   global twampLightStatus
   global twampLightHwStatus

   twampLightConfig = ConfigMount.mount(
      entityManager, 'monitor/twamp/twampLight/config',
      'Twamp::Light::Config', 'w' )

   twampLightStatus = LazyMount.mount(
      entityManager, 'monitor/twamp/twampLight/status',
      'Twamp::Light::Status', 'r' )

   twampLightHwStatus = LazyMount.mount(
      entityManager, 'monitor/twamp/twampLight/hwstatus',
      'Twamp::Light::TwampLightHwStatus', 'r' )
