# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import AgentDirectory
import BasicCli
import CliCommand
from CliDynamicSymbol import CliDynamicPlugin
import CliGlobal
import CliMatcher
from CliModel import cliPrinted
import CliParser
from CliPlugin.IntfCli import Intf
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
import CliToken.Monitor
import LazyMount
import ShowCommand
import SharedMem
import Smash
import Tac

twampLightHwStatus = None
gv = CliGlobal.CliGlobal( entityManager=None, shmemEm=None, dirMounter=None )
TwampLightDelaysModel = CliDynamicPlugin( "TwampLightDelaysModel" )

def twampLightSupportedGuard( mode, token ):
   if twampLightHwStatus.twampLightSupported:
      return None
   return CliParser.guardNotThisPlatform

# -----------------------------------------------------------------------------------
# show monitor twamp-light counters [ sender | reflector | drop ]
# -----------------------------------------------------------------------------------
class TwampLightCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor twamp-light counters ' +\
            '[ sender | reflector | drop [ detail ] ]'
   data = {
      'monitor': CliToken.Monitor.monitorMatcherForShow,
      'twamp-light': CliCommand.guardedKeyword( 'twamp-light',
                                                 helpdesc='Show TWAMP light',
                                                 guard=twampLightSupportedGuard ),
      'counters': 'Show TWAMP light counters',
      'sender': 'Show TWAMP light sender',
      'reflector': 'Show TWAMP light reflector',
      'drop': 'Show TWAMP light drop counters',
      'detail': 'Detailed drop counters'
   }

   handler = "TwampLightCountersModel.TwampLightCounters.create"
   cliModel = "TwampLightCountersModel.TwampLightCounters"

BasicCli.addShowCommandClass( TwampLightCountersCmd )

# -----------------------------------------------------------------------------------
# show monitor twamp-light [ ip IP ]
# -----------------------------------------------------------------------------------
class TwampLightCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor twamp-light [ ip IP ]'
   data = {
      'monitor': CliToken.Monitor.monitorMatcherForShow,
      'twamp-light': 'Show TWAMP light',
      'ip': 'Show results for specified IP Address',
      'IP': IpGenAddrMatcher(
                     helpdesc='Show results for specified IP Address' ),
   }

   handler = "TwampLightResultModel.ResultsModel.create"
   cliModel = "TwampLightResultModel.ResultsModel"

BasicCli.addShowCommandClass( TwampLightCmd )

# -----------------------------------------------------------------------------------
# show monitor twamp-light [ ip IP ] detail
# -----------------------------------------------------------------------------------
class TwampLightDetailedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor twamp-light [ ip IP ] detail '
   data = {
      'monitor': CliToken.Monitor.monitorMatcherForShow,
      'twamp-light': 'Show TWAMP light',
      'ip': 'Show results for specified IP Address',
      'IP': IpGenAddrMatcher(
                     helpdesc='Show results for specified IP Address' ),
      'detail': 'Detailed results'
   }

   handler = "TwampLightResultModel.DetailedResultsModel.create"
   cliModel = "TwampLightResultModel.DetailedResultsModel"

BasicCli.addShowCommandClass( TwampLightDetailedCmd )

# -------------------------------------------------------------------------------
# show monitor twamp-light ( ( ip IP ) | ( interface INTF ) ) raw [ count NUM ]
# --------------------------------------------------------------------------------
class ShowTwampLightDelays( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show monitor twamp-light ( ( ip IP ) | ( interface INTF ) )'
              ' raw [ count NUM ]' )
   data = {
      'monitor': CliToken.Monitor.monitorMatcherForShow,
      'twamp-light': 'Show TWAMP light',
      'ip': 'Show delays for specified IP Address',
      'IP': IpGenAddrMatcher(
               helpdesc='Show delays for specified IP Address' ),
      'interface': 'Show delays for specified interface',
      'INTF': Intf.matcher,
      'raw': 'Raw result delays',
      'count': 'Number of delays to show',
      'NUM': CliMatcher.IntegerMatcher(
         0, 255, helpdesc='Number of delays' ),
   }
   cliModel = "TwampLightDelaysModel.TwampLightResultsModel"

   @staticmethod
   def handler( mode, args ):
      if not AgentDirectory.agentIsRunning( mode.entityManager.sysname(),
                                            "TwampAgent" ):
         mode.addWarning( 'TwampAgent is not running' )
         return TwampLightDelaysModel.TwampLightResultsModel()
      AgentCommandRequest.runCliPrintSocketCommandWithArgs(
         mode.entityManager, "TwampAgent", "showTwampLightDelays", args, mode=mode,
         model=TwampLightDelaysModel.TwampLightResultsModel )
      return cliPrinted( TwampLightDelaysModel.TwampLightResultsModel )

BasicCli.addShowCommandClass( ShowTwampLightDelays )

def Plugin( entityManager ):
   global twampLightHwStatus

   gv.entityManager = entityManager
   gv.shmemEm = SharedMem.entityManager( sysdbEm=entityManager )

   gv.dirMounter = Tac.newInstance( 'TacSmash::DirMounter', gv.shmemEm,
      'monitor/twamp/twampLight/result', '', Smash.mountInfo( 'reader' ) )
   gv.dirMounter.doInitialize()
   twampLightHwStatus = LazyMount.mount(
      entityManager, 'monitor/twamp/twampLight/hwstatus',
      'Twamp::Light::TwampLightHwStatus', 'r' )
