#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import LazyMount
import CliParser
import CliCommand
from CliPlugin.RouterGeneralCli import ( RouterGeneralMode,
                                         routerGeneralCleanupHook )
import Tac
import Tracing

routingHwConfig = None
routingHwStatus = None

traceHandle = Tracing.Handle( 'UrpfCli' )
t1 = traceHandle.trace1 # Info

def urpfCliGuard( mode, token ):
   if routingHwStatus.looseUrpfFwdSnoopSupported or \
      routingHwStatus.strictUrpfFwdSnoopSupported:
      return None
   return CliParser.guardNotThisPlatform

def urpfModeGuard( mode, token ):
   if token == 'any' and routingHwStatus.looseUrpfFwdSnoopSupported:
      return None
   if token  == 'rx' and routingHwStatus.strictUrpfFwdSnoopSupported:
      return None
   return CliParser.guardNotThisPlatform

def looseUrpfFailFwdSettingIs( enable ):
   routingHwConfig.looseUrpfFailForceFwd = enable

def strictUrpfFailFwdSettingIs( enable ):
   routingHwConfig.strictUrpfFailForceFwd = enable

class CfgUrpfCmd( CliCommand.CliCommandClass ):
   syntax = 'verify unicast source reachable-via ( any | rx )'
   noOrDefaultSyntax = syntax
   data = {
      'verify': 'source verification method',
      'unicast': CliCommand.guardedKeyword( 'unicast',
                                            'Unicast RPF configuration',
                                            guard=urpfCliGuard ),
      'source': 'RPF mode',
      'reachable-via': 'RPF mode',
      'any': CliCommand.guardedKeyword( 'any',
                                        'RPF loose-mode',
                                        guard=urpfModeGuard ),
      'rx': CliCommand.guardedKeyword( 'rx',
                                       'RPF strict-mode',
                                       guard=urpfModeGuard ),
   }

   handler = ( 'UrpfCliHandler.urpfModeHandler' )
   noOrDefaultHandler = ( 'UrpfCliHandler.noOrDefaultUrpfModeHandler' )

RouterGeneralMode.addCommandClass( CfgUrpfCmd )

#---------------------------------------------------------------
# Remove all Urpf configs when the parent is removed
# i.e., "no router general" in config mode.
#---------------------------------------------------------------

def noOrDefaultRouterGeneralMode( mode=None ):
   looseUrpfFailFwdSettingIs( False )
   strictUrpfFailFwdSettingIs( False )

routerGeneralCleanupHook.addExtension( noOrDefaultRouterGeneralMode )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global routingHwConfig
   global routingHwStatus

   routingHwConfig = LazyMount.mount( entityManager,
                                      "routing/hardware/config",
                                      "Routing::Hardware::Config", "w" )
   routingHwStatus = LazyMount.mount( entityManager,
                                      "routing/hardware/status",
                                      "Routing::Hardware::Status", "r" )
