# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module contains the definition of CliParser Rule classes for matching
# virtual interface names.
#-------------------------------------------------------------------------------

import re

import CliMatcher
import CliParser
import CliParserCommon
import Tracing

t0 = Tracing.trace0

#------------------------------------------------------------------------------------
# Rule that matches a virtual interface name of the format <baseName><intfNumber>,
# and evaluates to the canonical interface name.
#
# This rule matches full interface names only, but provides completions so partial
# name can be autocompleted.  Whitespace between the <baseName> and <intfNumber> is
# also allowed.
#-----------------------------------------------------------------------------------

def _intfNumberMatcher( basename, lbound, ubound, subIntf=False, rangeFunc=None,
                        helpdesc=None,
                        subLbound=CliParser.subLowerBound,
                        subUbound=CliParser.subUpperBound ):
   if subIntf and rangeFunc:
      return CliMatcher.DottedIntegerMatcher(
         None, None,
         rangeFn=rangeFunc,
         subLbound=subLbound, subUbound=subUbound,
         helpname=None,
         helpdesc=( helpdesc or '%s sub interface number' % basename ) )
   elif subIntf:
      return CliMatcher.DottedIntegerMatcher(
         lbound, ubound,
         subLbound=subLbound, subUbound=subUbound,
         helpname=None,
         helpdesc=( helpdesc or '%s sub interface number' % basename ) )
   elif rangeFunc:
      return CliMatcher.DynamicIntegerMatcher(
         rangeFunc,
         helpname=None,
         helpdesc=( helpdesc or '%s interface number' % basename ) )
   else:
      return CliMatcher.IntegerMatcher(
         lbound, ubound,
         helpname=None,
         helpdesc=( helpdesc or '%s interface number' % basename ) )

#------------------------------------------------------------------------------------
# Type of matcher that matches a virtual interface name without whitespace, and
# evaluates to the canonical interface name.
#------------------------------------------------------------------------------------
class _VirtualIntfMatcher:
   __slots__ = ( 'basename_', 'alternates_', 'numberMatcher_',
                 'subSupported_', 'subIntfGuard_' )
   # Basename must be followed by a number
   nameRe_ = re.compile( r'([a-zA-Z-]+)([0-9].*)$' )

   def __init__( self, basename, numberMatcher, subIntf=False, subIntfGuard=None,
                 alternates=None ):
      self.basename_ = basename
      self.alternates_ = tuple( alternates ) if alternates else None
      self.numberMatcher_ = numberMatcher
      self.subSupported_ = subIntf
      self.subIntfGuard_ = subIntfGuard
      if subIntfGuard:
         assert subIntf, "subIntf should be True if subIntfGuard is provided"

   def _prefixOfAlternates( self, intfType ):
      if self.alternates_:
         for x in self.alternates_: # pylint: disable=not-an-iterable
            if x.lower().startswith( intfType ):
               return x
      return None

   def match( self, mode, context, token ):
      if ( '.' in token and ( not self.subSupported_ or
                              ( self.subIntfGuard_ and
                                self.subIntfGuard_[ 0 ]( mode, token )
                                is not None ) ) ):
         return None
      m = self.nameRe_.match( token )
      if m is not None:
         # We only match if the name has two chars or more, otherwise
         # we rely on autocomplete. This is to avoid Grammar Error
         # when multiple interface names start with the same letter. Note,
         # we assume there are no interface names have a common prefix of
         # two letters or more.
         intfType = m.group( 1 ).lower()
         if not ( len( intfType ) > 1 and
                  ( self.basename_.lower().startswith( intfType ) or
                    self._prefixOfAlternates( intfType ) ) ):
            return None
         number = self.numberMatcher_.match( mode, context, m.group( 2 ) )
         if number is not CliMatcher.noMatch:
            return '%s%s' % ( self.basename_, number.result )
      return None

   def completions( self, mode, context, token ):
      # We never offer completions of <basename>, as otherwise we will
      # duplicate the completions offered by the rule that matches
      # <basename> <number> (with whitespace).
      # We only offer completions if token contains the beginning of a number.
      # For example, 'Vl1' would return [ 'Vlan1', '<1-4096>' ].
      m = self.nameRe_.match( token )
      if m:
         intfType = m.group( 1 ).lower()
         intfFullType = None
         if self.basename_.lower().startswith( intfType ):
            intfFullType = self.basename_
         else:
            intfFullType = self._prefixOfAlternates( intfType )
         if intfFullType:
            compList = self.numberMatcher_.completions( mode, context, m.group( 2 ) )
            if compList:
               # add literal completion for the name
               c = CliParser.Completion( intfFullType + m.group( 2 ),
                                         self.basename_ + ' interface',
                                         literal=True, partial=True )
               # Sometimes numberMatcher_ also returns literal completions,
               # ignore those (e.g., we only want the <> range completions).
               compList = [ x for x in compList if not x.literal ] + [ c ]
            return compList

      return []

   def __str__( self ):
      return '<%s>' % self.basename_

# New parser
class VirtualIntfMatcherBase( CliMatcher.Matcher ):
   # Unlike _VirtualIntfMatcher, we match both the "<Basename><Number>" and
   # "<Basename> <Number>" forms to simplify things, similar to VirtualIntfRule.
   # It's easier than having to use an expression.
   # pylint: disable-next=redefined-slots-in-subclass
   __slots__ = ( 'intfNumberMatcher_', 'intfNameMatcher_', 'alternates_',
                 'basename_', 'helpdesc_', 'guard_', 'subIntf_' )

   def __init__( self, basename, intfNameMatcher, intfNumberMatcher,
                 helpdesc=None, alternates=None, guard=None, subIntf=False,
                 **kwargs ):
      CliMatcher.Matcher.__init__( self, helpdesc=helpdesc, **kwargs )
      self.basename_ = basename
      self.intfNumberMatcher_ = intfNumberMatcher
      self.intfNameMatcher_ = intfNameMatcher
      self.alternates_ = alternates
      self.helpdesc_ = helpdesc
      self.guard_ = guard
      self.subIntf_ = subIntf

   def __str__( self ):
      if self.subIntf_:
         return '<%s.sub>' % self.basename_
      else:
         return '<%s>' % self.basename_

   def _guardError( self, mode, token ):
      if self.guard_ is not None and mode.session_.guardsEnabled():
         return self.guard_( mode, token )
      return None

   def _begin( self, context ):
      # since context could be shared with IntfMatcher, we save state
      # by our own key.
      if context.state is None:
         context.state = {}

   def match( self, mode, context, token ):
      self._begin( context )
      result = CliMatcher.noMatch
      if self not in context.state:
         t0( "first token", token )
         r = self.intfNameMatcher_.match( mode, context, token )
         if r is not None:
            result = CliMatcher.MatchResult( r, str( r ) )
         # try the "<Basename> <Number>" syntax
         elif ( token.lower() == self.basename_.lower() or
                self.alternates_ and
                any( token.lower() == x.lower() for x in self.alternates_ ) ):
            result = CliMatcher.partialMatch

         if result is not CliMatcher.noMatch:
            # check guard error; we assume if we match and throw a GuardError,
            # nobody else can match.
            #
            # Note we have to include the parse result in the GuardError.
            context.state[ self ] = True

            guardCode = self._guardError( mode, token )
            if guardCode is not None:
               raise CliParserCommon.GuardError( guardCode, result )
      else:
         t0( "second token", token )
         r = self.intfNumberMatcher_.match( mode, context, token )
         if r is not CliMatcher.noMatch:
            r = self.basename_ + str( r )
            result = CliMatcher.MatchResult( r, r )

      return result

   def completions( self, mode, context, token ):
      self._begin( context )
      if self not in context.state:
         compl = self.intfNameMatcher_.completions( mode, context, token )
         if not compl and self.basename_.lower().startswith( token.lower() ):
            compl = [ CliParser.Completion(
               self.basename_,
               self.helpdesc_ or '%s interface' % self.basename_,
               literal=True ) ]
         if compl:
            guardCode = self._guardError( mode, token )
            if guardCode is not None:
               for c in compl:
                  c.guardCode = guardCode
      else:
         compl = self.intfNumberMatcher_.completions( mode, context, token )
      return compl

class VirtualIntfMatcher( VirtualIntfMatcherBase ):
   __slots__ = ()

   def __init__( self, basename, lbound, ubound, helpdesc=None, alternates=None,
                 subIntf=False, subIntfGuard=None, rangeFunc=None, guard=None,
                 subLbound=CliParser.subLowerBound,
                 subUbound=CliParser.subUpperBound, **kwargs ):
      intfNumberMatcher = _intfNumberMatcher( basename, lbound, ubound,
                                              subIntf=subIntf, rangeFunc=rangeFunc,
                                              helpdesc=helpdesc,
                                              subLbound=subLbound,
                                              subUbound=subUbound )
      intfNameMatcher = _VirtualIntfMatcher( basename, intfNumberMatcher,
                                             subIntf=subIntf,
                                             subIntfGuard=subIntfGuard,
                                             alternates=alternates )
      VirtualIntfMatcherBase.__init__( self, basename, intfNameMatcher,
                                       intfNumberMatcher,
                                       helpdesc=helpdesc,
                                       alternates=alternates,
                                       subIntf=subIntf,
                                       guard=guard, **kwargs )

class IntfState:
   __slots__ = ( 'matchers', 'guardError' )
   def __init__( self, matchers ):
      self.matchers = set( matchers )
      self.guardError = None

class IntfMatcher( CliMatcher.Matcher ):
   # This is a matcher that can combine multiple VirtualIntfMatchers.
   # It also filters out non-literal completions if more than one literal
   # completions are returned.
   __slots__ = ( 'matchers_', )

   def __init__( self, **kargs ):
      CliMatcher.Matcher.__init__( self, **kargs )
      self.matchers_ = set()

   def __ior__( self, matcher ):
      # make sure no matcher has a common prefix more than 1 character
      assert matcher is not self
      self.matchers_.add( matcher )
      return self

   def __str__( self ):
      return '|'.join( str( m ) for m in self.matchers_ )

   def _checkPrefix( self, name ):
      name = name[ : 2 ]
      for m in self.matchers_:
         assert m.basename_.startswith( name )
         if m.alternates_:
            assert not any( alt.startswith( name ) for alt in m.alternates_ )

   def _begin( self, context ):
      if context.state is None:
         context.state = {}
      # keep track of unguarded and guarded matchers
      context.state.setdefault( self, IntfState( self.matchers_ ) )
      return context.state[ self ]

   def match( self, mode, context, token ):
      # use context.state to keep track of remaining sub matchers
      result = CliMatcher.noMatch
      state = self._begin( context )
      remove = []
      for matcher in state.matchers :
         t0( "match", matcher, "token", token )
         try:
            r = matcher.match( mode, context, token )
         except CliParserCommon.GuardError as e:
            if state.guardError is None:
               state.guardError = e
            r = CliMatcher.noMatch

         if r is CliMatcher.noMatch:
            remove.append( matcher )
         else:
            t0( "matched", r )
            if r is CliMatcher.partialMatch:
               result = r
            else:
               # we should only have one final match
               context.state[ self ] = matcher
               return r

      if result is CliMatcher.noMatch:
         if state.guardError:
            raise state.guardError
      else:
         # do not change context state if we return noMatch
         for matcher in remove:
            state.matchers.remove( matcher )
      return result

   def completions( self, mode, context, token ):
      state = self._begin( context )
      compl = []
      for matcher in state.matchers:
         compl += matcher.completions( mode, context, token )
      compl = list( set( compl ) )

      if token:
         literals = [ x for x in compl if x.literal ]
         if len( literals ) > 1:
            return literals
      return compl

   def valueFunction( self, context ):
      matcher = context.state[ self ]
      return matcher.valueFunction( context )
