#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import ShowCommand
import CliParser
# pylint: disable-next=consider-using-from-import
import CliPlugin.VmTracerCli as VmTracerCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.LagIntfCli as LagIntfCli
from CliPlugin.MacAddr import MacAddrMatcher
from CliPlugin.VmTracerModels import VmTracerVmDebug, VmTracerAll, \
      VmTracerSessionStatuses, VmTracerDebugAll, NetworkScopes, VmTracerEsxDebug, \
      VmTracerVxlanSegmentRange, VmTracerVms, VmTracerInterfaces
from CliToken.Vxlan import vxlanMatcherForShow
from Intf.IntfRange import IntfRangeMatcher

matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Interface' )

matcherAll = CliMatcher.KeywordMatcher( 'all', helpdesc='All switches' )
matcherCounters = CliMatcher.KeywordMatcher( 'counters', helpdesc='Counters' )
matcherDebug = CliMatcher.KeywordMatcher( 'debug',
      helpdesc='Dump contents of vi object stored in agent' )
matcherEsx = CliMatcher.KeywordMatcher( 'esx', helpdesc='Esx host' )
matcherEsxName = CliMatcher.QuotedStringMatcher(
      pattern=CliParser.excludePipeTokensPattern,
      helpname='ESX Name',
      helpdesc='Name of the ESX whose interfaces to display' )
matcherMac = CliMatcher.KeywordMatcher( 'mac',
      helpdesc='MAC of VM whose details to display' )
matcherName = CliMatcher.KeywordMatcher( 'name', helpdesc='VM Name' )
matcherPool = CliMatcher.KeywordMatcher( 'pool', helpdesc='Segment Pool' )
matcherSegment = CliMatcher.KeywordMatcher( 'segment', helpdesc='VXLAN segment' )
matcherSession = CliMatcher.KeywordMatcher( 'session',
      helpdesc='Details about a session' )
matcherSessionName = CliMatcher.PatternMatcher(
      pattern='[A-Za-z0-9_-]+',
      helpdesc='VmTracer session name',
      helpname='WORD' )
matcherVm = CliMatcher.KeywordMatcher( 'vm', helpdesc='VMs active' )
matcherVmName = CliMatcher.QuotedStringMatcher(
   pattern=CliParser.excludePipeTokensPattern,
   helpname='VM Name',
   helpdesc='Name of VM whose interfaces to display' )
matcherVmtracer = CliMatcher.KeywordMatcher( 'vmtracer',
      helpdesc='VmTracer information' )
matcherVnic = CliMatcher.KeywordMatcher( 'vnic', helpdesc='VNic' )
nodeVmtracer = CliCommand.Node( matcher=matcherVmtracer,
      guard=VmTracerCli.vmTracerSupportedGuard )
matcherIntfRange = IntfRangeMatcher(
      explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType,
                          LagIntfCli.LagAutoIntfType, ) )

#--------------------------------------------------------------------------------
# show vmtracer all [ mac MAC ]
#--------------------------------------------------------------------------------
class VmtracerAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer all [ mac MAC ]'
   data = {
      'vmtracer': nodeVmtracer,
      'all': matcherAll,
      'mac': matcherMac,
      'MAC': MacAddrMatcher(),
   }
   handler = VmTracerCli.doShowAll
   cliModel = VmTracerAll

BasicCli.addShowCommandClass( VmtracerAllCmd )

#--------------------------------------------------------------------------------
# show vmtracer debug [ session SESSION ]
#--------------------------------------------------------------------------------
class VmtracerDebugCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer debug [ session SESSION ]'
   data = {
      'vmtracer': nodeVmtracer,
      'debug': matcherDebug,
      'session': matcherSession,
      'SESSION': matcherSessionName,
   }
   handler = VmTracerCli.doShowVmTracerAllDebug
   cliModel = VmTracerDebugAll
   hidden = True

BasicCli.addShowCommandClass( VmtracerDebugCmd )

#--------------------------------------------------------------------------------
# show vmtracer debug esx ESX_NAME [ session SESSION ]
#--------------------------------------------------------------------------------
class VmtracerDebugEsxCliCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer debug esx ESX_NAME [ session SESSION ]'
   data = {
      'vmtracer': nodeVmtracer,
      'debug': matcherDebug,
      'esx': matcherEsx,
      'ESX_NAME': matcherEsxName,
      'session': matcherSession,
      'SESSION': matcherSessionName,
   }
   handler = VmTracerCli.doShowVmTracerEsxDebug
   cliModel = VmTracerEsxDebug
   hidden = True

BasicCli.addShowCommandClass( VmtracerDebugEsxCliCmd )

#--------------------------------------------------------------------------------
# show vmtracer debug vm VM_NAME [ session SESSION ]
#--------------------------------------------------------------------------------
class VmtracerDebugVmCliCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer debug vm VM_NAME [ session SESSION ]'
   data = {
      'vmtracer': nodeVmtracer,
      'debug': matcherDebug,
      'vm': matcherVm,
      'VM_NAME': matcherVmName,
      'session': matcherSession,
      'SESSION': matcherSessionName,
   }
   handler = VmTracerCli.doShowVmTracerVmDebug
   cliModel = VmTracerVmDebug
   hidden = True

BasicCli.addShowCommandClass( VmtracerDebugVmCliCmd )

#--------------------------------------------------------------------------------
# show vmtracer logical-switch [ SWITCH_NAME ]
#--------------------------------------------------------------------------------
class VmtracerLogicalSwitchCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer logical-switch [ SWITCH_NAME ]'
   data = {
      'vmtracer': nodeVmtracer,
      'logical-switch': 'Vmware logical switch',
      'SWITCH_NAME': CliMatcher.QuotedStringMatcher( 
                        pattern=CliParser.excludePipeTokensPattern,
                        helpdesc='Name of switch to display' )
   }
   handler = VmTracerCli.doShowLogicalSwitch
   cliModel = NetworkScopes

BasicCli.addShowCommandClass( VmtracerLogicalSwitchCmd )

#--------------------------------------------------------------------------------
# show vmtracer session [ SESSION ]
#--------------------------------------------------------------------------------
class VmtracerSessionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer session [ SESSION ]'
   data = {
      'vmtracer': nodeVmtracer,
      'session': matcherSession,
      'SESSION': matcherSessionName,
   }
   handler = VmTracerCli.doShowVmTracerSessionWord
   cliModel = VmTracerSessionStatuses

BasicCli.addShowCommandClass( VmtracerSessionCmd )

#--------------------------------------------------------------------------------
# show vmtracer session SESSION ( vcenter | vsm ) [ detail ]
#--------------------------------------------------------------------------------
class VmtracerSessionVcenterVsmCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer session SESSION ( vcenter | vsm ) [ detail ]'
   data = {
      'vmtracer': nodeVmtracer,
      'session': matcherSession,
      'SESSION': matcherSessionName,
      'vcenter': 'VCenter info',
      'vsm': 'VShield Manager Info',
      'detail': 'Detailed session information',
   }
   handler = VmTracerCli.doShowVmTracerSessionWordVCenterVsmDetail
   cliModel = VmTracerSessionStatuses

BasicCli.addShowCommandClass( VmtracerSessionVcenterVsmCmd )

#--------------------------------------------------------------------------------
# show vmtracer vm [ detail | ( mac MAC ) | ( name VM_NAME ) ]
#--------------------------------------------------------------------------------
class VmtracerVmCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer vm [ detail | ( mac MAC ) | ( name VM_NAME ) ]'
   data = {
      'vmtracer': nodeVmtracer,
      'vm': matcherVm,
      'detail': 'Display information in detail',
      'mac': matcherMac,
      'MAC': MacAddrMatcher(),
      'name': matcherName,
      'VM_NAME': matcherVmName
   }
   handler = VmTracerCli.doShowVm
   cliModel = VmTracerVms

BasicCli.addShowCommandClass( VmtracerVmCmd )

#--------------------------------------------------------------------------------
# show vmtracer vm VM_NAME
#--------------------------------------------------------------------------------
class VmtracerVmHiddenCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer vm VM_NAME'
   data = {
      'vmtracer': nodeVmtracer,
      'vm': matcherVm,
      'VM_NAME': matcherVmName
   }
   handler = VmTracerCli.doShowVm
   cliModel = VmTracerVms
   # The "show vmtracer vm <name> variant is hidden
   # to maintain backwards compatibility
   hidden = True

BasicCli.addShowCommandClass( VmtracerVmHiddenCmd )

#--------------------------------------------------------------------------------
# show vmtracer vm mac MAC vnic counters
#--------------------------------------------------------------------------------
class VmtracerVmMacVnicCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer vm ( ( mac MAC ) | ( name VM_NAME ) ) vnic counters'
   data = {
      'vmtracer': nodeVmtracer,
      'vm': matcherVm,
      'mac': matcherMac,
      'MAC': MacAddrMatcher(),
      'name': matcherName,
      'VM_NAME': matcherVmName,
      'vnic': matcherVnic,
      'counters': matcherCounters,
   }
   handler = VmTracerCli.doShowVmVnicCounters

BasicCli.addShowCommandClass( VmtracerVmMacVnicCountersCmd )

#--------------------------------------------------------------------------------
# show vmtracer vm VM_NAME vnic counters
#--------------------------------------------------------------------------------
class VmtracerVmVnicCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer vm VM_NAME vnic counters'
   data = {
      'vmtracer': nodeVmtracer,
      'vm': matcherVm,
      'VM_NAME': matcherVmName,
      'vnic': matcherVnic,
      'counters': matcherCounters,
   }
   handler = VmTracerCli.doShowVmVnicCounters
   hidden = True

BasicCli.addShowCommandClass( VmtracerVmVnicCountersCmd )

#--------------------------------------------------------------------------------
# show vmtracer vxlan segment [ pool ]
#--------------------------------------------------------------------------------
class VmtracerVxlanSegmentCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer vxlan segment [ pool ]'
   data = {
      'vmtracer': nodeVmtracer,
      'vxlan': vxlanMatcherForShow,
      'segment': matcherSegment,
      'pool': matcherPool,
   }
   handler = VmTracerCli.doShowVmTracerVxlanSegmentWrapper
   cliModel = NetworkScopes

BasicCli.addShowCommandClass( VmtracerVxlanSegmentCmd )

#--------------------------------------------------------------------------------
# show vmtracer vxlan segment pool POOL
#--------------------------------------------------------------------------------
class VmtracerVxlanSegmentPoolCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer vxlan segment pool POOL'
   data = {
      'vmtracer': nodeVmtracer,
      'vxlan': vxlanMatcherForShow,
      'segment': matcherSegment,
      'pool': matcherPool,
      'POOL': CliMatcher.QuotedStringMatcher(
         pattern=CliParser.excludePipeTokensPattern,
         helpdesc='Segment Pool Name' )
   }
   handler = VmTracerCli.doShowVmTracerVxlanSegmentPoolWord

BasicCli.addShowCommandClass( VmtracerVxlanSegmentPoolCmd )

#--------------------------------------------------------------------------------
# show vmtracer vxlan segment range
#--------------------------------------------------------------------------------
class VmtracerVxlanSegmentRangeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer vxlan segment range'
   data = {
      'vmtracer': nodeVmtracer,
      'vxlan': vxlanMatcherForShow,
      'segment': matcherSegment,
      'range': 'VNI and Multicast ranges',
   }
   handler = VmTracerCli.doShowVmTracerVxlanSegmentRange
   cliModel = VmTracerVxlanSegmentRange

BasicCli.addShowCommandClass( VmtracerVxlanSegmentRangeCmd )

#--------------------------------------------------------------------------------
# show vmtracer vxlan vm
#--------------------------------------------------------------------------------
class VmtracerVxlanVmCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer vxlan vm'
   data = {
      'vmtracer': nodeVmtracer,
      'vm': matcherVm,
      'vxlan': vxlanMatcherForShow,
   }
   handler = VmTracerCli.doShowVmTracerVxlanVm

BasicCli.addShowCommandClass( VmtracerVxlanVmCmd )

#--------------------------------------------------------------------------------
# show vmtracer interface [ INTFS ] [ detail ]
#--------------------------------------------------------------------------------
class VmtracerInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer interface [ INTFS ] [ detail ]'
   data = {
      'vmtracer' : nodeVmtracer,
      'interface' : matcherInterface,
      'INTFS' : matcherIntfRange,
      'detail' : 'Detailed display',
   }

   handler = VmTracerCli.doShowInterface
   cliModel = VmTracerInterfaces

BasicCli.addShowCommandClass( VmtracerInterfaceCmd )

#--------------------------------------------------------------------------------
# show vmtracer interface [ INTFS ] host
#--------------------------------------------------------------------------------
class VmtracerInterfaceHostCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer interface [ INTFS ] host'
   data = {
      'vmtracer' : nodeVmtracer,
      'interface' : matcherInterface,
      'INTFS' : matcherIntfRange,
      'host' : 'Host information',
   }

   handler = VmTracerCli.doShowHost

BasicCli.addShowCommandClass( VmtracerInterfaceHostCmd )

#--------------------------------------------------------------------------------
# show vmtracer interface ( INTFS ) vm VMNAME
#--------------------------------------------------------------------------------
class VmtracerInterfaceVmVmnameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer interface ( INTFS ) vm VMNAME'
   data = {
      'vmtracer' : nodeVmtracer,
      'vm' : matcherVm,
      'interface' : matcherInterface,
      'INTFS' : matcherIntfRange,
      'VMNAME' : CliMatcher.QuotedStringMatcher(
                 helpdesc='Name of VM whose interfaces to display',
                 helpname='VM Name', pattern=CliParser.excludePipeTokensPattern ),
   }

   handler = VmTracerCli.doShowInterface
   cliModel = VmTracerInterfaces

BasicCli.addShowCommandClass( VmtracerInterfaceVmVmnameCmd )

#--------------------------------------------------------------------------------
# show vmtracer interface [ INTFS ] vnic counters
#--------------------------------------------------------------------------------
class VmtracerInterfaceVnicCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vmtracer interface [ INTFS ] vnic counters'
   data = {
      'vmtracer' : nodeVmtracer,
      'interface' : matcherInterface,
      'INTFS' : matcherIntfRange,
      'vnic' : 'VNic',
      'counters' : matcherCounters,
   }

   handler = VmTracerCli.doShowInterfaceVnic

BasicCli.addShowCommandClass( VmtracerInterfaceVnicCountersCmd )
