# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This should be the one VRF CLI plugin to rule them all.
'''

import errno
import os
import subprocess
from functools import partial
from itertools import chain

import Cell
import CliCommand
import CliMatcher
import CliParser
import CliParserCommon
import CmdExtension
import LazyMount
import ManagedSubprocess
import Tac
from Arnet.NsLib import DEFAULT_NS
from CliModel import GeneratorDict, Model, UncheckedModel
from IpLibConsts import ( # pylint: disable-msg=unused-import
      ALL_VRF_NAME,
      DEFAULT_VRF,
      DEFAULT_VRF_OLD, # Unused.
      VRFNAMES_RESERVED as RESERVED_VRF_NAMES, # Also unused.
)
import Tracing

t0 = Tracing.trace0

allVrfConfig = None
allVrfStatusLocal = None

def getVrfNames( mode=None ):
   '''Returns an unsorted iterable of the names of the configured VRFs.
   '''
   return allVrfConfig.vrf

def getAllVrfNames( mode=None ):
   '''Returns an unsorted iterable of the names of the configured VRFs,
   including the default VRF.
   '''
   return chain( getVrfNames( mode ), ( DEFAULT_VRF, ) )

def getAllPlusReservedVrfNames( mode=None ):
   '''Returns an unsorted iterable of the names of the configured VRFs,
   including the default VRF and the reserved word 'all'.
   '''
   return chain( getAllVrfNames( mode ), ( ALL_VRF_NAME, ) )

def vrfExists( vrf, collection=None ):
   '''Function is pretty straightforward.
   The `collection` parameter can be either the ip/vrf/config mountpoint,
   from which we automatically look at the `vrf` attribute,
   or any other collection type.'''
   if not vrf:
      # pylint: disable-next=consider-using-f-string
      raise ValueError( 'Invalid VRF name: %r' % vrf )

   collection = collection or allVrfConfig
   try:
      # pylint: disable=unsupported-membership-test
      return vrf == DEFAULT_VRF or vrf in getattr( collection, 'vrf', collection )
   except IndexError: # len( vrf ) > 100
      return False

def getVrfNameFromIntfConfig( ipConfig, intf ):
   ipIntfConfig = ipConfig.ipIntfConfig.get( intf )
   if ipIntfConfig is None:
      return DEFAULT_VRF
   return ipIntfConfig.vrf

def nsFromVrf( vrfName=DEFAULT_VRF ):
   ns = DEFAULT_NS
   if vrfName == DEFAULT_VRF:
      return ns
   status = allVrfStatusLocal.vrf.get( vrfName )
   if status:
      return status.networkNamespace
   return None

vrfKwForShow = CliMatcher.KeywordMatcher( 'vrf',
      helpdesc='Display VRF state' )
vrfMatcher = CliMatcher.DynamicNameMatcher( getVrfNames,
      helpdesc='VRF name' )
vrfMatcherExcludeAll = CliMatcher.DynamicNameMatcher( getVrfNames,
      helpdesc='VRF name',
      pattern=r'^(?!all$)[A-Za-z0-9_.:{}\[\]-]+' )
vrfMatcherAll = CliMatcher.KeywordMatcher( ALL_VRF_NAME,
      helpdesc='All Virtual Routing and Forwarding instances' )
vrfMatcherDefault = CliMatcher.KeywordMatcher( DEFAULT_VRF,
      helpdesc='Default Virtual Routing and Forwarding instance' )

class VrfNameExprFactory( CliCommand.CliExpressionFactory ):
   '''Generates a `VRF | DEFAULT_VRF | all` expression, where:
   'VRF' is a dynamic list of VRFs, which defaults to all VRFS;
   'DEFAULT_VRF' is the default VRF; and
   'all' is a kewyord for all VRFs.
   The latter two are added based on the `defaultVrf` and `allVrf` flags.

   Similar to VrfExprFactory below, Some CLI commands that use VrfNameExprFactory are
   expecting vrfName 'all' without setting the explicit inclAllVrf flag to be True.
   Related to BUG754191
   '''
   def __init__( self, maxMatches=0, inclAllVrf=None, inclDefaultVrf=False ):
      '''Create a dictionary of desired matchers:
      dynamic (from f'n), default VRF, and all VRFs.
      '''
      CliCommand.CliExpressionFactory.__init__( self )
      self.maxMatches = maxMatches
      self.matchers_ = {
         'DYNAMIC': vrfMatcher if
                    inclAllVrf is None or inclAllVrf else
                    vrfMatcherExcludeAll,
      }
      if inclAllVrf:
         self.matchers_[ 'ALL' ] = vrfMatcherAll
      if inclDefaultVrf:
         self.matchers_[ 'DEFAULT' ] = vrfMatcherDefault

   def generate( self, name ):
      '''For each matcher, unique-fy its key and wrap it in a `Node` with `alias`.
      '''
      Node = partial( CliCommand.Node, alias=name, maxMatches=self.maxMatches )
      class VrfExpression( CliCommand.CliExpression ):
         data = { name + '_' + key: Node( matcher=matcher )
                  for key, matcher in self.matchers_.items() }
         expression = ' | '.join( data )

      return VrfExpression

class VrfExprFactory( CliCommand.CliExpressionFactory ):
   '''Generates a `vrf VRF` expression.

      Some CLI commands that use VrfExprFactory are expecting vrfName 'all'
      without setting the explicit inclAllVrf flag to be True.
      Temporary fix is to default the flag to None and only not match
      'all' when it is explicitly set to False. This allows backwards compatibility
      while commands that need to ignore 'all' can still do so. Related to BUG754191
   '''
   def __init__( self, keyword='vrf', helpdesc='Configure VRF',
                 guard=None, hidden=False, sharedMatchObj=None, maxMatches=0,
                 inclAllVrf=None, inclDefaultVrf=False ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.vrfNode_ = CliCommand.guardedKeyword( keyword,
                                                 helpdesc=helpdesc,
                                                 guard=guard,
                                                 hidden=hidden,
                                                 sharedMatchObj=sharedMatchObj,
                                                 maxMatches=maxMatches,
                                                 noResult=True )
      self.vrfExprFactory_ = VrfNameExprFactory( inclAllVrf=inclAllVrf,
                                                 inclDefaultVrf=inclDefaultVrf,
                                                 maxMatches=maxMatches )

   def generate( self, name ):
      kwName = name + '_VRF_KW'
      exprFactoryName = name + '_VRF'

      class VrfExpression( CliCommand.CliExpression ):
         expression = kwName + ' ' + exprFactoryName
         data = {
            kwName: self.vrfNode_,
            exprFactoryName: self.vrfExprFactory_
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            vrf = args.pop( exprFactoryName, None )
            if vrf:
               args[ name ] = vrf

      return VrfExpression

class _VrfCliModel( Model ):
   pass

class _VrfCliUncheckedModel( UncheckedModel ):
   pass

def generateVrfCliModel( cliModel, desc, uncheckedModel=False,
                         revision=None, baseModel=None ):
   '''
   This is used for generating Capi Models that uses the VrfExecCmdDec.
   All Cli functions that are decorated by VrfExecCmdDec will not be returning
   the models to the correct parent function because the decorator can run
   the same Cli function on multiple Vrfs. Therefore, rather than creating a
   top level model for every single Cli command that will be the exactly the same
   as each other, this function will automatically generate it.

   The name of the model is 'Vrf' + model name and contains a generator dictionary
   attribute which will expect the render function to iterate through and call
   render on.
   '''
   vrfModels = GeneratorDict( keyType=str, valueType=cliModel,
                              help=desc )

   def render( self ):
      for _, model in self.vrfs:
         model.render()

   if not revision:
      revision = cliModel.__revision__

   className = 'Vrf%s' % cliModel.__name__ # pylint: disable=consider-using-f-string
   classBody = { '__revision__': revision,
                 '__streamable__': cliModel.__streamable__,
                 'vrfs': vrfModels,
                 'render': render }

   if baseModel is None:
      baseModel = _VrfCliUncheckedModel if uncheckedModel else _VrfCliModel

   return type( className, ( baseModel, ), classBody )

vrfMap = None

class VrfExecCmdDec:
   '''
   Decorator class that can be used to process
   Cli commands of the type "show <command> vrf [VRFNAME|default|all]"
   or "show <command>" in a VRF routing-context mode

   This is to be used in conjunction with the Vrf Cli Parser rules defined
   above. The parsed VRF name token is returned in the keyword argument called
   'vrfName' and the decorated function returned by this class assumes the
   'vrfName' argument is present

   Example usage:
   to extend the 'show ip bgp' command with 'vrf [VRFNAME|default|all]'
   add the 'VrfExecCmdDec' decorator to the doShowXXX function and pass
   a @getVrfsFunc argument to the decorator. @getVrfsFunc is a function pointer
   for a function that returns the list of configured VRF names

   @VrfExecCmdDec( getVrfsFunc )
   def doShowIpBgp( mode, addr=None, longerPrefixes=None, detail=None,
                    vrfName=None ):
   '''
   def __init__( self, getVrfsFunc=None, cliModel=None ):
      '''The function to be decorated is not passed into the constructor'''
      self.getVrfsFunc = getVrfsFunc
      self.modelType = cliModel
      assert ( cliModel is None or
               issubclass( cliModel, ( _VrfCliModel, _VrfCliUncheckedModel ) ) )

   def __call__( self, func ):
      '''return a decoroated function that wraps @func
         arguments to the wrapped function are passed as is'''
      def vrfExecCmdFunc( *args, **kwargs ):
         '''
         first argument in the args list must be of type 'CliParser.Mode'
         and one of the keyword arguments must be named 'vrfName'
         '''
         if list( kwargs ) == [ 'args' ]:
            kwargRef = kwargs[ 'args' ]
            vrfNameKw = 'VRF'
            if vrfNameKw not in kwargRef:
               kwargRef[ vrfNameKw ] = None
         else:
            kwargRef = kwargs
            vrfNameKw = 'vrfName'

         if not isinstance( args[ 0 ], CliParser.Mode ) or \
            vrfNameKw not in kwargRef:
            # call func as is since we can't do anything better
            func( *args, **kwargs )
         v = kwargRef.get( vrfNameKw )
         mode = args[ 0 ]
         # actual list of VRFs that the command needs to be execd over
         vrfList = []
         # determine the VRF in the current routing context
         vrfName = vrfMap.lookupCliModeVrf( mode, v )
         if not vrfName:
            vrfList.append( DEFAULT_VRF )
         elif vrfName == ALL_VRF_NAME:
            vrfList.append( DEFAULT_VRF )
            for v in sorted( self.getVrfsFunc() ):
               if vrfName != DEFAULT_VRF:
                  vrfList.append( v )
         else:
            vrfList.append( vrfName )

         def execFunc():
            for v in vrfList:
               kwargRef[ vrfNameKw ] = v
               model = func( *args, **kwargs )
               if not model:
                  continue
               yield v, model

         length = len( vrfList )
         if self.modelType:
            model = self.modelType()
            model.vrfs = execFunc()
            return model
         else:
            for index, vrf in enumerate( vrfList ):
               kwargRef[ vrfNameKw ] = vrf
               func( *args, **kwargs )
               if index < ( length - 1 ):
                  # separate output between 2 vrfs by an empty line
                  print()

         return None

      return vrfExecCmdFunc

class CliVrfMapper:
   def __init__( self, allVrfStatusLocal_, uid, gid ):
      self.uid = uid
      self.gid = gid
      if allVrfStatusLocal_ is not None:
         self.allVrfStatusLocal = allVrfStatusLocal_
      else:
         self.allVrfStatusLocal = allVrfStatusLocal

   def setCliVrf( self, session, vrfName=DEFAULT_VRF, nsName=DEFAULT_NS ):
      session.sessionDataIs( 'vrf', ( vrfName, nsName ) )
      # Also cast to env in case of shell restart (say for the "watch" command)
      os.environ[ "CLI_VRF_NAME" ] = vrfName

   def getCliSessVrf( self, session ):
      vinf = session.sessionData( 'vrf', defaultValue=( DEFAULT_VRF, DEFAULT_NS ) )
      return vinf[ 0 ]

   def isDefaultVrf( self, vrfName ):
      return vrfName == DEFAULT_VRF

   def getVrfNamespace( self, vrfName ):
      assert vrfName != ''
      if vrfName == DEFAULT_VRF:
         return DEFAULT_NS
      vrfstat = self.allVrfStatusLocal.vrf.get( vrfName )
      if not ( vrfstat and vrfstat.state == 'active' ):
         # pylint: disable-next=consider-using-f-string
         raise OSError( errno.ENOENT, "VRF '%s' is not active" % vrfName )
      return vrfstat.networkNamespace

   def lookupCliSessVrf( self, session, vrfName ):
      assert vrfName != ''
      if vrfName is None:
         vrfName = self.getCliSessVrf( session )
      assert vrfName != None # pylint: disable=singleton-comparison
      return vrfName

   def lookupCliModeVrf( self, mode, vrfName ):
      return self.lookupCliSessVrf( mode.session, vrfName )

class CliVrfCmdExtension( CmdExtension.CmdExtension ):
   def __init__( self, cliVrfMap=None ):
      CmdExtension.CmdExtension.__init__( self )
      self.cliVrfMap = cliVrfMap

   def setVrfMap( self, cliVrfMap ):
      self.cliVrfMap = cliVrfMap

   def _getNsName( self, vrfName, cliSession ):
      assert vrfName != ''
      if vrfName == DEFAULT_VRF:
         return DEFAULT_NS
      assert self.cliVrfMap, "initVrfMap() is not called"
      vrfName = self.cliVrfMap.lookupCliSessVrf( cliSession, vrfName )
      assert vrfName
      return self.cliVrfMap.getVrfNamespace( vrfName )

   def _genVrfCmdLine( self, cliSession, cmdLine, kwargs ):
      # useSudo/runAsRoot is for Tac.run() that handles asRoot itself.
      # useSudo: whether we should use "sudo -E" as prefix; if not,
      # caller is going to take care of it.
      # runAsRoot: whether the passed in command is supposed to run as root.

      # returns whether command is modified
      nsName = kwargs.pop( 'nsName', None )
      vrfName = kwargs.pop( 'vrfName', None )

      asRoot = kwargs.get( 'asRoot', False ) # passed by runCmd()
      useSudo = kwargs.pop( 'useSudo', True )
      assert not ( asRoot and useSudo )
      system = kwargs.pop( 'system', False )

      assert vrfName != ''
      assert isinstance( cmdLine, list )

      if nsName is None:
         nsName = self._getNsName( vrfName, cliSession )

      if nsName != DEFAULT_NS:
         if not asRoot and useSudo and len( cmdLine ) > 1 and cmdLine[ 0 ] == 'sudo':
            # If the command starts with sudo, it's run as root.
            # This isn't exactly accurate, but we detect "sudo" or "sudo -E"
            # without other options which is probably good enough.
            if cmdLine[ 1 ] == '-E' or not cmdLine[ 1 ].startswith( '-' ):
               asRoot = True
               cmdLine.pop( 0 )
               if cmdLine[ 0 ] == '-E':
                  cmdLine.pop( 0 )

         prefixCmds = [ 'ip', 'netns', 'exec', nsName ]
         if useSudo:
            prefixCmds = [ 'sudo', '-E' ] + prefixCmds
         if not asRoot:
            uidStr = '{}#{}'.format( '\\' if system else '', os.getuid() )
            prefixCmds.extend( [ 'sudo', '-E', '-u', uidStr ] )

         # update the cmdLine list
         for cmd in reversed( prefixCmds ):
            cmdLine.insert( 0, cmd )

      if 'MOCK_VRF_CMD' in os.environ:
         # Just print out the command line and exit
         raise CliParserCommon.AlreadyHandledError(
            ' '.join( cmdLine ),
            CliParserCommon.AlreadyHandledError.TYPE_INFO )

      t0( ' '.join( cmdLine ), "useSudo", useSudo, "asRoot", asRoot )
      return nsName != DEFAULT_NS

   @CmdExtension.setDefaultArgs
   def extendCmd( self, execCmd, session, **kwargs ):
      return self._genVrfCmdLine( session, execCmd, kwargs )

   @CmdExtension.setDefaultArgs
   def runCmd( self, execCmd, session, **kwargs ):
      kwargs[ 'useSudo' ] = False
      asRoot = self._genVrfCmdLine( session, execCmd, kwargs )
      if asRoot:
         kwargs[ 'asRoot' ] = True
      return Tac.run( execCmd, **kwargs )

   @CmdExtension.setDefaultArgs
   def subprocessPopen( self, execCmd, session, **kwargs ):
      self._genVrfCmdLine( session, execCmd, kwargs )
      return subprocess.Popen( args=execCmd, **kwargs )

   @CmdExtension.setDefaultArgs
   def managedSubprocessPopen( self, execCmd, session, **kwargs ):
      self._genVrfCmdLine( session, execCmd, kwargs )
      return ManagedSubprocess.Popen( execCmd, **kwargs )

   @CmdExtension.setDefaultArgs
   def system( self, execCmd, session, **kwargs ):
      if isinstance( execCmd, str ):
         execCmd = execCmd.split()
      kwargs[ 'system' ] = True
      self._genVrfCmdLine( session, execCmd, kwargs )
      cmdStr = kwargs.get( 'env', '' )
      if cmdStr:
         cmdStr += ' '
      cmdStr += ' '.join( execCmd )
      return os.system( cmdStr )

vrfCmdExt = CliVrfCmdExtension()

# add extender now so we can handle the extra nsName/vrfName parameters
# but full support is only enabled when initVrfMap() is called.
CmdExtension.addCmdExtender( vrfCmdExt )

def initVrfMap( allVrfStatus=None ):
   global vrfMap
   vrfMap = CliVrfMapper( allVrfStatus,
                          os.getuid(), os.getgid() )
   vrfCmdExt.setVrfMap( vrfMap )

def Plugin( em ):
   global allVrfConfig, allVrfStatusLocal
   allVrfConfig = LazyMount.mount( em, 'ip/vrf/config', 'Ip::AllVrfConfig', 'wi' )
   allVrfStatusLocal = LazyMount.mount( em, Cell.path( 'ip/vrf/status/local' ),
                                        'Ip::AllVrfStatusLocal', 'r' )
   initVrfMap()
