#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
from Arnet import IpGenPrefix
import Cell
import CliParser
# pylint: disable-next=consider-using-from-import
import CliPlugin.VrfLeakCliModels as VrfLeakCliModels
from CliPlugin.RouterGeneralCli import ( RouterGeneralVrfMode,
                                         routerGeneralVrfCleanupHook )
from ConfigConsistencyChecker import UndefinedReferenceChecker
import ConfigMount
import LazyMount
import SmashLazyMount
import TableOutput
import Tac
import Tracing
import VrfLeakAgent

# pkgdeps: rpm VrfLeak-lib

t0 = Tracing.trace0

entityManager = None
vrfLeakConfig = None
vrfLeakStatus = None
vrfNameStatus = None
vrfIdStatus = None

class VrfLeakModelet( CliParser.Modelet ):
   def getPerVrfConfig( self ):
      t0( 'VrfLeakModelet::getPerVrfConfig for', self.mode.vrfName )
      pvc = vrfLeakConfig.vrf.get( self.mode.vrfName )
      if not pvc:
         t0( 'creating PVC' )
         pvc = vrfLeakConfig.vrf.newMember( self.mode.vrfName )
      return pvc

   def configureVrfLeak( self, args ):
      assert vrfLeakConfig
      pvc = self.getPerVrfConfig()
      assert pvc

      sourceVrfName = args[ 'VRFNAME' ]
      mapName = args.get( 'MAPNAME', '' )
      rcf = args.get( 'FUNCTION', '' )
      if rcf:
         assert rcf[ -2 : ] == '()'
         rcf = rcf[ : -2 ]
      policyName = mapName if mapName else rcf
      t0( 'adding leak from', sourceVrfName, 'to', self.mode.vrfName, 'using',
          policyName )

      if sourceVrfName in pvc.leakFrom:
         lfc = pvc.leakFrom[ sourceVrfName ]
      else:
         lfc = pvc.leakFrom.newMember( sourceVrfName )

      assert lfc
      lfc.routeMap = mapName
      lfc.rcf = rcf

   def noVrfLeak( self, args ):
      pvc = self.getPerVrfConfig()
      assert pvc

      sourceVrfName = args[ 'VRFNAME' ]
      t0( 'removing leak from', sourceVrfName, 'to', self.mode.vrfName )

      if sourceVrfName in pvc.leakFrom:
         del pvc.leakFrom[ sourceVrfName ]

      if len( pvc.leakFrom ) == 0:
         pvc = None
         t0( 'removing PVC for', self.mode.vrfName )
         del vrfLeakConfig.vrf[ self.mode.vrfName ]

RouterGeneralVrfMode.addModelet( VrfLeakModelet )

def VrfLeakRouterGeneralVrfCleanup( vrfName ):
   if vrfName in vrfLeakConfig.vrf:
      del vrfLeakConfig.vrf[ vrfName ]

routerGeneralVrfCleanupHook.addExtension( VrfLeakRouterGeneralVrfCleanup )

def showLeakVrfId( mode, args ):
   assert vrfNameStatus
   if not vrfNameStatus.nameToIdMap:
      mode.addError( 'No VRF name to id map. Are we in ribd mode?' )
      return None

   result = VrfLeakCliModels.VrfLeakId()
   for ( name, vrfId ) in vrfNameStatus.nameToIdMap.vrfNameToId.items():
      result.vrfs[ name ] = vrfId

   return result

fl = TableOutput.Format( justify='left' )
fr = TableOutput.Format( justify='right' )

def vrfNameFromId( vrfId ):
   ve = vrfIdStatus.vrfIdToName.get( vrfId )
   if ve:
      return ve.vrfName

   return 'UNKNOWN (%s)' % vrfId # pylint: disable=consider-using-f-string

def vrfIdFromName( mode, vrfName ):
   vrfId = vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName )
   if not vrfId:
      # pylint: disable-next=consider-using-f-string
      mode.addError( 'VRF %s not found in VRF id map' % vrfName )
      return None

   return vrfId

def showVrfLeakFlapping( mode, args ):
   source = args.get( 'SRC_VRF' )
   dest = args.get( 'DEST_VRF' )
   vrf = args.get( 'VRF' )
   prefix = args.get( 'PREFIX' )

   if not vrfNameStatus.nameToIdMap:
      print( 'No VRF name to id map.  Are we in ribd mode?' )
      return

   displayAll = False
   if not source and not dest and not vrf and not prefix:
      displayAll = True

   if vrf and ( source or dest ):
      mode.addError( 'Must supply either wildcard VRF or source/destination' )
      return

   sourceVrfId = None
   if source:
      sourceVrfId = vrfIdFromName( mode, source )
      if not sourceVrfId:
         return

   destVrfId = None
   if dest:
      destVrfId = vrfIdFromName( mode, dest )
      if not destVrfId:
         return

   wildVrfId = None
   if vrf:
      wildVrfId = vrfIdFromName( mode, vrf )
      if not wildVrfId:
         return

   table = TableOutput.createTable( ( 'Age', 'Source VRF', 'Destination VRF',
                                      'Prefix', 'Created At' ) )
   table.formatColumns( fr, fl, fl, fl )

   now = Tac.now()

   if prefix:
      assert isinstance( prefix, ( Tac.Type( 'Arnet::Ip6AddrWithMask' ), str ) )
      prefix = IpGenPrefix( str( prefix ) )

      print( f'---Prefix type is {type( prefix )} and {prefix}' )

   for ( vpp, bornAt ) in vrfLeakStatus.flappingBlacklist.items():
      display = True

      # the various options have AND semantics
      if not displayAll:
         if sourceVrfId and vpp.srcVrf != sourceVrfId:
            display = False
         if destVrfId and vpp.dstVrf != destVrfId:
            display = False

         # already validated that we can't have wild and source/dest
         # pylint: disable-next=consider-using-in
         if wildVrfId and ( vpp.srcVrf != wildVrfId and
                            vpp.dstVrf != wildVrfId ):
            display = False

         if prefix and vpp.prefix != prefix:
            display = False

      if display:
         age = now - bornAt

         srcName = vrfNameFromId( vpp.srcVrf )
         dstName = vrfNameFromId( vpp.dstVrf )

         table.newRow( int( age ), srcName, dstName, vpp.prefix, bornAt )

   print( table.output() )

def showTechVrfLeakRoute( mode, args ):
   # note that the code in VrfLeakSm that responds to this command is
   # sensitive to the order in which the parameters are sent, so don't
   # change it without fixing that
   cmd = 'route'
   if 'prefix' in args:
      assert 'exact-prefix' not in args
      cmd += ' prefix ' + str( args[ 'PREFIX' ] )
   elif 'exact-prefix' in args:
      cmd += ' exact-prefix ' + str( args[ 'PREFIX' ] )

   if 'VRF' in args:
      assert 'SRC_VRF' not in args
      assert 'DEST_VRF' not in args
      cmd += ' eitherVrf ' + str( args[ 'VRF' ] )
   else:
      if 'SRC_VRF' in args:
         cmd += ' srcVrf ' + str( args[ 'SRC_VRF' ] )

      if 'DEST_VRF' in args:
         cmd += ' dstVrf ' + str( args[ 'DEST_VRF' ] )

   AgentCommandRequest.runSocketCommand( entityManager, VrfLeakAgent.name(),
                                         'vrfLeak', cmd )

def showTechVrfLeakOverview( mode, args ):
   AgentCommandRequest.runSocketCommand( entityManager, VrfLeakAgent.name(),
                                         'vrfLeak', 'overview' )

def clearFlapping( mode, args ):
   vrfLeakConfig.clearBlacklistVersion += 1

class VrfLeakRouteMapReferenceGatherer:
   """
   This class gather references to route maps in 'leak routes' config commands.
   """
   @staticmethod
   def gather( feature ):
      references = set()
      for leakToVrf in vrfLeakConfig.vrf:
         perVrfConfig = vrfLeakConfig.vrf[ leakToVrf ]
         for leakFromVrf in perVrfConfig.leakFrom:
            leakFromConfig = perVrfConfig.leakFrom[ leakFromVrf ]
            routeMap = leakFromConfig.routeMap
            if routeMap:
               references.add( routeMap )
      return references

UndefinedReferenceChecker.addReferenceGatherer( [ "Route map" ],
                                                VrfLeakRouteMapReferenceGatherer )

def Plugin( em ):

   global entityManager, vrfLeakConfig, vrfLeakStatus
   global vrfNameStatus, vrfIdStatus

   entityManager = em
   vrfLeakConfig = ConfigMount.mount( em, 'routing/vrfleak/config',
                                     'Routing::VrfLeak::Config', 'w' )
   vrfLeakStatus = LazyMount.mount( entityManager,
                                    Cell.path( 'routing/vrfleak/status' ),
                                    'Routing::VrfLeak::StatusLocal', 'r' )
   vrfNameStatus = LazyMount.mount( em, Cell.path( 'vrf/vrfNameStatus' ),
                                    'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )
   vrfIdStatus = SmashLazyMount.mount( em, "vrf/vrfIdMapStatus",
                                       "Vrf::VrfIdMap::Status",
                                       SmashLazyMount.mountInfo( 'reader' ),
                                       autoUnmount=True )
