# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliParser
import CliCommand
from CliCommon import AlreadyHandledError
from CliMatcher import KeywordMatcher, DynamicNameMatcher
from CliPlugin.ClassificationCliLib import protectedFieldSetNamesRegex
from CliPlugin.Ip6AddrMatcher import Ip6PrefixMatcher
from CliPlugin.IpAddrMatcher import IpPrefixMatcher, PREFIX_OVERLAP_REJECT
from CliPlugin.TrafficPolicyCliModel import TrafficPolicyPrefixFieldSetModel
from CliPlugin.VrfSelectionPolicyCliModel import VrfSelectionPolicyModel
import CliPlugin.VrfSelectionPolicyCliLib as cliLib
from Arnet import IpGenPrefix
import ShowCommand
import sys
import Tac

def showVrfSelectionPolicyGuard( mode, token ):
   if ( cliLib.vrfSelPolicyHwCapabilities and
        cliLib.vrfSelPolicyHwCapabilities.vrfSelectionSupported ):
      return None
   return CliParser.guardNotThisPlatform

vrfMatcher = KeywordMatcher( 'vrf', helpdesc='Display VRF state' )
selectionMatcher = KeywordMatcher( 'selection',
                                   helpdesc='Display VRF selection policy' )
fieldSetMatcher = KeywordMatcher( 'field-set',
                                  helpdesc='VRF selection policy field set' )

def showVrfSelectionPolicyHandler( mode, args ):
   statusPath = cliLib.vrfSelPolicyHwCapabilities.vrfSelectionStatusPath
   filterParams = Tac.Value( "TrafficPolicyCli::FilterParams", "", False, False,
                             False, False, byteCountersSupported=False )
   helper = Tac.newInstance( 'VrfSelectionPolicy::ShowVrfSelectionPolicy',
                             filterParams, cliLib.vrfSelPolicyConfig,
                             cliLib.vrfSelPolicyStatus,
                             statusPath,
                             None, cliLib.gv.fieldSetConfig,
                             None, # appRecConfig
                             None, None, None,
                             cliLib.vrfSelPolicyIntfConfig, None, None,
                             None, True, False, None, None,
                             cliLib.vrfSelPolicyHwRouteStatus,
                             None, None, None, None )
   helper.policyJsonDictKeyword = "vrfSelectionPolicies"
   helper.policyFeatureName = "VRF selection policy "
   helper.render( sys.stdout.fileno(), mode.session_.outputFormat(),
                  mode.session_.requestedModelRevision() )
   return VrfSelectionPolicyModel

class ShowVrfSelectionPolicy( ShowCommand.ShowCliCommandClass ):
   syntax = '''show vrf selection policy'''

   data = {
         'vrf': vrfMatcher,
         'selection': selectionMatcher,
         'policy': CliCommand.guardedKeyword( 'policy',
                                    helpdesc='Display VRF selection policy',
                                    guard=showVrfSelectionPolicyGuard ),
   }
   cliModel = VrfSelectionPolicyModel
   handler = showVrfSelectionPolicyHandler

BasicCli.addShowCommandClass( ShowVrfSelectionPolicy )

# ------------------------------------------------------------------------
# show vrf selection policy field-set ( ipv4 | ipv6 ) prefix [ <name> ]
#   [ <prefix> [ exact ] ]
# ------------------------------------------------------------------------
def showVrfSelectionPolicyFieldSetHandler( mode, args ):
   # statusInput will try to grab some status that has field-set to policy mapping.
   # Platform agents only fill in one status.
   statusInput = None

   header = "VRF selection policy"
   allowExceptPrefixes = False
   ipv4 = 'ipv4' in args
   fieldSetName = args.get( 'FIELD_SET_NAME', '' )
   filterOutput = 'PREFIX' in args
   exact = 'exact' in args
   prefix = IpGenPrefix( '0.0.0.0/0' )
   if filterOutput:
      if args.get( 'PREFIX' ) is None:
         mode.addError( 'Host part of prefix must be zero' )
         raise AlreadyHandledError()
      prefix = IpGenPrefix( str( args.get( 'PREFIX' ) ) )
   helper = Tac.newInstance( 'TrafficPolicyCli::ShowTrafficPolicyPrefixFieldSet',
                             cliLib.gv.fieldSetConfig, fieldSetName, ipv4,
                             filterOutput, prefix, exact, statusInput,
                             header, allowExceptPrefixes )
   model = TrafficPolicyPrefixFieldSetModel
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   revision = mode.session_.requestedModelRevision()
   helper.render( fd, fmt, revision )

   return model

class ShowVrfSelectionPolicyIpPrefixFieldSetBase( ShowCommand.ShowCliCommandClass ):
   _baseData = {
      'vrf': vrfMatcher,
      'selection': selectionMatcher,
      'policy': CliCommand.guardedKeyword( 'policy',
                                           helpdesc='Display VRF selection policy',
                                           guard=showVrfSelectionPolicyGuard ),
      'field-set': fieldSetMatcher,
      'ipv4': 'IPv4 field sets',
      'ipv6': 'IPv6 field sets',
      'prefix': 'Prefix field set',
      'exact': 'exact match'
   }
   cliModel = TrafficPolicyPrefixFieldSetModel
   handler = showVrfSelectionPolicyFieldSetHandler

def getIpPrefixFieldSetNames( mode ):
   return cliLib.gv.fieldSetConfig.fieldSetIpPrefix

ipPrefixFieldSetNameMatcher = DynamicNameMatcher(
   getIpPrefixFieldSetNames,
   'IP prefix field-set name',
   # Note: pattern may not be appropriate when SDK-defined field sets are supported
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class ShowVrfSelectionPolicyIpv4PrefixFieldSet(
      ShowVrfSelectionPolicyIpPrefixFieldSetBase ):
   syntax = 'show vrf selection policy field-set ipv4 prefix [ FIELD_SET_NAME ] '\
            '[ PREFIX [ exact ] ]'
   data = {
      'FIELD_SET_NAME': ipPrefixFieldSetNameMatcher,
      "PREFIX": IpPrefixMatcher( 'IPv4 prefix', overlap=PREFIX_OVERLAP_REJECT ),
   }
   data.update( ShowVrfSelectionPolicyIpPrefixFieldSetBase._baseData )

BasicCli.addShowCommandClass( ShowVrfSelectionPolicyIpv4PrefixFieldSet )

def getIpv6PrefixFieldSetNames( mode ):
   return cliLib.gv.fieldSetConfig.fieldSetIpv6Prefix

ipv6PrefixFieldSetNameMatcher = DynamicNameMatcher(
   getIpv6PrefixFieldSetNames,
   'IPv6 prefix field-set name',
   # Note: pattern may not be appropriate when SDK-defined field sets are supported
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class ShowVrfSelectionPolicyIpv6PrefixFieldSet(
      ShowVrfSelectionPolicyIpPrefixFieldSetBase ):
   syntax = 'show vrf selection policy field-set ipv6 prefix [ FIELD_SET_NAME ]'\
            '[ PREFIX [ exact ] ]'
   data = {
      'FIELD_SET_NAME': ipv6PrefixFieldSetNameMatcher,
      "PREFIX": Ip6PrefixMatcher( 'IPv6 prefix', overlap=PREFIX_OVERLAP_REJECT ),
   }
   data.update( ShowVrfSelectionPolicyIpPrefixFieldSetBase._baseData )

BasicCli.addShowCommandClass( ShowVrfSelectionPolicyIpv6PrefixFieldSet )

