#!/usr/bin/env python3
# Copyright (c) 2012-2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import re
import sys
import Arnet
import BasicCli
import CliCommand
import CliMatcher
import CliToken.Bfd
import CliToken.Clear
import CliToken.Vxlan
import CliParser
import CliParserCommon
from CliPlugin import (
      AclCli,
      BridgingCli,
      ConfigConvert,
      ConfigTagCommon,
      IntfCli,
      IpAddrMatcher,
      Ip6AddrMatcher,
      IpGenAddrMatcher,
      IraCommonCli,
      IraIp6Cli,
      IraIpCli,
      IraIpRouteCliLib,
      IraIp6RouteCliLib,
      LoopbackIntfCli,
      DpsIntfCli,
      MacAddr,
      RouteMapCli,
      TechSupportCli,
      VirtualIntfRule,
      VlanCli,
      VrfCli,
)

from CliPlugin.IntfModel import (
   InterfaceCounters,
   InterfaceCountersRate,
   InterfaceCountersRates,
   VxlanInterfacesVniCounters
)
from CliPlugin.VxlanControllerModel import (
      VxlanMacVtepListPair,
      VxlanMacVtepPair,
      VxlanVniStatus,
      VxlanArpEntry,
      VxlanArpTable,
)
from CliPlugin.IraIp6RouteCliLib import (
      manageIpv6Route,
      noIpv6RouteTableForVrfMsg,
      routeMatcherForIpv6Config,
      ipv6PrefixMatcher,
)
from CliPlugin import VxlanIntfModel
from CliPlugin import VxlanModel
from CliPlugin.EthIntfCli import EthIntf, EthPhyIntf
from CliPlugin.VxlanConfigSanity import (
      ConfigCheckItem,
      GenConfigCheckItems,
      warnExplainedMsg,
      dynVlanVniWarnMsg,
      vlanNotCreatedFmt,
      vlanIsInternalFmt,
      vlanIsPeerlinkFmt,
      dynVlanVniConflictFmt,
      noRemoteVtepVlanFloodlistFmt,
      noVniInVrfToVniFormat,
      noVrfInVrfToVniFormat,
      noDynVlanForVrfToVniFormat,
      RouteTrie,
      underlayDefaultVrfId,
)
from CliPlugin.TcpMssCli import CfgTcpMssCeilingCmdBase
from CliPlugin.PolicyMapCliLib import matcherType
from CliDynamicSymbol import LazyCallback
import CliSession
import ConfigMount
import Ethernet
from Intf import IntfRange
import LazyMount
import MultiRangeRule
import Smash
import SmashLazyMount
import Shark
import SharedMem
import VxlanCliLib
import RoutingConsts
import Tac
import Tracing
from collections import Counter, defaultdict
from itertools import chain
from CliExtensions import CliHook
from IntfRangePlugin.VxlanIntf import VxlanAutoIntfType, vxlanRangeFn
from MlagMountHelper import MlagStatusLazyMounter
import Toggles.VxlanToggleLib
from Toggles.RoutingLibToggleLib import toggleFibGenMountPathEnabled
from VxlanVniLib import VniFormat
import VxlanVniLib
from Ark import synchronized
import threading
import six
from six.moves import zip
from six.moves import range
from VxlanCliLib import icmpHelperInputConfigClientName

t0 = Tracing.trace0

# pkgdeps library RoutingLib

configSanityFeatureDir = None
vxlanStatusDir = None
vxlanConfigDir = None
vxlanEcnConfig = None
vxlanFloodsetConfig = None
vxlanCounterConfigDir = None
vxAgentCounter = None
vxHwStatusDir = None
vxlanFdbStatus = None
vxlanFdbMultiVtepStatus = None
swGroupIdColl = None
vtiConfigDir = None
vtiStatusDir = None
bridgingHwCapabilities = None
bridgingHwEnabled = None
bridgingConfig = None
smashBridgingStatus = None
arpTableStatus = None
fhrpStatus = None
vrMacStatus = None
bridgingCliConfig = None
vlanXlateStatusDir = None
vxlanVniStatusDir = None
vxlanVniFdbStatusDir = None
vxlanClientDir = None
vxlanCliClientConfig = None
vxlanEvpnDynamicVlans = None
serviceConfigDir = None
vxlanControllerConfig = None
controllerErrorStatus = None
vniToVlanMap = None
mlagStatus = None
mlagHostTable = None
mlagVxlanStatus = None
mlagProtoStatus = None
em = None
vcsStateClientView = None
lRStatus = None
cvxClientStatus = None
hwCounterFeatureStatusDir = None
vxlanHwCounter = None
ip6Config = None
ipConfig = None
ipStatus = None
vxlanClientConvergenceStatus = None
remoteVniToVlanStatus = None
remoteVtepHwConfig = None
vtepHwStatus = None
ethAddrZero = Tac.Type( "Arnet::EthAddr" ).ethAddrZero
ipAddrZero = Tac.Type( 'Arnet::IpAddr' ).ipAddrZero
ipAddrWithMask = Tac.Type( 'Arnet::IpAddrWithMask' )
zeroAddrWithMask = ipAddrWithMask( '0.0.0.0', 0 )
ipsecConfig = None
routingHwStatusCommon = None
routingHwStatus = None
l2RibFloodSet = None
l3Config = None
l3StatusDir = None
vniPolicerHwStatusDir = None
vniPolicerEncapCounterTable = None
vniPolicerDecapCounterTable = None
vniPolicerEncapSnapshotCounterTable = None
vniPolicerDecapSnapshotCounterTable = None
macRewriteCliConfig = None
macRewriteCapability = None
vxlanCountersConfig = None
vxlanVniCountersDir = None
vxlanVniCountersCheckpointDir = None
vxlanAggregateCountersDir = None
vxlanAggregateCountersCheckpointDir = None
pbrVniIntfConfig = None
pbrCliConfig = None
configTagInput = None
vtepSipValidationStatusCli = None
routingStatus = None
routing6Status = None
forwardingStatusMulti = None
forwarding6StatusMulti = None
forwardingGenStatusMulti = None
forwardingStatus = None
forwarding6Status = None
forwardingGenStatus = None
icmpHelperInputConfigDir = None
routeTrieLockConfigSanity = threading.Lock()

vxlanBridgingConfigCheckCallback = []
vxlanRoutingConfigCheckCallback = []

# Callback for Trident4 platforms to warn of VXLAN configuration in unsupported
# UFT modes
warnVxlanUnsupportedUFTModeHook = CliHook()

vxlanEncapType = Tac.Type( "Vxlan::VxlanEncap" )
VtepType = Tac.Type( "L2Rib::VtepType" )
VxlanTunnelType = Tac.Type( "Vxlan::VxlanTunnelType" )
BfdInterval = Tac.Type( 'Bfd::BfdInterval' )
BfdMultiplier = Tac.Type( 'Bfd::BfdMultiplier' )
VxlanSrcPortRange = Tac.Type( 'Vxlan::VxlanSrcPortRange' )
L2RibSourceDecoder = Tac.Value( 'L2Rib::SourceDecoder' )
ExtendedVlanId = Tac.Type( 'Bridging::ExtendedVlanId' )
VrfNameAndIp = Tac.Type( "L3::VrfNameAndIp" )
VrfIpAndMac = Tac.Type( "Vxlan::Input::VrfIpAndMac" )
VniVtiPair = Tac.Type( "Vxlan::VniVtiPair" )
VniVtiAndMac = Tac.Type( "Vxlan::Input::VniVtiAndMac" )
AFType = Tac.Type( 'Arnet::AddressFamily' )
operState = Tac.Type( "ConfigTag::ConfigTagState" )
TristateBoolean = Tac.Type( "Ark::TristateBoolean" )
VrfVniConfigTagKey = Tac.Type( "Vxlan::VrfVniConfigTagKey" )
RemoveOrDisassociate = Tac.Type( "ConfigTag::RemoveOrDisassociate" )
FapId = Tac.Type( 'FlexCounters::FapId' )

# To box vxlanCtrlConfig
vxlanCtrlCfgBox = []

DEFAULT_VRF = VrfCli.DEFAULT_VRF

matcherAny = CliMatcher.KeywordMatcher( 'any',
      helpdesc='Learn from any addresses' )
matcherLearnRestrict = CliMatcher.KeywordMatcher( 'learn-restrict',
      helpdesc='VXLAN learning restrictions' )
vniMatcher = VxlanVniLib.VniMatcher( 'VXLAN Network Identifier',
      vxlanCtrlCfgBox )
vniRangeHelp = 'VXLAN Network Identifier (VNI) or range(s) of VNIs'
vniMultiMatcher = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( 1, 2**32 - 2 ),
      noSingletons=False,
      numberFormat=MultiRangeRule.NumberFormat.DOTTED,
      DoParseClass=VxlanVniLib.VniDottedDoParse,
      helpdesc=vniRangeHelp )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='VLAN configuration' )
matcherVrf = CliMatcher.KeywordMatcher( 'vrf',
      helpdesc='Set VXLAN decapsulation VRF behaviour' )
ipsecProfileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if ipsecConfig is None else ipsecConfig.ipsecProfile,
      helpdesc='VXLAN security profile name',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
vniAddRemoveMatcher = CliMatcher.EnumMatcher( { 'add': 'Add a VNI',
                                                'remove': 'Remove a VNI' } )
dynamicMatcherForConfig = CliMatcher.KeywordMatcher( 'dynamic',
      helpdesc='Dynamic' )
ribBypassMatcherForConfig = CliMatcher.KeywordMatcher( 'rib-bypass',
      helpdesc='RIB bypass' )
vniMatcherForConfig = CliMatcher.KeywordMatcher( 'vni',
                            helpdesc='VXLAN Network Identifier configuration' )

configTagExpr = ConfigTagCommon.ConfigTagExpr

@Tac.memoize
def vxlanCliHelper():
   LazyMount.force( bridgingHwCapabilities )
   return Tac.newInstance( "Vxlan::VxlanCliPluginHelper",
         vtiConfigDir, IraIpRouteCliLib.iraIpRouteCliHelper(),
         bridgingHwCapabilities, IraIpRouteCliLib.noRouteTableForVrfMsg )

@Tac.memoize
def configTagInputAllocator():
   return Tac.newInstance( "ConfigTag::ConfigTagInputAllocator" )

class SwTunnelGroupType( object ):
   singleMemberGroup = 0
   multiMemberGroup = 1

def ribBypassSupportedGuard( mode, token ):
   rhs = vxlanCliHelper().iraIpRouteCliHelper.routingHardwareStatus
   if rhs.staticVxlanRouteRibBypassSupported:
      return None
   return CliParser.guardNotThisPlatform

def isVxlan1InterfaceGuard( mode, token ):
   if mode.intf.name == "Vxlan1":
      return None
   return 'only supported for Vxlan1'

def vxlanMultiVtepMlagGuard( mode, token ):
   if bridgingHwCapabilities.vxlanMultiVtepMlagSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanSupported:
      return None
   return CliParser.guardNotThisPlatform

def tunnelSipValidationSupportedGuard( mode, token ):
   if bridgingHwCapabilities.tunnelSipValidationSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanTunnelSipValidationSupportedGuard( mode, token ):
   if bridgingHwCapabilities.tunnelSipValidationSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanEncapIpv6SupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanEncapIpv6ConfigSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanV6RemoteVtepSupportedGuard( mode, token ):
   if ':' not in token or bridgingHwCapabilities.vxlanEncapIpv6ConfigSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanVirtualVtepCmdSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanEncapIpv6ConfigSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanMcastDecapOnlySupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanMcastDecapOnlySupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanUnderlayMcastSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanUnderlayMcastSupported:
      return None
   return CliParser.guardNotThisPlatform

def pimAsmVxlanUnderlayIifSwitchGuard( mode, token ):
   if token == 'asm' and bridgingHwCapabilities.vxlanUnderlayIifSwitchSupported:
      return None
   elif token == 'ssm':
      return None
   return CliParser.guardNotThisPlatform

def vxlanIgmpSnoopingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanIgmpSnoopingSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanMcastUnderlayHerSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanMcastUnderlayHerSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanMcastIpv6OverlaySupportedGuard( mode, token ):
   if token != 'ipv6' or bridgingHwCapabilities.vxlanMcastIpv6OverlaySupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanMcastIpv6UnderlaySupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanMcastIpv6UnderlaySupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanFloodOverUnderlayMcastSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanFloodOverUnderlayMcastSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanMcastRoutingOverUnderlayMcastSupportedGuard( mode, token ):
   if routingHwStatus.vxlanMcastRoutingOverUnderlayMcastSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanRoutingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanRoutingSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanControllerClientSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanControllerClientSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanDecapFilterSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanDecapFilterSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanDecapFilterNonDefaultVrfSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanDecapFilterNonDefaultVrfSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanDecapRouteAllSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanDecapRouteAllSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanUdpPortSupportedGuard( mode, token ):
   if not bridgingHwCapabilities.vxlanMultiVtiMultiPortSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return None

def vxlanL3EvpnSupportedGuard( mode, token ):
   if ( bridgingHwCapabilities.vxlanRoutingSupported and
        bridgingHwCapabilities.vxlanL3EvpnSupported ):
      return None
   return CliParser.guardNotThisPlatform

def vtepCountersSupported( mode, token ):
   if bridgingHwCapabilities.vxlanVtepCtrSupported:
      return None
   return CliParser.guardNotThisPlatform

def vtepCountersDirectionFilterSupported( mode, token ):
   # This enables/disables the direction filter in the `show vxlan counters vtep`
   # command. It is set to not VNI-RemoteVTEP counters supported because only BFN
   # supports VNI-RemoteVTEP counters and VTEP counters in the encap direction,
   # and would like to disable the direction filter. Other platforms use the
   # direction filter. For now, it is pointless to create a new hardware capability
   # for this, but should change if more platforms want to disable direction filter
   # in the future.
   if not bridgingHwCapabilities.vxlanVtepVniCtrSupported:
      return None
   return CliParser.guardNotThisPlatform

def vtepVniCountersSupported( mode, token ):
   if bridgingHwCapabilities.vxlanVtepVniCtrSupported:
      return None
   return CliParser.guardNotThisPlatform

def vniCountersSupported( mode, token ):
   if ( bridgingHwCapabilities.vxlanVniEncapCtrSupported or
         bridgingHwCapabilities.vxlanVniDecapCtrSupported or
         bridgingHwCapabilities.vxlanVniCtrSupported ):
      return None
   return CliParser.guardNotThisPlatform

def vniCountersDirectionSupported( mode, token ):
   if token == 'encap':
      if ( bridgingHwCapabilities.vxlanVniEncapCtrSupported or
           bridgingHwCapabilities.vxlanVniCtrSupported ):
         return None
      return CliParser.guardNotThisPlatform
   else:
      if ( bridgingHwCapabilities.vxlanVniDecapCtrSupported or
           bridgingHwCapabilities.vxlanVniCtrSupported ):
         return None
      return CliParser.guardNotThisPlatform

def vxlanVniPolicerSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanVniPolicerSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanVniGroupPolicerSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanVniGroupPolicerSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanVniEncapPolicingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanVniEncapPolicingSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanVniDecapPolicingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanVniDecapPolicingSupported:
      return None
   return CliParser.guardNotThisPlatform

def vniPolicingSmashCountersSupported():
   return bridgingHwCapabilities.vniPolicingSmashCountersSupported

def vxlanVtepMaskCliSupported( mode, token ):
   if bridgingHwCapabilities.vxlanVtepMaskCliSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanSrcPortSupported( mode, token ):
   if bridgingHwCapabilities.vxlanSrcPortSupported != 'notSupported':
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanRangeSrcPortSupported( mode, token ):
   if bridgingHwCapabilities.vxlanSrcPortSupported == \
         'offsetAllValuesRangePowerOfTwo':
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanFixedSrcPortSupported( mode, token ):
   if bridgingHwCapabilities.vxlanSrcPortSupported == \
         'anySingleValue':
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanEcnPropagationSupported( mode, token ):
   if bridgingHwCapabilities.vxlanEcnPropagationSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanDecapDscpEcnPropagationDisabledGuard( mode, token ):
   if bridgingHwCapabilities.vxlanDecapDscpEcnPropagationDisableSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanDscpEcnRewriteBridgedEnabledGuard( mode, token ):
   if bridgingHwCapabilities.vxlanDscpEcnRewriteBridgedEnableSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanOuterDscpFromInnerDscpSupported( mode, token ):
   if bridgingHwCapabilities.vxlanOuterDscpFromInnerDscpSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanInnerDscpFromOuterDscpSupported( mode, token ):
   if bridgingHwCapabilities.vxlanInnerDscpFromOuterDscpSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanDecapUcastMcastFloodsetExcludeCpuSupported( mode, token ):
   if bridgingHwCapabilities.vxlanDecapUcastMcastFloodsetExcludeCpuSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanIpv6UnderlaySupportedGuard( mode, token ):
   if ':' not in token or bridgingHwCapabilities.vxlanIpv6UnderlaySupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlan32BitVniSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanHeader32BitVniSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanSourceIpRewriteSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanSourceIpRewriteSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanSourceIpRewriteSupportedShowGuard( mode, token ):
   if bridgingHwCapabilities.vxlanSourceIpRewriteSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanVtepToVtepBridgingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanVtepToVtepBridgingSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanSecurityGuard( mode, token ):
   if bridgingHwCapabilities.vxlanSecuritySupported:
      return None
   return CliParser.guardNotThisPlatform

def natWithVxlanSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanSupported and \
      bridgingHwCapabilities.natWithVxlanSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanTunnelVniAclSupportedGuard( mode, token ):
   if ( bridgingHwCapabilities.vxlanRoutingSupported and
        bridgingHwCapabilities.vxlanL3EvpnSupported and
       ( bridgingHwCapabilities.vxlanTunnelIngressDecapAclSupported or
         bridgingHwCapabilities.vxlanTunnelIngressIpv6DecapAclSupported or
         bridgingHwCapabilities.vxlanTunnelEgressIpv4VniAclSupported or
         bridgingHwCapabilities.vxlanTunnelEgressIpv6VniAclSupported ) ):
      return None
   return CliParser.guardNotThisPlatform

def nodeVniRoutedGuard( mode, token ):
   if vxlanTunnelVniAclSupportedGuard( mode, token ) is None or \
         natWithVxlanSupportedGuard( mode, token ) is None:
      return None
   return CliParser.guardNotThisPlatform

def vxlanTunnelIngressIpv4VniAclSupported( mode, token ):
   if bridgingHwCapabilities.vxlanTunnelIngressDecapAclSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanTunnelEgressIpv4VniAclSupported( mode, token ):
   if bridgingHwCapabilities.vxlanTunnelEgressIpv4VniAclSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanTunnelIngressIpv6VniAclSupported( mode, token ):
   if bridgingHwCapabilities.vxlanTunnelIngressIpv6DecapAclSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanTunnelEgressIpv6VniAclSupported( mode, token ):
   if bridgingHwCapabilities.vxlanTunnelEgressIpv6VniAclSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanIpv6TcpMssPlatformGuard( mode, token ):
   if bridgingHwCapabilities.vxlanIpv6TcpMssSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanTcpMssPlatformGuard( mode, token ):
   if ( bridgingHwCapabilities.vxlanTcpMssSupported or
        bridgingHwCapabilities.vxlanIpv6TcpMssSupported ):
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanTunnelIpv4VniAclSupported( mode, token ):
   if ( bridgingHwCapabilities.vxlanTunnelIngressDecapAclSupported or
        bridgingHwCapabilities.vxlanTunnelEgressIpv4VniAclSupported ):
      return None
   return CliParser.guardNotThisPlatform

def vxlanTunnelIpv6VniAclSupported( mode, token ):
   if ( bridgingHwCapabilities.vxlanTunnelIngressIpv6DecapAclSupported or
        bridgingHwCapabilities.vxlanTunnelEgressIpv6VniAclSupported ):
      return None
   return CliParser.guardNotThisPlatform

def vxlanSharedRouterMacSupported( mode, token ):
   if ( bridgingHwCapabilities.vxlanRoutingSupported and
        bridgingHwCapabilities.vxlanL3EvpnSupported ):
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def srcMacRewriteSupportedGuard( mode, token ):
   if macRewriteCapability.overlaySrcMacRewriteSupported or \
         macRewriteCapability.intfOverlaySrcMacRewriteSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def overlaySrcMacRewriteSupportedGuard( mode, token ):
   if macRewriteCapability.overlaySrcMacRewriteSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def intfOverlaySrcMacRewriteSupportedGuard( mode, token ):
   if macRewriteCapability.intfOverlaySrcMacRewriteSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def vxlanVniCountersSupportedGuard( mode, token ):
   if bridgingHwEnabled.vxlanVniIngressCountersEnabled or \
      bridgingHwEnabled.vxlanVniEgressCountersEnabled:
      return None
   return CliParser.guardNotThisPlatform

vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF which is mapped to a VNI',
      guard=vxlanL3EvpnSupportedGuard )
#
# VTI interface name to Arnet::IntfId cache.
#
# We use this to avoid building a short lived Tac object everytime we map a
# VTI name to interface Id. Mapping from a interface name string to interface
# id integer does not change.
#

class VtiNameToId( object ):
   __slots__ = [ 'nameIntfCache' ]

   def __init__( self ):
      self.nameIntfCache = {}

   def get( self, intfName ):
      if intfName not in self.nameIntfCache:
         vtiIntfId = Tac.Value( "Arnet::IntfId", intfName )
         self.nameIntfCache[ intfName ] = vtiIntfId

      return self.nameIntfCache[ intfName ]

# Global cache object for vti name intf caching
vtiNameToId = VtiNameToId()

class VniToVlanMap( object ):
   '''
   Builds and caches vni-to-vlan map for all VTIs. Cache structure is:

   { <interface> :  { expireTime : <time>,
                      vniToVlanMap : {...} },
   ...
   }

   The map is refreshed on access if cached time is older than the
   configured expiry time.
   '''

   # Default cache retention duration
   RETENTION = 30.0

   def __init__( self ):
      self.cache = {}

   def populateCache( self ):
      if not self.cache:
         self.scan()

   # Build vni to vlan mapping from vtiStatusDir. If an entry is not found,
   # maybe due to state propagation delay, we force refresh the cache.
   def scanVti( self, intfName ):
      vlanVniMap = vtiStatusDir.vtiStatus[ intfName ].extendedVlanToVniMap
      revMap = { vs.vni: vlan for vlan, vs in six.iteritems( vlanVniMap ) }

      if intfName not in self.cache:
         self.cache[ intfName ] = {}

      self.cache[ intfName ][ 'expiry' ] = Tac.now() + self.RETENTION
      self.cache[ intfName ][ 'revMap' ] = revMap

   def scan( self ):
      for intfId in vtiStatusDir.vtiStatus:
         self.scanVti( intfId )

   # get keys ( VNIs ) present
   def getKeys( self, intfId ):
      self.scan() # updates mapping for existing VTIs
      if intfId not in self.cache:
         self.cache[ intfId ] = { 'expiry': Tac.now() + self.RETENTION,
                                  'revMap': {} }
      return list( self.cache[ intfId ][ 'revMap' ] )

   def getVlan( self, vni, intfId ):
      # if the Vxlan intf has gone away clear out the old cache
      if intfId not in vtiStatusDir.vtiStatus:
         if intfId in self.cache:
            del self.cache[ intfId ]
         return None

      self.populateCache()

      # Refresh the cache if retention time expired or 'vni' is missing
      if ( Tac.now() > self.cache[ intfId ][ 'expiry' ] ) or \
         ( vni not in self.cache[ intfId ][ 'revMap' ] ):
         self.scanVti( intfId )

      return self.cache[ intfId ][ 'revMap' ].get( vni )

   def setVni( self, vni, vlan, intfId ):
      self.populateCache()
      if intfId not in self.cache:
         self.cache[ intfId ] = { 'expiry': Tac.now() + self.RETENTION,
                                  'revMap': {} }
      self.cache[ intfId ][ 'revMap' ][ vni ] = vlan

   def delVni( self, vni, intfId ):
      self.populateCache()
      if ( intfId in self.cache ) and ( vni in self.cache[ intfId ][ 'revMap' ] ):
         del self.cache[ intfId ][ 'revMap' ][ vni ]

def _vxlanSessionCommitHandler( mode, vxlanIntf ):
   if vxlanIntf in vtiConfigDir.vtiConfig:
      vxlanCounterConfigDir.vxlanCounterConfig.newMember( vxlanIntf )
   else:
      del vxlanCounterConfigDir.vxlanCounterConfig[ vxlanIntf ]

# CLI handler for `show interfaces counters vni`
def getVxlanVniDeltaCountersCheckpoint( mode, intf ):
   zeroOut, checkpoints = {}, {}
   # pylint: disable-msg=W0212
   deltaCounter = IntfCli._deltaCounterDir( mode ).get( intf.name )
   if deltaCounter is None:
      zeroOut, checkpoints = None, None
   else:
      for key, checkpoint in six.iteritems( deltaCounter.intfCounter ):
         # The aggregate delta counters are stored in the same collection as the
         # per-VNI counters. Skip the aggregate.
         if key == intf.name:
            continue
         vniVtiPair = Tac.const( Tac.newInstance( 'Vxlan::VniVtiPair',
                                                  int( key ), intf.name ) )
         zeroOut[ vniVtiPair ] = checkpoint is None
         checkpoints[ vniVtiPair ] = checkpoint
   return ( zeroOut, checkpoints )

def showVxlanVniCounters( mode, args ):
   intfs = args.get( 'INTFS' )
   vni = args.get( 'VNI' )
   direction = args.get( 'incoming' ) or args.get( 'outgoing' )

   intfsCounters = VxlanInterfacesVniCounters()
   if direction is not None:
      # pylint: disable-msg=W0212
      intfsCounters._outputType = direction

   intfs = IntfCli.perVniCounterSupportedIntfs( mode, intfs, None )
   if not intfs:
      return intfsCounters

   for x in intfs:
      if 'delta' in args:
         zeroOut, checkpoint = getVxlanVniDeltaCountersCheckpoint( mode, x )
         intfCountersModel = x.updateVxlanInterfaceCountersModel(
            checkpoint=checkpoint, zeroOut=zeroOut, vni=vni )
         intfsCounters.interfaces[ x.status().intfId ] = intfCountersModel
      else:
         checkpoint = x.getLatestVniCounterCheckpoint()
         intfCountersModel = x.updateVxlanInterfaceCountersModel(
            checkpoint=checkpoint, vni=vni )
         intfsCounters.interfaces[ x.status().intfId ] = intfCountersModel
         x.updateLastVniCounters( mode )

   return intfsCounters

# CLI handler for `show interfaces counters vni rates`
def showVxlanVniCountersRates( mode, args ):
   intfs = args.get( 'INTFS' )
   vni = args.get( 'VNI' )

   intfsCountersRates = InterfaceCountersRates()

   intfs = IntfCli.perVniCountersRateSupportedIntfs( mode, intfs, None )
   if not intfs:
      return intfsCountersRates

   for x in intfs:
      intfCountersRates = x.updateVxlanInterfaceCountersRateModel( vni )
      if intfCountersRates is None:
         mode.addWarning( "Counters unavailable for {}".format( x ) )
         continue
      intfsCountersRates.interfaces[ x.name_ ] = intfCountersRates

   return intfsCountersRates

# CLI hook to clear VTI counters
def clearVxlanIntfCountersHook( mode, intfs, sessionOnly, allIntfs ):
   vxlanRequest = vxlanCountersConfig

   if not bridgingHwEnabled.vxlanCountersEnabled or not vxlanRequest:
      return

   # Do nothing and return, the Ale hook will handle this, we don't want the
   # reactors to fire twice.
   if allIntfs:
      return

   # We don't support per-session clear.
   # If we do in the future, check for session clear here.

   for intf in intfs:
      intfId = Tac.newInstance( "Arnet::IntfId", intf.name )
      vxlanRequest.vtiClearRequest[ intfId ] = Tac.now()

IntfCli.registerClearCountersHook( clearVxlanIntfCountersHook )

def clearVxlanVniCounters( mode, intf, vni ):
   intfs = IntfCli.perVniCounterSupportedIntfs( mode, intf, None )
   if not intfs:
      return

   vxlanRequest = vxlanCountersConfig

   if not ( bridgingHwEnabled.vxlanVniIngressCountersEnabled or
            bridgingHwEnabled.vxlanVniEgressCountersEnabled ) or not vxlanRequest:
      return

   # We don't support per-session clear.
   # If we do in the future, check for session clear here.

   for x in intfs:
      intfId = Tac.newInstance( "Arnet::IntfId", x.name )
      if intfId not in vxlanRequest.vxlanClearCountersRequest:
         vxlanRequest.vxlanClearCountersRequest.newMember( intfId )
      vxlanRequest.vxlanClearCountersRequest[ intfId ]\
         .vniClearRequest[ int( vni ) ] = Tac.now()

class VxlanIntf( IntfCli.VxlanVirtualIntf ):
   def __init__( self, name, mode ):
      m = re.match( r'Vxlan(\d+)$', name )
      self.vtiId = int( m.group( 1 ) )
      IntfCli.VxlanVirtualIntf.__init__( self, name, mode )
      self.vxlanStatusDir = vxlanStatusDir
      self.vxlanConfigDir = vxlanConfigDir
      self.vxlanCounterConfigDir = vxlanCounterConfigDir
      self.vtiConfigDir = vtiConfigDir
      self.vtiStatuses = vtiStatusDir.vtiStatus
      self.vtiStatus = None
      self.l2RibFloodSet = l2RibFloodSet
      self.learnStatus = vxHwStatusDir.learnStatus
      self.vxlanCliClientConfig = vxlanCliClientConfig

   def createPhysical( self, startupConfig=False ):
      if self.mode_.session.inConfigSession():
         # The same callback can be used for destroyPhysical and createPhysical and
         # only one instance is needed hence the common 'VXLAN' key for both
         CliSession.registerSessionOnCommitHandler(
            self.mode_.session_.entityManager,
            'VXLAN-{}'.format( self.name ),
            lambda m, onSessionCommit: _vxlanSessionCommitHandler( m, self.name ) )
      else:
         self.vxlanCounterConfigDir.vxlanCounterConfig.newMember( self.name )

      self.vtiConfigDir.vtiConfig.newMember( self.name )
      self.vxlanConfigDir.vxlanConfig.newMember( self.name )
      # Hook is populated on Trident4 platforms to warn of conflicting configuration
      warnVxlanUnsupportedUFTModeHook.notifyExtensions( self.mode_ )

   def lookupPhysical( self ):
      self.vtiStatus = self.vtiStatuses.get( self.name, None )
      return self.vtiStatus is not None

   def hardware( self ):
      return "vxlan"

   def addrStr( self ):
      return None

   def bandwidth( self ):
      return 0

   def getIntfStatusModel( self ):
      return VxlanIntfModel.VxlanInterfaceStatus( name=self.status().intfId )

   def getReplicationModeStr( self ):
      # If Vxlan interface is down, return 'unknown' to preserve existing behavior.
      if self.vtiStatus.operStatus != 'intfOperUp':
         return 'unknown'

      if self.l2RibFloodSet.source is None:
         return 'unknown'

      sourceBgp = L2RibSourceDecoder.sourceFromEnum( 'sourceBgp' )
      sourceVcs = L2RibSourceDecoder.sourceFromEnum( 'sourceVcs' )
      sourceVxlanStatic = L2RibSourceDecoder.sourceFromEnum( 'sourceVxlanStatic' )
      sourceVxlanDynamic = L2RibSourceDecoder.sourceFromEnum( 'sourceVxlanDynamic' )
      if self.l2RibFloodSet.source.source & sourceBgp:
         return 'headendVcs'
      elif self.l2RibFloodSet.source.source & sourceVcs:
         return 'headendVcs'
      elif self.l2RibFloodSet.source.source & sourceVxlanStatic:
         return 'headendNonVcs'
      elif self.l2RibFloodSet.source.source & sourceVxlanDynamic:
         return 'headendNonVcs'
      else:
         return 'unknown'

   def showPhysical( self, mode, intfStatusModel ):
      vtiConfig = getVtiConfig( self.name )
      vxlanConfig = getVxlanConfig( self.name )
      intfStatusModel.srcIpIntf = vtiConfig.srcIpIntf
      intfStatusModel.vArpVtepSrcIpIntf = vtiConfig.varpVtepSrcIpIntf
      encapV4 = self.vtiStatus.vxlanEncap == vxlanEncapType.vxlanEncapIp4
      encapV6 = self.vtiStatus.vxlanEncap == vxlanEncapType.vxlanEncapIp6
      encapDual = self.vtiStatus.vxlanEncap == vxlanEncapType.vxlanEncapDual

      if ( encapV4 or encapDual ):
         intfStatusModel.srcIpAddr = self.vtiStatus.localVtepAddr
      if ( encapV6 or encapDual ):
         intfStatusModel.srcIpAddrV6 = self.vtiStatus.localVtepAddr6
      if ( ( encapV4 or encapDual ) and self.vtiStatus.vArpVtepAddr != '0.0.0.0' ):
         intfStatusModel.vArpVtepAddr = self.vtiStatus.vArpVtepAddr
      if ( ( encapV6 or encapDual ) and not self.vtiStatus.vArpVtepAddr6.isZero ):
         intfStatusModel.vArpVtepAddrV6 = self.vtiStatus.vArpVtepAddr6
      intfStatusModel.udpPort = self.vtiStatus.udpPort
      intfStatusModel.secUdpPort = self.vtiStatus.secUdpPort
      intfStatusModel.dataPathLearningMode = self.vtiStatus.datapathLearning
      intfStatusModel.controllerClientMode = self.vtiStatus.controllerClientMode
      intfStatusModel.floodLearnedAll = vtiConfig.floodLearnedAll
      intfStatusModel.arpReplyRelayMode = self.vtiStatus.arpReplyRelay
      intfStatusModel.arpLocalAddress = self.vtiStatus.arpLocalAddress
      intfStatusModel.controllerControlPlane = self.vtiStatus.controllerControlPlane
      intfStatusModel.vtepAddrMask = self.vtiStatus.vtepAddrMask
      intfStatusModel.vniInDottedNotation = \
            vxlanControllerConfig.vniInDottedNotation
      intfStatusModel.portDot1qVniMapping = self.vtiStatus.portDot1qVniMapping
      intfStatusModel.remoteVxlanDomain = self.vtiStatus.remoteVxlanDomain
      intfStatusModel.replicationMode = self.getReplicationModeStr()
      intfStatusModel.tcpMssV4 = self.vtiStatus.tcpMssV4
      intfStatusModel.tcpMssV6 = self.vtiStatus.tcpMssV6
      # pylint: disable-msg=W0212
      intfStatusModel._mlagActive = mlagStatus.mlagState in \
            ( 'primary', 'secondary' )
      # pylint: enable-msg=W0212
      intfStatusModel.mlagSharedRouterMacAddr = \
            self.vtiStatus.mlagSharedRouterMacAddr
      intfStatusModel.staticFloodlists = ( bool( vxlanConfig.vlanToVtepList ) or
                                           bool( vxlanConfig.floodVtepList ) or
                                           bool( vxlanConfig.floodVtepList6 ) )
      grps = ""
      if self.vtiStatus.mcastGrpDecap:
         grps = ' '.join( sorted( self.vtiStatus.mcastGrpDecap,
                        key=lambda x: Arnet.IpAddress( x ).value ) )
      intfStatusModel.mcastGrpDecap = grps

      # copy the vlanToVniMap into the Capi model from Sysdb
      def giveNextVlanVspMap( vlanToVniMap ):
         for vlan, vniSource in six.iteritems( self.vtiStatus.extendedVlanToVniMap ):
            vsp = VxlanIntfModel.VxlanInterfaceStatus.VniSourcePair()
            vsp.vni = vniSource.vni
            vsp.source = vniSource.source
            yield vlan, vsp

      intfStatusModel.vlanToVniMap = giveNextVlanVspMap(
         self.vtiStatus.extendedVlanToVniMap )

      if self.name == 'Vxlan1':
         # Construct VLAN to VTEP list from L2Rib::FloodSetOutput
         for vfs in self.l2RibFloodSet.vlanFloodSet.values():
            for fs in vfs.floodSet.values():
               vteps = VxlanIntfModel.VxlanInterfaceStatus.VxlanRemoteVtepAddrs()
               # Construct GeneratorList for IPv4/IPv6 flood entries only when IPv4/
               # IPv6 underlay is configured.
               if not encapV6 and self.vtiStatus.localVtepAddr != '0.0.0.0':
                  vteps.remoteVtepAddr = ( dest.vxlan.vtepAddr.v4Addr
                     for dest in fs.destSet if dest.destType == 'destTypeVxlan' and
                                               dest.vxlan.vtepAddr.af == 'ipv4' )
               if not encapV4 and not self.vtiStatus.localVtepAddr6.isZero:
                  vteps.remoteVtepAddr6 = ( dest.vxlan.vtepAddr.v6Addr
                     for dest in fs.destSet if dest.destType == 'destTypeVxlan' and
                                               dest.vxlan.vtepAddr.af == 'ipv6' )
               intfStatusModel.vlanToVtepList[ vfs.vlanId ] = vteps

         # Copy the vlanToLearnRestrict into the Capi model
         # Handle different cases of the learnStatusList entries going away
         for vlan in self.learnStatus.learnStatusList:
            lr = VxlanIntfModel.VxlanInterfaceStatus.VxlanVtepPrefixes()
            try:
               lr.learnFrom = self.learnStatus.learnStatusList[ vlan ].learnFrom
               if lr.learnFrom == 'learnFromDefault':
                  raise KeyError
               prefixList = self.learnStatus.learnStatusList[ vlan ].prefixList
               lr.prefixList = ( p.ipPrefix for p in prefixList )
            except KeyError:
               continue
            intfStatusModel.vlanToLearnRestrict[ vlan ] = lr

      # Copy vrfToVniMap to the Capi model
      for vrfName in self.vtiStatus.vrfToVniMap:
         vni = self.vtiStatus.vrfToVniMap[ vrfName ]
         intfStatusModel.vrfToVniMap[ vrfName ] = int( vni )

      # Copy alternateDecapVniToVrfMap to the Capi model
      for vni, vrf in six.iteritems( self.vtiStatus.alternateDecapVniToVrfMap ):
         intfStatusModel.decapVniToVrfMap[ int( vni ) ] = vrf

      # Copy MLAG source-interface and address
      if vtiConfig.mlagSrcIpIntf:
         intfStatusModel.mlagSrcIpIntf = vtiConfig.mlagSrcIpIntf
         if ( encapV4 or encapDual ):
            intfStatusModel.mlagSrcIpAddr = self.vtiStatus.mlagVtepAddr
         if ( encapV6 or encapDual ):
            intfStatusModel.mlagSrcIpAddrV6 = self.vtiStatus.mlagVtepAddr6

      intfStatusModel.use32BitVni = self.vtiStatus.use32BitVni
      intfStatusModel.vtepToVtepBridging = self.vtiStatus.vtepToVtepBridging
      intfStatusModel.vtepSetForSourcePruning = (
            vtep for vtep in self.vtiStatus.vtepSetForSourcePruning )
      intfStatusModel.vtepSourcePruningAll = self.vtiStatus.vtepSourcePruningAll
      intfStatusModel.decapFilterMode = vtiStatusDir.decapFilterMode
      intfStatusModel.decapFilterIntf = (
            intf for intf in vtiStatusDir.decapFilterIntf )
      intfStatusModel.decapRouteAll = vtiStatusDir.decapRouteAll
      intfStatusModel.mcastRouting = self.vtiStatus.mcastRouting

      # Convert vtiStatus.vccImportVlans into canonical string and
      # copy into intfStatusModel.vccImportVlan
      intfStatusModel.vccImportVlan = \
            MultiRangeRule.multiRangeToCanonicalString(
                  self.vtiStatus.vccImportVlans )
      if encapV4:
         intfStatusModel.vxlanEncapsulation = 'ipv4'
      elif encapV6:
         intfStatusModel.vxlanEncapsulation = 'ipv6'
      else:
         intfStatusModel.vxlanEncapsulation = 'ipv4AndIpv6'

      intfStatusModel.vtepToSecProfileTotal = len( self.vtiStatus.vtepToSecProfile )
      if self.vtiStatus.bfdEnabled:
         intfStatusModel.txInterval = self.vtiStatus.bfdIntervalParams.minTx
         intfStatusModel.rxInterval = self.vtiStatus.bfdIntervalParams.minRx
         intfStatusModel.detectMult = self.vtiStatus.bfdIntervalParams.mult
      intfStatusModel.bfdPrefixList = self.vtiStatus.bfdPrefixList

      intfStatusModel.vccArpProxy = self.vtiStatus.vccArpProxy
      intfStatusModel.vccNdProxy = self.vtiStatus.vccNdProxy

      intfStatusModel.vxlanArpProxy = \
            self.vxlanCliClientConfig.vxlanEncapsulatedArpSnooping.value if \
            self.vxlanCliClientConfig.vxlanEncapsulatedArpSnooping.isSet else False
      intfStatusModel.vxlanNdProxy = \
            self.vxlanCliClientConfig.vxlanEncapsulatedNdSnooping.value if \
            self.vxlanCliClientConfig.vxlanEncapsulatedNdSnooping.isSet else False

   def destroyPhysical( self ):
      if self.mode_.session.inConfigSession():
         # The same callback can be used for destroyPhysical and createPhysical and
         # only one instance is needed hence the common 'VXLAN' key for both
         CliSession.registerSessionOnCommitHandler(
            self.mode_.session_.entityManager,
            'VXLAN-{}'.format( self.name ),
            lambda m, onSessionCommit: _vxlanSessionCommitHandler( m, self.name ) )
      else:
         del self.vxlanCounterConfigDir.vxlanCounterConfig[ self.name ]

      del self.vxlanConfigDir.vxlanConfig[ self.name ]
      del self.vtiConfigDir.vtiConfig[ self.name ]
      adjustVxlanControllerServiceEnable()

   @staticmethod
   def getAllPhysical( sysdbRoot ):
      intfs = []
      for name in vtiStatusDir.vtiStatus:
         intf = VxlanIntf( name, sysdbRoot )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   #----------------------------------------------------------------------------
   # Returns the vtiConfig object for this interface.
   #----------------------------------------------------------------------------
   def config( self ):
      return self.vtiConfigDir.vtiConfig.get( self.name )

   #----------------------------------------------------------------------------
   # Returns the vtiStatus object for this interface.
   #----------------------------------------------------------------------------
   def status( self ):
      return self.vtiStatus

   #----------------------------------------------------------------------------
   # VXLAN aggregate counters related methods
   #----------------------------------------------------------------------------
   def getIntfCounterDir( self ):
      return vxlanAggregateCountersDir

   def getCounterCheckpoint( self, className=None, sessionOnly=False ):
      # Currently only BFN supports this without sessions, this can be extended
      # in the future to be like the other intf types
      if vxlanAggregateCountersCheckpointDir is None:
         return None
      checkpoint = vxlanAggregateCountersCheckpointDir.counterSnapshot.get(
         self.name )
      return checkpoint

   def getLatestCounterCheckpoint( self ):
      return self.getCounterCheckpoint( sessionOnly=False )

   def clearCounters( self, sessionOnly=False ):
      # This is used for sessionOnly counters, not needed now and can be extended in
      # the future to be like the other intf types
      return

   def ingVxlanAggregateCountersSupported( self ):
      return bridgingHwEnabled.vxlanAggregateIngressCountersEnabled

   def ingVxlanAggregateCountersRateSupported( self ):
      return bridgingHwEnabled.vxlanAggregateIngressCountersRateEnabled

   def egrVxlanAggregateCountersSupported( self ):
      return bridgingHwEnabled.vxlanAggregateEgressCountersEnabled

   def egrVxlanAggregateCountersRateSupported( self ):
      return bridgingHwEnabled.vxlanAggregateEgressCountersRateEnabled

   def countersSupported( self ):
      return self.ingVxlanAggregateCountersSupported() or \
             self.egrVxlanAggregateCountersSupported()

   def countersRateSupported( self ):
      return self.ingVxlanAggregateCountersRateSupported() or \
             self.egrVxlanAggregateCountersRateSupported()

   def counter( self ):
      if not self.countersSupported():
         return None
      countersDir = self.getIntfCounterDir()
      intfCounter = countersDir.counter.get( self.name )
      return intfCounter

   # This is the handler for VXLAN aggregate counters under
   # `show interfaces counters`
   def updateInterfaceCountersModel( self, checkpoint=None, notFoundString=None,
                                     zeroOut=False ):
      intfCountersModel = InterfaceCounters(
            _name=self.status().intfId,
            _ingressCounters=self.ingVxlanAggregateCountersSupported(),
            _egressCounters=self.egrVxlanAggregateCountersSupported(),
            _umCounters=False,
            _bumCounters=False,
            _vxlanCounters=self.countersSupported() )

      def stat( attr, supported=True ):
         if not supported:
            return 0
         currentValue = getattr( self.counter().statistics, attr, None )
         if currentValue is None:
            return notFoundString
         checkpointValue = 0
         if checkpoint:
            checkpointValue = getattr( checkpoint.statistics, attr, None )

         return currentValue - checkpointValue

      # Multicast and broadcast aggregate counters are not supported on VXLAN
      # interfaces, so they are a "don't care" (the attributes will get filtered out
      # in the CLI model render anyway).
      intfCountersModel.inMulticastPkts = 0
      intfCountersModel.inBroadcastPkts = 0
      intfCountersModel.outMulticastPkts = 0
      intfCountersModel.outBroadcastPkts = 0
      if not zeroOut:
         intfCountersModel.inOctets = stat( 'inOctets',
               self.ingVxlanAggregateCountersSupported() )
         intfCountersModel.inUcastPkts = stat( 'inUcastPkts',
               self.ingVxlanAggregateCountersSupported() )
         intfCountersModel.outOctets = stat( 'outOctets',
               self.egrVxlanAggregateCountersSupported() )
         intfCountersModel.outUcastPkts = stat( 'outUcastPkts',
               self.egrVxlanAggregateCountersSupported() )
      else:
         intfCountersModel.inOctets = 0
         intfCountersModel.inUcastPkts = 0
         intfCountersModel.outOctets = 0
         intfCountersModel.outUcastPkts = 0

      return intfCountersModel

   # This is the handler for VXLAN aggregate counters rate under
   # `show interfaces counters rates`
   def updateInterfaceCountersRateModel( self ):
      if not self.countersRateSupported() or self.counter() is None:
         return None

      # pylint: disable-msg=unused-variable
      def ratePct( bps, pps ):
         '''
         Uses the link bandwidth to determine the % of theoretical bitrate
         acheived including the 12 byte inter-frame gap and the 8 byte preamble

         This returns None for now since VXLAN interfaces are not physical interfaces
         and self.bandwidth() returns 0. This also means that rate percent will
         always be N/A for a VXLAN interface.

         Add rate percent calculation in the future here when self.bandwidth()
         returns non-zero for non-physical interfaces
         '''
         return None

      def rateValue( attr ):
         counter = self.counter()
         currentValue = getattr( counter.rates, attr )
         return currentValue

      intfCountersRateModel = InterfaceCountersRate( _name=self.name )
      intfCountersRateModel.description = self.description()
      # pylint: disable=round-builtin
      intfCountersRateModel.interval = int( round(
            IntfCli.getActualLoadIntervalValue( self.config().loadInterval ) ) )
      intfCountersRateModel.inBpsRate = rateValue( 'inBitsRate' )
      intfCountersRateModel.inPpsRate = rateValue( 'inPktsRate' )
      # replace with the following when self.bandwidth() returns non-zero
      # ratePct( intfCountersRateModel.inBpsRate, intfCountersRateModel.inPpsRate )
      intfCountersRateModel.inPktsRate = None
      intfCountersRateModel.outBpsRate = rateValue( 'outBitsRate' )
      intfCountersRateModel.outPpsRate = rateValue( 'outPktsRate' )
      # replace with the following when self.bandwidth() returns non-zero
      # ratePct( intfCountersRateModel.inBpsRate, intfCountersRateModel.inPpsRate )
      intfCountersRateModel.outPktsRate = None

      return intfCountersRateModel

   #----------------------------------------------------------------------------
   # VXLAN per-VNI counters related methods
   #----------------------------------------------------------------------------
   def getIntfVniCounterDir( self ):
      # Currently only BFN supports this, it will be None for other platforms
      return vxlanVniCountersDir

   def getVniCounterCheckpoint( self, className=None, sessionOnly=False ):
      # Currently only BFN supports this without sessions, this can be extended
      # in the future to be like the other intf types
      if vxlanVniCountersCheckpointDir is None:
         return None
      counterCheckpoints = {}
      for vniVtiPair, checkpoint in six.iteritems(
            vxlanVniCountersCheckpointDir.counterSnapshot ):
         if vniVtiPair.vti == self.status().intfId:
            counterCheckpoints[ vniVtiPair ] = checkpoint
      return counterCheckpoints

   def getLatestVniCounterCheckpoint( self ):
      return self.getVniCounterCheckpoint( sessionOnly=False )

   def clearVniCounters( self, sessionOnly=False ):
      # This is used for sessionOnly counters, not needed now and can be extended in
      # the future to be like the other intf types
      return

   def storeLastVniCounters( self, mode, className=None, vnis=None ):
      if vnis is None:
         return []
      # pylint: disable-msg=W0212
      delta = IntfCli._deltaCounterDir( mode ).get( self.name )
      if delta is None:
         if className is None:
            delta = Tac.newInstance( "Interface::IntfCounterDir", self.name )
         else:
            delta = Tac.newInstance( "%sDir" % ( className ), self.name )

      counters = {}

      for vni in vnis:
         counter = delta.intfCounter.get( str( vni ) )
         if counter is None:
            counter = delta.newIntfCounter( str( vni ) )
         counters[ str( vni ) ] = counter

      # pylint: disable-msg=W0212
      IntfCli._deltaCounterDir( mode )[ self.name ] = delta

      return counters

   def updateLastVniCounters( self, mode ):
      counters = self.vniCounters()
      lastCounters = self.storeLastVniCounters( mode, className=None,
                        vnis=[ vniVtiPair.vni for vniVtiPair in counters ] )
      for vniVtiPair, counter in six.iteritems( counters ):
         if counter and str( vniVtiPair.vni ) in lastCounters:
            lastCounters[ str( vniVtiPair.vni ) ].statistics = counter.statistics

   def ingVniCountersSupported( self ):
      return bridgingHwEnabled.vxlanVniIngressCountersEnabled

   def ingVniCountersRateSupported( self ):
      return bridgingHwEnabled.vxlanVniIngressCountersRateEnabled

   def egrVniCountersSupported( self ):
      return bridgingHwEnabled.vxlanVniEgressCountersEnabled

   def egrVniCountersRateSupported( self ):
      return bridgingHwEnabled.vxlanVniEgressCountersRateEnabled

   def perVniCountersSupported( self ):
      return self.ingVniCountersSupported() or self.egrVniCountersSupported()

   def perVniCountersRateSupported( self ):
      return self.ingVniCountersRateSupported() or self.egrVniCountersRateSupported()

   def vniCounters( self ):
      '''Returns the VxlanCounter objects for this interface
         Note that `self.getIntfVniCounterDir` returns all the counters across
         all VXLAN interfaces, and this function filters out the VNI counters for
         this VTI only'''
      countersDir = self.getIntfVniCounterDir()
      counters = {}
      for vniVtiPair, counter in six.iteritems( countersDir.counter ):
         if vniVtiPair.vti == self.status().intfId:
            counters[ vniVtiPair ] = counter
      return counters

   # This is the handler for VXLAN per-VNI counters under
   # `show interfaces counters vni`
   def updateVxlanInterfaceCountersModel( self, checkpoint=None, notFoundString=None,
                                          zeroOut=None, vni=None ):
      intfCountersModel = VxlanModel.VxlanInterfaceCounters(
         _name=self.status().intfId,
         _ingressCounters=self.ingVniCountersSupported(),
         _egressCounters=self.egrVniCountersSupported(),
         _vxlanVniCounters=self.perVniCountersSupported() )

      counters = self.vniCounters()

      def stat( attr, key, supported=True ):
         if not supported:
            return 0
         currentValue = getattr( counters[ key ].statistics, attr, None )
         if currentValue is None:
            return notFoundString
         checkpointValue = 0
         if checkpoint and key in checkpoint:
            checkpointValue = getattr( checkpoint[ key ].statistics, attr, None )
         return currentValue - checkpointValue

      for vniVtiPair in counters:
         if vni is not None and vniVtiPair.vni != int( vni ):
            continue
         vniCounters = VxlanModel.VxlanInterfaceVniCounters()
         if not zeroOut or ( vniVtiPair in zeroOut and not zeroOut[ vniVtiPair ] ):
            vniCounters.inOctets = stat( 'inOctets', vniVtiPair,
                  self.ingVniCountersSupported() )
            vniCounters.inPackets = stat( 'inUcastPkts', vniVtiPair,
                  self.ingVniCountersSupported() )
            vniCounters.outOctets = stat( 'outOctets', vniVtiPair,
                  self.egrVniCountersSupported() )
            vniCounters.outPackets = stat( 'outUcastPkts', vniVtiPair,
                  self.egrVniCountersSupported() )
         else:
            vniCounters.inOctets = 0
            vniCounters.inPackets = 0
            vniCounters.outOctets = 0
            vniCounters.outPackets = 0
         intfCountersModel.vniCounterIs( str( vniVtiPair.vni ), vniCounters )

      return intfCountersModel

   # This is the handler for VXLAN per-VNI counters under
   # `show interfaces counters vni rates`
   def updateVxlanInterfaceCountersRateModel( self, vni=None ):
      if self.vniCounters() is None:
         return None

      counters = self.vniCounters()
      intfCountersRateModel = VxlanModel.VxlanInterfaceCountersRate(
            _name=self.name )

      # pylint: disable-msg=unused-variable
      def ratePct( bps, pps ):
         '''
         Uses the link bandwidth to determine the % of theoretical bitrate
         acheived including the 12 byte inter-frame gap and the 8 byte preamble

         This returns None for now since VXLAN interfaces are not physical interfaces
         and self.bandwidth() returns 0. This also means that rate percent will
         always be N/A for a VXLAN interface.

         Add rate percent calculation in the future here when self.bandwidth()
         returns non-zero for non-physical interfaces
         '''
         return None

      def rateValue( attr, key ):
         counter = counters[ key ]
         currentValue = getattr( counter.rates, attr )
         return currentValue

      for vniVtiPair in counters:
         if vni is not None and vniVtiPair.vni != int( vni ):
            continue
         vniCountersRate = VxlanModel.VxlanInterfaceVniCountersRate()
         vniCountersRate.description = self.description()
         # pylint: disable=round-builtin
         vniCountersRate.interval = int( round(
               IntfCli.getActualLoadIntervalValue( self.config().loadInterval ) ) )
         vniCountersRate.inBpsRate = rateValue( 'inBitsRate', vniVtiPair )
         vniCountersRate.inPpsRate = rateValue( 'inPktsRate', vniVtiPair )
         # replace with the following when self.bandwidth() returns non-zero
         # ratePct( vniCountersRate.inBpsRate, vniCountersRate.inPpsRate )
         vniCountersRate.inPktsRate = None
         vniCountersRate.outBpsRate = rateValue( 'outBitsRate', vniVtiPair )
         vniCountersRate.outPpsRate = rateValue( 'outPktsRate', vniVtiPair )
         # replace with the following when self.bandwidth() returns non-zero
         # ratePct( vniCountersRate.outBpsRate, vniCountersRate.outPpsRate )
         vniCountersRate.outPktsRate = None
         intfCountersRateModel.vniCounterRateIs(
            str( vniVtiPair.vni ), vniCountersRate )

      return intfCountersRateModel

   def routingSupported( self ):
      return False

   def routingCurrentlySupported( self ):
      return False

   def setDefault( self ):
      IntfCli.VxlanVirtualIntf.setDefault( self )
      vtiConfig = getVtiConfig( self.name )
      vxlanConfig = getVxlanConfig( self.name )
      vtiConfig.mcastGrpDecap.clear()
      vtiConfig.vtepAddrMask = Tac.newInstance( "Arnet::IpAddr", 0xFFFFFFFF )
      vtiConfig.srcIpIntf = ''
      vtiConfig.vlanToVniMap.clear()
      vtiConfig.ttl = vtiConfig.defaultTtl
      vtiConfig.udpPort = vtiConfig.vxlanWellKnownPort
      vtiConfig.secUdpPort = vtiConfig.vxlanSecWellKnownPort
      vtiConfig.srcPortRange = Tac.Value( "Vxlan::VxlanSrcPortRange" )
      vtiConfig.secSrcPortRange = VxlanSrcPortRange( 0, 1 )
      vtiConfig.controllerClientMode = False
      vtiConfig.use32BitVni = False
      vtiConfig.vtepSetForSourcePruning.clear()
      vtiConfig.vtepSourcePruningAll = False
      vtiConfig.floodLearnedAll = False
      vtiConfig.mcastRouting = False
      vtiConfig.vxlanEncap = bridgingHwCapabilities.vxlanDefaultEncap
      vtiConfig.vxlanEncapConfigured = False
      vtiConfig.cliVccImportVlans = '1-4094'
      vtiConfig.cliVccImportVlans = '1-4094'
      vtiConfig.vccArpProxy = False
      vtiConfig.vccNdProxy = False
      adjustVxlanControllerServiceEnable()

      vxlanConfig.floodVtepList.clear()
      vxlanConfig.floodVtepList6.clear()

      vxlanConfig.vlanToVtepList.clear()
      # Get a default empty learn list
      vxlanConfig.learnFrom = 'learnFromDefault'
      vxlanConfig.vlanToLearnRestrict.clear()
      vtiConfig.vrfToVniMap.clear()
      vtiConfig.alternateDecapVniToVrfMap.clear()
      vtiConfig.vrfToIpAddr.clear()
      vtiConfig.vrfToIp6Addr.clear()

      vxlanConfig.vniToIpAclMap.clear()
      vxlanConfig.vniToIpv6AclMap.clear()
      vxlanConfig.vniToEgressIpAclMap.clear()
      vxlanConfig.vniToEgressIpv6AclMap.clear()
      vtiConfig.arpLocalAddress = False
      vtiConfig.mlagSrcIpIntf = ''
      vtiConfig.varpVtepSrcIpIntf = ''
      vtiConfig.vtepToSecProfile.clear()
      vtiConfig.bfdParams = Tac.Value( 'Vxlan::BfdParamsConfig' )
      vtiConfig.bfdPrefixList = ''
      vtiConfig.decapFilterMode = 'filterEnabled'
      vtiConfig.decapFilterIntf.clear()
      vtiConfig.decapFilterNonDefaultVrf = False
      vtiConfig.decapRouteAll = False
      vxlanConfig.vniQosInfo.clear()

      vtiConfig.mlagSharedRouterMacAddr = ethAddrZero
      vtiConfig.mlagSharedRouterMacConfig = 'disabled'
      vtiConfig.tcpMssV4 = 0
      vtiConfig.tcpMssV6 = 0
      pbrVniIntfConfig.vni.clear()

      vtiConfig.vniNatProfile.clear()
      vtiConfig.virtualArpOptimizedFloodingEnabled = True
      if icmpHelperInputConfigClientName in icmpHelperInputConfigDir.entityPtr:
         icmpHelperInputConfigDir.deleteEntity( icmpHelperInputConfigClientName )

      # Only reset these params if the interface being defaulted is Vxlan1
      # The encapsulated arp/nd snooping and vtep-to-vtep flags are global, so
      # only reset them if the interface being defaulted is Vxlan1, not any other
      # VTI
      if self.name == 'Vxlan1':
         vxlanCliClientConfig.vxlanEncapsulatedArpSnooping = \
               TristateBoolean.valueInvalid()
         vxlanCliClientConfig.vxlanEncapsulatedNdSnooping = \
               TristateBoolean.valueInvalid()
         vxlanCliClientConfig.vxlanVtepToVtepBridgingEnabled = \
               TristateBoolean.valueInvalid()

#----------------------------------------------------------------------------
# The rule for matching Vxlan Tunnel interface names. When this pattern matches,
# it returns an instance of the VxlanIntf class.
#
# This rule gets added to the Intf.rule when this class is registered with
# the Intf class by calling Intf.addPhysicalIntfType.
#----------------------------------------------------------------------------
VxlanIntf.matcher = VirtualIntfRule.VirtualIntfMatcher(
      'Vxlan', None, None,
      rangeFunc=vxlanRangeFn,
      value=lambda mode, intf: VxlanIntf( intf, mode ),
      helpdesc="VXLAN Tunnel Interface", guard=vxlanSupportedGuard )

class VxlanIntfModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, VxlanIntf )

#-------------------------------------------------------------------------------
# Register the VxlanIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( VxlanIntf, VxlanAutoIntfType )

IntfRange.registerIntfTypeGuard( VxlanAutoIntfType, vxlanSupportedGuard )

# utility functions
def getVtiConfig( intfName, create=True ):
   vtiIntfId = vtiNameToId.get( intfName )
   vtiConfig = vtiConfigDir.vtiConfig.get( vtiIntfId )
   if not vtiConfig and create:
      vtiConfig = vtiConfigDir.vtiConfig.newMember( vtiIntfId )
   return vtiConfig

def adjustVxlanControllerServiceEnable():
   """ Synchronizes the CVX controller 'enabled' config attribute with the
   per-vti equivalent, 'controllerClientMode'. This is called whenever vti
   config's controllerClientMode is changed or a vti config is deleted. """

   vxlanControllerService = serviceConfigDir.service[ "Vxlan" ]
   for vtiConfig in vtiConfigDir.vtiConfig.values():
      if vtiConfig.controllerClientMode:
         vxlanControllerService.enabled = True
         return
   vxlanControllerService.enabled = False

def getVtiStatus( intfName ):
   vtiIntfId = vtiNameToId.get( intfName )
   vtiStatus = vtiStatusDir.vtiStatus.get( vtiIntfId )
   return vtiStatus

def getVxlanConfig( intfName ):
   vtiIntfId = vtiNameToId.get( intfName )
   vxlanConfig = vxlanConfigDir.vxlanConfig.get( vtiIntfId )
   return vxlanConfig

def getVxlanCounterConfig( intfName ):
   vtiIntfId = vtiNameToId.get( intfName )
   vxlanCounterConfig = vxlanCounterConfigDir.vxlanCounterConfig.get( vtiIntfId )
   return vxlanCounterConfig

def getVxlanStatus( intfName ):
   vtiIntfId = vtiNameToId.get( intfName )
   vxlanStatus = vxlanStatusDir.vxlanStatus.get( vtiIntfId )
   return vxlanStatus

def isVtiMissing( intfName ):
   vtiIntfId = vtiNameToId.get( intfName )
   return vtiConfigDir.vtiConfig.get( vtiIntfId ) is None

bfdNode = CliCommand.Node( matcher=CliToken.Bfd.matcherBfd,
      guard=isVxlan1InterfaceGuard )
vxlanNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'vxlan',
         helpdesc='Configure VXLAN parameters' ),
      guard=vxlanSupportedGuard )
controllerClientNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'controller-client',
         helpdesc='Set VXLAN controller client mode parameters' ),
      guard=vxlanControllerClientSupportedGuard )
decapFilterNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'filter',
         helpdesc='Configure VXLAN decapsulation filter' ),
      guard=vxlanDecapFilterSupportedGuard )
decapFilterNonDefaultVrfNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'non-default',
         helpdesc='Filter on non-default VRF' ),
      guard=vxlanDecapFilterNonDefaultVrfSupportedGuard )
decapRouteNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'route',
         helpdesc='Configure VXLAN decapsulation routing action' ),
      guard=vxlanDecapRouteAllSupportedGuard )

def sipValidationNode( vxlanMode=False ):
   if vxlanMode:
      helpdesc = 'Configure VTEPs for source IP validation'
      guard = vxlanTunnelSipValidationSupportedGuard
   else:
      helpdesc = 'Enable source IP validation for VTEP'
      guard = tunnelSipValidationSupportedGuard
   return CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'source-vtep-validation',
         helpdesc=helpdesc ), maxMatches=1, guard=guard )

# Nodes only available on Vxlan1
vlanVxlan1Node = CliCommand.Node( matcher=matcherVlan,
                                  guard=isVxlan1InterfaceGuard )
matcherUdpPort = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'udp-port',
         helpdesc='VXLAN UDP port' ),
      guard=vxlanUdpPortSupportedGuard )
matcherFlood = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'flood',
         helpdesc='VXLAN flooding behavior' ),
      guard=isVxlan1InterfaceGuard )

# Keep vlanRange so that other packages can import it
# Use `VniDottedDoParse` to return the values in the original order,
# as opposed to sorting and merging them.
vlanMultiMatcher = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( 1, 4094 ),
      noSingletons=False,
      DoParseClass=MultiRangeRule.DoParseNoMerge,
      helpdesc='VLAN ID(s)' )
vtepMatcherForConfig = CliMatcher.KeywordMatcher( 'vtep',
                                   helpdesc='Configure VXLAN Tunnel End Points' )
addRemoveMatcher = CliMatcher.EnumMatcher( { 'add': 'Add a VTEP',
                                             'remove': 'Remove a VTEP' } )
vxlanIntfType = [ VxlanAutoIntfType ]
ipGenPrefixMatcher = IpGenAddrMatcher.IpGenPrefixMatcher(
      'IP address (or prefix) of remote VTEP',
      helpdesc4='IPv4 address (or prefix) of remote VTEP',
      helpdesc6='IPv6 address (or prefix) of remote VTEP', allowAddr=True )

def setVxlanShutdown( modeList ):
   # We can only create a single Vxlan interface at the moment
   m = re.match( r'Vxlan(\d+)$', modeList[ 0 ].intf.name )
   if m:
      vtiConfig = getVtiConfig( modeList[ 0 ].intf.name )
      serviceConfigDir.service[ "Vxlan" ].enabled = \
         vtiConfig.controllerClientMode and vtiConfig.adminEnabled

IntfCli.shutdownIfHook.addExtension( setVxlanShutdown )

# ----------------------------------------------------------------------------------
# 'tcp mss ceiling ([ipv4 <64-65495>] [ipv6 <64-65475>]) [ingress | egress ]' .
#
# enable tcp mss cli for vxlan
# ----------------------------------------------------------------------------------
class CfgVxlanTcpMssCeilingCmd( CfgTcpMssCeilingCmdBase ):
   data = {
      'tcp': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'tcp', helpdesc='TCP' ),
         guard=vxlanTcpMssPlatformGuard ),
   }
   data.update( CfgTcpMssCeilingCmdBase._baseData )
   data[ 'ipv6' ] = CliCommand.singleKeyword(
      'ipv6',
      helpdesc='Internet Protocol version 6',
      guard=vxlanIpv6TcpMssPlatformGuard )

   @staticmethod
   def _commonVxlanHandler( mode, args, vtiConfig ):
      if 'ingress' not in args:
         tcpMssV4Value = args.get( 'MSS4', [ 0 ] )[ 0 ]
         tcpMssV6Value = args.get( 'MSS6', [ 0 ] )[ 0 ]
         vtiConfig.tcpMssV4 = tcpMssV4Value
         vtiConfig.tcpMssV6 = tcpMssV6Value

   @staticmethod
   def _commonVxlanNoOrDefaultHandler( mode, vtiConfig ):
      vtiConfig.tcpMssV4 = 0
      vtiConfig.tcpMssV6 = 0

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      CfgVxlanTcpMssCeilingCmd._commonVxlanHandler( mode, args, vtiConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      CfgVxlanTcpMssCeilingCmd._commonVxlanNoOrDefaultHandler( mode, vtiConfig )

if Toggles.VxlanToggleLib.toggleVxlanTcpMssClampingSupportedEnabled():
   VxlanIntfModelet.addCommandClass( CfgVxlanTcpMssCeilingCmd )
#----------------------------------------------------------------------------------
# "[no|default] bfd vtep evpn interval INTERVAL min-rx MIN_RX multiplier MULTIPLIER
#
# sets the bfd interval for vxlan in non-vcs mode
#----------------------------------------------------------------------------------
def setVxlanIntfBfdParams( mode, args ):
   minTx = args[ 'INTERVAL' ]
   minRx = args[ 'MIN_RX' ]
   mult = args[ 'MULTIPLIER' ]
   vtiConfig = getVtiConfig( mode.intf.name )
   bfdIntervalParams = Tac.Value( 'Bfd::BfdIntervalConfig',
                                  minTx, minRx, mult )
   vtiConfig.bfdParams = Tac.Value( 'Vxlan::BfdParamsConfig',
                                    True, bfdIntervalParams )

def resetVxlanIntfBfdParams( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   bfdIntervalParams = Tac.Value( 'Bfd::BfdIntervalConfig',
                                  BfdInterval.defval,
                                  BfdInterval.defval,
                                  BfdMultiplier.defval )
   vtiConfig.bfdParams = Tac.Value( 'Vxlan::BfdParamsConfig',
                                    False, bfdIntervalParams )

class VxlanBfdIntfParamsCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd vtep evpn interval INTERVAL min-rx MIN_RX multiplier MULTIPLIER'
   noOrDefaultSyntax = 'bfd vtep evpn interval ...'
   data = {
      'bfd': bfdNode,
      'vtep': CliToken.Vxlan.bfdVtepMatcher,
      'evpn': CliToken.Vxlan.evpnMatcher,
      'interval': CliToken.Bfd.matcherInterval,
      'INTERVAL': CliToken.Bfd.matcherTxRxIntervalMs,
      'min-rx': CliToken.Bfd.matcherMinRx,
      'MIN_RX': CliToken.Bfd.matcherTxRxIntervalMs,
      'multiplier': CliToken.Bfd.matcherMultiplier,
      'MULTIPLIER': CliToken.Bfd.matcherMultiplierVal,
   }
   handler = setVxlanIntfBfdParams
   noOrDefaultHandler = resetVxlanIntfBfdParams

VxlanIntfModelet.addCommandClass( VxlanBfdIntfParamsCmd )

class VxlanBfdIntfFilterCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd vtep evpn prefix-list PREFIX_LIST'
   noOrDefaultSyntax = 'bfd vtep evpn prefix-list ...'
   data = {
      'bfd': bfdNode,
      'vtep': CliToken.Vxlan.bfdVtepMatcher,
      'evpn': CliToken.Vxlan.evpnMatcher,
      'prefix-list': 'Prefix list reference',
      'PREFIX_LIST': RouteMapCli.prefixListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.bfdPrefixList = args.get( 'PREFIX_LIST', '' )

   noOrDefaultHandler = handler

VxlanIntfModelet.addCommandClass( VxlanBfdIntfFilterCmd )

def _verifyMcastGrp( mode, mcastGrps ):
   for mcastGrp in mcastGrps:
      mcastAddr = Arnet.IpAddress( mcastGrp )
      validateMcastAddrResult = IpAddrMatcher.validateMulticastIpAddr(
                str( mcastAddr ), allowReserved=False )
      if validateMcastAddrResult:
         mode.addError( validateMcastAddrResult )
         return False
   return True

def addMcastGrpDecap( mode, args ):
   mcastGrpDecapList = args[ 'DECAP' ]
   if not _verifyMcastGrp( mode, mcastGrpDecapList ):
      return

   vtiConfig = getVtiConfig( mode.intf.name )
   for grp in mcastGrpDecapList:
      if grp not in vtiConfig.mcastGrpDecap:
         vtiConfig.mcastGrpDecap[ grp ] = True

def removeMcastGrpDecap( mode, args ):
   mcastGrpDecapList = args.get( 'DECAP' )
   vtiConfig = getVtiConfig( mode.intf.name )

   if mcastGrpDecapList:
      for grp in mcastGrpDecapList:
         del vtiConfig.mcastGrpDecap[ grp ]
   else:
      # Remove all if list is empty
      vtiConfig.mcastGrpDecap.clear()

class VxlanMulticastGroupDecapCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan multicast-group decap { DECAP }'
   noOrDefaultSyntax = 'vxlan multicast-group decap [ { DECAP } ]'
   data = {
      'vxlan': vxlanNode,
      'multicast-group': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'multicast-group',
            helpdesc='Flood multicast group' ),
         guard=vxlanMcastDecapOnlySupportedGuard ),
      'decap': 'Decap only',
      'DECAP': IpAddrMatcher.IpAddrMatcher( helpdesc='IP multicast group address' ),
   }
   handler = addMcastGrpDecap
   noOrDefaultHandler = removeMcastGrpDecap

VxlanIntfModelet.addCommandClass( VxlanMulticastGroupDecapCmd )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan multicast routing ipv4" command, in "config-vxlan-if"
#
# enables multicast routing on the vxlan interface
# turns default bridging on only for mcast packets, allows routing on the overlay
#-------------------------------------------------------------------------------
class VxlanMcastRoutingIpCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan multicast routing ipv4'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'multicast': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'multicast',
                                              helpdesc='Multicast vxlan commands' ),
                                    guard=isVxlan1InterfaceGuard ),
      'routing': 'Enable multicast routing',
      'ipv4': 'IPv4 related',
   }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.mcastRouting = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.mcastRouting = False

VxlanIntfModelet.addCommandClass( VxlanMcastRoutingIpCmd )

#-------------------------------------------------------------------------------
# "config-vxlan-if" vxlan decapsulation filter
#                   ( disabled | ( interface multiple-vrf disabled [ { INTF } ] ) |
#                   ( vrf default { INTF } ) )
#
# sets vxlan decap filter mode / parameters
#-------------------------------------------------------------------------------
def reconcileDecapFilterIntfList( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   for intf in vtiConfig.decapFilterIntf:
      if str( intf ) not in args[ 'INTF' ]:
         vtiConfig.decapFilterIntf.remove( intf )
   for intf in args[ 'INTF' ]:
      vtiConfig.decapFilterIntf.add( str( intf ) )

def vxlanDecapFilter( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   if 'vrf' in args and 'default' in args:
      vtiConfig.decapFilterMode = 'filterEnabledIntf'
      reconcileDecapFilterIntfList( mode, args )
   # The presence of 'interface' kw determines whether filter is disabled or relaxed
   elif 'interface' not in args:
      vtiConfig.decapFilterMode = 'filterDisabled'
      vtiConfig.decapFilterIntf.clear()
   else:
      # Filter is either relaxed for all intfs or just intfs in set
      if 'INTF' not in args:
         vtiConfig.decapFilterMode = 'filterRelaxedAll'
         vtiConfig.decapFilterIntf.clear()
      else:
         vtiConfig.decapFilterMode = 'filterRelaxedIntf'
         reconcileDecapFilterIntfList( mode, args )

def noVxlanDecapFilter( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.decapFilterMode = 'filterEnabled'
   vtiConfig.decapFilterIntf.clear()

class VxlanDecapFilterCmd( CliCommand.CliCommandClass ):
   syntax = ( 'vxlan decapsulation filter '
         '( disabled | ( interface multiple-vrf disabled [ { INTF } ] ) | '
         '( vrf default { INTF } ) )' )
   noOrDefaultSyntax = ( 'vxlan decapsulation filter ...' )
   data = {
         'vxlan': vxlanNode,
         'decapsulation': 'Set VXLAN decapsulation parameters',
         'filter': decapFilterNode,
         'disabled': 'Disable VXLAN decapsulation filter',
         'interface': 'Interface filter parameters',
         'multiple-vrf': 'Configure interfaces in multiple VRFs',
         'INTF': EthPhyIntf.ethMatcher,
         'vrf': matcherVrf,
         'default': 'Filter on default VRF',
   }
   handler = vxlanDecapFilter
   noOrDefaultHandler = noVxlanDecapFilter

VxlanIntfModelet.addCommandClass( VxlanDecapFilterCmd )

def vxlanSourceIpRewrite( mode, args ):
   vni = args[ 'VNI' ]
   vniNum = VniFormat( vni ).toNum()
   remoteVtep = args[ 'VTEP' ]
   rewriteAddr = args[ 'ADDR' ]

   vniAndRemoteVtep = Tac.Value( 'Vxlan::VniAndRemoteVtep', vniNum, remoteVtep )
   vxlanConfig = getVxlanConfig( mode.intf.name )
   vxlanConfig.vniVtepSrcIp[ vniAndRemoteVtep ] = rewriteAddr

def noVxlanSourceIpRewrite( mode, args ):
   vni = args[ 'VNI' ]
   vniNum = VniFormat( vni ).toNum()
   remoteVtep = args[ 'VTEP' ]
   vniAndRemoteVtep = Tac.Value( 'Vxlan::VniAndRemoteVtep', vniNum, remoteVtep )
   vxlanConfig = getVxlanConfig( mode.intf.name )
   if vniAndRemoteVtep in vxlanConfig.vniVtepSrcIp:
      del vxlanConfig.vniVtepSrcIp[ vniAndRemoteVtep ]

#-------------------------------------------------------------------------------
# "config-vxlan-if" vxlan vni VNI vtep VTEP encapsulation ipv4 source address ADDR
# aid/13978 has more details
#-------------------------------------------------------------------------------
class VxlanSourceIpRewrite( CliCommand.CliCommandClass ):
   syntax = ( 'vxlan vni VNI vtep VTEP encapsulation ipv4 source address ADDR' )
   noOrDefaultSyntax = ( 'vxlan vni VNI vtep VTEP ...' )
   data = {
      'vxlan': vxlanNode,
      'vni': vniMatcherForConfig,
      'VNI': vniMatcher,
      'vtep': CliCommand.Node( matcher=vtepMatcherForConfig,
         guard=vxlanSourceIpRewriteSupportedGuard ),
      'VTEP': IpAddrMatcher.IpAddrMatcher( 'IP address of remote VTEP' ),
      'encapsulation': 'Set VXLAN encapsulation parameters',
      'ipv4': 'VXLAN IPv4 underlay',
      'source': 'source',
      'address': 'IP address',
      'ADDR': IpAddrMatcher.IpAddrMatcher( 'Source IP address of underlay' )
   }
   handler = vxlanSourceIpRewrite
   noOrDefaultHandler = noVxlanSourceIpRewrite

VxlanIntfModelet.addCommandClass( VxlanSourceIpRewrite )

def showVxlanSourceIpRewrite( mode, args ):
   remoteVtep = args.get( 'VTEP', 'all' )
   model = VxlanModel.VxlanRewriteInfo()
   vxlanConfig = getVxlanConfig( 'Vxlan1' )
   if vxlanConfig:
      for ( vniAndRemoteVtep, srcIp ) in six.iteritems( vxlanConfig.vniVtepSrcIp ):
         if remoteVtep == 'all' or remoteVtep == vniAndRemoteVtep.remoteVtep:
            dataList = VxlanModel.VxlanEncapSourceIpRewriteModelList()
            if vniAndRemoteVtep.remoteVtep in model.remoteVteps:
               dataList = model.remoteVteps[ vniAndRemoteVtep.remoteVtep ]
            dataModel = VxlanModel.VxlanEncapSourceIpRewriteModel()
            dataModel.vni = vniAndRemoteVtep.vni
            dataModel.sourceIp = srcIp
            dataList.rewriteInfo.append( dataModel )
            model.remoteVteps[ vniAndRemoteVtep.remoteVtep ] = dataList
   return model

#-------------------------------------------------------------------------------
# "config-vxlan-if" vxlan decapsulation filter vrf non-default ipv4
#
# sets vxlan decap to only be on default VRF
#-------------------------------------------------------------------------------

def vxlanDecapFilterNonDefaultVrf( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.decapFilterNonDefaultVrf = True

def noVxlanDecapFilterNonDefaultVrf( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.decapFilterNonDefaultVrf = False

class VxlanDecapFilterNonDefaultVrfCmd( CliCommand.CliCommandClass ):
   syntax = ( 'vxlan decapsulation filter vrf non-default ipv4' )
   noOrDefaultSyntax = syntax
   data = {
         'vxlan': vxlanNode,
         'decapsulation': 'Set VXLAN decapsulation parameters',
         'filter': decapFilterNode,
         'vrf': matcherVrf,
         'non-default': decapFilterNonDefaultVrfNode,
         'ipv4': 'VXLAN IPv4 underlay',
   }
   handler = vxlanDecapFilterNonDefaultVrf
   noOrDefaultHandler = noVxlanDecapFilterNonDefaultVrf

VxlanIntfModelet.addCommandClass( VxlanDecapFilterNonDefaultVrfCmd )

# -------------------------------------------------------------------------------
# "config-vxlan-if" vxlan decapsulation route all
#
# enables routing all vxlan decapsulated IP packets
# -------------------------------------------------------------------------------
def vxlanDecapRouteAll( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.decapRouteAll = True

def noVxlanDecapRouteAll( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name, create=False )
   if vtiConfig:
      vtiConfig.decapRouteAll = False

class VxlanDecapRouteAllCmd( CliCommand.CliCommandClass ):
   syntax = ( 'vxlan decapsulation route all' )
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'decapsulation': 'Set VXLAN decapsulation parameters',
      'route': decapRouteNode,
      'all': 'All IP packets',
      }
   handler = vxlanDecapRouteAll
   noOrDefaultHandler = noVxlanDecapRouteAll

VxlanIntfModelet.addCommandClass( VxlanDecapRouteAllCmd )

#-------------------------------------------------------------------------------
# "config-vxlan-if" vxlan [ mlag ] source-interface ( INTERFACE | ( Loopback NUM ) )
#
# sets the source interface for vti
# sets the ip address of the MLAG VTEP IP for Multi-VTEP MLAG
#-------------------------------------------------------------------------------
class VxlanSourceInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan [ mlag ] source-interface ( LO_INTERFACE | DPS_INTERFACE )'
   noOrDefaultSyntax = 'vxlan [ mlag ] source-interface ...'
   data = {
      'vxlan': vxlanNode,
      'mlag': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'mlag',
          helpdesc='Configure shared VTEP IP to use when VTEP IPs are unique.' ),
         guard=vxlanMultiVtepMlagGuard ),
      'source-interface': 'VXLAN source interface',
      'LO_INTERFACE': LoopbackIntfCli.LoopbackIntf.matcher,
      'DPS_INTERFACE': DpsIntfCli.DpsIntf.matcher,
   }

   @staticmethod
   def handler( mode, args ):
      srcIntf = None
      if 'LO_INTERFACE' in args:
         srcIntf = args[ 'LO_INTERFACE' ]
      else:
         srcIntf = args[ 'DPS_INTERFACE' ]
      vtiConfig = getVtiConfig( mode.intf.name )
      if 'mlag' not in args:
         for conf in vtiConfigDir.vtiConfig.values():
            if ( conf.srcIpIntf == srcIntf.name and
                 conf.intfId != mode.intf.name ):
               mode.addError( "Interface %s is already assigned to %s" %
                              ( srcIntf.name, conf.intfId ) )
               return
         vtiConfig.srcIpIntf = srcIntf.name
      else:
         vtiConfig.mlagSrcIpIntf = srcIntf.name

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      if 'mlag' not in args:
         vtiConfig.srcIpIntf = ''
      else:
         vtiConfig.mlagSrcIpIntf = ''

VxlanIntfModelet.addCommandClass( VxlanSourceInterfaceCmd )

#-------------------------------------------------------------------------------
# "config-vxlan-if" vxlan virtual-vtep local-interface ( INTERFACE | (Loopback NUM) )
#
# sets the source interface for virtual vtep
#-------------------------------------------------------------------------------
class VxlanVarpVtepLocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan virtual-vtep local-interface INTERFACE'
   noOrDefaultSyntax = 'vxlan virtual-vtep local-interface ...'
   data = {
      'vxlan': vxlanNode,
      'virtual-vtep': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'virtual-vtep',
            helpdesc='Configure virtual VTEP IP address.' ),
         guard=vxlanVirtualVtepCmdSupportedGuard ),
      'local-interface': 'VXLAN virtual VTEP source interface',
      'INTERFACE': LoopbackIntfCli.LoopbackIntf.matcher,
   }

   @staticmethod
   def handler( mode, args ):
      srcIntf = args[ 'INTERFACE' ]
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.varpVtepSrcIpIntf = srcIntf.name

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.varpVtepSrcIpIntf = ''

VxlanIntfModelet.addCommandClass( VxlanVarpVtepLocalInterfaceCmd )

#-------------------------------------------------------------------------------
# vxlan flood vtep <vteps> - replaces the entire default vtep list
# vxlan flood vtep add <vteps> - add to default vtep list
# vxlan flood vtep remove <vteps> - remove from default vtep list
# no vxlan flood vtep - remove all vtep lists
# vxlan vlan <vlan> flood vtep <vteps> - replaces the entire vtep list
# vxlan vlan <vlan> flood vtep add <vteps> - add to existing list
# vxlan vlan <vlan> flood vtep remove <vteps> - remove from existing list
# no vxlan vlan <vlan> flood vtep - remove vtep list for a vlan
# no vxlan vlan flood vtep - remove all vtep lists
# show vxlan flood vtep [vlan <vlanId>]
#-------------------------------------------------------------------------------
def _verifyVtepAddrs( mode, vteps ):
   for vtep in vteps:
      ip = Tac.Value( 'Arnet::IpGenAddr', str( vtep ) )
      if ip.isMulticast or ip.isLoopback or ip.isLinkLocal or ip.isThisNetwork:
         mode.addError( "invalid VTEP address: %s" % vtep )
         return False
   return True

def convertVtepsToIpGenAddr( vteps ):
   if vteps and not isinstance( vteps[ 0 ], Tac.Type( "Arnet::IpGenAddr" ) ):
      vteps = [ Arnet.IpGenAddr( str( vtep ) ) for vtep in vteps ]
   return vteps

def setDefaultFloodVtep( mode, vteps ):
   '''Remove the old default vtep list and create a new one'''
   vxlanConfig = getVxlanConfig( mode.intf.name )

   if not _verifyVtepAddrs( mode, vteps ):
      return
   vteps = convertVtepsToIpGenAddr( vteps )
   v4Vteps = set( v for v in vteps if v.af == 'ipv4' )
   v6Vteps = set( v for v in vteps if v.af == 'ipv6' )
   if v4Vteps:
      for v in vxlanConfig.floodVtepList:
         if Arnet.IpGenAddr( v ) not in v4Vteps:
            del vxlanConfig.floodVtepList[ v ]
   else:
      vxlanConfig.floodVtepList.clear()

   if v6Vteps:
      for v in vxlanConfig.floodVtepList6:
         if Arnet.IpGenAddr( str( v ) ) not in v6Vteps:
            del vxlanConfig.floodVtepList6[ v ]
   else:
      vxlanConfig.floodVtepList6.clear()
   for v in vteps:
      if v.af == 'ipv4':
         if v.v4Addr not in vxlanConfig.floodVtepList:
            vxlanConfig.floodVtepList[ v.v4Addr ] = True
      elif v.v6Addr not in vxlanConfig.floodVtepList6:
         vxlanConfig.floodVtepList6[ v.v6Addr ] = True

def noSetDefaultFloodVtep( mode ):
   '''Remove the default flood vtep list'''
   setDefaultFloodVtep( mode, vteps=[] )

def addRemoveDefaultFloodVtep( mode, vteps, addOrRemove ):
   '''Add or remove vteps from the default floodVtepList'''
   vxlanConfig = getVxlanConfig( mode.intf.name )
   vteps = convertVtepsToIpGenAddr( vteps )

   s = set( Arnet.IpGenAddr( ip )
            for ip in chain( vxlanConfig.floodVtepList,
                             ( str( vtep6 ) for vtep6 in vxlanConfig.floodVtepList6 )
            ) )
   t = set( vteps )
   if addOrRemove == 'add':
      s = s.union( t )
   else:
      s = s.difference( t )
   vteps = list( s )
   setDefaultFloodVtep( mode, vteps )

def setFloodVtep( mode, vSet, vteps ):
   '''Remove the old list and create a new one'''
   if len( vSet ) > 1:
      vxlanConfigDir.vlanFloodVtepRangeSyntax = True
   vxlanConfig = getVxlanConfig( mode.intf.name )

   if not _verifyVtepAddrs( mode, vteps ):
      return

   vteps = list( vteps )
   vteps = convertVtepsToIpGenAddr( vteps )
   # sync the config with the provided vtep list
   for vlan in vSet:
      if vteps:
         if not vlan in vxlanConfig.vlanToVtepList:
            vxlanConfig.vlanToVtepList.newMember( vlan )
         for v in vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr:
            if Arnet.IpGenAddr( v ) not in vteps:
               del vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr[ v ]
         for v in vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr6:
            if Arnet.IpGenAddr( str( v ) ) not in vteps:
               del vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr6[ v ]
         for v in vteps:
            if v.af == 'ipv4':
               if v.v4Addr not in vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr:
                  vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr[ v.v4Addr
                        ] = True
            elif v.v6Addr not in vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr6:
               vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr6[ v.v6Addr ] = True
      else:
         if vlan in vxlanConfig.vlanToVtepList:
            del vxlanConfig.vlanToVtepList[ vlan ]

def addRemoveFloodVtep( mode, vSet, vteps, addOrRemove ):
   '''Add or remove vteps from the vlanToVtepList'''
   if len( vSet ) > 1:
      vxlanConfigDir.vlanFloodVtepRangeSyntax = True
   vxlanConfig = getVxlanConfig( mode.intf.name )
   vtiConfig = getVtiConfig( mode.intf.name )
   vteps = convertVtepsToIpGenAddr( vteps )
   invalidVlans = []
   for vlan in vSet:
      vs = VlanCli.VlanSet( mode=mode, vlanSetString='', vlanIds=[ vlan ] )
      # allow 'add/remove' in per vlan configuration
      if vlan in vxlanConfig.vlanToVtepList:
         s = set( [ Arnet.IpGenAddr( ip ) for ip in
                       vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr ] +
                  [ Arnet.IpGenAddr( str( ip ) ) for ip in
                       vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr6 ] )
         t = set( vteps )
         if addOrRemove == 'add':
            s = s.union( t )
         else:
            s = s.difference( t )
         setFloodVtep( mode, vs, s )
         continue

      # allow 'add' when default flood list is empty
      if not vxlanConfig.floodVtepList:
         if addOrRemove == 'add':
            setFloodVtep( mode, vs, vteps )
         continue

      # vlan without vni is nop and will not show error message
      if vlan in vtiConfig.vlanToVniMap:
         invalidVlans.append( vlan )

   if invalidVlans != []:
      invalidVlansStr = MultiRangeRule.multiRangeToCanonicalString( invalidVlans )
      if addOrRemove == 'add':
         mode.addError( "VLANs %s are using the default VTEP flood list. "
               "These VLANs do not have their own configured VTEP flood lists to "
               "add the specified VTEPs to." % invalidVlansStr )
      else:
         mode.addError( "VLANs %s are using the default VTEP flood list. "
               "These VLANs do not have their own configured VTEP flood lists to "
               "remove the specified VTEPs from." % invalidVlansStr )

def noSetFloodVtep( mode, vSet ):
   if len( vSet ) > 1:
      vxlanConfigDir.vlanFloodVtepRangeSyntax = True
   vxlanConfig = getVxlanConfig( mode.intf.name )
   for vlan in vSet:
      if vlan in vxlanConfig.vlanToVtepList:
         del vxlanConfig.vlanToVtepList[ vlan ]

def clearFloodVtep( mode, args ):
   vxlanConfig = getVxlanConfig( mode.intf.name )
   vxlanConfig.vlanToVtepList.clear()

def showFloodVtep( mode, args ):
   vSet = args.get( 'VLANS' )
   vxlanConfig = getVxlanConfig( 'Vxlan1' )
   vtiStatus = getVtiStatus( 'Vxlan1' )

   vxlanFloodVtep = Tac.newInstance( "VxlanCliCapi::VxlanFloodVtep",
                                     l2RibFloodSet, vxlanConfig )

   if vtiStatus:
      if ( vtiStatus.localVtepAddr != '0.0.0.0' and
           vtiStatus.vxlanEncap != vxlanEncapType.vxlanEncapIp6 ):
         vxlanFloodVtep.afFilter.add( AFType.ipv4 )
      if ( not vtiStatus.localVtepAddr6.isZero and
           vtiStatus.vxlanEncap != vxlanEncapType.vxlanEncapIp4 ):
         vxlanFloodVtep.afFilter.add( AFType.ipv6 )

   if vSet is not None:
      for vlan in vSet:
         vxlanFloodVtep.vlanFilter.add( vlan )

   if 'DOMAIN' in args and args[ 'DOMAIN' ] == 'local':
      vxlanFloodVtep.domainFilter.add( VtepType.controlPlaneVtep )
      vxlanFloodVtep.domainFilter.add( VtepType.dataPlaneVtep )
      vxlanFloodVtep.domainFilter.add( VtepType.umrVtep )
   elif 'DOMAIN' in args and args[ 'DOMAIN' ] == 'remote':
      vxlanFloodVtep.domainFilter.add( VtepType.remoteDomainVtep )

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   vxlanFloodVtep.render( fd, fmt )
   sys.stdout.flush()
   # VxlanFloodVtepListModel is a Deferred Model,
   # so do not have to populate the python Cli model.
   return VxlanModel.VxlanFloodVtepListModel

def convertVlanToFloodVtepSyntax( mode ):
   vxlanConfigDir.vlanFloodVtepRangeSyntax = True

ConfigConvert.registerConfigConvertCallback( convertVlanToFloodVtepSyntax )

class VxlanVlanFloodVtepCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'vxlan vlan flood vtep'
   data = {
      'vxlan': vxlanNode,
      'vlan': vlanVxlan1Node,
      'flood': matcherFlood,
      'vtep': vtepMatcherForConfig,
   }
   noOrDefaultHandler = clearFloodVtep

VxlanIntfModelet.addCommandClass( VxlanVlanFloodVtepCmd )

class VxlanVlanVlansetFloodVtepCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan [ vlan VLANS ] flood vtep [ ACTION ] { VTEP }'
   noOrDefaultSyntax = 'vxlan [ vlan VLANS ] flood vtep ...'
   data = {
      'vxlan': vxlanNode,
      'vlan': vlanVxlan1Node,
      'VLANS': VlanCli.vlanSetMatcher,
      'flood': matcherFlood,
      'vtep': vtepMatcherForConfig,
      'ACTION': addRemoveMatcher,
      'VTEP': IpGenAddrMatcher.IpGenAddrMatcher(
         helpdesc='IP address of remote VTEP' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'ACTION' in args:
         if 'vlan' in args:
            addRemoveFloodVtep( mode, args[ 'VLANS' ], args[ 'VTEP' ],
                                args[ 'ACTION' ] )
         else:
            addRemoveDefaultFloodVtep( mode, args[ 'VTEP' ], args[ 'ACTION' ] )
      elif 'vlan' in args:
         setFloodVtep( mode, args[ 'VLANS' ], args[ 'VTEP' ] )
      else:
         setDefaultFloodVtep( mode, args[ 'VTEP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'vlan' in args:
         noSetFloodVtep( mode, args[ 'VLANS' ] )
      else:
         noSetDefaultFloodVtep( mode )

VxlanIntfModelet.addCommandClass( VxlanVlanVlansetFloodVtepCmd )

#-------------------------------------------------------------------------------
# vxlan flood vtep learned data-plane - include active remote VTEPs in per-VLAN
#                                       flood-lists
#-------------------------------------------------------------------------------
class VxlanFloodVtepLearnedDataPlane( CliCommand.CliCommandClass ):
   syntax = 'vxlan flood vtep learned data-plane'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'flood': matcherFlood,
      'vtep': vtepMatcherForConfig,
      'learned': 'Learned remote VTEPs',
      'data-plane': 'Include learned remote VTEPs in VXLAN flood-lists',
   }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.floodLearnedAll = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.floodLearnedAll = False

VxlanIntfModelet.addCommandClass( VxlanFloodVtepLearnedDataPlane )

def showVxlanSecurityProfile( mode, args ):
   model = VxlanModel.VxlanSecurityProfileModel()
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return model
   for vtep, secProfile in six.iteritems( vtiStatus.vtepToSecProfile ):
      model.vteps[ vtep ] = secProfile
   return model

def addRemoveVtep( mode, vteps, addOrRemove, ipsecProfile=None ):
   '''Add or remove vteps from the default floodVtepList'''
   vtiConfig = getVtiConfig( mode.intf.name )

   if addOrRemove:
      for v in vteps:
         vtiConfig.vtepToSecProfile[ v ] = ipsecProfile
   else:
      invalidVteps = []
      for v in vteps:
         if v not in vtiConfig.vtepToSecProfile:
            invalidVteps.append( v )
            continue
         del vtiConfig.vtepToSecProfile[ v ]
      if invalidVteps != []:
         invalidVtepsList = ', '.join( sorted( ip.v4Addr for ip in invalidVteps ) )
         mode.addWarning( 'No security profile mapping for VTEP { %s }' %
                           invalidVtepsList )

class VxlanVtepsetSecurityProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vtep { VTEPS } ip security profile PROFILE'
   noOrDefaultSyntax = syntax.replace( 'PROFILE', '...' )
   data = {
      'vxlan': vxlanNode,
      'vtep': vtepMatcherForConfig,
      'VTEPS': IpGenAddrMatcher.IpGenAddrMatcher(
                 helpdesc='IP address of remote VTEP' ),
      'ip': 'IP config commands',
      'security': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'security',
            helpdesc='Security configuration' ),
         guard=vxlanSecurityGuard ),
      'profile': 'Configure security profile for VTEP',
      'PROFILE': ipsecProfileNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      addRemoveVtep( mode, args[ 'VTEPS' ], True, ipsecProfile=args[ 'PROFILE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      addRemoveVtep( mode, args[ 'VTEPS' ], False )

VxlanIntfModelet.addCommandClass( VxlanVtepsetSecurityProfileCmd )

def setVxlanSecUdpPort( mode, args ):
   udpPort = args[ 'UDP_PORT' ]
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.secUdpPort = udpPort

def noSetVxlanSecUdpPort( mode, args ):
   # Default to Vxlan udp port
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.secUdpPort = vtiConfig.vxlanSecWellKnownPort

class VxlanSecUdpPortUdpPortCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan security udp-port UDP_PORT'
   noOrDefaultSyntax = 'vxlan security udp-port ...'
   data = {
      'vxlan': vxlanNode,
      'security': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'security',
            helpdesc='Security configuration' ),
         guard=vxlanSecurityGuard ),
      'udp-port': matcherUdpPort,
      'UDP_PORT': CliMatcher.IntegerMatcher( 1024, 65535, helpdesc='UDP port' ),
   }
   handler = setVxlanSecUdpPort
   noOrDefaultHandler = noSetVxlanSecUdpPort

VxlanIntfModelet.addCommandClass( VxlanSecUdpPortUdpPortCmd )

def setVxlanSecUdpSrcPort( mode, args ):
   udpPort = args[ 'UDP_PORT' ]
   vtiConfig = getVtiConfig( mode.intf.name )
   length = 1
   vxlanSrcPortRange = VxlanSrcPortRange( udpPort, length )
   vtiConfig.secSrcPortRange = vxlanSrcPortRange

def noSetVxlanSecUdpSrcPort( mode, args ):
   # Default to 0 to enable source port entropy
   vtiConfig = getVtiConfig( mode.intf.name )
   length = 1
   vxlanSrcPortRange = VxlanSrcPortRange( 0, length )
   vtiConfig.secSrcPortRange = vxlanSrcPortRange

class VxlanSecUdpSrcPortUdpPortCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan security udp-port source UDP_PORT'
   noOrDefaultSyntax = 'vxlan security udp-port source ...'
   data = {
      'vxlan': vxlanNode,
      'security': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'security',
            helpdesc='Security configuration' ),
         guard=vxlanSecurityGuard ),
      'udp-port': matcherUdpPort,
      'source': 'Set source port for VXLANsec encapsulated packets',
      'UDP_PORT': CliMatcher.IntegerMatcher( 1024, 65535, helpdesc='UDP port' ),
   }
   handler = setVxlanSecUdpSrcPort
   noOrDefaultHandler = noSetVxlanSecUdpSrcPort

VxlanIntfModelet.addCommandClass( VxlanSecUdpSrcPortUdpPortCmd )

#-------------------------------------------------------------------------------
# vxlan learn-restrict any - learn from any IP address
# vxlan learn-restrict flood - use the flood list as learn list
# vxlan learn-restrict vtep <prefixes> - replaces the entire default prefix list
# vxlan learn-restrict vtep add <prefixes> - add to default prefix list
# vxlan learn-restrict vtep remove <prefixes> - remove from default prefix list
# vxlan learn-restrict vtep - learn from no prefixes i.e., learn from nobody
# no vxlan learn-restrict - revert to default i.e. learn from any
# vxlan vlan <vlan> learn-restrict any - learn from any IP address
# vxlan vlan <vlan> learn-restrict flood - use the vlan flood list as vlan learn list
# vxlan vlan <vlan> learn-restrict vtep <prefixes> - replaces the entire prefix list
# vxlan vlan <vlan> learn-restrict vtep add <prefixes> - add to existing list
# vxlan vlan <vlan> learn-restrict vtep remove <prefixes> - remove from existing list
# vxlan vlan <vlan> learn-restrict vtep - learn from no prefixes i.e., nobody
# no vxlan vlan <vlan> learn-restrict - remove per-vlan config
# no vxlan vlan learn-restrict vtep - remove per-vlan config for all vlans
# show vxlan learn-restrict vtep [vlan <vlanId>]
# show vxlan counters learn-restrict {prefix,brief,all} [vlan <vlanId>]
# clear vxlan counters learn-restrict [vlan <vlanId>]
#-------------------------------------------------------------------------------
def _verifyVtepPrefixes( mode, prefixes ):
   for p in prefixes:
      # The predicates return false if the prefix is too short to determine
      # i.e., 0.0.0.0/0 is not isMulticast etc.
      if p.isMulticast or p.isLoopback or p.isLinkLocal or p.isUnspecified:
         mode.addError( "invalid VTEP address prefix: %s" % p )
         return False
      if not bridgingHwCapabilities.vxlanIpv6UnderlaySupported and p.af == 'ipv6':
         mode.addError( "IPv6 encapsulation is not supported: %s" % p )
         return False
   return True

def setDefaultLearnRestrictNone( mode ):
   setDefaultLearnRestrict( mode, prefixes=[], learnFrom='learnFromList' )

def prefixesToVtepPrefixes( prefixes ):
   def _prefixToVtepPrefix( p ):
      return Tac.Value( 'Vxlan::VtepPrefix', p )
   if not prefixes:
      return []
   return [ _prefixToVtepPrefix( p ) for p in prefixes ]

def setDefaultLearnRestrict( mode, prefixes, learnFrom='learnFromList' ):
   '''Remove the old default prefix list and create a new one'''
   vxlanConfig = getVxlanConfig( mode.intf.name )

   if not _verifyVtepPrefixes( mode, prefixes ):
      return

   if not _verifyLearnFrom( mode, vxlanConfig.learnPrefixList, prefixes,
                               vxlanConfig.learnFrom, learnFrom ):
      return

   # Form a list of Vxlan::VtepPrefix
   vps = prefixesToVtepPrefixes( prefixes )
   # remove any prefix that don't exist in the new list
   if vxlanConfig.learnPrefixList:
      for vp in vxlanConfig.learnPrefixList:
         if not vps or vp not in vps:
            del vxlanConfig.learnPrefixList[ vp ]
   # add any vteps missing from the new list
   if vps:
      for vp in vps:
         if vp not in vxlanConfig.learnPrefixList:
            vxlanConfig.learnPrefixList[ vp ] = True
   vxlanConfig.learnFrom = learnFrom

def _verifyLearnFrom( mode, oldPrefixes, newPrefixes, oldLearnFrom, newLearnFrom ):
   if oldLearnFrom == newLearnFrom:
      # No change
      return True

   if oldLearnFrom == 'learnFromList' and oldPrefixes:
      mode.addWarning( 'Configured learn list will be ignored' )
   if newLearnFrom == 'learnFromList' and not newPrefixes:
      mode.addWarning( 'Configured learn list is empty. Will learn from nobody' )
   return True

def noSetDefaultLearnRestrict( mode, args ):
   '''Remove the default learn prefix list. Learn from any i.e. default.'''
   setDefaultLearnRestrict( mode, prefixes=[], learnFrom='learnFromDefault' )

def addRemoveDefaultLearnPrefix( mode, prefixes, addOrRemove ):
   '''Add or remove prefixes from the default learn prefix list'''
   vxlanConfig = getVxlanConfig( mode.intf.name )

   # Have to get ipPrefix attribute for each of the existing Vxlan::VtepPrefix
   k = list( vxlanConfig.learnPrefixList )
   s = set( w.ipPrefix for w in k )
   t = set( prefixes )
   if addOrRemove == 'add':
      s = s.union( t )
   else:
      s = s.difference( t )
   prefixes = list( s )
   setDefaultLearnRestrict( mode, prefixes, 'learnFromList' )

def setDefaultLearnFlood( mode ):
   '''Remove the default learn prefix list. Use flood list for learn.'''
   setDefaultLearnRestrict( mode, prefixes=[], learnFrom='learnFromFloodList' )

def setDefaultLearnAny( mode ):
   '''Remove the default learn prefix list. Learn from any.'''
   setDefaultLearnRestrict( mode, prefixes=[], learnFrom='learnFromAny' )

def setLearnRestrictNone( mode, vSet ):
   setLearnRestrict( mode, vSet, prefixes=[], learnFrom='learnFromList' )

def setLearnRestrict( mode, vSet, prefixes, learnFrom='learnFromList' ):
   '''Update the list'''
   vxlanConfig = getVxlanConfig( mode.intf.name )

   if not _verifyVtepPrefixes( mode, prefixes ):
      return

   # Form a list of Vxlan::VtepPrefix
   vps = prefixesToVtepPrefixes( prefixes )
   for vlan in vSet:
      if vlan in vxlanConfig.vlanToLearnRestrict:
         oldLF = vxlanConfig.vlanToLearnRestrict[ vlan ].learnFrom
         oldPref = vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList
      else:
         oldLF = 'learnFromAny'
         oldPref = []
      if not _verifyLearnFrom( mode, oldPref, prefixes, oldLF, learnFrom ):
         return

      if not vlan in vxlanConfig.vlanToLearnRestrict:
         vxlanConfig.vlanToLearnRestrict.newMember( vlan )
      # remove any prefix that don't exist in the new list
      if vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList:
         for vp in vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList:
            if not vps or vp not in vps:
               del vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList[ vp ]
      # add any vteps missing from the new list
      if vps:
         for vp in vps:
            if vp not in vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList:
               vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList[ vp ] = True
      vxlanConfig.vlanToLearnRestrict[ vlan ].learnFrom = learnFrom
      if learnFrom == 'learnFromDefault':
         del vxlanConfig.vlanToLearnRestrict[ vlan ]

def addRemoveLearnPrefix( mode, vSet, prefixes, addOrRemove ):
   '''Add or remove prefixes from the vlanToLearnRestrict'''
   vxlanConfig = getVxlanConfig( mode.intf.name )
   for vlan in vSet:
      if vlan in vxlanConfig.vlanToLearnRestrict:
         # Have to get ipPrefix attribute for each of the existing Vxlan::VtepPrefix
         k = list( vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList )
         s = set( w.ipPrefix for w in k )
         t = set( prefixes )
         if addOrRemove == 'add':
            s = s.union( t )
         else:
            s = s.difference( t )
         prefixes = list( s )
         vs = VlanCli.VlanSet( mode=mode, vlanSetString='', vlanIds=[ vlan ] )
         setLearnRestrict( mode, vs, prefixes, 'learnFromList' )

      elif addOrRemove == 'add':
         vs = VlanCli.VlanSet( mode=mode, vlanSetString='', vlanIds=[ vlan ] )
         setLearnRestrict( mode, vs, prefixes, 'learnFromList' )

def noSetLearnRestrict( mode, vSet ):
   '''Remove the per-vlan learn prefix list. Learn based on default
   (non-vlan) config.'''
   setLearnRestrict( mode, vSet, prefixes=[], learnFrom='learnFromDefault' )

def setLearnFlood( mode, vSet ):
   '''Remove the per-vlan learn prefix list. Use flood list for learn.'''
   setLearnRestrict( mode, vSet, prefixes=[], learnFrom='learnFromFloodList' )

def setLearnAny( mode, vSet ):
   '''Remove the per-vlan learn prefix list. Use flood list for learn.'''
   setLearnRestrict( mode, vSet, prefixes=[], learnFrom='learnFromAny' )

def clearLearnPrefix( mode, args ):
   '''Clear the learn prefix for all the vlans.'''
   vxlanConfig = getVxlanConfig( mode.intf.name )
   for vlan in vxlanConfig.vlanToLearnRestrict:
      vs = VlanCli.VlanSet( mode=mode, vlanSetString='', vlanIds=[ vlan ] )
      setLearnRestrict( mode, vs, prefixes=[], learnFrom='learnFromDefault' )

def showLearnRestrict( mode, args ):
   vSet = args.get( 'VLANS' )
   vxlanStatus = getVxlanStatus( 'Vxlan1' )
   learnModel = VxlanModel.VxlanLearnRestrictModel()
   if vxlanStatus:
      for vlan in vxlanStatus.vlanToLearnRestrict:
         if vSet is not None and vlan not in vSet:
            continue
         pl = VxlanModel.VxlanLearnRestrictModel.VxlanPrefixListModel()
         try:
            pl.learnFrom = vxlanStatus.vlanToLearnRestrict[ vlan ].learnFrom
            if pl.learnFrom == 'learnFromDefault':
               raise KeyError
            for ip in vxlanStatus.vlanToLearnRestrict[ vlan ].prefixList:
               pl.prefixList.append( ip.ipPrefix )
         except KeyError:
            continue
         learnModel.learnMap[ vlan ] = pl
   return learnModel

def showLearnCounters( mode, args ):
   counters = args[ 'COUNTER' ]
   vSet = args.get( 'VLANS' )
   learnStatus = vxHwStatusDir.learnStatus
   statusModel = VxlanModel.VxlanLearnCountersModel()
   statusModel.counters = counters
   if learnStatus:
      for vlan in learnStatus.learnStatusList:
         if vSet is not None and vlan not in vSet:
            continue
         pl = statusModel.VxlanPrefixStatusModel()
         try:
            status = learnStatus.learnStatusList[ vlan ]
            pl.learnFrom = status.learnFrom
            if pl.learnFrom == 'learnFromDefault':
               raise KeyError
            for ip in status.numMatches:
               pl.prefixList[ ip.ipPrefix ] = status.numMatches[ ip ]
            pl.numMatchAny = status.numMatchAny
            pl.numMatchFloodList = status.numMatchFloodList
            pl.numMatchList = status.numMatchList
            pl.numRejectFloodList = status.numRejectFloodList
            pl.numRejectList = status.numRejectList
         except KeyError:
            continue
         statusModel.learnCountersMap[ vlan ] = pl

   return statusModel

def clearLearnCounters( mode, args ):
   vSet = args.get( 'VLANS' )
   vxlanCounterConfig = getVxlanCounterConfig( mode.intf.name )
   learnStatus = vxHwStatusDir.learnStatus
   now = Tac.now()

   if learnStatus:
      for vlan in learnStatus.learnStatusList:
         if vSet is not None and vlan not in vSet:
            continue
         try:
            del vxlanCounterConfig.vlanToClearCounterRequestTime[ vlan ]
            vxlanCounterConfig.vlanToClearCounterRequestTime[ vlan ] = now
         except KeyError:
            pass

      for vlan in learnStatus.learnStatusList:
         if vSet is not None and vlan not in vSet:
            continue
         try:
            Tac.waitFor(
               lambda:
               vlan in learnStatus.learnStatusList and
               learnStatus.learnStatusList[ vlan ].lastClearTime >= now,
               description='learn-restrict clear counter request to complete',
               warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
         except Tac.Timeout:
            mode.addWarning( "learn-restrict counters may not have been reset yet" )

#--------------------------------------------------------------------------------
# vxlan learn-restrict ( any | flood | ( vtep [ { VTEP } ] ) )
#--------------------------------------------------------------------------------
class VxlanLearnRestrictCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan learn-restrict ( any | flood | ( vtep [ { PREFIX_ADDR } ] ) )'
   noOrDefaultSyntax = 'vxlan learn-restrict ...'
   data = {
      'vxlan': vxlanNode,
      'learn-restrict': matcherLearnRestrict,
      'any': matcherAny,
      'flood': matcherFlood,
      'vtep': vtepMatcherForConfig,
      'PREFIX_ADDR': ipGenPrefixMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if 'any' in args:
         return setDefaultLearnAny( mode )
      elif 'flood' in args:
         return setDefaultLearnFlood( mode )
      else:
         assert 'vtep' in args
         if 'PREFIX_ADDR' in args:
            return setDefaultLearnRestrict( mode, args[ 'PREFIX_ADDR' ] )
         else:
            return setDefaultLearnRestrictNone( mode )

   noOrDefaultHandler = noSetDefaultLearnRestrict

VxlanIntfModelet.addCommandClass( VxlanLearnRestrictCmd )

# Allow setting zero prefixes
class AddRemoveLearnPrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan [ vlan VLANS ] learn-restrict vtep ACTION { PREFIX_ADDR }'
   data = {
      'vxlan': vxlanNode,
      'vlan': vlanVxlan1Node,
      'VLANS': VlanCli.vlanSetMatcher,
      'learn-restrict': matcherLearnRestrict,
      'ACTION': addRemoveMatcher,
      'vtep': vtepMatcherForConfig,
      'PREFIX_ADDR': ipGenPrefixMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if 'vlan' in args:
         addRemoveLearnPrefix( mode, args[ 'VLANS' ], args[ 'PREFIX_ADDR' ],
                               args[ 'ACTION' ] )
      else:
         addRemoveDefaultLearnPrefix( mode, args[ 'PREFIX_ADDR' ], args[ 'ACTION' ] )

VxlanIntfModelet.addCommandClass( AddRemoveLearnPrefixCmd )

# Allow setting zero prefixes
class LearnRestrictNoneCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vlan VLANS learn-restrict vtep'
   data = {
            'vxlan': vxlanNode,
            'vlan': vlanVxlan1Node,
            'VLANS': VlanCli.vlanSetMatcher,
            'learn-restrict': matcherLearnRestrict,
            'vtep': vtepMatcherForConfig,
          }

   @staticmethod
   def handler( mode, args ):
      setLearnRestrictNone( mode, args[ 'VLANS' ] )

VxlanIntfModelet.addCommandClass( LearnRestrictNoneCmd )

class VxlanVlansetLearnRestrictCmd( CliCommand.CliCommandClass ):
   syntax = ( 'vxlan vlan VLANS learn-restrict '
                                       '( any | flood | ( vtep { PREFIX_ADDR } ) )' )
   noOrDefaultSyntax = 'vxlan vlan VLANS learn-restrict ...'
   data = {
      'vxlan': vxlanNode,
      'vlan': vlanVxlan1Node,
      'VLANS': VlanCli.vlanSetMatcher,
      'learn-restrict': matcherLearnRestrict,
      'any': matcherAny,
      'flood': matcherFlood,
      'vtep': vtepMatcherForConfig,
      'PREFIX_ADDR': ipGenPrefixMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if 'any' in args:
         return setLearnAny( mode, args[ 'VLANS' ] )
      elif 'flood' in args:
         return setLearnFlood( mode, args[ 'VLANS' ] )
      else:
         assert 'vtep' in args
         return setLearnRestrict( mode, args[ 'VLANS' ], args[ 'PREFIX_ADDR' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noSetLearnRestrict( mode, args[ 'VLANS' ] )

VxlanIntfModelet.addCommandClass( VxlanVlansetLearnRestrictCmd )

class ClearLearnPrefixCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'vxlan vlan learn-restrict vtep'
   data = {
      'vxlan': vxlanNode,
      'vlan': matcherVlan,
      'learn-restrict': matcherLearnRestrict,
      'vtep': vtepMatcherForConfig,
   }
   noOrDefaultHandler = clearLearnPrefix

VxlanIntfModelet.addCommandClass( ClearLearnPrefixCmd )

#--------------------------------------------------------------------------------
# clear vxlan counters learn-restrict [ vlan VLANS ]
#--------------------------------------------------------------------------------
class ClearVxlanCountersLearnRestrictCmd( CliCommand.CliCommandClass ):
   syntax = 'clear vxlan counters learn-restrict [ vlan VLANS ]'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'vxlan': vxlanNode,
      'counters': 'VXLAN counters',
      'learn-restrict': matcherLearnRestrict,
      'vlan': matcherVlan,
      'VLANS': VlanCli.vlanSetMatcher,
   }
   handler = clearLearnCounters

VxlanIntfModelet.addCommandClass( ClearVxlanCountersLearnRestrictCmd )

#-------------------------------------------------------------------------------
# vxlan vni <VNI> routed ip access-list <NAME> ( in | out )
# vxlan vni <VNI> routed ipv6 access-list <NAME> ( in | out )
#-------------------------------------------------------------------------------
vniDecapsulationMatcher = CliMatcher.KeywordMatcher( 'decapsulation',
                                 helpdesc='List of additional VNIs' )

def getAclMap( mode, args ):
   af = 'ipv6' if 'ipv6' in args else 'ip'
   direction = 'in' if 'in' in args else 'out'
   vxlanConfig = getVxlanConfig( mode.intf.name )
   aclMap = {
      'ip': {
         'in': vxlanConfig.vniToIpAclMap,
         'out': vxlanConfig.vniToEgressIpAclMap,
      },
      'ipv6': {
         'in': vxlanConfig.vniToIpv6AclMap,
         'out': vxlanConfig.vniToEgressIpv6AclMap,
      }
   }
   return aclMap[ af ][ direction ]

def setVniAcl( mode, args ):
   vni = args[ 'VNI' ]
   aclName = args[ 'ACL_NAME' ]
   aclMap = getAclMap( mode, args )
   vniNum = VniFormat( vni ).toNum()
   if not isValidVniWithError( mode, vniNum, mode.intf.name ):
      return
   aclMap[ vniNum ] = aclName

def noVniAcl( mode, args ):
   vni = args[ 'VNI' ]
   aclName = args.get( 'ACL_NAME' )
   vniNum = VniFormat( vni ).toNum()
   aclMap = getAclMap( mode, args )
   af = 'ipv6' if 'ipv6' in args else 'ip'
   if not isValidVniWithError( mode, vniNum, mode.intf.name ):
      return
   errorString = \
         "Access list %s not configured on VNI %d for %s traffic"
   currentAclName = aclMap.get( vniNum )
   direction = 'in' if 'in' in args else 'out'
   if aclName and currentAclName and aclName != currentAclName:
      dirAfStr = 'ingress IP' if direction == 'in' else 'egress IP'
      if af == 'ipv6':
         dirAfStr += 'v6'
      else:
         dirAfStr += ''
      mode.addError( errorString % ( aclName, vniNum, dirAfStr ) )
   else:
      del aclMap[ vniNum ]

nodeVniRouted = CliCommand.guardedKeyword( 'routed',
      helpdesc='Apply to interfaces for VXLAN routing',
      guard=nodeVniRoutedGuard )

class VxlanVniRoutedIpAccessListAclNameCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vni VNI routed ip access-list ACL_NAME ( in | out )'
   noOrDefaultSyntax = syntax.replace( 'ACL_NAME', '[ ACL_NAME ]' )
   data = {
      'vxlan': vxlanNode,
      'vni': vniMatcherForConfig,
      'VNI': vniMatcher,
      'routed': nodeVniRouted,
      'ip': CliCommand.Node(
         matcher=AclCli.ipKwForServiceAclMatcher,
         guard=vxlanTunnelIpv4VniAclSupported ),
      'access-list': AclCli.accessListKwMatcher,
      'ACL_NAME': AclCli.ipAclNameMatcher,
      'in': CliCommand.Node(
         matcher=AclCli.inKwMatcher,
         guard=vxlanTunnelIngressIpv4VniAclSupported ),
      'out': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'out',
            helpdesc='Outbound packets' ),
         guard=vxlanTunnelEgressIpv4VniAclSupported ),
   }
   handler = setVniAcl
   noOrDefaultHandler = noVniAcl

VxlanIntfModelet.addCommandClass( VxlanVniRoutedIpAccessListAclNameCmd )

class VxlanVniRoutedIpv6AccessListAclNameCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vni VNI routed ipv6 access-list ACL_NAME ( in | out )'
   noOrDefaultSyntax = syntax.replace( 'ACL_NAME', '[ ACL_NAME ]' )
   data = {
      'vxlan': vxlanNode,
      'vni': vniMatcherForConfig,
      'VNI': vniMatcher,
      'routed': nodeVniRouted,
      'ipv6': CliCommand.Node(
         matcher=AclCli.ipv6KwMatcherForServiceAcl,
         guard=vxlanTunnelIpv6VniAclSupported ),
      'access-list': AclCli.accessListKwMatcher,
      'ACL_NAME': AclCli.ip6AclNameMatcher,
      'in': CliCommand.Node(
         matcher=AclCli.inKwMatcher,
         guard=vxlanTunnelIngressIpv6VniAclSupported ),
      'out': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'out',
            helpdesc='Outbound packets' ),
         guard=vxlanTunnelEgressIpv6VniAclSupported ),
   }
   handler = setVniAcl
   noOrDefaultHandler = noVniAcl

VxlanIntfModelet.addCommandClass( VxlanVniRoutedIpv6AccessListAclNameCmd )

# -------------------------------------------------------------------------------
# vxlan vni <VNI> routed ip nat service-profile <PROFILE>
# -------------------------------------------------------------------------------
def attachVniNatProfile( mode, args ):
   vni = args[ 'VNI' ]
   profileName = args[ 'PROFILE' ]

   vniNum = VniFormat( vni ).toNum()
   if not isValidVniWithError( mode, vniNum, mode.intf.name ):
      return

   vtiConfig = getVtiConfig( mode.intf.name )

   # Add an entry for this VNI
   vtiConfig.vniNatProfile[ vniNum ] = profileName

def detachVniNatProfile( mode, args ):
   vni = args[ 'VNI' ]

   vniNum = VniFormat( vni ).toNum()
   if not isValidVniWithError( mode, vniNum, mode.intf.name ):
      return

   vtiConfig = getVtiConfig( mode.intf.name )

   # Delete the config if it exists
   del vtiConfig.vniNatProfile[ vniNum ]

class VxlanVniNatProfileCmd( CliCommand.CliCommandClass ):
   syntax = '''vxlan vni VNI routed ip nat service-profile PROFILE'''
   noOrDefaultSyntax = '''vxlan vni VNI routed ip nat service-profile ...'''

   data = {
      'vxlan': vxlanNode,
      'vni': vniMatcherForConfig,
      'VNI': vniMatcher,
      'routed': nodeVniRouted,
      'ip': CliCommand.guardedKeyword(
         'ip', helpdesc='IP config commands',
         guard=natWithVxlanSupportedGuard ),
      'nat': 'NAT config commands',
      'service-profile': 'NAT profile',
      'PROFILE': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
                                            helpdesc='NAT profile name',
                                            helpname='name' ),
   }

   handler = attachVniNatProfile
   noOrDefaultHandler = detachVniNatProfile

VxlanIntfModelet.addCommandClass( VxlanVniNatProfileCmd )

# -----------------------------------------------------------------------------------
# vxlan vni <VNI> policer ((profile <PROFILE>) | (group <POLICER>))  input
# vxlan vni <VNI> policer ((profile <PROFILE>) | (group <POLICER>))  output
# [no | default] vxlan vni <VNI> policer [ ( [ ( (profile PROFILE) [ DIR ] ) |\
#                                                ( (group POLICER) [ DIR ] ) ] ) |\
#                                                DIR ]'
# -----------------------------------------------------------------------------------
vniPolicer = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'policer',
      helpdesc='Policer configuration on the vni' ),
      guard=vxlanVniPolicerSupportedGuard )
vniProfile = CliMatcher.KeywordMatcher( 'profile',
      helpdesc="Associate vni with a policer profile" )
vniProfileName = CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
     helpdesc='Policer profile name',
     helpname='NAME' )
vniGroup = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'group',
      helpdesc="Associate vni with a group policer" ),
      guard=vxlanVniGroupPolicerSupportedGuard )
vniPolicerName = CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
     helpdesc='Policer name',
     helpname='NAME' )
inputNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'input',
      helpdesc='Applied to incoming traffic' ),
      guard=vxlanVniDecapPolicingSupportedGuard )
outputNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'output',
      helpdesc='Applied to outgoing traffic' ),
      guard=vxlanVniEncapPolicingSupportedGuard )

def showVxlanVniPolicers( mode, args ):
   vtiInput = args.get( 'INTF' )
   vniInput = args.get( 'VNI' )

   if vniInput is not None:
      vniInputNum = int( vniInput )
      vtiName = 'Vxlan1'
      if vtiInput is not None:
         vtiName = str( vtiInput )
      if not isValidVniWithError( mode, vniInputNum, vtiName ):
         return None

   def filterInterface( vtiName ):
      if vtiInput is not None:
         return str( vtiInput ) == vtiName
      else:
         return True

   def filterVni( vniVal ):
      if vniInput:
         vniInputNum = VniFormat( vniInput ).toNum()
         return vniInputNum == vniVal
      else:
         return True

   def buildVniVlanBindings( vtiStatus, vniVlanBindings ):
      for vlan, vs in six.iteritems( vtiStatus.extendedVlanToVniMap ):
         if filterVni( int( vs.vni ) ):
            vniVlanBindings[ int( vs.vni ) ] = vlan

   def populateModelEntry( vti, vni, vniVlanBindings, vxlanConfig ):

      vniQosInfoModel = VxlanModel.VxlanVniQosInfoModel()

      vniQosInfoModel.extVlan = vniVlanBindings.get( vni )
      vniExt = Tac.Value( "Vxlan::VniExt", vni )
      vti = Tac.Value( "Arnet::IntfId", vti )
      vniVti = Tac.Value( "Vxlan::VniVtiPair", vniExt, vti )

      iPolicerStatus = False
      oPolicerStatus = False
      if vniVti in vniPolicerHwStatusDir.vniPolicerHwStatus:
         iPolicerStatus =\
               vniPolicerHwStatusDir.vniPolicerHwStatus[ vniVti ].ingPolicerState
         oPolicerStatus =\
               vniPolicerHwStatusDir.vniPolicerHwStatus[ vniVti ].egrPolicerState

      iPolicer = vxlanConfig.vniQosInfo[ vni ].iPolicer
      if iPolicer:
         vniQosInfoModel.iPolicerInfo = VxlanModel.VxlanVniPolicer()
         vniQosInfoModel.iPolicerInfo.name = iPolicer.policerName

         if iPolicer.policerMode:
            vniQosInfoModel.iPolicerInfo.mode =\
               VxlanModel.VxlanVniPolicerModeEnum( value='group' )
         else:
            vniQosInfoModel.iPolicerInfo.mode =\
               VxlanModel.VxlanVniPolicerModeEnum( value='profile' )

         if iPolicerStatus:
            vniQosInfoModel.iPolicerInfo.status =\
               VxlanModel.VxlanVniPolicerHwStateEnum( hwState='active' )
         else:
            vniQosInfoModel.iPolicerInfo.status =\
               VxlanModel.VxlanVniPolicerHwStateEnum( hwState='inactive' )

      oPolicer = vxlanConfig.vniQosInfo[ vni ].oPolicer
      if oPolicer:
         vniQosInfoModel.oPolicerInfo = VxlanModel.VxlanVniPolicer()
         vniQosInfoModel.oPolicerInfo.name = oPolicer.policerName

         if oPolicer.policerMode:
            vniQosInfoModel.oPolicerInfo.mode =\
               VxlanModel.VxlanVniPolicerModeEnum( value='group' )
         else:
            vniQosInfoModel.oPolicerInfo.mode =\
               VxlanModel.VxlanVniPolicerModeEnum( value='profile' )

         if oPolicerStatus:
            vniQosInfoModel.oPolicerInfo.status =\
               VxlanModel.VxlanVniPolicerHwStateEnum( hwState='active' )
         else:
            vniQosInfoModel.oPolicerInfo.status =\
               VxlanModel.VxlanVniPolicerHwStateEnum( hwState='inactive' )

      return vniQosInfoModel

   def populateModel( vtiName, vxlanConfig, vniVlanBindings, vxlanConfigMapping ):

      if vniInput is not None:
         vniNum = int( vniInput )
         if vniNum in vxlanConfig.vniQosInfo:
            vxlanConfigMapping.vniQosInfo[ vniNum ] =\
                  populateModelEntry( vtiName, vniNum, vniVlanBindings, vxlanConfig )
         return

      for vni in vxlanConfig.vniQosInfo:
         vxlanConfigMapping.vniQosInfo[ vni ] =\
               populateModelEntry( vtiName, vni, vniVlanBindings, vxlanConfig )

   # populate model
   model = VxlanModel.VxlanVniPolicersModel()
   if not vtiStatusDir or not vtiStatusDir.vtiStatus:
      return model

   if vtiInput is not None:
      vtiName = str( vtiInput )
      vxlanConfig = getVxlanConfig( vtiName )
      if vxlanConfig:
         vniVlanBindings = {}
         vtiStatus = getVtiStatus( vtiName )
         buildVniVlanBindings( vtiStatus, vniVlanBindings )
         vxlanConfigMapping = VxlanModel.VxlanConfigModel()
         populateModel( vtiName, vxlanConfig, vniVlanBindings, vxlanConfigMapping )
         model.vxlanConfig[ vtiName ] = vxlanConfigMapping
      return model

   for vtiName, vtiStatus in six.iteritems( vtiStatusDir.vtiStatus ):
      vxlanConfig = getVxlanConfig( vtiName )
      if vxlanConfig and filterInterface( vtiName ):
         vniVlanBindings = {}
         buildVniVlanBindings( vtiStatus, vniVlanBindings )
         vxlanConfigMapping = VxlanModel.VxlanConfigModel()
         populateModel( vtiName, vxlanConfig, vniVlanBindings, vxlanConfigMapping )
         model.vxlanConfig[ vtiName ] = vxlanConfigMapping
   return model

def showVxlanVniPolicersCounters( mode, args ):
   vtiInput = args.get( 'VXLANINTF' )
   vniInput = args.get( 'VNI' )
   dirInput = args.get( 'DIR', 'all' )

   model = VxlanModel.VxlanVniPolicersCountersModel()
   model.policerDirIs( dirInput )

   def matchVtiInput( vti ):
      if vtiInput is None:
         return True
      return str( vtiInput ) == vti

   def matchVniInput( vni ):
      if vniInput is None:
         return True
      return int( vniInput ) == vni

   def retrieveDecapSmashCounter( vniVti ):
      iPolicerCounter = None
      if vniPolicerDecapCounterTable:
         counter = vniPolicerDecapCounterTable.counter.get( vniVti )
         if counter is not None:
            iPolicerCounter = Tac.nonConst( counter )
            if vniPolicerDecapSnapshotCounterTable:
               snapshot = vniPolicerDecapSnapshotCounterTable.counter.get( vniVti )
               if snapshot is not None:
                  iPolicerCounter.pktInCount -= snapshot.pktInCount
                  iPolicerCounter.byteInCount -= snapshot.byteInCount
                  iPolicerCounter.pktDropCount -= snapshot.pktDropCount
                  iPolicerCounter.byteDropCount -= snapshot.byteDropCount
      return iPolicerCounter

   def retrieveEncapSmashCounter( vniVti ):
      oPolicerCounter = None
      if vniPolicerEncapCounterTable:
         counter = vniPolicerEncapCounterTable.counter.get( vniVti )
         if counter is not None:
            oPolicerCounter = Tac.nonConst( counter )
            if vniPolicerEncapSnapshotCounterTable:
               snapshot = vniPolicerEncapSnapshotCounterTable.counter.get( vniVti )
               if snapshot is not None:
                  oPolicerCounter.pktInCount -= snapshot.pktInCount
                  oPolicerCounter.byteInCount -= snapshot.byteInCount
                  oPolicerCounter.pktDropCount -= snapshot.pktDropCount
                  oPolicerCounter.byteDropCount -= snapshot.byteDropCount
      return oPolicerCounter

   def createVniPolicingCountersSubmodel( hwPolicerCounter ):
      counters = VxlanModel.VniPolicingCounters()
      counters.conformedPackets = hwPolicerCounter.pktInCount
      counters.conformedBytes = hwPolicerCounter.byteInCount
      counters.exceededPackets = hwPolicerCounter.pktDropCount
      counters.exceededBytes = hwPolicerCounter.byteDropCount
      return counters

   def populateCountersModel( vtiName, vni, vniQosInfo ):
      vniExt = Tac.Value( "Vxlan::VniExt", vni )
      vti = Tac.Value( "Arnet::IntfId", vtiName )
      vniVti = Tac.Value( "Vxlan::VniVtiPair", vniExt, vti )

      if vniVti not in vniPolicerHwStatusDir.vniPolicerHwStatus:
         return

      vniPolicerHwStatus = vniPolicerHwStatusDir.vniPolicerHwStatus[ vniVti ]

      # Populate model with input policer info
      iPolicer = vniQosInfo.iPolicer
      iPolicerState = vniPolicerHwStatus.ingPolicerState
      if vniPolicingSmashCountersSupported():
         # Sand Gen4 based platforms use smash tables for rendering counters
         iPolicerCounter = retrieveDecapSmashCounter( vniVti )
      else:
         iPolicerCounter = vniPolicerHwStatus.ingPolicerCounter
      iPolicerDir = dirInput == 'input' or dirInput == 'all'
      if iPolicer and iPolicerState and iPolicerCounter and iPolicerDir:
         policerModeEnumValue = 'group' if iPolicer.policerMode else 'profile'
         countersInfoModel = VxlanModel.VxlanVniPolicersCountersInfoModel()
         countersInfoModel.policerName = iPolicer.policerName
         countersInfoModel.mode = \
            VxlanModel.VxlanVniPolicerModeEnum( value=policerModeEnumValue )
         countersInfoModel.counters = \
            createVniPolicingCountersSubmodel( iPolicerCounter )

         if vtiName not in model.ingInterfaces:
            model.ingInterfaces[ vtiName ] = \
               VxlanModel.VxlanVniPolicersAllCountersInfoModel()
         model.ingInterfaces[ vtiName ].vniPolicers[ vni ] = countersInfoModel

      # Populate model with output policer info
      oPolicer = vniQosInfo.oPolicer
      oPolicerState = vniPolicerHwStatus.egrPolicerState
      if vniPolicingSmashCountersSupported():
         # Sand Gen4 based platforms read counters from smash tables
         oPolicerCounter = retrieveEncapSmashCounter( vniVti )
      else:
         oPolicerCounter = vniPolicerHwStatus.egrPolicerCounter
      oPolicerDir = dirInput == 'output' or dirInput == 'all'
      if oPolicer and oPolicerState and oPolicerCounter and oPolicerDir:
         policerModeEnumValue = 'group' if oPolicer.policerMode else 'profile'
         countersInfoModel = VxlanModel.VxlanVniPolicersCountersInfoModel()
         countersInfoModel.policerName = oPolicer.policerName
         countersInfoModel.mode = \
            VxlanModel.VxlanVniPolicerModeEnum( value=policerModeEnumValue )
         countersInfoModel.counters = \
            createVniPolicingCountersSubmodel( oPolicerCounter )

         if vtiName not in model.egrInterfaces:
            model.egrInterfaces[ vtiName ] = \
               VxlanModel.VxlanVniPolicersAllCountersInfoModel()
         model.egrInterfaces[ vtiName ].vniPolicers[ vni ] = countersInfoModel

   for vtiName in list( vtiStatusDir.vtiStatus ):
      vxlanConfig = getVxlanConfig( vtiName )
      if vxlanConfig and matchVtiInput( vtiName ):
         for vni, vniQosInfo in six.iteritems( vxlanConfig.vniQosInfo ):
            if vniQosInfo and matchVniInput( vni ):
               populateCountersModel( vtiName, vni, vniQosInfo )

   return model

def attachVniPolicer( mode, args ):

   vni = args[ 'VNI' ]
   if 'profile' in args:
      sharedMode = False
      policerName = args[ 'PROFILE' ]
   elif 'group' in args:
      sharedMode = True
      policerName = args[ 'POLICER' ]

   if 'input' in args:
      direction = 'input'
   elif 'output' in args:
      direction = 'output'
   else:
      assert 0, "Invalid direction"

   vniNum = VniFormat( vni ).toNum()
   if not isValidVniWithError( mode, vniNum, mode.intf.name ):
      return

   vxlanConfig = getVxlanConfig( mode.intf.name )
   if not vniNum in vxlanConfig.vniQosInfo:
      vxlanConfig.vniQosInfo.newMember( vniNum )

   if direction == 'input':
      vxlanConfig.vniQosInfo[ vniNum ].iPolicer = ( policerName, sharedMode )
   else:
      vxlanConfig.vniQosInfo[ vniNum ].oPolicer = ( policerName, sharedMode )

def detachVniPolicer( mode, args ):

   vni = args[ 'VNI' ]
   vniNum = VniFormat( vni ).toNum()
   if not isValidVniWithError( mode, vniNum, mode.intf.name ):
      return

   vxlanConfig = getVxlanConfig( mode.intf.name )
   if not vniNum in vxlanConfig.vniQosInfo:
      return
   else:
      qosConfig = vxlanConfig.vniQosInfo[ vniNum ]

   if 'profile' in args:
      sharedMode = False
      policerName = args[ 'PROFILE' ]
   elif 'group' in args:
      sharedMode = True
      policerName = args[ 'POLICER' ]
   else:
      policerName = ""
      sharedMode = False

   # Detach specified direction policer provided name matches
   # If direction not specified, try detaching in both directons
   if 'input' in args:
      direction = 'input'
   elif 'output' in args:
      direction = 'output'
   else:
      direction = ""

   if policerName:
      if qosConfig.iPolicer and direction != 'output':
         if qosConfig.iPolicer.policerMode == sharedMode and \
               qosConfig.iPolicer.policerName == policerName:
            qosConfig.iPolicer = None

      if qosConfig.oPolicer and direction != 'input':
         if qosConfig.oPolicer.policerMode == sharedMode and \
               qosConfig.oPolicer.policerName == policerName:
            qosConfig.oPolicer = None
   else:
      if direction != 'output':
         qosConfig.iPolicer = None

      if direction != 'input':
         qosConfig.oPolicer = None

   # Delete the configuration from collection if no policer profile attached to vni
   if not ( qosConfig.oPolicer or qosConfig.iPolicer ):
      del vxlanConfig.vniQosInfo[ vniNum ]

class VxlanVniPolicerProfileCmd( CliCommand.CliCommandClass ):
   syntax = '''vxlan vni VNI policer ( ( profile PROFILE ) | ( group POLICER ) )
            ( input | output )'''
   noOrDefaultSyntax = '''vxlan vni VNI policer
                        [ ( profile PROFILE [ input | output ] )
                        | ( group POLICER [ input | output ] )
                        | ( input | output ) ]'''
   data = {
      'vxlan': vxlanNode,
      'vni': vniMatcherForConfig,
      'VNI': vniMatcher,
      'policer': vniPolicer,
      'profile': vniProfile,
      'PROFILE': vniProfileName,
      'group': vniGroup,
      'POLICER': vniPolicerName,
      'input': inputNode,
      'output': outputNode,
   }
   handler = attachVniPolicer
   noOrDefaultHandler = detachVniPolicer

VxlanIntfModelet.addCommandClass( VxlanVniPolicerProfileCmd )

# -----------------------------------------------------------------------------------
# vxlan vni <VNI> service-policy type pbr input <PMAP_NAME>
# -----------------------------------------------------------------------------------

def guardPbrOnVni( mode, token ):
   if routingHwStatus.pbrSupported and routingHwStatus.pbrOnVniSupported:
      return None
   return CliParser.guardNotThisPlatform

nodePbr = CliCommand.guardedKeyword( 'pbr', helpdesc='Pbr type',
                                     guard=guardPbrOnVni )

matcherPmapNamePbr = CliMatcher.DynamicNameMatcher(
   lambda mode: pbrCliConfig.pmapType.pmap, 'Pbr Policy Map Name' )

def guardServicePolicyPbr( mode, token ):
   if routingHwStatus.pbrSupported and mode.intf.name == 'Vxlan1':
      return None

   return CliParser.guardNotThisPlatform

def setPbrOnVni( mode, args ):
   vni = args[ 'VNI' ]
   vniNum = VniFormat( vni ).toNum()
   if not isValidVniWithError( mode, vniNum, mode.intf.name ):
      return

   # pbrVniIntfConfig can accept VNI with both type of VLAN (static, dynamic)
   # but for VNI with static VLAN, config will be ignored.
   pbrName = args[ 'PMAP_NAME' ]
   pbrVniIntfConfig.vni[ vniNum ] = pbrName

   dynVlanSet = VlanCli.Vlan.getDynVlanSet( mode )
   vlan = vniToVlanMap.getVlan( int( vniNum ), mode.intf.name )
   if vlan and vlan not in dynVlanSet:
      mode.addWarning( "PBR policy on VNI %d with static VLAN %d will be "
                        "ignored" % ( vniNum, vlan ) )

def noPbrOnVni( mode, args ):
   vni = args[ 'VNI' ]
   vniNum = VniFormat( vni ).toNum()
   if not isValidVniWithError( mode, vniNum, mode.intf.name ):
      return

   inputPbrName = args.get( 'PMAP_NAME' )
   currentPbrName = pbrVniIntfConfig.vni.get( vniNum )
   if inputPbrName and currentPbrName and inputPbrName != currentPbrName:
      return

   del pbrVniIntfConfig.vni[ vniNum ]

class VxlanPbrOnVniCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vni VNI service-policy type pbr input PMAP_NAME'
   noOrDefaultSyntax = syntax.replace( 'PMAP_NAME', '[ PMAP_NAME ]' )
   data = {
      'vxlan': vxlanNode,
      'vni': vniMatcherForConfig,
      'VNI': vniMatcher,
      'service-policy': CliCommand.guardedKeyword( 'service-policy',
         helpdesc='Service policy configuration on VNI',
         guard=guardServicePolicyPbr ),
      'type': matcherType,
      'pbr': nodePbr,
      'input': 'Apply the policy map to overlay ingress packets',
      'PMAP_NAME': matcherPmapNamePbr,
   }
   handler = setPbrOnVni
   noOrDefaultHandler = noPbrOnVni

VxlanIntfModelet.addCommandClass( VxlanPbrOnVniCmd )

# -----------------------------------------------------------------------------------
# vxlan vni <VNI> traffic-policy input <PMAP_NAME>
#
# Note: this is implemented in TrafficPolicy/CliPlugin due to the following circular
# dependency when implemented here:
# Vxlan -> TrafficPolicy -> Qos -> Dcb -> Lldp -> Snmp -> Vxlan
# -----------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# vxlan vlan <vlanRange> vni <vniRange>
# vxlan vlan add <vlanRange> vni <vniRange>
# vxlan vlan remove <vlanRange> vni <vniRange>
# [no|default] vxlan vlan <vlanRange> vni
#
# sets the vni to vlan mappings under interface vxlan
#-------------------------------------------------------------------------------
def validateVlanToVniAdd( mode, vlanList, vniList ):
   vtiConfig = getVtiConfig( mode.intf.name )
   # Command fails unless all mappings in range are valid
   for vlan, vni in zip( vlanList, vniList ):
      vni = VniFormat( vni ).toNum()
      if not isValidVniWithError( mode, vni, mode.intf.name ):
         return
      if not mode.session_.startupConfig():
         exisitingVlan = vtiConfig.vniInVlanCheck( vlan, vni )
         if exisitingVlan != 0:
            mode.addError( "VLAN %d is already mapped to VNI %d" %
                           ( exisitingVlan, vni ) )
            return
      # If a vlan is in use as an internal vlan by a routed port,
      # vlan to Vni creation for that vlan is not supported.
      vc = bridgingConfig.vlanConfig.get( vlan )
      # In interactive mode, warn user if VLAN conflicts with internal VLAN
      if vc and vc.internal and mode.session_.interactive_:
         mode.addError( "VLAN %d is already used as an internal VLAN." % ( vlan ) )
         return
      # If the mlag pair is configured ('primary', 'secondary', or negotiating
      # 'inactive' ) a the peer link SVI vlan cannot be mapped
      if mlagStatus.mlagState in ( 'primary', 'secondary', 'inactive' ):
         if mlagStatus.localInterface:
            vlanIntf = mlagStatus.localInterface.intfId
            vlanId = vlanIntf[ len( 'Vlan' ) : ]
            if vlanId == str( vlan ):
               mode.addError( "VLAN %d is already used as the peer-link SVI VLAN." %
                              ( vlan ) )
               return
      # If the VNI is used in a vrfToVniMap, then reject
      for vrf in vtiConfig.vrfToVniMap:
         if vtiConfig.vrfToVniMap[ vrf ] == vni:
            mode.addError( "VNI %s is already used to map VRF %s" %
                           ( str( vni ), vrf ) )
            return
      if vni in vtiConfig.alternateDecapVniToVrfMap:
         vrf = vtiConfig.alternateDecapVniToVrfMap[ vni ]
         mode.addError( "VNI %s is already used to map VRF %s" %
                         ( str( vni ), vrf ) )
   # The old syntax for setting vlan to vni mapping accepted single values only:
   # e.g. (config-if-Vx1)$ vxlan vlan <vlan> vni <vni>
   # The new syntax accepts ranges:
   # e.g. (config-if-Vx1)$ vxlan vlan <vlanList> vni <vniList>
   # If the new syntax is used, we set vlanVniRangeSyntax in VtiConfigDir so
   # that command is stored as single line in running-config. Otherwise, we
   # store mappings in individual lines.
   if len( vlanList ) > 1:
      vtiConfigDir.vlanVniRangeSyntax = True
   for vlan, vni in zip( vlanList, vniList ):
      vni = VniFormat( vni ).toNum()
      vtiConfig.vlanToVniMap[ vlan ] = vni
      vniToVlanMap.setVni( vni, vlan, mode.intf.name )

def validateVlanToVniRemove( mode, vlanList, vniList ):
   vtiConfig = getVtiConfig( mode.intf.name )
   if vniList is not None:
      # All deletions must be validated before we can commit anything
      for vlan, vni in zip( vlanList, vniList ):
         # If vlan is not mapped, ignore the delete request
         if vlan not in vtiConfig.vlanToVniMap:
            continue
         vni = VniFormat( vni ).toNum()
         if not isValidVniWithError( mode, vni, mode.intf.name ):
            return
         if vtiConfig.vlanToVniMap[ vlan ] != vni:
            mode.addError( "vlan %d is not mapped to vni %s" %
                  ( vlan, vni ) )
            return
   for vlan in vlanList:
      # Delete vni to vlan mapping
      if vlan in vtiConfig.vlanToVniMap:
         vni = vtiConfig.vlanToVniMap[ vlan ]
         vniToVlanMap.delVni( vni, mode.intf.name )
      # We tolerate overconstrained vlan list for usability,
      # ie. `$ vxlan vlan 1-20 vni` if only 1-10,14-20 are valid
      # If vni list is passed, explicit checking is done above
      try:
         del vtiConfig.vlanToVniMap[ vlan ]
      except KeyError:
         continue

def setVlanToVniMapping( mode, vlanList, vniList ):
   if len( vlanList ) != len( vniList ):
      mode.addError( "Number of VLANs must equal number of VNIs" )
      return
   vtiConfig = getVtiConfig( mode.intf.name )
   # If mapping with new syntax already exists, add keyword is needed
   if vtiConfigDir.vlanVniRangeSyntax and vtiConfig.vlanToVniMap:
      mode.addError( "VLAN to VNI mapping already exists. Use 'add/remove' "
                     "keyword to modify existing map" )
      return
   # Validate before adding mappings
   validateVlanToVniAdd( mode, vlanList, vniList )

def updateVlanToVniMapping( mode, vlanList, vniList, addOrRemove ):
   if len( vlanList ) != len( vniList ):
      mode.addError( "Number of VLANs must equal number of VNIs" )
      return
   if addOrRemove == 'add':
      validateVlanToVniAdd( mode, vlanList, vniList )
   else:
      validateVlanToVniRemove( mode, vlanList, vniList )

def noSetVlanToVniMapping( mode, vlanList, vniList=None ):
   # If a vniList is passed, we must validate the mappings
   if vniList is not None and len( vlanList ) != len( vniList ):
      mode.addError( "Number of VLANs must be equal to number of VNIs" )
      return
   validateVlanToVniRemove( mode, vlanList, vniList )

def convertVlanToVniMapSyntax( mode ):
   # New syntax must persist even if Vxlan interface is removed and readded
   vtiConfigDir.vlanVniRangeSyntax = True

#-------------------------------------------------------------------------------
# Register convertVlanToVniMapSyntax via "config convert new-syntax"
#-------------------------------------------------------------------------------
ConfigConvert.registerConfigConvertCallback( convertVlanToVniMapSyntax )

class VlanVniMultiRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vlan [ ACTION ] VLAN_LIST vni VNI_LIST'
   noOrDefaultSyntax = 'vxlan vlan VLAN_LIST vni [ VNI_LIST ]'
   data = {
      'vxlan': vxlanNode,
      'vlan': vlanVxlan1Node,
      'ACTION': CliMatcher.EnumMatcher( {
         'add': 'Add VLAN to VNI mapping',
         'remove': 'Remove VLAN to VNI mapping',
      } ),
      'VLAN_LIST': vlanMultiMatcher,
      'vni': vniMatcherForConfig,
      'VNI_LIST': vniMultiMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      vnis = args.get( 'VNI_LIST' )
      if vnis is None:
         return

      # Validate VNI list size before handlers
      vniList = list( vnis.values() )
      if len( vniList ) > 4094:
         mode.addError( "VNI list size cannot exceed VLAN range" )
         raise CliParserCommon.AlreadyHandledError
      else:
         args[ 'VNI_LIST' ] = vniList

   @staticmethod
   def handler( mode, args ):
      vlanList = list( args[ 'VLAN_LIST' ].values() )
      vniList = args[ 'VNI_LIST' ]
      if 'ACTION' in args:
         updateVlanToVniMapping( mode, vlanList, vniList, args[ 'ACTION' ] )
      else:
         setVlanToVniMapping( mode, vlanList, vniList )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vlanList = list( args[ 'VLAN_LIST' ].values() )
      vniList = args.get( 'VNI_LIST' )
      noSetVlanToVniMapping( mode, vlanList, vniList )

VxlanIntfModelet.addCommandClass( VlanVniMultiRangeCmd )

def vniEligibleForVrfMapping( mode, vrfName, vniDottedFormat, decap=False ):
   '''
   Verify if the received VNI is valid and eligible to be vrf mapped
   '''
   vtiConfig = getVtiConfig( mode.intf.name )
   vni = VniFormat( vniDottedFormat ).toNum()
   if not isValidVniWithError( mode, vni, mode.intf.name ):
      return None

   if not mode.session_.startupConfig():
      for ( existingVrf, existingVni ) in vtiConfig.vrfToVniMap.items():
         if existingVni == vni and existingVrf != vrfName:
            mode.addError( "VRF %s is already mapped to VNI %d" %
                           ( existingVrf, vni ) )
            return None
         if decap and existingVni == vni:
            mode.addError( "VRF %s is already mapped to VNI %d" %
                           ( existingVrf, vni ) )
            return None
      if not decap and vni in vtiConfig.alternateDecapVniToVrfMap:
         mode.addError( "VRF %s is already mapped to alternate VNI %d" %
                        ( vtiConfig.alternateDecapVniToVrfMap[ vni ], vni ) )
         return None

   # If the VNI is used in a vlanToVniMap, then reject
   for vlan, mappedVni in six.iteritems( vtiConfig.vlanToVniMap ):
      if mappedVni == vni:
         mode.addError( "VNI %s is already used to map vlan %d" %
                        ( vniDottedFormat, vlan ) )
         return None
   return vni

#-------------------------------------------------------------------------------
# The "[no|default] vxlan vrf <vlanId> vni <vni>, in "config-if-Vx"
#
# sets the vni to vlan mapping
#-------------------------------------------------------------------------------

def setVrfToVniMapping( mode, args ):
   vniDottedFormat = args[ 'VNI' ]
   vrfName = args[ 'VRF' ]
   vtiConfig = getVtiConfig( mode.intf.name )
   configTag = args.get( 'CONFIG_TAG', None )
   ConfigTagCommon.commandTagConfigCheck( mode, configTag )
   vni = vniEligibleForVrfMapping( mode, vrfName, vniDottedFormat )
   if not vni:
      return
   vrfVni = Tac.const( Tac.newInstance( 'Vxlan::VrfVniConfigTagKey',
      vrfName, vni ) )

   # remove previous config tag association with vrf if they are no longer valid
   oldVni = vtiConfig.vrfToVniMap.get( vrfName, None )
   if oldVni:
      oldVrfVni = Tac.const( Tac.newInstance( 'Vxlan::VrfVniConfigTagKey',
         vrfName, oldVni ) )
      oldCommandTag = vtiConfig.vrfVniConfigTagConfig.get( oldVrfVni, None )
      if oldCommandTag:
         # avoid sysdb notification if config remains same
         if oldVni == vni:
            if configTag != oldCommandTag:
               del vtiConfig.vrfVniConfigTagConfig[ oldVrfVni ]
         else:
            del vtiConfig.vrfVniConfigTagConfig[ oldVrfVni ]

   # create vrf to vni mapping
   vtiConfig.vrfToVniMap[ vrfName ] = vni

   # create ( vrf, vni ) association with config tag
   if configTag:
      vtiConfig.vrfVniConfigTagConfig[ vrfVni ] = configTag
      # create an input for configTag if it doesn't exist
      res = configTagInputAllocator().newConfigTagInputEntry( configTagInput,
            configTag, vrfName )
      if not res:
         mode.addErrorAndStop( configTagInputAllocator().errorMsg )

def noSetVrfToVniMapping( mode, args ):
   vrfName = args[ 'VRF' ]
   vniDottedFormat = args.get( 'VNI' )
   vtiConfig = getVtiConfig( mode.intf.name )

   # If vrf is not mapped, ignore the delete request
   if vrfName not in vtiConfig.vrfToVniMap:
      return

   if vniDottedFormat is not None:
      vni = VniFormat( vniDottedFormat ).toNum()
      if not isValidVniWithError( mode, vni, mode.intf.name ):
         return

      if vtiConfig.vrfToVniMap[ vrfName ] != vni:
         return

   # delete vrf to vni mapping
   if vrfName in vtiConfig.vrfToVniMap:
      vni = vtiConfig.vrfToVniMap[ vrfName ]
   del vtiConfig.vrfToVniMap[ vrfName ]

   # delete ( vrf, vni ) association with config tag if it exists
   vrfVni = Tac.const( Tac.newInstance( 'Vxlan::VrfVniConfigTagKey', vrfName, vni ) )
   vrfVniConfigTag = vtiConfig.vrfVniConfigTagConfig.get( vrfVni, None )
   if vrfVniConfigTag:
      del vtiConfig.vrfVniConfigTagConfig[ vrfVni ]

def setVrfToIp( mode, args ):
   vrfName = args[ 'VRF' ]
   addrWithMask = Arnet.AddrWithMask( args[ 'IP_ADDR' ] )
   # Need to convert to IpAddr because addrWithMask.address returns a string in
   # python
   addr = Arnet.IpAddr( addrWithMask.address )
   vtiConfig = getVtiConfig( mode.intf.name )

   # Skip sanity checks during startup
   if not mode.session.skipConfigCheck():
      # Nothing to do
      if vtiConfig.vrfToIpAddr.get( vrfName ) == addrWithMask:
         return

      if addr.isZero:
         mode.addError( 'Address must not be zero' )
         return

      if addr.isMulticast or addr.isBroadcast:
         mode.addError( 'Address must be unicast' )
         return

      if addrWithMask.len < 31 and addrWithMask.address == addrWithMask.allZerosAddr:
         mode.addError( 'Address must not have a zero host number' )
         return

      if IpAddrMatcher.isMartianIpAddr( addrWithMask.address ):
         mode.addError( 'Not a valid host address' )
         return

      if IpAddrMatcher.isInvalidThisNetworkAddress( addrWithMask ):
         mode.addError( 'Not a valid local router address or subnet mask' )
         return

      # Check for overlap with other configured IP addresses. We cannot use
      # Ira::AddressAssignmentChecker here because we might not have allocated a
      # VLAN yet which means that we don't know what the interface name should be.
      def shouldReject( other, intfId ):
         if other.len == 0:
            return False

         if other.address == addrWithMask.address:
            mode.addError( 'Address %s is already assigned to interface %s' %
                           ( addr, intfId ) )
            return True

         if other.contains( addr ) or addrWithMask.contains( other.address ):
            mode.addError(
               'Subnet %s overlaps with existing subnet %s of interface %s' %
               ( addrWithMask, other, intfId ) )
            return True

         return False

      for ipIntfConfig in ipConfig.ipIntfConfig.values():
         # Ignore interface if it is the interface we're configuring now or if it
         # is in a different VRF (in which case IP addresses may overlap)
         if ipIntfConfig.configSource == 'evpn' or ipIntfConfig.vrf != vrfName:
            continue

         if shouldReject( ipIntfConfig.addrWithMask, ipIntfConfig.intfId ):
            return

         if shouldReject( ipIntfConfig.virtualAddrWithMask, ipIntfConfig.intfId ):
            return

         for secondary in ipIntfConfig.secondaryWithMask:
            if shouldReject( secondary, ipIntfConfig.intfId ):
               return

         for virtual in ipIntfConfig.virtualSecondaryWithMask:
            if shouldReject( virtual, ipIntfConfig.intfId ):
               return

   vtiConfig.vrfToIpAddr[ vrfName ] = addrWithMask

def noSetVrfToIp( mode, args ):
   vrfName = args[ 'VRF' ]
   vtiConfig = getVtiConfig( mode.intf.name )
   del vtiConfig.vrfToIpAddr[ vrfName ]

def setVrfToIp6( mode, args ):
   vrfName = args[ 'VRF' ]
   addrWithMask = Arnet.Ip6AddrWithMask( args[ 'IP6_ADDR' ] )
   addr = addrWithMask.address
   vtiConfig = getVtiConfig( mode.intf.name )

   # Skip sanity checks during startup
   if not mode.session.skipConfigCheck():
      # Nothing to do
      if vtiConfig.vrfToIp6Addr.get( vrfName ) == addrWithMask:
         return

      if addr.isMulticast:
         mode.addError( 'Address must be unicast' )
         return

      if addr.isUnspecified:
         mode.addError( 'Invalid unicast address' )
         return

      if addr.isLoopback:
         mode.addError( 'Invalid address for this interface' )
         return

      if addr.isLinkLocal:
         mode.addError( 'Cannot configure link local address on this interface' )
         return

      for intfConfig in ip6Config.intf.values():
         # This is the interface we're configuring now
         if intfConfig.configSource == 'evpn':
            continue

         # IP addresses in different VRFs may overlap
         if intfConfig.vrf != vrfName:
            continue

         # Check for overlap with other configured IPv6 addresses. We cannot use
         # Ira::Address6AssignmentChecker here because we might not have allocated a
         # VLAN yet which means that we don't know what the interface name should be.
         for other in chain( intfConfig.addr, intfConfig.virtualAddr ):
            if other.address == addrWithMask.address:
               mode.addError( 'Address %s is already assigned to interface %s' %
                              ( addr, intfConfig.intfId ) )
               return

            if addrWithMask.overlaps( other ):
               mode.addError(
                  'Subnet %s overlaps with existing subnet %s of interface %s' %
                  ( addrWithMask, other, intfConfig.intfId ) )
               return

   vtiConfig.vrfToIp6Addr[ vrfName ] = addrWithMask

def noSetVrfToIp6( mode, args ):
   vrfName = args[ 'VRF' ]
   vtiConfig = getVtiConfig( mode.intf.name )
   del vtiConfig.vrfToIp6Addr[ vrfName ]

def removePrevConfigTagVrfVniAssociation( mode, vrfName, vni, configTag ):
   vtiConfig = getVtiConfig( mode.intf.name )
   oldVrfName = vtiConfig.alternateDecapVniToVrfMap.get( vni, None )
   if oldVrfName:
      oldVrfVni = Tac.const( Tac.newInstance( 'Vxlan::VrfVniConfigTagKey',
         oldVrfName, vni ) )
      oldCommandTag = vtiConfig.vrfVniConfigTagConfig.get( oldVrfVni, None )
      if oldCommandTag:
         # avoid sysdb notification if config remains same
         if oldVrfName == vrfName:
            if configTag != oldCommandTag:
               del vtiConfig.vrfVniConfigTagConfig[ oldVrfVni ]
         else:
            del vtiConfig.vrfVniConfigTagConfig[ oldVrfVni ]

def mapDecapVniAdd( mode, vrfName, vniSet, configTag=None ):
   vtiConfig = getVtiConfig( mode.intf.name )
   validVnis = []
   for vniDottedFormat in vniSet:
      vni = vniEligibleForVrfMapping( mode, vrfName, vniDottedFormat, decap=True )
      if not vni:
         # A warning is already displayed in vniEligibleForVrfMapping()
         # No need to do it again
         return
      validVnis.append( vni )
   for vni in validVnis:
      vrfVni = Tac.const( Tac.newInstance( 'Vxlan::VrfVniConfigTagKey',
         vrfName, vni ) )
      if ConfigTagCommon.configTagSupportedByPlatform( mode, configTag ):
         removePrevConfigTagVrfVniAssociation( mode, vrfName, vni, configTag )
      vtiConfig.alternateDecapVniToVrfMap[ vni ] = vrfName
      if configTag:
         # create ( vrf, vni ) association with config tag
         vtiConfig.vrfVniConfigTagConfig[ vrfVni ] = configTag
         # create an input for configTag if it doesn't exist
         res = configTagInputAllocator().newConfigTagInputEntry( configTagInput,
               configTag, vrfName )
         if not res:
            mode.addErrorAndStop( configTagInputAllocator().errorMsg )

def mapDecapVniRemoveConfigTag( mode, vrfName, validVnis ):
   vtiConfig = getVtiConfig( mode.intf.name )
   for vni in validVnis:
      vrfVni = Tac.const( Tac.newInstance( 'Vxlan::VrfVniConfigTagKey',
         vrfName, vni ) )
      # delete ( vrf, vni ) association with config tag if it exists
      vrfVniConfigTag = vtiConfig.vrfVniConfigTagConfig.get( vrfVni, None )
      if vrfVniConfigTag:
         del vtiConfig.vrfVniConfigTagConfig[ vrfVni ]

def mapDecapVniRemove( mode, vrfName, vniSet ):
   vtiConfig = getVtiConfig( mode.intf.name )
   validVnis = []
   for vniDottedFormat in vniSet:
      if not vniDottedFormat:
         return
      vni = VniFormat( vniDottedFormat ).toNum()
      if not isValidVniWithError( mode, vni, mode.intf.name ):
         return
      if ( vni not in vtiConfig.alternateDecapVniToVrfMap or
            vrfName != vtiConfig.alternateDecapVniToVrfMap[ vni ] ):
         continue
      validVnis.append( vni )
   for vni in validVnis:
      del vtiConfig.alternateDecapVniToVrfMap[ vni ]
   if ConfigTagCommon.configTagSupportedByPlatform( mode, None ):
      mapDecapVniRemoveConfigTag( mode, vrfName, validVnis )

def getVniSet( vniList ):
   # Multiple space separated
   if not vniList:
      return set()
   return set( vniList )

def setDefaultVrfToMultipleVniMapping( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vrfName = args[ 'VRF' ]
   configTag = args.get( 'CONFIG_TAG', None )
   ConfigTagCommon.commandTagConfigCheck( mode, configTag )
   vnis = getVniSet( args[ 'DECAPVNI' ] )
   vrfDecapVnis = set()
   for vni, vrf in six.iteritems( vtiConfig.alternateDecapVniToVrfMap ):
      if vrf == vrfName:
         vrfDecapVnis.add( vni )
   removeVnis = vrfDecapVnis - vnis
   # add all the vnis again to allow updating configTags
   addVnis = vnis
   mapDecapVniAdd( mode, vrfName, addVnis, configTag )
   mapDecapVniRemove( mode, vrfName, removeVnis )

def setVrfToMultipleVniMapping( mode, args ):
   addOrRemove = args[ 'ACTION' ]
   vrfName = args[ 'VRF' ]
   configTag = args.get( 'CONFIG_TAG', None )
   ConfigTagCommon.commandTagConfigCheck( mode, configTag )
   vniSet = getVniSet( args[ 'DECAPVNI' ] )
   if addOrRemove == 'add':
      mapDecapVniAdd( mode, vrfName, vniSet, configTag )
   else:
      mapDecapVniRemove( mode, vrfName, vniSet )

def noSetVrfToMultipleVniMapping( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vrfName = args[ 'VRF' ]
   vniSet = getVniSet( args.get( 'DECAPVNI', [] ) )
   validVnis = []
   for vniDottedFormat in vniSet:
      vni = VniFormat( vniDottedFormat ).toNum()
      if not vni:
         mode.addError( "Invalid VNI %s entered" % ( str( vni ) ) )
         return
      if vni not in vtiConfig.alternateDecapVniToVrfMap or \
         vtiConfig.alternateDecapVniToVrfMap[ vni ] != vrfName:
         continue
      validVnis.append( vni )
   if not validVnis:
      # Delete all decap VNIs
      for vni, vrf in vtiConfig.alternateDecapVniToVrfMap.items():
         if vrf == vrfName:
            del vtiConfig.alternateDecapVniToVrfMap[ vni ]
            vrfVni = Tac.const( Tac.newInstance( 'Vxlan::VrfVniConfigTagKey',
               vrfName, vni ) )
            vrfVniConfigTag = vtiConfig.vrfVniConfigTagConfig.get( vrfVni, None )
            if vrfVniConfigTag:
               del vtiConfig.vrfVniConfigTagConfig[ vrfVni ]
      return
   for vni in validVnis:
      # Delete individual decap VNIs but not all
      del vtiConfig.alternateDecapVniToVrfMap[ vni ]
      vrfVni = Tac.const( Tac.newInstance( 'Vxlan::VrfVniConfigTagKey',
         vrfName, vni ) )
      vrfVniConfigTag = vtiConfig.vrfVniConfigTagConfig.get( vrfVni, None )
      if vrfVniConfigTag:
         del vtiConfig.vrfVniConfigTagConfig[ vrfVni ]

class VrfToMultipleVniConfigTagState(
      ConfigTagCommon.ConfigTagDependentBase ):
   def processTaggedMapping( self, mode, tag, removeOrDisassociate ):
      # Check if the tag exists in commandTag CLI input.
      if not configTagInput.configTagEntry.get( tag, None ):
         return
      # Remove the association of the tag with the vrf/vni mappings
      # and remove the mappings as well if keepConfig is false.
      for vti in vtiConfigDir.vtiConfig:
         vtiConfig = getVtiConfig( vti )

         def processVrfVni( vrf, vni, alternateDecap=False ):
            vrfVni = VrfVniConfigTagKey( vrf, vni )
            configTag = vtiConfig.vrfVniConfigTagConfig.get( vrfVni )
            if configTag == tag:
               if removeOrDisassociate == RemoveOrDisassociate.remove:
                  if alternateDecap:
                     del vtiConfig.alternateDecapVniToVrfMap[ vni ]
                  else:
                     del vtiConfig.vrfToVniMap[ vrf ]
               elif removeOrDisassociate == RemoveOrDisassociate.disassociate:
                  pass
               else:
                  assert False, "unhandled action"
               del vtiConfig.vrfVniConfigTagConfig[ vrfVni ]

         for vrf, vni in vtiConfig.vrfToVniMap.items():
            processVrfVni( vrf, vni )
         for vni, vrf in vtiConfig.alternateDecapVniToVrfMap.items():
            processVrfVni( vrf, vni, alternateDecap=True )

   def removeTaggedConfig( self, mode, tag ):
      self.processTaggedMapping( mode, tag, RemoveOrDisassociate.remove )

   def disassociateConfigFromTag( self, mode, tag ):
      self.processTaggedMapping( mode, tag, RemoveOrDisassociate.disassociate )

   def processAndValidateConfig( self, mode, commandTagInfo ):
      pass

ConfigTagCommon.ConfigTagState.registerConfigTagSupportingClass(
      VrfToMultipleVniConfigTagState )

class VrfToMultipleVniMappingCmd( CliCommand.CliCommandClass ):
   syntax = '''vxlan VRF vni
               ( ( VNI )
               | ( decapsulation [ ACTION ] DECAPVNI ) )
               [ CONFIG_TAG_EXPR ]
               '''
   noOrDefaultSyntax = '''vxlan VRF vni
                          [ ( VNI )
                          | ( decapsulation [ DECAPVNI ] ) ]
                          [ CONFIG_TAG_EXPR ]
                          '''
   data = {
      'vxlan': vxlanNode,
      'VRF': vrfExprFactory,
      'vni': vniMatcherForConfig,
      'VNI': vniMatcher,
      'decapsulation': vniDecapsulationMatcher,
      'ACTION': vniAddRemoveMatcher,
      'DECAPVNI': vniMultiMatcher,
      'CONFIG_TAG_EXPR': configTagExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      vnis = args.get( 'DECAPVNI' )
      if vnis is None:
         args[ 'DECAPVNI' ] = []
      else:
         vniList = list( vnis.values() )
         args[ 'DECAPVNI' ] = vniList

   @staticmethod
   def handler( mode, args ):
      configTag = args.get( 'CONFIG_TAG', None )
      if configTag:
         ConfigTagCommon.checkTagInRemovedOrDisassociatedState( mode,
               configTag, featureCheck=True )
      if 'VNI' in args:
         # Evpn Vrf to Vni map
         setVrfToVniMapping( mode, args )
      elif 'ACTION' in args:
         setVrfToMultipleVniMapping( mode, args )
      else:
         setDefaultVrfToMultipleVniMapping( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'decapsulation' not in args:
         noSetVrfToVniMapping( mode, args )
      else:
         noSetVrfToMultipleVniMapping( mode, args )

VxlanIntfModelet.addCommandClass( VrfToMultipleVniMappingCmd )

class VrfToIpMappingCmd( CliCommand.CliCommandClass ):
   syntax = '''vxlan VRF
               ( ( ip address IP_ADDR )
               | ( ipv6 address IP6_ADDR ) )'''
   noOrDefaultSyntax = 'vxlan VRF ( ( ip ... ) | ( ipv6 ... ) )'
   data = {
      'vxlan': vxlanNode,
      'VRF': vrfExprFactory,
      'ip': 'Dynamic SVI IPv4 configuration',
      'ipv6': 'Dynamic SVI IPv6 configuration',
      'address': 'IP address',
      'IP_ADDR': IpAddrMatcher.ipPrefixMatcher,
      'IP6_ADDR': Ip6AddrMatcher.ip6PrefixMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if 'IP_ADDR' in args:
         setVrfToIp( mode, args )
      elif 'IP6_ADDR' in args:
         setVrfToIp6( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'ip' in args:
         noSetVrfToIp( mode, args )
      elif 'ipv6' in args:
         noSetVrfToIp6( mode, args )

if Toggles.VxlanToggleLib.toggleVxlanDynSviIpEnabled():
   VxlanIntfModelet.addCommandClass( VrfToIpMappingCmd )
#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# The "[no] mac address-table static vlan <vlan> interface Vxlan<n>
#           vtep <ipAddr>
#
# configures a static host in vxlan fdb in global config mode
#-------------------------------------------------------------------------------
def setStaticVxlanHost( mode, args ):
   macAddr = args[ 'MAC_ADDR' ]
   vlanId = args[ 'VLAN_ID' ]
   intfList = args[ 'INTFS' ]
   vtepAddr = args[ 'VTEP' ]

   if not Ethernet.isUnicast( macAddr ):
      mode.addError( 'Adding a static multicast address is not supported' )
      return

   # FIXME: Why use `IntfRangeMatcher`?
   intfNames = list( intfList )
   if len( intfNames ) > 1:
      mode.addError( 'Multiple interfaces are not allowed' )
      return

   intfName = intfNames.pop()
   if isVtiMissing( intfName ):
      mode.addError( 'Interface %s does not exist' % intfName )
      return

   if ( vtepAddr.isAddrZero or not vtepAddr.isUnicast ):
      mode.addError( 'Invalid unicast address for VTEP' )
      return

   macAddr = Ethernet.convertMacAddrToCanonical( macAddr )

   macVlanPair = Tac.Value( "Vxlan::MacVlanPair", macAddr, vlanId.id )
   vti = vtiNameToId.get( intfName )

   vxlanConfigDir.fdbConfig.configuredHost.addMember( Tac.Value(
         "Vxlan::ConfiguredHost", macVlanPair, vti, vtepAddr ) )

   # delete l2 static mac entry
   fdbConfig = bridgingCliConfig.fdbConfig.newMember( vlanId.id )
   table = fdbConfig.configuredHost
   if table.get( macAddr ):
      del table[ macAddr ]
   # Hook is populated on Trident4 platforms to warn of conflicting configuration
   BridgingCli.warnMacTableUnsupportedUFTModeHook.notifyExtensions( mode )

def noSetStaticVxlanHost( mode, macAddr, vlanId, intfList ):
   macAddr = Ethernet.convertMacAddrToCanonical( macAddr )
   macVlanPair = Tac.Value( "Vxlan::MacVlanPair", macAddr, vlanId.id )
   configuredHost = vxlanConfigDir.fdbConfig.configuredHost

   if configuredHost.get( macVlanPair ):
      del configuredHost[ macVlanPair ]
      return True
   else:
      if mode:
         mode.addWarning( 'Address not found' )
      return False

def filterVxlanConfiguredMacAddr( host, vlanId, macAddr, vteps ):
   if vlanId:
      if vlanId != host.macVlanPair.vlanId:
         return False
   if macAddr:
      if Ethernet.convertMacAddrToCanonical( macAddr ) != \
             host.macVlanPair.macAddr:
         return False
   if vteps:
      if host.remoteVtepGenAddr not in vteps:
         return False
   return True

# verify that intf is not vxlan
def bridgingOkToAddHandler( mode, macAddr, vlanId, intfNamesOrDrop=None ):
   if intfNamesOrDrop is None or intfNamesOrDrop == 'drop':
      return True
   for intfName in intfNamesOrDrop:
      if 'Vxlan' in intfName:
         if len( intfNamesOrDrop ) == 1:
            mode.addError( "vtep must be specified for %s" % intfName )
         else:
            mode.addError( "%s is not allowed with other interfaces" % intfName )
         return False
   return True

# When ebra static mac is added to one of the local interfaces,
# remove corresponding vxlan static mac address.
def bridgingSetStaticHandler( macAddr, vlanId, intfsOrDrop=None ):
   # remove the vxlan host for the same macVlan pair
   noSetStaticVxlanHost( None, macAddr, vlanId, None )
   return False

# When ebra static mac is removed via 'no mac addr static vlan <vlan>', i.e.,
# without vtep token, Ebra CliPlugin will handle the command and
# notify vxlan to remove the addr
def bridgingDelStaticHandler( macAddr, vlanId, intfsOrDrop=None ):
   return noSetStaticVxlanHost( None, macAddr, vlanId, None )

# When 'show mac address-table configured' is issued, Ebra CliPlugin
# will inform Vxlan through this function to display Vxlan configured
# entries
def bridgingGetConfiguredHandler( macAddr=None, vlanId=None,
                                  intfsOrDrop=None, vteps=None ):

   # Get the configured host object for static vxlan entries
   configuredHosts = vxlanConfigDir.fdbConfig.configuredHost

   vxlanMatchedHosts = []
   bridgingRemoteHosts = []
   if macAddr and vlanId:
      macVlanPair = Tac.Value( "Vxlan::MacVlanPair", macAddr, vlanId )
      macEntry = configuredHosts.get( macVlanPair )
      if macEntry:
         if filterVxlanConfiguredMacAddr( macEntry, vlanId,
                                          macAddr, vteps ):
            vxlanMatchedHosts = [ macEntry ]
   else:
      vxlanMatchedHosts = [ macEntry for macEntry in
                            six.itervalues( configuredHosts )
                            if filterVxlanConfiguredMacAddr( macEntry,
                                                             vlanId,
                                                             macAddr,
                                                             vteps ) ]
   # Create new mac address dictionary, as expected by Ebra
   # for configured mac entries
   for host in vxlanMatchedHosts:
      bridgingRemoteHosts.append( Tac.Value(
         "Bridging::ConfiguredHost", address=host.macVlanPair.macAddr,
         intf=host.vtiIntfId, entryType='configuredStaticMac' ) )

   return bridgingRemoteHosts

# Register the static (configured) mac config hooks with the Ebra
# provided hooks.
BridgingCli.bridgingCheckStaticMacHook.addExtension( bridgingOkToAddHandler )
BridgingCli.bridgingAddStaticMacHook.addExtension( bridgingSetStaticHandler )
BridgingCli.bridgingDelStaticMacHook.addExtension( bridgingDelStaticHandler )
BridgingCli.bridgingExtraStaticMacHook.addExtension( bridgingGetConfiguredHandler )

vtepGenAddrNode = CliCommand.Node(
      IpGenAddrMatcher.IpGenAddrMatcher( 'IPv4 or IPv6 address of remote VTEP' ),
      guard=vxlanV6RemoteVtepSupportedGuard )

#--------------------------------------------------------------------------------
# [ no | default ] ( ( mac address-table ) | mac-address-table ) static MAC_ADDR
#                                              vlan VLAN_ID interface INTFS vtep VTEP
#--------------------------------------------------------------------------------
class SetStaticVxlanHostCmd( CliCommand.CliCommandClass ):
   syntax = 'MAC_ADDR_TABLE static MAC_ADDR vlan VLAN_ID interface INTFS vtep VTEP'
   noOrDefaultSyntax = syntax.replace( 'VTEP', '...' )
   data = {
      'MAC_ADDR_TABLE': BridgingCli.MacAddrTableExprForConfig,
      'static': BridgingCli.staticKwMatcher,
      'MAC_ADDR': MacAddr.macAddrMatcher,
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'interface': BridgingCli.matcherInterface,
      'INTFS': IntfRange.IntfRangeMatcher( explicitIntfTypes=vxlanIntfType ),
      'vtep': vtepMatcherForConfig,
      'VTEP': vtepGenAddrNode,
   }
   handler = setStaticVxlanHost

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noSetStaticVxlanHost( mode, args[ 'MAC_ADDR' ], args[ 'VLAN_ID' ],
                            args[ 'INTFS' ] )

BasicCli.GlobalConfigMode.addCommandClass( SetStaticVxlanHostCmd )

IntfCli.IntfConfigMode.addModelet( VxlanIntfModelet )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan ttl <ttl>" command, in "config-vxlan-if"
#
# sets the TTL for multicast forwarded Vxlan traffic
#-------------------------------------------------------------------------------
class TtlCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan ttl TTL'
   data = {
            'vxlan': vxlanNode,
            'ttl': 'Multicast TTL',
            'TTL': CliMatcher.IntegerMatcher( 0, 255, helpdesc='Multicast TTL' )
          }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.ttl = args[ 'TTL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.ttl = vtiConfig.defaultTtl

# BUG48350
#VxlanIntfModelet.addCommandClass( TtlCmd )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan udp-port <port>" command, in "config-vxlan-if"
#
# sets the UDP port for Vxlan encapsulated traffic
#-------------------------------------------------------------------------------
def setUdpPort( mode, args ):
   udpPort = args[ 'UDP_PORT' ]
   if 0 <= udpPort <= 1023:
      mode.addError( 'invalid udp-port ' )
      return
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.udpPort = udpPort

def noSetUdpPort( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.udpPort = vtiConfig.vxlanWellKnownPort

class VxlanUdpPortUdpportCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan udp-port UDP_PORT'
   noOrDefaultSyntax = 'vxlan udp-port [ UDP_PORT ]'
   data = {
      'vxlan': vxlanNode,
      'udp-port': matcherUdpPort,
      'UDP_PORT': CliMatcher.IntegerMatcher( 1024, 65535, helpdesc='UDP port' ),
   }
   handler = setUdpPort
   noOrDefaultHandler = noSetUdpPort

VxlanIntfModelet.addCommandClass( VxlanUdpPortUdpportCmd )

#-------------------------------------------------------------------------------
# "[no] vxlan source-vtep-validation [ ACTION ] { VTEP } , in "config-vxlan-if"
#
# sets the list of VTEPs for SIP validation
# -------------------------------------------------------------------------------
def sipValidationVtepsAdd( vteps ):
   for vtep in vteps:
      vtepSipValidationStatusCli.remoteVtepAddr[ vtep ] = 1

def sipValidationVtepsRemove( vteps ):
   for vtep in vteps:
      vtepAddr = \
         vtepSipValidationStatusCli.remoteVtepAddr.get( vtep, None )
      if vtepAddr:
         del vtepSipValidationStatusCli.remoteVtepAddr[ vtep ]

def setSipValidationVteps( mode, args ):
   addOrRemove = args[ 'ACTION' ]
   vteps = args[ 'VTEPS' ]
   if addOrRemove == 'add':
      sipValidationVtepsAdd( vteps )
   else:
      sipValidationVtepsRemove( vteps )

def setDefaultSipValidationVteps( mode, args ):
   # replace the current list of VTEPs with the given list
   vteps = set( args[ 'VTEPS' ] )
   currVteps = set( vtepSipValidationStatusCli.remoteVtepAddr )
   removeVteps = currVteps - vteps
   sipValidationVtepsRemove( removeVteps )
   sipValidationVtepsAdd( vteps )

def noSetSipValidationVteps( mode, args ):
   vteps = args.get( 'VTEPS', vtepSipValidationStatusCli.remoteVtepAddr )
   sipValidationVtepsRemove( vteps )

class VxlanSipValidationCmd( CliCommand.CliCommandClass ):
   syntax = '''vxlan source-vtep-validation [ ACTION ] { VTEPS }'''
   noOrDefaultSyntax = '''vxlan source-vtep-validation [ ACTION ] [ { VTEPS } ]'''

   data = {
      'vxlan': vxlanNode,
      'source-vtep-validation': sipValidationNode( vxlanMode=True ),
      'ACTION': addRemoveMatcher,
      'VTEPS': IpGenAddrMatcher.IpGenAddrMatcher(
         helpdesc='IP address of remote VTEP' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'ACTION' in args:
         setSipValidationVteps( mode, args )
      else:
         setDefaultSipValidationVteps( mode, args )

   noOrDefaultHandler = noSetSipValidationVteps

VxlanIntfModelet.addCommandClass( VxlanSipValidationCmd )

# -------------------------------------------------------------------------------
# The "[no|default] vxlan udp-port source offset <offset> range <range>" command,
# in "config-vxlan-if"
#
# sets the source UDP port range for Vxlan encapsulated traffic encapsulated by
# the switch
#-------------------------------------------------------------------------------
def setUdpSourcePortRange( mode, args ):
   offset = args[ 'OFFSET' ]
   portLength = args[ 'PORT_LENGTH' ]

   if portLength + offset > 0x10000:
      mode.addError( 'offset + length must not be greater than 0x10000' )
      return
   if bridgingHwCapabilities.vxlanSrcPortSupported == \
         'offsetAllValuesRangePowerOfTwo':
      # verify that portLength is a power of 2, 0 is allowed
      if bin( portLength ).count( '1' ) > 1:
         mode.addError( 'length must be a power of 2, e.g. 0x1000' )
         return
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.srcPortRange = \
         Tac.Value( "Vxlan::VxlanSrcPortRange", offset, portLength )

def noSetUdpSourcePortRange( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.srcPortRange = Tac.Value( "Vxlan::VxlanSrcPortRange" )

class UdpSourcePortRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan udp-port source offset OFFSET length PORT_LENGTH'
   noOrDefaultSyntax = 'vxlan udp-port source offset ...'
   data = {
      'vxlan': vxlanNode,
      'udp-port': matcherUdpPort,
      'source': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'source',
            helpdesc='Set source port range for VXLAN encapsulated packets' ),
         guard=vxlanSrcPortSupported ),
      'offset': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'offset',
            helpdesc='Start of source port range' ),
         guard=vxlanRangeSrcPortSupported ),
      'OFFSET': CliMatcher.IntegerMatcher( 0, 65535, helpdesc='UDP port offset' ),
      'length': 'Length of source port range',
      'PORT_LENGTH': CliMatcher.IntegerMatcher( 0, 65535,
         helpdesc='UDP port length' ),
   }
   handler = setUdpSourcePortRange
   noOrDefaultHandler = noSetUdpSourcePortRange

VxlanIntfModelet.addCommandClass( UdpSourcePortRangeCmd )

# -------------------------------------------------------------------------------
# The "[no|default] vxlan udp-port source <udpSourcePort>" command,
# in "config-vxlan-if"
#
# sets a fixed source UDP port for Vxlan encapsulated traffic encapsulated by
# the switch
# -------------------------------------------------------------------------------
def setFixedUdpSourcePort( mode, args ):
   offset = args[ 'UDP_SRC_PORT' ]
   portLength = 1
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.srcPortRange = \
         Tac.Value( "Vxlan::VxlanSrcPortRange", offset, portLength )

class VxlanFixedUdpSourcePortCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan udp-port source UDP_SRC_PORT'
   noOrDefaultSyntax = 'vxlan udp-port source ...'
   data = {
      'vxlan': vxlanNode,
      'udp-port': matcherUdpPort,
      'source': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'source',
            helpdesc='Set source port range for VXLAN encapsulated packets' ),
         guard=vxlanSrcPortSupported ),
      'UDP_SRC_PORT': CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 1024, 65535,
                                                helpdesc='UDP source port' ),
         guard=vxlanFixedSrcPortSupported ),
   }
   handler = setFixedUdpSourcePort
   noOrDefaultHandler = noSetUdpSourcePortRange

VxlanIntfModelet.addCommandClass( VxlanFixedUdpSourcePortCmd )

#-------------------------------------------------------------------------------
# "[no|default] vxlan vtep ipv4 address-mask <mask>" command, in "config-vxlan-if"
#-------------------------------------------------------------------------------
def setVtepIpv4AddressMask( mode, args ):
   addressMask = args[ 'MASK' ]
   try:
      Arnet.Mask( addressMask )
   except ValueError:
      mode.addError( "invalid address mask: " + addressMask )
      return
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.vtepAddrMask = Arnet.IpAddress( addressMask )

def noSetVtepIpv4AddressMask( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.vtepAddrMask = Tac.newInstance( "Arnet::IpAddr", 0xFFFFFFFF )

class VxlanVtepIpv4AddressMaskMaskCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vtep ipv4 address-mask MASK'
   noOrDefaultSyntax = 'vxlan vtep ipv4 address-mask ...'
   data = {
      'vxlan': vxlanNode,
      'vtep': CliCommand.Node( matcher=vtepMatcherForConfig,
         guard=vxlanVtepMaskCliSupported ),
      'ipv4': 'Configure VXLAN vtep IPv4',
      'address-mask': 'Configure vtep address-mask',
      'MASK': IpAddrMatcher.IpAddrMatcher( helpdesc='address mask' ),
   }
   handler = setVtepIpv4AddressMask
   noOrDefaultHandler = noSetVtepIpv4AddressMask

VxlanIntfModelet.addCommandClass( VxlanVtepIpv4AddressMaskMaskCmd )

def setVtepsForSourcePruning( vtiConfig, vteps ):
   for v in vtiConfig.vtepSetForSourcePruning:
      if v not in vteps:
         vtiConfig.vtepSetForSourcePruning.remove( v )
   for v in vteps:
      if v not in vtiConfig.vtepSetForSourcePruning:
         vtiConfig.vtepSetForSourcePruning.add( v )

#-------------------------------------------------------------------------------
# "[no|default] vxlan bridging vtep-to-vtep [ source-vtep tx disabled
# [ { VTEPS } ] ]" command, in "config-vxlan-if"
#
# "[no|default] vxlan bridging vtep-to-vtep" -
# Enables vxlan vtep-to-vtep bridging.
#
# "[no|default] vxlan bridging vtep-to-vtep source-vtep tx disabled" -
# Enables source pruning for all vteps
#
# "[no|default] vxlan bridging vtep-to-vtep source-vtep tx disabled { VTEPS }" -
# Enables source pruning for the list of vteps
#-------------------------------------------------------------------------------
class VxlanVtepToVtepBridging( CliCommand.CliCommandClass ):
   syntax = 'vxlan bridging vtep-to-vtep [ source-vtep tx disabled [ { VTEPS } ] ]'
   noOrDefaultSyntax = 'vxlan bridging vtep-to-vtep [ source-vtep tx disabled ] ...'
   data = {
         'vxlan': vxlanNode,
         'bridging': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                          'bridging',
                                          helpdesc='VXLAN bridging' ),
                                      guard=vxlanVtepToVtepBridgingSupportedGuard ),
         'vtep-to-vtep': 'Enable VTEP to VTEP bridging',
         'source-vtep': 'Configure source VTEP properties',
         'tx': 'Allow bridged packets to go back to source VTEPs',
         'disabled': 'Disable bridged packets to go back to source VTEPs',
         'VTEPS': IpAddrMatcher.IpAddrMatcher( 'IP address of remote VTEP' )
   }

   @staticmethod
   def handler( mode, args ):
      # Enable Vxlan bridging
      vxlanCliClientConfig.vxlanVtepToVtepBridgingEnabled = \
            TristateBoolean.valueSet( True )
      if 'source-vtep' not in args:
         return

      vtiConfig = getVtiConfig( mode.intf.name )

      # Source pruning for all vteps
      vteps = args.get( 'VTEPS' )
      if not vteps:
         vtiConfig.vtepSourcePruningAll = True
         setVtepsForSourcePruning( vtiConfig, [] )
         return

      # Source pruning for vteps in the VTEPS list
      if not _verifyVtepAddrs( mode, vteps ):
         return
      vtiConfig.vtepSourcePruningAll = False
      setVtepsForSourcePruning( vtiConfig, vteps )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      if 'source-vtep' not in args:
         vxlanCliClientConfig.vxlanVtepToVtepBridgingEnabled = \
               TristateBoolean.valueInvalid()
      setVtepsForSourcePruning( vtiConfig, [] )
      vtiConfig.vtepSourcePruningAll = False

VxlanIntfModelet.addCommandClass( VxlanVtepToVtepBridging )

#-------------------------------------------------------------------------------
# "[no|default] vxlan encapsulation ipv6", in "config-if-Vx"
#
# By default, VXLAN encapsulation (underlay) is assumed to be of type ipv4. This
# CLI will set the encapsulation to be ipv6 for all VNIs. The 'no' form of the CLI
# will set the encapsulation back to its default value (which is ipv4).
#-------------------------------------------------------------------------------
vxlanEncapNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'encapsulation',
         helpdesc='VXLAN encapsulation type' ),
      guard=vxlanEncapIpv6SupportedGuard )

class VxlanEncapsulation( CliCommand.CliCommandClass ):
   syntax = '''vxlan encapsulation
               ( ( ipv4 [ DUAL_V6 ] ) | ( ipv6 [ DUAL_V4 ] ) ) '''
   noOrDefaultSyntax = '''vxlan encapsulation
               [ ( ( ipv4 [ DUAL_V6 ] ) | ( ipv6 [ DUAL_V4 ] ) ) ]'''
   data = {
         'vxlan': vxlanNode,
         'encapsulation': vxlanEncapNode,
         'ipv4': 'Use IPv4 for VXLAN Encapsulation',
         'ipv6': 'Use IPv6 for VXLAN Encapsulation',
         'DUAL_V6': CliMatcher.KeywordMatcher( 'ipv6',
            helpdesc="Use IPv4 and IPv6 for VXLAN Encapsulation" ),
         'DUAL_V4': CliMatcher.KeywordMatcher( 'ipv4',
            helpdesc="Use IPv4 and IPv6 for VXLAN Encapsulation" ),
   }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.vxlanEncapConfigured = True
      if 'DUAL_V6' in args or 'DUAL_V4' in args:
         vtiConfig.vxlanEncap = vxlanEncapType.vxlanEncapDual
      elif 'ipv6' in args:
         vtiConfig.vxlanEncap = vxlanEncapType.vxlanEncapIp6
      else:
         vtiConfig.vxlanEncap = vxlanEncapType.vxlanEncapIp4

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      encap = vtiConfig.vxlanEncap
      dualConfig = 'DUAL_V6' in args or 'DUAL_V4' in args
      noOption = ( 'ipv6' not in args and 'ipv4' not in args )
      if noOption or ( dualConfig and encap == vxlanEncapType.vxlanEncapDual ) or \
         ( encap == vxlanEncapType.vxlanEncapIp6 and 'ipv6' in args and
           'DUAL_V4' not in args ):
         vtiConfig.vxlanEncap = bridgingHwCapabilities.vxlanDefaultEncap
         vtiConfig.vxlanEncapConfigured = False

VxlanIntfModelet.addCommandClass( VxlanEncapsulation )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan header vni 32-bit" command, in "config-vxlan-if"
#
# Triggers the switch to treat the reserved 8-bits in the vxlan header as an
# extension to form a 32-bit VNI
#-------------------------------------------------------------------------------
def set32BitVni( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.use32BitVni = True

def noSet32BitVni( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.use32BitVni = False

class VxlanHeaderVni32BitCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan header vni 32-bit'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'header': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'header',
            helpdesc='VXLAN header extensions' ),
         guard=vxlan32BitVniSupportedGuard ),
      'vni': vniMatcherForConfig,
      '32-bit': 'Allow 32-bit VXLAN Network Identifier Configuration',
   }
   handler = set32BitVni
   noOrDefaultHandler = noSet32BitVni

VxlanIntfModelet.addCommandClass( VxlanHeaderVni32BitCmd )

#-------------------------------------------------------------------------------
# [ no | default ] vxlan controller-client
# in "config-vxlan-if"
#
# sets the switch in vxlan controller client mode
#-------------------------------------------------------------------------------
def setControllerMode( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.vccDataPathLearning = False
   vtiConfig.controllerClientMode = True
   adjustVxlanControllerServiceEnable()

def noSetControllerMode( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.vccDataPathLearning = False
   vtiConfig.controllerClientMode = False
   vtiConfig.vccArpProxy = False
   vtiConfig.vccNdProxy = False
   adjustVxlanControllerServiceEnable()

class VxlanControllerClientCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan controller-client'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'controller-client': controllerClientNode,
   }
   handler = setControllerMode
   noOrDefaultHandler = noSetControllerMode

VxlanIntfModelet.addCommandClass( VxlanControllerClientCmd )

# -------------------------------------------------------------------------------
# The "[no|default] vxlan decap floodset unknown-multicast
#                                        unknown-unicast exclude cpu"
# command, in "config-vxlan-if" mode.
# -------------------------------------------------------------------------------
def setDecapUcastMcastFloodsetExcludeCpu( mode, args ):
   vxlanFloodsetConfig.decapUcastMcastFloodsetExcludeCpu = True

def clearDecapUcastMcastFloodsetExcludeCpu( mode, args ):
   vxlanFloodsetConfig.decapUcastMcastFloodsetExcludeCpu = False

class VxlanDecapUcastMcastFloodsetExcludeCpuCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan decap floodset unknown-multicast unknown-unicast exclude cpu'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'decap': 'Decapsulation settings',
      'floodset': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'floodset',
                                            helpdesc='Floodset settings' ),
         guard=vxlanDecapUcastMcastFloodsetExcludeCpuSupported ),
      'unknown-multicast': 'Unknown multicast packets',
      'unknown-unicast': 'Unknown unicast packets',
      'exclude': 'Exclude from floodset',
      'cpu': 'CPU'
   }
   handler = setDecapUcastMcastFloodsetExcludeCpu
   noOrDefaultHandler = clearDecapUcastMcastFloodsetExcludeCpu

VxlanIntfModelet.addCommandClass( VxlanDecapUcastMcastFloodsetExcludeCpuCmd )
#-------------------------------------------------------------------------------
# The "[no|default] vxlan qos ecn propagation" command, in "enable" mode.
#-------------------------------------------------------------------------------
def setEcnPropagation( mode, args ):
   vxlanEcnConfig.ecnPropagation = True

def clearEcnPropagation( mode, args ):
   vxlanEcnConfig.ecnPropagation = False

class VxlanQosEcnPropagationCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan qos ecn propagation'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'qos': 'Qos settings',
      'ecn': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'ecn', helpdesc='Ecn settings' ),
         guard=vxlanEcnPropagationSupported ),
      'propagation': 'Set ecn propagation',
   }
   handler = setEcnPropagation
   noOrDefaultHandler = clearEcnPropagation

VxlanIntfModelet.addCommandClass( VxlanQosEcnPropagationCmd )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan qos dscp ecn propagation decapsulation disabled"
# command, in "enable" mode.
# -------------------------------------------------------------------------------
def setDecapDscpEcnPropagationDisabled( mode, args ):
   vxlanEcnConfig.dscpEcnPropagation = False

def clearDecapDscpEcnPropagationDisabled( mode, args ):
   vxlanEcnConfig.dscpEcnPropagation = True

class VxlanQosDscpEcnPropagationDecapDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan qos dscp ecn propagation decapsulation disabled'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'qos': 'Qos settings',
      'dscp': 'DSCP settings',
      'ecn': 'ECN settings',
      'propagation': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'propagation',
            helpdesc='Set DSCP and ECN propagation' ),
         guard=vxlanDecapDscpEcnPropagationDisabledGuard ),
      'decapsulation': 'Only for VXLAN Decapsulation',
      'disabled': 'Disable DSCP and ECN propagation'
   }
   handler = setDecapDscpEcnPropagationDisabled
   noOrDefaultHandler = clearDecapDscpEcnPropagationDisabled

VxlanIntfModelet.addCommandClass( VxlanQosDscpEcnPropagationDecapDisabledCmd )

# -------------------------------------------------------------------------------
# The "[no|default] vxlan qos dscp ecn rewrite bridged enabled"
# command, in "enable" mode.
# -------------------------------------------------------------------------------
def setDscpEcnRewriteBridgedEnabled( mode, args ):
   vxlanEcnConfig.dscpEcnRewriteBridged = True

def clearDscpEcnRewriteBridgedEnabled( mode, args ):
   vxlanEcnConfig.dscpEcnRewriteBridged = False

class VxlanQosDscpEcnRewriteBridgedEnabledCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan qos dscp ecn rewrite bridged enabled'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'qos': 'Qos settings',
      'dscp': 'DSCP settings',
      'ecn': 'ECN settings',
      'rewrite': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'rewrite',
         helpdesc='Rewrite DSCP and ECN' ),
         guard=vxlanDscpEcnRewriteBridgedEnabledGuard ),
      'bridged': 'For VXLAN bridged packets',
      'enabled': 'Enable DSCP and ECN rewrite'
   }
   handler = setDscpEcnRewriteBridgedEnabled
   noOrDefaultHandler = clearDscpEcnRewriteBridgedEnabled

VxlanIntfModelet.addCommandClass( VxlanQosDscpEcnRewriteBridgedEnabledCmd )

# -------------------------------------------------------------------------------
# The "[no|default] vxlan qos dscp propagation encapsulation" command,
# in "config-vxlan-if" mode.
#-------------------------------------------------------------------------------
def setOuterDscpFromInnerDscp( mode, args ):
   vxlanConfigDir.outerDscpFromInnerDscp = True

def clearOuterDscpFromInnerDscp( mode, args ):
   vxlanConfigDir.outerDscpFromInnerDscp = False

class VxlanQosDscpPropagationEncapsulationCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan qos dscp propagation encapsulation'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'qos': 'Qos settings',
      'dscp': 'DSCP settings',
      'propagation': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'propagation',
            helpdesc='Set DSCP propagation to VXLAN packet' ),
         guard=vxlanOuterDscpFromInnerDscpSupported ),
      'encapsulation': 'Propagate only during encapsulation',
   }
   handler = setOuterDscpFromInnerDscp
   noOrDefaultHandler = clearOuterDscpFromInnerDscp

if Toggles.VxlanToggleLib.toggleVxlanOuterDscpFromInnerDscpEnabled():
   VxlanIntfModelet.addCommandClass( VxlanQosDscpPropagationEncapsulationCmd )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan qos map dscp to traffic-class decapsulation" command,
# in "config-vxlan-if" mode.
#-------------------------------------------------------------------------------
def setTrafficClassFromOuterDscp( mode, args ):
   vxlanConfigDir.trafficClassFromOuterDscp = True

def clearTrafficClassFromOuterDscp( mode, args ):
   vxlanConfigDir.trafficClassFromOuterDscp = False

class VxlanQosMapDscpToTrafficClassDecapsulationCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan qos map dscp to traffic-class decapsulation'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'qos': 'Qos settings',
      'map': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'map',
            helpdesc='Mapping QoS parameters' ),
         guard=vxlanInnerDscpFromOuterDscpSupported ),
      'dscp': 'Map vxlan packet DSCP value',
      'to': 'QoS parameter to map vxlan packet DSCP to',
      'traffic-class': 'Map vxlan packet DSCP value to traffic-class',
      'decapsulation': 'Map only during decapsulation',
   }
   handler = setTrafficClassFromOuterDscp
   noOrDefaultHandler = clearTrafficClassFromOuterDscp

if Toggles.VxlanToggleLib.toggleVxlanOuterDscpFromInnerDscpEnabled():
   VxlanIntfModelet.addCommandClass( VxlanQosMapDscpToTrafficClassDecapsulationCmd )

#-------------------------------------------------------------------------------
# The "show vxlan qos" command.
#-------------------------------------------------------------------------------
def showVxlanQos( mode, args ):
   qosModel = VxlanModel.VxlanQosModel()
   if bridgingHwCapabilities.vxlanEcnPropagationSupported:
      qosModel.ecnPropagation = vxlanEcnConfig.ecnPropagation
   else:
      qosModel.ecnPropagation = None
   if bridgingHwCapabilities.vxlanDecapDscpEcnPropagationDisableSupported:
      qosModel.decapDscpEcnPropagationDisabled = \
            not vxlanEcnConfig.dscpEcnPropagation
   else:
      qosModel.decapDscpEcnPropagationDisabled = None
   if bridgingHwCapabilities.vxlanDscpEcnRewriteBridgedEnableSupported:
      qosModel.dscpEcnRewriteBridgedEnabled = vxlanEcnConfig.dscpEcnRewriteBridged
   else:
      qosModel.dscpEcnRewriteBridgedEnabled = None
   qosModel.routingDscpRewriteSupported = \
         bridgingHwCapabilities.vxlanRoutingDscpRewriteSupported
   if Toggles.VxlanToggleLib.toggleVxlanOuterDscpFromInnerDscpEnabled():
      qosModel.outerDscpFromInnerDscp = vxlanConfigDir.outerDscpFromInnerDscp
      qosModel.trafficClassFromOuterDscp = vxlanConfigDir.trafficClassFromOuterDscp
   return qosModel

#-------------------------------------------------------------------------------
# The "[no|default] vxlan virtual-router encapsulation mac-addr [ <MAC> |
#                                                                 mlag-system-id ]
# command, in "config-vxlan-if"
#
# This command enables VXLAN Shared Router MAC. The MAC is also refered as MLAG
# shared router MAC when MLAG is configured. It can be configured with one of 2
# options:
#  (1) set VXLAN shared router mac using the value of local mlag-system-id
#  (2) set VXLAN shared router mac to the virtual mac provided by configure
# Option 1 is only availible when MLAG is configured.
#
# Note: When configured with option (1), this command only has effect when MLAG
# is configured and "active".
#-------------------------------------------------------------------------------
def setVxlanSharedRouterMacAuto( mode ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.mlagSharedRouterMacAddr = ethAddrZero
   vtiConfig.mlagSharedRouterMacConfig = 'autoGenerated'

def setVxlanSharedRouterMacExplicit( mode, macAddr ):
   if not Ethernet.isUnicast( macAddr ) or \
      macAddr == '00:00:00:00:00:00':
      mode.addError( 'VXLAN shared router MAC cannot be a multicast or broadcast'
                     'MAC address' )
      return
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.mlagSharedRouterMacAddr = macAddr
   vtiConfig.mlagSharedRouterMacConfig = 'explicitConfig'

def noSetVxlanSharedRouterMacMode( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.mlagSharedRouterMacAddr = ethAddrZero
   vtiConfig.mlagSharedRouterMacConfig = 'disabled'

class VxlanSharedRouterMacCmd( CliCommand.CliCommandClass ):
   syntax = '''vxlan virtual-router encapsulation mac-address
               ( MAC_ADDR | mlag-system-id )'''
   noOrDefaultSyntax = 'vxlan virtual-router encapsulation mac-address ...'
   data = {
      'vxlan': vxlanNode,
      'virtual-router': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'virtual-router',
            helpdesc='Configure a virtual router' ),
         guard=vxlanSharedRouterMacSupported ),
      'encapsulation': 'VXLAN encapsulation',
      'mac-address': 'Virtual router MAC address',
      'MAC_ADDR': MacAddr.MacAddrMatcher( helpdesc='Ethernet address' ),
      'mlag-system-id': 'Configure MLAG shared router MAC with System ID',
   }

   @staticmethod
   def handler( mode, args ):
      if 'MAC_ADDR' in args:
         return setVxlanSharedRouterMacExplicit( mode, args[ 'MAC_ADDR' ] )
      else:
         return setVxlanSharedRouterMacAuto( mode )

   noOrDefaultHandler = noSetVxlanSharedRouterMacMode

VxlanIntfModelet.addCommandClass( VxlanSharedRouterMacCmd )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan controller-client import vlan" command in
# "config-vxlan-if" mode.
#
# The full syntax of this command is:
#
#   vxlan controller-client import vlan none
#   vxlan controller-client import vlan <import_enabled_vlans>
#   vxlan controller-client import vlan add <import_enabled_vlans>
#   vxlan controller-client import vlan remove <import_disabled_vlans>
#   (no|default) vxlan controller-client import vlan ...
#-------------------------------------------------------------------------------
importEnabledVlanSetMatcher = MultiRangeRule.MultiRangeMatcher(
      lambda: ( 1, 4094 ),
      False,
      'VCS import-enabled VLAN IDs',
      value=VlanCli.vlanIdListFunc )

importDisabledVlanSetMatcher = MultiRangeRule.MultiRangeMatcher(
      lambda: ( 1, 4094 ),
      False,
      'VCS import-disabled VLAN IDs',
      value=VlanCli.vlanIdListFunc )

def setVccImportVlans( mode, importVlanOp ):
   vtiConfig = getVtiConfig( mode.intf.name )
   if not vtiConfig:
      return

   importVlanSet = VlanCli.VlanSet( mode, vtiConfig.cliVccImportVlans )
   ( vlanOp, vlanSet ) = importVlanOp

   if vlanOp == 'set':
      importVlanSet = vlanSet
   elif vlanOp == 'add':
      importVlanSet |= vlanSet
   else:
      assert vlanOp == 'remove'
      importVlanSet -= vlanSet

   s = MultiRangeRule.multiRangeToCanonicalString( importVlanSet )
   vtiConfig.cliVccImportVlans = s

def allVlanIds():
   return frozenset( range( 1, 4095 ) )

def noVlanIds():
   return frozenset()

class VxlanControllerClientImportVlan( CliCommand.CliCommandClass ):
   syntax = '''vxlan controller-client import vlan
               ( ( none )
               | ( remove IMPORT_DISABLED_VLANS )
               | ( [ add ] IMPORT_ENABLED_VLANS ) )'''
   noOrDefaultSyntax = '''vxlan controller-client import vlan ...'''

   data = {
      'vxlan': vxlanNode,
      'controller-client': controllerClientNode,
      'import': 'Set import VLAN characteristics when in controller client mode',
      'vlan': 'Set import VLAN when in controller client mode',
      'none': 'No VLANs',
      'add': 'Add VLANs to the current list',
      'remove': 'Remove VLANs from the current list',
      'IMPORT_DISABLED_VLANS': importDisabledVlanSetMatcher,
      'IMPORT_ENABLED_VLANS': importEnabledVlanSetMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'none' ):
         setVccImportVlans( mode, ( 'set', noVlanIds() ) )
      elif args.get( 'add' ):
         vlanSet = args.get( 'IMPORT_ENABLED_VLANS' )
         setVccImportVlans( mode, ( 'add', vlanSet ) )
      elif args.get( 'remove' ):
         vlanSet = args.get( 'IMPORT_DISABLED_VLANS' )
         setVccImportVlans( mode, ( 'remove', vlanSet ) )
      else:
         vlanSet = args.get( 'IMPORT_ENABLED_VLANS' )
         assert vlanSet
         setVccImportVlans( mode, ( 'set', vlanSet ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setVccImportVlans( mode, ( 'set', allVlanIds() ) )

VxlanIntfModelet.addCommandClass( VxlanControllerClientImportVlan )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan controller-client arp" command in
# "config-vxlan-if" mode.
#
# The full syntax of this command is:
#
#   vxlan controller-client arp proxy
#   (no|default) vxlan controller-client arp ...
#-------------------------------------------------------------------------------

class VxlanControllerClientArp( CliCommand.CliCommandClass ):
   syntax = '''vxlan controller-client arp proxy'''
   noOrDefaultSyntax = syntax

   data = {
      'vxlan': vxlanNode,
      'controller-client': controllerClientNode,
      'arp': 'ARP commands',
      'proxy': 'ARP proxying when in controller client mode',
   }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      if not vtiConfig.controllerClientMode:
         setControllerMode( mode, args )
      vtiConfig.vccArpProxy = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name, create=False )
      if vtiConfig:
         vtiConfig.vccArpProxy = False

if Toggles.VxlanToggleLib.toggleVxlanVccArpProxySupportedEnabled():
   VxlanIntfModelet.addCommandClass( VxlanControllerClientArp )

# -------------------------------------------------------------------------------
# The "[no|default] vxlan controller-client nd" command in
# "config-vxlan-if" mode.
#
# The full syntax of this command is:
#
#   vxlan controller-client nd proxy
#   (no|default) vxlan controller-client nd ...
# -------------------------------------------------------------------------------

class VxlanControllerClientNd( CliCommand.CliCommandClass ):
   syntax = '''vxlan controller-client nd proxy'''
   noOrDefaultSyntax = syntax

   data = {
      'vxlan': vxlanNode,
      'controller-client': controllerClientNode,
      'nd': 'ND commands',
      'proxy': 'Enable Neighbor Discovery Proxy',
   }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      if not vtiConfig.controllerClientMode:
         setControllerMode( mode, args )
      vtiConfig.vccNdProxy = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name, create=False )
      if vtiConfig:
         vtiConfig.vccNdProxy = False

if Toggles.VxlanControllerToggleLib.toggleVxlanVccNdProxySupportedEnabled():
   VxlanIntfModelet.addCommandClass( VxlanControllerClientNd )

forwardingKw = CliCommand.guardedKeyword(
      'forwarding', helpdesc='Configure VXLAN forwarding parameters',
      guard=srcMacRewriteSupportedGuard )

macRewriteVrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF name',
      guard=overlaySrcMacRewriteSupportedGuard )

localInterfaceKw = CliCommand.guardedKeyword(
      'local-interface', helpdesc='VXLAN Source Interface',
      guard=intfOverlaySrcMacRewriteSupportedGuard )

macRewriteVtiMatcher = VirtualIntfRule.VirtualIntfMatcher( 'Vxlan',
      None, None, rangeFunc=vxlanRangeFn,
      helpdesc='VXLAN Tunnel Interface' )

# -------------------------------------------------------------------------------
# The "[no|default] vxlan arp proxy" command in
# "config-vxlan-if" mode.
#
# The full syntax of this command is:
#
#   vxlan arp proxy
#   (no|default) vxlan arp proxy ...
# -------------------------------------------------------------------------------

class VxlanArpProxy( CliCommand.CliCommandClass ):
   syntax = '''vxlan arp proxy'''
   noOrDefaultSyntax = syntax

   data = {
      'vxlan': vxlanNode,
      'arp': CliCommand.guardedKeyword( 'arp',
                                        helpdesc='ARP commands',
                                        guard=isVxlan1InterfaceGuard ),
      'proxy': 'ARP proxying for requests received over VXLAN',
   }

   @staticmethod
   def handler( mode, args ):
      vxlanCliClientConfig.vxlanEncapsulatedArpSnooping = \
            TristateBoolean.valueSet( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vxlanCliClientConfig.vxlanEncapsulatedArpSnooping = \
            TristateBoolean.valueInvalid()

if Toggles.VxlanToggleLib.toggleVxlanArpProxyCliCommandEnabled():
   VxlanIntfModelet.addCommandClass( VxlanArpProxy )

# -------------------------------------------------------------------------------
# The "[no|default] vxlan nd proxy" command in
# "config-vxlan-if" mode.
#
# The full syntax of this command is:
#
#   vxlan nd proxy
#   (no|default) vxlan nd proxy ...
# -------------------------------------------------------------------------------

class VxlanNdProxy( CliCommand.CliCommandClass ):
   syntax = '''vxlan nd proxy'''
   noOrDefaultSyntax = syntax

   data = {
      'vxlan': vxlanNode,
      'nd': CliCommand.guardedKeyword( 'nd',
                                        helpdesc='Neighbor Discovery commands',
                                        guard=isVxlan1InterfaceGuard ),
      'proxy': 'ND proxying for requests received over VXLAN',
   }

   @staticmethod
   def handler( mode, args ):
      vxlanCliClientConfig.vxlanEncapsulatedNdSnooping = \
            TristateBoolean.valueSet( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vxlanCliClientConfig.vxlanEncapsulatedNdSnooping = \
            TristateBoolean.valueInvalid()

if Toggles.VxlanToggleLib.toggleVxlanNdProxyCliCommandEnabled():
   VxlanIntfModelet.addCommandClass( VxlanNdProxy )

# -------------------------------------------------------------------------------
# [no|default] vxlan forwarding match vrf <VRF-NAME> src-ip <IP> action rewrite
# src-mac <MAC> out
# -------------------------------------------------------------------------------

class OverlaySrcMacRewrite( CliCommand.CliCommandClass ):
   syntax = '''vxlan forwarding match VRF src-ip SIP
               action rewrite src-mac SMAC out'''
   noOrDefaultSyntax = 'vxlan forwarding match VRF src-ip SIP action rewrite ...'
   data = {
         'vxlan': 'Configure global VXLAN parameters',
         'forwarding': forwardingKw,
         'match': 'Specify match parameters',
         'VRF': macRewriteVrfExprFactory,
         'src-ip': 'Overlay source IP address',
         'SIP': IpAddrMatcher.ipAddrMatcher,
         'action': 'Specify action parameters',
         'rewrite': 'Rewrite source MAC address',
         'src-mac': 'Overlay rewritten source MAC address',
         'SMAC': MacAddr.macAddrMatcher,
         'out': 'Rewrite MAC address of outgoing VXLAN encapped packets'
   }

   @staticmethod
   def handler( mode, args ):
      t0( 'Mac rewrite config handler' )
      vrfName = args[ 'VRF' ]
      overlaySip = args[ 'SIP' ]
      overlayRewriteMac = args[ 'SMAC' ]

      vrfNameAndIp = VrfNameAndIp( vrfName, overlaySip )
      del macRewriteCliConfig.overlaySrcMacRewriteEntry[ vrfNameAndIp ]

      vrfIpAndMac = VrfIpAndMac( vrfNameAndIp, overlayRewriteMac )
      macRewriteCliConfig.overlaySrcMacRewriteEntry.addMember( vrfIpAndMac )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'Mac rewrite config no/default handler' )
      vrfName = args[ 'VRF' ]
      overlaySip = args[ 'SIP' ]

      vrfNameAndIp = VrfNameAndIp( vrfName, overlaySip )
      del macRewriteCliConfig.overlaySrcMacRewriteEntry[ vrfNameAndIp ]

BasicCli.GlobalConfigMode.addCommandClass( OverlaySrcMacRewrite )

# -------------------------------------------------------------------------------
# [no|default] vxlan forwarding match local-interface <VTI> vni <VNI> action rewrite
# src-mac <MAC> out
# -------------------------------------------------------------------------------
class IntfOverlaySrcMacRewrite( CliCommand.CliCommandClass ):
   syntax = '''vxlan forwarding match local-interface VTI vni VNI
               action rewrite src-mac SMAC out'''
   noOrDefaultSyntax = '''vxlan forwarding match local-interface VTI vni VNI
                          action rewrite ...'''
   data = {
         'vxlan': 'Configure global VXLAN parameters',
         'forwarding': forwardingKw,
         'match': 'Specify match parameters',
         'local-interface': localInterfaceKw,
         'VTI': macRewriteVtiMatcher,
         'vni': 'VXLAN Network Identifier',
         'VNI': vniMatcher,
         'action': 'Specify action parameters',
         'rewrite': 'Rewrite source MAC address',
         'src-mac': 'Overlay rewritten source MAC address',
         'SMAC': MacAddr.macAddrMatcher,
         'out': 'Rewrite MAC address of outgoing VXLAN encapped packets'
   }

   @staticmethod
   def handler( mode, args ):
      t0( 'VTI/VNI mac rewrite config handler' )
      vti = args[ 'VTI' ]
      vni = args[ 'VNI' ]
      overlayRewriteMac = args[ 'SMAC' ]

      vniVtiPair = VniVtiPair( int( vni ), vti )
      del macRewriteCliConfig.intfOverlaySrcMacRewriteEntry[ vniVtiPair ]

      vniVtiAndMac = VniVtiAndMac( vniVtiPair, overlayRewriteMac )
      macRewriteCliConfig.intfOverlaySrcMacRewriteEntry.addMember( vniVtiAndMac )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'VTI/VNI mac rewrite config no/default handler' )
      vti = args[ 'VTI' ]
      vni = args[ 'VNI' ]

      vniVtiPair = VniVtiPair( int( vni ), vti )
      del macRewriteCliConfig.intfOverlaySrcMacRewriteEntry[ vniVtiPair ]

BasicCli.GlobalConfigMode.addCommandClass( IntfOverlaySrcMacRewrite )

#-------------------------------------------------------------------------------
# [no|default] arp reply relay
#-------------------------------------------------------------------------------
def enableArpReplyRelay( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.arpReplyRelay = True

def disableArpReplyRelay( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.arpReplyRelay = False

class ArpReplyRelayCmd( CliCommand.CliCommandClass ):
   syntax = 'arp reply relay'
   noOrDefaultSyntax = syntax
   data = {
      'arp': 'Configure ARP feature',
      'reply': 'Configure ARP Reply',
      'relay': 'ARP Relay',
   }
   handler = enableArpReplyRelay
   noOrDefaultHandler = disableArpReplyRelay

VxlanIntfModelet.addCommandClass( ArpReplyRelayCmd )

#--------------------------------------------------------------------------------
# [ no | default ] arp source ip address local
#--------------------------------------------------------------------------------
def enableArpLocalAddress( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.arpLocalAddress = True

def disableArpLocalAddress( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.arpLocalAddress = False

class ArpSourceIpAddressLocalCmd( CliCommand.CliCommandClass ):
   syntax = 'arp source ip address local'
   noOrDefaultSyntax = syntax
   data = {
      'arp': 'Configure ARP feature',
      'source': 'Modify sender address in VXLAN encapsulated ARP requests',
      'ip': 'Modify sender IP in VXLAN encapsulated ARP requests',
      'address': 'Modify sender IP address in VXLAN encapsulated ARP requests',
      'local': ( 'Enable rewrite of virtual IP in VXLAN encapsulated ARP requests '
                 'with local IP before local bridging' ),
   }
   handler = enableArpLocalAddress
   noOrDefaultHandler = disableArpLocalAddress

VxlanIntfModelet.addCommandClass( ArpSourceIpAddressLocalCmd )

class StaticEvpnRouteConfigTagState( ConfigTagCommon.ConfigTagDependentBase ):
   def processTaggedRoutes( self, mode, tag, removeOrDisassociate, af='ipv4',
         dynamic=False ):
      taggedVrfConf = IraCommonCli.getConfigTagRouteKeyMount(
            af, dynamic ).vrfConfig
      tagInput = configTagInput.configTagEntry.get( tag, None )
      if not tagInput:
         return
      nextHop = {}
      vrfsUsingTag = tagInput.vrfsUsingTag
      for vrf in vrfsUsingTag:
         if af == 'ipv4':
            rt = IraIpRouteCliLib.staticRoutingTable( vrf, dynamic=dynamic )
         else:
            rt = IraIp6RouteCliLib.staticIpv6RoutingTable( vrf, dynamic=dynamic )
         tagConfig = taggedVrfConf.get( vrf, None )
         if not tagConfig:
            continue
         vxlanCliHelper().iraIpRouteCliHelper.removeOrDisassociateCmdTagRoutes( rt,
               tagConfig, tag, removeOrDisassociate )

   def removeTaggedConfig( self, mode, tag ):
      self.processTaggedRoutes( mode, tag, RemoveOrDisassociate.remove )
      self.processTaggedRoutes( mode, tag, RemoveOrDisassociate.remove,
         dynamic=True )

   def disassociateConfigFromTag( self, mode, tag ):
      self.processTaggedRoutes( mode, tag, RemoveOrDisassociate.disassociate )
      self.processTaggedRoutes( mode, tag, RemoveOrDisassociate.disassociate,
         dynamic=True )

   def processAndValidateConfig( self, mode, commandTagInfo ):
      pass

ConfigTagCommon.ConfigTagState.registerConfigTagSupportingClass(
               StaticEvpnRouteConfigTagState )

#-------------------------------------------------------------------------------
# [no|default] ip route vrf <vrfId> <prefix> vtep <vtepAddr> vni <vni>
# router-mac-address <router-mac>
#-------------------------------------------------------------------------------

def isValidVniWithError( mode, vni, vti ):
   if not vti:
      vti = 'Vxlan1'
   if not getVtiConfig( vti, create=False ):
      errorStr = "Interface %s doesn't exist." % vti
      mode.addError( errorStr )
      return False
   # Support for 32-bit VNI is inherited from Vxlan1
   vx1Config = getVtiConfig( 'Vxlan1', create=False )
   use32BitVni = vx1Config and vx1Config.use32BitVni
   minVni = 1
   maxVni = 0xFFFFFF if not use32BitVni else 0xFFFFFFFE
   if vni < minVni or vni > maxVni:
      errorStr = "Invalid VXLAN Network Identifier. "
      if vxlanCtrlCfgBox[ 0 ].vniInDottedNotation:
         errorStr += "The VNI should be of the form A.B.C where A, B and C are " \
                     "integers in the range 0-255. "
      else:
         errorStr += "Allowed range is: %d-%d. " % ( minVni, maxVni )
      mode.addError( errorStr )
      return False
   return True

def getVxlanIntf( localIntf ):
   vxlanIntfStr = localIntf.strWithLongTag() if localIntf else ''
   if vxlanIntfStr:
      intf = vtiNameToId.get( vxlanIntfStr )
   else:
      intf = Tac.Value( "Arnet::IntfId", "" )
   return intf

def manageNoIpv4StaticEvpnRoute( mode, args ):
   preference = args.get( 'PREFERENCE', None )
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   evpnNextHop = args[ 'EVPN_NEXTHOP' ]
   nexthop = {}
   if 'vtepAddr' in evpnNextHop and 'vni' in evpnNextHop and \
         'routerMac' in evpnNextHop:
      nexthop[ 'evpn' ] = {}
      nexthop[ 'evpn' ][ 'vtepAddr' ] = evpnNextHop[ 'vtepAddr' ]
      nexthop[ 'evpn' ][ 'vni' ] = evpnNextHop[ 'vni' ]
      nexthop[ 'evpn' ][ 'routerMac' ] = evpnNextHop[ 'routerMac' ]
      nexthop[ 'evpn' ][ 'dynamic' ] = evpnNextHop[ 'dynamic' ]
      localIntf = evpnNextHop.get( 'vxlanSrcIntf' )
      localIntfStr = localIntf.strWithLongTag() if localIntf else ""
      nexthop[ 'evpn' ][ 'local-intf' ] = localIntfStr

   noIpRoute = LazyCallback( "IraIpCliHandler.noIpRoute" )
   configTagSupported = \
      ConfigTagCommon.configTagSupportedByPlatform( mode, None )
   noIpRoute( mode, args[ 'PREFIX' ], nexthop, preference, vrfName,
                        None, configTag=configTagSupported )

def manageIpv4StaticEvpnRoute( mode, args ):
   prefix = args[ 'PREFIX' ]
   evpnNextHop = args[ 'EVPN_NEXTHOP' ]
   routeOptions = args.get( 'OPTION', {} )
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   vtepSipValidation = args[ 'source-vtep-validation' ]
   configTagStr = args.get( 'CONFIG_TAG', None )

   preference = routeOptions.get( 'preference',
                                  RoutingConsts.defaultStaticRoutePreference )
   tag = routeOptions.get( 'tag', RoutingConsts.defaultStaticRouteTag )
   nextHopName = routeOptions.get( 'nextHopName',
                                   RoutingConsts.defaultStaticRouteName )

   vxlanIntf = getVxlanIntf( evpnNextHop.get( 'vxlanSrcIntf', "" ) )
   routerMac = evpnNextHop[ 'routerMac' ]
   vni = int( evpnNextHop[ 'vni' ] )
   vtepAddr = evpnNextHop[ 'vtepAddr' ]
   dynamic = evpnNextHop[ 'dynamic' ]
   ribBypass = evpnNextHop[ 'ribBypass' ]
   if hasattr( vtepAddr, 'stringValue' ):
      # this can be a v4 or v6 address, v4 is a string, v6 is an
      # Arnet::Ip6Addr, we normalize to a string
      vtepAddr = vtepAddr.stringValue

   rt = IraIpRouteCliLib.staticRoutingTable( vrfName, dynamic=dynamic )
   if rt is None:
      mode.addError( IraIpRouteCliLib.noRouteTableForVrfMsg % vrfName +
                     " Create first." )
      return

   if configTagStr:
      ConfigTagCommon.commandTagConfigCheck( mode, configTagStr )
      ConfigTagCommon.checkTagInRemovedOrDisassociatedState( mode, configTagStr,
            featureCheck=True )

   error = not vxlanCliHelper().manageStaticEvpnRoute( rt, vrfName,
         prefix, vni, preference, tag, nextHopName, vtepSipValidation, vtepAddr,
         routerMac, vxlanIntf, ribBypass, mode.session.inConfigSession() )

   if error:
      mode.addError( vxlanCliHelper().errorMsg(
                     vxlanCtrlCfgBox[ 0 ].vniInDottedNotation ) )
   elif ConfigTagCommon.configTagSupportedByPlatform( mode, configTagStr ):
      if configTagStr:
         IraCommonCli.manageStaticEvpnRouteConfigTag( mode, prefix, preference,
               vrfName, dynamic, rt, tagStr=configTagStr )
      else:
         # If a route exists with a configTag association, remove the tag.
         IraCommonCli.manageStaticEvpnRouteConfigTag( mode, prefix, preference,
               vrfName, dynamic, rt )

vtepAddrNode = CliCommand.Node(
      IpGenAddrMatcher.IpGenAddrMatcher( 'IPv4 or IPv6 address of remote VTEP' ),
      guard=vxlanIpv6UnderlaySupportedGuard )
routerMacMatcherForConfig = CliMatcher.KeywordMatcher( 'router-mac-address',
                                        helpdesc='Remote router Mac address' )

class IpRouteEvpnCmdMultiVtep( CliCommand.CliCommandClass ):
   syntax = '''ip route [ VRF ] PREFIX vtep VTEP vni VNI
               router-mac-address MAC_ADDR
               [ { ROUTE_OPTIONS
                 | ( local-interface INTFS )
                 | dynamic
                 | rib-bypass
                 | source-vtep-validation } ]
               [ CONFIG_TAG_EXPR ]'''
   noOrDefaultSyntax = '''ip route [ VRF ] PREFIX vtep [ VTEP vni VNI
                          router-mac-address MAC_ADDR ]
                          [ { PREFERENCE
                            | ( local-interface INTFS )
                            | dynamic
                            | rib-bypass
                            | source-vtep-validation } ]
                          [ CONFIG_TAG_EXPR ] ...'''
   data = { 'ip': CliToken.Ip.ipMatcherForConfig,
            'route': IraIpRouteCliLib.routeMatcherForConfig,
            'VRF': IraIpCli.vrfExprFactoryForConfig,
            'PREFIX': IraIpRouteCliLib.prefixMatcher,
            'vtep': vtepMatcherForConfig,
            'source-vtep-validation': sipValidationNode(),
            'VTEP': vtepAddrNode,
            'vni': vniMatcherForConfig,
            'VNI': vniMatcher,
            'router-mac-address': routerMacMatcherForConfig,
            'MAC_ADDR': MacAddr.MacAddrMatcher(),
            'local-interface': 'VXLAN VTEP source interface',
            'INTFS': CliCommand.Node( IntfRange.IntfRangeMatcher(
                                      explicitIntfTypes=vxlanIntfType ),
                                      maxMatches=1 ),
            # For the noOrDefaultSyntax we can use PREFERENCE which is
            # emplaced in the data dict by the RouteOptionsExprFactory
            'ROUTE_OPTIONS': IraCommonCli.RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeNexthopName=True
                             ),
            'dynamic': CliCommand.Node( dynamicMatcherForConfig, maxMatches=1 ),
            'rib-bypass': CliCommand.Node(
               ribBypassMatcherForConfig, maxMatches=1,
               guard=ribBypassSupportedGuard ),
            'CONFIG_TAG_EXPR': configTagExpr
            }
   rootNodesCacheEnabled = True

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'EVPN_NEXTHOP' ] = {}
      if 'VTEP' in args:
         args[ 'EVPN_NEXTHOP' ][ 'vtepAddr' ] = args[ 'VTEP' ]
      if 'VNI' in args:
         args[ 'EVPN_NEXTHOP' ][ 'vni' ] = args[ 'VNI' ]
      if 'MAC_ADDR' in args:
         args[ 'EVPN_NEXTHOP' ][ 'routerMac' ] = args[ 'MAC_ADDR' ]
      if ( intfs := args.get( 'INTFS' ) ) and intfs.strWithLongTag() != 'Vxlan1':
         # routes with Vxlan1 as source VTI will have an empty vxlanSrcIntf
         args[ 'EVPN_NEXTHOP' ][ 'vxlanSrcIntf' ] = intfs
      args[ 'EVPN_NEXTHOP' ][ 'dynamic' ] = 'dynamic' in args
      args[ 'EVPN_NEXTHOP' ][ 'ribBypass' ] = 'rib-bypass' in args
      args[ 'source-vtep-validation' ] = 'source-vtep-validation' in args

   @staticmethod
   def handler( mode, args ):
      manageIpv4StaticEvpnRoute( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      manageNoIpv4StaticEvpnRoute( mode, args )

BasicCli.GlobalConfigMode.addCommandClass( IpRouteEvpnCmdMultiVtep )

class StaticEvpnRouteIpv6ConfigTagState( StaticEvpnRouteConfigTagState ):
   def removeTaggedConfig( self, mode, tag ):
      self.processTaggedRoutes( mode, tag, RemoveOrDisassociate.remove, af='ipv6' )
      self.processTaggedRoutes( mode, tag, RemoveOrDisassociate.remove,
         af='ipv6', dynamic=True )

   def disassociateConfigFromTag( self, mode, tag ):
      self.processTaggedRoutes( mode, tag, RemoveOrDisassociate.disassociate,
         af='ipv6' )
      self.processTaggedRoutes( mode, tag, RemoveOrDisassociate.disassociate,
         af='ipv6', dynamic=True )

ConfigTagCommon.ConfigTagState.registerConfigTagSupportingClass(
               StaticEvpnRouteIpv6ConfigTagState )

#-------------------------------------------------------------------------------
# [no|default] ipv6 route vrf <vrfId> <prefix> vtep <vtepAddr> vni <vni>
# router-mac-address <router-mac>
#-------------------------------------------------------------------------------

def manageNoIpv6StaticEvpnRoute( mode, args ):
   preference = args.get( 'PREFERENCE', None )
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   evpnNextHop = args[ 'EVPN_NEXTHOP' ]
   nexthop = {}
   if 'vtepAddr' in evpnNextHop and 'vni' in evpnNextHop and \
         'routerMac' in evpnNextHop:
      nexthop[ 'evpn' ] = {}
      nexthop[ 'evpn' ][ 'vtepAddr' ] = evpnNextHop[ 'vtepAddr' ]
      nexthop[ 'evpn' ][ 'vni' ] = evpnNextHop[ 'vni' ]
      nexthop[ 'evpn' ][ 'routerMac' ] = evpnNextHop[ 'routerMac' ]
      nexthop[ 'evpn' ][ 'dynamic' ] = evpnNextHop[ 'dynamic' ]
      localIntf = evpnNextHop.get( 'vxlanSrcIntf' )
      localIntfStr = localIntf.strWithLongTag() if localIntf else ""
      nexthop[ 'evpn' ][ 'local-intf' ] = localIntfStr

   noIp6Route = LazyCallback( "IraIp6CliHandler.noIp6Route" )
   configTagSupported = \
      ConfigTagCommon.configTagSupportedByPlatform( mode, None )
   noIp6Route( mode, args[ 'PREFIX' ], nexthop, preference, vrfName,
               None, configTag=configTagSupported )

def manageIpv6StaticEvpnRoute( mode, args ):
   prefix = args[ 'PREFIX' ]
   evpnNextHop = args[ 'EVPN_NEXTHOP' ]
   routeOptions = args.get( 'OPTION' )
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   configTagStr = args.get( 'CONFIG_TAG', None )
   vxlanIntf = getVxlanIntf( evpnNextHop.get( 'vxlanSrcIntf', "" ) )

   if not mode.session.inConfigSession():
      if not VrfCli.vrfExists( vrfName ):
         mode.addError( "%s Create first." % noIpv6RouteTableForVrfMsg % vrfName )
         return
      vni = int( evpnNextHop[ 'vni' ] )
      if not isValidVniWithError( mode, vni, vxlanIntf.stringValue ):
         return

   if configTagStr:
      ConfigTagCommon.commandTagConfigCheck( mode, configTagStr )
      ConfigTagCommon.checkTagInRemovedOrDisassociatedState( mode, configTagStr,
            featureCheck=True )

   evpnNextHop[ 'vxlanSrcIntf' ] = vxlanIntf

   nexthop = { 'evpn': evpnNextHop }
   configTagSupported = \
         ConfigTagCommon.configTagSupportedByPlatform( mode, configTagStr )
   manageIpv6Route( mode, prefix, nexthop, routeOptions, vrfName,
         configTag=configTagSupported, configTagStr=configTagStr )

class Ipv6RouteEvpnCmdMultiVtep( CliCommand.CliCommandClass ):
   syntax = '''ipv6 route [ VRF ] PREFIX vtep VTEP vni VNI
               router-mac-address MAC_ADDR
               [ { ROUTE_OPTIONS
                 | ( local-interface INTFS )
                 | dynamic
                 | rib-bypass
                 | source-vtep-validation } ]
               [ CONFIG_TAG_EXPR ]'''
   noOrDefaultSyntax = '''ipv6 route [ VRF ] PREFIX vtep [ VTEP vni VNI
                          router-mac-address MAC_ADDR ]
                          [ { PREFERENCE
                            | ( local-interface INTFS )
                            | dynamic
                            | rib-bypass
                            | source-vtep-validation } ]
                          [ CONFIG_TAG_EXPR ] ...'''
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': IraIp6Cli.vrfExprFactoryForConfig,
            'PREFIX': ipv6PrefixMatcher,
            'vtep': vtepMatcherForConfig,
            'source-vtep-validation': sipValidationNode(),
            'VTEP': vtepAddrNode,
            'vni': vniMatcherForConfig,
            'VNI': vniMatcher,
            'router-mac-address': routerMacMatcherForConfig,
            'MAC_ADDR': MacAddr.MacAddrMatcher(),
            'local-interface': 'VXLAN VTEP source interface',
            'INTFS': CliCommand.Node( IntfRange.IntfRangeMatcher(
                                      explicitIntfTypes=vxlanIntfType ),
                                      maxMatches=1 ),
            # For the noOrDefaultSyntax we can use PREFERENCE which is
            # emplaced in the data dict by the RouteOptionsExprFactory
            'ROUTE_OPTIONS': IraCommonCli.RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeNexthopName=True
                             ),
            'dynamic': CliCommand.Node( dynamicMatcherForConfig, maxMatches=1 ),
            'rib-bypass': CliCommand.Node(
               ribBypassMatcherForConfig, maxMatches=1,
               guard=ribBypassSupportedGuard ),
            'CONFIG_TAG_EXPR': configTagExpr
            }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'EVPN_NEXTHOP' ] = {}
      if 'VTEP' in args:
         args[ 'EVPN_NEXTHOP' ][ 'vtepAddr' ] = args[ 'VTEP' ].stringValue
      if 'VNI' in args:
         args[ 'EVPN_NEXTHOP' ][ 'vni' ] = args[ 'VNI' ]
      if 'MAC_ADDR' in args:
         args[ 'EVPN_NEXTHOP' ][ 'routerMac' ] = args[ 'MAC_ADDR' ]
      if ( intfs := args.get( 'INTFS' ) ) and intfs.strWithLongTag() != 'Vxlan1':
         # routes with Vxlan1 as source VTI will have an empty vxlanSrcIntf
         args[ 'EVPN_NEXTHOP' ][ 'vxlanSrcIntf' ] = intfs
      args[ 'EVPN_NEXTHOP' ][ 'dynamic' ] = 'dynamic' in args
      args[ 'EVPN_NEXTHOP' ][ 'ribBypass' ] = 'rib-bypass' in args
      args[ 'EVPN_NEXTHOP' ][ 'source-vtep-validation' ] =\
            'source-vtep-validation' in args

   @staticmethod
   def handler( mode, args ):
      manageIpv6StaticEvpnRoute( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      manageNoIpv6StaticEvpnRoute( mode, args )

BasicCli.GlobalConfigMode.addCommandClass( Ipv6RouteEvpnCmdMultiVtep )


#--------------------------------------------------------------------------------
# [ no | default ] virtual-router arp optimize flooding disabled
#--------------------------------------------------------------------------------
def enableVirtualArpOptimizedFlooding( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.virtualArpOptimizedFloodingEnabled = True

def disableVirtualArpOptimizedFlooding( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.virtualArpOptimizedFloodingEnabled = False

class VirtualArpOptimizedFloodingCmd( CliCommand.CliCommandClass ):
   syntax = 'virtual-router arp optimize flooding disabled'
   noOrDefaultSyntax = syntax
   data = {
      'virtual-router': 'Configure virtual-router options',
      'arp': ( 'Options for ARP and Neighbor Solicitation from sender virtual-router'
               ' mac address' ),
      'optimize': ( 'Options for optimizing ARP and Neighbor Solicitation from'
                    ' sender virtual-router mac address' ),
      'flooding': ( 'Options for optimizing the flooding of ARP and Neighbor'
                    ' Solicitation from sender virtual-router mac address' ),
      'disabled': ( 'Disable optimized flooding of ARP and Neighbor Solicitation'
                    ' from sender virtual-router mac address' )
   }
   handler = disableVirtualArpOptimizedFlooding
   noOrDefaultHandler = enableVirtualArpOptimizedFlooding

if Toggles.VxlanToggleLib.toggleVxlanVirtualArpOptimizedFloodingEnabled():
   VxlanIntfModelet.addCommandClass( VirtualArpOptimizedFloodingCmd )

# --------------------------------------------------------------------------------
# [ no | default ] icmp helper vrf
# --------------------------------------------------------------------------------
def getIcmpHelperInputConfig( create ):
   if icmpHelperInputConfigClientName in icmpHelperInputConfigDir.entityPtr:
      return icmpHelperInputConfigDir.entityPtr[ icmpHelperInputConfigClientName ]
   if not create:
      return None
   return icmpHelperInputConfigDir.createEntity( "TepIcmpHelper::InputConfig",
                                                 icmpHelperInputConfigClientName )

def enableIcmpHelper( mode, args ):
   icmpHelperConfig = getIcmpHelperInputConfig( create=True )
   if 'VRF' in args:
      vrfName = args[ 'VRF' ]
      icmpHelperConfig.vrf[ vrfName ] = True

def disableIcmpHelper( mode, args ):
   icmpHelperConfig = getIcmpHelperInputConfig( create=False )
   if icmpHelperConfig:
      if 'VRF' in args:
         vrfName = args[ 'VRF' ]
         if vrfName in icmpHelperConfig.vrf:
            del icmpHelperConfig.vrf[ vrfName ]
      else:
         icmpHelperInputConfigDir.deleteEntity( icmpHelperInputConfigClientName )

def icmpHelperSupportedGuard( mode, token ):
   if bridgingHwCapabilities.tepIcmpHelperDeviceSupported:
      return None
   return CliParser.guardNotThisPlatform

vrfIcmpHelperExprFactory = VrfCli.VrfExprFactory(
      helpdesc='Enable enhanced reporting of ICMP in OVERLAY VRF',
      guard=icmpHelperSupportedGuard )

class IcmpHelperCmd( CliCommand.CliCommandClass ):
   syntax = 'icmp helper [ VRF ]'
   noOrDefaultSyntax = syntax
   data = {
      'icmp': 'Configure ICMP options for VXLAN overlay and underlay packets',
      'helper': CliCommand.guardedKeyword( 'helper',
                 helpdesc='Enable helper for enhanced reporting of VXLAN ICMP',
                 guard=vxlanSupportedGuard ),
      'VRF': vrfIcmpHelperExprFactory
   }
   handler = enableIcmpHelper
   noOrDefaultHandler = disableIcmpHelper

if Toggles.VxlanToggleLib.toggleVtepIcmpHelperEnabled():
   VxlanIntfModelet.addCommandClass( IcmpHelperCmd )

#-------------------------------------------------------------------------------
# The "show vxlan counters software" command, in "enable" mode.
#-------------------------------------------------------------------------------
def createMapSwCounterIdToStr( countersToStr, countersToKey, countersToName,
                               cntrIdMap, strKey=False ):
   # countersToStr stores the description for the counter
   # countersToName stores the name of the counter
   def getSwKeyRepresentation( key ):
      return key if strKey else counterId.enumVal( key )

   def addSwCountersToKey( key, name=None ):
      countKey = getSwKeyRepresentation( key )
      countersToKey[ countKey ] = key
      countersToName[ countKey ] = name or key
      return countKey

   counterId = Tac.newInstance( "Vxlan::VxlanCounterId" )

   for cntr, value in six.iteritems( cntrIdMap ):
      key = value[ 0 ] if value[ 0 ] else cntr
      countKey = addSwCountersToKey( key, cntr )
      countersToStr[ countKey ] = value[ 1 ]

def showSwCtrs( mode, args ):
   counterIdToStr = {}
   counterIdToKey = {}
   counterIdToName = {}
   counterModel = VxlanModel.VxlanSwCounters()
   cntrIdMap = counterModel.getCntrIdMap()
   createMapSwCounterIdToStr( counterIdToStr, counterIdToKey,
                              counterIdToName, cntrIdMap )

   if not vxAgentCounter.vxlanCounter:
      for idx, _ in enumerate( counterIdToName ):
         setattr( counterModel, counterIdToName[ idx ], 0 )
   else:
      for cntr in six.itervalues( vxAgentCounter.vxlanCounter ):
         if cntr.key in counterIdToName:
            setattr( counterModel, counterIdToName[ cntr.key ], cntr.counterValue )

   return counterModel

#-------------------------------------------------------------------------------
# The "clear vxlan counters software" command, in "enable" mode.
#-------------------------------------------------------------------------------
def clearSwCounters( mode, args ):
   vxlanConfigDir.clearSwCounterRequestTime = Tac.now()

#-------------------------------------------------------------------------------
# The "show vxlan counters vtep [<vtep>] [encap|decap]" command,
# in "enable" mode.
#-------------------------------------------------------------------------------
# This is a hook for the platform to implement its own function to collect vtep
# counters. The hook will receive VxlanVtepCountersModel object and zero or
# more filled VxlanVtepCounters . If the hooks finds a counter to be already
# populated it should add its own values for that counter to it.

showVtepCountersHook = CliHook()

def showVtepCounters( mode, args ):
   ''' Create an object of the container and pass to platform hooks to
       update it. Platform may use more than one hook to collect the counters
       and update the model using addToAttr() calls.'''
   direction = args.get( 'DIRECTION' )
   vtep = None
   if 'unlearnt' in args:
      vtep = 'unlearnt'
   elif 'IP' in args:
      vtep = args[ 'IP' ]
      vtep = str( vtep )

   vtepCounters = VxlanModel.VxlanVtepCountersModel()
   showVtepCountersHook.notifyExtensions( mode=mode, inVtep=vtep,
         direction=direction, vtepCounters=vtepCounters )
   return vtepCounters

#-------------------------------------------------------------------------------
# The "clear vxlan counters vtep [<vtep>]" command,
# in "enable" mode.
#-------------------------------------------------------------------------------
# Clear Command for Vxlan VTEP counters.
# The command clears vxlan hardware counters by publishing state into
# vxlan/hardware/counter vtepClearRequest collection.
# The feature agents react to changes in the collection and clear the vteps

def clearVtepCounters( mode, args ):
   ''' clearVtepCounters populates the entity in vxlan/hardware/counter
       with either the vtep's Ip address, keyword 'unlearnt' or
       255.255.255.255 to clear all counters '''

   # if there is no vtep specified we use the special address 255.255.255.255 to
   # signal that all vteps should be cleared
   vtep = args.get( 'VTEP' )
   if 'unlearnt' in args:
      # if vtep is 'unlearnt' then we normalize to 0.0.0.0
      vtep = '0.0.0.0'

   if vtep is None:
      vtep = '255.255.255.255'
      vxlanHwCounter.vtepClearRequest.clear()
   else:
      vtep = str( vtep )
      del vxlanHwCounter.vtepClearRequest[ Arnet.IpGenAddr( vtep ) ]

   vxlanHwCounter.vtepClearRequest[ Arnet.IpGenAddr( vtep ) ] = True

#-------------------------------------------------------------------------------
# The "show vxlan counters vtep [<vtep>] vni [<vni>] [encap|decap]" command,
# in "enable" mode.
#-------------------------------------------------------------------------------
# This is a hook for the platform to implement its own function to collect
# VNI-RemoteVTEP counters. The hook will receive VxlanVtepCountersModel object
# and zero or more filled VxlanVtepVniCounters . If the hooks finds a counter to
# be already populated it should add its own values for that counter to it.

showVtepVniCountersHook = CliHook()

def showVtepVniCounters( mode, args ):
   ''' Create an object of the container and pass to platform hooks to update it.
       Platform may use more than one hook to collect the counters and update the
       model using addToAttr() calls.'''
   vtep = args.get( 'IP' ) # this is of type Arnet::IpGenAddr
   vni = args.get( 'VNI' )
   if vni:
      vni = int( vni )

   vtepVniCounters = VxlanModel.VxlanVtepVniCountersModel()
   showVtepVniCountersHook.notifyExtensions( mode=mode, vtep=vtep, vni=vni,
      vtepVniCounters=vtepVniCounters )
   return vtepVniCounters

#-------------------------------------------------------------------------------
# The "clear vxlan counters vtep [<vtep>] vni [<vni>]" command,
# in "enable" mode.
#-------------------------------------------------------------------------------
# Clear Command for Vxlan VNI-RemoteVTEP counters.
# The command clears vxlan hardware counters by publishing state into
# vxlan/hardware/counter vniRemoteVtepClearRequest collection.
# The feature agents react to changes in the collection and clear
# the VNI-RemoteVTEP pairs.

VniRemoteVtepPair = Tac.Type( 'Vxlan::VniRemoteVtepPair' )

def clearVtepVniCounters( mode, args ):
   ''' clearVtepCounters populates the entity in vxlan/hardware/counter
       with the following:
         1. valid VNI-RemoteVTEP pair for clear one case
         2. valid VNI and broadcast IP for clear all VNI case
         3. valid VTEP and VNI 0 for clear all VTEP case
         4. VNI 0 and broadcast IP for clear all VNI-RemoteVTEP pairs case'''

   vtep = args.get( 'VTEP', Arnet.IpGenAddr( '255.255.255.255' ) )
   vni = int( args.get( 'VNI', 0 ) )

   vniRemoteVtepPair = VniRemoteVtepPair( vni, vtep )
   del vxlanHwCounter.vniRemoteVtepClearRequest[ vniRemoteVtepPair ]
   vxlanHwCounter.vniRemoteVtepClearRequest[ vniRemoteVtepPair ] = True

#-------------------------------------------------------------------------------
# The "show vxlan counters vni [<vni>] [encap|decap]" command,
# in "enable" mode.
#-------------------------------------------------------------------------------
# This is a hook for the platform to implement its own function to collect vni
# counters. The hook will receive VxlanVniCountersModel object and zero or
# more filled VxlanVniCounters . If the hooks finds a counter to be already
# populated it should add its own values for that counter to it.
showVniCountersHook = CliHook()

def showVniCounters( mode, args ):
   ''' Create an object of the container and pass to platform hooks to
       update it. Platform may use more than one hook to collect the counters
       and update the model without stepping on each other.'''
   vniDottedFormat = args.get( 'VNI' )
   direction = args.get( 'DIRECTION' )
   vni = None
   if vniDottedFormat is not None:
      # vni passes to hooks is always a number
      vni = VniFormat( vniDottedFormat ).toNum()
      if not isValidVniWithError( mode, vni, 'Vxlan1' ):
         return None

   vniCounters = VxlanModel.VxlanVniCountersModel()
   vniCounters.vniInDottedNotation = \
            vxlanControllerConfig.vniInDottedNotation
   showVniCountersHook.notifyExtensions( mode=mode, inVni=vni,
         direction=direction, vniCounters=vniCounters )
   return vniCounters

#-------------------------------------------------------------------------------
# The "clear vxlan counters vni [<vni>]" command,
# in "enable" mode.
#-------------------------------------------------------------------------------
# Clear Command for Vxlan VNI counters.
# The command clears vxlan hardware counters by publishing state into
# vxlan/hardware/counter vniClearRequest collection.
# The feature agents react to changes in the collection and clear the vteps

def clearVniCounters( mode, args ):
   ''' clearVniCounters populates the entity in vxlan/hardware/counter
       with either the VNI's Id  or VNI's invalid Id to clear all counters '''
   vniDottedFormat = args.get( 'VNI' )

   # if there is no VNI specified we use the special value 'invalid'  to
   # signal that all VNIs should be cleared
   if vniDottedFormat is None:
      vniDottedFormat = 'invalid'

   vni = VniFormat( vniDottedFormat ).toNum()
   if vniDottedFormat != 'invalid' and \
         not isValidVniWithError( mode, vni, 'Vxlan1' ):
      return

   if vni == Tac.Value( "Vxlan::VniExtOrNone" ).invalidVni:
      vxlanHwCounter.vniClearRequest.clear()
   else:
      del vxlanHwCounter.vniClearRequest[ vni ]

   vxlanHwCounter.vniClearRequest[ vni ] = True

#-------------------------------------------------------------------------------
# The "show vxlan counters varp" command, in "enable" mode.
#-------------------------------------------------------------------------------
def createMapVarpCounterIdToStr( countersToStr, countersToKey,
                                 cntrIdMap, strKey=False ):
   # countersToStr maps a counter ID to its counter description.
   # varpCountersToName maps a counter ID to its counter name.

   def getVarpKeyRepresentation( key ):
      return key if strKey else counterId.enumVal( key )

   def addVarpCountersToKey( key ):
      countKey = getVarpKeyRepresentation( key )
      countersToKey[ countKey ] = key
      return countKey

   counterId = Tac.newInstance( "Vxlan::VarpCounterId" )
   for cntr, value in six.iteritems( cntrIdMap ):
      key = value[ 0 ] if value[ 0 ] else cntr
      countKey = addVarpCountersToKey( key )
      countersToStr[ countKey ] = value[ 1 ]

def showVarpCounters( mode, args ):
   counterIdToStr = {}
   counterIdToName = {}
   counterModel = VxlanModel.VxlanVarpCounters()
   cntrIdMap = counterModel.getCntrIdMap()
   createMapVarpCounterIdToStr( counterIdToStr, counterIdToName,
                                cntrIdMap )

   if not vxAgentCounter.varpCounter:
      idx = 0
      for cntr in six.iteritems( counterIdToName ):
         setattr( counterModel, counterIdToName[ idx ], 0 )
         idx += 1
   else:
      for cntr in six.iteritems( vxAgentCounter.varpCounter ):
         cntr = cntr[ 1 ]
         if cntr.key in counterIdToName:
            setattr( counterModel, counterIdToName[ cntr.key ], cntr.counterValue )

   return counterModel

#-------------------------------------------------------------------------------
# The "clear vxlan counters varp" command, in "enable" mode.
#-------------------------------------------------------------------------------
def clearVarpCounters( mode, args ):
   vxlanConfigDir.clearVarpCounterRequestTime = Tac.now()

#-------------------------------------------------------------------------------
# Build and return a VxlanVniStatus CAPI object filtered by macAddr and
# vlanId with entries from the passed in vxlanvni status object.
# -------------------------------------------------------------------------------
def buildVxlanVniStatusV2( vni, macAddr, vlanId, vxlanVniStatusV2,
                           intfName, vtepIpList=None ):
   vlan = vniToVlanMap.getVlan( int( vni ), intfName )

   # Skip this vni if there's no vlan mapping for it or if a vlanId
   # was specified and it doesn't match.
   if vlan is None or ( vlanId and vlanId.id != vlan ):
      return None

   vs = VxlanVniStatus()
   vs.vlan = vlan
   vs.vni = int( vni )

   # scan the macVtepTable
   for lhmacAddr, lhmvp in six.iteritems( vxlanVniStatusV2.macVtepTable ):

      # if no mac is specified print them all, else print the requested mac
      if macAddr and not MacAddr.compareMacs( macAddr, lhmacAddr ):
         continue

      # if no vtep is specified print them all, else print the requested vtep
      if vtepIpList and lhmvp.vtepIp not in vtepIpList:
         continue

      mvp = VxlanMacVtepPair()
      mvp.macAddr = lhmvp.macAddr
      mvp.vtepIp = lhmvp.vtepIp
      mvp.moveCount = lhmvp.moveCount
      vs.unicastHostTable.append( mvp )

   # scan the macVtepMultiPathTable
   for bhmacAddr, bhmvlp in six.iteritems( vxlanVniStatusV2.macVtepMultiPathTable ):
      # if no mac is specified print them all, else print the requested mac
      if macAddr and not MacAddr.compareMacs( macAddr, bhmacAddr ):
         continue

      # if no vtep is specified print them all, else print the requested vtep
      if vtepIpList:
         filteredVtepIpList = [ vtepIp for vtepIp in vtepIpList
                                          if vtepIp in bhmvlp.vtepIpList ]
         if not filteredVtepIpList:
            continue

      mvlp = VxlanMacVtepListPair()
      mvlp.macAddr = bhmacAddr
      mvlp.vtepIpListType = 'sendToAny'

      # append all VTEP IPs for this mac
      vteps = list( bhmvlp.vtepIpList )
      mvlp.vtepIpList.extend( vteps )

      vs.bumVtepListTable.append( mvlp )

   # scan the macVtepFloodListTable
   for bhmacAddr, bhmvlp in six.iteritems( vxlanVniStatusV2.macVtepFloodListTable ):
      # if mac is also present in macVtepMultiPathTable, skip it
      if bhmacAddr in vxlanVniStatusV2.macVtepMultiPathTable:
         continue

      # if no mac is specified print them all, else print the requested mac
      if macAddr and not MacAddr.compareMacs( macAddr, bhmacAddr ):
         continue

      # if no vtep is specified print them all, else print the requested vtep
      if vtepIpList:
         filteredVtepIpList = [ vtepIp for vtepIp in vtepIpList
                                          if vtepIp in bhmvlp.vtepIpFloodList ]
         if not filteredVtepIpList:
            continue

      mvlp = VxlanMacVtepListPair()
      mvlp.macAddr = bhmacAddr
      mvlp.vtepIpListType = 'sendToAll'

      # append all VTEP IPs for this mac
      vteps = list( bhmvlp.vtepIpFloodList )
      mvlp.vtepIpList.extend( vteps )

      vs.bumVtepListTable.append( mvlp )

   return vs

#-------------------------------------------------------------------------------
# show vxlan controller address-table advertised [ vlan <vlan> | mac <mac> ]
#-------------------------------------------------------------------------------
def showVxlanVniStatusAdvertised( mode, args ):
   vsd = VxlanModel.VxlanVniStatusDirModel()
   vsd.tableType = 'Advertised'
   vlanId = args.get( 'VLAN_ID' )
   macAddr = args.get( 'MAC_ADDR' )

   if vlanId:
      vni = None
      for intfId in vtiStatusDir.vtiStatus:
         vlanVniMap = vtiStatusDir.vtiStatus[ intfId ].extendedVlanToVniMap
         if vlanId.id in vlanVniMap:
            vni = vlanVniMap[ vlanId.id ].vni
            break
      if vni and vni in vxlanVniStatusDir.vniStatusV2:
         vs = buildVxlanVniStatusV2( vni, macAddr, vlanId,
                                     vxlanVniStatusDir.vniStatusV2[ vni ],
                                     'Vxlan1' )
         vsd.vniStatus.append( vs )
      return vsd

   # scan each vni
   for vni, vniStatus in six.iteritems( vxlanVniStatusDir.vniStatusV2 ):

      vsV1 = buildVxlanVniStatusV2( vni, macAddr, vlanId, vniStatus,
                                    'Vxlan1' )
      if vsV1:
         vsd.vniStatus.append( vsV1 )

   return vsd

#-------------------------------------------------------------------------------
# show vxlan controller address-table received
#                                      [ vlan <vlan> | mac <mac> | vtep <vtep> ]
#-------------------------------------------------------------------------------
def showVxlanVniStatusReceived( mode, args ):
   vsd = VxlanModel.VxlanVniStatusDirModel()
   vsd.tableType = 'Received'
   vlanId = args.get( 'VLAN_ID' )
   macAddr = args.get( 'MAC_ADDR' )
   vteps = args.get( 'VTEP' )

   if not vxlanVniFdbStatusDir:
      return vsd

   vfsd = vxlanVniFdbStatusDir

   # scan each vni
   for vni in vfsd:
      vsV1 = buildVxlanVniStatusV2( vni, macAddr, vlanId, vfsd[ vni ],
                                    'Vxlan1', vtepIpList=vteps )
      if vsV1:
         vsd.vniStatus.append( vsV1 )

   return vsd

def showVxlanControlServiceStatus( mode, args ):
   statuses = { 'oobStateConnecting': 'connecting',
                'oobStateNegotiating': 'negotiating',
                'oobStateEstablished': 'established',
                'oobStateDisconnecting': 'disconnecting',
                'oobStateShuttingDown': 'disconnecting',
                'oobStateShutdown': 'disconnecting' }

   vcsStatus = VxlanModel.VxlanControlServiceStatus()
   controllerStatus = None
   for controller in cvxClientStatus.controllerStatus:
      if cvxClientStatus.controllerStatus[ controller ].leader:
         controllerStatus = cvxClientStatus.controllerStatus[ controller ]
         break
   if not controllerStatus:
      vcsStatus.status = 'disconnecting'
   else:
      status = controllerStatus.serviceStatusDir.service.get( 'Vxlan' )
      vcsStatus.status = statuses[ controllerStatus.connectionStatus.state ] if \
                            status and status.enabled else 'disconnecting'
   vcsStatus.resyncInProgress = vcsStateClientView.convergenceInProgress
   vcsStatus.purgeInProgress = (
         vxlanClientConvergenceStatus.unicastDeferredDeletionInProgress |
         vxlanClientConvergenceStatus.bumDeferredDeletionInProgress )
   return vcsStatus

#-------------------------------------------------------------------------------
# Build and return a VxlanArpTable CAPI object filtered by ip and vlanId with
# entries from the passed in vxlanVniStatus object.
#-------------------------------------------------------------------------------
def buildVxlanArpTable( vrf, vlanId, ip, macAddr, ipToMacByVni, arpType, intfName,
                        nd ):
   def makeArpTable( vni, ipToMac ):
      af = AFType.ipv6 if nd else AFType.ipv4
      entries = ( ( k, v ) for ( k, v ) in six.iteritems( ipToMac )
                  if not ip or str( ip ) == str( k ) )
      entries = ( ( k, v ) for ( k, v ) in entries
                  if not macAddr or MacAddr.compareMacs( macAddr, v.macAddr ) )
      entries = ( ( k, v ) for ( k, v ) in entries if k.af == af )

      arpTable = VxlanArpTable()
      arpTable.addresses = { k: VxlanArpEntry( macAddress=v.macAddr,
                                               preference=v.preference,
                                               changes=v.changeCount )
                             for ( k, v ) in entries }
      return arpTable

   def accept( vlan ):
      if not( vlan ) or ( vlanId and vlanId.id != vlan ):
         return False

      if vrf and vrf not in ipStatus.vrfIpIntfStatus:
         return False

      if vrf:
         return 'Vlan%d' % vlan in ipStatus.vrfIpIntfStatus[ vrf ].ipIntfStatus

      return True

   col = ( ( vni, vniToVlanMap.getVlan( vni, intfName ), ipToMac )
          for ( vni, ipToMac ) in ipToMacByVni )

   protocol = "nd" if nd else "arp"
   model = VxlanModel.VxlanArpTableModel( tableType=arpType, protocol=protocol )
   model.vlanToArpTable = { vlan: makeArpTable( vni, ipToMac )
                            for ( vni, vlan, ipToMac ) in col if accept( vlan ) }
   return model

#-------------------------------------------------------------------------------
# show vxlan controller arp advertised [ vlan <vlan> | ip <A.B.C.D> ]
# show vxlan controller nd advertised [ vlan <vlan> | ip <A.B.C.D> ]
#-------------------------------------------------------------------------------
def showVxlanIpToMacAdvertised( mode, args ):
   vrf = args.get( 'VRF' )
   vlanId = args.get( 'VLAN_ID' )
   ip = args.get( 'IP' )
   macAddr = args.get( 'MAC_ADDR' )
   nd = 'nd' in args
   ipToMacByVni = ( ( k, v.ipToMacTable )
                    for ( k, v ) in six.iteritems( vxlanVniStatusDir.vniStatusV2 ) )
   return buildVxlanArpTable( vrf, vlanId, ip, macAddr, ipToMacByVni,
                              'advertised', 'Vxlan1', nd )

#-------------------------------------------------------------------------------
# show vxlan controller arp received [ vlan <vlan> | ip <A.B.C.D> ]
# show vxlan controller nd received [ vlan <vlan> | ip <A.B.C.D> ]
#-------------------------------------------------------------------------------
def showVxlanIpToMacReceived( mode, args ):
   vrf = args.get( 'VRF' )
   vlanId = args.get( 'VLAN_ID' )
   ip = args.get( 'IP' )
   macAddr = args.get( 'MAC_ADDR' )
   nd = 'nd' in args
   ipToMacByVni = ( ( int( k ), v.ipToMacTable )
                    for ( k, v ) in six.iteritems( vxlanVniFdbStatusDir ) )
   return buildVxlanArpTable( vrf, vlanId, ip, macAddr, ipToMacByVni,
                              'received', 'Vxlan1', nd )

#-------------------------------------------------------------------------------
# show vxlan controller logical-router [ name <lr-name> ] [ vni <vni> ]
# show vxlan controller logical-router uplink [ name <lr-name> ]
# show vxlan controller logical-router routes [ name <lr-name> ]
#-------------------------------------------------------------------------------
def isLocalVtepAddr( vtepIp ):
   for vtiStatus in vtiStatusDir.vtiStatus.values():
      if vtiStatus.localVtepAddr == vtepIp:
         return True
   return False

def addLogicalRouter( logicalRouter, lRModel, lRName, vniFilter, processIpPort,
                      processUplinkPort, processRoute ):
   lRModel.routingTable[ lRName ] = VxlanModel.VxlanRoutingTable()
   lRModel.routingTable[ lRName ].lRCreateLocalIpPortsOnly = \
       logicalRouter.createLocalIpPortsOnly
   if processIpPort:
      for ipPort in logicalRouter.ipPort.values():
         if ipPort.vtep.stringValue != '127.0.0.1' and \
            not isLocalVtepAddr( ipPort.vtep.stringValue ):
            continue
         if vniFilter and vniFilter != ipPort.vni:
            continue
         modelIpPort = VxlanModel.LRIpPort( vni=ipPort.vni,
                                            macAddr=ipPort.mac )
         lRModel.routingTable[ lRName ].lRPort[ ipPort.ip.stringValue ] = modelIpPort

   if processUplinkPort:
      for uplinkPort in logicalRouter.ipUplinkPort.values():
         if uplinkPort.vtep.stringValue != '127.0.0.1' and \
            not isLocalVtepAddr( uplinkPort.vtep.stringValue ):
            continue
         if uplinkPort.portVlan:
            # Assuming vlan id is same in all keys
            vlanId = list( uplinkPort.portVlan )[ 0 ].vlanId
            interfaceList = [ x.port for x in uplinkPort.portVlan ]
            modelIpUplinkPort = VxlanModel.LRIpUplinkPort(
               vlan=vlanId,
               interfaces=interfaceList,
               macAddr=uplinkPort.mac )
            lRModel.routingTable[ lRName ].lRUplinkPort[
               uplinkPort.ip.stringValue ] = modelIpUplinkPort

   if processRoute:
      for route in logicalRouter.route.values():
         modelIpRoute = VxlanModel.LRIpRoute()
         for nexthop in route.nexthop:
            modelIpRoute.nexthop.append( nexthop )
         lRModel.routingTable[ lRName ].lRRoute[ route.prefix.stringValue ] = \
            modelIpRoute

def processLRStatus( lRNameFilter, vniFilter, processIpPort,
                     processUplinkPort, processRoute ):
   lRModel = VxlanModel.VxlanLogicalRouterModel()
   # pylint: disable-msg=W0212
   lRModel._vniDotted = vxlanControllerConfig.vniInDottedNotation
   if lRNameFilter:
      if lRNameFilter in lRStatus.entityPtr:
         logicalRouter = lRStatus[ lRNameFilter ]
         addLogicalRouter( logicalRouter, lRModel, lRNameFilter, vniFilter,
                           processIpPort, processUplinkPort, processRoute )
      return lRModel

   for lRName in lRStatus.entityPtr:
      logicalRouter = lRStatus[ lRName ]
      addLogicalRouter( logicalRouter, lRModel, lRName, vniFilter,
                        processIpPort, processUplinkPort, processRoute )
   return lRModel

def showVxlanLogicalRouter( mode, args ):
   vniFilter = None
   lRNameFilter = None
   if 'LOGICAL_ROUTER' in args:
      lRNameFilter = args[ 'LOGICAL_ROUTER' ][ 0 ]
   if 'VNI' in args:
      vniFilter = int( args[ 'VNI' ][ 0 ] )
      if not isValidVniWithError( mode, vniFilter, 'Vxlan1' ):
         return None

   if vniFilter:
      return processLRStatus( lRNameFilter=lRNameFilter,
                              vniFilter=vniFilter,
                              processIpPort=True,
                              processUplinkPort=False,
                              processRoute=False )
   else:
      return processLRStatus( lRNameFilter=lRNameFilter,
                              vniFilter=None,
                              processIpPort=True,
                              processUplinkPort=True,
                              processRoute=True )

def showVxlanUplinkPorts( mode, args ):
   return processLRStatus( lRNameFilter=args.get( 'LOGICAL_ROUTER' ),
                           vniFilter=None,
                           processIpPort=False,
                           processUplinkPort=True,
                           processRoute=False )

def showVxlanRoutes( mode, args ):
   return processLRStatus( lRNameFilter=args.get( 'LOGICAL_ROUTER' ),
                           vniFilter=None,
                           processIpPort=False,
                           processUplinkPort=False,
                           processRoute=True )

#-------------------------------------------------------------------------------
# The "show vxlan address-table" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show vxlan address-table [ mlag-peer ] [static|dynamic|received|unicast|evpn]
#                            [address <mac_addr>] [vtep <ip_addr>] [vlan <vlan_id>]
#-------------------------------------------------------------------------------
def getMovesAndLastMoveTime( vlanId, macAddr, host ):
   fid = bridgingConfig.vidToFidMap.get( vlanId )
   if not fid:
      fid = vlanId
   smashKey = Tac.Value( 'Bridging::HostKey', fid, macAddr )
   macEntry = smashBridgingStatus.smashFdbStatus.get( smashKey )
   if macEntry is None:
      return ( 0, 0 )
   else:
      return ( macEntry.moves, macEntry.lastMoveTime )

def filterVxlanMacAddr( host, vlanId, macAddr, hostType, vteps,
                        vtepType=SwTunnelGroupType.singleMemberGroup ):
   if vlanId:
      if vlanId.id != host.macVlanPair.vlanId:
         return False
   if macAddr:
      if Ethernet.convertMacAddrToCanonical( macAddr ) != \
            host.macVlanPair.macAddr:
         return False
   if hostType:
      if hostType != 'evpn' and vtepType == SwTunnelGroupType.multiMemberGroup:
         return False
      if hostType == 'dynamic' and host.entryType != 'learnedMac':
         return False
      if hostType == 'static' and host.entryType != 'configuredMac':
         return False
      if hostType == 'received' and host.entryType != 'receivedMac':
         return False
      if hostType == 'unicast' and \
            not Ethernet.isUnicast( host.macVlanPair.macAddr ):
         return False
      if hostType == 'evpn' and host.entryType != 'evpnMac':
         return False
   if vteps:
      if vtepType is SwTunnelGroupType.singleMemberGroup:
         if str( host.remoteVtepAddr ) not in vteps:
            return False
      else:
         swId = host.swTunnelGroupId.index
         vtepList = swGroupIdColl.getTunnelIpAddrList( swId )
         vteps = set( vteps )
         return any( str( ip ) in vteps for ip in vtepList.tunnelIpGenAddr )

   return True

def showVxlanMacAddr( mode, args ):
   showAddrTable = Tac.newInstance( 'Vxlan::ShowVxlanAddressTable',
                                   bridgingConfig.force(),
                                   vxlanConfigDir.force(),
                                   vxlanFdbStatus,
                                   vxlanFdbMultiVtepStatus.force(),
                                   swGroupIdColl.force(),
                                   smashBridgingStatus,
                                   mlagHostTable.force() )

   vlanId = args.get( 'VLAN_ID' )
   if vlanId:
      showAddrTable.vlanId = Tac.Value( 'Bridging::VlanIdOrNone', vlanId.id )
   macAddr = args.get( 'MAC_ADDR' )
   if macAddr:
      showAddrTable.hasMacAddr = True
      macAddrTac = Tac.ValueConst( 'Arnet::EthAddr', stringValue=macAddr )
      showAddrTable.macAddr = macAddrTac
   else:
      showAddrTable.hasMacAddr = False
   vteps = args.get( 'VTEP' )
   if vteps:
      for v in vteps:
         vtepIp = Arnet.IpGenAddr( str( v ) )
         showAddrTable.vtepSet.add( vtepIp )
   hostType = args.get( 'HOST_TYPE' )
   hostTypeConvert = { 'received': 'receivedHost',
                       'evpn': 'evpnHost',
                       'static': 'staticHost',
                       'dynamic': 'dynamicHost',
                       'unicast': 'unicastHost',
                       'configured': 'configuredHost' }
   if hostType and hostType in hostTypeConvert:
      showAddrTable.hostType = hostTypeConvert[ hostType ]
   else:
      showAddrTable.hostType = 'unknownHost'

   showAddrTable.isMlag = 'mlag-peer' in args

   # Add entries to table buffer according to CLI arguments.
   showAddrTable.populateMacTable()

   # Get the file descriptor C++ will output to.
   fd = sys.stdout.fileno()
   # Get the format (CLI text or JSON) C++ will generate.
   fmt = mode.session_.outputFormat()
   # Get the revision number of JSON.
   revision = mode.session_.requestedModelRevision()
   # Do actual printing in C++.
   showAddrTable.render( fd, fmt, revision )
   sys.stdout.flush()

   return VxlanModel.VxlanMacAddresses

def showVxlanMacAddrCount( mode, args ):
   vtepInputs = args.get( 'VTEP' )
   if vtepInputs is not None:
      vtepInputs = set( str( v ) for v in vtepInputs )
   addrsModel = VxlanModel.VxlanAddressTableCount()
   learnedHosts = vxlanFdbStatus.learnedHost
   multiVtepHosts = vxlanFdbMultiVtepStatus.multiVtepLearnedHost

   counter = Counter()
   for macVlanPair, learnedHost in six.iteritems( learnedHosts ):
      host = multiVtepHosts.get( macVlanPair )
      if host is None:
         vtep = str( learnedHost.remoteVtepAddr )
         if vtepInputs is None or vtep in vtepInputs:
            counter.update( [ vtep ] )
      else:
         swId = host.swTunnelGroupId.index
         vtepList = swGroupIdColl.getTunnelIpAddrList( swId )
         vteps = set( str( ip ) for ip in vtepList.tunnelIpGenAddr )
         if vtepInputs is not None:
            vteps = vteps.intersection( set( vtepInputs ) )
         counter.update( list( vteps ) )

   addrsModel.vtepCounts = dict( counter )
   return addrsModel

#-------------------------------------------------------------------------------
# The "show vxlan vtep" command, in "enable" mode.
#
# The full syntax for this command is:
#   show vxlan vtep [ type TUNNEL_TYPE ] [ detail ]
#-------------------------------------------------------------------------------
def showVxlanVteps( mode, args ):
   vtepModel = VxlanModel.VxlanVtepsModel()
   vtepModel.detailIs( 'detail' in args )
   tunnelTypeFilter = args.get( 'TUNNEL_TYPE' )
   vtiIntfIds = list( vxHwStatusDir.vxlanHwStatus )
   tunnelTypes = defaultdict( set )
   allVteps = set()
   for tunnelType, vtepStatus in vtepHwStatus.vtepStatus.items():
      if ( tunnelTypeFilter and
           VxlanCliLib.tunnelTypeToString( tunnelType ) != tunnelTypeFilter ):
         continue
      allVteps |= set( vtepStatus.vtepList )
      tunnelTypeStr = VxlanCliLib.tunnelTypeToString( tunnelType )
      for vtep in vtepStatus.vtepList:
         tunnelTypes[ vtep ].add( tunnelTypeStr )
   vtepModel.vtepTunnelTypes = {}
   for vtepIp, vtepTypeSet in six.iteritems( tunnelTypes ):
      tunnelTypeList = VxlanModel.TunnelTypeList()
      tunnelTypeList.tunnelTypes = [
            VxlanModel.TunnelType( tunnelType=t ) for t in vtepTypeSet ]
      vtepModel.vtepTunnelTypes[ vtepIp ] = tunnelTypeList
   sortedVteps = sorted( allVteps, key=lambda x: x.sortKey )
   for vtiIntfId in vtiIntfIds:
      status = vxHwStatusDir.vxlanHwStatus.get( vtiIntfId, None )
      if not status:
         continue
      vi = VxlanModel.VxlanVtepsModel.VxlanInterface( vteps=sortedVteps )
      vtepModel.interfaces[ vtiIntfId ] = vi
   if vtepModel.detail():
      shouldRenderEvpnDomain = any( status.remoteVxlanDomain
                                    for status in vtiStatusDir.vtiStatus.values() )
      for vtep, config in remoteVtepHwConfig.vtepConfig.items():
         # It's possible for a VTEP to have been classified before a tunnel is
         # established, and in some cases, a tunnel will never be established
         # (eg. local VTEP) - don't include detail info for such VTEPs
         if vtep not in allVteps:
            continue

         learnedVia = ( 'dataPlane' if config.vtepType == VtepType.dataPlaneVtep
                        else 'controlPlane' )
         evpnDomain = ( 'remoteDomain' if config.vtepType ==
            VtepType.remoteDomainVtep
               else 'localDomain' )
         macAddressLearning = ( 'datapath' if config.learningEnabled
                                else 'controlPlane' )
         remoteVtepConfig = VxlanModel.VxlanVtepsModel.RemoteVtepConfig(
            learnedVia=learnedVia, macAddressLearning=macAddressLearning,
            evpnDomain=evpnDomain if shouldRenderEvpnDomain else None )
         vtepModel.vteps[ vtep.stringValue ] = remoteVtepConfig
   return vtepModel

def showVxlanVniSummary( mode, args ):
   model = VxlanModel.VxlanVniSummaryModel()
   if not vtiStatusDir or not vtiStatusDir.vtiStatus:
      # early exit if vtiStatus is empty
      return model
   for vtiStatus in six.itervalues( vtiStatusDir.vtiStatus ):
      for vniSourcePair in six.itervalues( vtiStatus.extendedVlanToVniMap ):
         source = vniSourcePair.source or 'static'
         model.vxlanVniSources[ source ] = \
            model.vxlanVniSources.get( source, 0 ) + 1
   return model

def showVxlanVtepsSummary( mode, args ):
   model = VxlanModel.VxlanVtepsSummaryModel()
   allVteps = set()
   for vtepStatus in vtepHwStatus.vtepStatus.values():
      allVteps |= set( vtepStatus.vtepList )
   vtiIntfIds = list( vxHwStatusDir.vxlanHwStatus )
   for vtiIntfId in vtiIntfIds:
      status = vxHwStatusDir.vxlanHwStatus.get( vtiIntfId, None )
      if not status:
         continue
      model.vxlanInterfaces[ vtiIntfId ] = len( allVteps )
   return model

#-------------------------------------------------------------------------------
# The "show vxlan vni" command, in "enable" mode.
#
# The full syntax for this command is:
#   show vxlan vni [<vni>] [dot1q <vlan_id>] [interface <interface>]
#                      [source <source>]
#-------------------------------------------------------------------------------
def showVxlanVni( mode, args ):
   filters = {}
   vniFilter = args.get( 'VNI' )
   if 'INTFS' in args:
      filters[ 'intfs' ] = args[ 'INTFS' ][ 0 ]
   if 'SOURCE' in args:
      filters[ 'source' ] = args[ 'SOURCE' ][ 0 ]
   if 'dot1q' in args:
      if 'VLANS' in args:
         filters[ 'dot1q' ] = args[ 'VLANS' ][ 0 ]
      else:
         filters[ 'dot1q' ] = 'untagged'

   if vniFilter is not None:
      vni = int( vniFilter )
      if not isValidVniWithError( mode, vni, 'Vxlan1' ):
         return None

   def filterIntfs( intfName, intfs ):
      # filter out MLAG peerlink
      if ( mlagStatus.mlagState != 'disabled' and mlagStatus.peerLinkIntf and
           mlagStatus.peerLinkIntf.intfId == intfName ):
         return False
      if intfs:
         return intfName in intfs
      else:
         # default, match Ethernet, Port-Channel, and Switch interfaces
         return intfName.startswith( ( 'Ethernet', 'Port', 'Vxlan', 'Switch' ) )

   def filterVlanTag( vlan, filters ):
      vlanTagFilter = filters.get( 'dot1q' )
      if vlanTagFilter:
         if isinstance( vlanTagFilter, VlanCli.VlanSet ):
            return vlan in vlanTagFilter
         else:
            # check if vlan is untagged
            return vlan == 0
      else:
         return True

   def filterVni( vni ):
      if vniFilter:
         vniValue = VniFormat( vniFilter ).toNum()
         return vni == vniValue
      else:
         return True

   def filterSource( source, filters ):
      # Can this source be allowed to render?
      sourceFilter = filters.get( 'source' )
      if sourceFilter:
         if sourceFilter == 'static':
            # CLI configured VLAN to VNI binding will not have a source.
            # calling CLI configured bindings 'static'
            return not source
         return source == sourceFilter
      else:
         return True

   def checkXlateConsistency( vxStatus, ingressVlan, internalVlan, model ):
      # error check to make sure hwIngressVlanXlate matches
      key = Tac.Value( "Bridging::VlanXlateKey", ingressVlan, 0 )
      ingressXlate = vxStatus.hwIngressVlanXlate.get( key )
      if not ingressXlate or ingressXlate.vlanId != internalVlan:
         model.error = 'ingress and egress VLAN translation do not match'

   def buildVniBindingsToVlan( vtiStatus, mapping ):
      # For a VtiStatus, walk over all VLAN <-> VNI map and add it to
      # result provided the VLAN is not an internal VLAN with an SVI.
      # mapping is VxlanVniVlanCollection CLI model which is the output.
      dynVlanSet = VlanCli.Vlan.getDynVlanSet( mode )
      for vlan, vs in six.iteritems( vtiStatus.extendedVlanToVniMap ):
         evid = ExtendedVlanId( vlan )
         # filter out vni and source. If source is evpn, ignore it as
         # it gets handled as part of building VNIs bound to VRF.
         # Extended VLANs can't be statically defined so skip them too.
         if ( not filterVni( vs.vni ) or
              not filterSource( vs.source, filters ) or
              vs.source == "evpn" or evid.inExtendedRange() ):
            continue
         vlanModel = VxlanModel.VxlanVniVlanMappingModel()
         vlanModel.vlan = vlan
         # if source is not defined, assume it is configured by CLI
         source = vs.source or 'static'
         vlanModel.source = source
         vlanModel.dynamicVlan = source != 'static' and vlan in dynVlanSet
         ports = []
         vc = bridgingConfig.vlanConfig.get( vlan )
         if vc:
            ports = [ EthIntf( port, mode ) for port in vc.intf ]
         for intf in ports:
            # filter out interface
            if not filterIntfs( intf.name, intfNames ):
               continue

            intfConfig = bridgingConfig.switchIntfConfig.get( intf.name )
            if intfConfig:
               xlateData = None
               intfModel = VxlanModel.VxlanIntfDot1qModel()
               vxStatus = vlanXlateStatusDir.vlanXlateStatus.get( intf.name )
               if vxStatus:
                  key = Tac.Value( "Bridging::VlanXlateKey", vlan, 0 )
                  xlateData = vxStatus.hwEgressVlanXlate.get( key )
                  if xlateData:
                     checkXlateConsistency( vxStatus, xlateData.vlanId, vlan,
                                            intfModel )
               if 'Vxlan' in intf.name:
                  intfModel.dot1q = vlan
               elif intfConfig.switchportMode == 'trunk':
                  intfModel.dot1q = xlateData.vlanId if xlateData else vlan
               elif intfConfig.switchportMode == 'access':
                  intfModel.dot1q = xlateData.vlanId if xlateData else 0
               elif intfConfig.switchportMode == 'dot1qTunnel':
                  intfModel.dot1q = 0
                  intfModel.dot1qTunnel = True
               else:
                  # not sure what to do with other switchport modes
                  # excluding them for now
                  continue

               # filter out dot1q tag
               if filterVlanTag( intfModel.dot1q, filters ):
                  vlanModel.interfaces[ intf.name ] = intfModel

         if ( ( ports and not vlanModel.interfaces ) or
              ( not ports and ( 'intfs' in filters or
                                'dot1q' in filters ) ) ):
            # no interfaces found after filter. do not add this entry
            continue

         mapping.vniBindings[ int( vs.vni ) ] = vlanModel

   def buildVniBindingsToVrf( vtiStatus, mapping ):
      # For the VtiStatus, provide the mapping's vniBindingsToVrf
      # collection which contains both local and remote VNIs.
      if not filterSource( "evpn", filters ):
         t0( "evpn source didn't pass the filters", filters )
         return
      if 'intfs' in filters or 'dot1q' in filters:
         t0( "intfs and dot1q filters render empty vniBindingsToVrf" )
         return
      # Cache the reverse map of SBD-VLAN -> VRF Name.
      sbdVlanToVrfMap = defaultdict( list )
      for ( vrfName, sbdVlan ) in six.iteritems( vtiStatus.vrfToSBDVlan ):
         sbdVlanToVrfMap[ sbdVlan ].append( vrfName )

      def getVrfForSbdVlan( sbdVlan ):
         vrfList = sbdVlanToVrfMap.get( vlan, [] )
         return vrfList[ 0 ] if len( vrfList ) == 1 else None

      # Cache the reverse map of VNI -> VRF Name.
      vniToVrfMap = {}
      for vrfName, vni in six.iteritems( vtiStatus.vrfToVniMap ):
         vniToVrfMap[ vni ] = vrfName
      for vni, vrfName in six.iteritems( vtiStatus.alternateDecapVniToVrfMap ):
         vniToVrfMap[ vni ] = vrfName

      def getVrfForVni( vni ):
         return vniToVrfMap.get( vni )

      for vlan, vs in six.iteritems( vtiStatus.extendedVlanToVniMap ):
         # Only "evpn" source is allowed for these VNI bindings.
         if vs.source != "evpn":
            continue
         # Apply VNI and source filter if available.
         if not filterVni( vs.vni ):
            continue
         vrfModel = VxlanModel.VxlanVniVrfMappingModel()
         vrfModel.source = vs.source
         vrfModel.vlan = vlan
         vrfName = getVrfForVni( vs.vni )
         if vrfName is None:
            t0( "vniSource", vs, "vlan", vlan, "contains invalid mapping to VRFs",
                sbdVlanToVrfMap.get( vlan ) )
         else:
            vrfModel.vrfName = vrfName
         mapping.vniBindingsToVrf[ int( vs.vni ) ] = vrfModel
      # Now build the map using RemoteVniToVlanStatus. Note that the
      # source is still "evpn" for filter purpose. We list it as evpn
      # multicast for clarity of the provisioning model.
      source = "evpn multicast"
      for vniSbdVlan in remoteVniToVlanStatus.remoteVniToVlan.values():
         # Apply VNI and source filter if available.
         if not filterVni( vniSbdVlan.remoteVni ):
            continue
         vrfModel = VxlanModel.VxlanVniVrfMappingModel()
         vrfModel.source = source
         vlan = vniSbdVlan.vlan
         vrfModel.vlan = vlan
         vrfName = getVrfForSbdVlan( vlan )
         if vrfName is None:
            t0( "Remote VNI, SBD VLAN", vniSbdVlan,
                "contains invalid mapping to VRFs", sbdVlanToVrfMap.get( vlan ) )
         else:
            vrfModel.vrfName = vrfName
         if int( vniSbdVlan.remoteVni ) not in mapping.vniBindingsToVrf:
            mapping.vniBindingsToVrf[ int( vniSbdVlan.remoteVni ) ] = vrfModel
         else:
            t0( "There appears to be a local configuration for", vniSbdVlan )

   intfNames = set( filters.get( 'intfs', () ) )

   # build vlan to vni mapping
   model = VxlanModel.VxlanVniModel(
         _vniInDottedNotation=vxlanControllerConfig.vniInDottedNotation )
   if not vtiStatusDir or not vtiStatusDir.vtiStatus:
      # early exit if vtiStatus is empty
      return model

   for vxlanIntfName, vtiStatus in six.iteritems( vtiStatusDir.vtiStatus ):
      mapping = VxlanModel.VxlanVniVlanCollection()
      buildVniBindingsToVlan( vtiStatus, mapping )
      buildVniBindingsToVrf( vtiStatus, mapping )
      model.vxlanIntfs[ vxlanIntfName ] = mapping

   return model

#-------------------------------------------------------------------------------
# VxlanStaticMacCleaner is used to remove the static macs associated with vti
# when the vti is deleted
#-------------------------------------------------------------------------------
class VxlanStaticMacCleaner( IntfCli.IntfDependentBase ):
   #----------------------------------------------------------------------------
   # Destroys the vxlan static mac
   #----------------------------------------------------------------------------
   def setDefault( self ):
      intfName = self.intf_.name
      configuredHost = vxlanConfigDir.fdbConfig.configuredHost
      hostsInVti = [ host for host in six.itervalues( configuredHost )
                     if host.vtiIntfId == intfName ]
      for host in hostsInVti:
         del configuredHost[ host.macVlanPair ]

def _showTechGuard():
   return bridgingHwCapabilities.vxlanSupported and vxlanConfigDir.vxlanConfig

def _showTechVtepCounterGuard():
   return _showTechGuard() and bridgingHwCapabilities.vxlanVtepCtrSupported

#-------------------------------------------------------------------------------
# The "show vxlan config-sanity" command, in "enable" mode.
#
# The full syntax of this command is:
#   show vxlan config-sanity [ brief | detail ]
#
# Refer to ConfigCheckItem, ConfigCheckCategory and ConfigSanityModel CAPI
# models. A VxlanConfigSanity check function should perform a single,
# focused check on a certain config or function related to VXLAN, and return
# a single, or a list of ConfigCheckItem (i.e. an example of a list of items
# would be VLAN-VNI mappings, where each VLAN-VNI map takes an entry). A new
# check function should be added to "vxlanConfigChecks" dictionary defined
# below.
#
# The actual output rendered on the CLI or returned via CAPI is built using
# this "vxlanConfigChecks" collection, together with logic found under
# buildCommonConfigSanityModel(). CLI rendering is performed by
# ConfigSanityModel.render() method
#
# Please use decorators found in VxlanConfigSanity.py to wrap around your
# check functions. This is mainly to prevent code duplication.
#-------------------------------------------------------------------------------

def getVxlanVlans( vti, excludedSource='evpn' ):
   vxlanVlans = set()
   vtiStatus = getVtiStatus( vti )
   vnis = vniToVlanMap.getKeys( vti )
   for vni in vnis:
      if vni:
         vlan = vniToVlanMap.getVlan( vni, vti )
         if vlan not in vtiStatus.extendedVlanToVniMap:
            continue
         if vtiStatus.extendedVlanToVniMap[ vlan ].source == excludedSource:
            continue
         vxlanVlans.add( vniToVlanMap.getVlan( vni, vti ) )
   return vxlanVlans

def isL2EvpnConfigured():
   if 'evpn' in vxlanClientDir:
      return vxlanClientDir[ 'evpn' ].enabled
   return False

def isL3EvpnConfigured():
   return bool( vxlanEvpnDynamicVlans.vlans )

def isEvpnConfigured():
   return isL2EvpnConfigured() or isL3EvpnConfigured()

def vrfForInterface( intfName ):
   if intfName in l3StatusDir.intfStatus:
      return l3StatusDir.intfStatus[ intfName ].vrf
   else:
      return None # interface not configured

@GenConfigCheckItems( 'Loopback IP Address', 1 )
def checkLocalVtepConfigured( itemNameBase ):
   items = []
   for vti in vtiStatusDir.vtiStatus:
      vtiConfig = getVtiConfig( vti, create=False )
      vtiStatus = getVtiStatus( vti )

      if not vtiStatus:
         # probably will never hit this, but need to probe vtiStatus to get
         # localVtepAddr
         item = VxlanModel.ConfigCheckItem( name=itemNameBase,
                  checkPass=False,
                  detail='VtiStatus for {} does not exist!'.format( vti ) )
         items.append( item )
      elif not vtiConfig.srcIpIntf:
         item = VxlanModel.ConfigCheckItem( name=itemNameBase,
                  checkPass=False,
                  detail='Source interface for {} not configured'.format( vti ) )
         items.append( item )
      else:
         invalidIp4 = ''
         invalidIp6 = ''
         expectIp4 = vtiStatus.vxlanEncap == vxlanEncapType.vxlanEncapIp4 or \
            vtiStatus.vxlanEncap == vxlanEncapType.vxlanEncapDual
         expectIp6 = vtiStatus.vxlanEncap == vxlanEncapType.vxlanEncapIp6 or \
            vtiStatus.vxlanEncap == vxlanEncapType.vxlanEncapDual

         if expectIp4 and \
               vtiStatus.localVtepAddr in [ '0.0.0.0', '255.255.255.255' ]:
            invalidIp4 = vtiStatus.localVtepAddr
         if expectIp6 and vtiStatus.localVtepAddr6.isUnspecified:
            invalidIp6 = vtiStatus.localVtepAddr6.stringValue

         if invalidIp4 or invalidIp6:
            item = VxlanModel.ConfigCheckItem( name=itemNameBase,
                     checkPass=False,
                     detail='Invalid IP %s for %s, encapsulation %s' %
                     ( invalidIp4 + ' ' + invalidIp6, vti, vtiStatus.vxlanEncap ) )
            items.append( item )

         vrf = vrfForInterface( vtiConfig.srcIpIntf )
         if vrf and vrf != DEFAULT_VRF:
            item = VxlanModel.ConfigCheckItem( name=itemNameBase,
                  checkPass=False,
                  detail='Source interface for %s is in a non-default VRF (VRF %s)' %
                  ( vti, vrf ) )
            items.append( item )


   # if no warnings has been added here, everything is fine
   if not items:
      item = VxlanModel.ConfigCheckItem( name=itemNameBase, checkPass=True )
      items.append( item )

   return items

# get errors from resolving VLAN-VNI Mappings
def getDynVlanVniErrors( vccMode ):
   vlanErrors = set()
   vniErrors = set()

   if not vccMode:
      return vlanErrors, vniErrors

   for vtepSrc in controllerErrorStatus:
      errorStatus = controllerErrorStatus.get( vtepSrc )
      if not errorStatus:
         return None
      for port in errorStatus.portError:
         portError = errorStatus.portError.get( port )
         if not portError:
            continue
         for vlan in portError.vlanVniError:
            vniOrNone = portError.vlanVniError.get( vlan )
            vlanErrors.add( vlan )
            if vniOrNone:
               vniErrors.add( vniOrNone )

   return vlanErrors, vniErrors

@GenConfigCheckItems( 'VLAN-VNI Map', 2 )
def checkVlanVniMap( itemNameBase ):
   items = []
   for vti in vtiStatusDir.vtiStatus:

      vtiStatus = getVtiStatus( vti )
      vtiConfig = getVtiConfig( vti )
      vlanVniMap = vtiConfig.vlanToVniMap

      # display an error if no VLAN-VNI map is found in vniToVlanMap
      if not vniToVlanMap.getKeys( vti ):
         item = VxlanModel.ConfigCheckItem( name=itemNameBase,
                  checkPass=False,
                  detail='No VLAN-VNI mapping in {}'.format( vti ) )
         items.append( item )

      if vlanVniMap: # otherwise check each mappings
         vlanErrors, vniErrors = getDynVlanVniErrors(
                                          vtiStatus.controllerClientMode )

         for vlan, vni in sorted( six.iteritems( vlanVniMap ) ):
            evid = ExtendedVlanId( vlan )
            if evid.inDot1qRange() and vlan not in bridgingConfig.vlanConfig:
               items.append(
                  VxlanModel.ConfigCheckItem(
                     name=itemNameBase,
                     hasWarning=True,
                     detail=vlanNotCreatedFmt % vlan
                  )
               )
            elif evid.inDot1qRange():
               vlanConfig = bridgingConfig.vlanConfig[ vlan ]

               if vlanConfig.internal:
                  items.append( VxlanModel.ConfigCheckItem(
                     name=itemNameBase,
                     hasWarning=True,
                     detail=vlanIsInternalFmt % vlan ) )

               vlanIdFromVlanIntf = Tac.Value( "Arnet::VlanIntfId" ).vlanId

               vlanIntf = mlagStatus.localInterface
               if vlanIntf and vlanIdFromVlanIntf( vlanIntf.intfId ) == vlan:
                  items.append(
                     VxlanModel.ConfigCheckItem(
                        name=itemNameBase,
                        hasWarning=True,
                        detail=vlanIsPeerlinkFmt % vlan
                     )
                  )

            item = VxlanModel.ConfigCheckItem()
            item.name = itemNameBase
            item.checkPass = True

            # check if dynamic errors exist
            if vlan in vlanErrors:
               item.detail = dynVlanVniConflictFmt % ( 'VLAN', vlan )
               item.hasWarning = True
            elif vni in vniErrors:
               item.detail = dynVlanVniConflictFmt % ( 'VNI', vni )
               item.hasWarning = True

            # do not show all VLAN-VNI mappings as this information can be
            # retrieved using other commands, only show the ones with dynamic
            # vlan-vni conflicts
            if item.hasWarning:
               items.append( item )

   # if no warnings has been added here, everything is fine
   if not items:
      item = VxlanModel.ConfigCheckItem( name=itemNameBase, checkPass=True )
      items.append( item )

   return items

@GenConfigCheckItems( 'Flood List', 3 )
def checkFloodListConfigured( itemNameBase ):
   items = []
   # pylint: disable=too-many-nested-blocks
   for vti in vtiStatusDir.vtiStatus:
      vtiStatus = getVtiStatus( vti )
      vxlanVlans = getVxlanVlans( vti )

      # Determine list of VLANs that have at least one valid flood entry.
      validFloodlistVlans = set()
      floodlistConfigured = False

      for vfs in l2RibFloodSet.vlanFloodSet.values():
         floodlistConfigured = True
         # Skip VLANs not present in VLAN to VNI map
         if vfs.vlanId not in vxlanVlans:
            continue
         # Break whenever first valid flood entry is found
         for fs in vfs.floodSet.values():
            for dest in fs.destSet:
               if dest.destType != 'destTypeVxlan':
                  continue
               # Ignore local VTEP addrs
               vtepAddr = dest.vxlan.vtepAddr
               if vtepAddr.af == 'ipv4':
                  if vtepAddr.v4Addr not in ( vtiStatus.localVtepAddr,
                                              vtiStatus.vArpVtepAddr,
                                              vtiStatus.mlagVtepAddr ):
                     validFloodlistVlans.add( vfs.vlanId )
                     break
               elif vtepAddr.af == 'ipv6':
                  if vtepAddr.v6Addr not in ( vtiStatus.localVtepAddr6,
                                              vtiStatus.vArpVtepAddr6,
                                              vtiStatus.mlagVtepAddr6 ):
                     validFloodlistVlans.add( vfs.vlanId )
                     break

      itemPass = vtiStatus.controllerClientMode
      fromCvx = ' from CVX' if vtiStatus.controllerClientMode else ''
      # checks before checking each VXLAN VLANs
      if not vxlanVlans:
         if not isEvpnConfigured():
            item = VxlanModel.ConfigCheckItem(
                     name=itemNameBase,
                     checkPass=itemPass, hasWarning=True,
                     detail='No VXLAN VLANs' + fromCvx + ' in {}'.format( vti ) )
            items.append( item )
      elif not floodlistConfigured:
         if not isEvpnConfigured():
            item = VxlanModel.ConfigCheckItem(
                     name=itemNameBase,
                     checkPass=itemPass, hasWarning=True,
                     detail='No flood list configured' + fromCvx )
            items.append( item )

      if isL2EvpnConfigured():
         continue

      # check each VXLAN-VLANs has proper remote VTEP in flood list
      for vlan in vxlanVlans:
         if vlan not in validFloodlistVlans:
            item = VxlanModel.ConfigCheckItem(
                     name=itemNameBase,
                     checkPass=itemPass, hasWarning=True,
                     detail=noRemoteVtepVlanFloodlistFmt % vlan )
            items.append( item )

   if isL2EvpnConfigured():
      return items

   if not items:
      item = VxlanModel.ConfigCheckItem(
               name=itemNameBase,
               checkPass=True, hasWarning=False,
               detail='' )
      items.append( item )

   return items

def isSviVirtual( vlanId ):
   vlanStr = 'Vlan%d' % vlanId

   # status updated with 'ip address virtual'
   ipIntfStatus = ipStatus.ipIntfStatus.get( vlanStr )
   if ipIntfStatus and ipIntfStatus.useVirtualAddr:
      return True

   # status updated with 'ip virtual-router address'
   if vlanStr in fhrpStatus.vrIntfStatus:
      return True

   return False

# check if any Vxlan SVIs are using virtual IP
# 1. SVI configured with ip addr and "ip virtual-router"
# 2. SVI configured with "ip address virtual"
def virtualIpInSvis():
   vxlanVlans = getVxlanVlans( 'Vxlan1' )

   for vlan in vxlanVlans:
      if isSviVirtual( vlan ):
         return True

   return False

@GenConfigCheckItems( 'Routing', 4 )
def checkVirtualRouting( itemNameBase ):
   vxlanConfig = getVxlanConfig( 'Vxlan1' )
   vtiStatus = getVtiStatus( 'Vxlan1' )

   items = []
   if not vxlanConfig or not vtiStatus:
      return items
   vccMode = vtiStatus.controllerClientMode
   if virtualIpInSvis():
      # check for VARP MAC
      if vrMacStatus.varpVirtualMac == ethAddrZero:
         items.append( VxlanModel.ConfigCheckItem(
                           name=itemNameBase, checkPass=vccMode, hasWarning=True,
                           detail='Virtual MAC is not configured' ) )
      # check for VTEP IP
      if vtiStatus.vArpVtepAddr == ipAddrZero and vtiStatus.vArpVtepAddr6.isZero:
         if not isEvpnConfigured():
            items.append( VxlanModel.ConfigCheckItem(
                          name=itemNameBase, checkPass=vccMode, hasWarning=True,
                          detail='Virtual VTEP IP is not configured' ) )

   # if no warnings up to this point, all checks passed
   if not items:
      item = VxlanModel.ConfigCheckItem(
               name=itemNameBase,
               checkPass=True, hasWarning=False,
               detail='' )
      items.append( item )
   return items

@GenConfigCheckItems( 'VNI VRF ACL', 5 )
def checkVniVrfAcl( itemNameBase ):
   items = []
   vxlanConfig = getVxlanConfig( 'Vxlan1' )
   vtiStatus = getVtiStatus( 'Vxlan1' )

   if not vxlanConfig or not vtiStatus:
      return items

   vnis = list( vxlanConfig.vniToIpAclMap )
   vrfToVniMap = vtiStatus.vrfToVniMap

   if not vnis:
      # if no ingress VNI ACLs configured, everything is fine
      pass
   elif not vrfToVniMap:
      # display an error if no VRF-VNI map is found
      item = VxlanModel.ConfigCheckItem( name=itemNameBase,
               checkPass=False, detail='No VRF-VNI mapping' )
      items.append( item )
   else:
      # otherwise check each mappings
      for vni in vnis:
         if vni in vrfToVniMap.values():
            continue
         items.append( VxlanModel.ConfigCheckItem(
                           name=itemNameBase,
                           checkPass=False,
                           detail=noVniInVrfToVniFormat % vni ) )

   # if no warnings has been added here, everything is fine
   if not items:
      item = VxlanModel.ConfigCheckItem( name=itemNameBase, checkPass=True )
      items.append( item )

   return items

@GenConfigCheckItems( 'Decap VRF-VNI Map', 6 )
def checkDecapVniToVrfMap( itemNameBase ):
   items = []
   for vti in vtiStatusDir.vtiStatus:
      vtiConfig = getVtiConfig( vti )
      vtiStatus = getVtiStatus( vti )

      decapVrfNames = set( vtiConfig.alternateDecapVniToVrfMap.values() )
      vrfToVniMap = vtiStatus.vrfToVniMap

      if not decapVrfNames:
         # if no alternate decap VNI-to-VRF map configured, everything is fine
         pass
      elif not vrfToVniMap:
         # display an error if no VRF-VNI map is found
         item = VxlanModel.ConfigCheckItem( name=itemNameBase,
                  checkPass=False, detail='No VRF-VNI mapping' )
         items.append( item )
      else:
         # otherwise check each mappings
         for vrfName in decapVrfNames:
            if vrfName in vrfToVniMap:
               continue
            items.append( VxlanModel.ConfigCheckItem(
                              name=itemNameBase,
                              checkPass=False,
                              detail=noVrfInVrfToVniFormat % vrfName ) )

   # if no warnings has been added here, everything is fine
   if not items:
      item = VxlanModel.ConfigCheckItem( name=itemNameBase, checkPass=True )
      items.append( item )

   return items

@GenConfigCheckItems( 'VRF-VNI Dynamic VLAN', 7 )
def checkVrfToVniDynVlanMap( itemNameBase ):
   items = []
   for vti in vtiStatusDir.vtiStatus:
      vtiConfig = getVtiConfig( vti )
      vtiStatus = getVtiStatus( vti )

      vrfToVniMap = vtiConfig.vrfToVniMap

      if not vrfToVniMap:
         continue

      vnisWithVlan = set( vniSource.vni for
                          vniSource in vtiStatus.extendedVlanToVniMap.values() )

      for vrfName, vni in six.iteritems( vrfToVniMap ):
         if vni in vnisWithVlan:
            continue
         items.append( VxlanModel.ConfigCheckItem(
                           name=itemNameBase,
                           checkPass=False,
                           detail=noDynVlanForVrfToVniFormat %
                                  ( vrfName, vni ) ) )

   # if no warnings has been added here, everything is fine
   if not items:
      item = VxlanModel.ConfigCheckItem( name=itemNameBase, checkPass=True )
      items.append( item )

   return items

def arpResolved( vrfId, intfId, destIp ):
   ipGenAddr = Arnet.IpGenAddr( str( destIp ) )
   arpKey = Tac.newInstance( 'Arp::Table::ArpKey', vrfId, ipGenAddr, intfId )
   if ipGenAddr.af == 'ipv4':
      return arpKey in arpTableStatus.arpEntry
   else:
      return arpKey in arpTableStatus.neighborEntry

def blackholeVia( via ):
   return via.hop == ipAddrZero and via.intfId == ''

# returns an ConfigCheckItem containing results and descriptions from checking
# route and arp to ipAddr
def ipRouteCheckHelper( vrf, ipAddr, itemName, routeTrie ):
   ipAddr = str( ipAddr )
   item = VxlanModel.ConfigCheckItem( name=itemName )
   ipGenPrefix = Arnet.IpGenPrefix( ipAddr )
   route, vias = routeTrie.getRoute( ipGenPrefix )

   if not vias or not route:
      item.detail = 'No route to ' + ipAddr
      item.checkPass = False
      return item

   nexthopArpResolved = False
   # handle ECMP case
   for via in vias:
      lookupIp = ipAddr if Arnet.IpGenAddr( str( via.hop ) ).isAddrZero else via.hop
      if blackholeVia( via ):
         continue
      elif arpResolved( vrf, via.intfId, lookupIp ):
         nexthopArpResolved = True
         break

   if not nexthopArpResolved:
      item.detail = 'Unresolved ARPs to ' + ipAddr
      item.checkPass = False

   return item

@GenConfigCheckItems( 'Remote VTEP', 1 )
def checkRemoteVtepRoute( itemNameBase, routeTrie ):
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return []

   floodVtepsToCheck = set()
   for vfs in l2RibFloodSet.vlanFloodSet.values():
      for fs in vfs.floodSet.values():
         for dest in fs.destSet:
            if dest.destType == 'destTypeVxlan':
               floodVtepsToCheck.add( dest.vxlan.vtepAddr )

   # we don't have to check configured VTEP addresses of this switch
   # including the local VTEP address and VARP VTEP address
   floodVtepsToCheck -= set( Arnet.IpGenAddr( str( addr ) ) for addr in
      ( vtiStatus.localVtepAddr, vtiStatus.localVtepAddr6,
        vtiStatus.vArpVtepAddr, vtiStatus.vArpVtepAddr6,
        vtiStatus.mlagVtepAddr, vtiStatus.mlagVtepAddr6 ) )

   items = []
   for vtep in floodVtepsToCheck:
      # underlay reachability is currently only in vrf 0 ( default vrf )
      item = ipRouteCheckHelper( underlayDefaultVrfId, vtep,
                                 itemNameBase, routeTrie )
      if not item.checkPass:
         items.append( item )

   if not items:
      items.append( VxlanModel.ConfigCheckItem( name=itemNameBase ) )

   return items

def vlanIpAddrSet( vlanId ):
   vlanStr = 'Vlan%d' % vlanId

   # this is how Ira shows interface-level IP addresses
   # first go through status, then config
   if vlanStr in ipStatus.ipIntfStatus:
      if ipStatus.ipIntfStatus[ vlanStr ].activeAddrWithMask != zeroAddrWithMask:
         return True

   if vlanStr in ipConfig.ipIntfConfig:
      if ipConfig.ipIntfConfig[ vlanStr ].addrWithMask != zeroAddrWithMask:
         return True

   return False

def vxlanRoutingEnabled( vtiStatus ):
   # VXLAN routing is enabled whenever a VRF to VNI mapping is configured or a VLAN
   # mapped to VNI has an assigned IP address.
   if vtiStatus.vrfToVniMap:
      return True

   vxlanVlan = getVxlanVlans( 'Vxlan1' )
   if not vxlanVlan:
      return False

   for vlan in vxlanVlan:
      if vlanIpAddrSet( vlan ) or isSviVirtual( vlan ):
         return True

   return False

def registerVxlanBridgingConfigCheckCallback( callback ):
   # platform-specific code should register a callback here
   if callback:
      vxlanBridgingConfigCheckCallback.append( callback )

def registerVxlanRoutingConfigCheckCallback( callback ):
   # platform-specific code should register a callback here
   if callback:
      vxlanRoutingConfigCheckCallback.append( callback )

@GenConfigCheckItems( 'VXLAN Bridging', 1 )
def checkVxlanBridging( itemNameBase ):
   vtiStatus = getVtiStatus( 'Vxlan1' )

   items = []
   if not vtiStatus:
      return [ VxlanModel.ConfigCheckItem( name=itemNameBase ) ]

   if not bridgingHwCapabilities.vxlanSupported:
      items.append( VxlanModel.ConfigCheckItem( name=itemNameBase,
                              detail='VXLAN Bridging not supported' ) )
      return items

   # call callback functions registered by platform-specific Cli plugins
   for checkCallback in vxlanBridgingConfigCheckCallback:
      checkCallback( itemNameBase, items )

   if not items:
      items.append( VxlanModel.ConfigCheckItem( name=itemNameBase ) )

   return items

@GenConfigCheckItems( 'VXLAN Routing', 1 )
def checkVxlanRouting( itemNameBase ):
   vtiStatus = getVtiStatus( 'Vxlan1' )

   items = []
   if not vtiStatus:
      return [ VxlanModel.ConfigCheckItem( name=itemNameBase ) ]

   if not bridgingHwCapabilities.vxlanRoutingSupported:
      items.append( VxlanModel.ConfigCheckItem( name=itemNameBase,
                              detail='VXLAN Routing not supported' ) )
      return items

   if not vxlanRoutingEnabled( vtiStatus ):
      items.append( VxlanModel.ConfigCheckItem( name=itemNameBase,
                              detail='VXLAN Routing not enabled' ) )
      return items

   # call callback functions registered by platform-specific Cli plugins
   for checkCallback in vxlanRoutingConfigCheckCallback:
      checkCallback( itemNameBase, items )

   if not items:
      items.append( VxlanModel.ConfigCheckItem( name=itemNameBase ) )

   return items

@ConfigCheckItem( 'CVX Server', 1 )
def cvxServerHostCheck( item, routeTrie ):
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return item

   if not vtiStatus.controllerClientMode:
      item.detail = 'Not in controller client mode'
      return item

   leaderController = None
   for controller in cvxClientStatus.controllerStatus:
      if cvxClientStatus.controllerStatus[ controller ].leader:
         leaderController = controller
         break

   if not leaderController:
      item.checkPass = False
      item.detail = 'No leader Found'
      return item

   # underlay reachability is currently only in vrf 0 ( default vrf )
   return ipRouteCheckHelper( underlayDefaultVrfId, leaderController,
                              item.name, routeTrie )

#-------------------------------------------------------------------------------
#
# VXLAN MLAG check ( including checking for multi-VTEP MLAG configs )
#
# We check for default VTEP address and MLAG VTEP address for Vxlan1 interface
# on local switch and peer switch. We have the following 4 scenarios, depending
# on what's configured locally and on the peer
#
# 1. Local ( default ), peer ( default )
#   * check that local_default == peer_default
#
# 2. Local ( default, mlag ), peer ( default, mlag )
#   * check that local_mlag == peer_mlag
#   * check that local_default != peer_default
#
# 3. Local ( default ), peer ( default, mlag )
#   * check that local_default in [ peer_default, peer_mlag ]
#
# 4. Local ( default, mlag ), peer ( default )
#   * this is inverse of case 3
#
# Note that we can keep config-sanity logic simple by show errors on only one of
# the switches invovled in MLAG for case 3&4 ( the one with MLAG VTEP IP, since
# the one without MLAG VTEP IP can simply skip checking for MLAG VTEP IP )
#
#-------------------------------------------------------------------------------

def mlagConnected():
   return mlagStatus.mlagState in [ 'primary', 'secondary' ] \
          and mlagStatus.negotiationStatus == 'Connected'

def localMlagVtepIpConfigured():
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return False
   return vtiStatus.mlagVtepAddr not in [ '0.0.0.0', '255.255.255.255' ] or \
      not vtiStatus.mlagVtepAddr6.isZero

def getPeerVtepAddrByType( vtepType ):
   return mlagVxlanStatus.vtepTypeAddr.get( vtepType, None )

def getPeerVtepAddr6ByType( vtepType ):
   return mlagVxlanStatus.vtepTypeAddr6.get( vtepType, None )

def isNullAddr( addr ):
   return addr is None or Arnet.IpGenAddr( str( addr ) ).isAddrZero

def validateAddr( addr, validAddrList ):
   'given a list of valid'
   if all( isNullAddr( a ) for a in validAddrList ):
      return isNullAddr( addr )
   return addr in validAddrList

@ConfigCheckItem( 'Peer VTEP IP', 1 )
def peerVtepIpCheck( item ):
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return item

   if not mlagConnected():
      item.detail = 'MLAG peer is not connected'
      return item

   peerVtepAddr = getPeerVtepAddrByType( 'defaultVtep' )
   peerMlagVtepAddr = getPeerVtepAddrByType( 'mlagVtep' )
   peerVtepAddr6 = getPeerVtepAddr6ByType( 'defaultVtep' )
   peerMlagVtepAddr6 = getPeerVtepAddr6ByType( 'mlagVtep' )
   if not peerVtepAddr and not peerVtepAddr6:
      item.detail = 'No VTEP IP from peer'
      item.checkPass = False
      return item

   if localMlagVtepIpConfigured():
      if peerMlagVtepAddr or peerMlagVtepAddr6:
         # when MLAG VTEP IP is configured, there needs to be three distinct VTEP
         # IPs on the MLAG: the MLAG VTEP IP and one VTEP IP from each VTEP.
         if vtiStatus.localVtepAddr == peerVtepAddr or \
            vtiStatus.localVtepAddr6 == peerVtepAddr6:
            item.detail = 'Peer has same VTEP IP'
            item.checkPass = False
      else:
         # peer has no MLAG VTEP IP configured
         if not ( validateAddr( peerVtepAddr,
                                [ vtiStatus.localVtepAddr,
                                  vtiStatus.mlagVtepAddr ] ) and
                  validateAddr( peerVtepAddr6,
                                [ vtiStatus.localVtepAddr6,
                                  vtiStatus.mlagVtepAddr6 ] ) ):
            item.detail = 'Invalid peer VTEP IP'
            item.checkPass = False
   else:
      # when no MLAG VTEP IP is configured, local VTEP IP need to be the same
      # as peer VTEP IP or peer MLAG IP
      if not ( validateAddr( vtiStatus.localVtepAddr,
                             [ peerVtepAddr, peerMlagVtepAddr ] ) and
               validateAddr( vtiStatus.localVtepAddr6,
                             [ peerVtepAddr6, peerMlagVtepAddr6 ] ) ):
         item.detail = 'Invalid local VTEP IP'
         item.checkPass = False

   return item

@ConfigCheckItem( 'MLAG VTEP IP', 2 )
def mlagVtepIpCheck( item ):
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return item
   if not mlagConnected():
      return item
   if not localMlagVtepIpConfigured():
      return item

   peerVtepAddr = getPeerVtepAddrByType( 'defaultVtep' )
   peerMlagVtepAddr = getPeerVtepAddrByType( 'mlagVtep' )
   peerVtepAddr6 = getPeerVtepAddr6ByType( 'defaultVtep' )
   peerMlagVtepAddr6 = getPeerVtepAddr6ByType( 'mlagVtep' )
   if not peerVtepAddr and not peerVtepAddr6:
      item.detail = 'No VTEP IP from peer'
      item.checkPass = False
      return item

   if not peerMlagVtepAddr and not peerMlagVtepAddr6:
      if not ( validateAddr( peerVtepAddr,
                             [ vtiStatus.localVtepAddr,
                               vtiStatus.mlagVtepAddr ] ) and
               validateAddr( peerVtepAddr6,
                             [ vtiStatus.localVtepAddr6,
                               vtiStatus.mlagVtepAddr6 ] ) ):
         item.detail = 'Invalid peer VTEP IP'
         item.checkPass = False
   elif not ( validateAddr( vtiStatus.mlagVtepAddr, [ peerMlagVtepAddr ] ) and
              validateAddr( vtiStatus.mlagVtepAddr6, [ peerMlagVtepAddr6 ] ) ):
      item.detail = 'Peer has different MLAG VTEP IP'
      item.checkPass = False

   return item

@ConfigCheckItem( 'Peer VLAN-VNI', 3 )
def peerVlanVniCheck( item ):
   vtiConfig = getVtiConfig( 'Vxlan1' )
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiConfig or not vtiStatus:
      return item
   if not mlagConnected():
      return item

   # vlan-vni mapping gets sent from primary to secondary, if we are primary
   # prompt the user to check output of this command from the secondary
   if mlagStatus.mlagState == 'primary':
      item.detail = 'Check this command from the peer'
      return item

   errorItem = VxlanModel.ConfigCheckItem(
                  name=item.name,
                  checkPass=False, hasWarning=False,
                  detail='VLAN-VNI Mapping not identical' )

   # VtiStatus contains configured and dynamic mapping
   if ( len( vtiStatus.extendedVlanToVniMap ) !=
        len( mlagVxlanStatus.vniVtiToDynVlanMap ) ):
      return errorItem

   for vlan, vsp in sorted( six.iteritems( vtiStatus.extendedVlanToVniMap ) ):
      vniVti = VniVtiPair( vsp.vni, vsp.vti )
      if vniVti not in mlagVxlanStatus.vniVtiToDynVlanMap:
         return errorItem
      if vlan != mlagVxlanStatus.vniVtiToDynVlanMap[ vniVti ]:
         return errorItem

   return item

@ConfigCheckItem( 'MLAG Inactive State', 5 )
def mlagInactiveCheck( item ):
   vtiConfig = getVtiConfig( 'Vxlan1' )
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiConfig or not vtiStatus:
      return item
   if mlagStatus.mlagState != 'inactive':
      return item
   if ( Tac.now() - mlagProtoStatus.lastStateChangeTime ) >= 600:
      return VxlanModel.ConfigCheckItem(
            name=item.name, checkPass=True, hasWarning=True,
            detail="MLAG state is inactive" )
   return VxlanModel.ConfigCheckItem(
             name=item.name, checkPass=False, hasWarning=True,
             detail="MLAG state is inactive. Dynamic SVIs won't be allocated \n"
                    "until either MLAG configuration is disabled, the \n"
                    "MLAG state converges to primary or secondary or \n"
                    "10 minutes has elapsed" )

@ConfigCheckItem( 'Virtual VTEP IP', 4 )
def virtualVtepCheck( item ):
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return item
   if not mlagConnected():
      return item

   varpVtepAddr = getPeerVtepAddrByType( 'varpVtep' )
   varpVtepAddr6 = getPeerVtepAddr6ByType( 'varpVtep' )
   if not ( validateAddr( vtiStatus.vArpVtepAddr, [ varpVtepAddr ] ) and
            validateAddr( vtiStatus.vArpVtepAddr6, [ varpVtepAddr6 ] ) ):
      return VxlanModel.ConfigCheckItem( name=item.name,
                                         checkPass=False, hasWarning=False,
                                         detail='Peer vVTEP IP does not match' )

   return item

# collection to hold all config-sanity checks
vxlanConfigChecks = {
   'localVtep': {
      'priority': 1,
      "description": "Local VTEP Configuration Check",
      "checks": [
         checkLocalVtepConfigured,
         checkVlanVniMap,
         checkFloodListConfigured,
         checkVirtualRouting,
         checkVniVrfAcl,
         checkVrfToVniDynVlanMap,
         checkDecapVniToVrfMap,
      ]
   },
   'remoteVtep': {
      'priority': 2,
      "description": "Remote VTEP Configuration Check",
      "checks": [
         checkRemoteVtepRoute,
      ]
   },
   'pd': {
      'priority': 3,
      "description": "Platform Dependent Check",
      "checks": [
         checkVxlanBridging,
         checkVxlanRouting,
      ]
   },
   'cvx': {
      'priority': 4,
      "description": "CVX Configuration Check",
      "checks": [
         cvxServerHostCheck,
      ]
   },
   'mlag': {
      'priority': 5,
      "description": "MLAG Configuration Check",
      "detail": "Run 'show mlag config-sanity' to verify MLAG config",
      "checks": [
         peerVtepIpCheck,
         mlagVtepIpCheck,
         virtualVtepCheck,
         peerVlanVniCheck,
         mlagInactiveCheck,
      ]
   }
}

# Checks that need a RouteTrie built from FIB
routeTrieChecks = [ checkRemoteVtepRoute, cvxServerHostCheck ]

def checkDynVlanVniWarning( configSanityModel ):
   category = configSanityModel.categories.get( 'localVtep', None )
   if category:
      for item in category.items:
         if item.name.startswith( 'VLAN-VNI Map' ) and item.hasWarning:
            return True
   return False

def checkWarnings( configSanityModel, cliMode ):
   # check presence of any warning
   hasWarning = any( c.hasWarning for
                     c in six.itervalues( configSanityModel.categories ) )
   if hasWarning:
      cliMode.addWarning( warnExplainedMsg )

   hasDynVlanVniWarning = checkDynVlanVniWarning( configSanityModel )
   if hasDynVlanVniWarning:
      cliMode.addWarning( dynVlanVniWarnMsg )

def saveConfigSanityModelToSysdb( model, mode ):
   # Save config-sanity output to Sysdb so that it can be streamed via TerminAttr
   vxlanFeature = configSanityFeatureDir.features.newMember( 'vxlan' )
   newSeqNum = 1 if vxlanFeature.seqNum == sys.maxsize else vxlanFeature.seqNum + 1

   def capiToConfigSanityResult( checkPass, hasWarning ):
      if not checkPass:
         return 'Fail'
      elif hasWarning:
         return 'Warn'
      return 'Pass'

   itemCache = {}

   # Delete stale entries
   for tag, category in six.iteritems( vxlanFeature.categories ):
      if tag not in model.categories:
         del vxlanFeature.categories[ tag ]
         continue
      categoryItemNames = set( i.name for i in model.categories[ tag ].items )
      for itemName in category.items:
         if itemName not in categoryItemNames:
            del vxlanFeature.categories[ tag ].items[ itemName ]

   for tag, category in six.iteritems( model.categories ):
      vxlanFeature.categories.newMember( tag )
      vxlanFeature.categories[ tag ].description = category.description
      vxlanFeature.categories[ tag ].result = capiToConfigSanityResult(
            category.allCheckPass, category.hasWarning )
      vxlanFeature.categories[ tag ].priority = category.priority()

      for item in category.items:
         result = capiToConfigSanityResult( item.checkPass, item.hasWarning )
         i = Tac.Value( 'ConfigSanity::Item', item.name, item.detail, result )
         itemTag = item.name
         if item.name in itemCache:
            itemCache[ item.name ] += 1
            itemTag += '_' + str( itemCache[ item.name ] )
         else:
            itemCache[ item.name ] = 1
         vxlanFeature.categories[ tag ].items[ itemTag ] = i

   for warning in mode.session.warnings_:
      vxlanFeature.warnings[ warning ] = True

   vxlanFeature.seqNum = newSeqNum

   return model

def showConfigSanity( mode, args ):
   # pylint: disable-msg=W0612
   model = VxlanModel.ConfigSanityModel()
   if 'brief' in args:
      model.detailLevelIs( 'brief' )
   elif 'detail' in args:
      model.detailLevelIs( 'detailed' )

   if not vtiConfigDir.vtiConfig:
      mode.addMessage( 'No VXLAN interface' )
      # Write ConfigSanityModel to Sysdb so that CVP can stream state via
      # TerminAttr Prevent write to Sysdb when supervisor is not active
      if em.redundancyStatus().mode == 'active':
         saveConfigSanityModelToSysdb( model, mode )
      return model

   routeTrie = createRouteTrie()
   # perform all config-sanity checks and build model
   for checkTag, check in six.iteritems( vxlanConfigChecks ):
      category = VxlanModel.ConfigCheckCategory()
      category.description = check[ 'description' ]
      for doCheck in check[ 'checks' ]:
         if doCheck in routeTrieChecks:
            item = doCheck( routeTrie )
         else:
            item = doCheck()
         # there are 2 types of sanity check functions, one type returns a list
         # while the other type returns a single CheckItem
         if isinstance( item, list ):
            category.items.extend( item )
         else:
            category.items.append( item )
      category.allCheckPass = all( i.checkPass for i in category.items )
      category.hasWarning = any( i.hasWarning for i in category.items )
      category.priorityIs( check.get( 'priority', 200 ) )
      category.detail = check.get( 'detail', '' )
      model.categories[ checkTag ] = category

   checkWarnings( model, mode )

   # Write ConfigSanityModel to Sysdb so that CVP can stream state via TerminAttr
   # Prevent write to Sysdb when supervisor is not active
   if em.redundancyStatus().mode == 'active':
      saveConfigSanityModelToSysdb( model, mode )
   return model

# This command requires the building of a new RouteTrie object from FIB, in order
# to not use too much memory in the case of this command running many times
# in parallel, a lock is used to make sure only one instance of command is building
# this a RouteTrie at a time.
@synchronized( routeTrieLockConfigSanity )
def createRouteTrie():
   fecModeStatus = Tac.newInstance( 'Smash::Fib::FecModeStatus', 'fms' )
   _ = Tac.newInstance( 'Ira::FecModeSm', l3Config, fecModeStatus )
   multiAgentMode = bool( fecModeStatus.fecMode == 'fecModeUnified' )

   fwdStatus = None
   fwd6Status = None
   fwdGenStatus = None
   if toggleFibGenMountPathEnabled():
      fwdGenStatus = ( forwardingGenStatusMulti if multiAgentMode
                       else forwardingGenStatus )
   else:
      fwdStatus = forwardingStatusMulti if multiAgentMode else forwardingStatus
      fwd6Status = forwarding6StatusMulti if multiAgentMode else forwarding6Status

   newRouteTrie = RouteTrie( routingStatus, fwdStatus, routing6Status,
                             fwd6Status, fwdGenStatus )
   return newRouteTrie

#-------------------------------------------------------------------------------
# end of "show vxlan config-sanity" command section
#-------------------------------------------------------------------------------

def showVxlanFloodTraffic( mode, args ):
   vtiStatus = getVtiStatus( 'Vxlan1' )
   model = VxlanModel.VxlanFloodTrafficModel()
   if not vtiStatus:
      return model

   model.defaultFlooding = not vtiStatus.vxlanFloodTraffic.enabled
   model.arpFlooding = model.defaultFlooding or vtiStatus.vxlanFloodTraffic.arp
   model.ndFlooding = model.defaultFlooding or vtiStatus.vxlanFloodTraffic.nd
   model.unknownMacFlooding = model.defaultFlooding
   return model

def Plugin( entityManager ):
   global configSanityFeatureDir
   global vxlanConfigDir, vxHwStatusDir, vtiConfigDir, vtiStatusDir
   global vxlanFdbStatus
   global vxlanFdbMultiVtepStatus
   global swGroupIdColl
   global vxlanEcnConfig
   global vxlanFloodsetConfig
   global vxlanCounterConfigDir
   global bridgingHwCapabilities, bridgingHwEnabled
   global bridgingConfig, smashBridgingStatus
   global arpTableStatus
   global fhrpStatus, vrMacStatus
   global bridgingCliConfig, vxAgentCounter
   global vxlanVniStatusDir
   global vxlanVniFdbStatusDir
   global serviceConfigDir
   global vxlanControllerConfig
   global controllerErrorStatus
   global vniToVlanMap
   global mlagStatus
   global mlagHostTable
   global mlagVxlanStatus
   global mlagProtoStatus
   global em
   global vxlanStatusDir
   global vcsStateClientView
   global lRStatus
   global cvxClientStatus
   global hwCounterFeatureStatusDir
   global vlanXlateStatusDir
   global vxlanHwCounter
   global ip6Config
   global ipConfig
   global ipStatus
   global vxlanClientConvergenceStatus
   global vxlanClientDir
   global vxlanCliClientConfig
   global vxlanEvpnDynamicVlans
   global remoteVniToVlanStatus
   global remoteVtepHwConfig
   global ipsecConfig
   global vtepHwStatus
   global routingHwStatusCommon
   global routingHwStatus
   global l2RibFloodSet
   global l3Config
   global l3StatusDir
   global vniPolicerHwStatusDir
   global vniPolicerEncapCounterTable
   global vniPolicerDecapCounterTable
   global vniPolicerEncapSnapshotCounterTable
   global vniPolicerDecapSnapshotCounterTable
   global macRewriteCliConfig
   global macRewriteCapability
   global vxlanCountersConfig
   global vxlanVniCountersDir, vxlanVniCountersCheckpointDir
   global vxlanAggregateCountersDir, vxlanAggregateCountersCheckpointDir
   global pbrVniIntfConfig
   global pbrCliConfig
   global configTagInput
   global vtepSipValidationStatusCli
   global routingStatus
   global routing6Status
   global forwardingStatusMulti
   global forwarding6StatusMulti
   global forwardingGenStatusMulti
   global forwardingStatus
   global forwarding6Status
   global forwardingGenStatus
   global icmpHelperInputConfigDir

   em = entityManager
   readerInfo = SmashLazyMount.mountInfo( 'reader' )

   routingStatus = SmashLazyMount.mount( em, "routing/status",
                                          'Smash::Fib::RouteStatus',
                                          readerInfo )
   routing6Status = SmashLazyMount.mount( em, "routing6/status",
                                          'Smash::Fib6::RouteStatus',
                                          readerInfo )

   forwardingStatusMulti = SmashLazyMount.mount( em, "forwarding/unifiedStatus",
                                                'Smash::Fib::ForwardingStatus',
                                                readerInfo )
   forwarding6StatusMulti = SmashLazyMount.mount( em, "forwarding6/unifiedStatus",
                                                 'Smash::Fib6::ForwardingStatus',
                                                 readerInfo )
   forwardingGenStatusMulti = SmashLazyMount.mount(
                                 em, "forwardingGen/unifiedStatus",
                                 'Smash::FibGen::ForwardingStatus', readerInfo )

   forwardingStatus = SmashLazyMount.mount( em, "forwarding/status",
                                                'Smash::Fib::ForwardingStatus',
                                                readerInfo )
   forwarding6Status = SmashLazyMount.mount( em, "forwarding6/status",
                                                 'Smash::Fib6::ForwardingStatus',
                                                 readerInfo )
   forwardingGenStatus = SmashLazyMount.mount(
                                 em, "forwardingGen/status",
                                 'Smash::FibGen::ForwardingStatus', readerInfo )

   configSanityFeatureDir = LazyMount.mount( entityManager, 'configsanity/feature',
                                             'ConfigSanity::FeatureDir', 'w' )
   vxAgentCounter = SmashLazyMount.mount( entityManager, "vxlan/counter",
                                          "Vxlan::VxlanAgentCounter",
                                          readerInfo )
   smashBridgingStatus = SmashLazyMount.mount( entityManager, "bridging/status",
                                               "Smash::Bridging::Status",
                                               readerInfo )
   # Lazy mount not supported for Shark yet.
   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   shmemMg = shmemEm.getMountGroup()
   l2RibFloodSet = shmemMg.doMount( 'bridging/l2Rib/floodOutput',
                                    'L2Rib::FloodSetOutput',
                                    Shark.mountInfo( 'shadow' ) )

   l3Config = LazyMount.mount( entityManager, 'l3/config',
                               'L3::Config', 'ri' )

   l3StatusDir = LazyMount.mount( entityManager, 'l3/intf/status',
                                 'L3::Intf::StatusDir', 'r' )

   arpTableStatus = SmashLazyMount.mount( entityManager, "arp/status",
                                          'Arp::Table::Status',
                                           readerInfo )
   fhrpStatus = LazyMount.mount( entityManager,
                                 'routing/fhrp/status',
                                 'Routing::Fhrp::StatusV4',
                                 'r' )
   vrMacStatus = LazyMount.mount( entityManager,
                                  'routing/fhrp/vrMacStatus',
                                  'Routing::Fhrp::VirtualRouterMacStatus',
                                  'r' )
   pbrCliConfig = LazyMount.mount( entityManager, 'pbr/input/pmap/cli',
                                   'Pbr::PbrConfig', 'r' )
   pbrVniIntfConfig = ConfigMount.mount( entityManager,
                                         "pbr/input/intf/config/vxlan",
                                         "PolicyMap::IntfConfig", 'w' )
   vxlanStatusDir = LazyMount.mount( entityManager,
                                     "vxlan/status",
                                     "Vxlan::VxlanStatusDir", "r" )
   vxlanConfigDir = ConfigMount.mount( entityManager,
                                       "vxlan/config",
                                       "Vxlan::VxlanConfigDir", "w" )
   vxlanEcnConfig = ConfigMount.mount( entityManager,
                                       "vxlan/ecnConfig",
                                       "Vxlan::EcnConfig", "w" )
   vxlanFloodsetConfig = ConfigMount.mount( entityManager,
                                            "vxlan/floodsetConfig",
                                            "Vxlan::VxlanFloodsetConfig", "w" )
   vxlanCounterConfigDir = LazyMount.mount( entityManager,
                                            "vxlan/counterconfig",
                                            "Vxlan::VxlanCounterConfigDir", "w" )
   vxHwStatusDir = LazyMount.mount( entityManager,
                                     "vxlan/hardware/status",
                                     "Vxlan::VxlanHwStatusDir", "r" )
   vxlanFdbStatus = SmashLazyMount.mount( entityManager,
                                          "vxlan/hardware/fdbStatus",
                                          "Smash::Vxlan::FdbStatus",
                                          readerInfo )
   vxlanFdbMultiVtepStatus = LazyMount.mount( entityManager,
                                     "vxlan/hardware/fdbMultiVtepStatus",
                                     "Vxlan::FdbMultiVtepStatus", "r" )
   swGroupIdColl = LazyMount.mount( entityManager,
                                    "vxlan/hardware/swTunnelGroupIdColl",
                                    "Vxlan::SwTunnelGroupIdColl", "r" )
   vtiConfigDir = ConfigMount.mount( entityManager,
                                     "interface/config/eth/vxlan",
                                     "Vxlan::VtiConfigDir", "w" )
   vtiStatusDir = LazyMount.mount( entityManager,
                                   "interface/status/eth/vxlan",
                                   "Vxlan::VtiStatusDir", "r" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   bridgingHwEnabled = LazyMount.mount( entityManager, "bridging/hwenabled",
                                        "Bridging::HwEnabled", "r" )
   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
   bridgingCliConfig = ConfigMount.mount( entityManager,
                                        "bridging/input/config/cli",
                                        "Bridging::Input::CliConfig", "w" )
   vlanXlateStatusDir = LazyMount.mount( entityManager, "bridging/vlanxlate/status",
                                         "Bridging::VlanXlateStatusDir", "r" )
   vxlanVniStatusDir = LazyMount.mount( entityManager,
                                     "vxlan/version2/vniStatusDir",
                                     "VxlanController::VniStatusDirV2", "r" )

   vxlanClientDir = LazyMount.mount( entityManager, "vxlan/clientDir",
                                     "Tac::Dir", "ri" )
   vxlanCliClientConfig = ConfigMount.mount( entityManager,
                                             "vxlan/clientDir/cli",
                                             "Vxlan::VxlanClientConfig",
                                             "wc" )
   vxlanEvpnDynamicVlans = LazyMount.mount( entityManager,
                                            "bridging/input/dynvlan/vlan/evpn",
                                            "Bridging::Input::VlanIdSet", "r" )
   # LazyMount mlag/status, Mlag::Status and its dependent paths
   mlagStatus = MlagStatusLazyMounter( entityManager )
   mlagHostTable = LazyMount.mount( entityManager,
                                    "mlag/hostTable",
                                    "Mlag::HostTable", "r" )
   mlagVxlanStatus = LazyMount.mount( entityManager,
                                      'mlag/vxlan/status',
                                      'Mlag::VxlanStatus', 'r' )
   mlagProtoStatus = LazyMount.mount( entityManager,
                                      "mlag/proto",
                                      "Mlag::ProtoStatus", "r" )
   # To let the CVX infrastructure to know that Vxlan controller service is
   # enabled/disabled on the switch
   serviceConfigDir = ConfigMount.mount( entityManager,
                                           "mgmt/controller/service/config",
                                           "Controller::ServiceConfigDir", "w" )

   vxlanVniFdbStatusDir = entityManager.mount(
      "vxlancontroller/version2/vni", "Tac::Dir", "ri" )
   lRStatus = LazyMount.mount( entityManager,
                               "vxlancontroller/version2/logicalRouter",
                               "Tac::Dir", "ri" )
   vxlanControllerConfig = LazyMount.mount( entityManager,
                                     "vxlancontroller/config",
                                     "VxlanController::Config", "r" )
   controllerErrorStatus = LazyMount.mount( entityManager,
                                      "interface/status/vxlan/errorStatus/version2",
                                      "Tac::Dir", "ri" )
   vcsStateClientView = LazyMount.mount( entityManager,
                                         "vxlan/version2/vcsStateClientView",
                                         "VxlanController::VcsStateClientViewV2",
                                         "r" )
   cvxClientStatus = LazyMount.mount( entityManager, "mgmt/controller/status",
                                           "ControllerClient::Status", "r" )
   hwCounterFeatureStatusDir = LazyMount.mount( entityManager,
                                      "flexCounter/featureStatusDir",
                                      'Tac::Dir', "ri" )
   vxlanHwCounter = LazyMount.mount( entityManager, "vxlan/hardware/counter",
                                        "Vxlan::VxlanHwCounter", "w" )

   ip6Config = LazyMount.mount( entityManager, "ip6/config", "Ip6::Config", "r" )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )

   vxlanClientConvergenceStatus = LazyMount.mount(
      entityManager,
      "vxlan/clientConvergenceStatus",
      "Vxlan::VxlanClientConvergenceStatus", "r" )

   # Boxing vxlanCtrlConfig since VniMatcher needs it at mod-load
   vxlanCtrlCfgBox.append( vxlanControllerConfig )

   vniToVlanMap = VniToVlanMap()

   mountPath = Tac.Type( 'Vxlan::RemoteVniToVlanStatus' ).mountPath( "irb",
                                                                     "Vxlan1" )
   remoteVniToVlanStatus = SmashLazyMount.mount( entityManager, mountPath,
                                                 "Vxlan::RemoteVniToVlanStatus",
                                                 readerInfo )
   ipsecConfig = LazyMount.mount( entityManager, 'ipsec/ike/config',
                                  'Ipsec::Ike::Config', 'r' )
   remoteVtepHwConfig = LazyMount.mount( entityManager, "vxlan/remoteVtepHwConfig",
                                         "Vxlan::RemoteVtepHwConfigDir", "r" )
   vtepHwStatus = LazyMount.mount( entityManager, "vxlan/vtepHwStatus",
                                   "Vxlan::VtepHwStatusDir", "r" )
   routingHwStatusCommon = LazyMount.mount( entityManager,
                                            "routing/hardware/statuscommon",
                                            "Routing::Hardware::StatusCommon", "r" )
   routingHwStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                      "Routing::Hardware::Status", "r" )

   vniPolicerHwStatusDir = LazyMount.mount( entityManager,
                                            "vxlan/hardware/vniPolicingStatus",
                                            "Vxlan::VniPolicingHwStatusDir", "r" )

   # Mount VNI policer smash counter tables as Sand Gen4 based platforms
   # read counters from smash tables
   readerInfo = SmashLazyMount.mountInfo( 'reader' )
   mountPath = f'flexCounters/counterTable/VniPolicingEncap/{FapId.allFapsId}'
   vniPolicerEncapCounterTable = SmashLazyMount.mount(
         entityManager, mountPath, "Vxlan::VniPolicingEncapCountersTable",
         readerInfo )

   mountPath = f'flexCounters/snapshotTable/VniPolicingEncap/{FapId.allFapsId}'
   vniPolicerEncapSnapshotCounterTable = SmashLazyMount.mount(
         entityManager, mountPath, "Vxlan::VniPolicingEncapCountersTable",
         readerInfo )

   mountPath = f'flexCounters/counterTable/VniPolicingDecap/{FapId.allFapsId}'
   vniPolicerDecapCounterTable = SmashLazyMount.mount(
         entityManager, mountPath, "Vxlan::VniPolicingDecapCountersTable",
         readerInfo )

   mountPath = f'flexCounters/snapshotTable/VniPolicingDecap/{FapId.allFapsId}'
   vniPolicerDecapSnapshotCounterTable = SmashLazyMount.mount(
         entityManager, mountPath, "Vxlan::VniPolicingDecapCountersTable",
         readerInfo )

   macRewriteCliConfig = ConfigMount.mount( em,
                                            'vxlan/hardware/srcMacRewrite/config',
                                            'Vxlan::Input::MacRewriteConfig',
                                            'w' )
   macRewriteCapability = LazyMount.mount( em,
                                           'vxlan/hardware/srcMacRewrite/capability',
                                           'Vxlan::MacRewriteCapability',
                                           'r' )
   vxlanCountersConfig = LazyMount.mount( em,
                                          'vxlan/hardware/counters/config',
                                          'Vxlan::VxlanCountersConfig',
                                          'w' )
   configTagInput = ConfigMount.mount( entityManager, 'configTag/input/cli',
                                     'ConfigTag::ConfigTagInput',
                                     'w' )
   vtepSipValidationStatusCli = ConfigMount.mount( entityManager,
         'vxlan/vtepValidationStatus/cli',
         'Vxlan::VtepSipValidationStatus',
         'w' )

   # ICMP helper config dir
   icmpHelperInputConfigDir = LazyMount.mount( em,
                                               "tepicmphelper/input/config",
                                               "Tac::Dir", "wi" )

   smi = Smash.mountInfo( 'reader' )
   vxlanVniCountersDir = shmemEm.doMount( 'interface/counter/vxlan/vni/current',
                                       'Smash::Interface::AllVxlanCounterDir', smi )
   vxlanVniCountersCheckpointDir = shmemEm.doMount(
      'interface/counter/vxlan/vni/snapshot',
      'Smash::Interface::AllVxlanCounterSnapshotDir', smi )
   vxlanAggregateCountersDir = shmemEm.doMount(
      'interface/counter/vxlan/aggregate/current',
      'Smash::Interface::AllIntfCounterDir', smi )
   vxlanAggregateCountersCheckpointDir = shmemEm.doMount(
      'interface/counter/vxlan/aggregate/snapshot',
      'Smash::Interface::AllIntfCounterSnapshotDir', smi )

   # Register vxlan cli commands with "show tech-support"
   TechSupportCli.registerShowTechSupportCmd(
      '2013-05-20 12:20:55',
      cmds=[ 'show interface vxlan 1-$',
             'show vxlan counter software',
             'show vxlan counter varp',
             'show vxlan vtep',
             'show vxlan address-table',
             'show vxlan config-sanity detail',
            ],
      cmdsGuard=_showTechGuard )

   # Time stamp added to maintain historical ordering of commands in show tech
   TechSupportCli.registerShowTechSupportCmd(
      '2016-12-21 06:20:00',
      cmds=[ 'show vxlan flood vtep' ],
      cmdsGuard=_showTechGuard )

   # Register commands in show tech-support extended evpn
   TechSupportCli.registerShowTechSupportCmd(
      '2017-11-03 12:06:10',
      cmds=[ 'show interface vxlan $',
             'show vxlan address-table',
           ],
      cmdsGuard=_showTechGuard,
      extended='evpn' )

   # Register VTEP counter command
   TechSupportCli.registerShowTechSupportCmd(
      '2020-12-04 15:56:14',
      cmds=[ 'show vxlan counter vtep' ],
      cmdsGuard=_showTechVtepCounterGuard )

   # Register summary commands for show tech-support summary
   TechSupportCli.registerShowTechSupportCmd(
      '2020-06-15 13:31:06',
      summaryCmds=[ 'show vxlan vni summary' ],
      summaryCmdsGuard=_showTechGuard )

   TechSupportCli.registerShowTechSupportCmd(
      '2020-07-02 17:12:23',
      summaryCmds=[ 'show vxlan vtep summary' ],
      summaryCmdsGuard=_showTechGuard )

def validateMulticastAddr( addr ):
   if addr.af == "ipv4":
      return Arnet.IpGenPrefix( "224.0.0.0/4" ).contains( addr )
   else:
      return Arnet.IpGenPrefix( "ff00::/8" ).contains( addr )

def ipPimParseSg( groupOrSource1, groupOrSource2 ):
   """Given two IPv4/v6 addresses, where one is unicast and the other is multicast,
   returns an (S,G) pair. Where S is the unicast address and G is the multicast
   address"""

   if ( groupOrSource1 is None ) and ( groupOrSource2 is None ):
      return ( None, None )

   addrType1 = ""
   addrType2 = ""
   addr1 = groupOrSource1
   addr2 = groupOrSource2
   if groupOrSource1:
      groupOrSource1 = Arnet.IpGenAddr( groupOrSource1 )
      addrType1 = "M" if validateMulticastAddr( groupOrSource1 ) else "U"

   if groupOrSource2:
      groupOrSource2 = Arnet.IpGenAddr( groupOrSource2 )
      addrType2 = "M" if validateMulticastAddr( groupOrSource2 ) else "U"

   if addrType1 == addrType2:
      raise ValueError

   if addrType1 == "M":
      # The first arg is a multicast address
      return ( addr2, addr1 )
   else:
      # The second arg is a multicast address
      return ( addr1, addr2 )
