# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# Functions related to Vxlan Config Sanity Checking

from __future__ import absolute_import, division, print_function
import Arnet
from CliPlugin import VxlanModel
from CliPlugin.IraCommonCli import AddressFamily
import IpUtils
import Tac
from Toggles.RoutingLibToggleLib import toggleFibGenMountPathEnabled

# pylint: disable=W0212

warnExplainedMsg = ( 'Your configuration contains warnings. This does not'
                     ' mean misconfigurations. But you may wish to re-check'
                     ' your configurations.' )
dynVlanVniWarnMsg = ( 'There are dynamic VLAN-VNI mapping errors.'
                      ' See syslog for more detail.' )

vlanNotCreatedFmt = 'VLAN %d does not exist'

vlanIsInternalFmt = 'VLAN %d is used as an internal VLAN'

vlanIsPeerlinkFmt = 'VLAN %d is the peer-link SVI VLAN'

dynVlanVniConflictFmt = 'dynamic %s %d conflict'

noRemoteVtepVlanFloodlistFmt = 'No remote VTEP in VLAN %d'

noVniInVrfToVniFormat = 'No VRF-VNI mapping for VNI %d'

noVrfInVrfToVniFormat = 'No VRF-VNI mapping for VRF %s'

noDynVlanForVrfToVniFormat = 'No dynamic VLAN-VNI mapping for VRF %s VNI %s'

underlayDefaultVrfId = 0

#-------------------------------------------------------------------------------
# decorators for sanity check functions
#
# use:
#  @ConfigCheckItem if function is returning one ConfigCheckItem
#  @GenConfigCheckItems if function is returning a list of ConfigCheckItems.
#     When this decorator is used, the functions themselves have to perform
#     object creation
#-------------------------------------------------------------------------------

# wrap a check function with ConfigCheckItem creation
def ConfigCheckItem( itemName, priority ):
   """Function wrapper for config check functions. Prevents code duplication"""
   def nameWrap( f ):
      def funcWrap( routeTrie=None ):
         item = VxlanModel.ConfigCheckItem()
         item._priority = priority
         item.name = itemName
         if routeTrie:
            item = f( item, routeTrie )
         else:
            item = f( item )
         return item
      return funcWrap
   return nameWrap

# wrap a config check function that generates ConfigCheckItems
def GenConfigCheckItems( itemNameBase, priority ):
   def nameWrap( f ):
      def funcWrap( routeTrie=None ):
         if routeTrie:
            items = f( itemNameBase, routeTrie )
         else:
            items = f( itemNameBase )
         for item in items:
            item._priority = priority
         return items
      return funcWrap
   return nameWrap

#-------------------------------------------------------------------------------
# helper classes
#
#-------------------------------------------------------------------------------

class RouteTrie( object ):
   def __init__( self, routingStatus, forwardingStatus,
                 routing6Status, forwarding6Status, forwardingGenStatus ):
      self.routingStatus = routingStatus
      self.forwardingStatus = forwardingStatus
      self.routing6Status = routing6Status
      self.forwarding6Status = forwarding6Status
      self.forwardingGenStatus = forwardingGenStatus
      self.trie = self._newTrie( AddressFamily.ipv4 )
      self.trie6 = self._newTrie( AddressFamily.ipv6 )
      self.trieBuilder = self._newTrieBuilder()
      self.trieBuilder6 = self._newTrieBuilder( af='ipv6' )

   def _newTrie( self, af ):
      return Tac.newInstance( 'Routing::TrieGen', 'trie', af )

   def _newTrieBuilder( self, af='ipv4' ):
      if af == 'ipv4':
         return Tac.newInstance( 'Routing::TrieGenBuilder', self.routingStatus, None,
                                 self.forwardingStatus, None,
                                 self.forwardingGenStatus, None, None, self.trie )
      else:
         return Tac.newInstance( 'Routing::TrieGenBuilder', None,
                                 self.routing6Status, None, self.forwarding6Status,
                                 self.forwardingGenStatus, None, None, self.trie6 )

   def getRoute( self, prefix ):
      while True:
         if prefix.af == AddressFamily.ipv4:
            prefixMatch = self.trie.longestMatch( prefix ).v4Prefix
            maxLen = 32
            routingStatus = self.routingStatus
            forwardingStatus = self.forwardingStatus
         elif prefix.af == AddressFamily.ipv6:
            prefixMatch = self.trie6.longestMatch( prefix ).v6Prefix
            maxLen = 128
            routingStatus = self.routing6Status
            forwardingStatus = self.forwarding6Status

         if toggleFibGenMountPathEnabled():
            forwardingStatus = self.forwardingGenStatus

         if not routingStatus or not forwardingStatus:
            return None, None
         if prefixMatch.isNullPrefix:
            break

         if prefixMatch in routingStatus.route:
            route = routingStatus.route[ prefixMatch ]
            if not route:
               return None, None
            fec = forwardingStatus.fec.get( route.fecId )
            if not fec:
               return None, None
            else:
               vias = []
               for i in range( len( fec.via ) ):
                  if fec.via[ i ].intfId:
                     vias.append( fec.via[ i ] )
               return route, vias

         if prefixMatch.len == 0:
            break

         ip = IpUtils.IpAddress( prefixMatch.address )
         ip = IpUtils.IpAddress( ip.toNum() &
                                 ( 0xFFFFFFFF << ( maxLen - prefixMatch.len + 1 ) ) )
         prefix = Arnet.IpGenPrefix( str( ip ) + '/' + str( prefixMatch.len - 1 ) )

      return None, None
