#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from CliModel import Dict, Str, Bool, Model, Int, GeneratorList, GeneratorDict, Enum
from CliPlugin.IntfModel import InterfaceStatus
from VxlanVniLib import VniFormat
from ArnetModel import Ip4Address, Ip6Address, IpGenericPrefix
from operator import itemgetter
import collections
from ArnetModel import MacAddress
import Ethernet
from IntfModels import Interface
from Arnet import IpGenAddr, sortIntf
import six

largeInt = int if six.PY3 else long # pylint:disable=long-builtin

#-------------------------------------------------------------------------------
# VxlanIntf specific Eapi Models
#-------------------------------------------------------------------------------
class VxlanInterfaceStatus( InterfaceStatus ):
   class VxlanRemoteVtepAddrs( Model ):
      remoteVtepAddr = GeneratorList(
            valueType=Ip4Address, help='List of IPv4 vteps' )
      remoteVtepAddr6 = GeneratorList(
            valueType=Ip6Address, help='List of IPv6 vteps' )

   class VniSourcePair( Model ):
      vni = Int( help="Vni" )
      source = Str( help='Mapping source' )

   class VxlanVtepPrefixes( Model ):
      learnFrom = Enum( values=( 'learnFromAny', 'learnFromList',
                                 'learnFromFloodList', 'learnFromDefault' ),
                        help='Origin of VTEP learn restriction' )
      prefixList = GeneratorList( valueType=IpGenericPrefix,
                                  help='List of vtep (prefixes)' )

   vniInDottedNotation = Bool( default=False,
                               help='True if vni is displayed in dotted notation' )
   srcIpIntf = Str( help='Source interface for vxlan', optional=True )
   vArpVtepAddr = Str( help='IP address of virtual vtep', optional=True )
   srcIpAddr = Str( help='IP address of the source interface', optional=True )
   srcIpAddrV6 = Ip6Address( help='IPv6 address of the source interface',
                             optional=True )
   udpPort = Int( help='VXLAN UDP port' )
   secUdpPort = Int( help='VXLANsec UDP port' )
   mcastGrpDecap = Str( help='IP multicast group list for decap', optional=True )
   vlanToVniMap = GeneratorDict( keyType=int, valueType=VniSourcePair,
                                 help='Vlan to VNI source pair mapping' )
   vlanToVtepList = Dict( keyType=int, valueType=VxlanRemoteVtepAddrs,
                        help='Per Vlan Headend Vtep list' )
   staticFloodlists = Bool( help='Static Floodlists Configured' )
   replicationMode = Str( help='Replication/Flood Mode', optional=True )
   vlanToLearnRestrict = Dict( keyType=int, valueType=VxlanVtepPrefixes,
                        help='Per Vlan Learn list', optional=True )
   vrfToVniMap = Dict( keyType=str, valueType=largeInt,
                       help='VRF to VNI mapping', optional=True )
   decapVniToVrfMap = Dict( keyType=largeInt, valueType=str,
                            help='Decapsulation VNI to VRF mapping', optional=True )

   dataPathLearningMode = Bool( help='Datapath Learning Mode' )
   controllerClientMode = Bool( help='Controller Client Mode' )
   controllerControlPlane = Bool( help='Controller Control Plane' )
   vccImportVlan = Str( help='Import-enabled VLANs' )
   vccArpProxy = Bool( help='ARP proxying when in Vxlan Controller Client Mode' )
   vccNdProxy = Bool( help='Neighbor Discovery proxying when in Vxlan Controller'
                           ' Client Mode' )
   vxlanArpProxy = Bool( help='ARP proxying for requests received over VXLAN' )
   vxlanNdProxy = Bool( help='ND proxying for requests received over VXLAN' )
   arpReplyRelayMode = Bool( help='ARP Reply Relay Mode' )
   arpLocalAddress = Bool( help='Enable rewrite of virtual IP with local IP in '
                           'VXLAN ARP requests' )
   portDot1qVniMapping = Bool( help='VNI mapping to port, dot1q tag pairs' )
   # _mlagActive is hidden in CAPI output following instruction in AID2150.
   _mlagActive = Bool( help="MLAG is configured and active", optional=True )
   mlagSharedRouterMacAddr = MacAddress( help='MAC address of the VXLAN Shared'
                                              ' Router Mac' )
   vtepAddrMask = Str( help='IP address mask assigned to VTEP',
                       optional=True )
   mlagSrcIpIntf = Interface( help='Source interface for MLAG VTEP IP',
                              optional=True )
   mlagSrcIpAddr = Ip4Address( help='IP address of the MLAG source interface',
                               optional=True )
   mlagSrcIpAddrV6 = Ip6Address( help='IPv6 address of the MLAG source interface',
                               optional=True )
   use32BitVni = Bool( help='VNI length in the VXLAN header' )
   vtepToVtepBridging = Bool( help='VTEP to VTEP bridging is enabled' )
   vtepSetForSourcePruning = GeneratorList( valueType=Ip4Address,
         help='Source pruning VTEP list' )
   vtepSourcePruningAll = Bool(
         help='Source pruning for all VTEPs' )
   decapFilterMode = Enum( values=( 'filterEnabled', 'filterEnabledIntf',
                                    'filterRelaxedAll', 'filterRelaxedIntf',
                                    'filterDisabled' ),
                           help='VXLAN decapsulation filter mode' )
   decapFilterIntf = GeneratorList( valueType=Interface,
                           help='VXLAN decapsulation filter excluded interfaces' )
   decapRouteAll = Bool( help='Routing all VXLAN decasulated packets is enabled' )
   mcastRouting = Bool( help='Multicast routing is enabled' )
   vxlanEncapsulation = Enum( values=( 'ipv4', 'ipv6', 'ipv4AndIpv6' ),
                              help='VXLAN encapsulation type' )
   vArpVtepSrcIpIntf = Interface( help='Source interface for virtual VTEP IP',
                                     optional=True )
   vArpVtepAddrV6 = Ip6Address( help='IPv6 address of the virtual VTEP'
                                        'source interface',
                                        optional=True )
   floodLearnedAll = Bool( help='Flood VTEPs learned via datapath learning',
                           optional=True )
   vtepToSecProfileTotal = Int( help='Total active secure VTEPs', optional=True )
   txInterval = Int( help="Rate in microseconds at which BFD control packets are "
                     "sent", optional=True )
   rxInterval = Int( help="Rate in microseconds at which BFD control packets are "
                     "received", optional=True )
   detectMult = Int( help="Number of consecutive BFD packets missed before failure",
                     optional=True )
   bfdPrefixList = Str( help="Prefix list for filtering BFD sessions",
                        optional=True )
   remoteVxlanDomain = Bool( help="VXLAN domain is remote", optional=True )
   tcpMssV4 = Int( help="VXLAN TCP MSS value for IPV4 packet",
                        optional=True )
   tcpMssV6 = Int( help="VXLAN TCP MSS value for IPV6 packet",
                        optional=True )

   def renderSrcIpAddr( self, mlag=False ):
      if not self.mlagSrcIpIntf and mlag:
         # render is optional for mlagSrcIpIntf
         return

      if mlag:
         srcIpIntf = self.mlagSrcIpIntf.stringValue if self.mlagSrcIpIntf else ''
      else:
         srcIpIntf = self.srcIpIntf
      mlagPrefix = "MLAG " if mlag else ""
      srcIntfConfigured = srcIpIntf or "not configured"
      printBuf = "  %sSource interface is %s" % ( mlagPrefix, srcIntfConfigured )

      if srcIpIntf:
         srcIpAddrs = []
         if mlag:
            if self.vxlanEncapsulation != 'ipv6':
               srcIpAddr = str( self.mlagSrcIpAddr )
               if not IpGenAddr( srcIpAddr ).isAddrZero:
                  srcIpAddrs.append( srcIpAddr )

            if self.vxlanEncapsulation != 'ipv4':
               srcIpAddr = str( self.mlagSrcIpAddrV6 )
               if not IpGenAddr( srcIpAddr ).isAddrZero:
                  srcIpAddrs.append( srcIpAddr )
         else:
            if self.vxlanEncapsulation != 'ipv6':
               srcIpAddr = str( self.srcIpAddr )
               if not IpGenAddr( srcIpAddr ).isAddrZero:
                  srcIpAddrs.append( srcIpAddr )

            if self.vxlanEncapsulation != 'ipv4':
               srcIpAddr = str( self.srcIpAddrV6 )
               if not IpGenAddr( srcIpAddr ).isAddrZero:
                  srcIpAddrs.append( srcIpAddr )

         srcIpAddr = ' and '.join( srcIpAddrs )
         if srcIpAddr:
            printBuf += " and is active with %s" % srcIpAddr
         else:
            printBuf += " and is inactive"
      print( printBuf )

   def renderUdpPort( self ):
      print( "  Listening on UDP port %d" % self.udpPort )

   def renderVxlanEncapsulation( self ):
      if self.vxlanEncapsulation != 'ipv4':
         if self.vxlanEncapsulation == 'ipv4AndIpv6':
            print( "  Vxlan Encapsulation is IPv4 and IPv6" )
         else:
            print( "  Vxlan Encapsulation is IPv6" )

   def renderVarpVtepAddr( self ):
      if self.vArpVtepSrcIpIntf:
         print( "  Virtual VTEP source interface is %s " % self.vArpVtepSrcIpIntf )
         varpVtepAddrs = []
         if self.vxlanEncapsulation != 'ipv6' and self.vArpVtepAddr:
            varpVtepIpAddr = str( self.vArpVtepAddr )
            if not IpGenAddr( varpVtepIpAddr ).isAddrZero:
               varpVtepAddrs.append( varpVtepIpAddr )

         if self.vxlanEncapsulation != 'ipv4' and self.vArpVtepAddrV6:
            varpVtepIpAddr = str( self.vArpVtepAddrV6 )
            if not IpGenAddr( varpVtepIpAddr ).isAddrZero:
               varpVtepAddrs.append( varpVtepIpAddr )
         varpVtepIpAddr = ' and '.join( varpVtepAddrs )
         if varpVtepIpAddr:
            print( "  Virtual VTEP IP address is %s" % varpVtepIpAddr )
      elif self.vArpVtepAddr:
         print( "  Virtual VTEP IP address is %s" % self.vArpVtepAddr )

   def renderVtepAddrMask( self ):
      if not self.vtepAddrMask or self.vtepAddrMask == '255.255.255.255':
         return
      print( "  VTEP address mask is %s" % self.vtepAddrMask )

   def renderMcastGrpDecapAddr( self ):
      if not self.mcastGrpDecap:
         return

      print( "  Multicast group decap address list is: " )
      grps = self.mcastGrpDecap.split()
      log = ''
      hdr = ' ' * 4
      for grp in grps:
         log += '%-15s' % grp
         if len( hdr + log ) > 72:
            print( hdr + log )
            log = ''
            hdr = ' ' * 4
      if log:
         print( hdr + log )

   def renderDatapathLearningMode( self ):
      if self.controllerClientMode and not self.dataPathLearningMode:
         print( "  Remote MAC learning via VCS" )
      elif self.controllerControlPlane:
         if self.dataPathLearningMode:
            print( "  Remote MAC learning via EVPN and Datapath" )
         else:
            print( "  Remote MAC learning via EVPN" )
      elif self.dataPathLearningMode:
         print( "  Remote MAC learning via Datapath" )
      else:
         print( "  Remote MAC learning is disabled" )

   def renderVccImportVlan( self ):
      if not self.controllerClientMode:
         return

      if self.vccImportVlan == '':
         print( "  Remote MAC learning from Vxlan Controller Service disabled "
                "in all VLANs" )
      else:
         print( "  Remote MAC learning from Vxlan Controller Service enabled in "
                "VLAN(s): %s" % self.vccImportVlan )

   def renderVccArpProxy( self ):
      if not self.controllerClientMode:
         return
      print( "  ARP proxying %s in Vxlan Controller Client Mode" %
             ( "enabled" if self.vccArpProxy else "disabled" ) )

   def renderVccNdProxy( self ):
      if not self.controllerClientMode:
         return
      print( "  Neighbor Discovery proxying %s in Vxlan Controller Client Mode" %
             ( "enabled" if self.vccNdProxy else "disabled" ) )

   def renderVxlanArpProxy( self ):
      if not self.vtepToVtepBridging and not self.vxlanArpProxy:
         return
      print( "  ARP proxying %s for requests received over VXLAN" %
             ( "enabled" if self.vxlanArpProxy else "disabled" ) )

   def renderVxlanNdProxy( self ):
      if not self.vtepToVtepBridging and not self.vxlanNdProxy:
         return
      print( "  ND proxying %s for requests received over VXLAN" %
             ( "enabled" if self.vxlanNdProxy else "disabled" ) )

   def renderArpReplyRelayMode( self ):
      if self.arpReplyRelayMode:
         print( "  ARP Reply relay is enabled" )

   def renderArpLocalAddress( self ):
      if self.arpLocalAddress:
         print( "  Rewrite of virtual IP with local IP in VXLAN ARP is enabled " )

   def renderPortDot1qVniMapping( self ):
      mapping = "port, dot1q tag pairs" if self.portDot1qVniMapping else "VLANs"
      print( "  VNI mapping to %s" % mapping )

   def renderVlanToVniMap( self ):
      vlanToVniMap = sorted( self.vlanToVniMap, key=itemgetter( 0 ) )
      notConfigured = "not configured" if not vlanToVniMap else ""

      print( "  Static VLAN to VNI mapping is %s" % notConfigured )

      # Loop through all mapping through sorted Vlan key
      vlanToVniMapBySource = collections.defaultdict( list )
      for vlan, vsp in vlanToVniMap:
         vni = vsp.vni
         source = vsp.source
         vlanToVniMapBySource[ source ].append( ( vlan, vni ) )

      def renderMap( vlanToVniSourceMap ):
         i = 1
         display = "   "
         for vlan, vni in vlanToVniSourceMap:
            vniStr = VniFormat( vni, self.vniInDottedNotation )
            display += "{:18s}".format( " [%d, %s]" % ( vlan, vniStr ) )
            if not i % 4:
               print( display )
               display = "   "
            i += 1

         if display:
            print( display )

      # Loop through sorted source key, "" or static should be first
      for source, sourceVlanToVniMap in \
            sorted( six.iteritems( vlanToVniMapBySource ) ):
         if source:
            print( "  Dynamic VLAN to VNI mapping for '%s' is" % source )

         renderMap( sourceVlanToVniMap )

      if vlanToVniMapBySource:
         print( "  Note: All Dynamic VLANs used by VCS are internal VLANs." )
         print( "        Use 'show vxlan vni' for details." )

   def renderFloodVtepList( self ):
      if not self.vlanToVtepList:
         return

      print( '  Headend replication flood vtep list is:' )
      for ( vlan, vl ) in sorted( six.iteritems( self.vlanToVtepList ),
                                  key=lambda k__: k__[ 0 ] ):
         hdr = '  %4d' % vlan
         trailer = ''
         # Note: we're not sorting the ip addresses
         for ip in vl.remoteVtepAddr:
            trailer += ' %-15s' % ip
            if len( hdr + trailer ) > 72:
               print( hdr + trailer )
               trailer = ''
               hdr = ' ' * 6
         for ip in vl.remoteVtepAddr6:
            trailer += ' %-15s' % ip.stringValue
            if len( hdr + trailer ) > 72:
               print( hdr + trailer )
               trailer = ''
               hdr = ' ' * 6
         if trailer:
            print( hdr + trailer )

   def renderReplicationMode( self ):
      mode_to_string = { 'unknown': "not initialized yet",
                        'headendNonVcs': "headend with Flood List Source: CLI",
                        'headendVcs': "headend with Flood List Source: %s" %
                        self.floodlistSourceString() }
      print( "  Replication/Flood Mode is %s" %
             mode_to_string[ self.replicationMode ] )

   def floodlistSourceString( self ):
      sources = []
      if self.controllerClientMode:
         sources.append( 'VCS' )
         if self.controllerControlPlane:
            sources.append( 'EVPN' )
      elif self.controllerControlPlane:
         sources.append( 'EVPN' )
         if self.staticFloodlists:
            sources.append( 'CLI' )
      if self.floodLearnedAll:
         sources.append( 'Datapath learning' )
      return ', '.join( sources )

   def renderReplicationList( self ):
      mode_to_render = { 'unknown': None,
                         'headendNonVcs': self.renderFloodVtepList,
                         'headendVcs': self.renderFloodVtepList }
      if self.replicationMode in mode_to_render and \
            mode_to_render[ self.replicationMode ]:
         mode_to_render[ self.replicationMode ]()

   def renderLearnFrom( self ):
      mode_to_string = { 'learnFromAny': "any IP address",
                        'learnFromFloodList': "only those in flood list",
                        'learnFromList': "only those in explicit learn list",
                        'learnFromDefault': "<default value>" }
      print( "  Learn VTEPS from %s" % mode_to_string[ self.learnFrom ] )

   def renderVrfToVniMap( self ):
      notConfigured = "not configured" if not self.vrfToVniMap else ""
      print( "  Static VRF to VNI mapping is %s" % notConfigured )

      # Loop through all mapping through sorted Vrf name
      for ( vrf, vni ) in sorted( six.iteritems( self.vrfToVniMap ),
                                  key=lambda k__1: k__1[ 0 ] ):
         vniStr = VniFormat( vni, self.vniInDottedNotation )
         print( "   [%s, %s]" % ( vrf, vniStr ) )

   def renderDecapVniToVrfMap( self ):
      if not self.decapVniToVrfMap:
         return
      print( "  Static VRF to alternate decapsulation VNIs mapping is" )

      vrfToVnisMap = collections.defaultdict( list )
      for vni, vrf in six.iteritems( self.decapVniToVrfMap ):
         vrfToVnisMap[ vrf ].append( vni )

      # Loop through all mapping through sorted Vrf name
      for vrf, vnis in sorted( six.iteritems( vrfToVnisMap ) ):
         vniStr = ', '.join( str( VniFormat( vni, self.vniInDottedNotation ) ) for
                             vni in sorted( vnis ) )
         print( "   [%s, %s]" % ( vrf, vniStr ) )

   def renderVxlanSharedRouterMacAddr( self ):
      if self.mlagSharedRouterMacAddr != '00:00:00:00:00:00':
         macName = ( "MLAG Shared Router MAC" if self._mlagActive
                     else "Shared Router MAC" )
         macAddr = Ethernet.convertMacAddrCanonicalToDisplay(
                      self.mlagSharedRouterMacAddr.stringValue )
         print( "  %s is %s" % ( macName, macAddr ) )

   def renderVxlanHeaderVniLength( self ):
      if not self.use32BitVni:
         return
      print( "  VXLAN header VNI length is 32 bits" )

   def renderVtepToVtepBridging( self ):
      if not self.vtepToVtepBridging:
         return

      if self.remoteVxlanDomain:
         print( "  VTEP to VTEP bridging to remote domain is enabled" )
      else:
         print( "  VTEP to VTEP bridging is enabled" )

      if self.vtepSourcePruningAll:
         print( "  Source pruning for all VTEPs" )
         return

      vtepSetForSourcePruning = list( self.vtepSetForSourcePruning )
      if not vtepSetForSourcePruning:
         return
      print( '  Source pruning VTEP list is:' )
      trailer = '  '
      for ip in vtepSetForSourcePruning:
         trailer += ' %-15s' % ip
         if len( trailer ) > 72:
            print( trailer )
            trailer = '  '
      print( trailer )

   def renderMcastRouting( self ):
      if not self.mcastRouting:
         return
      print( "  Multicast routing is enabled" )

   def renderDecapFilter( self ):
      if self.decapFilterMode == 'filterEnabled':
         return
      filterStr = '  Decapsulation filter'
      if self.decapFilterMode == 'filterDisabled':
         filterStr += ' is disabled'
      elif self.decapFilterMode == 'filterRelaxedAll':
         filterStr += ' excludes interfaces in multiple VRFs'
      else:
         excludedIntfs = ', '.join( str( intf ) for intf in sortIntf(
            self.decapFilterIntf ) )
         # decapFilterIntf is shared between modes filterRelaxedIntf and
         # filterEnabledIntf
         if self.decapFilterMode == 'filterRelaxedIntf':
            filterStr += ' excludes %s' % excludedIntfs
         elif self.decapFilterMode == 'filterEnabledIntf':
            filterStr += ' only includes %s in default VRF' % excludedIntfs
      print( filterStr )

   def renderVxlanSecurityStatus( self ):
      if self.vtepToSecProfileTotal:
         print( '  VTEP VXLAN security: active for %d VTEPs. '
                'Listening on UDP port %d' %
                ( self.vtepToSecProfileTotal, self.secUdpPort ) )

   def renderVxlanBfdParams( self ):
      bfdStr = ''
      if self.txInterval and self.rxInterval and self.detectMult:
         bfdStr = ( '  BFD is enabled with transmit interval %d,'
                    ' receive interval %d, multiplier %d' ) % ( self.txInterval,
                                                                self.rxInterval,
                                                                self.detectMult )
      if self.bfdPrefixList:
         bfdStr += ', VTEP prefix list %s' % self.bfdPrefixList
      if bfdStr:
         print( bfdStr )

   def renderVxlanTcpMss( self ):
      if self.tcpMssV4:
         print( ' IPV4 TCP MSS egress ceiling is %s ' % self.tcpMssV4 )
      if self.tcpMssV6:
         print( ' IPV6 TCP MSS egress ceiling is %s ' % self.tcpMssV6 )

   def renderVxlanDecapRouteAll( self ):
      if self.decapRouteAll:
         print( ' Routing all VXLAN decapsulated packets is enabled' )

   def renderLinkTypeSpecific( self ):
      self.renderSrcIpAddr()
      self.renderSrcIpAddr( mlag=True )
      self.renderUdpPort()
      self.renderVxlanEncapsulation()
      self.renderVarpVtepAddr()
      self.renderReplicationMode()
      self.renderDatapathLearningMode()
      self.renderArpReplyRelayMode()
      self.renderArpLocalAddress()
      self.renderPortDot1qVniMapping()
      self.renderVlanToVniMap()
      self.renderVrfToVniMap()
      self.renderDecapVniToVrfMap()
      self.renderReplicationList()
      self.renderMcastGrpDecapAddr()
      self.renderVxlanSharedRouterMacAddr()
      self.renderVtepAddrMask()
      self.renderVxlanHeaderVniLength()
      self.renderVtepToVtepBridging()
      self.renderMcastRouting()
      self.renderVccImportVlan()
      self.renderDecapFilter()
      self.renderVxlanSecurityStatus()
      self.renderVxlanBfdParams()
      self.renderVxlanTcpMss()
      self.renderVccArpProxy()
      self.renderVccNdProxy()
      self.renderVxlanArpProxy()
      self.renderVxlanNdProxy()
      self.renderVxlanDecapRouteAll()

   def render( self ):
      self.renderHeader()
      self.renderHardware()
      self.renderLinkTypeSpecific()
      self.renderUptime()
