#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from operator import attrgetter, itemgetter
import Tac
from ArnetModel import MacAddress, IpGenericAddress, IpGenericPrefix
from Arnet import IpGenPrefix, IpGenAddr, sortIntf
from CliMode.Intf import IntfMode
from CliModel import ( Dict, Enum, List, Model, Str, Float, Int, Bool,
                       DeferredModel, Submodel )
from CliPlugin.BridgingCliModel import _BaseTableEntry
from CliPlugin.IntfModel import ( VirtualInterfaceCountersBase,
                                  InterfaceCountersRateBase )
from CliPlugin.VxlanControllerModel import VxlanVniStatus, VxlanArpTable
from VxlanVniLib import VniFormat, vniToString
from TableOutput import createTable, FormattedCell, Format, TableFormatter
from IntfModels import Interface
import Toggles.VxlanToggleLib
import six
from six.moves import map
from six.moves import range

vxlanTunnelTypes = { 'unicast': 'Unicast tunnels',
                     'flood': 'Flood list tunnels',
                     'ecmp': 'ECMP tunnels',
                     'multicast': 'multicast tunnels' }

largeInt = int if six.PY3 else long # pylint:disable=long-builtin

class TunnelType( Model ):
   tunnelType = Enum( values=vxlanTunnelTypes, help='VXLAN tunnel type' )

class TunnelTypeList( Model ):
   tunnelTypes = List( valueType=TunnelType, help='List of tunnel types' )

class VxlanVtepsModel( Model ):
   class RemoteVtepConfig( Model ):
      learnedVia = Enum( help="Source from which remote VTEP was learned",
                         values=( 'controlPlane', 'dataPlane' ) )
      macAddressLearning = Enum( help='Source for MAC address learning for VTEP',
                                 values=( 'controlPlane', 'datapath' ) )
      evpnDomain = Enum( help="Domain in which VTEP is located",
                         values=( 'remoteDomain', 'localDomain' ),
                         optional=True )

   class VxlanInterface( Model ):
      vteps = List( valueType=IpGenericAddress, help="List of VTEPs" )

   def detailIs( self, detail ):
      self._detail = detail

   def detail( self ):
      return self._detail

   vteps = Dict( keyType=IpGenericAddress, valueType=RemoteVtepConfig,
                 help="A mapping of remote VTEP to details", optional=True )
   interfaces = Dict( keyType=Interface, valueType=VxlanInterface,
                      help="Mapping from VXLAN interface to list of VTEP addresses" )
   vtepTunnelTypes = Dict( keyType=IpGenericAddress, valueType=TunnelTypeList,
                           help="Mapping of VTEP to list of tunnel types" )
   _detail = Bool( help="Display detailed information", optional=True )

   def render( self ):
      fmt = Format( justify='left' )
      fmt.noPadLeftIs( True )

      shouldRenderEvpnDomain = any( x.evpnDomain for x in self.vteps.values() )

      for intfName, intf in six.iteritems( self.interfaces ):
         print( "Remote VTEPS for %s:\n" % intfName )

         if self._detail:
            if shouldRenderEvpnDomain:
               table = createTable( ( 'VTEP', 'Learned Via', 'MAC Address Learning',
                                       'EVPN Domain',
                                       'Tunnel Type(s)' ) )
               table.formatColumns( fmt, fmt, fmt, fmt, fmt )
            else:
               table = createTable( ( 'VTEP', 'Learned Via', 'MAC Address Learning',
                                       'Tunnel Type(s)' ) )
               table.formatColumns( fmt, fmt, fmt, fmt )

         else:
            table = createTable( ( 'VTEP', 'Tunnel Type(s)' ) )
            table.formatColumns( fmt, fmt )

         learnedViaToStr = { 'controlPlane': 'control plane',
                              'dataPlane': 'data plane' }
         macLearningToStr = { 'controlPlane': 'control plane',
                              'datapath': 'datapath' }
         evpnDomainToStr = { 'remoteDomain': 'remote',
                                 'localDomain': 'local' }
         for vtep in intf.vteps:
            vtep = vtep.stringValue
            if vtep in self.vtepTunnelTypes:
               tunnelTypes = ( ', '.join(
                  tunnelType.tunnelType for tunnelType in
                  self.vtepTunnelTypes[ vtep ].tunnelTypes ) )
            else:
               tunnelTypes = 'unicast'
            if self._detail:
               if vtep in self.vteps:
                  if shouldRenderEvpnDomain:
                     table.newRow(
                        vtep,
                        learnedViaToStr[ self.vteps[ vtep ].learnedVia ],
                        macLearningToStr[ self.vteps[ vtep ].macAddressLearning ],
                        evpnDomainToStr[ self.vteps[ vtep ].evpnDomain ],
                        tunnelTypes )
                  else:
                     table.newRow(
                        vtep,
                        learnedViaToStr[ self.vteps[ vtep ].learnedVia ],
                        macLearningToStr[ self.vteps[ vtep ].macAddressLearning ],
                        tunnelTypes )
               else:
                  if shouldRenderEvpnDomain:
                     table.newRow( vtep, 'unknown', 'unknown', 'unknown',
                                       tunnelTypes )
                  else:
                     table.newRow( vtep, 'unknown', 'unknown', tunnelTypes )
            else:
               table.newRow( vtep, tunnelTypes )
         print( table.output() )
         print( "Total number of remote VTEPS: ", len( intf.vteps ) )

class VxlanVtepsSummaryModel( Model ):
   vxlanInterfaces = Dict( keyType=Interface, valueType=int,
                           help="Total VTEPs by interface" )

   def render( self ):
      table = createTable( ( 'VXLAN Interface', 'VTEP Count' ) )
      intfFormat = Format( justify='left' )
      intfFormat.noPadLeftIs( True )
      countFormat = Format( justify='right' )
      table.formatColumns( intfFormat, countFormat )
      total = 0
      for vxlanIntf, vtepCount in six.iteritems( self.vxlanInterfaces ):
         table.newRow( vxlanIntf, vtepCount )
         total = total + vtepCount
      table.newRow()
      table.newRow( 'Total', total )
      print( table.output() )

class VxlanCountersCollectionBaseModel( Model ):

   def tableOutputFormat( self, maxWidth, justify, isHeading=False ):
      outformat = Format( isHeading=isHeading, maxWidth=maxWidth,
            justify=justify, wrap=True, align="bottom" )
      if justify == 'left':
         outformat.noPadLeftIs( True )
      outformat.noTrailingSpaceIs( True )
      return outformat

   def getCounterList( self, counters, attrList, attrsDisplayOrder ):
      counterList = []
      for attrName in attrsDisplayOrder:
         if attrName not in attrList:
            continue
         value = getattr( counters, attrName )
         if value is None:
            # attrName in attrList but value not found can happen if platform
            # ran out of resource and does not have counters in encap or
            # decap direction
            value = '-'
         counterList.append( value )
      return counterList

   # sortedObjKeys supplied here is sorted list of keys from objCollection
   # sorted as the caller sees fit based on the key type
   def renderOutput( self, objCollection, sortedObjKeys, attrsToDisplayIn,
         attrsDisplayOrder, attrsToFormatMapping, featureEnabled ):
      # first column is always vtepObjectAttrFormat
      attrsToDisplay = [ "objectKey" ]
      attrsToDisplay.extend( attrsToDisplayIn )

      headerList = []
      formatList = []

      for attr in attrsDisplayOrder:
         if attr not in attrsToDisplay:
            continue
         attrFormat = attrsToFormatMapping[ attr ]
         headerList.append( attrFormat[ 0 ] )
         formatList.append( attrFormat[ 1 ] )

      table = createTable( headerList )
      table.formatColumns( *formatList )

      # if feature is disabled we display just the header
      if not featureEnabled:
         return table

      # Now display the per vtep counters using the formatStr
      for objKey in sortedObjKeys:
         counters = objCollection[ objKey ]
         displayArgs = [ objKey ]
         displayArgs.extend( self.getCounterList( counters, attrsToDisplayIn,
                                                  attrsDisplayOrder ) )
         table.newRow( *displayArgs )
      return table

   def addRow( self, table, objKey, counters, attrsToDisplayIn, attrsDisplayOrder ):
      displayArgs = [ objKey ]
      displayArgs.extend( self.getCounterList( counters, attrsToDisplayIn,
                                               attrsDisplayOrder ) )
      table.newRow( *displayArgs )

class VxlanCountersBaseModel( Model ):
   def addToAttr( self, attrName, valueToAdd ):
      currentValue = getattr( self, attrName )
      if not currentValue:
         currentValue = 0
      setattr( self, attrName, currentValue + valueToAdd )

class VxlanVtepCountersModel( VxlanCountersCollectionBaseModel ):

   class VxlanVtepCounters( VxlanCountersBaseModel ):
      # All counters defined here are optional as they may or may not be
      # availabe on a given platform.
      decapBytes = Int( help="Bytes decapped from VTEP",
            optional=True )
      decapKnownUcastPkts = Int( help="Layer2 Known Unicast packets "
            "decapped from VTEP", optional=True )
      decapBUMPkts = Int( help="Layer2 BUM (Broadcast, Unknown, "
            "and Multicast) packets decapped from VTEP",
            optional=True )
      # Some platforms( eg XP ) may not support all counters
      # combine both BUM and know unicast decap packets into
      # single field
      decapPkts = Int( help="Packets decapped from VTEP", optional=True )

      decapDropExcptPkts = Int( help="Dropped and Exception packets "
            "decapped from VTEP", optional=True )
      encapBytes = Int( help="Bytes encapped to VTEP",
            optional=True )
      encapPkts = Int( help="Packets encapped to VTEP",
            optional=True )
      encapDropExcptPkts = Int( help="Dropped and Exception packets "
                                "encapped to VTEP", optional=True )
      encapBumPkts = Int( help="BUM Packets encapped to VTEP",
                          optional=True )

   vteps = Dict( keyType=IpGenericAddress, valueType=VxlanVtepCounters,
         help="Mapping between IP address and VTEP counters" )

   unlearntVteps = Submodel( valueType=VxlanVtepCounters,
                          help="Decap Counters for unlearnt VTEPs",
                             optional=True )

   # A platform may want to display only selected attributes in show CLI
   # output. Following contain list of attributes that should be displayed.
   _decapAttrsToDisplay = List( valueType=str,
         help="List of decap attributes to be displayed" )
   _encapAttrsToDisplay = List( valueType=str,
         help="List of encap attributes to be displayed" )

   # Following are two cases for missing encap or decap category of counters in
   # VxlanVtepCounters objects received in the model:
   # 1. Feature is disabled - in this case we want to display empty header in
   #    rendering
   # 2. Feature ran out of resource - in this case we want to display Vtep with
   #    '-' for missing counters.
   # Following knobs help differentiate whether to display just headers
   # or Vteps with '-' for counters.
   _decapFeatureEnabled = Bool( default=False,
                            help='True if Decap feature is enabled' )
   _encapFeatureEnabled = Bool( default=False,
                            help='True if Encap feature is enabled' )

   def decapFeatureEnabledIs( self, enabled ):
      self._decapFeatureEnabled = enabled

   def encapFeatureEnabledIs( self, enabled ):
      self._encapFeatureEnabled = enabled

   # Called by hooks to populate _decapAttrsToDisplay
   def decapAttrsToDisplay( self, attrlist ):
      for attr in attrlist:
         if attr not in self._decapAttrsToDisplay:
            self._decapAttrsToDisplay.append( attr )

   # Called by hooks to populate _encapAttrsToDisplay
   def encapAttrsToDisplay( self, attrlist ):
      for attr in attrlist:
         if attr not in self._encapAttrsToDisplay:
            self._encapAttrsToDisplay.append( attr )

   def addVtep( self, key ):
      key = str( key )
      # check if already exists
      if key in self.vteps:
         vtep = self.vteps[ key ]
      else:
         # create a new entry
         vtep = VxlanVtepCountersModel.VxlanVtepCounters()
         self.vteps[ key ] = vtep
      return vtep

   def addUnlearntVteps( self ):
      self.unlearntVteps = VxlanVtepCountersModel.VxlanVtepCounters()
      return self.unlearntVteps

   def render( self ):
      # attrFormat - Define string to use in the header and table output
      # format for the attributes column.
      # The objectKey goes in the first column as heading key. It is
      # left aligned. Rest of the attributes must be right aligned.
      vtepObjectAttrFormat = ( "VTEP",
         self.tableOutputFormat( maxWidth=15, justify='left', isHeading=True ) )
      decapBytesAttrFormat = ( "Decap Bytes",
         self.tableOutputFormat( maxWidth=18, justify='right' ) )
      decapKnownUcastPktsAttrFormat = ( "Decap Known Unicast Packets",
         self.tableOutputFormat( maxWidth=15, justify='right' ) )
      decapBUMPktsAttrFormat = ( "Decap BUM Packets",
         self.tableOutputFormat( maxWidth=15, justify='right' ) )
      decapPktsAttrFormat = ( "Decap Packets",
         self.tableOutputFormat( maxWidth=13, justify='right' ) )
      decapDropExcptPktsAttrFormat = ( "Decap Drop Or Exception Packets",
         self.tableOutputFormat( maxWidth=13, justify='right' ) )
      encapBytesAttrFormat = ( "Encap Bytes",
         self.tableOutputFormat( maxWidth=18, justify='right' ) )
      encapPktsAttrFormat = ( "Encap Packets",
         self.tableOutputFormat( maxWidth=15, justify='right' ) )
      encapBumPktsAttrFormat = ( "Encap BUM Packets",
         self.tableOutputFormat( maxWidth=15, justify='right' ) )
      encapDropExcptPktsAttrFormat = ( "Encap Drop Or Exception Packets",
         self.tableOutputFormat( maxWidth=13, justify='right' ) )

      # mapping of attributes to attrFormat
      attrsToFormatMapping = {
            "objectKey": vtepObjectAttrFormat,
            "decapBytes": decapBytesAttrFormat,
            "decapKnownUcastPkts": decapKnownUcastPktsAttrFormat,
            "decapBUMPkts": decapBUMPktsAttrFormat,
            "decapPkts": decapPktsAttrFormat,
            "decapDropExcptPkts": decapDropExcptPktsAttrFormat,
            "encapBytes": encapBytesAttrFormat,
            "encapPkts": encapPktsAttrFormat,
            "encapBumPkts": encapBumPktsAttrFormat,
            "encapDropExcptPkts": encapDropExcptPktsAttrFormat,
      }

      # following determines the order of display columns of _decapAttrsToDisplay
      # in the rendered table output
      decapAttrsDisplayOrder = [ "objectKey",
                                 "decapBytes",
                                 "decapKnownUcastPkts",
                                 "decapBUMPkts",
                                 "decapPkts",
                                 "decapDropExcptPkts",
                               ]

      # following determines the order of display columns of _encapAttrsToDisplay
      # in the rendered table output
      encapAttrsDisplayOrder = [ "objectKey",
                                 "encapBytes",
                                 "encapPkts",
                                 "encapBumPkts",
                                 "encapDropExcptPkts",
                               ]

      sortedVteps = sorted( self.vteps.keys(), key=lambda x: IpGenAddr( x ).sortKey )
      if self._decapAttrsToDisplay:
         table = \
            self.renderOutput( self.vteps, sortedVteps, self._decapAttrsToDisplay,
                  decapAttrsDisplayOrder, attrsToFormatMapping,
                  self._decapFeatureEnabled )
         if self._decapFeatureEnabled and self.unlearntVteps:
            self.addRow( table, 'unlearnt', self.unlearntVteps,
                         self._decapAttrsToDisplay, decapAttrsDisplayOrder )
         print( table.output() )
      if self._encapAttrsToDisplay:
         table = \
            self.renderOutput( self.vteps, sortedVteps, self._encapAttrsToDisplay,
                  encapAttrsDisplayOrder, attrsToFormatMapping,
                  self._encapFeatureEnabled )
         print( table.output() )

# These models are used for the VNI-RemoteVTEP counters feature, via the
# `show vxlan counters vtep [ VTEP ] vni [ VNI ]` CLI command
class VniRemoteVtepCounters( VxlanCountersBaseModel ):
   encapPkts = Int( help="Packets encapped to VTEP-VNI pair")

class PerVtepVniCounters( Model ):
   vnis = Dict( keyType=str, valueType=VniRemoteVtepCounters,
               help='Mapping between VNI and its counters per VTEP' )

   def addVni( self, key ):
      key = str( key )
      if key in self.vnis:
         vni = self.vnis[ key ]
      else:
         vni = VniRemoteVtepCounters()
         self.vnis[ key ] = vni
      return vni

class VxlanVtepVniCountersModel( VxlanCountersCollectionBaseModel ):
   vteps = Dict( keyType=IpGenericAddress, valueType=PerVtepVniCounters,
                help='Mapping between remote VTEP IP address and VNI' )

   def addVtep( self, key ):
      key = str( key )
      if key in self.vteps:
         vtep = self.vteps[ key ]
      else:
         vtep = PerVtepVniCounters()
         self.vteps[ key ] = vtep
      return vtep

   # Override base class because we have a two-level hierarchy
   def renderOutput( self, objCollection, sortedObjKeys, attrsToDisplayIn,
         attrsDisplayOrder, attrsToFormatMapping, featureEnabled ):
      # First column is always VTEP, second column is always VNI
      attrsToDisplay = [ 'vtep', 'vni' ]
      attrsToDisplay.extend( attrsToDisplayIn )

      headerList = []
      formatList = []

      for attr in attrsDisplayOrder:
         if attr not in attrsToDisplay:
            continue
         attrFormat = attrsToFormatMapping[ attr ]
         headerList.append( attrFormat[ 0 ] )
         formatList.append( attrFormat[ 1 ] )

      table = createTable( headerList )
      table.formatColumns( *formatList )

      # If feature is disabled we display just the header
      if not featureEnabled:
         return table

      # Display per vtep per vni counters using the formatStr
      # This is where it is different from the base class implementation
      for vtep in sortedObjKeys:
         vtepCtr = objCollection[ vtep ]
         for vni, counter in sorted( vtepCtr.vnis.items() ):
            displayArgs = [ vtep, vni ]
            displayArgs.extend(
               self.getCounterList( counter, attrsToDisplayIn, attrsDisplayOrder ) )
            table.newRow( *displayArgs )

      return table

   def render( self ):
      vtepObjectAttrFormat = ( 'VTEP',
         self.tableOutputFormat( maxWidth=15, justify='left', isHeading=True ) )
      vniObjectAttrFormat = ( 'VNI',
         self.tableOutputFormat( maxWidth=15, justify='left', isHeading=True ) )
      encapPktsAttrFormat = ( 'Encap Packets',
         self.tableOutputFormat( maxWidth=15, justify='right' ) )

      attrsToFormatMapping = {
         'vtep': vtepObjectAttrFormat,
         'vni': vniObjectAttrFormat,
         'encapPkts': encapPktsAttrFormat,
      }

      encapAttrsDisplayOrder = [ 'vtep', 'vni', 'encapPkts' ]

      # As of now this only supports encapPkt counter because it is only used by BFN
      # Add more checks for encap/decap attrs to display and potentially change the
      # hook in BfnTunnel as well if other platforms want to support other counters.

      table = self.renderOutput(
         self.vteps,
         sorted( self.vteps.keys(), key=lambda x: IpGenAddr( x ).sortKey ),
         [ 'encapPkts' ],
         encapAttrsDisplayOrder,
         attrsToFormatMapping,
         True )
      print( table.output() )

# This is an older VNI counters model that is only in use on Strata, meant to be a
# debug command for VNI hardware counters.
# This model is returned for the `show vxlan counters vni` command.
# Newer VXLAN capable devices (especially those with multi-VTI support) should no
# longer use this model for VNI counters, and instead use the newer
# VxlanInterfaceCounters and VxlanInterfaceCountersRate models defined below, which
# will be part of the `show interfaces counters` command.
class VxlanVniCountersModel( VxlanCountersCollectionBaseModel ):

   class VxlanVniCounters( VxlanCountersBaseModel ):
      # All counters defined here are optional as they may or may not be
      # availabe on a given platform.
      decapBytes = Int( help="Bytes decapped for a VNI",
            optional=True )
      decapPkts = Int( help="Packets decapped for a VNI",
            optional=True )
      decapKnownUcastPkts = Int( help="Layer2 Known Unicast packets "
            "decapped for a VNI", optional=True )
      decapBUMPkts = Int( help="Layer2 BUM (Broadcast, Unknown, "
            "and Multicast) packets decapped for a VNI",
            optional=True )
      decapDropExcptPkts = Int( help="Dropped and Exception packets "
            "decapped for a VNI", optional=True )
      encapBytes = Int( help="Bytes encapped for a VNI",
            optional=True )
      encapPkts = Int( help="Packets encapped for a VNI",
            optional=True )
      encapBUMPkts = Int( help="Layer2 BUM (Broadcast, Unknown, "
            "and Multicast) packets for a VNI before replication for encap",
            optional=True )
      encapDropPkts = Int( help="Dropped and Exception packets "
            "encapped for a VNI", optional=True )

   vnis = Dict( keyType=str, valueType=VxlanVniCounters,
         help="Mapping between VNI and its counters" )

   vniInDottedNotation = Bool( default=False,
                            help='True if VNI string is in dotted notation' )

   # A platform may want to display only selected attributes in show CLI
   # output. Following contain list of attributes that should be displayed.
   _decapAttrsToDisplay = List( valueType=str,
         help="List of decap attributes to be displayed" )
   _encapAttrsToDisplay = List( valueType=str,
         help="List of encap attributes to be displayed" )

   # Following are two cases for missing encap or decap category of counters in
   # VxlanVniCounters objects received in the model:
   # 1. Feature is disabled - in this case we want to display empty header in
   #    rendering
   # 2. Feature ran out of resource - in this case we want to display Vni with
   #    '-' for missing counters.
   # Following knobs help differentiate whether to display just headers
   # or Vnis with '-' for counters.
   _decapFeatureEnabled = Bool( default=False,
                            help='True if Decap feature is enabled' )
   _encapFeatureEnabled = Bool( default=False,
                            help='True if Encap feature is enabled' )

   def decapFeatureEnabledIs( self, enabled ):
      self._decapFeatureEnabled = enabled

   def encapFeatureEnabledIs( self, enabled ):
      self._encapFeatureEnabled = enabled

   # Called by hooks to populate _decapAttrsToDisplay
   def decapAttrsToDisplay( self, attrlist ):
      for attr in attrlist:
         if attr not in self._decapAttrsToDisplay:
            self._decapAttrsToDisplay.append( attr )

   # Called by hooks to populate _encapAttrsToDisplay
   def encapAttrsToDisplay( self, attrlist ):
      for attr in attrlist:
         if attr not in self._encapAttrsToDisplay:
            self._encapAttrsToDisplay.append( attr )

   def addVni( self, vni ):
      vniFormat = VniFormat( vni, self.vniInDottedNotation )
      vniStr = str( vniFormat )
      # check if already exists
      if vniStr in self.vnis:
         vniCtr = self.vnis[ vniStr ]
      else:
         # create a new entry
         vniCtr = VxlanVniCountersModel.VxlanVniCounters()
         self.vnis[ vniStr ] = vniCtr
      return vniCtr

   def render( self ):
      # attrFormat - Define string to use in the header and table output
      # format for the attributes column.
      # The objectKey goes in the first column as heading key. It is
      # left aligned. Rest of the attributes must be right aligned.
      vniObjectAttrFormat = ( "VNI",
         self.tableOutputFormat( maxWidth=15, justify='left', isHeading=True ) )
      decapBytesAttrFormat = ( "Decap Bytes",
         self.tableOutputFormat( maxWidth=18, justify='right' ) )
      decapPktsAttrFormat = ( "Decap Packets",
         self.tableOutputFormat( maxWidth=15, justify='right' ) )
      decapKnownUcastPktsAttrFormat = ( "Decap Known Unicast Packets",
         self.tableOutputFormat( maxWidth=15, justify='right' ) )
      decapBUMPktsAttrFormat = ( "Decap BUM Packets",
         self.tableOutputFormat( maxWidth=15, justify='right' ) )
      decapDropExcptPktsAttrFormat = ( "Decap Drop Or Exception Packets",
         self.tableOutputFormat( maxWidth=13, justify='right' ) )
      encapBytesAttrFormat = ( "Encap Bytes",
         self.tableOutputFormat( maxWidth=18, justify='right' ) )
      encapPktsAttrFormat = ( "Encap Packets",
         self.tableOutputFormat( maxWidth=15, justify='right' ) )
      encapBUMPktsAttrFormat = ( "Encap BUM Packets",
         self.tableOutputFormat( maxWidth=15, justify='right' ) )
      encapDropPktsAttrFormat = ( "Encap Drop Packets",
         self.tableOutputFormat( maxWidth=13, justify='right' ) )

      # mapping of attributes to attrFormat
      attrsToFormatMapping = {
            "objectKey": vniObjectAttrFormat,
            "decapBytes": decapBytesAttrFormat,
            "decapPkts": decapPktsAttrFormat,
            "decapKnownUcastPkts": decapKnownUcastPktsAttrFormat,
            "decapBUMPkts": decapBUMPktsAttrFormat,
            "decapDropExcptPkts": decapDropExcptPktsAttrFormat,
            "encapBytes": encapBytesAttrFormat,
            "encapPkts": encapPktsAttrFormat,
            "encapBUMPkts": encapBUMPktsAttrFormat,
            "encapDropPkts": encapDropPktsAttrFormat,
      }

      # following determines the order of display columns of _decapAttrsToDisplay
      # in the rendered table output
      decapAttrsDisplayOrder = [ "objectKey",
                                 "decapBytes",
                                 "decapPkts",
                                 "decapKnownUcastPkts",
                                 "decapBUMPkts",
                                 "decapDropExcptPkts",
                               ]

      # following determines the order of display columns of _encapAttrsToDisplay
      # in the rendered table output
      encapAttrsDisplayOrder = [ "objectKey",
                                 "encapBytes",
                                 "encapPkts",
                                 "encapBUMPkts",
                                 "encapDropPkts",
                               ]

      vniInts = [ VniFormat( x, self.vniInDottedNotation ).toNum()
                  for x in self.vnis ]
      sortedVnis = [ vniToString( x, self.vniInDottedNotation ) for x in
                     sorted( vniInts ) ]

      if self._decapAttrsToDisplay:
         table = \
            self.renderOutput( self.vnis, sortedVnis, self._decapAttrsToDisplay,
                  decapAttrsDisplayOrder, attrsToFormatMapping,
                  self._decapFeatureEnabled )
         print( table.output() )
      if self._encapAttrsToDisplay:
         table = \
            self.renderOutput( self.vnis, sortedVnis, self._encapAttrsToDisplay,
                  encapAttrsDisplayOrder, attrsToFormatMapping,
                  self._encapFeatureEnabled )
         print( table.output() )

# This is the new VNI counters model that will be part of `show interfaces counters`.
# This model will be able to handle per-VNI counters for multi-VTI topologies.
# New implementations for VNI counters should start using this model instead of the
# above VxlanVniCountersModel.
class VxlanInterfaceVniCounters( Model ):
   inOctets = Int( help="Input octets" )
   inPackets = Int( help="Input packets" )

   outOctets = Int( help="Output octets" )
   outPackets = Int( help="Output packets" )

   lastUpdateTimestamp = Float( help="Time of last update", optional=True )

class VxlanInterfaceCounters( VirtualInterfaceCountersBase ):
   vnis = Dict( keyType=str, valueType=VxlanInterfaceVniCounters,
                help="Mapping between VNI and its counters" )

   def vniCounterIs( self, vni, counters ):
      self.vnis[ vni ] = counters

   def printRow( self, vti, vni, octets, packets ):
      # Currently the model does not handle BUM counters, it can be extended later on
      def valueOrNa( value ):
         return value if value is not None else 'n/a'

      counterDisplayFormat = "%-11s %-14s %18s %15s"
      print( counterDisplayFormat % ( vti, vni, valueOrNa( octets ),
                                      valueOrNa( packets ) ) )

   def renderHeader( self, direction='in' ):
      if self._vxlanVniCounters:
         heading = "Interface"
      else:
         assert False, "Unsupported counter type for VxlanInterfaceCounters model"

      if direction == 'in':
         if not self._ingressCounters:
            return
         self.printRow( heading, "VNI", "InOctets", "InPkts" )
      elif direction == 'out':
         if not self._egressCounters:
            return
         self.printRow( heading, "VNI", "OutOctets", "OutPkts" )
      else:
         assert False, "Unhandled direction"

   def renderIncoming( self ):
      if not self._ingressCounters:
         return
      for vni in sorted( self.vnis ):
         self.printRow( IntfMode.getShortname( self._name ), vni,
                        self.vnis[ vni ].inOctets, self.vnis[ vni ].inPackets )

   def renderOutgoing( self ):
      if not self._egressCounters:
         return
      for vni in sorted( self.vnis ):
         self.printRow( IntfMode.getShortname( self._name ), vni,
                        self.vnis[ vni ].outOctets, self.vnis[ vni ].outPackets )

class VxlanInterfaceVniCountersRate( Model ):
   description = Str( help="Port description" )
   interval = Int( help="Interval in seconds" )

   inBpsRate = Float( help="Input bps rate" )
   inPktsRate = Float( help="Input packets rate percentage", optional=True )
   inPpsRate = Float( help="Input pps rate" )

   outBpsRate = Float( help="Output bps rate" )
   outPktsRate = Float( help="Output packets rate percentage", optional=True )
   outPpsRate = Float( help="Output pps rate" )

   lastUpdateTimestamp = Float( help="Time of last update", optional=True )

class VxlanInterfaceCountersRate( InterfaceCountersRateBase ):
   vnis = Dict( keyType=str, valueType=VxlanInterfaceVniCountersRate,
                help="Mapping between VNI and its counters rate" )

   def vniCounterRateIs( self, vni, countersRate ):
      self.vnis[ vni ] = countersRate

   def renderHeader( self ):
      print( "%-11s %-15s %-11.11s %5s  %8s %6s %8s  %8s %6s %8s"
             % ( 'Interface', 'VNI', 'Name', 'Intvl', 'In Mbps', '%', 'In Kpps',
                 'Out Mbps', '%', 'OutKpps' ) )

   def renderRates( self ):
      formatStr = "%-11s %-15s %-11.11s %2d:%02d  %8.1f %6s %8d  %8.1f %6s %8d"
      for vni in sorted( self.vnis ):
         vniCountersRate = self.vnis[ vni ]
         minutes = vniCountersRate.interval // 60
         seconds = vniCountersRate.interval % 60
         inMbpsRate = vniCountersRate.inBpsRate / 1000 / 1000
         outMbpsRate = vniCountersRate.outBpsRate / 1000 / 1000
         inRatePctStr = ( "%.1f%%" % vniCountersRate.inPktsRate
            if vniCountersRate.inPktsRate is not None else '-' )
         outRatePctStr = ( "%.1f%%" % vniCountersRate.outPktsRate
            if vniCountersRate.outPktsRate is not None else '-' )
         inKppsRate = vniCountersRate.inPpsRate // 1000
         outKppsRate = vniCountersRate.outPpsRate // 1000
         print( formatStr % ( IntfMode.getShortname( self._name ), vni,
                              vniCountersRate.description, minutes, seconds,
                              inMbpsRate, inRatePctStr, inKppsRate,
                              outMbpsRate, outRatePctStr, outKppsRate ) )

class VxlanMacAddresses( DeferredModel ):
   __revision__ = 2

   class VxlanMacAddrListing( _BaseTableEntry ):
      addrType = Enum( values=( "dynamic", "static", "received", "evpn", "unknown" ),
                       help="Type of the MAC address" )
      interface = Str( help="VXLAN interface for this address" )
      vteps = List( valueType=IpGenericAddress, help="List of VTEPs "
                    "corresponding to this address" )
      moves = Int( help="Number of times this address has moved" )
      lastMove = Float( help="Time of the last move for this address. "
                        "A value of 0 indicates the host has not moved" )

   addresses = List( valueType=VxlanMacAddrListing,
                     help="List of VXLAN MAC addresses" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         for addr in dictRepr[ 'addresses' ]:
            addr[ 'vtep' ] = addr[ 'vteps' ][ 0 ] if addr[ 'vteps' ] else []
            del addr[ 'vteps' ]
      return dictRepr

class VxlanAddressTableCount( Model ):
   vtepCounts = Dict( keyType=IpGenericAddress, valueType=int,
                      help="Count of MAC addresses per VTEP" )

   def render( self ):
      fmt = '%-15s  %19d'
      print( 'VTEP             Total MAC Addresses' )
      print( '---------------  -------------------' )

      for vtep, count in six.iteritems( self.vtepCounts ):
         print( fmt % ( vtep, count ) )

class VxlanFloodVtepListModel( DeferredModel ):
   class VxlanVtepListModel( Model ):
      vteps = List( valueType=IpGenericAddress, help='VTEP IP address' )

   floodMap = Dict( keyType=str, valueType=VxlanVtepListModel,
                    help='Per VLAN flood list' )

class VxlanLearnRestrictModel( Model ):
   class VxlanPrefixListModel( Model ):
      learnFrom = Enum(
         values=( "learnFromAny", "learnFromFloodList", "learnFromList",
                  "learnFromDefault" ),
         help="VTEP prefixes from which we learn" )
      prefixList = List( valueType=IpGenericPrefix, help='VTEP IP address (prefix)' )

   learnMap = Dict( keyType=int, valueType=VxlanPrefixListModel,
                    help='Per vlan learn list' )

   def render( self ):
      print( '          Vxlan Learn-Restrict Prefix Table' )
      print( '--------------------------------------------------------' )
      print()
      print( 'VLAN  Mode   IP Address Prefix' )
      print( '----  -----  -------------------------------------------' )
      for vlan in sorted( self.learnMap ):
         model_to_short = { 'learnFromAny': 'Any',
                        'learnFromFloodList': 'Flood',
                        'learnFromList': 'List',
                        'learnFromDefault': '' }
         short = model_to_short[ self.learnMap[ vlan ].learnFrom ]
         hdr = '%-04d  %-5s  ' % ( vlan, short )
         hlen = len( hdr )
         ips = []
         if not self.learnMap[ vlan ].prefixList:
            print( '%s' % hdr )
         else:
            for ip in sorted( self.learnMap[ vlan ].prefixList,
                              key=lambda x: x.sortKey ):
               ips.append( ip )
               if len( ips ) == 3:
                  print( '%s%-18s  %-18s  %-18s' % ( hdr, ips[ 0 ], ips[ 1 ],
                                                    ips[ 2 ] ) )
                  ips = []
                  hdr = ' ' * hlen
            if ips:
               ips = ips + [ '' ] * ( 3 - len( ips ) )
               print( '%s%-18s  %-18s  %-18s' % ( hdr,
                     ips[ 0 ], ips[ 1 ], ips[ 2 ] ) )

# Similar to above but with the counters from learnStatus
class VxlanLearnCountersModel( Model ):
   class VxlanPrefixStatusModel( Model ):
      learnFrom = Enum(
         values=( "learnFromAny", "learnFromFloodList", "learnFromList",
                  "learnFromDefault" ),
         help="VTEP prefixes from which we learn" )
      prefixList = Dict( keyType=IpGenericPrefix, valueType=int,
                       help='Per vlan learn list with number of VTEPs accepted' )
      numMatchAny = Int( help="Number of VTEPs that matched with learnFromAny" )
      numMatchFloodList = Int(
         help="Number of VTEPs that matched learnFromFloodList" )
      numMatchList = Int( help="Number of VTEPs that matched learnFromList" )
      numRejectFloodList = Int(
         help="Number of VTEPs dropped due to no match for learnFromFloodList" )
      numRejectList = Int(
         help="Number of VTEPs dropped due to no match for learnFromList" )

   learnCountersMap = Dict( keyType=int, valueType=VxlanPrefixStatusModel,
                            help='Per VLAN learn list' )
   counters = Enum( values=( "prefix", "all", "brief" ),
                    help='Select set of per-vlan counters' )

   def renderPrefix( self ):
      print( '          Vxlan Learn-Restrict Status Table' )
      print( '-----------------------------------------------------------------' )
      print()
      print( 'VLAN  Mode   IP Address Prefix(numAccept)' )
      print( '----  -----  ----------------------------------------------------' )
      for vlan in sorted( self.learnCountersMap ):
         lm = self.learnCountersMap[ vlan ]
         model_to_short = { 'learnFromAny': 'Any',
                        'learnFromFloodList': 'Flood',
                        'learnFromList': 'List',
                        'learnFromDefault': '' }
         short = model_to_short[ lm.learnFrom ]
         hdr = '%-04d  %-5s  ' % ( vlan, short )
         hlen = len( hdr )
         ips = []
         if not lm.prefixList:
            print( '%s' % hdr )
         else:
            for ip in sorted( lm.prefixList,
                              key=lambda x: IpGenPrefix( x ).sortKey ):
               ips.append( '%s(%d)' % ( ip, lm.prefixList[ ip ] ) )
               if len( ips ) == 2:
                  print( '%s%-23s %-23s' % ( hdr, ips[ 0 ], ips[ 1 ] ) )
                  ips = []
                  hdr = ' ' * hlen
            if ips:
               ips = ips + [ '' ] * ( 2 - len( ips ) )
               print( '%s%-23s %-23s' % ( hdr, ips[ 0 ], ips[ 1 ] ) )

   def renderBrief( self ):
      print( '          Vxlan Learn-Restrict Status Table' )
      print( '-----------------------------------------------------------------' )
      print()
      print( 'VLAN  Mode    numAccept  numReject' )
      print( '----  -----  ----------------------------------------------------' )
      for vlan in sorted( self.learnCountersMap ):
         lm = self.learnCountersMap[ vlan ]
         model_to_short = { 'learnFromAny': 'Any',
                        'learnFromFloodList': 'Flood',
                        'learnFromList': 'List',
                        'learnFromDefault': '' }
         short = model_to_short[ lm.learnFrom ]
         hdr = '%-04d  %-5s  ' % ( vlan, short )
         numAccept = self.learnCountersMap[ vlan ].numMatchAny + \
             self.learnCountersMap[ vlan ].numMatchList + \
             self.learnCountersMap[ vlan ].numMatchFloodList
         numReject = self.learnCountersMap[ vlan ].numRejectList + \
             self.learnCountersMap[ vlan ].numRejectFloodList

         print( '%s%10d %10d' % ( hdr, numAccept, numReject ) )

   def renderAll( self ):
      print( '          Vxlan Learn-Restrict Status Table' )
      print( '--------------------------------------------------------------------' )
      print()
      print( 'VLAN  Mode     MatchAny  MatchList MatchFlood RejectList RejectFlood' )
      print( '----  -----  -------------------------------------------------------' )
      for vlan in sorted( self.learnCountersMap ):
         lm = self.learnCountersMap[ vlan ]
         model_to_short = { 'learnFromAny': 'Any',
                        'learnFromFloodList': 'Flood',
                        'learnFromList': 'List',
                        'learnFromDefault': '' }
         short = model_to_short[ lm.learnFrom ]
         hdr = '%-04d  %-5s  ' % ( vlan, short )
         print( '%s%10d %10d %10d %10d %10d' %
             ( hdr,
               self.learnCountersMap[ vlan ].numMatchAny,
               self.learnCountersMap[ vlan ].numMatchList,
               self.learnCountersMap[ vlan ].numMatchFloodList,
               self.learnCountersMap[ vlan ].numRejectList,
               self.learnCountersMap[ vlan ].numRejectFloodList ) )

   def render( self ):
      if self.counters == 'prefix':
         self.renderPrefix()
      elif self.counters == 'brief':
         self.renderBrief()
      elif self.counters == 'all':
         self.renderAll()

class VxlanVniStatusDirModel( Model ):
   __revision__ = 2
   tableType = Str( "Advertised or Received" )
   vniStatus = List( valueType=VxlanVniStatus,
                       help="Vxlan tables" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         for vs in dictRepr[ 'vniStatus' ]:
            for mvp in vs[ 'unicastHostTable' ]:
               mvp[ 'timeLearned' ] = 0.0
      return dictRepr

   def render( self ):
      h1 = '------------------------------------------------------------------------'
      h2 = '---- ----------------- --------------- -----'
      h3 = '---- ----------------- -------------------------------------------------'
      print( '          %s Mac Address Table' % self.tableType )
      print( h1 )
      print()
      print( 'VLAN Mac Address       VTEP            Moves' )
      print( h2 )
      cnt = 0
      fmt = '%4d %-17s %-15s %5d'
      vnis = sorted( self.vniStatus, key=attrgetter( 'vlan' ) )
      byMacAddr = attrgetter( 'macAddr' )
      for vs in vnis:
         for mvp in sorted( vs.unicastHostTable, key=byMacAddr ):
            cnt += 1
            print( fmt % ( vs.vlan, mvp.macAddr, mvp.vtepIp, mvp.moveCount ) )
      print( 'Total Mac Addresses for this criterion: %d' % cnt )

      print()
      print( '          %s Flood Table' % self.tableType )
      print( h1 )
      print()
      print( 'VLAN Mac Address       VTEP(s)' )
      print( h3 )
      cnt = 0
      for vs in vnis:
         for mvlp in sorted( vs.bumVtepListTable, key=byMacAddr ):
            print( '%4d %-17s' % ( vs.vlan, mvlp.macAddr ), end=' ' )
            # print vtep IPs 3 to a line
            ip_count = 0
            for vtep in sorted( mvlp.vtepIpList ):
               if ip_count and ip_count % 3 == 0:
                  print( '\n' + ' ' * 22, end=' ' )
               print( ' %-15s' % vtep, end=' ' )
               ip_count += 1
            cnt += 1
            print()
      print( 'Total Mac Addresses for this criterion: %d' % cnt )

class VxlanArpTableModel( Model ):
   tableType = Enum( values=( "advertised", "received" ),
                     help="Type of ARP or ND entries contained in this table" )
   vlanToArpTable = Dict( keyType=int, valueType=VxlanArpTable,
                          help="A mapping from VLAN ID to ARP or ND table" )
   protocol = Enum( values=( "arp", "nd" ),
                    help="Type of address protocol" )

   def render( self ):
      def fmt( minWidth, maxWidth ):
         f = Format( justify='left', minWidth=minWidth, maxWidth=maxWidth )
         f.noPadLeftIs( True )
         return f
      protocolToName = { 'arp': 'ARP',
                         'nd': 'Neighbor Discovery' }
      topTable = createTable( ( '%s %s Table' %
                                ( self.tableType.capitalize(),
                                  protocolToName[ self.protocol ] ), ) )
      topTable.formatColumns( Format( justify='center', minWidth=77, maxWidth=77 ) )

      bodyTable = createTable( ( 'VLAN', 'IP Address', 'MAC Address', 'Changes' ) )
      bodyTable.formatColumns( fmt( 4, 4 ), fmt( 10, 39 ), fmt( 17, 17 ),
                               fmt( 7, 7 ) )
      count = 0

      def sortFn( col ):
         return sorted( six.iteritems( col ), key=lambda i: i[ 0 ] )

      for ( vlan, arpTable ) in sortFn( self.vlanToArpTable ):
         for ( ipAddress, arpEntry ) in sortFn( arpTable.addresses ):
            count += 1

            bodyTable.newRow( *( vlan, ipAddress, arpEntry.macAddress,
                                 arpEntry.changes ) )

      print( topTable.output() )
      print( bodyTable.output() )
      print( 'Total IP addresses for this criterion: %d' % count )

class VxlanIntfDot1qModel( Model ):
   dot1q = Int( help="Dot1q tagged VLAN" )
   error = Str( help="Error with VLAN translation",
                optional=True )
   dot1qTunnel = Bool( help="Dot1q-tunneled VLAN",
                       optional=True )

class VxlanVniVlanMappingModel( Model ):
   vlan = Int( help="VLAN ID" )
   dynamicVlan = Bool( help="VLAN is dynamically configured",
                       optional=True )
   source = Str( help="Source of VNI to VLAN mapping, ( VCS, DYNVTEP, STATIC )" )
   interfaces = Dict( keyType=Interface, valueType=VxlanIntfDot1qModel,
                      help="Ports associated with a VNI" )

class VxlanVniVrfMappingModel( Model ):
   vrfName = Str( help="VRF which is associated with the VNI", optional=True )
   vlan = Int( help="Dynamic VLAN ID" )
   source = Str( help="Source of VNI to dynamic VLAN mapping" )

class VxlanVniVlanCollection( Model ):
   vniBindings = Dict( keyType=largeInt, valueType=VxlanVniVlanMappingModel,
                       help="VNI to VLAN mapping" )
   vniBindingsToVrf = Dict( keyType=largeInt, valueType=VxlanVniVrfMappingModel,
                            help="VNI to VRF mapping" )

class VxlanVniSummaryModel( Model ):
   vxlanVniSources = Dict( keyType=str, valueType=int,
                           help="A mapping of VXLAN source to source count" )

   def render( self ):
      if not self.vxlanVniSources:
         return # nothing to output

      print( "VNI to VLAN mapping counts" )
      table = createTable( ( 'Source', 'Mapping Count' ) )
      sourceColFormat = Format( justify='left' )
      sourceColFormat.noPadLeftIs( True )
      countColFormat = Format( justify='right' )
      table.formatColumns( sourceColFormat, countColFormat )
      totalCount = 0
      for source, count in sorted( six.iteritems( self.vxlanVniSources ) ):
         table.newRow( source, count )
         totalCount = totalCount + count
      table.newRow()
      table.newRow( 'Total mappings', totalCount )
      print( table.output() )

class VxlanVniModel( Model ):
   vxlanIntfs = Dict( keyType=Interface, valueType=VxlanVniVlanCollection,
                      help="A mapping of VXLAN interface to VNI collection" )
   _vniInDottedNotation = Bool( default=False,
                                help='True if VNI is displayed in dotted notation' )

   def render( self ):
      def fmt():
         f = Format( justify='left' )
         f.noPadLeftIs( True )
         return f

      def fmtPortTag( port ):
         if port.error:
            return port.error
         if port.dot1qTunnel:
            return 'tunneled'
         tag = port.dot1q
         if tag == 0:
            return 'untagged'
         elif tag:
            return str( tag )
         else:
            return ''

      def fmtVlan( mappingModel ):
         vlanStr = str( mappingModel.vlan )
         if mappingModel.dynamicVlan:
            vlanStr += '*'
         return vlanStr

      def printVniBindingsToVlan( vxlanIntf, vniBindingsToVlan ):
         print( "VNI to VLAN Mapping for %s" % vxlanIntf )
         table = createTable( ( 'VNI', 'VLAN', 'Source', 'Interface',
                                '802.1Q Tag' ) )
         table.formatColumns( *[ fmt() for _ in range( 0, 5 ) ] )
         dynamicVlanRendered = False
         sortedKeys = sorted( vniBindingsToVlan )
         for vniKey in sortedKeys:
            entry = vniBindingsToVlan[ vniKey ]
            if entry.dynamicVlan:
               dynamicVlanRendered = True
            vniStr = VniFormat( vniKey, self._vniInDottedNotation )
            first = True
            portKeys = sorted( entry.interfaces )
            for pKey in portKeys:
               port = entry.interfaces[ pKey ]
               if first:
                  table.newRow( vniStr, fmtVlan( entry ), entry.source,
                                pKey, fmtPortTag( port ) )
                  first = False
               else:
                  table.newRow( '', '', '', pKey, fmtPortTag( port ) )

            # handle case where interfaces is empty
            if not entry.interfaces:
               table.newRow( vniStr, fmtVlan( entry ), entry.source, '', '' )
         print( table.output() )
         if dynamicVlanRendered:
            print( "Note: * indicates a Dynamic VLAN" )

      def printVniBindingsToVrf( vxlanIntf, vniBindingsToVrf ):
         print( "VNI to dynamic VLAN Mapping for %s" % vxlanIntf )
         table = createTable( ( 'VNI', 'VLAN', 'VRF', 'Source' ) )
         table.formatColumns( *[ fmt() for _ in range( 0, 5 ) ] )
         sortedKeys = sorted( vniBindingsToVrf )
         for vniKey in sortedKeys:
            entry = vniBindingsToVrf[ vniKey ]
            vniStr = VniFormat( vniKey, self._vniInDottedNotation )
            vlanStr = str( entry.vlan )
            vrfName = entry.vrfName if entry.vrfName is not None else ''
            table.newRow( vniStr, vlanStr, vrfName, entry.source )
         print( table.output() )

      if not self.vxlanIntfs:
         # nothing to output
         return

      for vxlanIntf, data in six.iteritems( self.vxlanIntfs ):
         printVniBindingsToVlan( vxlanIntf, data.vniBindings )
         printVniBindingsToVrf( vxlanIntf, data.vniBindingsToVrf )

class VxlanQosModel( Model ):
   __revision__ = 2
   ecnPropagation = Bool( default=False,
                          help='VXLAN ECN propagation is enabled',
                          optional=True )
   decapDscpEcnPropagationDisabled = Bool( default=False,
                          help='VXLAN DSCP and ECN propagation is disabled for'
                               ' VXLAN decapsulated packets',
                          optional=True )
   dscpEcnRewriteBridgedEnabled = Bool( default=False,
                          help='VXLAN DSCP and ECN rewrite is enabled for'
                               ' VXLAN bridged packets',
                          optional=True )
   if Toggles.VxlanToggleLib.toggleVxlanOuterDscpFromInnerDscpEnabled():
      outerDscpFromInnerDscp = Bool( default=False,
                              help='Outer DSCP is derived from Inner DSCP' )
      trafficClassFromOuterDscp = Bool( default=False,
                           help='Traffic Class is derived from Outer DSCP' )
   routingDscpRewriteSupported = Bool( default=True,
                           help='DSCP rewrite is supported on VXLAN routed packets' )

   def render( self ):
      if self.ecnPropagation != None:
         status = "Enabled" if self.ecnPropagation else "Disabled"
         print( "VXLAN ECN Propagation is %s." % status )

      if self.decapDscpEcnPropagationDisabled is not None or \
         self.dscpEcnRewriteBridgedEnabled is not None:
         # This section is only applicable to Jericho2 family
         if self.decapDscpEcnPropagationDisabled:
            print( "VXLAN DSCP and ECN propagation on decapsulation is disabled." )
         else:
            print( "VXLAN DSCP and ECN propagation is enabled." )
         status = "enabled" if self.dscpEcnRewriteBridgedEnabled else "disabled"
         print( "DSCP and ECN rewrite is %s for VXLAN bridged packets." %
                status )
         dscpSource = "ingress packet DSCP" if self.dscpEcnRewriteBridgedEnabled \
                      else "Traffic Class"
         print( "Outer header DSCP for bridged packets is derived "
                 "from %s." % dscpSource )
         print( "Outer header DSCP for routed packets is derived "
                 "from Traffic Class if DSCP rewrite is enabled "
                 "and from the ingress packet DSCP if DSCP rewrite is disabled." )
         print( "Traffic Class is derived from outer DSCP for "
                "decapsulated packets." )
         return

      if not self.routingDscpRewriteSupported:
         # This section is only applicable to Jericho family
         print( "Outer header DSCP for bridged packets is derived "
                "from Traffic Class." )
         print( "Outer header DSCP for routed packets is derived "
                "from the ingress packet DSCP." )
         print( "Traffic Class is derived from outer DSCP for "
                "decapsulated packets." )
         return

      if Toggles.VxlanToggleLib.toggleVxlanOuterDscpFromInnerDscpEnabled():
         outerDscpStatus = "Inner DSCP" if self.outerDscpFromInnerDscp else \
                           "Traffic Class"
         trafficClassStatus = "Outer DSCP" if self.trafficClassFromOuterDscp else \
                              "Inner DSCP"
         print( "Outer header DSCP is derived from %s." % outerDscpStatus )
         print( "Traffic Class is derived from %s." % trafficClassStatus )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         if 'ecnPropagation' not in dictRepr:
            dictRepr[ 'ecnPropagation' ] = False
      return dictRepr

class VxlanSwCounters( Model ):
   __revision__ = 3

   rxForEncapBytes = Int( help="Number of bytes received for encapsulation" )
   rxForEncapPkts = Int( help="Number packets received for encapsulation" )
   rxForEncapBytesSockHiPrio = Int( help="Number of bytes received from high "
                                    "priority input queue for encapsulation" )
   rxForEncapPktsSockHiPrio = Int( help="Number of packets received from high "
                                   "priority input queue for encapsulation" )
   rxForEncapBytesSockLoPrio = Int( help="Number of bytes received from low "
                                    "priority input queue for encapsulation" )
   rxForEncapPktsSockLoPrio = Int( help="Number of packets received from low "
                                   "priority input queue for encapsulation" )
   encapBytes = Int( help="Number of bytes sent to known remote vteps" )
   encapPkts = Int( help="Number of packets sent to known remote vteps" )

   encapTxTotalBytes = Int( help="Number of bytes sent after encapsulation" )
   encapTxTotalPkts = Int( help="Number of packets sent after encapsulation" )
   encapDot1qTunnel = Int( help="Number of packets dot1Q received for "
                           "encapsulation" )
   arpPkts = Int( help="Number of ARP packets sent to ARP resolver" )
   encapReadErr = Int( help="Number of read error packets received for "
                       "encapsulation" )
   encapDiscardRunt = Int( help="Number of discard packets failing size check "
                           "for encapsulation" )
   encapDiscardVlanRange = Int( help="Number of discard packets failing VLAN "
                                "range check for encapsulation" )
   encapDiscardVlanMap = Int( help="Number of discard packets missing VLAN-2-VNI "
                              "mapping for encapsulation" )
   encapDiscardMlagPeerLink = Int( help="Number of discard packets coming in "
                                   "on a MLAG peer link for encapsulation" )
   encapNoRemoteVtep = Int( help="Number of discard packets destinated to empty "
                            "flood list for encapsulation" )
   encapNoRemoteVtepBfd = Int( help="Number of discard packets destinated to empty "
                               "BFD flood list for encapsulation" )
   encapSendErr = Int( help="Number of send error packets for encapsulation" )
   encapSendDiscardHigh = Int( help="Number of high priority send error packets for "
                               "encapsulation" )
   encapSendDiscardLow = Int( help="Number of low priority send error packets for "
                               "encapsulation" )
   encapTimeout = Int( help="Number of times the sender activity loop yield " )
   encapSockDrop = Int( help="Number of pkts dropped by DMA driver socket" )
   arpSendErr = Int( help="Number of send error for ARP packets" )
   encapTxAnyRemoteVtep = Int( help="Number of BUM packets sent to first vtep "
                               "on flood list" )
   arpRxqRcvd = Int( help="Number of ARP request packets received" )
   arpRxqRcvdFromVarpMac = Int( help="Number of ARP request packets received from a "
         "virtual mac" )
   arpReqNoHER = Int( help="Number of ARP request packets that were not headend "
                      "replicated " )
   ipToMacMlagSent = Int( help="Number of Ip to MAC bindings sent to secondary"
                          " Mlag peer" )
   ipToMacMlagRecd = Int( help="Number of Ip to MAC bindings received from primary"
                          " Mlag peer" )
   arpSupressCacheMiss = Int( help="Number of Arp cache misses during Arp "
                           "suppression" )
   arpSupressCache1Miss = Int( help="Number of Arp initial cache misses during Arp "
                           "suppression" )
   arpSupressDeny = Int( help="Number of Arp supression packets denied due to ACL" )
   arpSuppressSkipLocalMac = Int( help="Number of ARP cache hits skipped during"
                                       " suppression due to target being local mac" )
   arpSuppressSkipLocalIp = Int( help="Number of ARP cache hits skipped during"
                                " suppression due to target IP address being local" )
   arpSuppressSkipUnknownMac = Int( help="Number of ARP cache hits skipped during"
                                " suppression due to sender address being unknown"
                                " unicast" )
   arpSupressTx = Int( help="Number of ARP cache hits during Arp suppression" )
   arpSupressTxErr = Int( help="Number of ARP reply errors during Arp suppression" )
   arpSupressTxBcast = Int( help="Number of ARP reply sent to bcast mac during Arp"
                            " suppression" )
   arpReplyRelayNoHer = Int( help="Number of ARP reply packets that were not headend"
                      " replicated due to ARP relay" )
   arpSuppressSkipDynamicArp = Int( help="Number of dynamic ARP cache hits skipped"
                                    " during suppression due to sender virtual MAC" )
   arpSuppressSkipProbe = Int( help="Number of ARP probes skipped ARP suppression" )
   garpPeerSyncMsgCount = Int( help='Number of GRAT ARP sync messages received' )
   garpPeerSyncMsgErrCount = Int( help='Number of GRAT ARP sync messages received '
                                        'in error' )
   garpPeerSyncMsgTxCount = Int( help='Number of GRAT ARP sync messges sent' )
   garpTgtSviSuppress = Int( help='Number of GRAT ARP pkts from SVI suppressed' )
   installedGarpCount = Int( help='Number of GRAT ARP bindings added to ARP' )
   installedGarpErrCount = Int( help='Number of GRAT ARP bindings added to ARP in '
                                     'error' )
   rarpRxCount = Int( help='Number of RARP pkts' )
   rarpRxInvalidCount = Int( help='Number of RARP pkts that were invalid' )
   nsReqNoHER = Int( help="ND NS pkts skipped HER since target Ip matched SVI Ip" )
   nsSuppressCacheMiss = Int( help="Number of Neighbor cache misses during ND "
                                   "NS proxy" )
   nsSuppressCache1Miss = Int( help="Number of Neighbor initial cache misses during"
                                    " ND NS proxy" )
   nsSuppressDeny = Int( help="Number of ND NS proxy packets denied due to ACL" )
   nsSuppressSkipDynamicArp = Int( help="Number of dynamic Neighbor hits skipped"
                                        " during ND proxy due to source link"
                                        " address being router MAC" )
   nsSuppressSkipLocalMac = Int( help="Number of Neighbor hits skipped during "
                                      "ND NS proxy due to target link layer being "
                                      "local" )
   nsSuppressSkipLocalIp = Int( help="Number of Neighbor hits skipped during"
                                     " ND NS proxy due to target address being"
                                     " local" )
   nsSuppressSkipUnknownMac = Int( help="Number of Neighbor hits skipped during"
                                     " ND NS proxy due to sender MAC address being"
                                     " unknown unicast" )
   nsSuppressTxErr = Int( help="Number of Neighbor Advertisement errors during ND "
                               "NS proxy" )
   nsSuppressSkipInvalidPkt = Int( help="Number of Neighbor Solicitation pkts found "
                                    "invalid" )
   nsSuppressNonUnicastSource = Int( help="Number of Neighbor Solicitation pkts from"
                                          " unspecified source" )
   nsSuppressDad = Int( help="Number of Neighbor Solicitation DAD pkts suppressed" )
   nsSuppressDadVarpIp = Int( help="Number of Neighbor Solicitation DAD pkts "
                                   "from kernel to virtual IP suppressed" )
   nsSuppressVirtualLinkLocal = Int( help="Number of Neighbor Solicitation pkts "
                                   "for virtual Link Local IP suppressed" )
   nsNASent = Int( help="Number of Neighbor Advertisement proxy pkts sent" )
   nsSuppressTx = Int( help="Number of Neighbor Solicitations suppressed" )
   nsRx = Int( help="Number of Neighbor Solicitation pkts received" )
   naRx = Int( help="Number of Neighbor Advertisement pkts received" )
   naSuppressTx = Int( help="Number of Neighbor Advertisement pkts suppressed" )
   naSuppressSkipInvalidPkt = Int( help="Number of Neighbor Advertisements found"
                                        " as invalid" )
   naSuppressSkipInvalidOptPkt = Int( help="Number of Neighbor Advertisements found"
                                           " with invalid or absent options hdr" )
   naTgtSvi = Int( help="Number of Neighbor Advertisements not suppressed as target"
                        " is SVI" )
   naTgtSviSuppress = Int( help="Number of Neighbor Advertisements suppressed"
                                " as target is SVI" )
   rsSuppressTx = Int( help="Number of Router Solicitation pkts suppressed" )
   installedNaCount = Int( help="Number of dynamic neighbor entries added to"
                                " neighbor cache" )
   installedNaErrCount = Int( help="Number of dynamic neighbor entries added to "
                                   "neighbor cache with error" )
   encapDiscardLocalMac = Int( help="Number of packets discarded due to "
                                    "destination being local" )
   noIpv4LocalVtepAddr = Int( help="Number of times IPv4 local Vtep address not "
                                   "found while encapsulating the packets" )
   noIpv6LocalVtepAddr = Int( help="Number of times IPv6 local Vtep address not "
                                   "found while encapsulating the packets" )
   noIpv4RemoteVtepAddr = Int( help="Number of times IPv4 remote Vtep address not "
                                   "found while encapsulating the packets" )
   noIpv6RemoteVtepAddr = Int( help="Number of times IPv6 remote Vtep address not "
                               "found while encapsulating the packets" )
   encapTxTotalIpv6Bytes = Int( help="Total number of bytes encapped with IPv6 "
                                "underlay" )
   encapTxTotalIpv6Pkts = Int( help="Total number of packets encapped with IPv6 "
                                "underlay" )
   encapHwFwd = Int( help="Number of software packets forwarded to remote VTEPs "
                          "via HW HER" )
   encapHwFwdErr = Int( help="Number of software packets failed in forwarding to "
                             "remote VTEPs via HW HER" )
   vtepToVtepSourcePrunedFrames = Int( help="Total number of bridged vtep-to-vtep "
                                       "pkts that were source pruned" )
   proxyDisabledBcastArpRx  = Int( help="Total number of broadcast ARPs received "
                                " from local MAC when ARP proxy is disabled" )
   proxyDisabledBcastArpSenderRemoteRx  = Int( help="Total number of broadcast ARPs "
                             "received from remote MAC when ARP proxy is disabled" )
   proxyDisabledBcastArpSenderNotLearnedRx  = Int( help="Total number of broadcast "
                             "ARPs received from unlearned MAC when ARP proxy is "
                             "disabled" )
   proxyDisabledBcastArpSenderVmacRx  = Int( help="Total number of broadcast ARPs "
                             "received from virtual MAC when ARP proxy is disabled" )

   proxyDisabledMcastNdRx  = Int( help="Total number of multicast NS or NA received "
                                " from local MAC when ND proxy is disabled" )
   proxyDisabledMcastNdSenderRemoteRx  = Int( help="Total number of multicast NS/NA "
                             "received from remote MAC when ND proxy is disabled" )
   proxyDisabledMcastNdSenderNotLearnedRx  = Int( help="Total number of multicast "
                             "NS/NA received from unlearned MAC when ND proxy is "
                             "disabled" )
   proxyDisabledMcastNdSenderVmacRx  = Int( help="Total number of multicast NS/NA "
                             "received from virtual MAC when ND proxy is disabled" )
   floodingSuppressCount = Int( help="Number of packets suppressed from getting "
                                "flooded into Vxlan fabric" )
   installedArpL2Count = Int( help="Number of bridged ARP entries installed" )
   installedArpL2EntryAgedOut = Int( help="Number of bridged ARP entries removed "
                                     "due to host mac aging" )
   installedArpL2RefreshSent = Int( help="Number of bridged ARP entry refresh pkts "
                                    "sent" )
   installedArpL2RefreshSentErr = Int( help="Number of bridged ARP entry refresh "
                                       "failures" )
   installedArpL2ToL3Count = Int( help="Number of bridged ARP entries migrated from "
                                       "L2 to L3" )
   installedArpL2ToL3ErrCount = Int( help="Number of bridged ARP entries migrated "
                                          "from L2 to L3 in error" )
   dynamicMacEventSent = Int( help="Number of dynamic MAC learned events sent to "
                                  "platform" )
   arpReplyL2RefreshNoHer = Int( help="Number of ARP reply refresh pkts not headend "
                                   "replicated" )
   arpSuppressSkipMplsMac = Int( help="Number of ARP cache hits skipped during "
         "suppression due to target being behind MPLS tunnel" )
   arpSuppressCacheHitMacMiss = Int( help="Number of ARP cache hits skipped during "
         "suppression due to target MAC being not installed" )
   noHERCount = Int( help="Number of frames with no HER requested by dma-driver" )
   encapSchedSwHER = Int( help="Number of software packets forwarded to remote VTEPs"
                          " via scheduled software HER" )
   encapVarpSwHER = Int( help="Number of software packets from VARP MAC for HER " )

   # Internal function used by cli
   def getCntrIdMap( self ):
      # Below is a counters id map. Key is same as counter name in this model above.
      # Value is a tuple in which the first value is tacc definition of the counter
      # that is used to build version 1 dict with that counter id. If null then
      # that means it didn't exist in version 1.
      # Second value in the tuple is description of the counter as printed in CLI
      # This map consolidates all the info about counters in one place as opposed to
      # being spread out in multiple places before ga.macon-dev
      cntrIdMap = {
         'arpPkts': ( 'arp_pkts', 'Pkts to ARP resolver' ),
         'arpReqNoHER': ( '', 'Arp request packets that were not headend '
                          'replicated' ),
         'arpRxqRcvd': ( '', 'Arp request pkts received' ),
         'arpRxqRcvdFromVarpMac': ( '',
                                    'Arp request pkts received from a virtual mac' ),
         'arpSendErr': ( 'arp_send_err', 'Arp send error' ),
         'encapBytes': ( 'encap_bytes', 'Encapped bytes' ),
         'encapDiscardMlagPeerLink': ( 'encap_discard_mlag_peer_link',
                                       'Discard mlag peer link ( encap side )' ),
         'encapDiscardRunt': ( 'encap_discard_runt',
                               'Discard runt pkts ( encap side )' ),
         'encapDiscardVlanMap': ( 'encap_discard_vlan_map',
                                  'Discard vlan map ( encap side )' ),
         'encapDiscardVlanRange': ( 'encap_discard_vlan_range',
                                    'Discard vlan range ( encap side )' ),
         'encapDiscardLocalMac': ( 'encap_discard_local_mac',
                                   'Discard pkts destined to local mac address'
                                   '( encap side )' ),
         'encapDot1qTunnel': ( 'encap_dot1qtunnel',
                               'Rx dot1Q Tunnel pkts for encapsulation' ),
         'encapNoRemoteVtep': ( 'encap_no_remote_vtep',
                                'Discard pkts due to empty flood list' ),
         'encapNoRemoteVtepBfd': ( 'encap_no_remote_vtep_bfd',
                                   'Discard pkts due to empty bfd up flood list' ),
         'encapPkts': ( 'encap_pkts', 'Encapped packets' ),
         'encapReadErr': ( 'encap_read_err',
                           'Read error for pkts coming in for encapsulation' ),
         'encapSendDiscardHigh': ( 'encap_send_discard_high',
                                   'Encap discard high priority' ),
         'encapSendDiscardLow': ( 'encap_send_discard_low',
                                  'Encap discard low priority' ),
         'encapSendErr': ( 'encap_send_err', 'Encap send error' ),
         'encapTimeout': ( 'encap_timeout', 'Encap timeout' ),
         'encapSockDrop': ( '', 'Number of Pkts dropped by DMA socket' ),
         'encapTxAnyRemoteVtep': ( 'encap_tx_any_remote_vtep',
                                   'Encap send to first vtep on flood list' ),
         'encapTxTotalBytes': ( 'encap_tx_total_bytes', 'Tx bytes after '
                                'encapsulation' ),
         'encapTxTotalPkts': ( 'encap_tx_total_pkts', 'Tx pkts after '
                               'encapsulation' ),
         'ipToMacMlagSent': ( '', 'IpToMac bindings sent' ),
         'ipToMacMlagRecd': ( '', 'IpToMac bindings received' ),
         'rxForEncapBytes': ( 'rx_for_encap_bytes', 'Rx bytes for encapsulation' ),
         'rxForEncapBytesSockHiPrio': ( 'rx_for_encap_bytes_sock_high',
                                        'Rx high priority bytes for encapsulation' ),
         'rxForEncapBytesSockLoPrio': ( 'rx_for_encap_bytes_sock_low',
                                        'Rx low priority bytes for encapsulation' ),
         'rxForEncapPkts': ( 'rx_for_encap_pkts', 'Rx pkts for encapsulation' ),
         'rxForEncapPktsSockHiPrio': ( 'rx_for_encap_pkts_sock_high',
                                       'Rx high priority pkts for encapsulation' ),
         'rxForEncapPktsSockLoPrio': ( 'rx_for_encap_pkts_sock_low',
                                       'Rx low priority pkts for encapsulation' ),
         'arpSuppressSkipLocalMac': ( '', 'Arp suppression not applied to ARP'
                                     ' requests as mac for the target IP address is'
                                     ' local' ),
         'arpSuppressSkipLocalIp': ( '', 'Arp suppression not applied to ARP'
                                     ' requests as the target IP address is local' ),
         'arpSuppressSkipUnknownMac': ( '', 'Arp suppression not applied to ARP'
                                        ' requests as the sender MAC address is'
                                        ' unknown unicast' ),
         'arpSupressCacheMiss': ( '', 'Arp cache misses during Arp suppression' ),
         'arpSupressCache1Miss': ( '', 'Arp cache initial misses during Arp'
                                   ' suppression' ),
         'arpSupressDeny': ( '', 'Arp supression denied due to ACL' ),
         'arpSupressTx': ( '', 'Arp cache hits during Arp suppression' ),
         'arpSupressTxErr': ( '', 'Arp reply send errors during Arp suppression ' ),
         'arpSupressTxBcast': ( '', 'Arp reply sent to broadcast mac during Arp'
                                ' suppression ' ),
         'arpReplyRelayNoHer': ( '', 'Arp reply packets that were not headend '
                          'replicated due to arp relay' ),
         'arpSuppressSkipDynamicArp': ( '', 'Arp suppression not applied as arp is'
                                         ' dynamic and sender is my virtual MAC' ),
         'arpSuppressSkipProbe': ( '', 'Arp suppression not applied to probe' ),
         'garpPeerSyncMsgCount': ( '', 'GRAT ARP sync messages received' ),
         'garpPeerSyncMsgErrCount': ( '', 'GRAT ARP sync messages received in '
                                           'error' ),
         'garpPeerSyncMsgTxCount': ( '', 'GRAT ARP sync messages sent' ),
         'garpTgtSviSuppress': ( '', 'GRAT ARP pkts from SVI suppressed' ),
         'installedGarpCount': ( '', 'total dynamic ARP cache entries added' ),
         'installedGarpErrCount': ( '', 'total dynamic ARP cache entries added in '
                                         'error' ),
         'rarpRxCount': ( '', 'total RARP pkts received' ),
         'rarpRxInvalidCount': ( '', 'total RARP invalid pkts received' ),
         'nsReqNoHER': ( '', 'ND NS pkts skipped HER as target Ip matched SVI IP' ),
         'nsSuppressSkipDynamicArp': ( '', 'ND NS proxy not applied as neighbor'
                                         ' entry is dynamic' ),
         'nsSuppressSkipLocalMac': ( '', 'ND NS proxy not applied as target link is'
                                         ' local' ),
         'nsSuppressSkipLocalIp': ( '', 'ND NS proxy not applied as target IP is'
                                        ' local' ),
         'nsSuppressSkipUnknownMac': ( '', 'ND NS proxy not applied as sender link'
                                           ' not in fdb' ),
         'nsSuppressCacheMiss': ( '', 'ND NS proxy neighbor remote binding misses' ),
         'nsSuppressCache1Miss': ( '', 'ND NS proxy neighbor cache misses' ),
         'nsSuppressDeny': ( '', 'ND NS proxy denied due to ACL' ),
         'nsSuppressTxErr': ( '', 'ND NS proxy errors during transmit' ),
         'nsSuppressSkipInvalidPkt': ( '', 'ND NS proxy not applied as pkt is'
                                            ' invalid' ),
         'nsSuppressDad': ( '', 'ND NS proxy DAD frames suppressed' ),
         'nsSuppressDadVarpIp': ( '', 'ND NS DAD frames for varp IP suppressed' ),
         'nsSuppressVirtualLinkLocal': ( '',
                                          'ND NS frames for virtual LL suppressed' ),
         'nsNASent': ( '', 'ND NS proxy neighbor advt sent' ),
         'nsSuppressNonUnicastSource': ( '', 'ND NS pkts from unspecified source' ),
         'nsSuppressTx': ( '', 'ND NS pkts total suppressed' ),
         'nsRx': ( '', 'ND NS pkts total received' ),
         'naRx': ( '', 'ND NA pkts total received' ),
         'naSuppressTx': ( '', 'ND NA pkts total suppressed' ),
         'naSuppressSkipInvalidPkt': ( '', 'ND NA pkts invalid' ),
         'naSuppressSkipInvalidOptPkt': ( '', 'ND NA pkts options hdr invalid or '
                                              'absent' ),
         'naTgtSvi': ( '', 'ND NA pkts not suppressed as source is SVI' ),
         'naTgtSviSuppress': ( '', 'ND NA pkts suppressed as source is SVI' ),
         'rsSuppressTx': ( '', 'ND RS pkts total suppressed' ),
         'installedNaCount': ( '', 'total dynamic neigbhor cache entries added' ),
         'installedNaErrCount': ( '', 'total dynamic neighbor cache entries added '
                                       'in error' ),
         'noIpv4LocalVtepAddr': ( '', 'local V4 vtep address not found while '
                                   'encapping' ),
         'noIpv6LocalVtepAddr': ( '', 'local V6 vtep address not found while '
                                   'encapping' ),
         'noIpv4RemoteVtepAddr': ( '', 'remote V4 vtep address not found while '
                                   'encapping' ),
         'noIpv6RemoteVtepAddr': ( '', 'remote V6 vtep address not found while '
                                   'encapping' ),
         'encapTxTotalIpv6Bytes': ( '', 'Tx bytes after IPv6 encapsulation' ),
         'encapTxTotalIpv6Pkts': ( '', 'Tx pkts after IPv6 encapsulation' ),
         'encapHwFwd': ( '', 'SW pkts forwarded to remote VTEPs via HW HER' ),
         'encapHwFwdErr': ( '', 'SW pkts forwarding to remote VTEPs via HW HER '
                                 'failed' ),
         'floodingSuppressCount': ( '', 'Packets suppressed from getting flooded' ),
         'installedArpL2Count': ( '', 'bridged ARP entries learned' ),
         'installedArpL2EntryAgedOut': ( '', 'bridged ARP entries removed due to '
                                         'host mac aging' ),
         'installedArpL2RefreshSent': ( '', 'bridged ARP entry refresh sent' ),
         'installedArpL2RefreshSentErr': ( '', 'bridged ARP entry refresh failed' ),
         'installedArpL2ToL3Count': ( '', 'bridged ARP entries migrated from '
                                          'L2 to L3' ),
         'installedArpL2ToL3ErrCount': ( '', 'bridged ARP entries failed migration '
                                          'from L2 to L3' ),
         'vtepToVtepSourcePrunedFrames': ( '', 'bridged vtep-to-vtep pkts source '
                                           'pruned' ),
         'proxyDisabledBcastArpRx': ( '', 'broadcast ARPs from local MAC, ARP '
                                       'proxy disabled' ),
         'proxyDisabledBcastArpSenderRemoteRx': ( '', 'broadcast ARPs from remote '
                                                   'MAC, ARP proxy disabled' ),
         'proxyDisabledBcastArpSenderNotLearnedRx': ( '', 'broadcast ARPs from '
                                                       'unlearned MAC, ARP proxy '
                                                       'disabled' ),
         'proxyDisabledBcastArpSenderVmacRx' : ( '', 'broadcast ARPs from virtual '
                                                 'MAC, ARP proxy disabled' ),

         'proxyDisabledMcastNdRx' : ( '', 'multicast NS/NA from local MAC, ND proxy '
                                      'disabled' ),
         'proxyDisabledMcastNdSenderRemoteRx' : ( '', 'multicast NS/NA from remote '
                                                  'MAC, ND proxy disabled' ),
         'proxyDisabledMcastNdSenderNotLearnedRx' : ( '', 'multicast NS/NA from '
                                                      'unlearned MAC, ND proxy '
                                                      'disabled' ),
         'proxyDisabledMcastNdSenderVmacRx' : ( '', 'multicast NS/NA from virtual '
                                                'MAC, ND proxy disabled'
                                              ),
         'dynamicMacEventSent' : ( '',  'dynamic MAC learned events sent'),
         'arpReplyL2RefreshNoHer' : ( '', 'ARP reply refresh pkts not headend '
                                      'replicated' ),
         'arpSuppressSkipMplsMac': ( '',
               'ARP suppression not applied to ARP requests as MAC for the target '
               'IP address is behind MPLS tunnel' ),
         'arpSuppressCacheHitMacMiss': ( '',
               'ARP suppression not applied to ARP requests as MAC for the target '
               'IP address is not installed' ),
         'noHERCount' : ( '',  'no HER requested by dma-driver'),
         'encapSchedSwHER': ( '', 'SW pkts forwarded to remote VTEPs via scheduled '
                                  'HER' ),
         'encapVarpSwHER': ( '', 'SW pkts from VARP MAC for HER' ),
         }
      return cntrIdMap

   def getRemoved( self, revision ):
      'get list of counters present only in previous versions'

      if revision == 1:
         return [
            'Decapped bytes',
            'Discard bytes not for vtep ( decap side )',
            'Rx bytes before decapsulation',
            'Discard vlan map ( decap side )',
            'Discard VNI range ( decap side )',
            'Discard pkts with invalid vxlan header',
            'Discard pkts not for vtep ( decap side )',
            'Decapped packets',
            'Rx pkts before decapsulation',
            'Discard runt pkts ( decap side )',
            'Decap socket error',
            'Decap timeout',
         ]
      if revision == 2:
         return [
            'arpRxqUnmodSent',
            'arpRxqRewriteSent',
            'arpRxqLocalMacSent',
            'decapBytes',
            'decapBytesFilter',
            'decapBytesTotal',
            'decapDiscardVlanMap',
            'decapDiscardVniRange',
            'decapDiscardVxhdr',
            'decapPktFilter',
            'decapPkts',
            'decapPktsTotal',
            'decapRunt',
            'decapSockErr',
            'decapTimeout',
            'rxForV6EncapPkts',
            'rxForV6EncapBytes',
            'rxForV6EncapPktsSockHigh',
            'rxForV6EncapBytesSockHigh',
            'rxForV6EncapPktsSockLow',
            'rxForV6EncapBytesSockLow',
         ]
      return []

   # Get number of counters in a particular revision. Useful for testing.
   def getNumCounters( self, revision ):
      count = 0
      cntrMap = self.getCntrIdMap()
      if revision == 1:
         count = sum( bool( r1Name ) for r1Name, _ in six.itervalues( cntrMap ) )
      else:
         count = len( cntrMap )
      return count + len( self.getRemoved( revision ) )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # build enumerated list of counters and dict of counter descriptions
         swCounterId = Tac.newInstance( "Vxlan::VxlanCounterId" )
         vxlanCounterDict = {}
         vxlanCounters = []
         for key, ( r1Name, desc ) in six.iteritems( self.getCntrIdMap() ):
            if not r1Name:
               continue
            cntrId = swCounterId.enumVal( r1Name )
            cntrValue = dictRepr.get( key, 0 )
            vxlanCounterDict[ cntrId ] = desc
            vxlanCounters.append( { "key": cntrId, "cntrValue": cntrValue } )
         # add back any removed counters
         rmv = self.getRemoved( 1 )
         for cntrId, desc in enumerate( rmv, max( vxlanCounterDict ) + 1 ):
            vxlanCounterDict[ cntrId ] = desc
            vxlanCounters.append( { "key": cntrId, "cntrValue": 0 } )

         retVal = { "vxlanCounters": vxlanCounters,
                    "vxlanCounterDict": vxlanCounterDict
                    }
         return retVal
      dictRepr.update( { key: 0 for key in self.getRemoved( revision ) } )
      return dictRepr

   def printSwCounter( self, attrName, name ):
      attr = getattr( self, attrName )
      if attr:
         counter = attr
      else:
         counter = 0
      print( "%-*s :  %d" % ( 52, name, counter ) )

   def render( self ):
      cntrIdMap = self.getCntrIdMap()
      swCounterId = Tac.newInstance( "Vxlan::VxlanCounterId" )
      for key, value in sorted(
            six.iteritems( cntrIdMap ),
            key=lambda x: swCounterId.enumVal( x[ 1 ][ 0 ] or x[ 0 ] ) ):
         self.printSwCounter( key, value[ 1 ] )

class VxlanVarpCounters( Model ):
   __revision__ = 3

   arpCacheInstalled = Int( help="Number of ARP packets installed in ARP cache" )
   arpCacheInstallErr = Int( help="Number of ARP cache installation failure" )
   arpCacheInstallConflict = Int( help="Number of ARP cache installation conflicts" )
   arpCacheSyncedTx = Int( help="Number of ARP cache sent to MLAG peer" )
   arpCacheSyncedRx = Int( help="Number of ARP cache received from MLAG peer" )
   arpReplyPktSyncedTx = Int( help="Number of ARP reply packets synced over "
                              "transmission" )
   arpReplyPktSyncedRx = Int( help="Number of ARP reply packets synced over "
                              "reception" )
   arpDynamicRemoteArpNotifySyncedTx = Int( help="Number of Remote ARPs which have "
                           "synced to MLAG peer" )
   arpDynamicRemoteArpNotifySyncedRx = Int( help="Number of remote ARPs which have "
                        "been received from MLAG peer" )
   arpCacheSyncedInstallErr = Int( help="Number of ARP packets sync install"
                                   " failures" )
   arpCacheSyncedInstallConflict = Int( help="Number of ARP packets sync install"
                                        " conflicts" )
   vxlanEncapArpRx = Int( help="Number of VXLAN ARP packets received" )
   arpReplyToVarpMacVarpVtepRx = Int( help="Number of ARP reply packets for VARP "
                                      "mac and VARP vtep" )
   arpReplyRx = Int( help="Number of naked ARP reply packets received" )
   arpReplyHerTx = Int( help="Number of ARP reply packets received for HER "
                        "encapsulation" )
   arpReplyTx = Int( help="Number of ARP reply packets received for known VTEP"
                     "encapsulation" )
   arpReqDecapAndTx = Int( help="Number of VXLAN ARP request decapsulated and "
                           "sent" )
   arpReqDecapAndTxErr = Int( help="Number of VXLAN ARP request failure in "
                              "decapsulated and sent" )
   arpReqDecapAndTxPortIdErr = Int( help="Number of VXLAN ARP request failure due "
                                    "to unknown CPU port id" )
   ndNsDecapAndTx = Int( help="Number of VXLAN NS decapsulated and sent" )
   ndNsDecapAndTxErr = Int( help="Number of VXLAN NS failure in "
                              "decapsulated and sent" )
   pktTooShortRx = Int( help="Number of received packets failed minimum frame size" )
   unexpectedVlanIdRx = Int( help="Number of received packets failed due to "
                                 "unexpected VLAN range" )
   unexpectedEtherTypeRx = Int( help="Number of received packets failed due to "
                                "unexpected ethertype" )
   unexpectedNonIpRx = Int( help="Number of received packets failed due to non "
                            "IP packet" )
   unexpectedIpProtoRx = Int( help="Number of received packets failed due to "
                              "unexpected IP protocol value" )
   unexpectedUdpPortRx = Int( help="Number of received packets failed due to "
                              "unexpected UDP destination port number" )
   unexpectedInnerNotArpRx = Int( help="Number of received packets failed due "
                                  "to inner ARP or IPv6 ND frames missing" )
   unexpectedArpReplyRx = Int( help="Number of received ARP reply packets failure" )
   invalidIpChecksumRx = Int( help="Number of received packets failed due to "
                              "invalid IP checksum " )
   invalidVxlanHdrRx = Int( help="Number of received packets failed due to "
                            "invalid VXLAN Header check" )
   invalidArpRx = Int( help="Number of received ARP packets failure" )
   noVlanToVniMapErr = Int( help="Number of received packets failed due to "
                            "missing VLAN-2-VNI map" )
   arpReplySrcIpErr = Int( help="Number of received ARP reply packets failed to "
                           "source IP error check" )
   arpReplyBumListErr = Int( help="Number of received ARP reply packets failed "
                             "to BUM processing error" )
   arpReplyMsgToArpMonErr = Int( help="Number of received ARP reply packets failed "
                                 "to ARP monitor processing error" )
   prependDecapForTxRaw = Int( help="Number of packets decapped and sent on txraw "
                               "interface from the CPU" )
   prependDecapForFabric = Int( help="Number of packets decapped and sent on fabric "
                                "interface from the CPU" )
   prependDecapForTxFwd = Int( help="Number of packets decapped and sent on txfwd "
                               "interface from the CPU" )

   arpRxqRcvd = Int( help="Number of received ARP request packets" )
   arpRxqRcvdFromVarpMac = Int( help="Number of ARP request packets received from a "
         "virtual mac" )
   arpReplyRcvd = Int( help="Number of ARP reply packets received" )
   arpReplyUnmodSent = Int( help="Number of ARP reply packets sent with original "
                            "source VARP MAC" )
   arpReplyRewriteDiscard = Int( help="Number of ARP reply packets discard due to "
                                 "missing rewrite pending list match" )
   arpReplyRelayTx = Int( help="Number of ARP reply relay packets sent" )
   arpLocalAddressTx = Int( help="Number of ARP request packets rewritten with local"
                            " IP and bridged" )

   arpReplyToBridgeMacReplicated = Int( help="Number of ARP reply packets rewritten "
                                        "and replicated" )
   arpReplyToVarpMacNoHer = Int( help="Number of ARP reply packets were not headend "
                                 "replicated" )
   dropArpReqForVarpIpToNonVarpVtep = Int( help="Number of ARP request packets for "
                                           "virtual IP to local VTEP IP dropped" )
   dropArpReqForNonSwitchIpToVarpVtep = Int( help="Number of ARP request packets "
                                             "for non-switch IP to VARP VTEP IP "
                                             "dropped" )
   naRx = Int( help="Number of naked neighbor advertisement packets received" )
   invalidNaRx = Int( help="Number of naked neighbor advertisement packets "
                      "received that were invalid" )
   invalidNaOptRx = Int( help="Number of naked neighbor advertisement packets "
                         "received with invalid or no option hdr" )
   naToVarpMacNoHer = Int( help="Number of neighbor advertisement packets that were"
                           " not headend replicated" )
   neighCacheSyncedTx = Int( help="Number of neighbor entries sent to "
                             "MLAG peer" )
   neighCacheInstalled = Int( help="Number of neighbor entries installed in IPv6 "
                              "neighbor cache" )
   neighCacheInstallErr = Int( help="Number of IPv6 neighbor cache "
                               "installation failures" )
   neighCacheInstallConflict = Int( help="Number of IPv6 neighbor cache "
                                    "installation conflicts" )
   neighCacheSyncedRx = Int( help="Number of neighbor entries received from "
                             "MLAG peer" )
   neighSyncMsgErrCount = Int( help="Number of neighbor sync messages received "
                               "in error" )
   neighCacheSyncedInstallErr = Int( help="Number of neighbor entries sync "
                                     "install failures" )
   neighCacheSyncedInstallConflict = Int( help="Number of neighbor entries sync "
                                          "install conflicts" )
   ipv4UnderlayPktsRx = Int( help="VXLAN encapped packets received with IPv4 "
                             "underlay" )
   ipv6UnderlayPktsRx = Int( help="VXLAN encapped packets received with IPv6 "
                             "underlay" )
   vxlanEvpnPeerFilterDecapped = Int( help="Number of VXLAN ARP packets filtered "
         "from multihoming peer and decapped" )
   localAAInstalledWithoutActiveRemote = Int( help="Number of ActiveLocal hosts "
         "installed into ARP cache separately from remote binding" )
   vxlanSecEncapArpRx = Int( help="Number of VxlanSec ARP packets received" )
   vxlanSecArpReqForVarpIpDrop = Int( help="Number of VxlanSec ARP requests to "
                                      "VARP IP dropped" )
   vxlanSecArpReplyTx = Int( help="Number of VxlanSec ARP replies sent" )
   vxlanSecArpReplyTxErr = Int( help="Number of VxlanSec ARP replies sent "
                                     "in error" )
   invalidNdRx = Int( help="Number of invalid inner IPv6 NS or NA packets received" )
   vxlanEncapNdRx = Int( help="Number of inner IPv6 NS or NA packets "
                              "received" )
   vxlanSecEncapNdRx = Int( help="Number of VxlanSec inner IPv6 NS or NA packets "
                                 "received" )
   vxlanSecNdNsForVarpIpDrop = Int( help="Number of VxlanSec inner IPv6 NS requests"
                                         " to VARP IP dropped" )
   vxlanSecNdReplyTx = Int( help="Number of VxlanSec inner IPv6 NA reply sent" )
   vxlanSecNdReplyTxErr = Int( help="Number of VxlanSec inner IPv6 NA reply sent "
                                    "in error" )
   ndNsRxqRcvd = Int( help="Number of inner IPv6 NS requests received" )
   ndNaReplyRcvd = Int( help="Number of inner IPv6 NS replies received" )
   ndNsRxqRcvdFromVarpMac = Int( help="Number of inner IPv6 NS requests received "
                                       " from a virtual mac" )
   dropNdNsReqForNonSwitchIpToVarpVtep = Int( help="Number of IPv6 NS requests "
                                             "for non-switch IP to VARP VTEP IP "
                                             "dropped" )
   dropNdNsReqForVarpIpToNonVarpVtep = Int( help="Number of IPv6 NS requests for "
                                           "virtual IP to local VTEP IP dropped" )
   ndNaReplyToVarpMacVarpVtepRx = Int( help="Number of IPv6 NA reply packets for "
                                            "VARP mac and VARP vtep received" )
   ndNaReplyTx = Int( help="Number of IPv6 NA reply packets sent for known VTEP "
                           "encapsulation" )
   arpProbeDropForVarpIP = \
         Int( help="Number of inner ARP probes for VARP IP dropped" )
   ndDadDropForVarpIP = Int( help="Number of inner IPv6 DAD for VARP IP dropped" )
   arpPublishGarpInstalled = Int( help="Number of ARP entries installed from "
                                       "Gratuitous ARPs when ARP publish is "
                                       "enabled" )
   arpPublishGarpInstallErr = Int( help="Number of ARP entries failed to install "
                                        "from Gratuitous ARPs when ARP publish is "
                                        "enabled" )
   arpPublishGarpInstallConflict = Int( help="Number of ARP entries failed to "
                                             "install from Gratuitous ARPs due to "
                                             "conflict when ARP publish is "
                                             "enabled" )
   arpPublishGarpRx = Int( help="Number of Gratuitous ARPs received from a local "
                                "MAC when ARP publish is enabled" )
   arpPublishGarpSenderRemoteRx = Int( help="Number of Gratuitous ARPs received "
                                            "from a remote MAC when ARP publish is "
                                            "enabled" )
   arpReplyHerHwFwdErr = Int( help="Number of ARP or IPv6 NA reply pkts sent via H/W"
                                   " HER in error" )
   arpReplyHerHwFwd = Int( help="Number of ARP or IPv6 NA reply pkts sent via H/W"
                                "HER" )

   scheduledSyncMsgRxErrCount = Int( help="Number of scheduled sync msg errors" )
   scheduledSyncMsgRxGapCount = Int( help="Number of scheduled sync msg pkts dropped"
                                          " in transit" )
   scheduledSyncRxInstallErrCount = Int( help="Number of scheduled sync binding"
                                              " install errors" )
   scheduledSyncRxInstallConflictCount = Int( help="Number of scheduled sync binding"
                                                   " install conflicts" )
   scheduledSyncRxInstallCount = Int( help="Number of scheduled sync bindings"
                                           " installed" )
   scheduledSyncRxPktCount = Int( help="Number of scheduled sync msgs received" )
   scheduledSyncTxCount = Int( help="Number of scheduled sync bindings sent" )
   scheduledSyncTxPktCount = Int( help="Number of scheduled sync msgs sent" )
   vxlanArpSockDropCount = Int( help="Number of VXLAN ARP/ND pkts dropped in"
                                     " socket" )

   # Internal function used by cli
   def getCntrIdMap( self ):
      # Below is a counters id map. Key is same as counter name in this model above.
      # Value is a tuple in which the first value is tacc definition of the counter
      # that is used to build version 1 dict with that counter id. If null then
      # that is didn't exist in version 1.
      # Second value in tuple is description of the counter as printed in CLI
      # This map consolidates all the info about counters in one place as opposed to
      # being spread out in multiple places before ga.macon-dev
      cntrIdMap = {
         "arpCacheInstalled": ( 'arpCacheInstalled', "ARP entries installed" ),
         "arpCacheInstallErr": ( "arpCacheInstallErr", "ARP entries discarded" ),
         "arpCacheInstallConflict": ( "", "ARP entries discarded due to conflicts" ),
         "arpCacheSyncedTx": ( "arpCacheSyncedTx", "ARP sync messages sent to MLAG"
                              " peer" ),
         "arpCacheSyncedRx": ( "arpCacheSyncedRx",
                               "ARP sync messages received from MLAG peer" ),
         "arpReplyPktSyncedTx": ( "",
                                  "ARP sync messages with reply sent to MLAG peer" ),
         "arpReplyPktSyncedRx": ( "", "ARP sync messages with reply received "
                                       "from MLAG peer" ),
         "arpDynamicRemoteArpNotifySyncedTx": ( "", "ARP sync messages type-2 sent "
                                                   "to MLAG peer" ),
         "arpDynamicRemoteArpNotifySyncedRx": ( "", "ARP sync messages type-2 "
                                                      "received from MLAG peer" ),
         "arpCacheSyncedInstallErr": ( "arpCacheSyncedInstallErr",
                                       "ARP sync entries discarded" ),
         "arpCacheSyncedInstallConflict": ( "",
                                            "ARP sync entries discarded due "
                                            "to conflicts" ),
         "vxlanEncapArpRx": ( "vxlanEncapArpRx", "VXLAN ARP packets received" ),
         "arpReplyToVarpMacVarpVtepRx": ( "arpReplyToVarpMacVarpVtepRx",
                                          "ARP replies destined to VARP VTEP IP" ),
         "arpReplyRx": ( "arpReplyRx", "ARP replies received over peer link" ),
         "arpReplyHerTx": ( "arpReplyHerTx", "ARP replies head end replicated" ),
         "arpReplyTx": ( "arpReplyTx", "ARP replies sent" ),
         "arpReqDecapAndTx": ( "arpReqDecapAndTx", "ARP requests not proxied" ),
         "arpReqDecapAndTxErr": ( "arpReqDecapAndTxErr",
                                  "ARP requests not proxied and failed to send" ),
         "arpReqDecapAndTxPortIdErr": ( "arpReqDecapAndTxPortIdErr",
                                        "ARP replies failed to send due to "
                                        "unknown CPU port" ),
         "ndNsDecapAndTx": ( "ndNsDecapAndTx", "NS packets not proxied" ),
         "ndNsDecapAndTxErr": ( "ndNsDecapAndTxErr",
                                "NS packets not proxied and failed to send" ),
         "pktTooShortRx": ( "pktTooShortRx", "ARP packets discarded due to packet "
                                             "being too short" ),
         "unexpectedVlanIdRx": ( "", "ARP packets discarded due to unexpected "
                                     "VLAN" ),
         "unexpectedEtherTypeRx": ( "unexpectedEtherTypeRx",
                                    "ARP packets discarded due to missing "
                                    "Arista ARP EtherType" ),
         "unexpectedNonIpRx": ( "unexpectedNonIpRx", "ARP packets discarded due to"
                                                " unexpected IP header protocol" ),
         "unexpectedIpProtoRx": ( "unexpectedIpProtoRx", "ARP packets discarded due"
                                                      " to missing UDP header" ),
         "unexpectedUdpPortRx": ( "unexpectedUdpPortRx", "ARP packets discarded due"
                                                         " to unexpected UDP port" ),
         "unexpectedInnerNotArpRx": ( "unexpectedInnerNotArpRx",
                                       "packets discarded due to inner payload"
                                       " is not ARP or IPv6 ND" ),
         "unexpectedArpReplyRx": ( "unexpectedArpReplyRx",
                                   "ARP packets discarded due to unexpected "
                                   "ARP request" ),
         "invalidIpChecksumRx": ( "invalidIpChecksumRx", "ARP packets discarded due"
                                    " to failed IPv4 checksum" ),
         "invalidVxlanHdrRx": ( "invalidVxlanHdrRx", "ARP packets discarded due to"
                                 " corrupt VXLAN header" ),
         "invalidArpRx": ( "invalidArpRx", "ARP packets discarded due to corrupt"
                                             " inner ARP header" ),
         "noVlanToVniMapErr": ( "noVlanToVniMapErr", "ARP packets discarded due to"
                                 " missing VLAN for VNI" ),
         "arpReplySrcIpErr": ( "arpReplySrcIpErr", "ARP packets discarded due to"
                                                   " missing source IP" ),
         "arpReplyBumListErr": ( "arpReplyBumListErr", "ARP packets discarded due"
                                                       " to empty flood set" ),
         "arpReplyMsgToArpMonErr": ( "arpReplyMsgToArpMonErr",
                                     "ARP replies to ARP monitor discarded" ),
         "prependDecapForTxRaw": ( "prependDecapForTxRaw",
                                   "ARP packets decapsulated and bridged locally"
                                   " on txraw interface" ),
         "prependDecapForTxFwd": ( "prependDecapForTxFwd",
                                   "ARP packets decapsulated and bridged locally"
                                   " on txfwd interface" ),
         "prependDecapForFabric": ( "prependDecapForFabric",
                                    "ARP packets decapsulated and bridged locally"
                                    " on fabric interface" ),
         "arpRxqRcvd": ( "", "ARP requests received" ),
         "arpRxqRcvdFromVarpMac": ( "",
            "ARP requests received from a virtual MAC" ),
         "arpReplyRcvd": ( "", "ARP replies received" ),
         "arpReplyUnmodSent": ( "", "ARP replies sent unmodified" ),
         "arpReplyRewriteDiscard": ( "", "ARP replies rewritten but discarded" ),
         "arpReplyRelayTx": ( "", "ARP relay replies sent" ),
         "arpLocalAddressTx": ( "", "ARP requests rewritten and bridged locally" ),
         "arpReplyToBridgeMacReplicated": ( "",
                                            "ARP replies rewritten and head"
                                            " end replicated" ),
         "arpReplyToVarpMacNoHer": ( "",
                                      "ARP replies to VARP MAC and not head end"
                                      " replicated" ),
         "dropArpReqForVarpIpToNonVarpVtep":
            ( "", "ARP requests for VARP VTEP IP to local VTEP IP dropped" ),
         "dropArpReqForNonSwitchIpToVarpVtep":
            ( "", "ARP requests for non-switch IP to VARP VTEP IP dropped" ),
         "naRx": ( "", "Neighbor Advertisements received" ),
         "invalidNaRx": ( "", "Neighbor Advertisements discarded due to"
                              " invalid packet" ),
         "invalidNaOptRx": ( "", "Neighbor Advertisements discarded due to"
                                 " missing options header" ),
         "naToVarpMacNoHer": ( "",
                                "Neighbor Advertisements not head end replicated" ),
         "neighCacheSyncedTx": ( "neighCacheSyncedTx",
                                 "Neighbor sync messages sent to MLAG peer" ),
         "neighCacheInstalled": ( 'neighCacheInstalled',
                                  "Neighbor entries installed" ),
         "neighCacheInstallErr": ( "neighCacheInstallErr",
                                    "Neighbor entries discarded" ),
         "neighCacheInstallConflict": ( "",
                                         "Neighbor entries discarded due to"
                                         " conflicts" ),
         "neighCacheSyncedRx": ( "neighCacheSyncedRx",
                                 "Neighbor sync messages received from MLAG peer" ),
         "neighSyncMsgErrCount": ( "", "Neighbor sync messages received in error" ),
         "neighCacheSyncedInstallErr": ( "neighCacheSyncedInstallErr",
                                         "Neighbor sync entries discarded" ),
         "neighCacheSyncedInstallConflict":
            ( "", "Neighbor sync entries discarded due to conflicts" ),
         "ipv4UnderlayPktsRx": ( "", "Packets received with IPv4 underlay" ),
         "ipv6UnderlayPktsRx": ( "", "Packets received with IPv6 underlay" ),
         "vxlanEvpnPeerFilterDecapped": ( "", "ARP packets filtered from "
               "multihoming peer and decapped" ),
         "localAAInstalledWithoutActiveRemote": ( "", "ActiveLocal hosts installed "
               "into ARP cache separately from remote binding" ),
         "vxlanSecEncapArpRx": ( "", "VxlanSec ARP packets received" ),
         "vxlanSecArpReqForVarpIpDrop": ( "", "VxlanSec ARP requests to VARP IP "
                                              "dropped" ),
         'vxlanSecArpReplyTx': ( "", "VxlanSec ARP replies sent" ),
         'vxlanSecArpReplyTxErr': ( "", "VxlanSec ARP replies sent in error" ),
         "invalidNdRx": ( "", "Inner IPv6 NS or NA packets skipped due to invalid "
                              "header" ),
         "vxlanEncapNdRx": ( "", "Inner IPv6 NS or NA packets received" ),
         "vxlanSecEncapNdRx": ( "", "VxlanSec inner IPv6 NS or NA packets "
                                    "received" ),
         "vxlanSecNdNsForVarpIpDrop": ( "", "VxlanSec inner NS requests to VARP IP "
                                            "dropped" ),
         'vxlanSecNdReplyTx': ( "", "VxlanSec IPv6 NA reply sent" ),
         'vxlanSecNdReplyTxErr': ( "", "VxlanSec IPv6 NA reply sent in error" ),
         'ndNsRxqRcvd' : ( "", "Inner IPv6 NS requests received" ),
         "ndNaReplyRcvd" : ( "", "Inner IPv6 NA replies received" ),
         'ndNsRxqRcvdFromVarpMac' : ( "", "Inner IPv6 NS requests received from a "
                                        "virtual MAC" ),
         "dropNdNsReqForNonSwitchIpToVarpVtep" : ( "",
               "Inner IPv6 Ns requests for non-switch IP to VARP VTEP IP dropped" ),
         "dropNdNsReqForVarpIpToNonVarpVtep": ( "",
               "Inner IPv6 NS requests for VARP VTEP IP to local VTEP IP dropped" ),
         "ndNaReplyToVarpMacVarpVtepRx": ( "",
               "Inner IPv6 NA replies destined to VARP VTEP IP" ),
         "ndNaReplyTx" : ( "", "IPv6 NA replies sent" ),
         "arpProbeDropForVarpIP": ( "", "Inner ARP probe for VARP IP dropped" ),
         "ndDadDropForVarpIP": ( "", "Inner IPv6 DAD for VARP IP dropped" ),
         'arpPublishGarpInstalled': ( '', 'Gratuitous ARPs installed '
                                          'when ARP publish is enabled' ),
         'arpPublishGarpInstallErr': ( '', 'Gratuitous ARPs install errors '
                                           'when ARP publish is enabled' ),
         'arpPublishGarpInstallConflict': ( '', 'Gratuitous ARPs install conflicts '
                                                'when ARP publish is enabled' ),
         'arpPublishGarpRx': ( '', 'Gratuitous ARPs from a local MAC '
                                   'when ARP publish is enabled' ),
         'arpPublishGarpSenderRemoteRx': ( '', 'Gratuitous ARPs from a remote MAC '
                                               'when ARP publish is enabled' ),
         'arpReplyHerHwFwdErr': ( '', 'ARP or IPv6 NA reply pkts sent via H/W HER '
                                        'in error' ),
         'arpReplyHerHwFwd': ( '', 'ARP or IPv6 NA reply pkts sent via H/W HER' ),
         'scheduledSyncMsgRxErrCount' : ( '', 'Scheduled sync msg errors' ),
         'scheduledSyncMsgRxGapCount' : ( '', 'Scheduled sync msg pkts dropped'
                                              ' in transit' ),
         'scheduledSyncRxInstallErrCount' : ( '', 'Scheduled sync binding'
                                                  ' install errors' ),
         'scheduledSyncRxInstallConflictCount' : ( '', 'Scheduled sync binding'
                                                       ' install conflicts' ),
         'scheduledSyncRxInstallCount' : ( '', 'Scheduled sync bindings'
                                               ' installed' ),
         'scheduledSyncRxPktCount' : ( '', 'Scheduled sync msgs received' ),
         'scheduledSyncTxCount' : ( '', 'Scheduled sync bindings sent' ),
         'scheduledSyncTxPktCount' : ( '', 'Scheduled sync msgs sent' ),
         'vxlanArpSockDropCount' : ( '', 'VXLAN ARP/ND packets dropped in socket' )
      }
      return cntrIdMap

   def getRemoved( self, revision ):
      'get list of counters present only in previous versions'

      if revision == 2:
         return [
            'arpRxqUnmodSent',
            'arpRxqRewriteSent',
            'arpReplyRewriteSent',
            'arpReplyRewriteAgingOut',
         ]
      return []

   # Get number of counters in a particular revision. Useful for testing.
   def getNumCounters( self, revision ):
      count = 0
      cntrMap = self.getCntrIdMap()
      if revision == 1:
         count = sum( bool( r1Name ) for r1Name, _ in six.itervalues( cntrMap ) )
      else:
         count = len( cntrMap )
      return count + len( self.getRemoved( revision ) )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         varpCounterId = Tac.newInstance( "Vxlan::VarpCounterId" )
         # Let's build a dict
         varpCounters = []
         for key, ( r1Name, _ ) in six.iteritems( self.getCntrIdMap() ):
            if not r1Name:
               continue
            cntrId = varpCounterId.enumVal( r1Name )
            cntrValue = dictRepr.get( key, 0 )
            varpCounters.append( { "key": cntrId, "cntrValue": cntrValue } )

         retVal = { "varpCounters": varpCounters }
         return retVal

      dictRepr.update( { key: 0 for key in self.getRemoved( revision ) } )
      return dictRepr

   def printVarpCounter( self, attrName, name ):
      attr = getattr( self, attrName )
      if attr:
         counter = attr
      else:
         counter = 0
      print( "%s: %d" % ( name, counter ) )

   def render( self ):
      cntrIdMap = self.getCntrIdMap()

      for key, value in sorted(
            six.iteritems( cntrIdMap ),
            key=lambda x: x[ 1 ][ 1 ] ):
         self.printVarpCounter( key, value[ 1 ] )

class VxlanControlServiceStatus( Model ):
   status = Enum( values=( "connecting", "negotiating", "established",
                           "disconnecting" ),
                  help="Connection status of Vxlan Control Service with CVX" )
   resyncInProgress = Bool( help="True if resync with CVX is in progress "
                            "after CVX restart", optional=True )
   purgeInProgress = Bool( help="True if purging of HW MAC table entries "
                           "received from CVX is in progress after CVX restart",
                           optional=True )

   def render( self ):
      print( "%-36s: %s" % ( "Controller connection status",
                                    self.status.capitalize() ) )
      print( "%-36s: %s" % ( "Resync in progress after CVX reboot",
                                    "Yes" if self.resyncInProgress else "No" ) )
      print( "%-36s: %s" % ( "Purge in progress after CVX reboot",
                                    "Yes" if self.purgeInProgress else "No" ) )

class LRIpUplinkPort( Model ):
   macAddr = MacAddress( help='Interface MAC address' )
   vlan = Int( help='Interface VLAN' )
   interfaces = List( valueType=Interface,
                      help='A list of physical interfaces' )

class LRIpRoute( Model ):
   nexthop = List( valueType=IpGenericAddress, help='List of nexthop addresses' )

class LRIpPort( Model ):
   vni = Int( help='Interface VNI' )
   macAddr = MacAddress( help='Interface MAC address' )

class VxlanRoutingTable( Model ):
   lRPort = Dict( valueType=LRIpPort,
                  help='A mapping from VNI to IP interface' )
   lRUplinkPort = Dict( valueType=LRIpUplinkPort,
                        help='A mapping from IP address to uplink interface' )
   lRRoute = Dict( valueType=LRIpRoute,
                   help='A mapping from IP prefix to IP route' )
   lRCreateLocalIpPortsOnly = Bool( default=False,
                                    help='Create local IP interface only' )

def createHdr( title ):
   hdrFormat = Format( isHeading=True, justify="left" )
   hdrFormat.noPadLeftIs( True )
   return FormattedCell( nCols=1, content=title, format=hdrFormat )

class VxlanLogicalRouterModel( Model ):
   __public__ = False
   _vniDotted = Bool( help='VNI display format' )
   routingTable = Dict( valueType=VxlanRoutingTable,
                        help='A mapping from logical router name to routing table' )

   def render( self ):
      fmt = Format( justify='left' )
      fmt.noPadLeftIs( True )
      lRPortTable = createTable( ( ( createHdr( "Name" ), "l" ),
                                   ( createHdr( "IP Address" ), "l" ),
                                   ( createHdr( "VNI" ), "l" ),
                                   ( createHdr( "Mac Address" ), "l" ) ) )
      lRPortTable.formatColumns( *( [ fmt ] * 4 ) )
      lRUplinkPortTable = createTable( ( ( createHdr( "Name" ), "l" ),
                                    ( createHdr( "IP Address" ), "l" ),
                                    ( createHdr( "Mac Address" ), "l" ),
                                    ( createHdr( "VLAN" ), "l" ),
                                    ( createHdr( "Ports" ), "l" ) ) )
      lRUplinkPortTable.formatColumns( *( [ fmt ] * 5 ) )
      lRRouteTable = createTable( ( ( createHdr( "Name" ), "l" ),
                                    ( createHdr( "Prefix" ), "l" ),
                                    ( createHdr( "Nexthop" ), "l" ) ) )
      lRRouteTable.formatColumns( *( [ fmt ] * 3 ) )

      portCount = 0
      uplinkPortCount = 0
      routeCount = 0
      for lRName, routingTable in sorted( six.iteritems( self.routingTable ) ):
         firstLine = True
         for ipAddr, ipPort in sorted( routingTable.lRPort.items() ):
            portCount = portCount + 1
            if firstLine:
               lRPortTable.newRow( lRName, ipAddr, vniToString( ipPort.vni,
                                   self._vniDotted ), ipPort.macAddr.displayString )
               firstLine = False
            else:
               lRPortTable.newRow( "", ipAddr, vniToString( ipPort.vni,
                                   self._vniDotted ), ipPort.macAddr.displayString )

         firstLine = True
         for ipAddr, ipUplinkPort in sorted( routingTable.lRUplinkPort.items() ):
            uplinkPortCount = uplinkPortCount + 1
            if firstLine:
               lRUplinkPortTable.newRow( lRName,
                                         ipAddr,
                                         ipUplinkPort.macAddr.displayString,
                                         str( ipUplinkPort.vlan ),
                                         ', '.join( map( str,
                                            sorted( ipUplinkPort.interfaces ) ) ) )
               firstLine = False
            else:
               lRUplinkPortTable.newRow( "", ipAddr,
                                         ipUplinkPort.macAddr.displayString,
                                         str( ipUplinkPort.vlan ),
                                         ', '.join( map( str,
                                            sorted( ipUplinkPort.interfaces ) ) ) )

         firstLine = True
         for prefix in sorted( routingTable.lRRoute ):
            ipRoute = routingTable.lRRoute[ prefix ]
            routeCount = routeCount + 1
            if firstLine:
               lRRouteTable.newRow( lRName, prefix,
                                    ', '.join( map( str,
                                                    sorted( ipRoute.nexthop ) ) ) )
               firstLine = False
            else:
               lRRouteTable.newRow( "", prefix,
                                    ', '.join( map( str,
                                                    sorted( ipRoute.nexthop ) ) ) )

      if portCount:
         print( lRPortTable.output() )
         print( "Total entries for this criterion: %d" % portCount )

      if uplinkPortCount:
         if portCount or routeCount:
            print( "\nUplink ports:" )
         print( lRUplinkPortTable.output() )
         print( "Total entries for this criterion: %d" % uplinkPortCount )

      if routeCount:
         if portCount or uplinkPortCount:
            print( "\nRoutes:" )
         print( lRRouteTable.output() )
         print( "Total entries for this criterion: %d" % routeCount )

class ControlPlaneConfigDirectionModel( Model ):
   direction = Enum(
         help='Direction of data sharing for a control plane',
         values=[ 'import', 'export', 'both' ] )

class ControlPlaneConfigSourceModel( Model ):
   sources = Dict(
         help='Sources of control plane configuration, keyed by source name',
         keyType=str,
         valueType=ControlPlaneConfigDirectionModel )

class VlanControlPlaneConfigModel( Model ):
   controlPlanes = Dict(
         help='Control planes active in a VLAN, keyed by control plane name',
         keyType=str,
         valueType=ControlPlaneConfigSourceModel )

class VxlanControlPlaneConfigModel( Model ):
   vlans = Dict(
         help='VLANs configured for VXLAN, keyed by VLAN',
         keyType=int,
         valueType=VlanControlPlaneConfigModel )

   def render( self ):
      tableHeadings = [ ( 'VLAN', 'l' ), ( 'Control Plane', 'l' ),
            ( 'Direction', 'l' ), ( 'Source', 'l' ) ]
      table = createTable( tableHeadings )
      for vlan, controlPlanes in sorted( self.vlans.items(), key=itemgetter( 0 ) ):
         firstControlPlaneInVlan = True
         for controlPlane, sources in sorted( controlPlanes.controlPlanes.items(),
               key=itemgetter( 0 ) ):
            firstSourceInControlPlane = True
            for source, direction in sorted( sources.sources.items(),
                  key=itemgetter( 0 ) ):
               vlanColumn = ''
               if firstControlPlaneInVlan:
                  vlanColumn = str( vlan )
                  firstControlPlaneInVlan = False
               controlPlaneColumn = ''
               if firstSourceInControlPlane:
                  controlPlaneColumn = controlPlane
                  firstSourceInControlPlane = False
               table.newRow( vlanColumn, controlPlaneColumn, direction.direction,
                     source )
      print( table.output() )

#-------------------------------------------------------------------------------
# Config Sanity Check Models
#-------------------------------------------------------------------------------

class ConfigCheckItem( Model ):
   name = Str( help='Name of item', default='Unknown Item' )
   checkPass = Bool( help='Item check passed', default=True )
   hasWarning = Bool( help='Item contains warning', default=False )
   detail = Str( help='Detail about this item', default='' )
   _priority = Int( help='Priority for sorting', default=1 )

class ConfigCheckCategory( Model ):
   description = Str( help='Description of category', default='Unknown Category' )
   allCheckPass = Bool( help='Category check passes when all items belonging'
                             ' to the category pass their checks',
                        default=True )
   detail = Str( help='Detail about this item', default='' )
   hasWarning = Bool( help='Category contains warning', default=False )
   items = List( help='List of items in this category',
                 valueType=ConfigCheckItem )
   _priority = Int( help='Priority for sorting', default=1 )

   def priority( self ):
      return self._priority

   def priorityIs( self, priority ):
      self._priority = priority

def resultString( checkPass, warning ):
   if not checkPass:
      return 'FAIL'
   if warning:
      return 'WARN'
   return 'OK'

class ConfigSanityModel( Model ):
   _detailLevel = Enum( help='Level of detail',
                        values=( 'normal', 'brief', 'detailed' ),
                        default='normal' )
   categories = Dict( help='Map of category names to check categories',
                      keyType=str, valueType=ConfigCheckCategory )

   def _setFormatProperties( self, fmt ):
      assert isinstance( fmt, Format )
      fmt.noPadLeftIs( True )
      fmt.padLimitIs( True )
      fmt.noTrailingSpaceIs( True )

   def _getHeadingFormat( self ):
      fmt = Format( isHeading=True, border=True )
      self._setFormatProperties( fmt )
      return fmt

   def _getColumnFormats( self ):
      sideFmt = Format( justify='left', minWidth=35, maxWidth=70,
                        isHeading=False, border=False )
      self._setFormatProperties( sideFmt )
      midFmt = Format( justify='center', minWidth=9, maxWidth=9,
                       isHeading=False, border=False )
      self._setFormatProperties( midFmt )
      return ( sideFmt, midFmt )

   def _buildTableFormatter( self, printHeading=True ):
      headings = [ "Category", "Result", "Detail" ]

      headingFmt = self._getHeadingFormat()
      sideColFmt, midColFmt = self._getColumnFormats()
      tbl = TableFormatter()

      if printHeading:
         tbl.startRow( headingFmt )
         tbl.newCell( *headings )

      # format columns to our liking
      tbl.formatColumns( sideColFmt, midColFmt, sideColFmt )

      return tbl

   def _skipRender( self ):
      if not self.categories:
         return True

      if self._detailLevel in ( 'normal', 'brief' ):
         allCategoryPass = all( c.allCheckPass for
                                c in six.itervalues( self.categories ) )
         categoryWarnings = any( c.hasWarning for
                                 c in six.itervalues( self.categories ) )
         if allCategoryPass and not categoryWarnings:
            return True

      return False

   def _skipCategoryRender( self, category ):
      # if no warning or failure in 'normal' or 'brief' level, skip
      # category output. If we're in the 'detailed' level, print
      # everything
      if self._detailLevel == 'detailed':
         return False
      if not category.allCheckPass or category.hasWarning:
         return False
      return True

   def detailLevelIs( self, detailLevel ):
      self._detailLevel = detailLevel

   def render( self ):
      if self._skipRender():
         return

      tbl = self._buildTableFormatter()

      # loop through categories and items
      for category in sorted( six.itervalues( self.categories ),
                              key=attrgetter( '_priority' ) ):

         if self._skipCategoryRender( category ):
            continue

         tbl.newRow( category.description,
                     resultString( category.allCheckPass, category.hasWarning ),
                     category.detail )

         for item in sorted( category.items, key=attrgetter( '_priority' ) ):
            # if in 'normal' mode, show only category-level outputs
            # if in 'brief' mode, show category, warnings and failures
            # if in 'detailed' mode, show everything
            failOrWarn = not item.checkPass or item.hasWarning
            if ( self._detailLevel == 'normal' and failOrWarn
                 or self._detailLevel == 'detailed' ):
               tbl.newRow( '  ' + item.name,
                           resultString( item.checkPass, item.hasWarning ),
                           item.detail )
      print( tbl.output() )

class VxlanSecurityProfileModel( Model ):
   vteps = Dict( keyType=IpGenericAddress, valueType=str,
                 help="A mapping of VXLAN VTEP to its security profile" )

   def render( self ):
      table = createTable( ( 'VTEP', 'Security Profile' ) )
      vtepFormat = Format( justify='left' )
      vtepFormat.noPadLeftIs( True )
      secProfileFormat = Format( justify='right' )
      table.formatColumns( vtepFormat, secProfileFormat )
      for vtep, secProfile in six.iteritems( self.vteps ):
         table.newRow( vtep, secProfile )
      print( table.output() )

class VxlanFloodTrafficModel( Model ):
   defaultFlooding = Bool( optional=True,
         help="True if default flooding of all traffic is enabled" )
   arpFlooding = Bool( optional=True,
         help="True if flooding of ARP traffic is enabled" )
   ndFlooding = Bool( optional=True,
         help="True if flooding of ND traffic is enabled" )
   unknownMacFlooding = Bool( optional=True,
         help="True if flooding of unknown MAC traffic is enabled" )

   def render( self ):
      if self.defaultFlooding is None:
         return
      print( "Default flooding is %s" %
             ( "enabled" if self.defaultFlooding else "disabled" ) )
      if self.defaultFlooding:
         return
      trafficEnabled = ""
      if self.arpFlooding:
         trafficEnabled = "ARP"
      if self.ndFlooding:
         if trafficEnabled:
            trafficEnabled += " & IPv6 Neighbor Discovery"
         else:
            trafficEnabled += "IPv6 Neighbor Discovery"
      if trafficEnabled:
         print( "Flooding is enabled for %s" % trafficEnabled )

class VxlanVniPolicerModeEnum( Model ):
   value = Enum( values=( "profile", "group" ),
                   help="Policer mode" )

class VxlanVniPolicerHwStateEnum( Model ):
   hwState = Enum( values=( "inactive", "active" ),
                   help="Policer hardware state" )

class VxlanVniPolicer( Model ):
   name = Str( help="Policer name" )
   mode = Submodel( valueType=VxlanVniPolicerModeEnum,
                     help="Policer mode configuration" )
   status = Submodel( valueType=VxlanVniPolicerHwStateEnum,
                     help="Policer hardware programming status" )

class VxlanVniQosInfoModel( Model ):
   extVlan = Int( help="VLAN-VNI-VRF mapping", optional=True )
   iPolicerInfo = Submodel( valueType=VxlanVniPolicer,
                            help="Ingress policer information",
                            optional=True )
   oPolicerInfo = Submodel( valueType=VxlanVniPolicer,
                            help="Egress policer information",
                            optional=True )

class VxlanConfigModel( Model ):
   vniQosInfo = Dict( keyType=int, valueType=VxlanVniQosInfoModel,
                    help="A mapping of VNI to its policer information" )

class VxlanVniPolicersModel( Model ):
   vxlanConfig = Dict( keyType=str, valueType=VxlanConfigModel,
                    help="A mapping of VXLAN interface to its configuration" )

   def render( self ):

      print( 'Legend: (P) Policer Profile, (G) Group Policer, * Not active' )
      table = createTable( ( 'INTF', 'VNI', 'VLAN',
                              'Ingress Policer', 'Egress Policer' ) )

      empty = True
      for vti in sorted( self.vxlanConfig ):
         for vni in sorted( self.vxlanConfig[ vti ].vniQosInfo ):
            vlan = self.vxlanConfig[ vti ].vniQosInfo[ vni ].extVlan
            iPolicerInfo = self.vxlanConfig[ vti ].vniQosInfo[ vni ].iPolicerInfo
            oPolicerInfo = self.vxlanConfig[ vti ].vniQosInfo[ vni ].oPolicerInfo

            suffix = ""
            iPolicer = '-'
            oPolicer = '-'
            if iPolicerInfo:
               suffix = '(G)' if iPolicerInfo.mode.value == 'group' else '(P)'
               if iPolicerInfo.status.hwState == 'inactive':
                  suffix = suffix + '*'
               iPolicer = iPolicerInfo.name + suffix

            if oPolicerInfo:
               suffix = '(G)' if oPolicerInfo.mode.value == 'group' else '(P)'
               if oPolicerInfo.status.hwState == 'inactive':
                  suffix = suffix + '*'
               oPolicer = oPolicerInfo.name + suffix

            table.newRow( vti, vni, vlan, iPolicer, oPolicer )
            if empty:
               empty = False

      if empty:
         print( "No matching entries found." )
      else:
         print( table.output() )

class VniPolicingCounters( Model ):
   # Green Packets and Bytes
   conformedPackets = Int( help="Conformed number of packets to policer", default=0 )
   conformedBytes = Int( help="Conformed number of bytes to policer", default=0 )
   # Red packets and Bytes
   exceededPackets = Int( help="Exceeded number of packets to policer", default=0 )
   exceededBytes = Int( help="Exceeded number of bytes to policer", default=0 )

class VxlanVniPolicersCountersInfoModel( Model ):
   policerName = Str( help="Policer name", optional=True )
   mode = Submodel( valueType=VxlanVniPolicerModeEnum,
                    help="Policer mode configuration" )
   counters = Submodel( valueType=VniPolicingCounters, help='VNI Policer Counters' )

   def modeStr( self ):
      if self.mode.value == 'group':
         return 'Group Policer'
      return 'Policer Profile'

   def countersDict( self ):
      return {
         "greenPkts": self.counters.conformedPackets,
         "greenBytes": self.counters.conformedBytes,
         "redPkts": self.counters.exceededPackets,
         "redBytes": self.counters.exceededBytes
      }

   def render( self ):
      output = ""
      policerString = "{} ( {} )".format( self.policerName, self.modeStr() )

      output += policerString

      indent = "\n  "
      conformedStr = "Below the rate limit {greenPkts} packets, {greenBytes} bytes"
      exceededStr = "Above the rate limit {redPkts} packets, {redBytes} bytes"
      policeStr = indent + conformedStr + indent + exceededStr

      countersDict = self.countersDict()
      output += policeStr.format( **countersDict )
      print( output )

class VxlanVniPolicersAllCountersInfoModel( Model ):
   vniPolicers = Dict( keyType=int, valueType=VxlanVniPolicersCountersInfoModel,
                       help="Per VNI policer collection" )

   def render( self ):
      for vni in sorted( self.vniPolicers ):
         print( "\nVNI {}: ".format( vni ), end='' )
         self.vniPolicers[ vni ].render()

class VxlanVniPolicersCountersModel( Model ):
   ingInterfaces = Dict( keyType=Interface,
                         valueType=VxlanVniPolicersAllCountersInfoModel,
                         help="Per VXLAN interface collection for ingress policers" )
   egrInterfaces = Dict( keyType=Interface,
                         valueType=VxlanVniPolicersAllCountersInfoModel,
                         help="Per VXLAN interface collection for egress policers" )
   _policerDir = Str( help="Print VNI policer info by policer direction" )

   def policerDirIs( self, policerDir ):
      self._policerDir = policerDir

   def render( self ):
      print( "\n===== VNI Policing Counters =====" )

      if self._policerDir == 'input' or self._policerDir == 'all':
         print( '\nIngress policing' )
         for intf in sortIntf( self.ingInterfaces ):
            print( '------------------' )
            print( 'Interface {}'.format( intf ) )
            self.ingInterfaces[ intf ].render()
         print( '------------------' )

      if self._policerDir == 'output' or self._policerDir == 'all':
         print( '\nEgress policing' )
         for intf in sortIntf( self.egrInterfaces ):
            print( '------------------' )
            print( 'Interface {}'.format( intf ) )
            self.egrInterfaces[ intf ].render()
         print( '------------------' )

class VxlanEncapSourceIpRewriteModel( Model ):
   vni = Int( help="VNI" )
   sourceIp = IpGenericAddress( help="Source VTEP IP address" )

class VxlanEncapSourceIpRewriteModelList( Model ):
   rewriteInfo = List( valueType=VxlanEncapSourceIpRewriteModel,
                       help="List of Rewrite Info" )

class VxlanRewriteInfo( Model ):
   remoteVteps = Dict( optional=True, keyType=IpGenericAddress,
                       valueType=VxlanEncapSourceIpRewriteModelList,
                       help="Rewrite information per remote VTEP" )

   def render( self ):
      table = createTable( ( 'Remote VTEP', 'VNI', 'Source IP' ) )
      colFormat = Format( justify='left' )
      colFormat.noPadLeftIs( True )
      table.formatColumns( colFormat, colFormat, colFormat )
      for remoteVtep, vniSipList in six.iteritems( self.remoteVteps ):
         for vniSip in vniSipList.rewriteInfo:
            table.newRow( remoteVtep, vniSip.vni, vniSip.sourceIp )
      print( table.output() )
