#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
import CliToken
import CliPlugin.BridgingCli as BridgingCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IntfModel as IntfModel
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
import CliPlugin.LagIntfCli as LagIntfCli
import CliPlugin.MacAddr as MacAddr
import CliPlugin.MlagShowCli as MlagShowCli
import CliPlugin.VlanCli as VlanCli
import CliPlugin.VrfCli as VrfCli
import CliPlugin.VxlanCli as VxlanCli
import CliPlugin.VxlanModel as VxlanModel
import Intf.IntfRange as IntfRange
import IntfRangePlugin.SwitchIntf as SwitchIntf
import ShowCommand
import Toggles.VxlanControllerToggleLib

matcherAddress = CliMatcher.KeywordMatcher( 'address',
      helpdesc='Filter by MAC address' )
matcherArp = CliMatcher.KeywordMatcher( 'arp',
      helpdesc='ARP entries' )
matcherController = CliMatcher.KeywordMatcher( 'controller',
      helpdesc='VXLAN control service' )
matcherCounters = CliMatcher.KeywordMatcher( 'counters',
      helpdesc='VXLAN Counters' )
matcherIp = CliMatcher.KeywordMatcher( 'ip',
      helpdesc='IP address' )
matcherLearnRestrict = CliMatcher.KeywordMatcher( 'learn-restrict',
      helpdesc='VXLAN learning restrictions' )
matcherName = CliMatcher.KeywordMatcher( 'name',
      helpdesc='Specify logical router name' )
matcherNd = CliMatcher.KeywordMatcher( 'nd',
      helpdesc='Neighbor Discovery entries' )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='VLAN configuration' )
matcherVtep = CliMatcher.KeywordMatcher( 'vtep',
      helpdesc='VXLAN Tunnel End Points' )
nodeLogicalRouter = CliCommand.guardedKeyword( 'logical-router',
      helpdesc='Logical router status',
      guard=VxlanCli.vxlanRoutingSupportedGuard )
nodeVxlan = CliCommand.guardedKeyword( 'vxlan',
      helpdesc='VXLAN configuration and statistics',
      guard=VxlanCli.vxlanSupportedGuard )
arpIpMatcher = IpGenAddrMatcher.IpGenAddrMatcher(
      helpdesc='IP address (or prefix) of remote VTEP',
      helpdesc4='ARP entry\'s IPv4 address',
      helpdesc6='ARP entry\'s IPv4 address' )
logicalRouterMatcher = CliMatcher.QuotedStringMatcher(
      helpdesc='Logical router name', helpname='WORD' )
directionMatcher = CliMatcher.EnumMatcher( {
      'encap': 'Encapsulation counters',
      'decap': 'Decapsulation counters',
   } )
domainMatcher = CliMatcher.KeywordMatcher( 'domain',
      helpdesc='Domain to be shown' )

vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF name' )
matcherFlood = CliMatcher.KeywordMatcher( 'flood',
      helpdesc='VXLAN flooding behavior' )
nodePolicers = CliCommand.guardedKeyword( 'policers', helpdesc='VNI Policers',
               guard=VxlanCli.vxlanVniPolicerSupportedGuard )
matcherPolicersCounters = CliMatcher.KeywordMatcher( 'counters',
                           helpdesc='VNI Policing Counters' )
matcherPolicersDirection = CliMatcher.EnumMatcher( {
                              'input': 'Applied to incoming traffic',
                              'output': 'Applied to outgoing traffic',
                           } )
matcherPolicerVni = CliMatcher.KeywordMatcher( 'vni',
      helpdesc='VNI filter' )
nodeVniCounters = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'vni',
            helpdesc='VXLAN Network Identifier counters' ),
      guard=VxlanCli.vxlanVniCountersSupportedGuard )

#--------------------------------------------------------------------------------
# show vxlan address-table [ mlag-peer ] [ HOST_TYPE ]
#                                        [ address MAC_ADDR ]
#                                        [ vlan VLAN_ID ]
#                                        [ vtep { VTEP } ]
#--------------------------------------------------------------------------------
class VxlanAddressTableCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan address-table [ mlag-peer ] [ HOST_TYPE ] '
                                                     '[ address MAC_ADDR ] '
                                                     '[ vlan VLAN_ID ] '
                                                     '[ vtep { VTEP } ]' )
   data = {
      'vxlan': nodeVxlan,
      'address-table': BridgingCli.matcherAddressTable,
      'mlag-peer': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'mlag-peer',
            helpdesc='MAC addresses learned from MLAG peer' ),
         guard=MlagShowCli.mlagSupportedGuard ),
      'address': matcherAddress,
      'HOST_TYPE': CliMatcher.EnumMatcher( {
         'received': 'Received MAC addresses',
         'evpn': 'EVPN Received MAC addresses',
         'static': 'Static entry type',
         'dynamic': 'Dynamic entry type',
         'unicast': 'Unicast entry type',
         'configured': 'Configured MAC entries',
      } ),
      'MAC_ADDR': MacAddr.macAddrMatcher,
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'vtep': matcherVtep,
      'VTEP': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='Vtep IP address',
                                               helpdesc4="Vtep's IPv4 address",
                                               helpdesc6="Vtep's IPv6 address" ),
   }
   handler = VxlanCli.showVxlanMacAddr
   cliModel = VxlanModel.VxlanMacAddresses

BasicCli.addShowCommandClass( VxlanAddressTableCmd )

#--------------------------------------------------------------------------------
# show vxlan address-table count [ vtep { VTEP } ]
#--------------------------------------------------------------------------------
class VxlanAddressTableCountCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan address-table count [ vtep { VTEP } ]'
   data = {
      'vxlan': nodeVxlan,
      'address-table': BridgingCli.matcherAddressTable,
      'count': 'Count of MAC addresses per VTEP',
      'vtep': matcherVtep,
      'VTEP': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='Vtep IP address',
                                               helpdesc4="Vtep's IPv4 address",
                                               helpdesc6="Vtep's IPv6 address" ),
   }
   handler = VxlanCli.showVxlanMacAddrCount
   cliModel = VxlanModel.VxlanAddressTableCount

BasicCli.addShowCommandClass( VxlanAddressTableCountCmd )

#--------------------------------------------------------------------------------
# show vxlan config-sanity [ brief | detail ]
#--------------------------------------------------------------------------------
class VxlanConfigSanityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan config-sanity [ brief | detail ]'
   data = {
      'vxlan': nodeVxlan,
      'config-sanity': 'VXLAN Config Sanity',
      'brief': 'Show brief information',
      'detail': 'More comprehensive output'
   }
   handler = VxlanCli.showConfigSanity
   cliModel = VxlanModel.ConfigSanityModel

BasicCli.addShowCommandClass( VxlanConfigSanityCmd )

#--------------------------------------------------------------------------------
# show vxlan controller address-table advertised [ vlan VLAN_ID ]
#                                                [ address MAC_ADDR ]
#--------------------------------------------------------------------------------
class VxlanControllerAddressTableAdvertisedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan controller address-table advertised [ vlan VLAN_ID ] '
                                                            '[ address MAC_ADDR ]' )
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'address-table': BridgingCli.matcherAddressTable,
      'address': matcherAddress,
      'advertised': 'Advertised MAC addresses',
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'MAC_ADDR': MacAddr.macAddrMatcher,
   }
   handler = VxlanCli.showVxlanVniStatusAdvertised
   cliModel = VxlanModel.VxlanVniStatusDirModel

BasicCli.addShowCommandClass( VxlanControllerAddressTableAdvertisedCmd )

#--------------------------------------------------------------------------------
# show vxlan controller address-table received [ vlan VLAN_ID ]
#                                              [ address MAC_ADDR ] [ vtep { VTEP } ]
#--------------------------------------------------------------------------------
class VxlanControllerAddressTableReceivedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan controller address-table received [ vlan VLAN_ID ] '
                                          '[ address MAC_ADDR ] [ vtep { VTEP } ]' )
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'address-table': BridgingCli.matcherAddressTable,
      'address': matcherAddress,
      'received': 'Received MAC addresses',
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'MAC_ADDR': MacAddr.macAddrMatcher,
      'vtep': matcherVtep,
      'VTEP': IpAddrMatcher.IpAddrMatcher( helpdesc='IP address of remote VTEP' ),
   }
   handler = VxlanCli.showVxlanVniStatusReceived
   cliModel = VxlanModel.VxlanVniStatusDirModel

BasicCli.addShowCommandClass( VxlanControllerAddressTableReceivedCmd )

#--------------------------------------------------------------------------------
# show vxlan controller arp advertised [ VRF ] [ vlan VLAN_ID ] [ ip IP ]
#                                                               [ address MAC_ADDR ]
#--------------------------------------------------------------------------------
class VxlanControllerArpAdvertisedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan controller arp advertised [ VRF ] '
                                 '[ vlan VLAN_ID ] [ ip IP ] [ address MAC_ADDR ]' )
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'arp': matcherArp,
      'advertised': 'Advertised ARP entries',
      'VRF': vrfExprFactory,
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'ip': matcherIp,
      'IP': arpIpMatcher,
      'address': matcherAddress,
      'MAC_ADDR': MacAddr.macAddrMatcher,
   }
   handler = VxlanCli.showVxlanIpToMacAdvertised
   cliModel = VxlanModel.VxlanArpTableModel

BasicCli.addShowCommandClass( VxlanControllerArpAdvertisedCmd )

#--------------------------------------------------------------------------------
# show vxlan controller nd advertised [ VRF ] [ vlan VLAN_ID ] [ ip IP ]
#                                                               [ address MAC_ADDR ]
# When toggle is no longer needed, it should be merged to
# VxlanControllerArpAdvertisedCmd to reduce duplication
#--------------------------------------------------------------------------------
class VxlanControllerNdAdvertisedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan controller nd advertised [ VRF ] '
                                 '[ vlan VLAN_ID ] [ ip IP ] [ address MAC_ADDR ]' )
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'nd': matcherNd,
      'advertised': 'Advertised Neighbor Discovery entries',
      'VRF': vrfExprFactory,
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'ip': matcherIp,
      'IP': arpIpMatcher,
      'address': matcherAddress,
      'MAC_ADDR': MacAddr.macAddrMatcher,
   }
   handler = VxlanCli.showVxlanIpToMacAdvertised
   cliModel = VxlanModel.VxlanArpTableModel

if Toggles.VxlanControllerToggleLib.toggleVxlanVccNdProxySupportedEnabled():
   BasicCli.addShowCommandClass( VxlanControllerNdAdvertisedCmd )

#--------------------------------------------------------------------------------
# show vxlan controller arp received [ VRF ] [ vlan VLAN_ID ] [ ip IP ]
#                                                               [ address MAC_ADDR ]
#--------------------------------------------------------------------------------
class VxlanControllerArpReceivedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan controller arp received [ VRF ] [ vlan VLAN_ID ] '
                                                 '[ ip IP ] [ address MAC_ADDR ]' )
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'arp': matcherArp,
      'received': 'Received ARP entries',
      'VRF': vrfExprFactory,
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'ip': matcherIp,
      'IP': arpIpMatcher,
      'address': matcherAddress,
      'MAC_ADDR': MacAddr.macAddrMatcher,
   }
   handler = VxlanCli.showVxlanIpToMacReceived
   cliModel = VxlanModel.VxlanArpTableModel

BasicCli.addShowCommandClass( VxlanControllerArpReceivedCmd )

#--------------------------------------------------------------------------------
# show vxlan controller nd received [ VRF ] [ vlan VLAN_ID ] [ ip IP ]
#                                                               [ address MAC_ADDR ]
# When toggle is no longer needed, it should be merged to
# VxlanControllerNdReceivedCmd to reduce duplication
#--------------------------------------------------------------------------------
class VxlanControllerNdReceivedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan controller nd received [ VRF ] [ vlan VLAN_ID ] '
                                                '[ ip IP ] [ address MAC_ADDR ]' )
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'nd': matcherNd,
      'received': 'Received ARP entries',
      'VRF': vrfExprFactory,
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'ip': matcherIp,
      'IP': arpIpMatcher,
      'address': matcherAddress,
      'MAC_ADDR': MacAddr.macAddrMatcher,
   }
   handler = VxlanCli.showVxlanIpToMacReceived
   cliModel = VxlanModel.VxlanArpTableModel

if Toggles.VxlanControllerToggleLib.toggleVxlanVccNdProxySupportedEnabled():
   BasicCli.addShowCommandClass( VxlanControllerNdReceivedCmd )

#--------------------------------------------------------------------------------
# show vxlan controller logical-router [ { ( vni VNI ) | ( name LOGICAL_ROUTER ) } ]
#--------------------------------------------------------------------------------
class VxlanControllerLogicalRouterCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan controller logical-router '
                                    '[ { ( vni VNI ) | ( name LOGICAL_ROUTER ) } ]' )
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'logical-router': nodeLogicalRouter,
      'vni': CliCommand.singleNode( VxlanCli.vniMatcherForConfig ),
      'VNI': VxlanCli.vniMatcher,
      'name': CliCommand.singleNode( matcherName ),
      'LOGICAL_ROUTER': logicalRouterMatcher
   }
   handler = VxlanCli.showVxlanLogicalRouter
   cliModel = VxlanModel.VxlanLogicalRouterModel

BasicCli.addShowCommandClass( VxlanControllerLogicalRouterCmd )

#--------------------------------------------------------------------------------
# show vxlan controller logical-router route [ name LOGICAL_ROUTER ]
#--------------------------------------------------------------------------------
class VxlanControllerLogicalRouterRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan controller logical-router route [ name LOGICAL_ROUTER ]'
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'logical-router': nodeLogicalRouter,
      'route': 'Logical router route',
      'name': matcherName,
      'LOGICAL_ROUTER': logicalRouterMatcher
   }
   handler = VxlanCli.showVxlanRoutes
   cliModel = VxlanModel.VxlanLogicalRouterModel

BasicCli.addShowCommandClass( VxlanControllerLogicalRouterRouteCmd )

#--------------------------------------------------------------------------------
# show vxlan controller logical-router uplink [ name LOGICAL_ROUTER ]
#--------------------------------------------------------------------------------
class VxlanControllerLogicalRouterUplinkCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan controller logical-router uplink [ name LOGICAL_ROUTER ]'
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'logical-router': nodeLogicalRouter,
      'uplink': 'Logical router uplink ports',
      'name': matcherName,
      'LOGICAL_ROUTER': logicalRouterMatcher
   }
   handler = VxlanCli.showVxlanUplinkPorts
   cliModel = VxlanModel.VxlanLogicalRouterModel

BasicCli.addShowCommandClass( VxlanControllerLogicalRouterUplinkCmd )

#--------------------------------------------------------------------------------
# show vxlan controller status
#--------------------------------------------------------------------------------
class VxlanControllerStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan controller status'
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'status': 'VXLAN control service status',
   }
   handler = VxlanCli.showVxlanControlServiceStatus
   cliModel = VxlanModel.VxlanControlServiceStatus

BasicCli.addShowCommandClass( VxlanControllerStatusCmd )

#--------------------------------------------------------------------------------
# show vxlan counters learn-restrict COUNTER [ vlan VLANS ]
#--------------------------------------------------------------------------------
class VxlanCountersLearnRestrictAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan counters learn-restrict COUNTER [ vlan VLANS ]'
   data = {
      'vxlan': nodeVxlan,
      'counters': matcherCounters,
      'learn-restrict': matcherLearnRestrict,
      'COUNTER': CliMatcher.EnumMatcher( {
         'all': 'Show all counters',
         'brief': 'Show brief information',
         'prefix': 'Match a prefix',
      } ),
      'vlan': matcherVlan,
      'VLANS': VlanCli.vlanSetMatcher,
   }
   handler = VxlanCli.showLearnCounters
   cliModel = VxlanModel.VxlanLearnCountersModel

BasicCli.addShowCommandClass( VxlanCountersLearnRestrictAllCmd )

#--------------------------------------------------------------------------------
# show vxlan flood vtep [ vlan VLANS ]
#--------------------------------------------------------------------------------
class VxlanFloodVtepCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan flood vtep [ vlan VLANS ] [ domain DOMAIN ]'
   data = {
      'vxlan': nodeVxlan,
      'flood': matcherFlood,
      'vtep': matcherVtep,
      'vlan': matcherVlan,
      'VLANS': VlanCli.vlanSetMatcher,
      'domain': domainMatcher,
      'DOMAIN': CliMatcher.EnumMatcher( {
         'local': 'Local domain',
         'remote': 'Remote domain',
      } ),
   }
   handler = VxlanCli.showFloodVtep
   cliModel = VxlanModel.VxlanFloodVtepListModel

BasicCli.addShowCommandClass( VxlanFloodVtepCmd )

#--------------------------------------------------------------------------------
# show vxlan learn-restrict vtep [ vlan VLANS ]
#--------------------------------------------------------------------------------
class VxlanLearnRestrictVtepCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan learn-restrict vtep [ vlan VLANS ]'
   data = {
      'vxlan': nodeVxlan,
      'learn-restrict': matcherLearnRestrict,
      'vtep': matcherVtep,
      'vlan': matcherVlan,
      'VLANS': VlanCli.vlanSetMatcher,
   }
   handler = VxlanCli.showLearnRestrict
   cliModel = VxlanModel.VxlanLearnRestrictModel

BasicCli.addShowCommandClass( VxlanLearnRestrictVtepCmd )

#--------------------------------------------------------------------------------
# show vxlan counters software
#--------------------------------------------------------------------------------
class VxlanCountersSoftwareCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan counters software'
   data = {
      'vxlan': nodeVxlan,
      'counters': matcherCounters,
      'software': 'Software forwarding counters',
   }
   handler = VxlanCli.showSwCtrs
   cliModel = VxlanModel.VxlanSwCounters

BasicCli.addShowCommandClass( VxlanCountersSoftwareCmd )

#--------------------------------------------------------------------------------
# show vxlan counters varp
#--------------------------------------------------------------------------------
class VxlanCountersVarpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan counters varp'
   data = {
      'vxlan': nodeVxlan,
      'counters': matcherCounters,
      'varp': 'Virtual ARP',
   }
   handler = VxlanCli.showVarpCounters
   cliModel = VxlanModel.VxlanVarpCounters

BasicCli.addShowCommandClass( VxlanCountersVarpCmd )

#--------------------------------------------------------------------------------
# show vxlan counters vni [ VNI ] [ DIRECTION ]
#--------------------------------------------------------------------------------
class VxlanCountersVniCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan counters vni [ VNI ] [ DIRECTION ]'
   data = {
      'vxlan': nodeVxlan,
      'counters': matcherCounters,
      'vni': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'vni', helpdesc='VNI counters' ),
         guard=VxlanCli.vniCountersSupported ),
      'VNI': VxlanCli.vniMatcher,
      'DIRECTION': CliCommand.Node(
          matcher=directionMatcher,
          guard=VxlanCli.vniCountersDirectionSupported ),
   }
   handler = VxlanCli.showVniCounters
   cliModel = VxlanModel.VxlanVniCountersModel

BasicCli.addShowCommandClass( VxlanCountersVniCmd )

#--------------------------------------------------------------------------------
# show vxlan counters vtep [ unlearnt | IP ] [ DIRECTION ]
#--------------------------------------------------------------------------------
class VxlanCountersVtepCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan counters vtep [ unlearnt | IP ] [ DIRECTION ]'
   data = {
      'vxlan': nodeVxlan,
      'counters': matcherCounters,
      'vtep': CliCommand.guardedKeyword( 'vtep',
                                         helpdesc='VTEP counters',
                                         guard=VxlanCli.vtepCountersSupported ),
      'unlearnt': 'Unlearnt VTEPs',
      'IP': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='VTEP IP address',
                                               helpdesc4="VTEP's IPv4 address",
                                               helpdesc6="VTEP's IPv6 address" ),
      'DIRECTION': CliCommand.Node(
         matcher=directionMatcher,
         guard=VxlanCli.vtepCountersDirectionFilterSupported )
   }
   handler = VxlanCli.showVtepCounters
   cliModel = VxlanModel.VxlanVtepCountersModel

BasicCli.addShowCommandClass( VxlanCountersVtepCmd )

#--------------------------------------------------------------------------------
# show vxlan counters vtep [ IP ] vni [ VNI ]
#--------------------------------------------------------------------------------
class VxlanCountersVtepVniCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan counters vtep [ IP ] vni [ VNI ]'
   data = {
      'vxlan': nodeVxlan,
      'counters': matcherCounters,
      'vtep': CliCommand.guardedKeyword( 'vtep',
                                         helpdesc='VTEP counters',
                                         guard=VxlanCli.vtepCountersSupported ),
      'IP': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='VTEP IP address',
                                               helpdesc4="VTEP's IPv4 address",
                                               helpdesc6="VTEP's IPv6 address" ),
      'vni': CliCommand.guardedKeyword( 'vni',
                                         helpdesc='VNI-VTEP counters',
                                         guard=VxlanCli.vtepVniCountersSupported ),
      'VNI': VxlanCli.vniMatcher
   }
   handler = VxlanCli.showVtepVniCounters
   cliModel = VxlanModel.VxlanVtepVniCountersModel

BasicCli.addShowCommandClass( VxlanCountersVtepVniCmd )

#--------------------------------------------------------------------------------
# show vxlan qos
#--------------------------------------------------------------------------------
class VxlanQosCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan qos'
   data = {
      'vxlan': nodeVxlan,
      'qos': 'Qos settings',
   }
   handler = VxlanCli.showVxlanQos
   cliModel = VxlanModel.VxlanQosModel

BasicCli.addShowCommandClass( VxlanQosCmd )

#--------------------------------------------------------------------------------
# show vxlan vni [ VNI ] [ { ( ( interface INTFS ) |
#                              ( source SOURCE ) |
#                              ( dot1q ( untagged | VLANS ) ) ) } ]
#--------------------------------------------------------------------------------
matcherVni = CliMatcher.KeywordMatcher( 'vni', helpdesc='VNI to VLAN mapping' )

class VxlanVniCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show vxlan vni [ VNI ]
               [ { ( interface INTFS )
                 | ( source SOURCE )
                 | ( dot1q ( untagged | VLANS ) ) } ]'''
   data = {
      'vxlan': nodeVxlan,
      'vni': matcherVni,
      'VNI': VxlanCli.vniMatcher,
      'interface': CliCommand.singleKeyword( 'interface',
         helpdesc='Interface selector' ),
      'INTFS': IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType,
                             LagIntfCli.LagAutoIntfType,
                             SwitchIntf.SwitchAutoIntfType ) ),
      'source': CliCommand.singleKeyword( 'source', helpdesc='Source filter' ),
      'SOURCE': CliMatcher.PatternMatcher( pattern='\\S+',
         helpdesc='source of VNI to VLAN binding', helpname='WORD' ),
      'dot1q': CliCommand.singleKeyword( 'dot1q', helpdesc='802.1Q Tag filter' ),
      'untagged': 'Untagged packets',
      'VLANS': VlanCli.vlanSetMatcher,
   }
   handler = VxlanCli.showVxlanVni
   cliModel = VxlanModel.VxlanVniModel

BasicCli.addShowCommandClass( VxlanVniCmd )

#--------------------------------------------------------------------------------
# show vxlan vni summary
#--------------------------------------------------------------------------------
class VxlanVniSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan vni summary' )
   data = {
      'vxlan': nodeVxlan,
      'vni': matcherVni,
      'summary': 'VXLAN VNI summary'
   }
   handler = VxlanCli.showVxlanVniSummary
   cliModel = VxlanModel.VxlanVniSummaryModel

BasicCli.addShowCommandClass( VxlanVniSummaryCmd )

#--------------------------------------------------------------------------------
# show vxlan vtep [ ( type TUNNELTYPE ) ] [ detail ]
#--------------------------------------------------------------------------------
class VxlanVtepCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan vtep [ ( type TUNNEL_TYPE ) ] [ detail ]' )
   data = {
      'vxlan': nodeVxlan,
      'vtep': matcherVtep,
      'type': CliCommand.singleKeyword( 'type', helpdesc='Tunnel type filter' ),
      'TUNNEL_TYPE': CliMatcher.EnumMatcher( VxlanModel.vxlanTunnelTypes ),
      'detail': CliCommand.singleKeyword( 'detail',
                                           helpdesc='Detailed VTEP information' ),
   }
   handler = VxlanCli.showVxlanVteps
   cliModel = VxlanModel.VxlanVtepsModel

class VxlanVtepSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan vtep summary'
   data = {
      'vxlan': nodeVxlan,
      'vtep': matcherVtep,
      'summary': 'VXLAN VTEP summary',
   }
   handler = VxlanCli.showVxlanVtepsSummary
   cliModel = VxlanModel.VxlanVtepsSummaryModel

BasicCli.addShowCommandClass( VxlanVtepCmd )
BasicCli.addShowCommandClass( VxlanVtepSummaryCmd )

#--------------------------------------------------------------------------------
# show vxlan security profile
#--------------------------------------------------------------------------------
class VxlanSecurityProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan security profile'
   data = {
      'vxlan': nodeVxlan,
      'flood': matcherFlood,
      'security': 'Security related configuration',
      'profile': 'Configured security profiles',
   }
   handler = VxlanCli.showVxlanSecurityProfile
   cliModel = VxlanModel.VxlanSecurityProfileModel

BasicCli.addShowCommandClass( VxlanSecurityProfileCmd )

#--------------------------------------------------------------------------------
# show vxlan flood traffic
#--------------------------------------------------------------------------------
class VxlanFloodTrafficCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan flood traffic'
   data = {
      'vxlan': nodeVxlan,
      'flood': matcherFlood,
      'traffic': 'Show flooding behavior for various traffic'
   }
   handler = VxlanCli.showVxlanFloodTraffic
   cliModel = VxlanModel.VxlanFloodTrafficModel

BasicCli.addShowCommandClass( VxlanFloodTrafficCmd )

# --------------------------------------------------------------------------------
# show vxlan policers vni [ VNI ] [ interface INTF ]
# --------------------------------------------------------------------------------
class VxlanVniPolicersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan policers vni [ VNI ] [ interface INTF ]'
   data = {
      'vxlan': nodeVxlan,
      'policers': nodePolicers,
      'vni': matcherPolicerVni,
      'VNI': VxlanCli.vniMatcher,
      'interface': BridgingCli.matcherInterface,
      'INTF': VxlanCli.VxlanIntf.matcher,
   }
   handler = VxlanCli.showVxlanVniPolicers
   cliModel = VxlanModel.VxlanVniPolicersModel

BasicCli.addShowCommandClass( VxlanVniPolicersCmd )

# --------------------------------------------------------------------------------
# show vxlan policers vni [ VNI ] [ interface VXLANINTF ] counters
# --------------------------------------------------------------------------------
class VxlanVniPolicersCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show vxlan policers vni
               [ VNI ] [ interface VXLANINTF ] counters [ DIR ]'''
   data = {
      'vxlan': nodeVxlan,
      'policers': nodePolicers,
      'vni': matcherPolicerVni,
      'VNI': VxlanCli.vniMatcher,
      'interface': BridgingCli.matcherInterface,
      'VXLANINTF': VxlanCli.VxlanIntf.matcher,
      'counters': matcherPolicersCounters,
      'DIR': matcherPolicersDirection,
   }
   handler = VxlanCli.showVxlanVniPolicersCounters
   cliModel = VxlanModel.VxlanVniPolicersCountersModel

BasicCli.addShowCommandClass( VxlanVniPolicersCountersCmd )

# -----------------------------------------------
# show vxlan vtep [ VTEP ] encapsulation
# aid/13978 has more details
# ----------------------------------------------
class VxlanSourceIpRewriteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show vxlan vtep [ VTEP ] encapsulation'''
   data = {
      'vxlan': nodeVxlan,
      'vtep': matcherVtep,
      'VTEP': CliCommand.Node(
         IpAddrMatcher.IpAddrMatcher( helpdesc='IP address of remote VTEP' ),
         guard=VxlanCli.vxlanSourceIpRewriteSupportedShowGuard ),
      'encapsulation': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'encapsulation',
            helpdesc='Show VXLAN encapsulation parameters' ),
         guard=VxlanCli.vxlanSourceIpRewriteSupportedShowGuard ),
   }
   handler = VxlanCli.showVxlanSourceIpRewrite
   cliModel = VxlanModel.VxlanRewriteInfo

BasicCli.addShowCommandClass( VxlanSourceIpRewriteCmd )

# -------------------------------------------------------------------------------
# The "show interfaces [Vxlan<1-7>] counters vni" command, in enable mode.
#  show interfaces [Vxlan<1-7>] counters vni [<VNI>]
#  show interfaces [Vxlan<1-7>] counters vni [<VNI>] incoming
#  show interfaces [Vxlan<1-7>] counters vni [<VNI>] outgoing
#  show interfaces [Vxlan<1-7>] counters vni [<VNI>] delta
#  show interfaces [Vxlan<1-7>] counters vni [<VNI>] delta incoming
#  show interfaces [Vxlan<1-7>] counters vni [<VNI>] delta outgoing
# -------------------------------------------------------------------------------
class ShowIntfVxlanVniCounters( ShowCommand.ShowCliCommandClass ):
   syntax = '''show interfaces [ INTFS ] counters vni [ VNI ]
               [ delta ] [ incoming | outgoing ]'''
   data = dict( interfaces=IntfCli.interfacesShowKw,
                INTFS=IntfRange.IntfRangeMatcher(
                     explicitIntfTypes=VxlanCli.vxlanIntfType ),
                counters=IntfCli.countersKw,
                vni=nodeVniCounters,
                VNI=VxlanCli.vniMatcher,
                delta='Interface counters delta',
                incoming='Incoming counters',
                outgoing='Outgoing counters' )
   cliModel = IntfModel.VirtualInterfacesCountersBase
   handler = VxlanCli.showVxlanVniCounters

BasicCli.addShowCommandClass( ShowIntfVxlanVniCounters )

# -------------------------------------------------------------------------------
# The "show interfaces [ INTFS ] counters vni [ VNI ] rates" command.
#   show interfaces counters vni rates
#   show interfaces INTFS counters rates vni rates
#   show interfaces INTFS counters rates vni VNI rates
# -------------------------------------------------------------------------------
class ShowIntfVxlanVniCountersRates( ShowCommand.ShowCliCommandClass ):
   syntax = "show interfaces [ INTFS ] counters vni [ VNI ] rates"
   data = dict( interfaces=IntfCli.interfacesShowKw,
                INTFS=IntfRange.IntfRangeMatcher(
                     explicitIntfTypes=VxlanCli.vxlanIntfType ),
                counters=IntfCli.countersKw,
                vni=nodeVniCounters,
                VNI=VxlanCli.vniMatcher,
                rates=IntfCli.countersRatesKw )
   cliModel = IntfModel.InterfaceCountersRates
   handler = VxlanCli.showVxlanVniCountersRates

BasicCli.addShowCommandClass( ShowIntfVxlanVniCountersRates )

# -------------------------------------------------------------------------------
# The "clear counters [interface] vni" command, in "privileged exec" mode.
# -------------------------------------------------------------------------------
class ClearVxlanVniCountersCmd( CliCommand.CliCommandClass ):
   syntax = "clear counters INTFS vni VNI [ session ]"
   data = dict( clear=CliToken.Clear.clearKwNode,
                counters=IntfCli.countersKw,
                INTFS=IntfRange.IntfRangeMatcher(
                     explicitIntfTypes=VxlanCli.vxlanIntfType ),
                vni=nodeVniCounters,
                VNI=VxlanCli.vniMatcher,
                session=IntfCli.counterSessionKw )

   @staticmethod
   def handler( mode, args ):
      intfs = args.get( 'INTFS' )
      vni = args.get( 'VNI' )
      VxlanCli.clearVxlanVniCounters( mode, intfs, vni )

BasicCli.EnableMode.addCommandClass( ClearVxlanVniCountersCmd )
