# Copyright (c) 2006-2008, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher
import CliCommand
import Logging
import Tac
import Tracing
import WaitForCleanup

t0 = Tracing.trace0
t1 = Tracing.trace1

# The default parameters of doWaitForCleanup are suited as per the need of cli
# wfc command.
# Return True if succeeded, else raise error. If however, ignoreTimeout is
# True, then instead of raising, return False on timeout errors.
def doWaitForCleanup( mode, agentList=None, verbose='concise', numIterations=1,
                     agentsToGrab=None, agentsToSkip=None,
                     ignoreTimeout=False, timeout=500.0, warnAfter=1.0,
                     maxDelay=1.0 ):
   # This is a thin wrapper around WaitForCleanup.wait().  The reason this lives
   # in the Cli package instead of in Sysdb is that we can't have Sysdb depend
   # on Cli.

   # if we are standalone, don't wait
   if mode.session_.isStandalone():
      return True

   for _ in range( numIterations ):
      try:
         WaitForCleanup.wait( mode.entityManager,
                             agentList=agentList,
                             agentsToGrab=agentsToGrab,
                             agentsToSkip=agentsToSkip,
                             sleep=not Tac.activityManager.inExecTime.isZero,
                             verbose=verbose, timeout=timeout,
                             maxDelay=maxDelay, warnAfter=warnAfter )
      except Tac.Timeout:
         if not ignoreTimeout:
            raise
         return False
   return True

agentListMatcher = CliMatcher.PatternMatcher( '[^/]+', helpname='WORD',
                                              helpdesc='Agent name' )

class WaitForCleanupCmd( CliCommand.CliCommandClass ):
   syntax = '''wait-for-cleanup [ { <agentList> } ] '''
   data = { 'wait-for-cleanup': 'Block until agents declare themselves as clean',
            '<agentList>': agentListMatcher }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      agents = args.get( '<agentList>' )
      clean = doWaitForCleanup( mode, agentList=agents, ignoreTimeout=True )
      if not clean:
         mode.addError( 'Command timed out' )

BasicCli.EnableMode.addCommandClass( WaitForCleanupCmd )

SYS_SYSTEM_CLEANUP_FAILED = Logging.LogHandle(
   "SYS_SYSTEM_CLEANUP_FAILED",
   severity=Logging.logNotice,
   fmt="System Cleanup after cli configuration by %s on %s (%s) failed",
   explanation="The system is taking a long time to clean up after the last "
               " configuration request from the Cli. This may mean an unusually "
               "large configuration change was requested from the Cli, which "
               "resulted in prolonged processing by the switching software. ",
   recommendedAction="No action is needed. However avoid loading the switch with "
               "large amount of configuration changes in quick succession. " )

def Plugin( entMan ):
   WaitForCleanup.doMounts( entMan )
