#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliGlobal
import CliParser
import LazyMount
import Tac

vtiStatusDirPath = 'interface/status/eth/vxlan'
avtCliConfigPath = 'avt/input/cli'

gv = CliGlobal.CliGlobal( dict( dpsHwCapability=None,
                                vtiStatusDir=None,
                                avtCliConfig=None ) )

def dpsSupported():
   if gv.dpsHwCapability and gv.dpsHwCapability.supported:
      return True
   return False

def pathSelectionSupportedGuard( mode, token ):
   if dpsSupported():
      return None
   return CliParser.guardNotThisPlatform

def getVrfNames( mode=None ):
   return gv.avtCliConfig.vrfConfig

def getAvtNames( mode=None ):
   return gv.avtCliConfig.avtProfile

def getAppNames( mode=None ):
   return ( app.appProfileName
            for config in gv.avtCliConfig.policyConfig.values()
            for app in config.appProfilePolicyRuleList.values() )

def getVniAvtIdImpl( vtiStatusDir, avtCliConfig, vrfName, avtName, verifyVni=True ):
   if not vrfName:
      return None, None, None

   # Get configured VNI for the given vrfName.
   vni = None
   if vtiStatusDir and vtiStatusDir.vtiStatus:
      vtiStatus = next( iter( vtiStatusDir.vtiStatus.values() ) )
      vni = vtiStatus.vrfToVniMap.get( vrfName )

   if ( verifyVni and ( vni is None or
                        vni is Tac.Type( "Vxlan::VniExtOrNone" ).invalidVni ) ):
      errStr = "VNI for VRF %s is not found" % vrfName
      return None, None, errStr

   avtVrfConfig = avtCliConfig.vrfConfig.get( vrfName )
   if avtVrfConfig is None:
      errStr = "Adaptive virtual topology is not enabled for VRF %s" % vrfName
      return vni, None, errStr

   if not avtName:
      return vni, None, None

   avtId = avtVrfConfig.avtNameToId.get( avtName )
   if avtId is None:
      errStr = "Adaptive virtual topology %s is not defined in VRF %s" % \
                ( avtName, vrfName )
      return vni, None, errStr

   return vni, avtId, None

def getVniAvtId( mode, vrfName, avtName, verifyVni=True ):
   vni, avt, err = getVniAvtIdImpl( gv.vtiStatusDir, gv.avtCliConfig,
                                    vrfName, avtName, verifyVni=verifyVni )
   if err:
      mode.addError( err )
   return vni, avt

def getAvtName( vrfName, inAvtId ):
   retAvtName = ''
   avtVrfConfig = gv.avtCliConfig.vrfConfig.get( vrfName )
   if avtVrfConfig:
      for avtName, avtId in avtVrfConfig.avtNameToId.items():
         if avtId == inAvtId:
            retAvtName = avtName
            break
   return retAvtName

def getVrfAvtName( inVniId, inAvtId ):
   retVrfName, retAvtName = '', ''

   if gv.vtiStatusDir and gv.vtiStatusDir.vtiStatus:
      vtiStatus = next( iter( gv.vtiStatusDir.vtiStatus.values() ) )
      for vrfName, vniId in vtiStatus.vrfToVniMap.items():
         if vniId == inVniId:
            retVrfName = vrfName
            break

   if retVrfName:
      retAvtName = getAvtName( retVrfName, inAvtId )

   return retVrfName, retAvtName

def getVniAvtIdUsingPyClient( client, vrfName, avtName, verifyVni=True,
                              printError=True ):
   entity = client.agentRoot().entity
   vtiStatusDir = entity[ vtiStatusDirPath ]
   avtCliConfig = entity[ avtCliConfigPath ]
   vni, avt, err = getVniAvtIdImpl( vtiStatusDir, avtCliConfig,
                                     vrfName, avtName, verifyVni=verifyVni )
   if err and printError:
      print( err )
   return vni, avt, err

def avtSupportedGuard( mode, token ):
   if dpsSupported():
      return None
   return CliParser.guardNotThisPlatform

def getSelfVtepImpl( vtiStatusDir ):
   vtiStatus = next( iter( vtiStatusDir.vtiStatus.values() ) )
   return vtiStatus.localVtepAddr

def getSelfVtep():
   return getSelfVtepImpl( gv.vtiStatusDir )

def getSelfVtepUsingPyClient( client ):
   entity = client.agentRoot().entity
   vtiStatusDir = entity[ vtiStatusDirPath ]
   return getSelfVtepImpl( vtiStatusDir )

def Plugin( em ):
   gv.dpsHwCapability = LazyMount.mount( em, 'dps/hwCapability',
                                         'Dps::HwCapability', 'r' )
   gv.vtiStatusDir = LazyMount.mount( em, vtiStatusDirPath,
                                      'Vxlan::VtiStatusDir', 'r' )
   gv.avtCliConfig = LazyMount.mount( em, avtCliConfigPath,
                                      'Avt::AvtCliConfig', 'r' )
