# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict
import re

import Cell
import CliGlobal
from CliPlugin.EthIntfCli import getAllIntfs, EthPhyIntf
from CliPlugin.FabricIntfCli import FabricIntf
import CliPlugin.XcvrAllStatusDir
import EthIntfLib
import ProductAttributes
import LazyMount
import MultiRangeRule
import Tac
from TypeFuture import TacLazyType
from XcvrLib import ( getXcvrSlotName,
                      getLineSideChannels,
                      isSingleLaneTransceiver )

LinkMode = TacLazyType( 'Interface::EthLinkMode' )
EthSpeed = TacLazyType( 'Interface::EthSpeed' )
EthLaneCount = TacLazyType( 'Interface::EthLaneCount' )
ethTypes = TacLazyType( 'Interface::EthTypesApi' )
XcvrType = TacLazyType( 'Xcvr::XcvrType' )
XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )
ClientConfig = TacLazyType(
   'Cfp2DcoHal::Cfp2DcoReg::GeneralModeControlLsb::ClientConfig' )

gv = CliGlobal.CliGlobal( xcvrStatusDir_=None,
                          entityManager=None,
                          lineSystemStatusSliceDir=None )

RS_FEC = 'reed-solomon'
RS544_BLOCK_SIZE = 544
RS528_BLOCK_SIZE = 528
# Map of CFP2DP04-DCO client configuration modes to error correction encoding,
# fec block size, and number of participating host lanes
generalModeControlToLabel = {
   ClientConfig.mode400Gaui8: ( RS_FEC, RS544_BLOCK_SIZE, 8 ),
   ClientConfig.mode200Gaui4: ( RS_FEC, RS544_BLOCK_SIZE, 4 ),
   ClientConfig.mode100Gaui2: ( RS_FEC, RS544_BLOCK_SIZE, 2 ),
   ClientConfig.mode100Caui4RsFec: ( RS_FEC, RS528_BLOCK_SIZE, 4 ),
   ClientConfig.mode100Caui4NoFec: ( 'none', None, 4 ),
}

def getAuxiliarySlots() -> list:
   """
   Returns auxiliary statuses.
   """
   return [ status for status in gv.xcvrStatusDir_.xcvrStatus if "Auxiliary" in
            status ]

def getAuxiliaryRenamedAsEthernet() -> list:
   """
   Returns auxiliary statuses with the Ethernet-name counterparts
   """
   auxStatuses = getAuxiliarySlots()
   return [ re.sub( "Auxiliary", "Ethernet", slotName ) for slotName in auxStatuses ]

# Helper function. Returns a tuple consisting of a list of interfaces in the range,
# and their corresponding names
def getAllIntfsWrapper( mode, intf: MultiRangeRule.IntfList, mod,
                        includeAuxiliary=False ) -> tuple[ MultiRangeRule.IntfList,
                                                           list[ str ] ]:
   auxSlotsInIntfRange = []
   if includeAuxiliary:
      # To allow auxiliary slots, we interfere before getAllIntfs is called.
      # Auxiliary slots do not have ethernet interfaces, and the ethernet-based CLI
      # infra is unaware of them. This forces us to do some extra filtering to
      # our CLIs work regardless of the requested port range!
      #
      # We need to:
      # 1) Find & include auxiliary slots in returned intfNames collection
      # 2) Remove auxiliary-slot intfs from the intfList before deferring to
      #    getAllIntfs. Otherwise an error message will be printed.
      auxSlots = getAuxiliaryRenamedAsEthernet()
      auxIrPaths = [ irPath for irPath in intf.irPathList_ if
                     f"Ethernet{irPath.sequence()[ 0 ]}" in auxSlots ]
      for irPath in auxIrPaths:
         # Include auxiliary slots if they are specified in requested range
         auxSlotsInIntfRange.append( f"Ethernet{irPath}" )
         # Remove intfs from the IntfList if they belong to an auxiliary slot
         intf.irPathList_.remove( irPath )

      if len( intf.irPathList_ ) == 0:
         # Return early if user is requesting ONLY auxiliary slots. getAllIntfs
         # errors if it's called with a list of no 'real' interfaces.
         return ( None, auxSlotsInIntfRange )

   intfType = ( EthPhyIntf, FabricIntf )
   intfs = getAllIntfs( mode, intf, mod, intfType,
                        exposeInactive=True, exposeUnconnected=True,
                        exposeInternal=True, )

   if not intfs:
      return ( None, None )
   intfNames = [ intf.name for intf in intfs ]
   return ( intfs, intfNames + auxSlotsInIntfRange )

def isDualLaserModule( status ) -> bool:
   if hasattr( status, "cfp2DcoNewStatus" ) and status.cfp2DcoNewStatus:
      return status.cfp2DcoNewStatus.dcoType == "cfp2DcoTypeDp04"
   return False

def isModular() -> bool:
   """
   Helper function. Returns True if run on a modular, False otherwise
   """
   return Cell.cellType() == "supervisor"

def getLineSystemStatus( xcvrStatus ):
   """
   Helper function. Returns LineSystemStatus corresponding to passed-in xcvrStatus.
   If no associated LineSystemStatus is found, return None.
   """
   sliceName = "FixedSystem"
   if isModular():
      sliceName = EthIntfLib.sliceName( xcvrStatus.name )
   lsStatusDir = gv.lineSystemStatusSliceDir[ sliceName ]
   lsStatus = lsStatusDir.status.get( re.sub( 'Ethernet', 'Port',
                                              xcvrStatus.name ) )
   return lsStatus

class IntfMapping():
   """
   Class that populates and contain the data necessary to display interfaces
   grouped together based on their current configuration. For example, given a
   100GBASE-LR4 QSFP module in a QSFP port configured to 50G-2 on /1 and /3, we
   would want to display /1 and /2 together, and display /3 and /4 together.

   Attributes:
      intfMap (dict):
         Dictionary (of the type defaultdict( dict )) that will contain the
         final port mapping.The port map ultimately contains the list optical
         channels for each interface that correspond to some interface.
         For example, in a general case of QSFP the port map might resemble:
            { 'Ethernet1/1' : { 'Ethernet1/1' : [ 0, 1, 2, 3 ] } }
         For more obscure cases, we can link interfaces to other interfaces.
         This might be done for Agile or CloudsRest2, and might look like:
            { 'Ethernet1' : { 'Ethernet1' : [ 0 ], 'Ethernet3' : [ 0 ],
                              'Ethernet5' : [ 0 ], 'Ethernet7' : [ 0 ] } }
         This means that we would print information for channel 0 of Et1 & Et3
         if the user requested info pertaining to Et1.
      intfNameOverride (dict):
         Dictionary (of the type defaultdict( dict )) that will override any
         interface names when they are being displayed.
      intfNameOverrideXcvrDom (dict):
         Same as intfNameOverride, except unique to the command 'show int
         transceiver dom'.
      portLaneMapOverride (dict):
         Dictionary (of the type defaultdict( dict )) that will override the
         lane mapping of a port. Sometimes a xcvr will only have one set of dom
         info. In these cases we want to show this info unassociated with any
         particular lane.
   """
   def __init__( self, requestedIntfs ):
      # Create empty mapping objects
      self.reqIntfs = set()
      self.intfMap = defaultdict( dict )
      self.hostMap = defaultdict( dict )

      self.xcvrStatus = gv.xcvrStatusDir_.xcvrStatus
      if not self.xcvrStatus:
         return

      self.reqIntfs = { i.name for i in requestedIntfs }
      self.reqPorts = { getXcvrSlotName( i ) for i in self.reqIntfs }

      # Sometimes the lane stats don't correctly correspond to the lane mapping
      # Stores a list of interfaces and the lane to display stats for, or use
      # the key 'xcvr' to indicate lane info for the xcvr as a whole.
      self.portLaneMapOverride = defaultdict( dict )

      # Affects the interface label. Ex: "Interface1" -> "Interface1 (channel 0)"
      self.intfNameOverride = {}
      # Override the interface label for "show interfaces transceiver dom"
      self.intfNameOverrideXcvrDom = {}

      ethIntfStatusDir = LazyMount.mount( gv.entityManager,
                                          'interface/status/eth/intf',
                                          'Interface::EthIntfStatusDir', 'r' )
      # EthIntfStatusDir throws an exception when an invalid interface name is
      # passed as a key. We just want to use this as a dictionary without this
      # bonus feature, so we convert it to a dict.
      self.epis = dict( ethIntfStatusDir.intfStatus )

      # Populate generic lists and dictionaries used by the intf mapping functions
      self.intfNamesDict = defaultdict( dict )
      self.linkModeStatusDict = {}
      self.anegLanes = {}
      self.anegSpeed = {}
      self.xcvrTypesDict = {}
      self.activeIntfs = []
      self.xcvrsPresent = []
      self.electricalLanesDict = {}
      self.opticalChannelsDict = {}
      self.xcvrTypesDict = {}
      self.xcvrMediaTypesDict = {}

      self.getCoherent = False
      self.modulationDict = {}

      # On Dropbear products we support Agile ports for 40G. The following product
      # attribute is a way to know whether we are on Dropbear or not and if so, we
      # enable the agile port aggregator to correctly display xvcr status on
      # subsumed interfaces as well.
      pa = ProductAttributes.productAttributes().productAttributes
      self.getAgile = pa.agentXcvrUsesResolvedConfig

      self.polsSlots = []

      self._populateInternalData()
      self._populateInternalDataCoherent()
      self._populateInternalDataPols()

      self._getIntfMap()

   def _populateInternalData( self ):
      for slotName, status in self.xcvrStatus.items():
         self.xcvrTypesDict[ slotName ] = status.xcvrType
         self.xcvrMediaTypesDict[ slotName ] = status.mediaType
         self.opticalChannelsDict[ slotName ] = getLineSideChannels( status )
         self.electricalLanesDict[ slotName ] = status.capabilities.maxChannels
         if status.presence == 'xcvrPresent':
            self.xcvrsPresent.append( slotName )
         for lane, intfName in status.xcvrConfig.intfName.items():
            if intfName not in self.epis:
               continue
            self.intfNamesDict[ slotName ][ lane ] = intfName
            lms = self.epis[ intfName ].linkModeStatus
            self.linkModeStatusDict[ intfName ] = lms
            anegLanes = self.epis[ intfName ].autonegStatus.laneCount
            self.anegLanes[ intfName ] = anegLanes
            anegSpeed = self.epis[ intfName ].autonegStatus.speed
            self.anegSpeed[ intfName ] = anegSpeed
            if self.epis[ intfName ].active:
               self.activeIntfs.append( intfName )

   def _populateInternalDataCoherent( self ):
      coherentSlices = []
      # This path exists on systems even if they don't support coherent optics
      coherentDir = LazyMount.mount( gv.entityManager, 'hardware/phy/status/data/'
                                     'coherent', 'Tac::Dir', 'ri' )
      coherentSliceDir = coherentDir.get( 'slice' )
      if coherentSliceDir:
         self.getCoherent = True
         coherentSlices = list( coherentSliceDir.values() )
      for coherentStatusDir in coherentSlices:
         phyCoherentStatus = coherentStatusDir.phyCoherentStatus
         for intfName in phyCoherentStatus:
            curStatus = phyCoherentStatus[ intfName ]
            self.modulationDict[ intfName ] = curStatus.modulationStatus

   def _populateInternalDataPols( self ):
      self.polsSlots = [ slotName for slotName, mediaType in
                         self.xcvrMediaTypesDict.items() if
                         mediaType == XcvrMediaType.xcvrAmpZr ]

   def _getIntfMap( self ):
      """
      Helper function that calls all mapping functions.
      """
      for curPort in self.reqPorts:
         if curPort not in self.xcvrsPresent:
            continue
         self._getIntfMapBase( curPort, self.xcvrTypesDict[ curPort ],
                               self.opticalChannelsDict[ curPort ],
                               self.electricalLanesDict[ curPort ],
                               self.intfNamesDict[ curPort ] )
      if self.getCoherent:
         self._getCoherentIntfMap()
      if self.getAgile:
         self._getAgileIntfMap()
      if self.polsSlots:
         self._getPolsIntfMap()

   def _getIntfMapFromLaneMapping( self, slotName, intfNamesDict ):
      """
      Function that populates self.intfMap and self.hostMap with the relevant intf
      to media/host lane mapping based on the LaneMapping published in Sysdb.
      """
      xcvrStatus = self.xcvrStatus[ slotName ]
      assert isinstance( xcvrStatus, Tac.Type( "Xcvr::CmisStatus" ) )
      laneMapping = xcvrStatus.laneMapping
      if laneMapping is None:
         # A laneMapping entity hasn't been created yet
         return
      # Lane is a zero-indexed lane number that corresponds to the current interface.
      # For example, when intfName == 'Et1/1', lane == 0
      for lane, intfName in intfNamesDict.items():
         # intfNamesDict has the correct correlation between lane and intfName
         # regardless of port renumbering status
         if intfName not in self.activeIntfs:
            # Could be transient state or inactive; don't allocate lanes
            continue
         if intfName not in self.reqIntfs:
            # This intfName was not requested by Cli
            continue
         # Any intfName that reaches this point is a valid one, and the lane
         # corresponds to the master host lane associated with that interface.
         linkMode = self.linkModeStatusDict.get( intfName, LinkMode.linkModeUnknown )
         ethLaneCount = ethTypes.linkModeToEthLaneCount( linkMode )
         numLanes = ethTypes.laneCountNumber( ethLaneCount )
         if not numLanes:
            # We couldn't find a known link mode
            continue

         if lane not in laneMapping.laneToChannel:
            # Skip interfaces associated with a lane that does not map to any
            # media channels.
            continue

         # We will find the set of media channels that correspond to the current
         # intfName
         # We will find the host lanes that correspond to the current intfName.
         # 'lane' is the starting host lane for the current interface, so we will
         # start counting from there
         intfHostLanes = list( range( lane, lane + numLanes ) )
         intfChannelSet = { laneMapping.laneToChannel[ l ].txId
                              for l in intfHostLanes }
         # Populate our maps with what we found
         self.intfMap[ intfName ][ intfName ] = sorted( intfChannelSet )
         self.hostMap[ intfName ] = intfHostLanes

   def _getIntfMapBase( self, slotName, xcvrType, opticalChannels, electricalLanes,
                        intfNamesDict ):
      """
      Function to return the most common interface mapping for a single interface.
      It also handles simple special cases.
      """
      remainingChannels = list( range( opticalChannels ) )

      if slotName not in self.xcvrsPresent:
         return

      # Handle 10x10 100G cases because it is ambiguous between 10x10G and 4x25G
      baseIntfName = slotName + '/1'

      # Special case for the remaining 10 lane 100G case, which is supported
      # only on CFP2 SR10
      if( xcvrType == XcvrType.cfp2 and opticalChannels == 10 and
          self.linkModeStatusDict[ baseIntfName ] ==
            LinkMode.linkModeForced100GbpsFull ):
         self.intfMap[ baseIntfName ][ baseIntfName ] = remainingChannels
         self.hostMap[ baseIntfName ] = remainingChannels
         return

      # 400G-FR4's may have a wierd 1x200g-4 application in which the lane mapping
      # changes: the upper 4 host lanes become inactive, and the lower 4 host lanes
      # use a straight-through mapping. Currently FR4 are getting their
      # xcvrStatus.laneMapping updated by XcvrCtrl. We will use the given
      # laneMapping to compute this IntfMapping. In the future we might decide to
      # have all CMIS transceivers use this method and avoid doing all the math
      # ourselves here
      if( self.xcvrMediaTypesDict.get( slotName ) ==
             XcvrMediaType.xcvr400GBaseFr4 and
          slotName in self.xcvrStatus ):
         self._getIntfMapFromLaneMapping( slotName, intfNamesDict )
         return

      # Lane is a zero-indexed lane number that corresponds to the current interface.
      # For example, when intfName == 'Et1/1' or 'Fc1/1', lane == 0
      elecIdx = 0
      for lane, intfName in intfNamesDict.items():
         if intfName not in self.activeIntfs:
            # Could be transient state or inactive; don't allocate lanes
            continue
         linkMode = self.linkModeStatusDict.get( intfName, LinkMode.linkModeUnknown )
         if linkMode == LinkMode.linkModeAutoneg:
            if( self.anegLanes.get( intfName, EthLaneCount.laneCountUnknown ) !=
                EthLaneCount.laneCountUnknown ):
               ethLaneCount = self.anegLanes[ intfName ]
            else:
               anegSpeed = self.anegSpeed.get( intfName, EthSpeed.speedUnknown )
               if anegSpeed != EthSpeed.speedUnknown:
                  ethLaneCount = ethTypes.nrzSpeedToEthLaneCount( anegSpeed )
               else:
                  if ( self.xcvrStatus is not None and
                       slotName in self.xcvrStatus and
                       isSingleLaneTransceiver( self.xcvrStatus[ slotName ] ) ):
                     # If module stuck in autonegotiation with lane count and speed
                     # unknown (e.g. rxLOS on 1000BASE-T SFP), report back single
                     # lane interface for laneMap.
                     ethLaneCount = EthLaneCount.laneCount1
                  else:
                     ethLaneCount = EthLaneCount.laneCountUnknown
         else:
            ethLaneCount = ethTypes.linkModeToEthLaneCount( linkMode )
         hostLanes = ethTypes.laneCountNumber( ethLaneCount )
         if not hostLanes:
            # lmUnknown
            continue

         if electricalLanes < opticalChannels:
            # We expected electricalLanes >= opticalChannels. In the unlikely
            # event that we get here, we want to prevent the cli from crashing
            # and still try and print relevant data, so we will use the lower
            # value to try and prevent indexing where we shouldn't.
            print( "Unhandled case. Format might not reflect actual configuration." )
            opticalChannels = electricalLanes

         if not opticalChannels or not electricalLanes:
            # Unexpected condition
            continue

         startLane = lane * opticalChannels / electricalLanes
         while remainingChannels and remainingChannels[ 0 ] < startLane:
            # If we aren't starting from /1, then we will want to dump the lanes
            # that we skipped over. We might also need to do this if we've skipped
            # past a lane as a result of the link mode being unknown.
            remainingChannels.pop( 0 )

         # Invariant: At this point, lane should be eligible to use
         # remainingChannels[ 0 ]

         if lane > elecIdx:
            # lane can sometimes jump farther ahead (for example, if a QSFP port
            # only has interfaces Et1/1 and Et1/3 available).
            elecIdx = lane

         opticalChIdxs = []
         # Determine the number of lanes in the group. We want to evenly distribute
         # lanes if opticalChannels != electricalLanes. For example, if we are
         # operating at 50G-2 on a 2 lane QSFP port (lanes = 2, opticalChannels = 2,
         # electricalLanes = 4) then we want to assign /1 to lane 0 and /3 to lane 1.
         # Also, we will always enter this loop, even if the math resolves to less
         # than 0. This could happen if we are printing a single lane speed. In the
         # previous example, set speed to 25G-1 (lanes = 1) and this will occur.
         lanesGroupedFloat = float( hostLanes * opticalChannels ) / electricalLanes
         lanesGrouped = int( lanesGroupedFloat )
         if lanesGrouped != lanesGroupedFloat or lanesGrouped == 0:
            lanesGrouped = hostLanes
         opticalChIdxs += remainingChannels[ : lanesGrouped ]
         remainingChannels = remainingChannels[ lanesGrouped : ]
         if intfName not in self.reqIntfs:
            # We break out here and not above because Alta does not set
            # sub-interfaces inactive ever. This allows /1 to run and allocate
            # remainingChannels, even though it wasn't requested by the user.
            # We then simply disregard the channels it allocated by continuing here.
            continue
         if opticalChIdxs:
            self.intfMap[ intfName ][ intfName ] = opticalChIdxs
            self.hostMap[ intfName ] = list( range( elecIdx, elecIdx + hostLanes ) )
         elecIdx += hostLanes
      return

   def _getCoherentIntfMap( self ):
      """
      Update the interface mappings for coherent interfaces (CloudsRest2 and
         DCO modules). This method only considers coherent media types, which are
         identified using self.xcvrMediaTypesDict and the xcvr slot name. The result
         is a nested interface => participating host electrical lanes dictionary
         keyed by a representative primary interface for the slot. Each module may
         use a different criteria in order to generate this mapping.

      Example output:

         { "Ethernet1/1" : { "Ethernet1/1" : [ 0 ], "Ethernet1/2" : [ 1 ], ... },
           ...
         }
      """
      for slotName in self.intfNamesDict:
         xcvrMediaType = self.xcvrMediaTypesDict[ slotName ]
         if xcvrMediaType not in ( XcvrMediaType.xcvr100GDwdmDco,
                                   XcvrMediaType.xcvr100GDwdmCoherentE,
                                   XcvrMediaType.xcvr100GDwdmCoherent,
                                   XcvrMediaType.xcvr400GDwdmDco ):
            continue
         intfNames = list( self.intfNamesDict[ slotName ].values() )

         # There is only one set of rxlos/txfault data for the xcvr
         self.portLaneMapOverride[ slotName ][ 'xcvr' ] = [ 0 ]
         for intfName in intfNames:
            self.portLaneMapOverride[ slotName ][ intfName ] = []
         modulation = self.modulationDict.get( slotName + '/1' )
         if xcvrMediaType == XcvrMediaType.xcvr100GDwdmDco:
            if( slotName in self.reqPorts and
                   modulation in [ 'modulation8Qam', 'modulation16Qam' ] ):
               self.intfMap[ slotName + '/1' ] = { slotName + '/1': [ 0 ],
                                                   slotName + '/2': [ 1 ] }
            continue
         if ( xcvrMediaType == XcvrMediaType.xcvr400GDwdmDco and
              slotName in self.reqPorts ):
            # CFP2DP04-DCO modules support 1, 2, or 4 active client interfaces
            # simultaneously, with each client consuming 8, 4, or 2 host electrical
            # lanes respectively. The number of active client interfaces depends on
            # the client interface mode configured in the GeneralModeControlLsb
            # register.
            primaryIntf = slotName + '/1'
            xS = gv.xcvrStatusDir_.xcvrStatus.get( getXcvrSlotName( primaryIntf ) )
            clientCfg = Tac.Value(
               'Cfp2DcoHal::Cfp2DcoReg::GeneralModeControlLsb',
               xS.cfp2DcoNewStatus.generalModeControlLsb ).clientCfg
            intfModulationDict = {
               # 1x400G-8, RS544 fec
               ClientConfig.mode400Gaui8: {
                  primaryIntf: [ 0 ],
               },
               # 2x200G-4, RS544 fec
               ClientConfig.mode200Gaui4: {
                  primaryIntf: [ 0 ],
                  slotName + '/5': [ 4 ],
               },
               # 4x100G-2, RS544 fec
               ClientConfig.mode100Gaui2: {
                  primaryIntf: [ 0 ],
                  slotName + '/3': [ 2 ],
                  slotName + '/5': [ 4 ],
                  slotName + '/7': [ 6 ],
               },
               # 2x100G-4, RS528 fec
               ClientConfig.mode100Caui4RsFec: {
                  primaryIntf: [ 0 ],
                  slotName + '/5': [ 4 ],
               },
               # 2x100G-4, fec disabled
               ClientConfig.mode100Caui4NoFec: {
                  primaryIntf: [ 0 ],
                  slotName + '/5': [ 4 ],
               },
            }
            # Check that for each interface in the map, there is a
            # corresponding intfStatus
            filteredDict = {}
            clientDict = intfModulationDict[ clientCfg ]
            for intfName in clientDict:
               if intfName in self.epis:
                  filteredDict[ intfName ] = clientDict[ intfName ]

            self.intfMap[ primaryIntf ] = filteredDict
            continue
         # CR2 in 8QAM groups two physical ports together to send 3 lanes
         # (3 interfaces) of 100G. The groups are interfaces adjacent,
         # such as EtX/1/1, EtX/1/2, and EtX/2/1 sending traffic and EtX/2/2
         # inactive, not sending anything. In the context of populating lane data,
         # we will refer to EtX/1/1 as the primary intf, EtX/1 as the primary
         # port, EtX/2/1 as the sub intf and EtX/2 as the sub port.
         # Only the primary port will contain lane 2 configured to 8qam
         intfName = slotName + '/2'
         if intfName not in intfNames:
            return
         modulation = self.modulationDict.get( intfName )
         if( modulation == 'modulation8Qam' and
              intfName.endswith( '/2' ) ):
            # Get the subsumed port name, we do this by adding 1 to the primary
            # port number (i.e. primary: EtX/1 sub: EtX/(1 + 1) => EtX/2)
            subsumedPort = slotName.split( '/' )
            subsumedPort[ -1 ] = str( int( subsumedPort[ -1 ] ) + 1 )
            subsumedPort = '/'.join( subsumedPort )
            # Create interface names from port names by appending lane id
            subIntfName = f'{subsumedPort}/1'
            inactiveSubIntf = f'{subsumedPort}/2'
            primaryIntfName = f'{slotName}/1'

            if not self.reqIntfs.intersection( [ primaryIntfName, intfName,
                                                 subIntfName ] ):
               if inactiveSubIntf in self.reqIntfs:
                  # To get here we must have requested the inactive sub intf.
                  # As a result, the generic mapping will have been created for the
                  # interfaces of this port, and we want to clear them because we
                  # don't actually want to display them.
                  self.intfMap.update( { subIntfName: {}, inactiveSubIntf: {} } )
               continue

            # Set lane stats for the sub port to be xcvr specific
            self.portLaneMapOverride[ subsumedPort ][ 'xcvr' ] = [ 0 ]
            self.portLaneMapOverride[ subsumedPort ][ subIntfName ] = []
            self.portLaneMapOverride[ subsumedPort ][ inactiveSubIntf ] = []
            # Display subIntf as part of group
            self.portLaneMapOverride[ primaryIntfName ][ subIntfName ] = [ 0 ]
            dwdmMapping = { primaryIntfName: [ 0 ], intfName: [ 0 ],
                            subIntfName: [ 0 ] }
            self.intfMap.update( { primaryIntfName: dwdmMapping,
                              subIntfName: dwdmMapping,
                              intfName: dwdmMapping,
                              inactiveSubIntf: {} } )
            cr2ShortName = primaryIntfName + ',' + intfName \
                              + ',' + subIntfName + ' (Channel 1)'
            cr2ShortName = cr2ShortName.replace( 'Ethernet', 'Et' )
            self.intfNameOverrideXcvrDom[ primaryIntfName ] = cr2ShortName
            self.intfNameOverrideXcvrDom[ intfName ] = cr2ShortName
            self.intfNameOverrideXcvrDom[ subIntfName ] = cr2ShortName
            self.hostMap.update( dwdmMapping )

   def _getAgileIntfMap( self ):
      """
      Updates the mapping for agile interfaces (SFP ports acting as one QSFP port).
      """

      for intfName in self.activeIntfs:
         # Only the primary intfs will appear to be operating at 40G
         if self.linkModeStatusDict[ intfName ] == LinkMode.linkModeForced40GbpsFull:
            if self.xcvrTypesDict[ getXcvrSlotName( intfName ) ] != 'sfpPlus':
               continue

            # We construct the list of subsumed interfaces on the assumption that
            # the primary intf is always the lowest (in intfId) of the 4.
            primary = Tac.Value( 'Arnet::IntfId', intfName )
            allLanes = []
            for i in range( 4 ):
               intf = Tac.Value( 'Arnet::IntfId', intfName )
               intf.intfId = primary.intfId + i
               allLanes.append( intf.stringValue )

            # if none of the found agile ports are actually requested by the CLI
            # show command, we continue on to the next one.
            if not any( intf in self.reqIntfs for intf in allLanes ):
               continue

            allLanes.sort()
            for laneId, curIntfName in enumerate( allLanes ):
               # Display rxlos/txfault data for the xcvr and not mapped to an intf
               self.portLaneMapOverride[ curIntfName ][ 'xcvr' ] = [ 0 ]
               self.portLaneMapOverride[ curIntfName ][ curIntfName ] = []
               # Overriding intfName->curIntfName is how we will prevent displaying
               # lane information grouped together and instead display the interfaces
               # separately (since each is its own port).
               self.portLaneMapOverride[ intfName ][ curIntfName ] = []
               self.intfMap[ curIntfName ] = {}
               self.intfMap[ allLanes[ 0 ] ][ curIntfName ] = [ 0 ]
               # The display name for the intf should contain the lane id and will
               # display the primary intf name
               self.intfNameOverride[ curIntfName ] = f'{intfName} (Lane {laneId})'
               self.intfNameOverrideXcvrDom[ curIntfName ] = \
                     f'{intfName} Lane {laneId} (Channel 1)'
               self.hostMap[ curIntfName ] = [ 0 ]

   def _getPolsIntfMap( self ):
      for polsSlot in self.polsSlots:
         for i in range( 8 ):
            laneId = i + 1
            intfId = f"{polsSlot}/{laneId}"
            if laneId in ( 1, 2 ) and intfId in self.reqIntfs:
               # Mark /1 and /2 as "owning" lane0 and lane1 respectively
               self.intfMap[ intfId ] = { intfId: [ i ] }
               # Change the interface names to Booster for /1 and
               # Pre-amp for /2.
               if laneId == 1:
                  self.intfNameOverride[ intfId ] = "Booster"
               else:
                  self.intfNameOverride[ intfId ] = "Pre-amp"
            else:
               self.intfMap.pop( intfId, None )

def _opt( value, defaultValue=None ):
   if ( value is not None and value != float( "-inf" ) ):
      return value
   else:
      return defaultValue

def subPort( intfName, key='Port ' ):
   return re.sub( 'Ethernet|Management|Fabric', key, getXcvrSlotName( intfName ) )

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.entityManager = em
   gv.xcvrStatusDir_ = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
   gv.lineSystemStatusSliceDir = \
      LazyMount.mount( em, "hardware/line/system/status/slice", "Tac::Dir", "ri" )
