# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import ShowCommand
import CliMatcher
import Intf
import BasicCli
import Tac
from Url import UrlMatcher
from CliPlugin.EthIntfCli import xcvrKw, xcvrShowKw
import IntfRangePlugin.XcvrSlot

# --------------------------------------------------------------------------------
#
# transceiver firmware update slot <SLOTID> ( image <FILE> [ inactive ] | abort  )
#
# Note that "slot <SLOTID>" uses a single matcher that matches two tokens.
# --------------------------------------------------------------------------------

matcherFirmware = CliMatcher.KeywordMatcher(
   'firmware', helpdesc='Manage transceiver firmware' )

matcherUpdate = CliMatcher.KeywordMatcher(
   'update', helpdesc='Perform firmware update' )

XcvrFirmwareUpdateSlotType = IntfRangePlugin.XcvrSlot.XcvrSlot(
   'slot', 'Transceiver slots to perform firmware update commands' )

slotRangeUpdateMatcher = Intf.IntfRange.IntfRangeMatcher(
   explicitIntfTypes=( XcvrFirmwareUpdateSlotType, ),
   dollarCompHelpText='Slot list end' )

fwBinaryFileMatcher = UrlMatcher( lambda fs: fs.scheme in [ 'file:', 'flash:' ],
                                  'Source file name',
                                  acceptSimpleFile=False )

class TransceiverFirmwareUpdate( CliCommand.CliCommandClass ):
   # This command checks that the slot has the correct capabilities within the
   # handler rather than a traditional guard in order to support a range of
   # transceivers. Given a range of slots, if one of the slots is not supported, the
   # command should still run for the other slots.
   #
   # This command also does a roundtrip with the CmisCliSanitizerSm where most of the
   # capability checks are done. This allows for both CLI error messages and trace
   # logs to be printed without duplicating the checking code. It also prevents cases
   # where the ConfigAgent is not up to date with XcvrAgent.
   syntax =\
      'transceiver firmware update SLOTS ( ( image FILE [ inactive ] ) | abort )'
   data = {
      'transceiver': xcvrKw,
      'firmware': matcherFirmware,
      'update': matcherUpdate,
      'SLOTS': slotRangeUpdateMatcher,
      'image': 'Path to the transceiver firmware image to use',
      'FILE': fwBinaryFileMatcher,
      'inactive': 'Keep the new firmware version on stand-by',
      'abort': 'Abort any running firmware update commands',
   }

   handler = "XcvrCmisFwCliHandler.transceiverFirmwareUpdateHandler"

# --------------------------------------------------------------------------------
#
# transceiver firmware activate-standby slot <SLOTID>
#
# Note that "slot <SLOTID>" uses a single matcher that matches two tokens.
# --------------------------------------------------------------------------------
XcvrFirmwareActivateStandbySlotType = IntfRangePlugin.XcvrSlot.XcvrSlot(
   'slot', 'Transceiver slots to perform firmware activate standby command' )

slotRangeActivateStandbyMatcher = Intf.IntfRange.IntfRangeMatcher(
   explicitIntfTypes=( XcvrFirmwareUpdateSlotType, ),
   dollarCompHelpText='Slot list end' )

class TransceiverFirmwareActivateStandby( CliCommand.CliCommandClass ):
   # This command checks that the slot has the correct capabilities within the
   # handler rather than a traditional guard in order to support a range of
   # transceivers. Given a range of slots, if one of the slots is not supported, the
   # command should still run for the other slots.
   #
   # This command also does a roundtrip with the CmisCliSanitizerSm where most of the
   # capability checks are done. This allows for both CLI error messages and trace
   # logs to be printed without duplicating the checking code. It also prevents cases
   # where the ConfigAgent is not up to date with XcvrAgent.
   syntax =\
      'transceiver firmware activate-standby SLOTS'
   data = {
      'transceiver': xcvrKw,
      'firmware': matcherFirmware,
      'activate-standby': 'Activate the standby firmware version',
      'SLOTS': slotRangeActivateStandbyMatcher,
   }

   handler = "XcvrCmisFwCliHandler.transceiverFirmwareActivateStandbyHandler"

# --------------------------------------------------------------------------------
#
# show transceiver status slot <SLOTID> firmware version
#
# Note that "slot <SLOTID>" uses a single matcher that matches two tokens.
# --------------------------------------------------------------------------------

xcvrFirmwareShowSlotType = IntfRangePlugin.XcvrSlot.XcvrSlot(
   "slot", "Show internal slot state" )

slotRangeShowMatcher = Intf.IntfRange.IntfRangeMatcher(
   explicitIntfTypes=( xcvrFirmwareShowSlotType, ),
   dollarCompHelpText="Slot list end" )

class ShowFirmwareVersion( ShowCommand.ShowCliCommandClass ):
   syntax = 'show transceiver status [ SLOTS ] firmware version'
   data = {
      'transceiver': xcvrShowKw,
      'status': 'Show transceiver status information',
      'SLOTS': slotRangeShowMatcher,
      'firmware': 'Show internal firmware state',
      'version': 'Show firmware version'
   }
   handler = "XcvrCmisFwCliHandler.handleShowFirmwareVersion"
   cliModel = "XcvrCmisFwModel.TransceiverFirmwareVersionSlotCollection"

# --------------------------------------------------------------------------------
#
# show transceiver status slot <SLOTID> firmware update
#
# Note that "slot <SLOTID>" uses a single matcher that matches two tokens.
# --------------------------------------------------------------------------------
class ShowFirmwareUpdate( ShowCommand.ShowCliCommandClass ):
   syntax = 'show transceiver status [ SLOTS ] firmware update'
   data = {
      'transceiver': xcvrShowKw,
      'status': 'Show transceiver status information',
      'SLOTS': slotRangeShowMatcher,
      'firmware': 'Show internal firmware state',
      'update': 'Show firmware update status'
   }
   handler = "XcvrCmisFwCliHandler.handleShowFirmwareUpdate"
   cliModel = "XcvrCmisFwModel.TransceiverFirmwareUpdateSlotCollection"

BasicCli.EnableMode.addCommandClass( TransceiverFirmwareUpdate )
BasicCli.EnableMode.addCommandClass( TransceiverFirmwareActivateStandby )
BasicCli.addShowCommandClass( ShowFirmwareVersion )
BasicCli.addShowCommandClass( ShowFirmwareUpdate )
