#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import MultiRangeRule
import ShowCommand
from Intf.IntfRange import intfRangeMatcher

import Toggles.XcvrToggleLib

from CliPlugin import IntfCli
from CliPlugin.EthIntfCli import xcvrShowKw

from CliPlugin.XcvrConfigCli import xcvrCmisGuard

IntfRangeKW = 'INTFRANGE'

# --------------------------------------------------------------------------------
# show transceiver traffic-loopback [interface <intf-specifier>] capabilities
# --------------------------------------------------------------------------------
class ShowCmisLoopbackCapsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( f'show transceiver traffic-loopback '
              f'capabilities [interface {IntfRangeKW}] ' )
   data = {
      'transceiver': xcvrShowKw,
      'traffic-loopback': 'Show transceiver loopback status and capabilities',
      'capabilities': 'Show loopback capabilities',
      'interface': 'Show loopback capabilities for specific interface(s)',
      IntfRangeKW: intfRangeMatcher,
   }

   handler = ( 'XcvrCmisLoopbackHandler.CmisLoopbackCapabilitiesShowHandler' )

   cliModel = ( 'XcvrCmisLoopbackModel.IntfCapabilities' )

# --------------------------------------------------------------------------------
# show transceiver traffic-loopback [interface <intf-specifier>] capabilities
# --------------------------------------------------------------------------------
class ShowCmisLoopbackStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = f'show transceiver traffic-loopback status [interface {IntfRangeKW}]'

   data = {
      'transceiver': xcvrShowKw,
      'traffic-loopback': 'Show transceiver loopback status and capabilities',
      'status': 'Show loopback configuration and operational status',
      'interface':
      'Show loopback configuration and operational status for specific interface(s)',
      IntfRangeKW: intfRangeMatcher,
   }

   handler = ( 'XcvrCmisLoopbackHandler.CmisLoopbackStatusShowHandler' )

   cliModel = ( 'XcvrCmisLoopbackModel.TrafficLoopback' )


# --------------------------------------------------------------------------------
# [ no | default ] traffic-loopback source {system|network} device transceiver
# (host|media) [lane <LANES>]
# --------------------------------------------------------------------------------
nodeTrafficLoopbackGuarded = CliCommand.guardedKeyword(
   'traffic-loopback', helpdesc='Configure loopback', guard=xcvrCmisGuard )
nodeTrafficLoopbackUnguarded = CliMatcher.KeywordMatcher(
   'traffic-loopback', helpdesc='Configure loopback' )

laneMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=lambda: ( 1, 8 ),
   noSingletons=False,
   value=lambda mode, genericRangeList: list( genericRangeList.values() ),
   helpdesc='lane number(s)' )

commonConfigData = {
   'trafficLoopback_GUARDED': nodeTrafficLoopbackGuarded,
   'trafficLoopback_UNGUARDED': nodeTrafficLoopbackUnguarded,
   'source': 'Configure traffic source',
   'device': 'Configure loopback device',
   'transceiver': 'implement loopback in the transceiver module',
   'lane': 'Specify lanes on which the loopback is configured',
   'LANES': laneMatcher,
}

class CmisLoopbackSystemConfigCommand( CliCommand.CliCommandClass ):
   syntax = ( 'trafficLoopback_GUARDED source system device transceiver '
              '(host|media) [lane LANES]' )
   noOrDefaultSyntax = (
      'trafficLoopback_UNGUARDED source system device transceiver '
      '[(host|media) [lane LANES]]' )
   data = dict(
      **{
         'system':
         'Traffic from local host',
         'host':
         'Loopback configured on the host side '
         '(CMIS Host Side Input Loopback) ',
         'media':
         'Loopback configured on the media side '
         '(CMIS Media Side Output Loopback) ',
      }, **commonConfigData )

   handler = ( 'XcvrCmisLoopbackHandler.CmisLoopbackSystemConfigCommandHandler' )
   noHandler = ( 'XcvrCmisLoopbackHandler.CmisLoopbackSystemConfigCommandNoHandler' )
   defaultHandler = noHandler

class CmisLoopbackNetworkConfigCommand( CliCommand.CliCommandClass ):
   syntax = ( 'trafficLoopback_GUARDED source network device transceiver '
              '(host|media) [lane LANES]' )
   noOrDefaultSyntax = (
      'trafficLoopback_UNGUARDED source network device transceiver '
      '[(host|media) [lane LANES]]' )
   data = dict(
      **{
         'network':
         'traffic from peer host',
         'host':
         'Loopback configured on the host side '
         '(CMIS Host Side Output Loopback) ',
         'media':
         'Loopback configured on the media side '
         '(CMIS Media Side Input Loopback) ',
      }, **commonConfigData )

   handler = ( 'XcvrCmisLoopbackHandler.CmisLoopbackNetworkConfigCommandHandler' )
   noHandler = (
      'XcvrCmisLoopbackHandler.CmisLoopbackNetworkConfigCommandNoHandler' )
   defaultHandler = noHandler


if Toggles.XcvrToggleLib.toggleCmisLoopbackEnabled():
   BasicCli.addShowCommandClass( ShowCmisLoopbackStatusCmd )
   BasicCli.addShowCommandClass( ShowCmisLoopbackCapsCmd )
   IntfCli.IntfConfigMode.addCommandClass( CmisLoopbackSystemConfigCommand )
   IntfCli.IntfConfigMode.addCommandClass( CmisLoopbackNetworkConfigCommand )
