# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import LazyMount
import CliParser
import CliMatcher
import CliCommand
import BasicCli
import ShowCommand
import Toggles.XcvrToggleLib
import CliPlugin.XcvrAllStatusDir
from CliPlugin.IntfCli import Intf
from CliPlugin.EthIntfCli import xcvrKw, xcvrShowKw
from CliPlugin.XcvrConfigCli import ( XcvrConfigModelet, primaryIntfGuard )
from XcvrLib import getXcvrStatus
from TypeFuture import TacLazyType
import CliGlobal

# aid/5819 - set up global variables that are initialized once in plugin load time
gv = CliGlobal.CliGlobal( xcvrConfigDir=None, xcvrStatusDir=None )
XcvrType = TacLazyType( 'Xcvr::XcvrType' )

coherentFecDegradeEnabled = Toggles.XcvrToggleLib.toggleCoherentFecDegradeEnabled()

# --------------------------------------------------------------------------------
# [ no | default ] transceiver error-correction coherent degrade <detected|excessive>
# raise RAISE clear CLEAR
# -------------------------------------------------------------- ------------------
minPreFecBerValue, maxPreFecBerValue = ( 1e-10, 9.99e-1 )

def coherentErrcorectionDegradeGuard( mode, token ):
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( mode.intf.name )
   if not xcvrName:
      return None

   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )

   # Guard this token for non-OSFP and non-QSFP-DD ports
   if xcvrStatus.xcvrType not in [ XcvrType.osfp, XcvrType.qsfpDd ]:
      return CliParser.guardNotThisXcvrSlotType

   # Guard for non-primary interfaces
   if primaryIntfGuard( mode, token ) is not None:
      return CliParser.guardNotThisInterface

   return None

matcherDegrade = CliMatcher.KeywordMatcher(
   'degrade', helpdesc='Configure forward error correction degrade alarms' )
nodeDegrade = CliCommand.Node( matcher=matcherDegrade,
                               guard=coherentErrcorectionDegradeGuard )

class TransceiverErrcorrectionCoherentDegradeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'transceiver error-correction coherent GUARD_DEGRADE '
              '( detected | excessive ) raise RAISE clear CLEAR ' )
   noOrDefaultSyntax = ( 'transceiver error-correction coherent UNGUARD_DEGRADE'
                         '( detected | excessive ) ' )
   data = {
      'transceiver': xcvrKw,
      'error-correction': 'Configure forward error correction',
      'coherent': 'Configure forward error correction coherent features',
      'GUARD_DEGRADE' : nodeDegrade,
      'UNGUARD_DEGRADE' : matcherDegrade,
      'detected': 'Configure forward error correction detected degrade alarm',
      'excessive': 'Configure forward error correction excessive degrade alarm',
      'raise': 'Set error correction raise threshold',
      'RAISE': CliMatcher.FloatMatcher(
         minPreFecBerValue, maxPreFecBerValue,
         helpdesc='Error correction raise threshold value', precisionString='%.2E' ),
      'clear': 'Set error correction clear threshold',
      'CLEAR': CliMatcher.FloatMatcher(
         minPreFecBerValue, maxPreFecBerValue,
         helpdesc='Error correction clear threshold value', precisionString='%.2E' )
   }

   handler = "XcvrCoherentFecDegradeCliHandler.coherentFecDegradeHandler"
   noOrDefaultHandler = "XcvrCoherentFecDegradeCliHandler.coherentFecDegradeHandler"

if coherentFecDegradeEnabled:
   XcvrConfigModelet.addCommandClass( TransceiverErrcorrectionCoherentDegradeCmd )

# --------------------------------------------------------------------------------
# show transceiver error-correction coherent degrade [ interface INTFS ]
# --------------------------------------------------------------------------------
class ShowErrcorrectionCoherentDegradeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show transceiver error-correction coherent degrade [ interface INTFS ]'
   data = {
      'transceiver': xcvrShowKw,
      'error-correction': 'Show forward error correction information',
      'coherent': 'Show coherent forward error correction information',
      'degrade': 'Show link degrade information',
      'interface': 'Show link degrade for a specific interface',
      'INTFS': Intf.rangeMatcher,
   }

   handler = "XcvrCoherentFecDegradeCliHandler.showXcvrCoherentFecDegrade"
   cliModel = "XcvrCoherentFecDegradeModel.TransceiverCoherentFecDegradeCollection"

if coherentFecDegradeEnabled:
   BasicCli.addShowCommandClass( ShowErrcorrectionCoherentDegradeCmd )

#------------------------------------------------------
# Plugin method
#------------------------------------------------------
def Plugin( em ):
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
   gv.xcvrConfigDir = LazyMount.mount( em, "hardware/xcvr/config/all",
                                    "Xcvr::AllConfigDir", "r" )
