# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import LazyMount
import CliParser
import CliMatcher
import CliCommand
import Toggles.XcvrToggleLib
import CliPlugin.XcvrAllStatusDir
from CliPlugin.EthIntfCli import xcvrKw
from CliPlugin.XcvrConfigCli import ( XcvrConfigModelet )
from XcvrLib import getXcvrStatus, isPrimaryIntf
from TypeFuture import TacLazyType
import CliGlobal

gv = CliGlobal.CliGlobal( xcvrStatusDir=None, xcvrConfigDir=None )

XcvrType = TacLazyType( 'Xcvr::XcvrType' )
xcvrLpoEnabled = Toggles.XcvrToggleLib.toggleXcvrLpoEnabled()

# --------------------------------------------------------------------------------
# Guard keywords ending with electrical
# transceiver electrical ...
# --------------------------------------------------------------------------------

def txElectricalGuard( mode, token ):
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( mode.intf.name )
   if not xcvrName:
      return None

   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if xcvrStatus.xcvrType not in [ XcvrType.qsfpDd, XcvrType.osfp ]:
      # Guard this token for non-QSFP-DD and non-OSFP ports
      return CliParser.guardNotThisXcvrSlotType

   if not isPrimaryIntf( mode.intf.name ):
      # Only primary interfaces can configure Tx electrical drive
      return CliParser.guardNotThisInterface

   # Don't use guard to check for inserted xcvr or xcvr features

   return None

# --------------------------------------------------------------------------------
# [ no | default ] transceiver electrical drive < linear | limiting >
# --------------------------------------------------------------------------------

class TransceiverElectricalDrive( CliCommand.CliCommandClass ):
   syntax = 'transceiver electrical drive_GUARDED DRIVE'
   noOrDefaultSyntax = 'transceiver electrical drive ...'
   data = {
      'transceiver': xcvrKw,
      'electrical': 'Configure transceiver electrical side parameters',
      'drive_GUARDED': CliCommand.guardedKeyword( 'drive',
            helpdesc='Configure system and transceiver transmitter options',
            guard=txElectricalGuard ),
      'drive': 'Configure system and transceiver transmitter options',
      'DRIVE': CliMatcher.EnumMatcher( {
            'linear': 'Force linear configuration of the inserted transceiver',
            'limiting': 'Force limiting configuration of the inserted transceiver',
      } )
   }

   handler = "XcvrElectricalCliHandler.transceiverElectricalDriveHandler"
   noOrDefaultHandler = (
      "XcvrElectricalCliHandler.transceiverElectricalDriveNoDefHandler" )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver electrical signal bandwidth < high | low >
# --------------------------------------------------------------------------------

class TransceiverElectricalSignal( CliCommand.CliCommandClass ):
   syntax = 'transceiver electrical signal_GUARDED bandwidth BANDWIDTH'
   noOrDefaultSyntax = 'transceiver electrical signal bandwidth ...'
   data = {
      'transceiver': xcvrKw,
      'electrical': 'Configure transceiver electrical side parameters',
      'signal_GUARDED': CliCommand.guardedKeyword( 'signal',
            helpdesc='Configure transceiver signal options',
            guard=txElectricalGuard ),
      'signal': 'Configure transceiver signal options',
      'bandwidth': 'Configure transceiver signal bandwidth options',
      'BANDWIDTH': CliMatcher.EnumMatcher( {
            'high': 'Force high bandwidth configuration of the LPO module',
            'low': 'Force low bandwidth configuration of the LPO module',
      } )
   }

   handler = (
      "XcvrElectricalCliHandler.transceiverElectricalSignalHandler" )
   noOrDefaultHandler = (
      "XcvrElectricalCliHandler.transceiverElectricalSignalNoDefHandler" )

if xcvrLpoEnabled:
   XcvrConfigModelet.addCommandClass( TransceiverElectricalDrive )
   XcvrConfigModelet.addCommandClass( TransceiverElectricalSignal )

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
   gv.xcvrConfigDir = LazyMount.mount( em, "hardware/xcvr/config/all",
                                    "Xcvr::AllConfigDir", "r" )
