# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# This module provides a getter function to an instance of a
# Hardware::XcvrController::AllStatusDir.  The returned instance of our AllStatusDir
# is guaranteed to (eventually) contain all the transceiver hardware status objects
# in Sysdb.  The returned AllStatusDir is guaranteed to (eventually) not contain
# any stale status entities from removed Linecards (or any other
# field-replaceable-units).
#
# CliPlugins which need to access transceiver hardware status information should
# import this module.  The intended use is to call the getter function only
# when the user needs the all collection.  The all collection is lazy-ily evaluated.
# The getter function should not be unconditionally called from a CliPlugin's Plugin
# function because mounting and initialization is expensive.

import threading
from Ark import synchronized
import Tac
import LazyMount

# Intended to be a private variable.  We have no real way to enforce this
# in python, so just don't touch it, or else!
_xcvrHardwareStatusAggregatorSm = None
_xcvrHardwareStatusAggregatorSmLock = threading.Lock()

# gyorgym:
# I've opted against using a singleton class pattern.  It seems unnecessary
# given that imported python modules are already singletons themselves.  In python,
# import creates a single copy of each module, and subsequent imports of the same
# name will always return the same module object.  Instead we have a
# "lazy evaluated" module global which is exposed through this getter function.
@synchronized( _xcvrHardwareStatusAggregatorSmLock )
def allXcvrHwStatusDir( entityManager ):
   """
   Returns a reference to an instance of AllStatusDir containing all available
   transceiver hardware status entities in Sysdb.

   Returns
   -------
   Hardware::XcvrController::AllStatusDir
   """
   global _xcvrHardwareStatusAggregatorSm
   if _xcvrHardwareStatusAggregatorSm is None:
      baseConfigDir = LazyMount.mount( entityManager,
                                       "hardware/xcvrController/config",
                                       "Tac::Dir", "ri" )
      baseStatusDir = LazyMount.mount( entityManager,
                                       "hardware/archer/xcvrController/status",
                                       "Tac::Dir", "ri" )
      # We need to force out mounts because we're going to be accessing our LazyMount
      # proxies from the activity thread.
      _xcvrHardwareStatusAggregatorSm = Tac.newInstance(
            "Hardware::XcvrController::HardwareStatusAggregatorSm",
            LazyMount.force( baseConfigDir ),
            LazyMount.force( baseStatusDir ) )
      _xcvrHardwareStatusAggregatorSm.start()
   return _xcvrHardwareStatusAggregatorSm.allStatusDir
