# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from XcvrLib import noneToNegInf
from IntfModels import Interface
import Ark
import Tac
from TypeFuture import TacLazyType
import TableOutput

SUPPORTED_FORMATS = frozenset( [ "default", "csv" ] )

CmisModuleState = TacLazyType( "Xcvr::CmisModuleState" )
DataPathState = TacLazyType( "Xcvr::DataPathState" )
MediaType = TacLazyType( "Xcvr::MediaType" )
TributaryId = TacLazyType( 'Xcvr::TributaryId' )
XcvrPresence = TacLazyType( "Xcvr::XcvrPresence" )

paramTypeToStr = {
      'temperature': 'Temperature',
      'caseTemperature': 'Case temperature',
      'voltage': 'Voltage',
      'txBias': 'TX bias current',
      'txPower': 'Optical TX power',
      'txPowerMin': 'TX power min',
      'txPowerMax': 'TX power max',
      'txPowerAvg': 'TX power avg',
      'rxPower': 'Optical RX power',
      'snr': 'SNR',
      'snrMin': 'SNR min',
      'snrMax': 'SNR max',
      'snrAvg': 'SNR avg',
      'residualISI': 'Residual ISI',
      'pam4Transitions': 'Level transitions',
      'tecCurrent': 'TEC current',
      'laserFreq': 'Frequency error',
      'laserTemp': 'Temperature error',
      'laserTemperature': 'Laser temperature',
      'preFecBERCurr': 'Pre-FEC BER',
      'preFecBERMin': 'Media pre-FEC BER min',
      'preFecBERMax': 'Media pre-FEC BER max',
      'preFecBERAvg': 'Media pre-FEC BER avg',
      'preFecBERMinHost': 'Host pre-FEC BER min',
      'preFecBERMaxHost': 'Host pre-FEC BER max',
      'preFecBERAvgHost': 'Host pre-FEC BER avg',
      'uncorrectedBERCurr': 'Uncorrected BER',
      'preFECBERAvg': 'Pre-FEC BER avg',
      'preFECBERMax': 'Pre-FEC BER max',
      'preFECBERMin': 'Pre-FEC BER min',
      'preFECBERHighAlarmExceeded': 'Pre-FEC BER high alarm exceeded',
      'preFECBERHighWarnExceeded': 'Pre-FEC BER high warn exceeded',
      'uncorrectedBERAvg': 'Uncorrected BER avg',
      'uncorrectedBERMax': 'Uncorrected BER max',
      'uncorrectedBERMin': 'Uncorrected BER min',
      'uncorrectedBERHighAlarmExceeded': 'Uncorrected BER high alarm exceeded',
      'uncorrectedBERHighWarnExceeded': 'Uncorrected BER high warn exceeded',
      'errFramesAvg': 'Media post-FEC errored frames ratio avg',
      'errFramesMin': 'Media post-FEC errored frames ratio min',
      'errFramesMax': 'Media post-FEC errored frames ratio max',
      'errFramesAvgHost': 'Host post-FEC errored frames ratio avg',
      'errFramesMinHost': 'Host post-FEC errored frames ratio min',
      'errFramesMaxHost': 'Host post-FEC errored frames ratio max',
      'errFramesCur': 'Post-FEC errored frames ratio',
      'tempAlarmExceeded': 'Temperature alarm exceeded',
      'tempWarnExceeded': 'Temperature warn exceeded',
      'netFecUncBlkCount': 'Post-FEC UCB',
      'rxChromDispersion': 'Chromatic dispersion',
      'rxDiffGroupDelay': 'Differential group delay',
      'rxStateOfPolarization': 'State of polarization',
      'rxNetworkPDL': 'Polarization dependent loss',
      'rxQFactor': 'Q-factor',
      'rxCarrierFreqOffset': 'Carrier frequency offset',
      'netOsnrEstimate': 'Received OSNR estimate',
      'laserAge': 'Laser age',
      'txLaserAge': 'TX laser age',
      'rxLaserAge': 'RX laser age',
      'txLaserPower': 'TX laser power',
      'rxLaserPower': 'RX laser power',
      'txLaserTemperature': 'TX laser temperature',
      'rxLaserTemperature': 'RX laser temperature',
      'rxChannelPower': 'RX channel power',
      'coherentTxPower': 'TX power',
      'totalRxPower': 'RX total power',
      'rxTotalPowerMin': 'RX total power min',
      'rxTotalPowerMax': 'RX total power max',
      'rxTotalPowerAvg': 'RX total power avg',
      'rollOff': 'Roll-off factor',
      'aggrTxPwr': 'Aggregate TX Power',
      'aggrRxPwr': 'Aggregate RX Power',
      'cdHighShortLink': 'Chromatic dispersion (short link)',
      'cdLowLongLink': 'Chromatic dispersion (long link)',
      'rxCdMin': 'Chromatic dispersion min',
      'rxCdMax': 'Chromatic dispersion max',
      'rxCdAvg': 'Chromatic dispersion avg',
      'osnr': 'Received OSNR estimate',
      'osnrMin': 'Received OSNR estimate min',
      'osnrMax': 'Received OSNR estimate max',
      'osnrAvg': 'Received OSNR estimate avg',
      'esnr': 'Received ESNR estimate',
      'esnrMin': 'Received ESNR estimate min',
      'esnrMax': 'Received ESNR estimate max',
      'esnrAvg': 'Received ESNR estimate avg',
      'dgd': 'Differential group delay',
      'dgdMin': 'Differential group delay min',
      'dgdMax': 'Differential group delay max',
      'dgdAvg': 'Differential group delay avg',
      'sopmd': 'SOPMD',
      'sopmdMin': 'SOPMD min',
      'sopmdMax': 'SOPMD max',
      'sopmdAvg': 'SOPMD avg',
      'pdl': 'Polarization dependent loss',
      'pdlMin': 'Polarization dependent loss min',
      'pdlMax': 'Polarization dependent loss max',
      'pdlAvg': 'Polarization dependent loss avg',
      'cfo': 'Carrier frequency offset',
      'cfoMin': 'Carrier frequency offset min',
      'cfoMax': 'Carrier frequency offset max',
      'cfoAvg': 'Carrier frequency offset avg',
      'evm': 'Error vector magnitude',
      'evmMin': 'Error vector magnitude min',
      'evmMax': 'Error vector magnitude max',
      'evmAvg': 'Error vector magnitude avg',
      'sopRoc': 'SOP rate of change',
      'sopRocMin': 'SOP rate of change min',
      'sopRocMax': 'SOP rate of change max',
      'sopRocAvg': 'SOP rate of change avg',
      'mer': 'Modulation error ratio',
      'txChannelPower': 'TX power',
      'cmisLaserFrequency': 'Laser frequency',
      'txModBiasXIMonitor': 'Modulation Bias XI',
      'txModBiasXQMonitor': 'Modulation Bias XQ',
      'txModBiasYIMonitor': 'Modulation Bias YI',
      'txModBiasYQMonitor': 'Modulation Bias YQ',
      'txModBiasXPhaseMonitor': 'Modulation Bias X Phase',
      'txModBiasYPhaseMonitor': 'Modulation Bias Y Phase',
      'rxSignalPowerMin': 'RX channel power min',
      'rxSignalPowerMax': 'RX channel power max',
      'rxSignalPowerAvg': 'RX channel power avg',
      'qFactor': 'Q-factor',
      'qFactorMin': 'Q-factor min',
      'qFactorMax': 'Q-factor max',
      'qFactorAvg': 'Q-factor avg',
      'qMargin': 'Q-margin',
      'qMarginMin': 'Q-margin min',
      'qMarginMax': 'Q-margin max',
      'qMarginAvg': 'Q-margin avg',
      'acquisitionCounterInphi': 'Acquisition counter',
      'clkRecoveryLoop': 'Clock recovery loop',
      'clkRecoveryLoopMin': 'Clock recovery loop min',
      'clkRecoveryLoopMax': 'Clock recovery loop max',
      'clkRecoveryLoopAvg': 'Clock recovery loop avg',
      'sopmdLg': 'SOPMD low granularity',
      'sopmdLgMin': 'SOPMD low granularity min',
      'sopmdLgMax': 'SOPMD low granularity max',
      'sopmdLgAvg': 'SOPMD low granularity avg',
      'unknown': 'Unknown',
      }

xcvrPresenceToStr = {
   XcvrPresence.xcvrPresenceUnknown: 'unknown',
   XcvrPresence.xcvrNotPresent: 'not present',
   XcvrPresence.xcvrPresent: 'present'
}

moduleStateToStr = {
   CmisModuleState.moduleUnknown: 'unknown',
   CmisModuleState.moduleLowPwr: 'low power',
   CmisModuleState.modulePwrUp: 'power up',
   CmisModuleState.moduleReady: 'ready',
   CmisModuleState.modulePwrDn: 'power down',
   CmisModuleState.moduleFault: 'fault'
}

dataPathStateToStr = {
   DataPathState.dataPathUnknown: 'unknown',
   DataPathState.dataPathDeactivated: 'deactivated',
   DataPathState.dataPathInit: 'init',
   DataPathState.dataPathDeinit: 'deinit',
   DataPathState.dataPathActivated: 'activated',
   DataPathState.dataPathTxTurnOn: 'TX turn on',
   DataPathState.dataPathTxTurnOff: 'TX turn off',
   DataPathState.dataPathInitialized: 'initialized'
}

# ----------------------------------------------------------------------------
#
# Helpers for "show interface [ <interface> ] transceiver [csv]"
#
# ----------------------------------------------------------------------------

def _createTable():
   headers = ( "Slot",
               "Channel",
               "Temp (Celsius)",
               "Voltage (Volts)",
               "Bias Current (mA)",
               "Optical Tx Power (dBm)",
               "Optical Rx Power (dBm)",
               "Last Update" )
   formats = ( TableOutput.Format( justify="left", maxWidth=13 ),
               TableOutput.Format( justify="right", maxWidth=7, wrap=True ),
               TableOutput.Format( justify="left", maxWidth=9, wrap=True,
                                   dotAlign=True ),
               TableOutput.Format( justify="left", maxWidth=7, wrap=True,
                                   dotAlign=True ),
               TableOutput.Format( justify="left", maxWidth=7, wrap=True,
                                   dotAlign=True ),
               TableOutput.Format( justify="left", maxWidth=8, wrap=True,
                                   dotAlign=True ),
               TableOutput.Format( justify="left", maxWidth=8, wrap=True,
                                   dotAlign=True ),
               TableOutput.Format( justify="right", maxWidth=11 ) )
   for fmt in formats:
      fmt.noPadLeftIs( True )
      fmt.padLimitIs( True )
      fmt.align_ = "bottom"
   table = TableOutput.createTable( headers )
   table.formatColumns( *formats )
   return table

def _printHeaderDefault():
   print( "Please consider using show interfaces transceiver dom." )
   print( "If device is externally calibrated, only calibrated values are printed." )
   print( "n/a: not applicable, Tx: transmit, Rx: receive." )
   print( "mA: milliamperes, dBm: decibels (milliwatts)." )

def _printHeaderCsv():
   print( "Last Update,Port (Interface Name),Xcvr Serial Number,"
           "Media type,Temperature (Celsius),Voltage (Volts),Current (mA),"
           "Tx Power (dBm),Rx Power (dBm)" )

# ----------------------------------------------------------------------------
#
# Helpers for "show interface [ <interface> ] transceiver detail [csv]"
#
# ----------------------------------------------------------------------------
def _createThreshParamTable( name, unit ):
   headers = ( "Slot",
               "Channel",
               f"{name} ({unit})",
               f"High Alarm Threshold ({unit})",
               f"High Warn Threshold ({unit})",
               f"Low Alarm Threshold ({unit})",
               f"Low Warn Threshold ({unit})" )
   formats = ( TableOutput.Format( justify="left", maxWidth=13 ),
               TableOutput.Format( justify="right", maxWidth=7 ),
               TableOutput.Format( justify="left", maxWidth=11, minWidth=11,
                                   wrap=True, dotAlign=True ),
               TableOutput.Format( justify="left", maxWidth=10, wrap=True,
                                   dotAlign=True ),
               TableOutput.Format( justify="left", maxWidth=9, wrap=True,
                                   dotAlign=True ),
               TableOutput.Format( justify="left", maxWidth=9, wrap=True,
                                   dotAlign=True ),
               TableOutput.Format( justify="left", maxWidth=9, wrap=True,
                                   dotAlign=True ) )
   for fmt in formats:
      fmt.noPadLeftIs( True )
      fmt.padLimitIs( True )
      fmt.align_ = "bottom"
   table = TableOutput.createTable( headers )
   table.formatColumns( *formats )
   return table

def _printThresHeaderPerParameterCsv( name, unit ):
   units = '(' + unit + ')'
   print( 'Port (Interface Name), ' + name + units +
          ' High Alarm Threshold, ' + units +
          ' High Warn Threshold, ' + units +
          ' Low Alarm Threshold, ' + units +
          ' Low Warn Threshold, ' + units )

def formatThreshold( threshold:float, overridden:bool, length:int,
                     alignLeft:bool = True ) -> str:
   """
   Returns a formatted left/right aligned threshold padded to length LENGTH.
   Threshold value is appended with an asterisk if OVERRRIDDEN
   is true.
   """
   MINCOLSIZE = 5
   assert length >= MINCOLSIZE
   threshStr = f"{threshold:.2f}"
   threshStr += "*" if overridden else ""
   align = '<' if alignLeft else '>'
   alignThreshStr = f"{threshStr:{align}{length}}"
   if len( alignThreshStr ) > length:
      # Edge case to deal with when the threshold is too long and gets cut off.
      # This should never really happen outside of testing as
      # threshold values should never be this long.
      if overridden:
         alignThreshStr = ( alignThreshStr[ : length - len( "..." ) - len( "*" ) ]
                             + "...*" )
      else:
         alignThreshStr = alignThreshStr[ : length - len( "..." ) ] + "..."

   return alignThreshStr


# ----------------------------------------------------------------------------
#
# Models for "show interface [ <interface> ] transceiver [detail] [csv]"
#
# ----------------------------------------------------------------------------
class InterfaceTransceiverBase( Model ):
   _slot = Str( help="Transceiver slot number" )
   _channel = Str( help="Transceiver optical channel number" )

   def setSlot( self, val: Str ):
      self._slot = val

   def getSlot( self ) -> Str:
      return self._slot

   def setChannel( self, val: Str ):
      self._channel = val

   def getChannel( self ) -> Str:
      return self._channel

   def renderModelNoDetail( self, table, intfName, printFmt ):
      assert printFmt in SUPPORTED_FORMATS, "Unrecognized DOM output format"
      numFields = { "default": 6, "csv": 8 }
      if printFmt == 'default':
         table.newRow( *( ( self._slot, self._channel ) +
                          ( "n/a", ) * numFields[ printFmt ] ) )
      elif printFmt == 'csv':
         fmt = "{},{},{},{},{},{},{},{},{}"
         numFields = 8
         print( fmt.format( *( ( "n/a", intfName, ) +
                  ( "n/a", ) * ( numFields - 1 ) ) ) )

   def renderModelDetailed( self, table, intfName, param ):
      numDataFields = 5
      table.newRow( *( ( self._slot, self._channel ) + ( "n/a", ) * numDataFields ) )

class InterfaceTransceiverDetailThresholds( Model ):
   highAlarm = Float( help="High alarm threshold for parameter", optional=True )
   highAlarmOverridden = Bool( help="High alarm hardware threshold is overridden by "
                                    "software config",
                               default=False )
   highWarn = Float( help="High warning threshold for parameter", optional=True )
   highWarnOverridden = Bool( help="High warning hardware threshold is overridden by"
                                   " software config",
                              default=False )
   lowAlarm = Float( help="Low alarm threshold for parameter", optional=True )
   lowAlarmOverridden = Bool( help="Low alarm hardware threshold is overridden by "
                                   "software config",
                              default=False )
   lowWarn = Float( help="Low warning threshold for parameter", optional=True )
   lowWarnOverridden = Bool( help="Low warning hardware threshold is overridden by "
                                  "software config",
                             default=False )

class InterfaceTransceiverDetails( Model ):
   temperature = \
       Submodel( valueType=InterfaceTransceiverDetailThresholds,
                 help="Temperature alarm and warning thresholds in Celsius.",
                 optional=True )

   voltage = \
       Submodel( valueType=InterfaceTransceiverDetailThresholds,
                 help="Voltage alarm and warning thresholds in Volts.",
                 optional=True )

   txBias = \
       Submodel( valueType=InterfaceTransceiverDetailThresholds,
                 help="Bias current alarm and warning thresholds in mA.",
                 optional=True )

   txPower = \
       Submodel( valueType=InterfaceTransceiverDetailThresholds,
                 help="TX power alarm and warning thresholds in dBm.",
                 optional=True )

   rxPower = \
       Submodel( valueType=InterfaceTransceiverDetailThresholds,
                 help="RX power alarm and warning thresholds in dBm.",
                 optional=True )

   totalRxPower = \
       Submodel( valueType=InterfaceTransceiverDetailThresholds,
                 help="Total RX power alarm and warning thresholds in dBm.",
                 optional=True )

   _domThresholdOverridden = Bool(
      help="Indicates when the DOM thresholds are set by configuration" )

   def getDomThresholdOverridden( self ):
      """
      Getter function for protected member variable. Returns a boolean.
      """
      return self._domThresholdOverridden

   def setDomThresholdOverridden( self, value ):
      """
      Setter function

      Parameters
      ----------
      value : bool
      """
      self._domThresholdOverridden = value

class InterfaceTransceiver( InterfaceTransceiverBase ):
   slot = Str( help="Transceiver slot number" )
   channel = Str( help="Transceiver optical channel number" )
   vendorSn = Str( help="Transceiver serial number" )
   mediaType = Str( help="Media type" )
   narrowBand = Bool( help="Narrowband Transceiver", optional=True )
   updateTime = Float( help="Last update time in UTC" )
   temperature = Float( help="Temperature in Celsius", optional=True )
   voltage = Float( help="Voltage in Volts", optional=True )
   txBias = Float( help="Current in mA", optional=True )
   txPower = Float( help="TX power in dBm", optional=True )
   rxPower = Float( help="RX power in dBm", optional=True )
   totalRxPower = Float( help="Total RX power in dBm", optional=True )
   details = Submodel( valueType=InterfaceTransceiverDetails,
                       help="Warning and alarm thresholds",
                       optional=True )

   def setSlot( self, val: Str ):
      super().setSlot( val )
      self.slot = val

   def setChannel( self, val: Str ):
      super().setChannel( val )
      self.channel = val

   def renderModelNoDetail( self, table, intfName, printFmt ):
      assert printFmt in SUPPORTED_FORMATS, "Unrecognized DOM output format"

      updateTime = Ark.timestampToStr( self.updateTime, now=Tac.utcNow() )
      if printFmt == 'default':
         fieldValues = [ self.slot,
                         self.channel,
                         noneToNegInf( self.temperature ),
                         noneToNegInf( self.voltage ),
                         noneToNegInf( self.txBias ),
                         noneToNegInf( self.txPower ),
                         noneToNegInf( self.rxPower ),
                         updateTime ]
      elif printFmt == 'csv':
         fieldValues = [ updateTime,
                         intfName,
                         self.vendorSn,
                         self.mediaType,
                         noneToNegInf( self.temperature ),
                         noneToNegInf( self.voltage ),
                         noneToNegInf( self.txBias ),
                         noneToNegInf( self.txPower ),
                         noneToNegInf( self.rxPower ) ]

      # Convert floats to strings so they can be printed
      for i, val in enumerate( fieldValues ):
         if val == float( "-inf" ):
            fieldValues[ i ] = "n/a"
         elif isinstance( val, float ):
            fieldValues[ i ] = f"{val:.2f}"

      if printFmt == 'default':
         table.newRow( *fieldValues )
      elif printFmt == 'csv':
         print( ",".join( fieldValues ) )

   def renderModelDetailed( self, table, intfName, param ):
      # Note that even though the CLI accepts the optional "csv"
      # keyword, the only difference between "CSV" and non-"CSV"
      # output is the heading. There is no difference in the
      # per-interface output.

      operValue = float( "-inf" )
      highAlarm = float( "-inf" )
      highWarn = float( "-inf" )
      lowAlarm = float( "-inf" )
      lowWarn = float( "-inf" )
      highAlarmOverridden = False
      highWarnOverridden = False
      lowAlarmOverridden = False
      lowWarnOverridden = False

      thresh = None
      if param == "Temperature":
         thresh = self.details.temperature
         operValue = self.temperature
      elif param == "Voltage":
         thresh = self.details.voltage
         operValue = self.voltage
      elif param == "Current":
         thresh = self.details.txBias
         operValue = self.txBias
      elif param == "Tx Power":
         thresh = self.details.txPower
         operValue = self.txPower
      elif param == "Rx Power":
         thresh = self.details.rxPower
         operValue = self.rxPower
      elif param == "Total Rx Power":
         thresh = self.details.totalRxPower
         operValue = self.totalRxPower

      if thresh:
         # Some xcvr types support only a subset of the thresholds in
         # the TAC model.
         highAlarm = thresh.highAlarm
         highAlarmOverridden = thresh.highAlarmOverridden
         highWarn = thresh.highWarn
         highWarnOverridden = thresh.highWarnOverridden
         lowAlarm = thresh.lowAlarm
         lowAlarmOverridden = thresh.lowAlarmOverridden
         lowWarn = thresh.lowWarn
         lowWarnOverridden = thresh.lowWarnOverridden

      operValue = noneToNegInf( operValue )
      highAlarm = noneToNegInf( highAlarm )
      highWarn = noneToNegInf( highWarn )
      lowAlarm = noneToNegInf( lowAlarm )
      lowWarn = noneToNegInf( lowWarn )

      # Build out the row output.
      rowOutput = [ self.slot, self.channel, f"{operValue:.2f}" ]
      # Adding all threshold strings
      for threshold, overridden in [ ( highAlarm, highAlarmOverridden ),
                                     ( highWarn, highWarnOverridden ),
                                     ( lowAlarm, lowAlarmOverridden ),
                                     ( lowWarn, lowWarnOverridden ) ]:
         # Add a space before the asterisk to treat it as a unit so that TableOutput
         # dotAlign can work.
         rowOutput.append( f"{threshold:.2f}" + ( "*" if overridden else "" ) )
      # Convert all -inf to n/a.
      rowOutput = [ "n/a" if val == "-inf" else val for val in rowOutput ]
      if self.narrowBand or param != "Total Rx Power":
         # Show "Total Rx Power" section only if transceiver is narrowband.
         table.newRow( *rowOutput )

class InterfacesTransceiver( Model ):
   _printFmt = Enum( values=SUPPORTED_FORMATS, help="Type of print format" )
   _detailed = Bool( help="Include warning and alarm thresholds" )
   interfaces = Dict( keyType=Interface, valueType=InterfaceTransceiverBase,
                     help="Mapping between Interface name and the transceiver info" )
   _domThresholdOverrideEnabled = Bool( help="Indicates whether the DOM threshold"
                                             " override is enabled" )

   def render( self ):
      if self._detailed:
         self._renderDetailed()
      else:
         self._renderNotDetailed()

   def _printThresHeaderDefault( self ):
      print( "Please consider using show interfaces transceiver dom thresholds." )
      print( "mA: milliamperes, dBm: decibels (milliwatts), "
            + "n/a: not applicable." )
      print( "A2D readouts (if they differ), are reported in parentheses." )
      print( "The threshold values are calibrated." )
      if self._domThresholdOverrideEnabled:
         print( "(*) Threshold has been overridden via configuration." )

   def _renderDetailed( self ):
      assert self._printFmt in SUPPORTED_FORMATS, "Unrecognized DOM output format"

      if not self.interfaces:
         return

      if self._printFmt == "default":
         self._printThresHeaderDefault()

      paramUnitList = [ ( 'Temperature', 'Celsius' ),
                        ( 'Voltage', 'Volts' ),
                        ( 'Current', 'mA' ),
                        ( 'Tx Power', 'dBm' ),
                        ( 'Rx Power', 'dBm' ) ]

      for interface in self.interfaces.values():
         if isinstance( interface, InterfaceTransceiver ) and interface.narrowBand:
            paramUnitList.append( ( 'Total Rx Power', 'dBm' ) )
            break

      for ( param, unit ) in paramUnitList:
         table = _createThreshParamTable( param, unit )
         for intf in Arnet.sortIntf( self.interfaces ):
            self.interfaces[ intf ].renderModelDetailed( table, intf, param )
         print( table.output(), end="" )

   def _renderNotDetailed( self ):
      assert self._printFmt in SUPPORTED_FORMATS, "Unrecognized DOM output format"
      printHeaderFn = { "default": _printHeaderDefault,
                        "csv": _printHeaderCsv }

      if not self.interfaces:
         # If there are no interfaces, skip even printing headers.
         return

      table = _createTable()

      printHeaderFn[ self._printFmt ]()
      for intf in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ intf ].renderModelNoDetail( table, intf, self._printFmt )

      if self._printFmt == 'default':
         print( table.output(), end="" )
