#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Cell
import EthIntfLib
import LazyMount
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.XcvrAllStatusDir
from CliPlugin import IntfCli
from CliPlugin.EthIntfCli import xcvrKw, xcvrShowKw
from CliPlugin.XcvrConfigCli import ( xcvrCmisGuard,
                                      diagKeyword,
                                      intfToXcvrName,
                                      XcvrConfigModelet )
import TypeFuture
from XcvrPrbsLib import ( prbsCliHostKw, prbsCliMediaKw, prbsCliGeneratorKw,
                          prbsCliCheckerKw, fecEnableKw, testKw,
                          patternKw )
from XcvrPrbsLib import ( PatternTypeEnum )

import CliGlobal

gv = CliGlobal.CliGlobal( testPatternSliceDir=None, xcvrStatusDir=None )
PresenceEnum = TypeFuture.TacLazyType( "Xcvr::XcvrPresence" )

# -------------------------------------------------------------------------------
#
# Common helper functions
#
# -------------------------------------------------------------------------------

def getTestPatternStatusDir( intfName ):
   isModular = Cell.cellType() == "supervisor"
   sliceName = "FixedSystem"
   if isModular and not Tac.Type( "Arnet::MgmtIntfId" ).isMgmtIntfId( intfName ):
      sliceName = EthIntfLib.sliceName( intfName )
   return gv.testPatternSliceDir.get( sliceName )

# Guards
def xcvrPrbsGuard( mode, token ):

   # Guard on CMIS modules and primary interfaces
   cmisGuardResult = xcvrCmisGuard( mode, token )
   if cmisGuardResult is not None:
      return cmisGuardResult

   # We need to make sure that we accept all commands if the module is not
   # inserted ( as long as the dut has cmis slots )
   xcvrName = intfToXcvrName( mode.intf.name )
   if not xcvrName:
      return None
   status = gv.xcvrStatusDir.xcvrStatus.get( xcvrName )
   if status.presence != PresenceEnum.xcvrPresent:
      return None

   # Guard on modules with no test pattern support ( evidenced by a lack of
   # test pattern capabilities ).
   intfName = mode.intf.name
   testPatternStatusDir = getTestPatternStatusDir( intfName )
   if intfName not in testPatternStatusDir.capabilities:
      return CliParser.guardNotThisPlatform
   return None

# Command family matchers
testKwDescription = 'Configure the interface in test mode'
testKwMatcher = CliMatcher.KeywordMatcher( testKw, helpdesc=testKwDescription )

prbsCmdDescription = 'PRBS test pattern generator/checker'
prbsCmdMatcher = CliMatcher.KeywordMatcher( patternKw, helpdesc=prbsCmdDescription )
prbsCmdGuardedMatcher = CliCommand.Node( matcher=prbsCmdMatcher,
                                         guard=xcvrPrbsGuard )

capabilitiesKw = 'capabilities'
capabilitiesDesc = 'A list of test pattern features supported by the module'
capabilitiesMatcher = CliMatcher.KeywordMatcher( capabilitiesKw,
                                                 helpdesc=capabilitiesDesc )

countersKw = 'counters'
countersDesc = 'Test Pattern results/statistics'
countersMatcher = CliMatcher.KeywordMatcher( countersKw,
                                             helpdesc=countersDesc )

detailKw = 'detail'
detailDesc = 'Detailed version of test pattern results/statistics'
detailMatcher = CliMatcher.KeywordMatcher( detailKw, helpdesc=detailDesc )

# ------------------------------------------------------------------------------
#
# [ no | default ] transceiver diag test pattern <PATTERN>
# {line | system} {transmitter | receiver} [fec]
#
# --------------------------------------------------------------------------------

# Matcher for the pseudo-random bit sequence the user wants to send / receive
# across the link
sequenceTypeMatcher = CliMatcher.EnumMatcher(
      { prbsType: f'Configure the {prbsType} test pattern'
        for prbsType in PatternTypeEnum.attributes } )

# Matcher for the side/link that the user wants activated
hostOrMediaMatcher = CliMatcher.EnumMatcher( {
   prbsCliHostKw: 'Physical link between the module and the phy. '
                  'Equivalent to the host side.',
   prbsCliMediaKw: 'Physical link between the module and its link partner. '
                   'Equivalent to the media side.'
   }
 )

# Matcher for the side functionality that the user wants activated,
# Only the generator can produce a pattern, and only the checker
# can fill the bit error counters
genOrCheckMatcher = CliMatcher.EnumMatcher( {
   prbsCliGeneratorKw: 'Generates and transmits the test pattern sequence. '
                       'Equivalent to the generator.',
   prbsCliCheckerKw: 'Receives, parses and counts errors in test pattern sequence. '
                     'Equivalent to the checker. '
   }
 )

class TransceiverDiagTestPatternCmd( CliCommand.CliCommandClass ):
   syntax = ( 'transceiver diag test PATTERN_GUARDED '
              'PATTERN_TYPE HOST_MEDIA GENERATOR_CHECKER [ fec ]' )

   # You have two modes of operation for the no/default command:
   # Mode 1 deletes all existing configuration by removing the underlying object
   # Mode 2 only deletes a specific side config by replacing
   # the object with its default/inactive version
   noOrDefaultSyntax = ( 'transceiver diag test PATTERN_UNGUARDED '
                         '[ PATTERN_TYPE ] [ HOST_MEDIA GENERATOR_CHECKER ] ...' )

   data = {
      'transceiver': xcvrKw,
      'diag': diagKeyword,
      'test': testKwMatcher,
      'PATTERN_UNGUARDED': prbsCmdMatcher,
      'PATTERN_GUARDED': prbsCmdGuardedMatcher,
      'HOST_MEDIA': hostOrMediaMatcher,
      'GENERATOR_CHECKER': genOrCheckMatcher,
      'PATTERN_TYPE': sequenceTypeMatcher,
      fecEnableKw: 'Enable Forward Error Correction'
   }

   handler = "XcvrPrbsCliHandler.transceiverDiagTestPatternCmdHandler"

   noOrDefaultHandler = ( "XcvrPrbsCliHandler."
      "transceiverDiagTestPatternCmdNoOrDefaultHandler" )

# ------------------------------------------------------------------------------
#
# show interfaces [<name>] transceiver diag test pattern
#
# ------------------------------------------------------------------------------

class ShowIntfXcvrTestPatternConfiguration( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces transceiver diag test pattern'
   data = { "transceiver": xcvrShowKw,
            "diag": diagKeyword,
            "test": testKw,
            "pattern": prbsCmdMatcher }
   handler = ( "XcvrPrbsCliHandler."
      "showInterfacesTransceiverTestPatternConfiguration" )
   cliModel = ( "XcvrPrbsModel."
      "InterfacesTransceiverTestPatternConfigurationCollection" )

# ------------------------------------------------------------------------------
#
# show interfaces [<name>] transceiver diag test pattern capabilities
#
# ------------------------------------------------------------------------------

class ShowIntfXcvrTestPatternCapabilities( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces transceiver diag test pattern capabilities'
   data = { "transceiver": xcvrShowKw,
            "diag": diagKeyword,
            "test": testKw,
            "pattern": prbsCmdMatcher,
            "capabilities": capabilitiesMatcher }
   handler = ( "XcvrPrbsCliHandler."
      "showInterfacesTransceiverTestPatternCapabilities" )
   cliModel = ( "XcvrPrbsModel."
      "InterfacesTransceiverTestPatternCapabilitiesCollection" )

# ------------------------------------------------------------------------------
#
# show interfaces [<name>] transceiver diag test pattern counters
#
# ------------------------------------------------------------------------------

class ShowIntfXcvrTestPatternCounters( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces transceiver diag test pattern counters [ detail ]'
   data = { 'transceiver': xcvrShowKw,
            'diag': diagKeyword,
            'test': testKw,
            'pattern': prbsCmdMatcher,
            'counters': countersMatcher,
            'detail': detailMatcher }

   handler = ( "XcvrPrbsCliHandler."
      "showInterfacesTransceiverTestPatternCounters" )
   cliModel = ( "XcvrPrbsModel."
      "InterfacesTransceiverTestPatternCountersCollection" )

# ------------------------------------------------------------------------------
#
# Plugin Method
#
# ------------------------------------------------------------------------------

XcvrConfigModelet.addCommandClass( TransceiverDiagTestPatternCmd )
BasicCli.addShowCommandClass( ShowIntfXcvrTestPatternCapabilities )
BasicCli.addShowCommandClass( ShowIntfXcvrTestPatternConfiguration )
BasicCli.addShowCommandClass( ShowIntfXcvrTestPatternCounters )

def Plugin( em ):
   gv.testPatternSliceDir = LazyMount.mount( em,
         "hardware/archer/xcvr/status/prbs/cmis/slice", "Tac::Dir", "ri" )
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
