# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import OrderedDict, namedtuple
from typing import Literal, Optional, Callable

import Arnet
import CliGlobal
import ConfigMount
import Tac
import Tracing
from TypeFuture import TacLazyType

import CliPlugin.XcvrAllStatusDir
from CliPlugin.XcvrCliLib import ( IntfMapping, isDualLaserModule,
                                  getAllIntfsWrapper, getLineSystemStatus )
from CliPlugin.XcvrModel import ( InterfaceTransceiverDetailThresholds,
                                  SUPPORTED_FORMATS )
from CliPlugin.XcvrShowDomModel import ( InterfacesTransceiverDom,
                                        InterfaceTransceiverDom,
                                        InterfaceTransceiverDomBase,
                                        InterfaceTransceiverDomParameter )
from XcvrLib import ( getLaneId, getXcvrStatus, isCmisTransceiver, negInfToNone,
                      getXcvrSlotName )

t0 = Tracing.trace0

XcvrType = TacLazyType( 'Xcvr::XcvrType' )
XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )
DomParamType = TacLazyType( 'Xcvr::EnhancedDomParamType' )
DomParamScope = TacLazyType( 'Xcvr::EnhancedDomParamScope' )
DomThreshold = TacLazyType( 'Xcvr::DomThreshold' )
QsfpLsDomScalarHal = TacLazyType( 'Xcvr::HalQsfpLsDomScalingFactorExponent' )
gv = CliGlobal.CliGlobal( xcvrStatusDir=None, xgc=None )

DomParameter = namedtuple( "DomParameter", [ "paramSupported", "paramName",
                                             "paramUnit", "paramValueGetter" ] )

# ----------------------------------------------------------------------------
#
# "show interfaces [ <interface> ] transceiver dom [ thresholds ]"
#
# ----------------------------------------------------------------------------

def xcvrDomMinPowerThreshold( status, param ) -> float:
   '''
   Determines the value to represent 'no power', aka, the DOM floor.
   In most cases, -30dBm is sufficient. However, there are some
   exceptions.
    - when values less than or equal to -30dBm are expected
    - for coherent modules
   Generically written to handle both rx and tx floors but at
   this time, only rx can be <-30dBm so tx option is not used.
   '''
   assert param in [ 'rx',
                     # 'tx' - Enable if needed and update ShowIntfXcvrDomFloorTest
                     # btest to cover.
                    ]

   # Special case for Coherent dual-laser (CFP2-DP04)
   if isDualLaserModule( status ) and param == 'rx':
      return Tac.Value( "Xcvr::PowerConstants" ).minimumCoherentRxPower

   PowerConstants = Tac.Type( "Xcvr::PowerConstants" )
   defaultMinPower = getattr( PowerConstants,
                              "minimum" + param.capitalize() + "Power" )

   if not status.domThresholdValid:
      return defaultMinPower

   domThreshold = status.domThreshold

   lowAlarm = negInfToNone( getattr( domThreshold, param + "PowerLowAlarm" ) )
   if not lowAlarm:
      return defaultMinPower

   if defaultMinPower >= lowAlarm:
      return Tac.Value( "Xcvr::PowerConstants" ).minimumRepresentablePower

   return defaultMinPower

def _getEnhancedThresholdModel( domThreshold, attr: str,
                                overrideEntry=None ) ->\
                                Optional[ InterfaceTransceiverDetailThresholds ]:

   def getDomThresholdData( thresholdSuffix: str ) ->\
                            tuple[ Optional[ float ], bool ]:
      attributeName = attr + thresholdSuffix
      return ( getattr( domThreshold, attributeName, None ),
               ( overrideEntry is not None and getattr( overrideEntry,
                                                       attributeName, None
                                                       ) is not None ) )

   alternativePrefixes = { "temperature": "temp" }
   attr = alternativePrefixes.get( attr, attr ) # Get alternative alarm prefix

   highAlarm, highAlarmO = getDomThresholdData( "HighAlarm" )
   highWarn, highWarnO = getDomThresholdData( "HighWarn" )
   lowAlarm, lowAlarmO = getDomThresholdData( "LowAlarm" )
   lowWarn, lowWarnO = getDomThresholdData( "LowWarn" )
   if None not in ( highAlarm, highWarn, lowAlarm, lowWarn ):
      return InterfaceTransceiverDetailThresholds( highAlarm=negInfToNone(
                                                      highAlarm ),
                                                   highWarn=negInfToNone( highWarn ),
                                                   lowAlarm=negInfToNone( lowAlarm ),
                                                   lowWarn=negInfToNone( lowWarn ),
                                                   highAlarmOverridden=highAlarmO,
                                                   highWarnOverridden=highWarnO,
                                                   lowAlarmOverridden=lowAlarmO,
                                                   lowWarnOverridden=lowWarnO )
   return None

def isDomThresholdUsed( status, paramType, thresholds: bool ) -> bool:
   paramTypeThresholdSupported = True
   if ( isDualLaserModule( status ) and
        paramType == Tac.Type( "Xcvr::EnhancedDomParamType" ).rxPower ):
      paramTypeThresholdSupported = False
   return thresholds and status.domThresholdValid and paramTypeThresholdSupported

def getXcvrParamType( status, paramType: str, isThreshold: bool = False ) -> str:
   """
   Helper method for reformatting parameter names to be used in
   paramTypeToStr to get exact output format

   Parameters
   ----------
   status : Xcvr::XcvrNewStatus object for corresponding module.
   paramType : str
      Name of general dom parameter that may need to be remapped.
   isThreshold : bool
      When true, this indicates that there is a difference in parameter naming
         between the dom parameter and its corresponding threshold attribute names.
         Return back the correct dom threshold param type name for this module.
   """

   domParamType = Tac.Type( "Xcvr::EnhancedDomParamType" )

   # Convert some params common for all coherent systems
   if( status.isCoherent( status.mediaType ) or
       isCmisTransceiver( status ) and
       status.vdmCapabilities.vdmSupported ):

      paramTypeToLabelEnhancedDom = {
         domParamType.temperature: 'caseTemperature',
         domParamType.laserTemp: 'laserTemperature',
         domParamType.laserTempAux: 'laserTemperature',
         domParamType.preFecBERAvg: 'preFECBERAvg',
         domParamType.preFecBERMin: 'preFECBERMin',
         domParamType.preFecBERMax: 'preFECBERMax',
      }

      if status.isCoherent( status.mediaType ):
         paramTypeToLabelCoherent = {
            domParamType.txPower: 'coherentTxPower',
            domParamType.rxSignalPower: 'rxChannelPower',
            domParamType.rxTotalPower: 'totalRxPower',
         }
         paramTypeToLabelEnhancedDom.update( paramTypeToLabelCoherent )

      if paramType in paramTypeToLabelEnhancedDom and not isThreshold:
         # paramTypes may have mismatching names between its
         # corresponding threshold attribute names. E.g. For DP04 DCO modules,
         # txPower is actually coherentTxPower while its thresholds still
         # reference txPower[High/Low][Alarm/Warn].
         return paramTypeToLabelEnhancedDom[ paramType ]

   # Convert some params of dual laser systems
   if isDualLaserModule( status ):
      paramTypeToLabelDualLaser = {
         domParamType.rxPower: 'rxChannelPower',
         domParamType.laserAge: 'txLaserAge',
      }
      if paramType in paramTypeToLabelDualLaser:
         return paramTypeToLabelDualLaser[ paramType ]

   # Default case - no conversion
   return paramType

def getXcvrChannel( status, channel ) -> str:
   """
   Helper method for reformatting appropriate channel to be printed.
   """
   if status.isCoherent( status.mediaType ):
      channel = '-'
   return channel

def _showInterfacesXcvrDom( mode, intf=None,
      mod=None, outputFmt=None, thresholds=False,
      onlyRenderPols=False ) -> InterfacesTransceiverDom:
   res = InterfacesTransceiverDom( _printFmt=outputFmt, _detailed=thresholds,
               _domThresholdOverrideEnabled=gv.xgc.domThresholdOverrideEnabled )
   assert outputFmt in SUPPORTED_FORMATS, "Unrecognized DOM output format"
   ( intfs, _ ) = getAllIntfsWrapper( mode, intf, mod )

   if not intfs:
      return res

   # Get the laneMappings for all the active interfaces
   intfMapping = IntfMapping( intfs )
   intfLaneMapping = intfMapping.intfMap
   if not intfLaneMapping:
      return res

   # Get the overriden display Names of the interfaces
   intfNameOverrideXcvrDom = intfMapping.intfNameOverrideXcvrDom

   if not intfLaneMapping:
      return res

   for intfToDisplay in Arnet.sortIntf( intfLaneMapping ):
      for intfName in Arnet.sortIntf( intfLaneMapping[ intfToDisplay ] ):
         channels = intfLaneMapping[ intfToDisplay ][ intfName ]
         xS = gv.xcvrStatusDir.xcvrStatus.get( getXcvrSlotName( intfName ) )
         xS = getXcvrStatus( xS )

         if not xS.xcvrMgrCapabilities.domDynamicValues:
            continue

         if xS.presence != "xcvrPresent":
            # Skip transceivers that aren't present
            continue

         if intfName in res.interfaces or not channels:
            continue

         laneId = getLaneId( intfName )
         laneId = laneId - 1 if laneId else laneId

         # For Dp04 modules, we display all of the shared module dom info using only
         # the primary interface at a global port level in the Cli output.
         if isDualLaserModule( xS ) and laneId != 0:
            continue

         domModel = _getIntfXcvrDomInfo( xS, laneId,
                                         channels, thresholds )
         domModel.displayName = intfNameOverrideXcvrDom.get( intfName ) or intfName
         # pylint:disable-msg=protected-access
         if not onlyRenderPols or ( onlyRenderPols and domModel._usePolsLabels ):
            res.interfaces[ intfName ] = domModel
            res._interfacesOrder.append( intfName )

   return res

# Populate and return the DOM model for the given xcvrStatus.
def _getIntfXcvrDomInfo( status, laneId: int, channels: list[ int ],
                         thresholds: bool ) -> InterfaceTransceiverDomBase:
   """
   Parameters
   ------------------
   status: Xcvr::XcvrNewStatus
   laneId: int
   channels: list[ int ]
   thresholds: bool
   Returns
   ------------------
   InterfaceTransceiverDomBase | InterfaceTransceiverDom
   """
   # Populate N/A for all params on interfaces with invalid domInfo.
   xcvrDomModel = InterfaceTransceiverDomBase()
   if status.domInfoValid[ laneId ]:
      # Populate the legacy DOM parameters such as temperature, voltage etc.
      xcvrDomModel = _xcvrCommonDomInfo( status, laneId, channels, thresholds )
      # Append enhanced dom parameters if they are present,
      # 100GE-DWDM2 and some of 100GBASE-FR/LR/DR.
      if( status.typeInUse == XcvrType.qsfpPlus and
          status.isEnhancedDomSupported() ):
         return _xcvrEnhancedDomInfo( status, thresholds, xcvrDomModel )

      # Append enhanced dom parameters if they are present ( 400GBASE-ZR )
      elif( isCmisTransceiver( status ) and
            status.vdmCapabilities.vdmSupported ):
         return _xcvrCmisEnhancedDomInfo( status, laneId, channels,
                                          thresholds, xcvrDomModel )

      # Append DCO DOM parameters.
      elif ( status.mediaType in ( XcvrMediaType.xcvr100GDwdmDco,
                                   XcvrMediaType.xcvr400GDwdmDco ) ):
         return _xcvrCfp2DcoDomInfo( status, xcvrDomModel, thresholds )
   return xcvrDomModel

def insertParameterInModel( model: InterfaceTransceiverDom,
                            paramType: str,
                            unit: Literal[ "C", "V", "dBm", "mA" ],
                            channel: str,
                            value: Optional[ float ],
                            threshold:
                            Optional[ InterfaceTransceiverDetailThresholds ] = None
                            ) -> InterfaceTransceiverDom:
   '''
   Helper function to insert a parameter in a InterfaceTransceiverDom
   If parameter exists, we are probably adding value for another channel
   If parameter doesn't exist, create one.

   Parameters
   ----------
   model: InterfaceTransceiverDom
   paramType: Str
      String of the paramType. For example, this may be "txPower" or "totalRxPower"
   unit: str
      Unit corresponding to the value passed in
   channel: str
      The channel corresponding to the value passed in
   value: float | None
   threshold: InterfaceTransceiverDetailThresholds | None
      Cli model that contains the threshold values
   '''
   # Set value to None if is "-inf". This prevents -inf from entering into the model.
   value = negInfToNone( value )
   if paramType in model.parameters:
      model.parameters[ paramType ].channels[ channel ] = value
   else:
      newParam = InterfaceTransceiverDomParameter()
      newParam.unit = unit
      newParam.channels[ channel ] = value
      newParam.threshold = threshold
      model.parameters[ paramType ] = newParam
   return model

def _addTotalRxPowerToModel( status, model: InterfaceTransceiverDom )\
      -> InterfaceTransceiverDom:
   if isDualLaserModule( status ):
      paramType = 'totalRxPower'
      paramChannel = '-'
      paramUnit = 'dBm'
      domInfo = status.domInfo[ 0 ]
      minimumRxPowerConstant = \
         Tac.Value( "Xcvr::PowerConstants" ).minimumCoherentRxPower
      if ( domInfo.totalRxPower < minimumRxPowerConstant and
           status.domCapabilities.totalRxPower ):
         totalRxPower = minimumRxPowerConstant
      else:
         totalRxPower = domInfo.totalRxPower
      _threshold = _getEnhancedThresholdModel( status.domThreshold, paramType )
      model = insertParameterInModel( model, paramType, paramUnit,
                                       paramChannel, totalRxPower, _threshold )
   return model

def copyRxPowerThresholds( channelThresholds, newDomThresholds ):
   ''' Helper method to populate the DOM threshold values with new Rx power
   DOM threshold values.

   Params:
   ------
   channelThresholds : Xcvr::DomThresholds
      DOM threshold of the channel.
   newDomThresholds : Xcvr::DomThresholds
      Place-holder of new DOM threshold.

   Returns:
   -------
   channelThresholds : Xcvr::DomThresholds
      Modified channelThreshold
   '''
   channelThresholds = Tac.nonConst( channelThresholds )
   for thresholdType in [ 'HighAlarm', 'HighWarn', 'LowAlarm', 'LowWarn' ]:
      thresholdName = "rxPower" + thresholdType
      setattr( channelThresholds, thresholdName,
         getattr( newDomThresholds, thresholdName, None ) )
   return channelThresholds

# Helper methods for _xcvrCommonDomInfo to deal with alternate code paths required
# to support POLS modules
def _channelDomThresholdsHelper( status, channel: int ):
   '''
   Helper method to return the appropriate DOM thresholds attribute depending on
   input factors:
     * Conventional optics will want the standard thresholds found at
       status.domThreshold
     * OSFP-LS want the standard thresholds for the booster datapath (channel 1),
       and the parsed alternate thresholds for the pre-amp datapath  (channel 2)
     * QSFP-LS want the same thing as OSFP-LS, but the parsed alternate thresholds
       are stored elsewhere in its xcvrStatus object.

   Parameters
   ----------
   status : Xcvr::QsfpNewStatus or Xcvr::CmisStatus
   channel : int, corresponding to a datapath/optical lane on the module; starts
             from 1

   Returns
   -------
   Xcvr::DomThreshold instance
   '''
   if status.mediaType == XcvrMediaType.xcvrAmpZr:
      if channel == 2:
         # On the pre-amp/splitter channel, LS modules use alternate DOM thresholds.
         # OSFP-LS vs QSFP-LS use different attributes to store alternate thresholds.
         if status.xcvrType == XcvrType.osfp:
            return status.eepromContents.altDomThreshold
         if status.xcvrType == XcvrType.qsfpPlus:
            return status.altDomThreshold
      if channel == 1 and status.xcvrType == XcvrType.qsfpPlus:
         # On QSFP-LS, the booster thresholds are not published statically and thus
         # are not scaled in the XcvrAgent. Introduce the scaling factor here.
         correctedThresh = Tac.nonConst( status.domThreshold )
         powerAttrs = [ i for i in correctedThresh.attributes
                        if i.startswith( ( "txPower", "rxPower" ) ) ]
         for attr in powerAttrs:
            paramValue = getattr( correctedThresh, attr )
            paramValue = _correctQsfpLsTxRxPowerData( status, paramValue )
            setattr( correctedThresh, attr, paramValue )
         return correctedThresh
   return status.domThreshold

def _correctOsfpLsRxPowerData( status, channel: int ) -> tuple:
   '''
   Helper method to fix RX-power-related data for OSFP-LS.

   For context, POLS modules have two data paths -- a booster and a pre-amp. DOM
   data for the booster and pre-amp are encoded differently between the OSFP-
   and QSFP-LS.

   For QSFP-LS, lane 1 DOM corresponds to the booster, and lane 2 DOM
   corresponds to the pre-amp. The pre-amp data uses the alternate DOM thresholds
   read from upper page 2. This organization is naturally supported by code for
   conventional optics.

   OSFP-LS has a more complex scheme. Lane 1 refers to line-side DOM and lane 2
   refers to local-side DOM. Line-side DOM consists of the booster's TX and the
   pre-amp's RX. Local-side DOM consists of the booster's RX and the pre-amp's TX.
   This means that what we publish as "Channel 1 (booster) Rx power" comes from
   lane 2, and same for "Channel 2 (pre-amp) Rx power" and lane 1. Performing the
   swap here instead of in the CAPI model seems easier.

   Parameters
   ----------
   status : Xcvr::XcvrNewStatus-derived object
   channel : int, corresponding to a datapath/optical lane on the module; starts
             from 1

   Returns
   -------
   rxPower : float, reported RX power from module converted to dBm
   channelThreshold : Xcvr::DomThreshold
   '''
   channelThreshold = _channelDomThresholdsHelper( status, channel )
   if channel == 1:
      # Pull value from channel 2, note, domInfo is indexed with 1 for the
      # channel.
      rxPower = status.domInfo[ 1 ].rxPower
      # We also have to swap the thresholds for rxPower only.
      domThresholdCh2 = status.eepromContents.altDomThreshold
      channelThreshold = copyRxPowerThresholds( channelThreshold,
                                                domThresholdCh2 )
   elif channel == 2:
      # Pull value from channel 1, note, domInfo is indexed with 0
      # for the channel.
      rxPower = status.domInfo[ 0 ].rxPower
      # We also have to swap the thresholds for rxPower only.
      channelThreshold = copyRxPowerThresholds( channelThreshold,
                                                status.domThreshold )
   else:
      # We shouldn't ever find ourselves here, but set up a default return value
      # for rxPower just in case instead of risking some kind of error in the CLI.
      # status.domInfo is 0-indexed while 'channel' is 1-indexed.
      rxPower = status.domInfo[ channel - 1 ].rxPower
   return rxPower, channelThreshold

def _correctQsfpLsTxRxPowerData( status, paramValue: float ) -> float:
   '''
   Helper method to fix laser powers reported by QSFP-LS

   Powers reported by POLS at their extremes go out of the bounds of what is
   encodable by the spec. The vendor scaled down the reported powers by a factor
   encoded in the EEPROM. This function adds back to scaling offset.

   Parameters
   ----------
   status : Xcvr::XcvrNewStatus-derived object
   paramValue : float, raw TX/RX power value as reported by the EEPROM

   Returns
   -------
   float, corrected TX/RX power with scaling factor included
   '''
   eepromContents = status.actualIdEepromContents
   page0UpperBa = bytearray( eepromContents.eepromPage00UpperData )
   scalarHal = QsfpLsDomScalarHal( page0UpperBa[ QsfpLsDomScalarHal.offset -
                                                 eepromContents.eepromPageSize ] )
   return scalarHal.addPowerScalar( paramValue )

def addSimpleDomParameter( model: InterfaceTransceiverDom, status, supported: bool,
                           paramName: str, unit: str, thresholdPrefix: str,
                           value: Callable, useThresholds: bool )\
                           -> InterfaceTransceiverDom:
   if not supported:
      return model
   paramType = getXcvrParamType( status, paramName )
   paramChannel = '-'
   thresholdModel = None
   if ( isDomThresholdUsed( status, paramName, useThresholds ) and
        thresholdPrefix is not None ):
      thresholdModel = _getEnhancedThresholdModel( status.domThreshold,
                                                   thresholdPrefix,
                                                   status.domThresholdOverrideEntry )
   return insertParameterInModel( model, paramType, unit,
                                  paramChannel, value(), thresholdModel )

# Create and populate legacy DOM parameters such as temperature, voltage etc...
# and return the model.
def _xcvrCommonDomInfo( status, laneId: int, channels: list[ int ],
                        useThresholds: bool ) -> InterfaceTransceiverDom:
   """
   Parameters
   ----------------
   status: XcvrNewStatus-derived status
   domInfo: Xcvr::DomInfo
   laneId: int
   channels: list[ int ]
   useThresholds: bool

   Returns
   ----------------
   InterfaceTransceiverDom
   """
   model = InterfaceTransceiverDom(
      _dualLaserModulePresent=isDualLaserModule( status ) )
   model.vendorSn = status.vendorInfo.vendorSn.strip()
   model.mediaType = status.mediaTypeString.strip()
   polsModule = status.mediaType == XcvrMediaType.xcvrAmpZr
   domInfo = status.domInfo[ laneId ]
   model.updateTime = Tac.utcNow() - ( Tac.now() - domInfo.updateTime )
   domCapabilities = status.domCapabilities

   model = addSimpleDomParameter( model, status, domCapabilities.temperature,
                                 "temperature", "C", "temp",
                                 lambda: domInfo.temperature, useThresholds )
   model = addSimpleDomParameter( model, status, domCapabilities.voltage,
                                 "voltage", "V", "voltage",
                                 lambda: domInfo.voltage, useThresholds )
   model = addSimpleDomParameter( model, status, domCapabilities.aggTxPower,
                                 "aggrTxPwr", "dBm", None,
                                 lambda: status.domAggPower.aggTxPower,
                                 useThresholds )
   model = addSimpleDomParameter( model, status, domCapabilities.aggRxPower,
                                 "aggrRxPwr", "dBm", None,
                                 lambda: status.domAggPower.aggRxPower,
                                 useThresholds )

   for chan in channels:
      # channels passed in are indexed starting from 0...
      ch = chan + 1
      paramChannel = getXcvrChannel( status, str( ch ) )
      channelThreshold = _channelDomThresholdsHelper( status, ch )

      # Insert Tx-Bias in the model
      paramType = getXcvrParamType( status, 'txBias' )
      paramUnit = 'mA'
      if not status.isCoherent( status.mediaType ):
         paramValue = status.domInfo[ chan ].txBias
         if not domCapabilities.txBias:
            paramValue = float( "-inf" )
         _threshold = None
         if isDomThresholdUsed( status, 'txBias', useThresholds ):
            if domCapabilities.txBias:
               _threshold = _getEnhancedThresholdModel( channelThreshold, "txBias",
                                                 status.domThresholdOverrideEntry )
            else:
               _threshold = InterfaceTransceiverDetailThresholds( highAlarm=None,
                                                   highWarn=None, lowAlarm=None,
                                                   lowWarn=None )
         model = insertParameterInModel( model, paramType, paramUnit,
                                         paramChannel, paramValue, _threshold )

      # Insert Rx total power in the model if applicable
      model = _addTotalRxPowerToModel( status, model )

      # For coherent xcvrs, Insert Tx and RX Power in the model
      if not status.isCoherent( status.mediaType ) or isDualLaserModule( status ):
         # Insert Tx Power in the model
         paramType = getXcvrParamType( status, 'txPower' )
         paramUnit = 'dBm'
         paramValue = status.domInfo[ chan ].txPower
         if polsModule and status.xcvrType == XcvrType.qsfpPlus:
            # Correct QSFP-LS TX power values
            paramValue = _correctQsfpLsTxRxPowerData( status, paramValue )
         if ( ( paramValue == float( "-inf" ) or
                paramValue < Tac.Value( "Xcvr::PowerConstants" ).minimumTxPower ) and
              domCapabilities.txPower ):
            paramValue = Tac.Value( "Xcvr::PowerConstants" ).minimumTxPower
         _threshold = None
         if isDomThresholdUsed( status, 'txPower', useThresholds ):
            thresholdsParamType = getXcvrParamType( status, 'txPower',
                                                    isThreshold=True )
            _threshold = _getEnhancedThresholdModel( channelThreshold,
                                                   thresholdsParamType,
                                                   status.domThresholdOverrideEntry )

         model = insertParameterInModel( model, paramType, paramUnit,
                                         paramChannel, paramValue, _threshold )

         # Insert Rx Power in the model
         paramType = getXcvrParamType( status, 'rxPower' )
         paramUnit = 'dBm'
         paramValue = status.domInfo[ chan ].rxPower
         if polsModule:
            # Correct POLS RX power values
            if status.xcvrType == XcvrType.osfp:
               paramValue, channelThreshold = \
                     _correctOsfpLsRxPowerData( status, ch )
            elif status.xcvrType == XcvrType.qsfpPlus:
               paramValue = _correctQsfpLsTxRxPowerData( status, paramValue )

         minimumRxPowerConstant = xcvrDomMinPowerThreshold( status, "rx" )
         if ( ( paramValue == float( "-inf" ) or
                paramValue < minimumRxPowerConstant ) and
              domCapabilities.rxPower ):
            paramValue = minimumRxPowerConstant
         _threshold = None
         if isDomThresholdUsed( status, 'rxPower', useThresholds ):
            _threshold = _getEnhancedThresholdModel( channelThreshold, paramType,
                                                status.domThresholdOverrideEntry )
         model = insertParameterInModel( model, paramType, paramUnit,
                                         paramChannel, paramValue, _threshold )

      # If POLS, include Aux2/3 as laser temp for booster/pre-amp respectively
      if polsModule and chan in [ 0, 1 ]:
         # At the moment, laser temperature is published in different locations
         # between OSFP-LS and QSFP-LS. This block can simplify after OSFP-LS
         # starts publishing status to LineSystemStatus.
         if status.xcvrType == XcvrType.osfp:
            auxInfo = status.auxInfo
            auxThreshold = status.auxThreshold
         elif status.xcvrType == XcvrType.qsfpPlus:
            lsStatus = getLineSystemStatus( status )
            # If the command happens to run shortly after the LS is installed but
            # before the controller has constructed the status, the object may not
            # yet exist.
            if lsStatus:
               auxInfo = lsStatus.ampLaserTemperature
               auxThreshold = lsStatus.ampLaserTempThresholds
            else:
               # In this scenario, just ignore. LineSystemStatus will likely exist
               # for subsequent runs.
               t0( "LineSystemStatus not found for ", status )
               continue
         else:
            # Unexpected module type, ignore
            continue
         # At this point, we know we have laser-temperature information one way or
         # another, so only now can we safely add it to the model.
         paramType = 'laserTemperature'
         paramUnit = 'C'
         paramName = { 0: "aux2", 1: "aux3" }[ chan ]
         paramValue = getattr( auxInfo, f"{paramName}Val" )
         if isDomThresholdUsed( status, 'laserTemperature', useThresholds ):
            _threshold = _getEnhancedThresholdModel( auxThreshold, paramName )
         model = insertParameterInModel( model, paramType, paramUnit,
                                         paramChannel, paramValue, _threshold )

   # pylint: disable-msg=protected-access
   model._usePolsLabels = polsModule
   model._paramOrder = list( model.parameters )
   model._hasMultiChannelDomParam = any( len( v.channels ) > 1 for v in
                                         model.parameters.values() )
   return model

# Appends the Cmis Enhanced DOM parameters to the xcvr DOM model that is passed in.
def _xcvrCmisEnhancedDomInfo( status, laneId: int, channels: list[ int ],
                              thresholds: bool, model: InterfaceTransceiverDom )\
                              -> InterfaceTransceiverDom:
   """
   Parameters
   ----------------
   status: Xcvr::XcvrNewStatus
   domInfo: Xcvr::DomInfo
   laneId: int
   channels: list[ int ]
   thresholds: bool

   Returns
   ----------------
   InterfaceTransceiverDom
   """
   _threshold = None
   addThreshold = thresholds
   enhancedDomVdmModule = Tac.enumValue( "Xcvr::EnhancedDomVdmParamLane",
                                          "enhancedDomVdmModule" )
   if status.enhancedDom:
      for idx, extDomConfig in status.enhancedDom.domConfig.items():
         extDomInfo = status.enhancedDom.domInfo[ idx ]
         if ( extDomInfo.valid and
              ( extDomConfig.lane in channels or
                extDomConfig.lane == enhancedDomVdmModule or
                ( status.capabilities.muxponderModeCapable and
                  status.muxponderMode ) ) ):

            # Coherent modules only display a subset of available parameters
            if status.isCoherent( status.mediaType ) and not extDomInfo.readable:
               continue
            # Statistical and Host observables are displayed by other commands
            if( extDomConfig.scope == DomParamScope.enhancedDomScopeHost or
                extDomInfo.perfMon ):
               continue

            paramType = getXcvrParamType( status, extDomConfig.paramType )
            paramUnit = extDomConfig.paramUnit
            paramValue = extDomInfo.paramValue
            # channels passed in are indexed starting from 0...
            paramChannel = str( extDomConfig.lane + 1 )
            # 400G-ZR shows all media parameters as 'globals'
            if( status.isCoherent( status.mediaType ) or
                extDomConfig.lane == enhancedDomVdmModule ):
               paramChannel = '-'

            _threshold = None
            # 400G-ZR implements only small subset of VDM thresholds
            thresholdValid = True
            if status.isCoherent( status.mediaType ):
               thresholdValid = extDomConfig.paramType in (
                  DomParamType.laserTemp,
                  DomParamType.rxTotalPower,
                  DomParamType.rxSignalPower )

            if ( addThreshold and thresholdValid and
                 idx in status.enhancedDom.domThresholds ):
               paramThresholds = status.enhancedDom.domThresholds[ idx ]
               # These two parameters don't have low alarm and low warn thresholds
               if paramType in ( DomParamType.preFecBERCurr,
                                 DomParamType.errFramesCur ):
                  _threshold = InterfaceTransceiverDetailThresholds(
                     highAlarm=paramThresholds.highAlarm,
                     highWarn=paramThresholds.highWarn )
               else:
                  _threshold = InterfaceTransceiverDetailThresholds(
                     highAlarm=paramThresholds.highAlarm,
                     highWarn=paramThresholds.highWarn,
                     lowWarn=paramThresholds.lowWarn,
                     lowAlarm=paramThresholds.lowAlarm )
            model = insertParameterInModel( model, paramType, paramUnit,
                                            paramChannel, paramValue, _threshold )

   # pylint: disable-msg=protected-access
   model._paramOrder = list( model.parameters )
   model._hasMultiChannelDomParam = any( len( v.channels ) > 1 for v in
                                         model.parameters.values() )
   return model

# Appends the DCO DOM parameters to the xcvr DOM model that is passed in.
def _xcvrCfp2DcoDomInfo( status, model: InterfaceTransceiverDom, thresholds: bool )\
                         -> InterfaceTransceiverDom:
   cfp2DcoStatus = status.cfp2DcoNewStatus
   # True when at least one dom parameter exists for multiple channels.
   isDp04 = ( cfp2DcoStatus.dcoType ==
              Tac.Type( "Xcvr::Cfp2DcoType" ).cfp2DcoTypeDp04 )
   addThreshold = thresholds and status.domThresholdValid and isDp04
   domThreshold = cfp2DcoStatus.dcoDomThreshold

   def positionInModel( model: InterfaceTransceiverDom, status,
                        attrKey: tuple[ bool, str ], valueDict: dict,
                        threshold: Optional[ InterfaceTransceiverDetailThresholds ]
                        ) -> InterfaceTransceiverDom:
      """
      Adds a transceiver dom parameter to model, deleting and re-inserting to ensure
      ordering
      """
      isDp04Attr, attr = attrKey
      if isDp04 and isDp04Attr:
         if attr in model.parameters:
            # Parameter exists in our model. Shuffle it to the end for ordering!
            model.parameters[ attr ] = model.parameters.pop( attr )
            return model
      if not isDp04 and isDp04Attr:
         return model
      if attr not in cfp2DcoStatus.attributes:
         assert f"Attribute {attr} is not a member of Cfp2DcoStatus"

      paramType = getXcvrParamType( status, valueDict[ attrKey ][ 0 ] )
      paramUnit = valueDict[ attrKey ][ 1 ]
      paramValue = float( getattr( cfp2DcoStatus, attr ) )
      paramChannel = '-'
      return insertParameterInModel( model, paramType, paramUnit, paramChannel,
                                     paramValue, threshold )

   domAttr = OrderedDict( [
         ( ( False, 'networkBer' ), ( 'preFecBERCurr', '', None ) ),
         ( ( False, 'netFecUncBlkCount' ), ( 'netFecUncBlkCount', '', None ) ),
         ( ( False, 'netOsnrEstimate' ), ( 'netOsnrEstimate', 'dB', None ) ),
         ( ( False, 'rxDiffGroupDelay' ), ( 'rxDiffGroupDelay', 'ps', None ) ),
         ( ( False, 'laserAge' ), ( 'laserAge', '%', None ) ),
         ( ( True, 'rxLaserAge' ), ( 'rxLaserAge', '%', None ) ),
         ( ( False, 'rxSnr' ), ( 'snr', 'dB', None ) ),
         ( ( False, 'rxStateOfPolarization' ), ( 'rxStateOfPolarization', 'rad/sec',
                                                 None ) ),
         ( ( True, 'rxNetworkPDL' ), ( 'rxNetworkPDL', 'dB', None ) ),
         ( ( False, 'rxQFactor' ), ( 'rxQFactor', 'dB', None ) ),
         ( ( False, 'rxChromDispersion' ), ( 'rxChromDispersion', 'ps/nm', None ) ),
         ( ( False, 'rxCarrierFreqOffset' ), ( 'rxCarrierFreqOffset', 'MHz',
                                               None ) ),
         ( ( False, 'rollOff' ), ( 'rollOff', '', None ) ),
         # These are commonDomParams being populated elsewhere, but mut be specified
         # here for proper ordering in output. Indicate this behavior by setting the
         # desired parameter type below to None.
         ( ( True, 'coherentTxPower' ), ( None, None, None ) ),
         ( ( True, 'totalRxPower' ), ( None, None, None ) ),
         ( ( True, 'rxChannelPower' ), ( None, None, None ) ),
         # END commonDomParams
         ( ( True, 'txLaserPower' ), ( 'txLaserPower', 'dBm', 'txLaserPower' ) ),
         ( ( True, 'rxLaserPower' ), ( 'rxLaserPower', 'dBm', 'rxLaserPower' ) ),
         ( ( True, 'txLaserTemperature' ), ( 'txLaserTemperature', 'C',
                                             'txLaserTemp' ) ),
         ( ( True, 'rxLaserTemperature' ), ( 'rxLaserTemperature', 'C',
                                             'rxLaserTemp' ) ),
   ] )
   # pylint: disable-msg=protected-access
   for attr, ( _, _, thresholdAttr ) in domAttr.items():
      _threshold = None
      if thresholdAttr and addThreshold:
         _threshold = _getEnhancedThresholdModel( domThreshold, thresholdAttr )
      model = positionInModel( model, status, attr, domAttr, _threshold )
   model._paramOrder = list( model.parameters )
   model._hasMultiChannelDomParam = any( len( v.channels ) > 1 for v in
                                        model.parameters.values() )
   return model

def _checkEnhancedDomThresholdValid( mediaType: str, paramType: str ) -> bool:
   """
   For 100GBASE-FR/DR/LR, the enhanced dom thresholds are not implemented,
   so we will not print these parameters in the CLI output.
   """
   return not( mediaType in ( XcvrMediaType.xcvr100GBaseFr,
                              XcvrMediaType.xcvr100GBaseDr,
                              XcvrMediaType.xcvr100GBaseLr,
                              XcvrMediaType.xcvr100GBaseEr, ) and
               paramType in ( DomParamType.preFecBERCurr,
                              DomParamType.uncorrectedBERCurr,
                              DomParamType.snr, DomParamType.tecCurrent,
                              DomParamType.laserTemp,
                              DomParamType.pam4Transitions,
         DomParamType.residualISI,
                              DomParamType.laserFreq ) )

# Appends the Enhanced DOM parameters to the xcvr DOM model that is passed in.
def _xcvrEnhancedDomInfo( status, thresholds: bool, model: InterfaceTransceiverDom )\
                          -> InterfaceTransceiverDom:
   """
   Parameters
   ---------------------
   domInfo: Xcvr::DomInfo
   laneId: int
   thresholds: bool
   mode: InterfaceTransceiverDom

   Returns
   ---------------------
   InterfaceTransceiverDom

   """
   _threshold = None
   addThreshold = thresholds and status.domThresholdValid

   if status.isEnhancedDomSupported():
      eMC = Tac.Value( 'Xcvr::Sff8436EnhancedMonitoringContents' )
      totalParameters = eMC.totalParameters
      for idx in range( 1, totalParameters + 1 ):
         extDom = status.enhancedDomInfo[ idx ]
         if extDom.enhancedDomInfoValid and not extDom.isPM:
            paramType = extDom.paramType
            paramUnit = extDom.paramUnit
            paramValue = extDom.monitoredParameter
            paramChannel = str( extDom.channel ) if extDom.channelSpecific else '-'
            extDomThreSupport = _checkEnhancedDomThresholdValid(
                                status.mediaType, paramType )
            if addThreshold and extDomThreSupport:
               # These two parameters don't have low alarm and low warn thresholds
               if paramType in ( DomParamType.preFecBERCurr,
                                 DomParamType.uncorrectedBERCurr ):
                  _threshold = InterfaceTransceiverDetailThresholds(
                        highAlarm=extDom.highAlarm, highWarn=extDom.highWarn )
               else:
                  _threshold = InterfaceTransceiverDetailThresholds(
                        highAlarm=extDom.highAlarm, highWarn=extDom.highWarn,
                        lowWarn=extDom.lowWarn, lowAlarm=extDom.lowAlarm )
            model = insertParameterInModel( model, paramType, paramUnit,
                                            paramChannel, paramValue, _threshold )

   # pylint: disable-msg=protected-access
   model._paramOrder = list( model.parameters )
   model._hasMultiChannelDomParam = any( len( v.channels ) > 1 for v in
                                         model.parameters.values() )
   return model

def showInterfacesXcvrDom( mode, args ) -> InterfacesTransceiverDom:
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   thresholds = 'thresholds' in args
   return _showInterfacesXcvrDom( mode, intf, mod, outputFmt='default',
                                  thresholds=thresholds )

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
   gv.xgc = ConfigMount.mount( em, "hardware/xcvr/xgc", "Xcvr::Xgc", "w" )
