# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from collections import namedtuple
import re
import Tac
from typing import Optional

import Arnet
import CliPlugin.TechSupportCli
import CliPlugin.XcvrAllStatusDir
import CliPlugin.IntfCli
import CliPlugin.EthIntfCli
import CliPlugin.EthIntfModel
from CliPlugin.XcvrCliLib import ( getAllIntfsWrapper, isDualLaserModule,
                                   IntfMapping,
                                   generalModeControlToLabel,
                                   getLineSystemStatus )
from CliPlugin.XcvrShowStatusModel import ( InterfacesTransceiverStatus,
      InterfacesTransceiverStatusValues,
      InterfacesTransceiverStatusExtCfp2Dco,
      InterfacesTransceiverStatusExtCfp2DcoIntf,
      InterfacesTransceiverStatusValueFormatModuleStateDec,
      InterfacesTransceiverStatusValueFormatTxTurnUpStateDec,
      InterfacesTransceiverStatusValueFormatRxTurnUpStateDec,
      InterfacesTransceiverStatusValueFormatModuleGeneralStatus,
      InterfacesTransceiverStatusValueFormatModuleFaultStatus,
      InterfacesTransceiverStatusValueFormatNetworkLaneVendorFAWS,
      InterfacesTransceiverStatusValueFormatNetworkLaneFaultStatus,
      InterfacesTransceiverStatusValueFormatLaneAlignment,
      InterfacesTransceiverStatusValueFormatNetworkLaneRxOTNStatus,
      InterfacesTransceiverStatusValueFormatTxModulatorBiasVOA,
      InterfacesTransceiverStatusValueFormatVendorSpecificFaultStatus,
      InterfacesTransceiverStatusIntfValueFormatIngressAlarmStatus,
      InterfacesTransceiverStatusIntfValueFormatEgressAlarmStatus,
      InterfacesTransceiverStatusValueFormatInt,
      InterfacesTransceiverStatusValueFormatTrib,
      InterfacesTransceiverStatusValueFormatStr,
      InterfacesTransceiverStatusInterfaceSpecific,
      InterfacesTransceiverStatusValueFormatBool,
      InterfacesTransceiverStatusValueFormatHex,
      InterfacesTransceiverStatusValueFormatScientificFloat,
      InterfacesTransceiverStatusChannelSpecific,
      InterfacesTransceiverStatusCoherentChannelSpecific,
      InterfacesTransceiverStatusFrequencyTuningChannelSpecific,
      InterfacesTransceiverStatusHostLaneSpecific )
from XcvrLib import ( getLineSideChannels,
                      getXcvrSlotName,
                      getXcvrStatus,
                      isCmisTransceiver )
import CliGlobal
import Tracing
from TypeFuture import TacLazyType
import Toggles.XcvrToggleLib

t0 = Tracing.trace0

RxLosOrTxFaultImplemented = namedtuple( 'rxLosOrTxFaultImplemented',
   'rx_los tx_fault' )
CoherentClientStatusAttr = namedtuple( 'ClientPcsAlarmStatusAttr',
                                       'name trueState falseState' )
XcvrType = TacLazyType( 'Xcvr::XcvrType' )
XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )
ClientConfig = TacLazyType(
   'Cfp2DcoHal::Cfp2DcoReg::GeneralModeControlLsb::ClientConfig' )
OpticalDetect = TacLazyType( 'Xcvr::OpticalDetect' )

gv = CliGlobal.CliGlobal( xcvrStatusDir_=None )

ethSpeedToStr = {
   'speedUnknown': 'unknown',
   'speed10Mbps': '10Mbps',
   'speed100Mbps': '100Mbps',
   'speed1Gbps': '1Gbps',
   'speed2p5Gbps': '2.5Gbps',
   'speed5Gbps': '5Gbps',
   'speed10Gbps': '10Gbps',
   'speed25Gbps': '25Gbps',
   'speed40Gbps': '40Gbps',
   'speed50Gbps': '50Gbps',
   'speed100Gbps': '100Gbps',
   'speed200Gbps': '200Gbps',
   'speed400Gbps': '400Gbps',
   'speed800Gbps': '800Gbps',
   }

# --------------------------------------------------------------------------------
#
# show transceiver status [ interface INTF ]
#
# --------------------------------------------------------------------------------
def _populateTransceiverStatusValues( intfName ):
   model = InterfacesTransceiverStatusValues()

   baseXcvrStatus = gv.xcvrStatusDir_.xcvrStatus[ getXcvrSlotName( intfName ) ]
   xcvrStatus = getXcvrStatus( baseXcvrStatus )

   model.setSerialNumber( xcvrStatus )
   model.setSerialNumber( xcvrStatus )
   model.setMediaType( xcvrStatus )
   if ( Toggles.XcvrToggleLib.toggleXcvrLpoEnabled() and xcvrStatus and
       xcvrStatus.isLpo ):
      model.setBandwidth( xcvrStatus )
   model.setPresence( xcvrStatus )
   model.setBadEepromChecksums( xcvrStatus )
   model.setResets( xcvrStatus )
   model.setInterrupts( xcvrStatus )
   model.setSmbusFailures( xcvrStatus )
   model.setAdapters( baseXcvrStatus )

   if isCmisTransceiver( xcvrStatus ):
      model.setDataPathFirmwareFault( xcvrStatus )
      model.setModuleFirmwareFault( xcvrStatus )
      model.setModuleState( xcvrStatus )
      for laneId in xcvrStatus.dataPathState:
         model.setLaneDataPathState( laneId, xcvrStatus )

   if xcvrStatus.domCapabilities.voltage:
      model.setVoltageAlarmAndWarn( xcvrStatus )
   if xcvrStatus.domCapabilities.temperature:
      model.setTemperatureAlarmAndWarn( xcvrStatus )

   return model

def _populatePolsTransceiverStatusValues( intfName ):
   """
   Only a subset of data applies to POLS, so we populate the model with
   POLS-relevant data in this function.

   Parameters
   ----------
   intfName : str
   """
   model = InterfacesTransceiverStatusValues()
   xcvrSlotName = getXcvrSlotName( intfName )
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir_.xcvrStatus[ xcvrSlotName ] )

   model.setSerialNumber( xcvrStatus )
   model.setMediaType( xcvrStatus )
   model.setPresence( xcvrStatus )
   model.setBadEepromChecksums( xcvrStatus )
   model.setResets( xcvrStatus )
   model.setInterrupts( xcvrStatus )
   model.setSmbusFailures( xcvrStatus )
   model.setAdapters( xcvrStatus )
   model.setVoltageAlarmAndWarn( xcvrStatus )
   model.setTemperatureAlarmAndWarn( xcvrStatus )

   if xcvrStatus.xcvrType == XcvrType.qsfpPlus:
      # QSFP-LS doesn't report many of the CMIS statuses that OSFP-LS does. It
      # does however publish data-path state for the booster and pre-amp in a
      # custom EEPROM region. Fetch the state from the corresponding
      # LineSystemStatus object.
      lsStatus = getLineSystemStatus( xcvrStatus )

      if lsStatus:
         # Lane 0 corresponds to the booster and lane 1 to the pre-amp
         boosterStateChange = lsStatus.boosterStateChange
         model.setLaneDataPathStateFromRawData( 0, lsStatus.boosterState,
                                                boosterStateChange.count,
                                                boosterStateChange.last )
         preAmpStateChange = lsStatus.preAmpStateChange
         model.setLaneDataPathStateFromRawData( 1, lsStatus.preAmpState,
                                                preAmpStateChange.count,
                                                preAmpStateChange.last )
   else:
      model.setDataPathFirmwareFault( xcvrStatus )
      model.setModuleFirmwareFault( xcvrStatus )
      model.setModuleState( xcvrStatus )

      for laneId in xcvrStatus.dataPathState:
         # laneId is 0-indexed and we only want the first two
         if laneId > 1:
            continue
         model.setLaneDataPathState( laneId, xcvrStatus )

   return model

def _populatePolsChannelStatus( model, intfName, intfMapping, primaryIntf ):
   intfSpecific = InterfacesTransceiverStatusInterfaceSpecific()
   slotName = getXcvrSlotName( primaryIntf )
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir_.xcvrStatus[ slotName ] )
   laneMap = intfMapping.intfMap
   intfLanes = laneMap[ primaryIntf ].get( intfName )
   # Populate only the channel parameters that apply to POLS
   channelAttrWhitelist = [ "txPowerLoWarn", "txPowerLoAlarm", "txPowerHiWarn",
                            "txPowerHiAlarm", "rxPowerLoWarn", "rxPowerLoAlarm",
                            "rxPowerHiWarn", "rxPowerHiAlarm" ]
   for laneId in intfLanes:
      chSpecific = InterfacesTransceiverStatusChannelSpecific()
      laneStats = xcvrStatus.laneStats.get( laneId )
      for attr in channelAttrWhitelist:
         cur = getattr( laneStats, attr )
         val = InterfacesTransceiverStatusValueFormatBool(
                  state=bool( cur.current ),
                  changes=cur.changes,
                  lastChange=cur.lastChange )
         setattr( chSpecific, attr, val )
      intfSpecific.channels[ laneId + 1 ] = chSpecific
   # Name override labels the datapaths as "Booster" and "Pre-amp"
   intfNameOverride = intfMapping.intfNameOverride
   intfDisplayName = intfNameOverride.get( intfName ) or intfName
   intfSpecific.interface = InterfacesTransceiverStatusValueFormatStr(
         state=intfDisplayName )
   # Populate base model with channel information and return
   model.interfaces[ intfDisplayName ] = intfSpecific
   return model

def displayOnlyRxLosTxFaultPinStates( xcvrStatus ):
   ''' Xcvr CLI helper to determine if 'show transceiver status interface'
   command should only render pin states for RX_LOS and TX_FAULT
   instead of all the DOM parameters.
   We want to catch all the SFP's that are not optics but do implement these options.
   For all the other optical media-types, isOpticalDomSupported check
   should be used to determine the eligibility of displaying DOM parameters.
   ----------------------------------------------------------------------------
   params:
      xcvrStatus : Xcvr::XcvrNewStatus
         status object of the transceiver slot.
   '''
   rxLosOrTxFaultImpl = RxLosOrTxFaultImplemented( False, False )
   if xcvrStatus.xcvrType == XcvrType.sfpPlus:
      optionsValue = xcvrStatus.sff8472IdEepromContents.options
      options = Tac.newInstance( "Xcvr::Sff8472Options", optionsValue )
      rxLosOrTxFaultImpl = RxLosOrTxFaultImplemented( options.rxLosImplemented,
            options.txFaultImplemented )
   return rxLosOrTxFaultImpl

def _populateCoherentXcvrStatusValues( intfName ):

   status = gv.xcvrStatusDir_.xcvrStatus[ getXcvrSlotName( intfName ) ]
   baseModel = None

   def populateModel( model, cfp2DcoStatus, attrs, attrType, isEventGroup,
                      trueState, falseState ):
      '''
      Helper method to populate commonly formatted alarm/warning models for the DP04
      dual laser module.
      '''
      for attr in attrs:
         attrValue = getattr( cfp2DcoStatus, attrType ).event[ attr ].stat
         if isEventGroup:
            current = attrValue.current
            lastChange = attrValue.lastChange
         else:
            current = attrValue.value
            lastChange = attrValue.whenTime
         submodel = InterfacesTransceiverStatusValueFormatStr(
               state=trueState if current else falseState,
               changes=attrValue.changes,
               lastChange=lastChange )
         setattr( model, attr, submodel )

   if status.mediaType == XcvrMediaType.xcvr100GDwdmDco:
      cfp2DcoStatus = status.cfp2DcoNewStatus
      baseModel = InterfacesTransceiverStatusExtCfp2Dco()
      baseModel.controllerState = InterfacesTransceiverStatusValueFormatStr(
            state=cfp2DcoStatus.controllerState )
      baseModel.controllerSubState = InterfacesTransceiverStatusValueFormatStr(
            state=cfp2DcoStatus.controllerSubState )
      baseModel.moduleState = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.moduleState )
      baseModel.txTurnUpState = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.txTurnUpState )
      baseModel.rxTurnUpState = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.rxTurnUpState )
      baseModel.moduleGeneral = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.moduleGeneralStatus )
      baseModel.moduleFaults = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.moduleFaultStatus )
      baseModel.vendorFaults = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.vendorSpecificFAWS )
      baseModel.networkVendorFaults = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.networkVendorSpecificFAWS )
      baseModel.laserFaultCode = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.laserFaultCode )
      baseModel.sopFastTracking = InterfacesTransceiverStatusValueFormatStr(
            state="enabled" if cfp2DcoStatus.sopFastTracking else "disabled" )

   elif isDualLaserModule( status ):
      # The following states apply only to the dual laser DP04 module
      baseModel = InterfacesTransceiverStatusExtCfp2Dco()
      baseModel.dualLaserModulePresentIs( True )
      cfp2DcoStatus = status.cfp2DcoNewStatus
      baseModel.controllerState = InterfacesTransceiverStatusValueFormatStr(
            state=cfp2DcoStatus.controllerState )
      baseModel.controllerSubState = InterfacesTransceiverStatusValueFormatStr(
            state=cfp2DcoStatus.controllerSubState )
      baseModel.moduleStateDec = \
         InterfacesTransceiverStatusValueFormatModuleStateDec()
      moduleStateEvents = [ 'init', 'lowPwr', 'hiPwrUp', 'txOff', 'txTurnOn',
                            'ready', 'fault', 'txTurnOff', 'hiPwrDown' ]
      populateModel( baseModel.moduleStateDec, cfp2DcoStatus, moduleStateEvents,
                     'moduleStateDec', True, 'active', 'inactive' )

      # TX turn up state
      baseModel.txTurnUpStateDec = \
         InterfacesTransceiverStatusValueFormatTxTurnUpStateDec()
      txTurnUpState = baseModel.txTurnUpStateDec
      txTurnUpModelInputs = [ ( 'txInit', 'ok', 'not initialized' ),
                              ( 'txDataPathLock', 'ok', 'no lock' ),
                              ( 'txLasReadyOff', 'ok', 'not ready' ),
                              ( 'txLaserReady', 'ok', 'not ready' ),
                              ( 'txModulatorConverge', 'ok', 'not converged' ),
                              ( 'txOutPwrAdj', 'ok', 'not adjusted' ), ]
      for ( txTurnUpAttr, trueState, falseState ) in txTurnUpModelInputs:
         populateModel( txTurnUpState, cfp2DcoStatus, [ txTurnUpAttr ],
                        'txTurnUpStateDec', False, trueState, falseState )

      # RX turn up state
      baseModel.rxTurnUpStateDec = \
         InterfacesTransceiverStatusValueFormatRxTurnUpStateDec()
      rxTurnUpState = baseModel.rxTurnUpStateDec
      rxTurnUpModelInputs = [ ( 'rxInit', 'ok', 'not initialized' ),
                              ( 'rxLoLaserReady', 'ok', 'not ready' ),
                              ( 'rxWaitForInput', 'ok', 'not ready' ),
                              ( 'adcOutput', 'up', 'down' ),
                              ( 'dispersionLock', 'ok', 'no lock' ),
                              ( 'rxDemodLock', 'ok', 'no lock' ), ]
      for ( rxTurnUpAttr, trueState, falseState ) in rxTurnUpModelInputs:
         populateModel( rxTurnUpState, cfp2DcoStatus, [ rxTurnUpAttr ],
                        'rxTurnUpStateDec', False, trueState, falseState )

      # Module general status
      baseModel.moduleGeneralStatus = \
         InterfacesTransceiverStatusValueFormatModuleGeneralStatus()
      moduleGeneralStatus = baseModel.moduleGeneralStatus
      modGenStatModelInputs = [ ( 'hiPowerOn', 'ok', 'power off' ),
                                ( 'outOfAlgn', 'unaligned', 'ok' ),
                                ( 'rxNetworkLOL', 'no lock', 'ok' ),
                                ( 'rxLOS', 'no signal', 'ok' ),
                                ( 'txHostLOL', 'no lock', 'ok' ),
                                ( 'txLOF', 'no signal', 'ok' ),
                                ( 'hwInterlock', 'max capacity', 'ok' ), ]
      for ( modStatAttr, trueState, falseState ) in modGenStatModelInputs:
         populateModel( moduleGeneralStatus, cfp2DcoStatus, [ modStatAttr ],
                        'modGeneralStatus', True, trueState, falseState )

      # Module fault status
      baseModel.moduleFaultStatus = \
         InterfacesTransceiverStatusValueFormatModuleFaultStatus()
      moduleFaultStatus = baseModel.moduleFaultStatus
      checkSum = cfp2DcoStatus.modFaultStat.event[ 'checkSum' ].stat
      moduleFaultStatus.checkSum = InterfacesTransceiverStatusValueFormatStr(
         state='ok' if not checkSum.current else 'failed',
         changes=checkSum.changes,
         lastChange=checkSum.lastChange )
      modFaultStatusAttrs = [ 'powerSupplyFault', 'pldFlashInitFault',
                              'modSpecFault', 'modOverTempFault' ]
      populateModel( baseModel.moduleFaultStatus, cfp2DcoStatus, modFaultStatusAttrs,
                     'modFaultStat', True, 'fault', 'ok' )

      # Dual laser alarms/warnings
      netAlarm1Attrs = [ 'txLaserPowerHighAlarm', 'txLaserPowerLowAlarm',
                         'txLaserTempLowAlarm', 'txLaserTempHighAlarm',
                         'rxPowerHighAlarm', 'rxPowerLowAlarm' ]
      populateModel( baseModel, cfp2DcoStatus, netAlarm1Attrs,
                     'netAlarmWarn1', True, 'alarm', 'ok' )
      netWarning1Attrs = [ 'txLaserPowerHighWarn', 'txLaserPowerLowWarn',
                           'txLaserTempHighWarn', 'txLaserTempLowWarn',
                           'rxPowerHighWarn', 'rxPowerLowWarn' ]
      populateModel( baseModel, cfp2DcoStatus, netWarning1Attrs,
                     'netAlarmWarn1', True, 'warn', 'ok' )
      netAlarm2Attrs = [ 'txModBiasHighAlarm', 'rxLaserPowerHighAlarm',
                         'rxLaserPowerLowAlarm', 'rxLaserTempHighAlarm',
                         'rxLaserTempLowAlarm' ]
      populateModel( baseModel, cfp2DcoStatus, netAlarm2Attrs,
                     'netAlarmWarn2', True, 'alarm', 'ok' )
      netWarning2Attrs = [ 'txModBiasHighWarn', 'rxLaserPowerHighWarn',
                           'rxLaserPowerLowWarn', 'rxLaserTempHighWarn',
                           'rxLaserTempLowWarn' ]
      populateModel( baseModel, cfp2DcoStatus, netWarning2Attrs,
                     'netAlarmWarn2', True, 'warn', 'ok' )

      # Network lane vendor specific faults/status
      baseModel.netLaneVendorSpecificFAWS = \
         InterfacesTransceiverStatusValueFormatNetworkLaneVendorFAWS()
      netLaneVendorSpecificFAWS = baseModel.netLaneVendorSpecificFAWS
      netVendorSpecFAWSModelInputs = [
         ( [ 'oaPumpBiasHighAlarm', 'oaPumpBiasLowAlarm', 'rxPhaseCtrlLoopHighAlarm',
             'rxPhaseCtrlLoopLowAlarm', 'txModBiasVOAHighAlarm',
             'rxTunedChannelPwrHighAlarm', 'rxTunedChannelPwrLowAlarm' ],
           'alarm', 'ok' ),
         ( [ 'oaPumpBiasHighWarn', 'oaPumpBiasLowWarn', 'rxPhaseCtrlLoopHighWarn',
             'rxPhaseCtrlLoopLowWarn', 'txModBiasVOAHighWarn',
             'rxTunedChannelPwrHighWarn', 'rxTunedChannelPwrLowWarn' ],
           'warn', 'ok' ),
         ( [ 'modBiasConvergeFault' ], 'fault', 'ok' ),
         ( [ 'adcOutputLow' ], 'low', 'ok' ), ]
      for ( fawsAttrs, trueState, falseState ) in netVendorSpecFAWSModelInputs:
         populateModel( netLaneVendorSpecificFAWS, cfp2DcoStatus,
                        fawsAttrs, 'networkVendorSpecificFAWSDec',
                        True, trueState, falseState )

      # Network lane fault/status
      baseModel.netLaneFaultStatus = \
         InterfacesTransceiverStatusValueFormatNetworkLaneFaultStatus()
      netLaneFaultStatus = baseModel.netLaneFaultStatus
      netLaneFaultAttrs = [ 'rxTECFault', 'rxFIFOFault', 'wavelengthUnlockFault',
                            'tecFault' ]
      populateModel( netLaneFaultStatus, cfp2DcoStatus, netLaneFaultAttrs,
                     'networkFaultStatus', True, 'fault', 'ok' )
      netLaneFaultStatModelInputs = [ ( 'rxLOL', 'no lock', 'ok' ),
                                      ( 'rxLOS', 'no signal', 'ok' ),
                                      ( 'txLOL', 'no lock', 'ok' ),
                                      ( 'txLOSF', 'no signal', 'ok' ), ]
      for ( netLaneFaultStatAttr, trueState,
            falseState ) in netLaneFaultStatModelInputs:
         populateModel( netLaneFaultStatus, cfp2DcoStatus,
                        [ netLaneFaultStatAttr ], 'networkFaultStatus',
                        True, trueState, falseState )

      # TX/RX lane alignment
      baseModel.laneAlignment = \
         InterfacesTransceiverStatusValueFormatLaneAlignment()
      laneAlignment = baseModel.laneAlignment
      netLaneTxAlignment = cfp2DcoStatus.txNetworkAlignmentStatus.event[
         'outOfAlgn' ].stat
      laneAlignment.netLaneTxAlignment = \
         InterfacesTransceiverStatusValueFormatStr(
            state='unaligned' if netLaneTxAlignment.current else 'ok',
            changes=netLaneTxAlignment.changes,
            lastChange=netLaneTxAlignment.lastChange )
      populateModel( laneAlignment, cfp2DcoStatus,
                     [ 'modemLockFault', 'modemSyncDetectFault' ],
                     'rxNetworkAlignmentStatus', True, 'fault', 'ok' )

      # Network lane RX OTN status
      baseModel.networkLaneRxOTNStatus = \
         InterfacesTransceiverStatusValueFormatNetworkLaneRxOTNStatus()
      networkLaneRxOTNStatus = baseModel.networkLaneRxOTNStatus
      rxNetOTNStatusAttrs = [ 'fastPMRxSignalDegradeAlarm',
         'fastPMRxSignalFailAlarm', 'fastPMTxSignalDegradeAlarm',
         'fastPMTxSignalFailAlarm', 'ingressFDDAlarm', 'ingressFEDAlarm' ]
      populateModel( networkLaneRxOTNStatus, cfp2DcoStatus, rxNetOTNStatusAttrs,
                     'rxNetOTNStatus', True, 'alarm', 'ok' )
      netLaneRxOTNStatModelInputs = [
         ( 'oduLCKMaintSignal', 'active', 'inactive' ),
         ( 'oduAISMaintSignal', 'active', 'inactive' ),
         ( 'sectionMonitoringBackDefect', 'defect', 'ok' ),
         ( 'outOfMultiframe', 'out of frame', 'ok' ),
         ( 'lossOfMultiframe', 'no frame', 'ok' ),
         ( 'outOfFrame', 'out of frame', 'ok' ),
         ( 'lossOfFrame', 'no frame', 'ok' ) ]
      for ( netLaneRxOTNStatAttr, trueState,
            falseState ) in netLaneRxOTNStatModelInputs:
         populateModel( networkLaneRxOTNStatus, cfp2DcoStatus,
                        [ netLaneRxOTNStatAttr ], 'rxNetOTNStatus', True, trueState,
                        falseState )

      # Network TX modulator bias VOA AWS
      baseModel.networkTxModulatorBiasVoaAWS = \
         InterfacesTransceiverStatusValueFormatTxModulatorBiasVOA()
      networkTxModulatorBiasVoaAWS = baseModel.networkTxModulatorBiasVoaAWS
      netTxModBiasVOAStatAlarms = [ 'loop1Alarm', 'loop2Alarm', 'loop3Alarm',
                                    'loop4Alarm', 'loop5Alarm', 'loop6Alarm',
                                    'loop7Alarm', 'loop8Alarm' ]
      populateModel( networkTxModulatorBiasVoaAWS, cfp2DcoStatus,
                     netTxModBiasVOAStatAlarms, 'netTxModBiasVOAStat', False,
                     'alarm', 'ok' )
      netTxModBiasVOAStatWarnings = [ 'loop1Warn', 'loop2Warn', 'loop3Warn',
                                      'loop4Warn', 'loop5Warn', 'loop6Warn',
                                      'loop7Warn', 'loop8Warn' ]
      populateModel( networkTxModulatorBiasVoaAWS, cfp2DcoStatus,
                     netTxModBiasVOAStatWarnings, 'netTxModBiasVOAStat', False,
                     'warn', 'ok' )

      # ADC and DAC fault codes
      adcFaultCode = cfp2DcoStatus.ntwkAdcDacFltCodes.event[
         'netADCFaultCode' ].stat
      baseModel.adcFaultCode = \
         InterfacesTransceiverStatusValueFormatHex(
            state=adcFaultCode.value,
            changes=adcFaultCode.changes,
            lastChange=adcFaultCode.whenTime )
      dacFaultCode = cfp2DcoStatus.ntwkAdcDacFltCodes.event[
         'netDACFaultCode' ].stat
      baseModel.dacFaultCode = \
         InterfacesTransceiverStatusValueFormatHex(
            state=dacFaultCode.value,
            changes=dacFaultCode.changes,
            lastChange=dacFaultCode.whenTime )

      # Net mod bias fault information
      convergenceIter = cfp2DcoStatus.netModBiasFaultInfo.event[
         'convergenceIter' ].stat
      baseModel.modulatorConvergenceIter = \
         InterfacesTransceiverStatusValueFormatInt(
            state=int( convergenceIter.value ),
            changes=convergenceIter.changes,
            lastChange=convergenceIter.whenTime )
      convergenceStep = cfp2DcoStatus.netModBiasFaultInfo.event[
         'convergenceStep' ].stat
      baseModel.modulatorConvergenceStep = \
         InterfacesTransceiverStatusValueFormatInt(
            state=int( convergenceStep.value ),
            changes=convergenceStep.changes,
            lastChange=convergenceStep.whenTime )
      faultCode = cfp2DcoStatus.netModBiasFaultInfo.event[
         'faultCode' ].stat
      baseModel.modulatorBiasFaultCode = \
         InterfacesTransceiverStatusValueFormatHex(
            state=faultCode.value,
            changes=faultCode.changes,
            lastChange=faultCode.whenTime )

      # Vendor specific fault status
      vendorSpecificFAWSAttrs = [ 'bootEepromCRCFault', 'devInitFault',
         'ctrlProcImgABBootFault', 'fwHwMismatchFault', 'powerSupplyRWFault',
         'cryptoDevRWFault', 'cpldDevRWFault', 'viMonitorDevRWFault',
         'dacRefClockRWFault' ]
      baseModel.vendorSpecificFaultStatus = \
         InterfacesTransceiverStatusValueFormatVendorSpecificFaultStatus()
      populateModel( baseModel.vendorSpecificFaultStatus, cfp2DcoStatus,
                     vendorSpecificFAWSAttrs, 'vendorSpecificFAWSDec', True,
                     'fault', 'ok' )
      vendorSpecificFAWS2Attrs = [ 'ctrlProcIntCommFault', 'modemProcIntCommFault',
         'opticsProcIntCommFault', 'txITLAFault', 'rxITLAFault',
         'adcCalibrationFault', 'dacCalibrationFault', 'picEepromRWFault',
         'picEepromCRCFault', 'apexRWFault', 'picXTiaRWFault', 'picYTiaRWFault',
         'picDriverRWFault' ]
      populateModel( baseModel.vendorSpecificFaultStatus, cfp2DcoStatus,
                     vendorSpecificFAWS2Attrs, 'vendorSpecificFAWS2', True,
                     'fault', 'ok' )

      # Laser fault codes
      rxLaserFaultCode = cfp2DcoStatus.rxLaserFaultCode.event[
         'rxLaserFaultCode' ].stat
      baseModel.rxLaserFaultCode = InterfacesTransceiverStatusValueFormatHex(
         state=rxLaserFaultCode.value,
         changes=rxLaserFaultCode.changes,
         lastChange=rxLaserFaultCode.whenTime )
      txLaserFaultCode = cfp2DcoStatus.txLaserFaultCode.event[
         'txLaserFaultCode' ].stat
      baseModel.laserFaultCode = InterfacesTransceiverStatusValueFormatHex(
         state=txLaserFaultCode.value,
         changes=txLaserFaultCode.changes,
         lastChange=txLaserFaultCode.whenTime )

   return baseModel

def populateModelAttrs( attrList, dataObject, modelObject ):
   '''
   Helper function for extracting the given attributes from the object containing
   the data and populating the model with these values.

   Parameters
   ----------
   attrList : [ str ]
      A list of attribute names that are being populated.
   dataObject : object
      The object that is used to extract the attributes' values from
   modelObject : object
      The model object whose attributes are being populated.
      Ex: an object of type InterfacesTransceiverStatusChannelSpecific or
      InterfacesTransceiverStatusHostLaneSpecific.
   '''
   for attr in attrList:
      cur = getattr( dataObject, attr )
      val = InterfacesTransceiverStatusValueFormatBool(
            state=bool( cur.current ),
            changes=cur.changes,
            lastChange=cur.lastChange )
      setattr( modelObject, attr, val )

def populateModelCoherentFecDegradeAttr( adminAttrs, alarmAttrs,
                                         dataObject, modelObject ):
   '''
   A wrapper to populating Coherent Fec Degrade  atrributes
   alarm status is populated only if the feature is administratively
   enabled

   Parameters
   ----------
   adminList : [ str ]
      A list of administrative attributes

   alarmList : [ str ]
      A list of alarm attributes

   dataObject : object
      The object that is used to extract the attributes' values from
   modelObject : object
      The model object whose attributes are being populated.
      Ex: an object of type InterfacesTransceiverStatusChannelSpecific or
      InterfacesTransceiverStatusHostLaneSpecific.
   '''
   for adminAttr, alarmAttr in zip( adminAttrs, alarmAttrs ):
      if Toggles.XcvrToggleLib.toggleCoherentFecDegradeEnabled():
         if getattr( dataObject, adminAttr ).current:
            populateModelAttrs( [ adminAttr, alarmAttr ], dataObject, modelObject )
         else:
            populateModelAttrs( [ adminAttr ], dataObject, modelObject )
      else:
         # Old model - populate only alarms
         populateModelAttrs( [ alarmAttr ], dataObject, modelObject )

def _cmisMediaChannelToHostElectricalLane( cmisStatus,
                                           channel: int ) -> Optional[ int ]:
   # For Cmis transceivers, there may be a different number of electrical and
   # optical lanes (named 'lanes' and 'channels' respectively by Xcvr controller).
   # Therefore, we use the mapping from channels to lanes in
   # xcvrStatus.laneMapping to find the corresponding lane for a given channel.
   if ( cmisStatus.capabilities.muxponderModeCapable and
        cmisStatus.muxponderMode ):
      # ZR/ZRP transceivers in muxponder mode have all their host lanes mapped
      # to the only existent media channel (0).
      return 0
   elif ( cmisStatus.laneMapping and
          ( rxChannelToLane :=
               cmisStatus.laneMapping.rxChannelToLane.get( channel ) ) ):
      # Use a mapping from media channels to host lanes to find the
      # corresponding host lane for a given media channel.
      return min( rxChannelToLane.hostLane )
   else:
      return None

def _mediaChannelToHostElectricalLane( xcvrStatus, channel: int ) -> Optional[ int ]:
   # For Cmis transceivers, there may be a different number of host electrical
   # lanes and optical media channels.
   if isCmisTransceiver( xcvrStatus ):
      return _cmisMediaChannelToHostElectricalLane( xcvrStatus, channel )

   # Otherwise, assume we have an equal number of host electrical lanes
   # and optical media lanes, which gives us a 1-to-1 mapping
   return channel

def getXcvrLaneStats( xcvrStatus, channels ):
   # This function is populating the model with media lane attributes
   chValues = {}
   flagsSupported = xcvrStatus.isOpticalDomSupported( xcvrStatus.mediaType )
   renderOnlyRxLosTxFault = displayOnlyRxLosTxFaultPinStates( xcvrStatus )
   for channel in channels: # pylint: disable=too-many-nested-blocks
      # We start by assuming we have an eqaul number of electrical and media lanes
      # with a 1-to-1 mapping
      laneId = _mediaChannelToHostElectricalLane( xcvrStatus, channel )
      if laneId is None:
         continue

      # These are indexed by laneId
      laneStats = xcvrStatus.laneStats.get( laneId )
      laneStatsEnt = xcvrStatus.laneStatsEnt.get( laneId )
      if ( ( flagsSupported or True in renderOnlyRxLosTxFault ) and
            laneStats is not None ):
         chSpecific = InterfacesTransceiverStatusChannelSpecific()

         if flagsSupported or renderOnlyRxLosTxFault.rx_los:
            chSpecific.rxLos = InterfacesTransceiverStatusValueFormatBool(
               state=laneStats.currentRxLosState,
               changes=laneStats.rxLosEventsSinceInsertion,
               lastChange=laneStats.lastRxLosEvent )

         if flagsSupported or renderOnlyRxLosTxFault.tx_fault:
            chSpecific.txFault = InterfacesTransceiverStatusValueFormatBool(
               state=laneStats.currentTxFaultState,
               changes=laneStats.txFaultEventsSinceInsertion,
               lastChange=laneStats.lastTxFaultEvent )

         # The following parameters should be displayed only if
         # flagsSuppported is set irrespective of renderOnlyRxLosTxFault values.
         if flagsSupported:
            attrs = [ 'rxCdrLol', 'txPowerHiAlarm', 'txPowerLoAlarm',
                      'txPowerHiWarn', 'txPowerLoWarn', 'txBiasHiAlarm',
                      'txBiasLoAlarm', 'txBiasHiWarn', 'txBiasLoWarn',
                      'rxPowerHiAlarm', 'rxPowerLoAlarm', 'rxPowerHiWarn',
                      'rxPowerLoWarn' ]

            if isCmisTransceiver( xcvrStatus ):
               populateModelAttrs( attrs, laneStats, chSpecific )

               # If Output status entity exists (CMIS5.0+), call populateModelAttrs
               # for media lane attributes of the entity
               if xcvrStatus.cmisOutputSignalState:
                  cmisTxOutputSignalStats = \
                     xcvrStatus.cmisOutputSignalState.cmisTxOutputSignalStats
                  if laneId in cmisTxOutputSignalStats:
                     populateModelAttrs( [ 'txOutputSignalStats' ],
                                         cmisTxOutputSignalStats[ laneId ],
                                         chSpecific )

               if xcvrStatus.opticalDetect == OpticalDetect.coherentDetect:
                  # We are populating the model with media lane coherent attributes
                  mediaStatsLaneId = 0 \
                        if ( xcvrStatus.capabilities.muxponderModeCapable and
                             xcvrStatus.muxponderMode ) else laneId
                  coherentAlarms = \
                        InterfacesTransceiverStatusCoherentChannelSpecific()
                  frequencyTuningAlarms = \
                        \
                        InterfacesTransceiverStatusFrequencyTuningChannelSpecific()

                  if xcvrStatus.cmisFrequencyTuningStats:
                     cmisFrequencyTuningStats = \
                           xcvrStatus.cmisFrequencyTuningStats.get(
                                 mediaStatsLaneId )
                     if cmisFrequencyTuningStats:
                        freqAttrs = [ 'tuningInProgress', 'tuningNotAccepted',
                                      'invalidChannel', 'tuningComplete' ]
                        populateModelAttrs( freqAttrs, cmisFrequencyTuningStats,
                                            frequencyTuningAlarms )

                  if xcvrStatus.cmisCoherentMediaStats:
                     cmisCoherentMediaStats = xcvrStatus.cmisCoherentMediaStats.get(
                           mediaStatsLaneId )
                     if cmisCoherentMediaStats:
                        coherentAttrs = [ 'txLossOfAlignment', 'txOutOfAlignment',
                                          'txCmuLossOfLock', 'txRefClkLossOfLock',
                                          'txDeskewLossOfLock', 'txFifoErr',
                                          'rxDemodulationLossOfLock',
                                          'rxChrDispCompensationLossOfLock',
                                          'rxLossOfAlignment', 'rxOutOfAlignment',
                                          'rxDeskewLossOfLock', 'rxFifoErr',
                                        ]
                        populateModelAttrs( coherentAttrs, cmisCoherentMediaStats,
                                            coherentAlarms )

                        coherentFecDegradeAdminAttrs = \
                           [ 'fecDetectedDegradeEnabled',
                             'fecExcessiveDegradeEnabled' ]
                        coherentFecDegradeAttrs = \
                           [ 'fecDetectedDegrade',
                             'fecExcessiveDegrade' ]

                        populateModelCoherentFecDegradeAttr(
                           coherentFecDegradeAdminAttrs,
                           coherentFecDegradeAttrs,
                           cmisCoherentMediaStats,
                           coherentAlarms )

                  setattr( chSpecific, 'coherentAlarms', coherentAlarms )
                  setattr( chSpecific, 'frequencyTuningAlarms',
                           frequencyTuningAlarms )

            elif laneStatsEnt is not None:
               populateModelAttrs( attrs, laneStatsEnt, chSpecific )

         chValues[ channel ] = chSpecific
   return chValues

def getHostLaneStats( xcvrStatus, hostLanes ):
   # This function is populating the model with host lane attributes
   hlValues = {}
   flagsSupported = xcvrStatus.isOpticalDomSupported( xcvrStatus.mediaType )
   for laneId in hostLanes: # pylint: disable=too-many-nested-blocks
      laneStats = xcvrStatus.laneStats.get( laneId )
      laneStatsEnt = xcvrStatus.laneStatsEnt.get( laneId )
      if flagsSupported and laneStats is not None:
         hlSpecific = InterfacesTransceiverStatusHostLaneSpecific()

         attrs = [ 'txCdrLol', 'txLos', 'txAdaptiveInputEqFault' ]
         if isCmisTransceiver( xcvrStatus ):
            populateModelAttrs( attrs, laneStats, hlSpecific )

            # If Output status entity exists (CMIS5.0+), call populateModelAttrs
            # for host lane attributes of the entity
            if xcvrStatus.cmisOutputSignalState:
               cmisRxOutputSignalStats = \
                  xcvrStatus.cmisOutputSignalState.cmisRxOutputSignalStats
               if laneId in cmisRxOutputSignalStats:
                  populateModelAttrs( [ 'rxOutputSignalStats' ],
                                      cmisRxOutputSignalStats.get( laneId ),
                                      hlSpecific )

            # Coherent modules with one media channel only display data
            # for lane 0, with exception of muxponder (4x100G, 2x200G-4, 1x200g-4)
            # modes. For muxponder mode, we only want to display this information for
            # the main lane of each interface because the transceiver reports this
            # based on the datapath for each applciation: i.e. 4x100G reports lanes
            # 0,2,4,6; 2x200G reports lanes 0,4
            numHostLanes = xcvrStatus.capabilities.maxChannels
            lineSideChannels = getLineSideChannels( xcvrStatus )
            numLanesPerInterface = 0 if lineSideChannels == 0 else\
                                        numHostLanes // lineSideChannels

            if ( xcvrStatus.enhancedDom and
                 numLanesPerInterface and
                 ( not xcvrStatus.isCoherent( xcvrStatus.mediaType ) or
                 ( lineSideChannels == 1 and laneId == 0 ) or
                 ( xcvrStatus.muxponderMode and
                   laneId % numLanesPerInterface == 0 ) ) ):
               domInfo = xcvrStatus.enhancedDom.domInfo
               fecAttrs = [ 'preFecBERCurrHost', 'errFramesCurHost' ]
               enhancedDomVdmModule = Tac.enumValue( "Xcvr::EnhancedDomVdmParamLane",
                                                     "enhancedDomVdmModule" )
               for attr in fecAttrs:
                  key = Tac.Value( 'Xcvr::CmisEnhancedDomKey', attr, laneId )
                  globalKey = Tac.Value( 'Xcvr::CmisEnhancedDomKey', attr,
                                         enhancedDomVdmModule )
                  cur = domInfo.get( key )
                  # If preFecBER is advertised as CMID4.0 Table 8-99 module
                  # parameter, store it under laneId 0, for consistency
                  # with 400GBASE-ZR and other coherent modules
                  # We assume there won't be a per-lane and a global
                  # pre-FEC BER at the same time
                  if cur is None and laneId == 0:
                     cur = domInfo.get( globalKey )
                  if cur is not None:
                     val = \
                        InterfacesTransceiverStatusValueFormatScientificFloat(
                           state=float( cur.paramValue ),
                           changes=None,
                           lastChange=None )
                     setattr( hlSpecific, attr, val )

         elif laneStatsEnt is not None:
            populateModelAttrs( attrs, laneStatsEnt, hlSpecific )

         hlValues[ laneId ] = hlSpecific
   return hlValues

def _populateIntfSpecificValues( model, intfName, intfMapping, primaryIntf ):
   intfSpecific = InterfacesTransceiverStatusInterfaceSpecific()
   laneMap = intfMapping.intfMap
   hostMap = intfMapping.hostMap
   portLaneMapOverride = intfMapping.portLaneMapOverride
   intfNameOverride = intfMapping.intfNameOverride
   slotName = getXcvrSlotName( intfName )
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir_.xcvrStatus[ slotName ] )
   intfLanes = laneMap[ intfName ].get( intfName )

   intfLanes = portLaneMapOverride.get( slotName, {} ).get( intfName, intfLanes )
   hostLanes = hostMap[ intfName ]

   primaryEpis = None
   if primaryIntf in intfMapping.epis:
      primaryEpis = intfMapping.epis[ primaryIntf ]
   else:
      primaryEpis = intfMapping.epis[ getXcvrSlotName( primaryIntf ) ]
   if primaryEpis:
      speed = primaryEpis.speed
      intfSpecific.operSpeed = InterfacesTransceiverStatusValueFormatStr(
            state=ethSpeedToStr[ speed ] )

   xcvrLaneStats = getXcvrLaneStats( xcvrStatus, intfLanes )
   for laneId, stats in xcvrLaneStats.items():
      mediaChannel = 0 if ( xcvrStatus.capabilities.muxponderModeCapable and
                            xcvrStatus.muxponderMode ) else laneId
      # In 4x100G ZR application we really only have one channel. All 4 interfaces
      # should display the same channel.
      intfSpecific.channels[ mediaChannel + 1 ] = stats
   hostLaneStats = getHostLaneStats( xcvrStatus, hostLanes )
   for laneId, stats in hostLaneStats.items():
      intfSpecific.hostLanes[ laneId + 1 ] = stats

   intfDisplayName = intfNameOverride.get( intfName ) or intfName
   intfSpecific.interface = InterfacesTransceiverStatusValueFormatStr(
         state=intfDisplayName )
   model.interfaces[ intfDisplayName ] = intfSpecific

   return model

def _populateCoherentIntfSpecificValues( model, intfName ):
   status = gv.xcvrStatusDir_.xcvrStatus[ getXcvrSlotName( intfName ) ]
   intfSpecific = InterfacesTransceiverStatusExtCfp2DcoIntf()

   if not isDualLaserModule( status ):
      if status.mediaType == XcvrMediaType.xcvr100GDwdmDco:
         lane = 1
         if status.xcvrConfig.intfName.get( 0 ) == intfName:
            lane = 0
         pcsAlarmState = getattr( status.cfp2DcoNewStatus,
                                  f'clientPcsAlarmStatus{lane}' )
         statusModel = InterfacesTransceiverStatusValueFormatHex(
            state=pcsAlarmState )
         intfSpecific.pcsAlarmStatus = statusModel
         model.interfaces[ intfName ].cfp2Dco = intfSpecific
      return model

   # The current field only applies for DP04 DCO modules
   intfSpecific.dualLaserModulePresentIs( True )
   cfp2DcoStatus = status.cfp2DcoNewStatus
   # There is no interface status on startup, so return nothing
   if intfName not in cfp2DcoStatus.cfp2DcoIntfStatus:
      return model

   clientCfg = Tac.Value( 'Cfp2DcoHal::Cfp2DcoReg::GeneralModeControlLsb',
                          cfp2DcoStatus.generalModeControlLsb ).clientCfg
   try:
      ( fec, fecBlockSize, laneCount ) = generalModeControlToLabel[ clientCfg ]
   except KeyError as e:
      raise AssertionError(
         f'Invalid ClientConfig type identified: {clientCfg}' ) from e
   intfSpecific.laneCount = InterfacesTransceiverStatusValueFormatInt(
      state=laneCount )
   intfSpecific.fec = InterfacesTransceiverStatusValueFormatStr( state=fec )
   intfStatus = cfp2DcoStatus.cfp2DcoIntfStatus[ intfName ]
   if fecBlockSize:
      intfSpecific.rsFecCodewordSize = \
         InterfacesTransceiverStatusValueFormatInt(
         state=fecBlockSize )
      clientFecUcb = intfStatus.clientFecUcb.event[ 'ucb' ].stat
      intfSpecific.fecUncorrectedBlocks = \
         InterfacesTransceiverStatusValueFormatInt(
         state=clientFecUcb.value,
         changes=clientFecUcb.changes,
         lastChange=clientFecUcb.whenTime )
      intfSpecific.preFecBer = InterfacesTransceiverStatusValueFormatStr(
         state=f"{intfStatus.clientTxBerPm:.2e}" )

   def populateClientPcsModel( status, model, attrs ):
      for ( attr, intfXcvrStatusValueFormat ) in attrs:
         # Construct model attribute name
         alarmStatStr = attr.name.replace( 'ingress', '' ).replace( 'egress', '' )
         cliModelAttrName = alarmStatStr[ 0 ].lower() + alarmStatStr[ 1 : ]
         attrValue = status.event[ attr.name ].stat
         attrModel = intfXcvrStatusValueFormat(
            state=attr.trueState if attrValue.current else attr.falseState,
            changes=attrValue.changes,
            lastChange=attrValue.lastChange )
         setattr( model, cliModelAttrName, attrModel )

   intfSpecific.ingressAlarmStatus = \
      InterfacesTransceiverStatusIntfValueFormatIngressAlarmStatus()
   intfSpecific.egressAlarmStatus = \
      InterfacesTransceiverStatusIntfValueFormatEgressAlarmStatus()
   fmtBool = InterfacesTransceiverStatusValueFormatBool
   fmtStr = InterfacesTransceiverStatusValueFormatStr
   ingressClientPcsAttrs = [
      ( CoherentClientStatusAttr( 'ingressRxLocalFault', True, False ), fmtBool ),
      ( CoherentClientStatusAttr( 'ingressRxRemoteFault', True, False ), fmtBool ),
      ( CoherentClientStatusAttr( 'ingressBlockLock', 'no block lock', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'ingressAlignmentMarkerLock', 'unaligned', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'ingressBipErrorsDetected', 'detected', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'ingressErroredBlocksDetected', 'detected', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'ingressLoa', 'unaligned', 'ok' ), fmtStr ),
      ( CoherentClientStatusAttr( 'ingressLos', 'no signal', 'ok' ), fmtStr ) ]
   egressClientPcsAttrs = [
      ( CoherentClientStatusAttr( 'egressHighBer', 'high BER', 'ok' ), fmtStr ),
      ( CoherentClientStatusAttr( 'egressRxLocalFault', True, False ), fmtBool ),
      ( CoherentClientStatusAttr( 'egressRxRemoteFault', True, False ), fmtBool ),
      ( CoherentClientStatusAttr( 'egressBlockLock', 'no block lock', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'egressAlignmentMarkerLock', 'unaligned', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'egressBipErrorsDetected', 'detected', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'egressErroredBlocksDetected', 'detected', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'egressLoa', 'unaligned', 'ok' ), fmtStr ), ]
   for ( client, attrs ) in [ ( 'ingressAlarmStatus', ingressClientPcsAttrs ),
                              ( 'egressAlarmStatus', egressClientPcsAttrs ) ]:
      clientPcsModel = getattr( intfSpecific, client )
      populateClientPcsModel( intfStatus.clientPcsAlarmStatus, clientPcsModel,
                              attrs )
   if intfName in status.tribSelectStatus:
      intfSpecific.tributary = InterfacesTransceiverStatusValueFormatTrib(
         state=status.tribSelectStatus[ intfName ] )
   model.interfaces[ intfName ].cfp2Dco = intfSpecific
   return model

def _showInterfacesXcvrStatus( mode, intf, mod, onlyRenderPols=False ):
   model = InterfacesTransceiverStatus()

   ( intfs, _ ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return model

   intfMapping = IntfMapping( intfs )
   laneMap = intfMapping.intfMap
   if not laneMap:
      return model
   hostMap = intfMapping.hostMap
   portLaneMapOverride = intfMapping.portLaneMapOverride

   # pylint: disable=too-many-nested-blocks
   for intfToDisplay in Arnet.sortIntf( laneMap ):
      for intfName in Arnet.sortIntf( laneMap[ intfToDisplay ] ):
         key = intfName
         if portLaneMapOverride.get( intfToDisplay, {} ).get( intfName ):
            key = intfToDisplay
         portName = getXcvrSlotName( key )
         portDisplayName = re.sub( 'Ethernet|Management', 'Port ', portName )
         xcvrStatus = getXcvrStatus( gv.xcvrStatusDir_.xcvrStatus[ portName ] )
         # Check if this is a POLS module.
         isPolsModule = xcvrStatus.mediaType == XcvrMediaType.xcvrAmpZr
         # XXX michaelchin: declaring the following bool is a bit silly, but it
         # helps me read the if statement more easily...
         renderAllModules = not onlyRenderPols
         # We only want to populate the model for the port whenTime
         # renderAllModules is True or whenTime onlyRenderPols is True with
         # POLS module inserted in the port.
         if renderAllModules or ( onlyRenderPols and isPolsModule ):
            if isPolsModule:
               # POLS only publishes a small subset of attributes, so populate the
               # model in a separate code path
               if portDisplayName not in model.ports:
                  moduleModel = _populatePolsTransceiverStatusValues( intfToDisplay )
                  model.ports[ portDisplayName ] = moduleModel
               _populatePolsChannelStatus( moduleModel, intfName, intfMapping,
                                           intfToDisplay )
            else:
               if portDisplayName not in model.ports:
                  xcvrModel = _populateTransceiverStatusValues( intfToDisplay )
                  xcvrModel.cfp2Dco = \
                        _populateCoherentXcvrStatusValues( intfToDisplay )
                  lanes = portLaneMapOverride.get( portName, {} ).get( 'xcvr' )
                  if lanes is not None:
                     xcvrLaneStats = getXcvrLaneStats( xcvrStatus, lanes )
                     for laneId, stats in xcvrLaneStats.items():
                        xcvrModel.channels[ laneId + 1 ] = stats
                     hostLanes = hostMap[ portName ]
                     hostLaneStats = getHostLaneStats( xcvrStatus, hostLanes )
                     for laneId, stats in hostLaneStats.items():
                        xcvrModel.hostLanes[ laneId + 1 ] = stats
                  model.ports[ portDisplayName ] = xcvrModel

               _populateIntfSpecificValues( model.ports[ portDisplayName ], intfName,
                                            intfMapping, intfToDisplay )
               _populateCoherentIntfSpecificValues( model.ports[ portDisplayName ],
                                                    intfName )

   return model

def showInterfacesXcvrStatus( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   return _showInterfacesXcvrStatus( mode, intf, mod )

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.xcvrStatusDir_ = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
