# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliToken.Reset
# pylint: disable-next=consider-using-from-import
import CliPlugin.ConfigMgmtMode as ConfigMgmtMode
import CliPlugin.DmfCli as DmfCli # pylint: disable=consider-using-from-import
from CliPlugin.ZtnModel import ConfigDigest
import ConfigMount
import LazyMount
import ShowCommand
import Tac
from TypeFuture import TacLazyType

ztnConfig = None
ztnStatus = None
ZtnState = TacLazyType( 'Ztn::SyncState' )
HashAlgo = TacLazyType( 'Ztn::HashAlgorithm' )

#--------------------------------------------------------------------------------
# reset management dmf controller sync
#--------------------------------------------------------------------------------

class ResetDmfControllerSyncCmd( ShowCommand.ShowCliCommandClass ):
   # This command is run by the DMF controller via CAPI and expects a return value
   # to be in the following JSON model
   #
   # Success case:
   # "result": [
   #   {
   #     "controllerDigest": "deadbeef...",
   #     "runningConfigDigest": "abcdef..."
   #   }
   # ]
   #
   # Failure case:
   # "error": {
   #   "data": [
   #     "errors": [ "Sync request timed out" ]
   #   ]
   # }
   #
   # Since Cli infrastructure doesn't support cliModel for Enable mode commands,
   # this command class is inheriting from ShowCliCommandClass as a workaround.
   syntax = 'reset management dmf controller sync'
   data = {
         'reset': CliToken.Reset.resetMatcher,
         'management': ConfigMgmtMode.managementKwMatcher,
         'dmf': DmfCli.dmfMatcher,
         'controller': 'DMF controller',
         'sync': 'Synchronize running-config and software image'
         }
   cliModel = ConfigDigest

   @staticmethod
   def handler( mode, args ):
      config = ConfigDigest()
      manifestTs = ztnStatus.manifestTs
      ztnConfig.forceSync += 1
      try:
         Tac.waitFor( lambda: ( ztnStatus.manifestTs > manifestTs and
                                ztnStatus.syncState == ZtnState.complete ),
               warnAfter=None, sleep=True, maxDelay=0.5, timeout=300 )
      except Tac.Timeout:
         mode.addError( 'Sync request timed out' )
      else:
         config.controllerDigest = ztnStatus.manifestConfigChecksum.get(
                                       HashAlgo.sha256, '' )
         config.runningConfigDigest = ztnStatus.runningCfgDigest.get(
                                       HashAlgo.sha1, '' )
      return config

BasicCli.addShowCommandClass( ResetDmfControllerSyncCmd )

def Plugin( em ):
   global ztnConfig
   global ztnStatus
   ztnConfig = ConfigMount.mount( em, 'ztn/config', 'Ztn::Config', 'w' )
   ztnStatus = LazyMount.mount( em, 'ztn/status', 'Ztn::Status', 'r' )
