# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ArnetModel
import datetime
import Tac
from CliModel import Bool
from CliModel import Enum
from CliModel import Float
from CliModel import List
from CliModel import Model
from CliModel import Str

class Digest( Model ):
   __public__ = False
   sha256 = Str( help="EOS software image SHA-256 hash" )

   def render( self ):
      print( 'Digest (SHA-256):', self.sha256 )

class ConfigDigest( Model ):
   controllerDigest = Str( help="SHA-256 digest of the config received from "
                                 "the controller" )
   runningConfigDigest = Str( help="SHA-1 digest of the running-config of "
                                   "the switch" )

class ZtnStatus( Model ):
   enabled = Bool( "ZTN enabled" )
   controllers = List( help="List of controllers used for ZTN",
                       valueType=ArnetModel.IpGenericAddress )
   manifestTimestamp = Float( "Time of manifest download in UTC" )
   state = Enum( help="State of ZTN on the switch",
                 values=Tac.Type( "Ztn::SyncState" ).attributes )
   errorMessage = Str( "Reason for ZTN handshake failure",
                       optional=True )

   @staticmethod
   def stateStr( state ):
      return { 'none': 'Inactive',
               'init': 'Initializing',
               'manifestDownloadFailed': 'Downloading manifest failed',
               'invalidManifest': 'Invalid manifest downloaded',
               'manifestReady': 'Manifest is processed',
               'configDownloadInProgress': 'Configuration is being downloaded '
                  'from the controller',
               'configDownloadFailed': 'Configuration download from the controller '
                  'failed',
               'configDownloaded': 'Configuration downloaded from the controller. '
                  'Applying on the switch is in progress',
               'configSaveComplete': 'Configuration applied on the switch '
                  'successfully',
               'invalidConfig': 'Configuration received from the controller '
                  'is invalid',
               'configSaveError': 'Error occurred while applying configuration '
                  'on the switch',
               'swiDownloadInProgress': 'Image is being downloaded from '
                  'the controller',
               'swiDownloadFailed': 'Image download from the controller failed',
               'swiDownloaded': 'Image has been downloaded from the controller. '
                  'Installation is in progress',
               'invalidSwi': 'Image from the controller is invalid and its '
                  'installation failed',
               'swiInstallComplete': 'Image installation is complete',
               'complete': 'Zerotouch handshake is complete',
               'rebooting': 'Switch is about to reboot',
                  }[ state ]

   def render( self ):
      if not self.enabled:
         print( "ZTN is disabled" )
         return
      else:
         print( "ZTN is active" )
      print( "Controllers:", " ".join( str( c ) for c in self.controllers ) )
      tLabel = datetime.timezone.utc
      dt = datetime.datetime.fromtimestamp( self.manifestTimestamp, tz=tLabel )
      dt = dt.astimezone()
      print( "Manifest timestamp:", dt.strftime( '%Y-%m-%d %Z %H:%M:%S.%f' ) )
      zerotouchStateStr = f"Zerotouch state: { self.stateStr( self.state)}"
      if self.errorMessage:
         zerotouchStateStr = f"{zerotouchStateStr} (reason: {self.errorMessage})"
      print( zerotouchStateStr )

class LogMsg( Model ):
   timestamp = Float( "Timestamp of the log" )
   message = Str( "Log message" )

class ZtnStatusDetail( Model ):
   enabled = Bool( "ZTN enabled" )
   platform = Str( "Platform name sent to the controller for establishing "
                   "connection" )
   systemMac = ArnetModel.MacAddress( "MAC address of the system sent to the "
                                      "controller for establishing connection" )
   configUrl = Str( "URL used by ZTN to download startup-config" )
   configDigest = Str( "SHA-256 digest of the downloaded startup-config" )
   imageUrl = Str( "URL used by ZTN to download EOS image" )
   imageDigest = Str( "SHA-256 digest of the downloaded image" )
   logMessages = List( valueType=LogMsg, help="Most recent ZTN log messages" )

   def render( self ):
      if not self.enabled:
         print( "ZTN is disabled" )
         return
      print( "Platform:", self.platform )
      print( "MAC:", self.systemMac )
      print( "Config URL:", self.configUrl )
      print( "Config digest:", self.configDigest )
      print( "SWI URL:", self.imageUrl )
      print( "SWI digest:", self.imageDigest )
      print( "Log:" )
      for log in self.logMessages:
         tLabel = datetime.timezone.utc
         dt = datetime.datetime.fromtimestamp( log.timestamp, tz=tLabel )
         dt = dt.astimezone()
         print( dt.strftime( '%Y-%m-%d %Z %H:%M:%S.%f' ), " ", log.message )
