# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import namedtuple
import enum
import io
import json
import tempfile

import CliCommon
import CliDiff
import ShowRunOutputModel
import Tac
import Tracing
from CliDiff import DiffModel

th = Tracing.defaultTraceHandle()
trace = th.trace0
traceTopSort = th.trace1
traceDetail = th.trace2
traceDiff = th.trace3

# Marker for commands that we want to skip saving.
SKIP_COMMAND_MARKER = '\xff'

ShowRunningConfigOptions = namedtuple(
   'option',
   [ 'saveAll', # save default config
     'saveAllDetail',
     'saveCleanConfig', # if False, do not show always-on default config
     'showNoSeqNum',
     'secureMonitor',
     'commandTag',
     'showProfileExpanded',
     'showFilteredRoot',
     'intfFilter',
     'expandMergeRange' ]
)
ShowRunningConfigOptions.__new__.__defaults__ = ( False, # saveAll
                                                  False, # saveAllDetail
                                                  True, # saveCleanConfig
                                                  False, # showNoSeqNum
                                                  False, # secureMonitor
                                                  None, # commandTag
                                                  False, # showProfileExpanded
                                                  False, # showFilteredRoot
                                                  None, # intfFilter
                                                  True, # expandMergeRange
                                                 )

ShowRunningConfigRenderOptions = namedtuple( 'renderOption', [ 'showSanitized',
                                                               'showJson',
                                                               'showHeader' ] )
ShowRunningConfigRenderOptions.__new__.__defaults__ = ( False, # showSanitized
                                                        False, # showJson,
                                                        True, # showHeader
                                                        )

class AcceptOption( enum.Enum ):
   ACCEPT_MERGE = 0
   ACCEPT_THEIRS = 1
   ACCEPT_MINE = 2

# -----------------------------------------------------------------------------------
# The model here is that the CLI save output comprises a set of SaveBlocks, of two
# kinds: CommandSequences and ModeCollections.  A CommandSequence comprises an
# ordered sequence of CLI commands.  A ModeCollection comprises a set of Mode
# instances, each of which recursively contains a set of SaveBlocks.
#
# SaveBlocks may have dependencies between them which are used to compute the order
# in which they are output.
#
# For examples of how to write a CliSave plugin, see AID95.
# -----------------------------------------------------------------------------------
class SaveBlock:
   """Abstract baseclass for all SaveBlocks, the basic unit from which the CLI save
   output is produced."""

   def generateSaveBlockModel( self, param ):
      """ Translate the SaveBlocks that potentially depend on Sysdb into a model
          that is completely self-contained """
      raise NotImplementedError

   def empty( self, param ):
      """ Returns True if there are no commands in this SaveBlock."""
      raise NotImplementedError

   def content( self, param ):
      """The content function returns a tuple to facilitate the range merge
      feature where multiple instances of a certain config mode (e.g., VLAN)
      can be displayed as a range if they have identical content."""
      raise NotImplementedError

sanitizedString = "<removed>"

class SensitiveCommand:
   """A command that contains both normal and sanitized output. It can be
   given to a CommandSequence."""
   __slots__ = ( 'format_', 'tokens_' )

   def __init__( self, formatStr, *tokens ):
      # Format string contains "{}" to be substituted by tokens, such as
      # "password {}". If the format string is constructed dynamically,
      # consider using CliSave.escapeFormatString() so it does not contain
      # any { or } characters.
      #
      # Note all tokens are sensitive and will be replaced by sanitizedString.
      assert not formatStr.endswith( '\n' )
      # <string>.format() doesn't care if there are more tokens, so let's
      # add a bit more sanity check. using >= to account for escaped {{}}
      # just in case.
      assert formatStr.count( '{}' ) >= len( tokens )
      self.format_ = formatStr
      self.tokens_ = tokens

   def normalOutput( self ):
      return self.format_.format( *self.tokens_ )

   def sanitizedOutput( self ):
      return self.format_.format( *( sanitizedString, ) * len( self.tokens_ ) )

   def output( self, options ):
      if options.showSanitized:
         return self.sanitizedOutput()
      return self.normalOutput()

   def __eq__( self, other ):
      return ( isinstance( other, SensitiveCommand ) and
               self.format_ == other.format_ and
               self.tokens_ == other.tokens_ )

   def __str__( self ):
      return self.format_

   def __repr__( self ):
      return f"SensitiveCommand({self.format_!r})"

class CommandSequence( SaveBlock ):
   """A SaveBlock that comprises a simple ordered sequence of CLI commands.  Commands
   may be added to the sequence via the 'addCommand' method, or via one of the
   convenience methods 'writeAttr' or 'writeBoolAttr'.

   This class should not be instantiated directly by plugins; instead, a command
   sequence name should be registered with a Mode subclass by calling
   addCommandSequence( name ) on that Mode subclass, and then a CommandSequence
   object should be obtained by doing mode[ name ] on an instance of that Mode
   subclass."""

   __slots__ = ( 'commands_', 'name_', 'useInsertionOrder_' )

   def __init__( self, name, useInsertionOrder=False ):
      SaveBlock.__init__( self )
      self.commands_ = []
      self.name_ = name
      self.useInsertionOrder_ = useInsertionOrder

   def generateSaveBlockModel( self, param ):
      cmds = [ c for c in self.commands_ if c != SKIP_COMMAND_MARKER ]
      return CommandSequenceModel( self.name_, cmds, self.useInsertionOrder_ )

   def addCommand( self, command ):
      # make sure people don't add an empty command. This generates an empty line on
      # the running-config which is wrong
      assert command

      # make sure people don't inadvertently create empty lines by adding a \n to
      # their commands, but do allow multiline commands that have "inner carriage
      # returns" in json output format (like 'banner motd' or capi's certificates)
      if isinstance( command, str ):
         assert not command.endswith( '\n' )
      self.commands_.append( command )

   def empty( self, param ):
      return not self.commands_

   def content( self, param ):
      return tuple( self.commands_ )

class ModeCollection( SaveBlock ):
   """A SaveBlock that comprises a set of Mode instances for a particular CLI
   mode.

   This class should not be instantiated directly by plugins; instead, a child Mode
   subclass should be registered with a parent Mode subclass by calling
   addChildMode( cls ) on that parent Mode subclass, and then a ModeCollection
   object should be obtained by doing mode[ cls ] on an instance of that parent
   Mode subclass."""

   def __init__( self, modeClass ):
      # TODO: would be nice to fix this assert
      # assert issubclass( modeClass, CliSaveMode.Mode )
      SaveBlock.__init__( self )
      self.modeClass_ = modeClass
      self.modeInstanceMap_ = {}

   def content( self, param ):
      # pylint: disable-next=consider-using-generator
      return tuple( [ ( mode.enterCmd(), saveBlock.content( mode.param_ ) )
                      for mode in self.modeInstanceMap_.values()
                      for saveBlock in mode.saveBlocks_ ] )

   def _generateModeRange( self, param ):
      # Merge modes with the same block together.
      # contentMap maps content to the first mode with unique commands.
      contentMap = {}
      # modeMap maps the first mode with unique commands to a list of
      # modes with identical content.
      modeMap = {}
      for i in sorted( self.modeInstanceMap_.values() ):
         if i.hideInactive( param ):
            continue
         if i.hideUnconnected( param ):
            continue
         if i.empty( param ):
            continue
         if not i.canMergeRange():
            modeMap[ i ] = None
            continue
         content = i.content( param )
         mode = contentMap.get( content )
         if mode:
            # we can merge
            modeMap[ mode ].append( i )
         else:
            contentMap[ content ] = i
            modeMap[ i ] = [ i ]

      saveBlockModel = ModeCollectionModel( self.modeClass_ )
      for mode in sorted( modeMap.keys() ):
         m = modeMap[ mode ]
         enterCmd = mode.enterRangeCmd( m ) if m else mode.enterCmd()
         saveBlockModel.addSaveBlockModel(
               self._generateModeSaveBlocks( param, mode, enterCmd=enterCmd ) )

      return saveBlockModel

   def _generateModeSaveBlocks( self, param, mode, enterCmd=None ):
      enterCmd = mode.enterCmd() if enterCmd is None else enterCmd
      saveBlockModel = ModeEntryModel( enterCmd, mode.comments( param ),
            mode.modeSeparator(), mode.revertCmd() )
      mode.expandMode( param )
      for b in mode.saveBlocks_:
         if b.empty( param ):
            continue
         saveBlockModel.addSaveBlockModel( b.generateSaveBlockModel( param ) )
      return saveBlockModel

   def generateSaveBlockModel( self, param ):
      if ( param.options.expandMergeRange and
            self.modeClass_.mergeRange and len( self.modeInstanceMap_ ) > 1 ):
         return self._generateModeRange( param )

      saveBlockModel = ModeCollectionModel( self.modeClass_ )

      # This has been perfed; please don't "simplify" w/o perfing.
      if self.modeInstanceMap_:
         for mode in sorted( self.modeInstanceMap_.values() ):
            if ( mode.hideInactive( param ) or
                 mode.hideUnconnected( param ) or
                 mode.empty( param ) ): # TODO: combine as one call?
               continue

            blocks = self._generateModeSaveBlocks( param, mode )
            saveBlockModel.addSaveBlockModel( blocks )

      return saveBlockModel

   def empty( self, param ):
      """The ModeCollection is empty (no commands) if all Modes in the collection
      are empty. Note that currently, Modes are never empty, since they always
      consist of at least their enterCmd(), but it felt wrong relying on that."""

      for i in self.modeInstanceMap_.values():
         if not i.empty( param ):
            return False
      return True

   def getOrCreateModeInstance( self, param ):
      if param not in self.modeInstanceMap_:
         if param is not None:
            assert None not in self.modeInstanceMap_, \
               "singleton instance has to use getSingletonInstance()"
         self.modeInstanceMap_[ param ] = self.modeClass_( param )
      return self.modeInstanceMap_[ param ]

   def getSingletonInstance( self ):
      # This is a special case for singleton modes that don't have keys
      # We just use a fixed key.
      if ( len( self.modeInstanceMap_ ) == 1 and
           list( self.modeInstanceMap_ )[ 0 ] is not None or
           len( self.modeInstanceMap_ ) > 1 ):
         assert False, "singleton instance has to use getSingletonInstance()"
      return self.getOrCreateModeInstance( None )

class CliMergeConflict( Exception ):
   def __init__( self, message, ancestorSaveBlock, theirSaveBlock, mySaveBlock ):
      self.message_ = message
      self.ancestorSaveBlock = ancestorSaveBlock
      self.theirSaveBlock = theirSaveBlock
      self.mySaveBlock = mySaveBlock
      super().__init__( self.message_ )

   def renderConflictMsg( self, stream, ancestorConfigName, theirConfigName,
                          myConfigName ):
      stream.write(
            'Merge conflict detected: unable to generate merged config\n' )
      stream.write( 'Please use \'show session-config diffs\' and '
            '\'show running-config diffs session-config ancestor\' for additional '
            'information.\nSpecific conflict is printed below:\n\n' )
      self.renderConflict( stream, ancestorConfigName, theirConfigName,
                           myConfigName )

   def renderConflict( self, stream, ancestorConfigName, theirConfigName,
                       myConfigName ):
      # This is always called with default render options
      renderOptions = ShowRunningConfigRenderOptions()
      with tempfile.NamedTemporaryFile( mode='w+' ) as ancestorFile:
         with tempfile.NamedTemporaryFile( mode='w+' ) as theirFile:
            with tempfile.NamedTemporaryFile( mode='w+' ) as myFile:
               if self.ancestorSaveBlock:
                  self.ancestorSaveBlock.render( ancestorFile, renderOptions, '' )
               if self.theirSaveBlock:
                  self.theirSaveBlock.render( theirFile, renderOptions, '' )
               if self.mySaveBlock:
                  self.mySaveBlock.render( myFile, renderOptions, '' )
               ancestorFile.flush()
               theirFile.flush()
               myFile.flush()
               diffs = Tac.run( [ 'diff3', '--text', '--strip-trailing-cr', '-m',
                  '-A', '-L', myConfigName, '-L', ancestorConfigName, '-L',
                  theirConfigName, myFile.name, ancestorFile.name, theirFile.name ],
                  stdout=Tac.CAPTURE, ignoreReturnCode=True )
               stream.write( diffs )

class SaveBlockModelBase:
   def render( self, stream, options, prefix ):
      """ Print the save block to the stream """
      raise NotImplementedError

   def getRenderOutput( self, options, prefix ):
      """ Render the output instead of to a stream to a list """
      f = io.StringIO()
      self.render( f, options, prefix )
      return f.getvalue().splitlines()

   def getDiffModel( self, diffModel, options, prefix, theirModel,
                     forceSortBlock=False ):
      """ Generate the DiffModel from save block diff """
      raise NotImplementedError

   def getMergedModel( self, ancestorModel, theirModel,
                       accept=AcceptOption.ACCEPT_MERGE ):
      """ Generate save block model for the 'merged' config """
      raise NotImplementedError

   def generateCliModel( self, cliModel, options ):
      """ Given a cliModel this function will fill in the cliModel"""
      raise NotImplementedError

   def separator( self ):
      """ Should a separator be printed """
      raise NotImplementedError

   def name( self ):
      """ a unique identifier for this save block """
      raise NotImplementedError

   def hasChanges( self, theirModel ):
      """ return a boolean if the saveblock has any differences """
      raise NotImplementedError

class CommandSequenceModel( SaveBlockModelBase ):
   __slots__ = ( 'commands_', 'name_', 'useInsertionOrder_' )

   def __init__( self, name, commands, useInsertionOrder ):
      self.name_ = name
      self.commands_ = commands
      self.useInsertionOrder_ = useInsertionOrder

   def _getCommand( self, command, options ):
      if isinstance( command, SensitiveCommand ):
         return command.output( options )
      else:
         return command

   def getCommands( self, options ):
      # return a generator
      return ( self._getCommand( c, options ) for c in self.commands_ )

   def render( self, stream, options, prefix ):
      for cmd in self.getCommands( options ):
         for line in cmd.split( '\n' ):
            stream.write( f'{prefix}{line}\n' )

   def getHeadCommands( self, options, prefix ):
      heads = []
      for cmd in self.getCommands( options ):
         head = cmd.split( '\n' )[ 0 ]
         heads.append( f'{prefix}{head}' )
      return heads

   def getDiffModel( self, diffModel, options, prefix, theirModel,
                     forceSortBlock=False ):
      assert isinstance( theirModel, CommandSequenceModel )
      assert self.name() == theirModel.name()
      assert self.useInsertionOrder_ == theirModel.useInsertionOrder_

      def filterFunc( tag, line ):
         # We don't care about lines that don't have a change
         return tag != DiffModel.COMMON

      CliDiff.diffLines(
         diffModel,
         list( theirModel.getCommands( options ) ),
         list( self.getCommands( options ) ),
         prefix=prefix, filterFunc=filterFunc,
         useInsertionOrder=self.useInsertionOrder_ )

   def getMergedModel( self, ancestorModel, theirModel,
                       accept=AcceptOption.ACCEPT_MERGE ):
      cmds = None
      if self.hasChanges( theirModel ):
         # this means that myModel and theirModel differ. If both models differ
         # from ancestor config then that's an error. However if only 1 differs
         # then take that one.
         if ( self.hasChanges( ancestorModel ) and
              theirModel.hasChanges( ancestorModel ) ):
            # this means that myConfig != theirConfig != ancestorConfig
            # raise an exception only if accept=ACCEPT_MERGE.
            # Otherwise, use mine or theirs.
            if accept == AcceptOption.ACCEPT_MERGE:
               raise CliMergeConflict( 'Merge conflict in Command Sequences',
                     ancestorModel, theirModel, self )
            if accept == AcceptOption.ACCEPT_THEIRS:
               cmds = theirModel.commands_
            elif accept == AcceptOption.ACCEPT_MINE:
               cmds = self.commands_
            else:
               assert False, f"Unsupported accept option {accept}"
         elif self.hasChanges( ancestorModel ):
            # this means that myConfig != ancestorConfig, however
            # theirConfig == ancestorConfig so print myConfig as the
            # authoritative one.
            cmds = self.commands_
         elif theirModel.hasChanges( ancestorModel ):
            # this means that theirConfig != ancestorConfig, however
            # myConfig == ancestorConfig so print theirConfig as the
            # authoritative one.
            cmds = theirModel.commands_
         else:
            assert False, 'How did we get here'
      else:
         # this means that theirModel and myModel are the same, so the ancestor
         # config doesn't matter
         cmds = self.commands_

      assert cmds is not None, 'cmds should be set'
      return CommandSequenceModel( self.name_, list( cmds ),
            self.useInsertionOrder_ )

   def hasChanges( self, theirModel ):
      if theirModel is None:
         return True
      return self.commands_ != theirModel.commands_

   def generateCliModel( self, cliModel, options ):
      for command in self.getCommands( options ):
         cliModel.cmds[ command ] = None

   def separator( self ):
      return False

   def name( self ):
      return self.name_

class ModeModelBase( SaveBlockModelBase ):
   __slots__ = ( 'saveBlocks_', )

   def __init__( self ):
      self.saveBlocks_ = []

   def render( self, stream, options, prefix ):
      firstTime = True
      prevNeedSeparator = False
      for saveBlock in self.saveBlocks_:
         needSeparator = saveBlock.separator()

         if self.printSeparator( prefix, firstTime, needSeparator,
               prevNeedSeparator ):
            stream.write( f'{prefix}!\n' )
         firstTime = False
         prevNeedSeparator = needSeparator
         saveBlock.render( stream, options, prefix )

   def generateCliModel( self, cliModel, options ):
      """ Given a cliModel this function will fill in the cliModel"""
      raise NotImplementedError

   def separator( self ):
      """ Should a separator be printed """
      raise NotImplementedError

   def name( self ):
      """ a unique identifier for this save block"""
      raise NotImplementedError

   def _getMergedSaveBlocks( self, ancestorModel, theirModel,
                             accept=AcceptOption.ACCEPT_MERGE ):
      mergedSaveBlocks = []
      ancestorSaveBlocks = { saveBlock.name(): saveBlock
            for saveBlock in ancestorModel.saveBlocks_ } if ancestorModel else {}
      mySaveBlocks = { saveBlock.name(): saveBlock
            for saveBlock in self.saveBlocks_ }
      theirSaveBlocks = { saveBlock.name(): saveBlock
            for saveBlock in theirModel.saveBlocks_ }

      useInsertionOrder = ( self.modeClass_.useInsertionOrder()
            if isinstance( self, ModeCollectionModel ) else False )
      if useInsertionOrder and self.hasChanges( theirModel ):
         # this means that mine and theirs are different AND mode collection
         # has a custom sort function. In this case 3-way merge is a bit more
         # limited. If myModel and theirModel differ, then only 1 of them is allowed
         # to have deviated from the ancestor config. It's a conflict if both of them
         # differ in different ways relative to the ancestor config.
         theirModelChanged = theirModel.hasChanges( ancestorModel )
         myModelChange = self.hasChanges( ancestorModel )
         if theirModelChanged and myModelChange:
            # this means that mine and theirs both made changes to the
            # same mode collection
            if accept == AcceptOption.ACCEPT_MERGE:
               raise CliMergeConflict( 'Their config and my config both made changes'
                     ' to the same set of modes', ancestorModel, theirModel, self )
            if accept == AcceptOption.ACCEPT_THEIRS:
               return theirModel.saveBlocks_
            elif accept == AcceptOption.ACCEPT_MINE:
               return self.saveBlocks_
            else:
               assert False, f"Unsupported accept option {accept}"

         # This means that only 1 model changed relative to the ancestor (or
         # they both changed in the same way). So no more conflict should happen
         if theirModelChanged: # use theirModel
            assert not myModelChange
            return theirModel.saveBlocks_
         elif myModelChange: # use myModel
            assert not theirModelChanged
            return self.saveBlocks_
         else:
            assert False, 'Who changed?!?!'

      saveBlockDiff = DiffModel()
      CliDiff.diffLines(
         saveBlockDiff,
         [ saveBlock.name() for saveBlock in theirModel.saveBlocks_ ],
         [ saveBlock.name() for saveBlock in self.saveBlocks_ ] )
      for block in saveBlockDiff.blocks_: # pylint:disable=too-many-nested-blocks
         tag = block.tag_
         name = block.block_[ 0 ]
         ancestorBlock = ancestorSaveBlocks.get( name )
         mySaveBlock = mySaveBlocks.get( name )
         theirSaveBlock = theirSaveBlocks.get( name )

         if tag == DiffModel.REMOVE:
            # if this assert hits it may be because of BUG876816
            assert mySaveBlock is None
            assert theirSaveBlock is not None
            # This means that that my save block was removed. Don't print out
            # myconfig. To figure out if you want print out theirSaveBlock
            # see if it's different than ancestorSaveConfig.
            if ancestorBlock is None:
               # this means that mySaveBlock and ancestorSaveBlock don't
               # exist. This means that theisSaveBlock was added cleanly
               # so render theirSaveBlock
               mergedSaveBlocks.append( theirSaveBlock )
            elif theirSaveBlock.hasChanges( ancestorBlock ):
               # this means that mySaveBlock removed this saveBlock, and
               # theirSaveBlock modified the saveblock.
               if accept == AcceptOption.ACCEPT_MERGE:
                  raise CliMergeConflict(
                     'Merge conflict: My config removed save block modified by '
                     'their config', ancestorBlock, theirSaveBlock, mySaveBlock )
               if accept == AcceptOption.ACCEPT_THEIRS:
                  mergedSaveBlocks.append( theirSaveBlock )
               elif accept == AcceptOption.ACCEPT_MINE:
                  pass
               else:
                  assert False, f"Unsupported accept option {accept}"
            else:
               # this means that ancestorBlock is the same as theirSaveBlock. This
               # means that myConfig removed this config, so don't print anything
               pass

         elif tag == DiffModel.ADD:
            assert mySaveBlock is not None
            # if this assert hits it may be because of BUG876816
            assert theirSaveBlock is None
            # this means that theirSaveBlock was removed. Don't print out
            # theirSaveBlock. To figure out if we want to print out mySaveBlock
            # see how it's different from ancestorSaveConfig.
            if ancestorBlock is None:
               # this mean that this saveblock didn't exist before and
               # myConfig added this. Just render the new saveblock!
               mergedSaveBlocks.append( mySaveBlock )
            elif mySaveBlock.hasChanges( ancestorBlock ):
               # this means that theirConfig deleted this save block
               # while myconfig modified.
               if accept == AcceptOption.ACCEPT_MERGE:
                  raise CliMergeConflict(
                     'Merge conflict: Their config removed save block modified by '
                     'my config', ancestorBlock, theirSaveBlock, mySaveBlock )
               if accept == AcceptOption.ACCEPT_THEIRS:
                  pass
               elif accept == AcceptOption.ACCEPT_MINE:
                  mergedSaveBlocks.append( mySaveBlock )
               else:
                  assert False, f"Unsupported accept option {accept}"
            else:
               # this means that ancestorBlock is the same as mySaveBlock. This
               # means that myConfig removed this config, so don't print anything
               pass

         elif tag == DiffModel.COMMON:
            # mine and theirs should exist, BUT ancestorBlock may or not
            # exist
            assert mySaveBlock is not None
            assert theirSaveBlock is not None
            if ancestorBlock is None:
               # this means that mySaveBlock and theirSaveBlock indepedently
               # added this save block. If mine and theirs are the same
               # then it should be good. If they are different,
               # 1. if it's a CommandSequence, it's a conflict
               # 2. Otherwise (i.e., it's either ModeCollection or ModeEntry),
               #    recurse down
               if isinstance( mySaveBlock, CommandSequenceModel ):
                  if mySaveBlock.hasChanges( theirSaveBlock ):
                     if accept == AcceptOption.ACCEPT_MERGE:
                        raise CliMergeConflict(
                              'Merge conflict: Their config and my config added a'
                              'new command-sequence save block but differ',
                              ancestorBlock, theirSaveBlock, mySaveBlock )
                     if accept == AcceptOption.ACCEPT_THEIRS:
                        mergedSaveBlocks.append( theirSaveBlock )
                     elif accept == AcceptOption.ACCEPT_MINE:
                        mergedSaveBlocks.append( mySaveBlock )
                     else:
                        assert False, f"Unsupported accept option {accept}"
                  else:
                     # this means that mine and theirs both got added
                     # but are the same, so just print out the entire
                     # subtree
                     mergedSaveBlocks.append( mySaveBlock )
               else:
                  # Recurse down
                  mergedSaveBlocks.append(
                     mySaveBlock.getMergedModel( ancestorBlock, theirSaveBlock,
                                                 accept=accept ) )
            else:
               # this means that everything is the same. Recurse down to find
               # any additional differences
               mergedSaveBlocks.append(
                     mySaveBlock.getMergedModel( ancestorBlock, theirSaveBlock,
                                                 accept=accept ) )
      # BUG972060: sort mergedSaveBlocks based on CliSave.Mode instanceKey()
      return mergedSaveBlocks

   def hasChanges( self, theirModel ):
      '''
         Detects if 2 sets of save blocks are different or not. This also include
         if the save blocks are ordered differently. Normally save blocks are
         stored in the same order because they are sorted by the instanceKey,
         and the instanceKey doesn't normally change for a mode. However
         some mode do change their instanceKey, so their sorted position
         change. In cases like this we want to be able to also flag that
         the save blocks are differnt as well.
      '''
      if theirModel is None:
         return True
      if len( self.saveBlocks_ ) != len( theirModel.saveBlocks_ ):
         return True

      for idx, mySaveBlock in enumerate( self.saveBlocks_ ):
         # iterate through all of the save blocks and see if and of the saveblocks
         # underneath have any changes
         theirSaveBlock = theirModel.saveBlocks_[ idx ]
         if ( mySaveBlock.name() != theirSaveBlock.name() or
               mySaveBlock.hasChanges( theirSaveBlock ) ):
            return True

      # yay there are no changes!
      return False

   def addSaveBlockModel( self, saveBlockModel ):
      self.saveBlocks_.append( saveBlockModel )

   def getDiffModel( self, diffModel, options, prefix, theirModel,
                     forceSortBlock=False ):
      trace( 'getDiffModel' )
      mySaveBlocks = { saveBlock.name(): saveBlock
            for saveBlock in self.saveBlocks_ }
      trace( 'my save blocks names', list( mySaveBlocks.keys() ) )
      theirSaveBlocks = { saveBlock.name(): saveBlock
            for saveBlock in theirModel.saveBlocks_ }
      trace( 'their blocks names', list( theirSaveBlocks.keys() ) )

      firstTime = True
      prevNeedSeparator = False

      saveBlockDiff = DiffModel()
      useInsertionOrder = ( self.modeClass_.useInsertionOrder()
            if isinstance( self, ModeCollectionModel ) else False )
      theirSaveBlockNames = [ saveBlock.name() for saveBlock in
                              theirModel.saveBlocks_ ]
      mySaveBlockNames = [ saveBlock.name() for saveBlock in self.saveBlocks_ ]
      if not useInsertionOrder and forceSortBlock:
         # Workaround for BUG972060
         # mySaveblock or theirSaveBlock may be out of order
         theirSaveBlockNames = sorted( theirSaveBlockNames )
         mySaveBlockNames = sorted( mySaveBlockNames )
      CliDiff.diffLines( saveBlockDiff, theirSaveBlockNames, mySaveBlockNames,
                         useInsertionOrder=useInsertionOrder )
      for block in saveBlockDiff.blocks_:
         tag = block.tag_
         name = block.block_[ 0 ]
         mySaveBlock = mySaveBlocks.get( name )
         theirSaveBlock = theirSaveBlocks.get( name )

         needSeparator = ( mySaveBlock.separator()
               if mySaveBlock else theirSaveBlock.separator() )
         printSeparator = self.printSeparator( prefix, firstTime, needSeparator,
               prevNeedSeparator )

         if tag == DiffModel.REMOVE:
            # myConfig has remove this mode
            if not useInsertionOrder:
               # if this assert hits it may be because of BUG876816
               assert mySaveBlock is None
            assert theirSaveBlock is not None

            if printSeparator:
               diffModel.append( DiffModel.REMOVE, f'{prefix}!' )
            # All ModeEntryModel must call append() with a list of lines
            if isinstance( theirSaveBlock, ModeEntryModel ):
               assert theirSaveBlock.revertCmd_ is not None
               diffModel.append( DiffModel.REMOVE,
                                 theirSaveBlock.getRenderOutput( options, prefix ),
                                 theirSaveBlock.revertCmd_ )
            elif isinstance( theirSaveBlock, ModeCollectionModel ):
               for childMode in theirSaveBlock.saveBlocks_:
                  # Only ModeEntryModel can be in ModeCollectionModel
                  assert isinstance( childMode, ModeEntryModel )
                  assert childMode.revertCmd_ is not None
                  diffModel.append( DiffModel.REMOVE,
                                    childMode.getRenderOutput( options, prefix ),
                                    childMode.revertCmd_ )
            elif isinstance( theirSaveBlock, CommandSequenceModel ):
               # For remove, only put head of each possible multi-line commands
               for headCmd in theirSaveBlock.getHeadCommands( options, prefix ):
                  diffModel.append( DiffModel.REMOVE, headCmd )
            else:
               assert False

         elif tag == DiffModel.ADD:
            # myConfig has added this mode
            assert mySaveBlock is not None
            if not useInsertionOrder:
               # if this assert hits it may be because of BUG876816
               assert theirSaveBlock is None

            if printSeparator:
               diffModel.append( DiffModel.ADD, f'{prefix}!' )
            diffModel.append( DiffModel.ADD,
                              mySaveBlock.getRenderOutput( options, prefix ) )

         elif tag == DiffModel.COMMON:
            # both modes exist so lets recurse down and try to find
            # differences
            assert mySaveBlock is not None
            assert theirSaveBlock is not None
            if not mySaveBlock.hasChanges( theirSaveBlock ):
               continue
            if printSeparator:
               diffModel.append( DiffModel.COMMON, f'{prefix}!' )

            mySaveBlock.getDiffModel( diffModel, options, prefix, theirSaveBlock,
                                      forceSortBlock )
         else:
            assert False

         firstTime = False
         prevNeedSeparator = needSeparator

   def printSeparator( self, prefix, firstTime, saveBlockSeparator,
         prevNeedSeparator ):
      raise NotImplementedError

class ModeCollectionModel( ModeModelBase ):
   __slots__ = ( 'modeClass_', )

   def __init__( self, modeClass ):
      super().__init__()
      self.modeClass_ = modeClass

   def addSaveBlockModel( self, saveBlockModel ):
      # A Mode collection can only contain modes
      assert not isinstance( saveBlockModel,
            ( ModeCollectionModel, CommandSequenceModel ) )
      assert isinstance( saveBlockModel, ModeEntryModel )

      # call the base class
      super().addSaveBlockModel( saveBlockModel )

   def getMergedModel( self, ancestorModel, theirModel,
                       accept=AcceptOption.ACCEPT_MERGE ):
      """ Generate save block model for the 'merged' config """
      result = ModeCollectionModel( self.modeClass_ )
      for saveBlock in self._getMergedSaveBlocks( ancestorModel, theirModel,
                                                  accept=accept ):
         result.addSaveBlockModel( saveBlock )
      return result

   def printSeparator( self, prefix, firstTime, saveBlockSeparator,
         prevNeedSeparator ):
      # Write a newline to separate this instance of the Mode subclass
      # from the previous instance in the ModeCollection.
      return not firstTime and ( prevNeedSeparator or saveBlockSeparator )

   def generateCliModel( self, cliModel, options ):
      for saveBlock in self.saveBlocks_:
         saveBlock.generateCliModel( cliModel, options )

   def separator( self ):
      return True

   def name( self ):
      # pylint: disable-next=consider-using-f-string
      return '%s-%d' % ( self.modeClass_, id( self.modeClass_ ) )

class ModeEntryModel( ModeModelBase ):
   __slots__ = ( 'enterCmd_', 'comment_', 'separator_', 'revertCmd_' )

   def __init__( self, enterCmd, comment, separator, revertCmd=None ):
      super().__init__()
      self.enterCmd_ = enterCmd
      self.comment_ = comment
      self.separator_ = separator
      self.revertCmd_ = revertCmd

   def addSaveBlockModel( self, saveBlockModel ):
      # A mode can't contain modes directly, but can have a ModeCollection
      # which can contain a mode
      assert not isinstance( saveBlockModel, ModeEntryModel )
      assert isinstance( saveBlockModel,
            ( ModeCollectionModel, CommandSequenceModel ) )

      # call the base class
      super().addSaveBlockModel( saveBlockModel )

   def render( self, stream, options, prefix ):
      if self.enterCmd_:
         # all mode except for the global config mode should have an enter cmd
         stream.write( f'{prefix}{self.enterCmd_}\n' )
         prefix = '%s   ' % prefix # pylint: disable=consider-using-f-string

      for line in self.commentContent( prefix ):
         stream.write( '%s\n' % line ) # pylint: disable=consider-using-f-string

      super().render( stream, options, prefix )

   def getDiffModel( self, diffModel, options, prefix, theirModel,
                     forceSortBlock=False ):
      if self.enterCmd_:
         # all mode except for the global config mode should have an enter cmd
         diffModel.append( DiffModel.COMMON, f'{prefix}{self.enterCmd_}' )
         prefix = '%s   ' % prefix # pylint: disable=consider-using-f-string

      # diff the comment
      myComment = self.commentContent( prefix )
      theirComment = theirModel.commentContent( prefix )
      if myComment or theirComment:
         CliDiff.diffLines( diffModel, theirComment, myComment )

      # diff all of the save blocks
      super().getDiffModel( diffModel, options, prefix, theirModel, forceSortBlock )

   def getMergedModel( self, ancestorModel, theirModel,
                       accept=AcceptOption.ACCEPT_MERGE ):
      """ Generate save block model for the 'merged' config """
      ancestorComment = ancestorModel.comment_ if ancestorModel else None
      theirComment = theirModel.comment_
      myComment = self.comment_

      comment = None
      if myComment != theirComment:
         # if the configs are different figure out how they compare to the ancestor
         # config

         # pylint: disable-next=consider-using-in
         if theirComment != ancestorComment and myComment != ancestorComment:
            # this means that mine and theirs both made changes to the same
            # comment.
            if accept == AcceptOption.ACCEPT_MERGE:
               raise CliMergeConflict( 'Their config and my config both made changes'
                     ' to the same comment', ancestorModel, theirModel, self )
            if accept == AcceptOption.ACCEPT_THEIRS:
               comment = theirComment
            elif accept == AcceptOption.ACCEPT_MINE:
               comment = myComment
            else:
               assert False, f"Unsupported accept option {accept}"
         elif theirComment != ancestorComment:
            # this means that myComment or theirComment changed, but one of them
            # is the same as the ancestorConfig. Let find out which one changed
            assert myComment == ancestorComment
            # this means theirs changed so print that one
            comment = theirComment
         elif myComment != ancestorComment:
            assert theirComment == ancestorComment
            # this means my comment changed, so print that one
            comment = myComment
         else:
            assert False, 'Which comment should we choose???'
      else:
         # this mean that mine and theirs are the same, it doesn't really matter
         # what the ancestor was if it was changed in the same way
         comment = myComment

      result = ModeEntryModel( self.enterCmd_, comment, self.separator_,
                               self.revertCmd_ )
      for saveBlock in self._getMergedSaveBlocks( ancestorModel, theirModel,
                                                  accept=accept ):
         result.addSaveBlockModel( saveBlock )
      return result

   def commentContent( self, prefix ):
      if self.comment_:
         # pylint: disable-next=consider-using-f-string
         return [ '{}{}{}{}'.format( prefix, CliCommon.commentAppendStr, ' ', line )
                  for line in self.comment_.splitlines() ]
      return []

   def printSeparator( self, prefix, firstTime, saveBlockSeparator,
         prevNeedSeparator ):
      # Write a newline to separate it from the previous SaveBlock
      # also always print out a separetor between saveblocks for global mode
      return not firstTime and ( not prefix or saveBlockSeparator )

   def generateCliModel( self, cliModel, options ):
      if self.enterCmd_:
         newModel = ShowRunOutputModel.Mode()
         cliModel.cmds[ self.enterCmd_ ] = newModel
         cliModel = newModel
      cliModel.header = None # only global config has the header field
      cliModel.comments = ( self.comment_.splitlines()
            if self.comment_ is not None else [] )
      for saveBlock in self.saveBlocks_:
         saveBlock.generateCliModel( cliModel, options )

   def separator( self ):
      return self.separator_

   def name( self ):
      return self.enterCmd_

   def hasChanges( self, theirModel ):
      if theirModel is None:
         return True
      if self.comment_ != theirModel.comment_:
         # if comments aren't the same then there are changes
         return True

      # return base cls implementation
      return super().hasChanges( theirModel )

class SaveBlockModelRenderer:
   @staticmethod
   def render( stream, options, headers, rootSaveBlock ):
      if options.showJson:
         SaveBlockModelRenderer._writeJsonToStream( headers, stream, options,
                                                    rootSaveBlock )
      else:
         # write the text to the stream
         for header in headers:
            stream.writelines( ( header, "\n!\n" ) )
         rootSaveBlock.render( stream, options, '' )
         stream.writelines( ( '!\n', 'end\n' ) )

   @staticmethod
   def _writeJsonToStream( headers, stream, options, rootSaveBlock ):
      # generate them model
      cliModel = ShowRunOutputModel.Mode()
      rootSaveBlock.generateCliModel( cliModel, options )
      cliModel.header = headers # add the header to the root

      # print the model to the stream
      stream.flush()
      # TODO: don't generate the model at all and have generateCliModel
      # actually generate json (with a name change)
      json.dump( cliModel.toDict(), stream )
      stream.flush()

   @staticmethod
   def _getDiffModel( options, theirHeaders, myHeaders,
         theirSaveBlock, mySaveBlock, forceSortBlock=False ):
      assert not options.showJson

      diffModel = CliDiff.DiffModel()

      if options.showHeader:
         if myHeaders != theirHeaders:
            myHeader = '\n'.join( myHeaders ).split( '\n' )
            theirHeader = '\n'.join( theirHeaders ).split( '\n' )

            def filterFunc( tag, line ):
               # skip empty lines
               return line.strip()
            CliDiff.diffLines( diffModel, theirHeader, myHeader,
                               filterFunc=filterFunc )

      mySaveBlock.getDiffModel( diffModel, options, '', theirSaveBlock,
                                forceSortBlock )
      return diffModel

   @staticmethod
   def renderDiff( stream, options, theirHeaders, myHeaders,
         theirSaveBlock, mySaveBlock ):
      diffModel = SaveBlockModelRenderer._getDiffModel( options,
                     theirHeaders, myHeaders, theirSaveBlock, mySaveBlock )
      diffModel.render( stream )

   @staticmethod
   def renderDiffCliCommands( stream, options, theirHeaders, myHeaders,
         theirSaveBlock, mySaveBlock, forceSortBlock=False ):
      diffModel = SaveBlockModelRenderer._getDiffModel( options,
                     theirHeaders, myHeaders, theirSaveBlock, mySaveBlock,
                     forceSortBlock )
      cliCommands = diffModel.getCliCommands()
      for cmd in cliCommands:
         stream.write( f'{cmd}\n' )
