# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave, re, MultiRangeRule
from CliMode.Aaa import RoleMode
from AaaDefs import authzBuiltinRoles
import SecretCli
from AaaPluginLib import primarySshKeyId

CliSave.GlobalConfigMode.addCommandSequence( 'Aaa.global' )
CliSave.GlobalConfigMode.addCommandSequence( 
   'Aaa.encrypt',
   # root password needs to be high-priority 
   before=[ 'config.priority', 'Aaa.global' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Aaa.users', after=[ 'Aaa.encrypt' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Aaa.netaccount' )

class RoleConfigMode( RoleMode, CliSave.Mode ):
   def __init__( self, param ):
      RoleMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RoleConfigMode, after=[ 'Aaa.global' ] )
RoleConfigMode.addCommandSequence( 'Aaa.role' )

# I can't use a specific CliSave.saver for Aaa::AuthenMethodList because I
# don't know from the type whether it's a login method list or an enable method
# list, so I do that from the main Aaa::Config saver.
@CliSave.saver( 'Aaa::Config', 'security/aaa/config' )
def saveAaaConfig( cfg, root, requireMounts, options ):
   cmds = root[ 'Aaa.global' ]
   saveAll = options.saveAll

   def _genAuthnCmd( ml, type ): # pylint: disable=redefined-builtin
      m = ml.method
      s = " ".join( [ m[ i ] for i in range( 0, len( m ) ) ] )
      # 'login' is a special alias for 'console'
      name = ml.name if ml.name != 'login' else 'console'
      return "aaa authentication %s %s %s" %( type, name, s ) 


   # If the default login method list has been configured to have anything
   # other than "local" in its list, we need to have it show up in
   # running-config.
   for authType, methodList, defaultMethodList in \
      [("login", cfg.loginMethodList, cfg.defaultLoginMethodList),
       ("enable", cfg.enableMethodList, cfg.defaultEnableMethodList)]: 
      ml = defaultMethodList
      if len( ml.method ) != 1 or ml.method[ 0 ] != "local" or saveAll:
         cmds.addCommand( _genAuthnCmd( ml, authType ) )

      # Output the named method lists
      for ml in methodList.values():
         cmds.addCommand( _genAuthnCmd( ml, authType ) )
      
   # Output dot1x method lists
   ml = cfg.defaultDot1xMethodList
   if len( ml.method ):
      cmds.addCommand( _genAuthnCmd( ml, "dot1x" ) )
   

   # aaa authentication policy logging on success/failure
   if cfg.loggingOnSuccess:
      cmds.addCommand( 'aaa authentication policy on-success log' )
   elif saveAll:
      cmds.addCommand( 'no aaa authentication policy on-success log' )

   if cfg.loggingOnFailure:
      cmds.addCommand( 'aaa authentication policy on-failure log' )
   elif saveAll:
      cmds.addCommand( 'no aaa authentication policy on-failure log' )

   # aaa authentication policy lockout failure <> [ window <> ] duration <>
   if cfg.lockoutTime:
      lockoutCmd = 'aaa authentication policy lockout %s'
      if cfg.lockoutWindow != cfg.lockoutWindowDefault or saveAll:
         lockAttrs = 'failure %d window %d duration %d' % ( cfg.maxLoginAttempts,
                                                            cfg.lockoutWindow,
                                                            cfg.lockoutTime )
         cmds.addCommand( lockoutCmd % lockAttrs )
      else:
         lockAttrs = 'failure %d duration %d' % ( cfg.maxLoginAttempts,
                                                  cfg.lockoutTime )
         cmds.addCommand( lockoutCmd % lockAttrs )
   elif saveAll:
      cmds.addCommand( 'no aaa authentication policy lockout' )

   # aaa authorization console
   if cfg.consoleAuthz != cfg.consoleAuthzDefault:
      cmds.addCommand( "aaa authorization serial-console" )
   elif saveAll:
      cmds.addCommand( "no aaa authorization serial-console" )

   def _genDot1xDynAuthzCmd( ml ):
      m = ml.method
      groupList = " ".join( [ m[i] for i in range( 0, len( m ) ) ] )
      return "aaa authorization dynamic dot1x additional-groups %s" % groupList
   
   # Include Dot1x Dynamic Authorization only command
   ml = cfg.coaOnlyGroupList
   if len( ml.method ):
      cmds.addCommand( _genDot1xDynAuthzCmd( ml ) )

   def _genDot1MbaMultiGroupAuthCmd( ml ):
      m = ml.method
      groupList = None
      for level, grp in m.items(): # pylint: disable=unused-variable
         grp = grp.split()[ 1: ]
         if groupList:
            groupList = groupList + " " + grp[ 0 ]
         else:
            groupList = " ".join( grp )
      return "aaa authentication dot1x mba multi-group %s" % groupList
   
   # Include Dot1x MBA multi group authentication command
   ml = cfg.mbaMultiGroupList
   if len( ml.method ):
      cmds.addCommand( _genDot1MbaMultiGroupAuthCmd( ml ) )

   # pylint: disable-next=redefined-builtin,inconsistent-return-statements
   def _genAuthzCmd( ml, type, service ):
      if ( ( len( ml ) == 1 and ml[ 0 ] == "none" and service == 'default' ) or \
           ( len(ml) == 0 and service == 'console' ) ):
         if saveAll:
            return f"no aaa authorization {type} {service}"
         else:
            return
      s = " ".join( ml[ i ] for i in range( 0, len( ml ) ) )
      return f"aaa authorization {type} {service} {s}"

   def _getCmdMethodLists( methods, mlKeyFunc ):
      methodList = [] # pylint: disable=unused-variable
      cmdRe = re.compile( "command(\\d+)" )
      # first get level -> ML
      cmdMls = {}
      for name in methods:
         ml = methods[ name ]
         match = re.match( cmdRe, name )
         if match is not None:
            level = int( match.group( 1 ), 10 )
            if level >= 0 and level <= 15: # pylint: disable=chained-comparison
               cmdMls[ level ] = ml

      # then merge levels to the same ML
      mlLevelMap = {}   # methodList key -> ( level list, methodList )
      for level, ml in sorted( cmdMls.items() ):
         mlKey = mlKeyFunc( ml )
         mlLevelMap.setdefault( mlKey, ( [], ml ) )
         mlLevelMap[ mlKey ][ 0 ].append( level )

      # return merged levels + ML
      return sorted( mlLevelMap.values() )
   
   execAuthzMethod = cfg.authzMethod.get('exec')
   if execAuthzMethod:
      for ml, service in \
         [ ( execAuthzMethod.defaultMethod, "default" ), 
           ( execAuthzMethod.consoleMethod, "console" ) ]:
         if ml:
            cmd = _genAuthzCmd( ml, "exec", service )
            if cmd:
               cmds.addCommand( cmd )

   def _methodListStr( methodColl ):
      return ','.join( method for method in  methodColl.values() )

   def _authzMethodListKey( ml, service ):
      if service == 'console':
         return _methodListStr( ml.consoleMethod ) 
      else:
         return _methodListStr( ml.defaultMethod )

   for commonLevels, ml in \
      _getCmdMethodLists( cfg.authzMethod, lambda m:
                                             _authzMethodListKey( m, 'default' ) ):
      if len( commonLevels ) == 16:
         levels = "all"
      else:
         levels = MultiRangeRule.multiRangeToCanonicalString( commonLevels )

      cmd = _genAuthzCmd( ml.defaultMethod, "commands %s" % levels, "default" )
      if cmd:
         cmds.addCommand( cmd )

   if cfg.suppressConfigCommandAuthz:
      cmds.addCommand( "no aaa authorization config-commands" )
   elif saveAll:
      cmds.addCommand( "aaa authorization config-commands" )

   # aaa accounting
   def _actionToToken( action ):
      if action == 'startStop':
         return 'start-stop'
      elif action == 'stopOnly':
         return 'stop-only'
      else:
         assert False, 'invalid action!'

   def _genMethodsStr( methodMap, multicastMap ):
      """method names with optional 'multicast' keyword suffix"""
      maybeMulticast = lambda m: " multicast" if multicastMap.get( m ) else ""
      orderedMethods = ( methodMap[ k ] for k in methodMap )
      return " ".join( m + maybeMulticast( m ) for m in orderedMethods )

   # pylint: disable-next=redefined-builtin,inconsistent-return-statements
   def _genConsoleAcctCmd( ml, type ):
      if not ml.consoleUseOwnMethod:
         if saveAll:
            # generate default for console accounting not set.
            return 'no aaa accounting %s console' % type
         else:
            return
      if ml.consoleAction == 'none':
         return 'aaa accounting %s console none' % type
      methodsStr = _genMethodsStr( ml.consoleMethod, ml.consoleMethodMulticast )
      return 'aaa accounting {} console {} {}'.format(
         type, _actionToToken( ml.consoleAction ), methodsStr )

   # pylint: disable-next=redefined-builtin,inconsistent-return-statements
   def _genDefaultAcctCmd( ml, type ):
      if ml.defaultAction == 'none':
         if saveAll:
            return 'no aaa accounting %s default' % type
         else:
            return
      methodsStr = _genMethodsStr( ml.defaultMethod, ml.defaultMethodMulticast )
      return 'aaa accounting {} default {} {}'.format(
         type, _actionToToken( ml.defaultAction ), methodsStr )

   # pylint: disable-next=redefined-builtin,inconsistent-return-statements
   def _genAcctCmd( ml, type, service ):
      if service == 'console':
         if type != 'system' and type != 'dot1x': # pylint: disable=consider-using-in
            return _genConsoleAcctCmd( ml, type )
         else:
            return
      else:
         return _genDefaultAcctCmd( ml, type )
      
   def _acctMethodListKey( ml, service ):
      if service == 'console':
         return "{} {} {}".format( ml.consoleUseOwnMethod,
                                   ml.consoleAction,
                                   _methodListStr( ml.consoleMethod ) )
      else:
         return "{} {}".format( ml.defaultAction,
                                _methodListStr( ml.defaultMethod ) )

   for service in ( 'console', 'default' ):
      for acctType in ( 'exec', 'system', 'dot1x' ):
         ml = cfg.acctMethod.get( acctType )
         if ml:
            cmd = _genAcctCmd( ml, acctType, service )
            if cmd:
               cmds.addCommand( cmd )

      # pylint: disable=cell-var-from-loop
      for commonLevels, ml in _getCmdMethodLists( cfg.acctMethod,
                                                  lambda m:
                                                  _acctMethodListKey( m, service ) ):
         if len( commonLevels ) == 16:
            levels = "all"
         else:
            levels = MultiRangeRule.multiRangeToCanonicalString( commonLevels )

         cmd = _genAcctCmd( ml, "commands %s" % levels, service )
         if cmd:
            cmds.addCommand( cmd )

   if cfg.acctCmdSecureMonitorEnabled != cfg.acctCmdSecureMonitorEnabledDefault:
      cmds.addCommand( "aaa accounting commands all secure-monitor none" )
   elif saveAll:
      cmds.addCommand( "no aaa accounting commands all secure-monitor" )

@CliSave.saver( 'LocalUser::Config', 'security/aaa/local/config',
      requireMounts = ( 'mgmt/ssh/config', ) )
def saveLocalUserConfig( entity, root, requireMounts, options ):
   cmds = root[ 'Aaa.encrypt' ]
   saveAll = options.saveAll
   sshConfig = requireMounts[ 'mgmt/ssh/config' ]

   if entity.encryptedEnablePasswd != entity.encryptedEnablePasswdDefault:
      cmd = SecretCli.getCliSaveCommand( "enable password {}",
                                         entity.encryptedEnablePasswd )
      cmds.addCommand( cmd )
   elif saveAll:
      # there is no enable password by default
      cmds.addCommand( 'no enable password' )

   # AaaSetPassword finds one of the following lines in startup-config
   # and sets root password accordingly. To avoid confusion with the content
   # of other commands (e.g., banner), we make sure that those aaa commands
   # are at the beginning of the running-config and always present (therefore
   # we generate 'no aaa root' even in the default case).
   aaaRootCommands = []
   if entity.encryptedRootPasswd != entity.encryptedRootPasswdDefault:
      if entity.encryptedRootPasswd == '':
         aaaRootCommands.append( 'aaa root nopassword' )
      else:
         cmd = SecretCli.getCliSaveCommand( "aaa root secret {}",
                                            entity.encryptedRootPasswd )
         aaaRootCommands.append( cmd )

   # Check if we need to render config in the old CLI
   if not sshConfig.useNewSshCliKey and 'root' in sshConfig.user:
      rootKeys = sshConfig.user[ 'root' ].sshAuthKeys

      for name, key in rootKeys.items(): # pylint: disable=unused-variable
         keyContents = rootKeys[ name ].keyContents
         if name == primarySshKeyId:
            cmd = 'aaa root ssh-key ' + keyContents
         else:
            cmd = 'aaa root ssh-key secondary ' + keyContents
         aaaRootCommands.append( cmd )

   if not aaaRootCommands and options.saveCleanConfig:
      aaaRootCommands.append( 'no aaa root' )

   for cmd in aaaRootCommands:
      cmds.addCommand( cmd )

   if entity.allowRemoteLoginWithEmptyPassword:
      cmds.addCommand(
         'aaa authentication policy local allow-nopassword-remote-login' )
   elif saveAll:
      # entity.allowRemoteLoginWithEmptyPassword is False by default
      cmds.addCommand(
         'no aaa authentication policy local allow-nopassword-remote-login' )

   if 'admin' not in entity.acct:
      cmds.addCommand( "no username admin" )

   # Check if we need to render SSH principal config in the old CLI
   if not sshConfig.useNewSshCliPrincipal:
      if 'root' in sshConfig.user:
         principals = sorted( sshConfig.user[ 'root' ].sshAuthPrincipals )

         # Check if we have any SSH principals configured
         if principals := ' '.join( principals ):
            cmds.addCommand( f'username root ssh principal {principals}' )
         elif saveAll:
            cmds.addCommand( "no username root ssh principal" )
      elif saveAll:
         cmds.addCommand( "no username root ssh principal" )

   # output username configuration in alphabetical order
   for user in entity.acct.values():
      saveUserAccount( user, sshConfig, root, options )

   # aaa authorization policy local default-role <role-name>
   c = "aaa authorization policy local default-role"
   if entity.defaultRole != entity.defaultRoleDefault:
      cmds.addCommand( c + " " + entity.defaultRole )
   elif saveAll:
      cmds.addCommand( "no " + c )

   # Roles
   for name, role in entity.role.items():
      if not saveAll and name in authzBuiltinRoles:
         continue
      role = entity.role[ name ]
      mode = root[ RoleConfigMode ].getOrCreateModeInstance( name )
      cmds = mode[ 'Aaa.role' ]
      for seq, rule in role.rule.items():
         modeKey = ' mode %s' % rule.modeKey if rule.modeKey != '' else ''
         cmd = '%d %s%s command %s' % ( seq, rule.action, modeKey, rule.regex )
         cmds.addCommand( cmd )

# These definitions an go away once we have constAttr access in Python
defaultPrivLevel = 1

def saveUserAccount( entity, sshConfig, root, options ):
   cmds = root[ 'Aaa.users' ]
   saveAll = options.saveAll

   # The "admin" account is special: it always exists, even if it has
   # not been configured. If it has not explicitly been configured, it
   # has a blank password, and does not show up in the running-config
   # or startup-config.
   if( entity.name == "admin" and entity.role == 'network-admin' and
         not entity.encryptedPasswd and entity.privilegeLevel == defaultPrivLevel
         and ( "admin" not in sshConfig.user or
            len( sshConfig.user[ "admin" ].sshAuthKeys ) == 0 )
         and not saveAll ):
      return
   cmd = "username %s" % entity.name
   if entity.privilegeLevel != entity.privilegeLevelDefault or saveAll:
      cmd += " privilege %d" % entity.privilegeLevel
   if entity.role != entity.roleDefault:
      cmd += " role " + entity.role
   if entity.shell:
      cmd += " shell " + entity.shell
   if entity.encryptedPasswd == '':
      cmd += " nopassword"
   else:
      # cmd could contain curly braces, so escape those
      formatStr = CliSave.escapeFormatString( cmd ) + " secret {}"
      cmd = SecretCli.getCliSaveCommand( formatStr,
                                         entity.encryptedPasswd )
   cmds.addCommand( cmd )

   # Check if we need to render SSH key config in the old CLI
   if not sshConfig.useNewSshCliKey and entity.name in sshConfig.user:
      userSshKeys = sshConfig.user[ entity.name ].sshAuthKeys
      for name, key in userSshKeys.items():
         if name == primarySshKeyId:
            cmds.addCommand( f"username {entity.name} ssh-key {key.keyContents}" )
         else:
            cmds.addCommand(
                  f"username {entity.name} ssh-key secondary {key.keyContents}" )

   # Check if we need to render SSH principal config in the old CLI
   if not sshConfig.useNewSshCliPrincipal and entity.name in sshConfig.user:
      principals = sorted( sshConfig.user[ entity.name ].sshAuthPrincipals )
   
      if principals := ' '.join( principals ):
         cmds.addCommand( f'username {entity.name} ssh principal {principals}' )
