# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.ArBgp import BgpSessionTrackerMode
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.RoutingBgpCliSave import RouterBgpBaseConfigMode
from RouteMapLib import isAsdotConfigured
import Tac
from Toggles import (
   ArBgpToggleLib,
   RouterGeneralCliToggleLib
)
import Tracing

th = Tracing.Handle( 'ArBgpCliSave' )
t0 = th.trace0
EthIntfId = Tac.Type( 'Arnet::EthIntfId' )
PortChannelIntfId = Tac.Type( 'Arnet::PortChannelIntfId' )
LoopbackIntfId = Tac.Type( 'Arnet::LoopbackIntfId' )

class SessionTrackerConfigSaveMode( BgpSessionTrackerMode, CliSave.Mode ):
   def __init__( self, params ):
      BgpSessionTrackerMode.__init__( self, params )
      CliSave.Mode.__init__( self, params )

RouterBgpBaseConfigMode.addChildMode( SessionTrackerConfigSaveMode )
SessionTrackerConfigSaveMode.addCommandSequence( 'Bgp.tracker.config' )

def saveSessionTrackerConfig( tracker, trackerConfig, saveAll=False ):
   cmds = []
   if ArBgpToggleLib.toggleSessionTrackerUpAllModeEnabled():
      if trackerConfig.trackAllPeersUp:
         cmds.append( 'neighbor all' )
      elif saveAll:
         cmds.append( 'no neighbor all' )
   if trackerConfig.recoveryDelaySecPresent:
      # pylint: disable-next=consider-using-f-string
      cmds.append( "recovery delay {} seconds".format(
                      trackerConfig.recoveryDelaySec ) )
   elif saveAll:
      cmds.append( 'no recovery delay' )

   return cmds

@CliSave.saver( 'Routing::Bgp::SessionTrackerCliConfigDir',
                'routing/arbgp/sessionTracker/inst',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config' ) )
def saveBgpSessionTrackerConfig( sessionTrackerConfigDir, root, requireMounts,
                                 options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   if bgpConfig.asNumber == 0:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )

   for tracker in sorted( sessionTrackerConfigDir.sessionTrackerConfig ):
      trackerConfig = sessionTrackerConfigDir.sessionTrackerConfig[ tracker ]
      trackerMode = bgpMode[ SessionTrackerConfigSaveMode ].getOrCreateModeInstance(
         tracker )
      trackerModeCmds = trackerMode[ 'Bgp.tracker.config' ]
      cmds = saveSessionTrackerConfig( tracker, trackerConfig, options.saveAll )
      for cmd in cmds:
         trackerModeCmds.addCommand( cmd )

IntfConfigMode.addCommandSequence( 'Bgp.tracker.ethIntf' )

def allSessionTrackerIntfNames( sessionTrackerIntfConfigDir, includeEligible=False,
                                requireMounts=None ):
   # if includeEligible, then all Ethernet, Port-Channel and Loopback
   # interfaces are included.  Otherwise, only interface with this feature
   # configured are included.
   assert requireMounts
   intfConfigDir = requireMounts[ 'interface/config/all' ]
   intfNames = []
   if includeEligible:
      for intfName in intfConfigDir.intfConfig:
         if ( PortChannelIntfId.isPortChannelIntfId( intfName ) or
              EthIntfId.isEthIntfId( intfName ) ):
            intfNames.append( intfName )
         if ( ArBgpToggleLib.toggleSessionTrackerOnLoopbackEnabled() and
              LoopbackIntfId.isLoopbackIntfId( intfName ) ):
            intfNames.append( intfName )
   else:
      intfNames = list( sessionTrackerIntfConfigDir.intfConfig )
   return intfNames

@CliSave.saver( 'Routing::Bgp::SessionTrackerIntfCliConfigDir',
                'routing/arbgp/sessionTracker/intf',
                requireMounts=( 'interface/config/all', ) )
def saveBgpSessionTrackerIntfConfig( sessionTrackerIntfConfigDir, root,
                                     requireMounts, options ):
   t0( "Saving intf config for session tracker" )
   configuredIntfNames = []
   saveAllDetail = options.saveAllDetail
   configuredIntfNames = \
      allSessionTrackerIntfNames( sessionTrackerIntfConfigDir,
                                  includeEligible=saveAllDetail,
                                  requireMounts=requireMounts )

   for intf in configuredIntfNames:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      ethIntfCmds = mode[ 'Bgp.tracker.ethIntf' ]
      intfConfig = sessionTrackerIntfConfigDir.intfConfig.get( intf, None )
      if not intfConfig:
         if saveAllDetail:
            ethIntfCmds.addCommand( "no bgp session tracker" )
         else:
            continue
      else:
         trackerName = intfConfig.sessionTracker
         ethIntfCmds.addCommand( f"bgp session tracker {trackerName}" )

CliSave.GlobalConfigMode.addCommandSequence( 'Agent.bgp.threadconfig' )

@CliSave.saver( 'Routing::Bgp::ThreadConfig', 'agent/bgp/threadconfig' )
def saveBgpAgentThreadConfig( entity, root, requireMounts, options ):
   cmds = root[ 'Agent.bgp.threadconfig' ]
   val = entity.singleThreaded
   if val:
      cmds.addCommand( 'agent bgp threads single' )
   elif entity.numBestpathThreads != entity.numBestpathThreadsDefault or \
        entity.numInputThreads != entity.numInputThreadsDefault:
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'agent bgp threads bestpath {} input {}'.format(
            entity.numBestpathThreads, entity.numInputThreads ) )
   elif options.saveAll:
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'agent bgp threads bestpath {} input {}'.format(
         entity.numBestpathThreadsDefault, entity.numInputThreadsDefault ) )

@CliSave.saver( 'Routing::General::Config', 'routing/general/config/global',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config' ) )
def saveNhResRouteIgpNhCostConfig( routerGeneralConfig, root, requireMounts,
                                   options ):
   if not RouterGeneralCliToggleLib.toggleNhResolutionRouteIgpNhCostEnabled():
      return

   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured, ) )

   cmds = []
   if routerGeneralConfig.nhResRouteIgpNhCostBgp:
      cmds.append( 'next-hop resolution route igp-nexthop-cost protocol bgp' )
   elif options.saveAll:
      cmds.append( 'no next-hop resolution route' )

   for cmd in cmds:
      bgpMode[ 'Bgp.config' ].addCommand( cmd )
