# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
from TypeFuture import TacLazyType
from CliMode.Avt import (
      RouterAdaptiveVirtualTopologyModeBase,
      RouterAvtPolicyConfigModeBase,
      RouterAvtPolicyAppProfileConfigModeBase,
      RouterAvtProfileConfigModeBase,
      RouterAvtVrfConfigModeBase,
      DEFAULT_OUTLIER_THRESH_JITTER,
      DEFAULT_OUTLIER_THRESH_LATENCY,
      DEFAULT_OUTLIER_THRESH_LOSSRATE,
      DEFAULT_OUTLIER_THRESH_LOAD
      )
from CliMode.Dps import ( JITTER_SCALE,
                          LATENCY_SCALE,
                          LOSS_RATE_SCALE,
                          LOSS_RATE_ADJUSTMENT,
                          LOAD_SCALE )
from Toggles.WanTECommonToggleLib import ( toggleAvtPathOutlierEliminationEnabled,
                                           toggleAvtLowestLoadMetricEnabled )

AvtRole = TacLazyType( 'Avt::AvtRole' )
TrafficClass = TacLazyType( "WanTE::TrafficClass" )
Dscp = TacLazyType( "WanTE::Dscp" )
PolicyConfig = TacLazyType( 'SfeInternetExit::PolicyConfig' )
MetricOrder = TacLazyType( "Avt::MetricOrder" )

class RouterAdaptiveVirtualTopologySaveMode( RouterAdaptiveVirtualTopologyModeBase,
      CliSave.Mode ):
   def __init__( self, param ):
      RouterAdaptiveVirtualTopologyModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

class RouterAvtProfileConfigSaveMode( RouterAvtProfileConfigModeBase,
      CliSave.Mode ):
   def __init__( self, param ):
      RouterAvtProfileConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterAvtVrfConfigSaveMode( RouterAvtVrfConfigModeBase,
      CliSave.Mode ):
   def __init__( self, param ):
      RouterAvtVrfConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterAvtPolicyConfigSaveMode( RouterAvtPolicyConfigModeBase,
      CliSave.Mode ):
   def __init__( self, param ):
      RouterAvtPolicyConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RouterAvtPolicyAppProfileConfigSaveMode(
      RouterAvtPolicyAppProfileConfigModeBase,
      CliSave.Mode ):
   def __init__( self, param ):
      RouterAvtPolicyAppProfileConfigModeBase.__init__( self, param[ 0 ],
                                                        param[ 1 ], param[ 2 ] )
      CliSave.Mode.__init__( self, param )

def registerAllCommands():

   CliSave.GlobalConfigMode.addChildMode( RouterAdaptiveVirtualTopologySaveMode )
   RouterAdaptiveVirtualTopologySaveMode.addCommandSequence( 'Avt.RouterAvt' )

   RouterAdaptiveVirtualTopologySaveMode\
         .addChildMode( RouterAvtProfileConfigSaveMode )
   RouterAvtProfileConfigSaveMode.addCommandSequence( 'Avt.AvtProfile' )

   RouterAdaptiveVirtualTopologySaveMode \
         .addChildMode( RouterAvtVrfConfigSaveMode )
   RouterAvtVrfConfigSaveMode.addCommandSequence( 'Avt.VrfConfig' )

   RouterAdaptiveVirtualTopologySaveMode \
         .addChildMode( RouterAvtPolicyConfigSaveMode )
   RouterAvtPolicyConfigSaveMode.addCommandSequence( 'Avt.PolicyConfig' )

   RouterAvtPolicyConfigSaveMode \
         .addChildMode( RouterAvtPolicyAppProfileConfigSaveMode )
   RouterAvtPolicyAppProfileConfigSaveMode \
         .addCommandSequence( 'Avt.PolicyConfig.AppProfile' )

registerAllCommands()

@CliSave.saver( 'Avt::AvtCliConfig', 'avt/input/cli' )
def saveConfig( entity, root, sysdbRoot, options ):

   if not entity.avtConfigured:
      return

   saveAll = options.saveAll
   avtMode = root[ RouterAdaptiveVirtualTopologySaveMode ] \
      .getSingletonInstance()

   avtCmds = avtMode[ 'Avt.RouterAvt' ]
   if entity.role != AvtRole.unknown:
      strMapper = {
            AvtRole.transit: 'transit zone',
            AvtRole.edge: 'edge',
            AvtRole.pathfinder: 'pathfinder',
            AvtRole.regionBR: 'transit region',
            }
      cmd = 'topology role %s' % strMapper[ entity.role ]
      if entity.vxlanGateway:
         cmd += ' gateway vxlan'
      avtCmds.addCommand( cmd )
   elif saveAll:
      avtCmds.addCommand( 'no topology role' )

   if entity.region != '':
      avtCmds.addCommand( 'region %s id %d' % ( entity.region, entity.regionId ) )

   if entity.zone != '':
      avtCmds.addCommand( 'zone %s id %d' % ( entity.zone, entity.zoneId ) )

   if entity.site != '':
      avtCmds.addCommand( 'site %s id %d' % ( entity.site, entity.siteId ) )

   for vrfName in sorted( entity.vrfConfig.keys() ):
      vrfConfigMode = avtMode[ RouterAvtVrfConfigSaveMode ]\
            .getOrCreateModeInstance( vrfName )
      vrfConfigCmds = vrfConfigMode[ 'Avt.VrfConfig' ]
      vrfConfig = entity.vrfConfig[ vrfName ]

      if vrfConfig.policyName != '':
         vrfConfigCmds.addCommand( 'avt policy %s' % vrfConfig.policyName )

      for avtName in sorted( vrfConfig.avtNameToId.keys() ):
         vrfConfigCmds.addCommand( 'avt profile %s id %d'
               % ( avtName, vrfConfig.avtNameToId[ avtName ] ) )

   for policyName in sorted( entity.policyConfig.keys() ):
      policyConfigMode = avtMode[ RouterAvtPolicyConfigSaveMode ]\
            .getOrCreateModeInstance( policyName )
      policyConfig = entity.policyConfig[ policyName ]

      for ruleKey in policyConfig.appProfilePolicyRuleList.keys():
         ruleConfig = policyConfig.appProfilePolicyRuleList[ ruleKey ]
         appProfileConfigMode = \
            policyConfigMode[ RouterAvtPolicyAppProfileConfigSaveMode ]\
               .getOrCreateModeInstance(
                     ( policyName, ruleKey, ruleConfig.appProfileName ) )
         appProfileConfigCmds = appProfileConfigMode[ 'Avt.PolicyConfig.AppProfile' ]

         if ruleConfig.actionName != '':
            appProfileConfigCmds.addCommand(
                  'avt profile %s' % ruleConfig.actionName )

         if ruleConfig.trafficClass != TrafficClass.null:
            appProfileConfigCmds.addCommand(
                  'traffic-class %s' % ruleConfig.trafficClass )

         if ruleConfig.dscp != Dscp.null:
            appProfileConfigCmds.addCommand(
                  'dscp %s' % ruleConfig.dscp )

   for avtName in sorted( entity.avtProfile.keys() ):
      profileMode = avtMode[ RouterAvtProfileConfigSaveMode ]\
            .getOrCreateModeInstance( avtName )
      profileCmds = profileMode[ 'Avt.AvtProfile' ]
      avtProfile = entity.avtProfile[ avtName ]

      # When avtProfile is created, CliSavPlugin shows default
      # internetExitPolicy as system-default-internet-exit-policy
      if avtProfile.internetExitPolicyName != PolicyConfig.defaultName:
         profileCmds.addCommand( 'internet-exit policy %s'
               % avtProfile.internetExitPolicyName )
      else:
         # We are showing only when options.saveAll has set
         if saveAll:
            profileCmds.addCommand(
                 'internet-exit policy %s' % avtProfile.internetExitPolicyName )

      if avtProfile.loadBalanceGroupName != '':
         profileCmds.addCommand( 'path-selection load-balance %s'
               % avtProfile.loadBalanceGroupName )

      if toggleAvtPathOutlierEliminationEnabled():
         if avtProfile.metricOrder != MetricOrder.lossrate:
            profileCmds.addCommand( 'metric order %s'
                                    % avtProfile.metricOrder )
         elif saveAll:
            profileCmds.addCommand( 'metric order loss-rate' )

         if avtProfile.outlierDisabled:
            profileCmds.addCommand( 'path-selection outlier elimination disabled' )
         elif saveAll:
            profileCmds.addCommand( 'path-selection outlier elimination' )

         if avtProfile.outlierThreshold.latency != DEFAULT_OUTLIER_THRESH_LATENCY \
            or saveAll:
            profileCmds.addCommand( 'path-selection outlier elimination threshold'
                                    ' latency %d milliseconds' %
                    ( int( avtProfile.outlierThreshold.latency / LATENCY_SCALE ) ) )

         if avtProfile.outlierThreshold.jitter != DEFAULT_OUTLIER_THRESH_JITTER \
            or saveAll:
            profileCmds.addCommand( 'path-selection outlier elimination threshold'
                                    ' jitter %d milliseconds' %
                    ( int( avtProfile.outlierThreshold.jitter / JITTER_SCALE ) ) )

         if avtProfile.outlierThreshold.loss != DEFAULT_OUTLIER_THRESH_LOSSRATE \
            or saveAll:
            loss = avtProfile.outlierThreshold.loss
            loss -= LOSS_RATE_ADJUSTMENT
            profileCmds.addCommand( 'path-selection outlier elimination threshold'
                                    ' loss-rate %.2f percent' %
                                    ( float( loss ) / LOSS_RATE_SCALE ) )

         if toggleAvtLowestLoadMetricEnabled():
            if avtProfile.outlierThreshold.load != DEFAULT_OUTLIER_THRESH_LOAD \
               or saveAll:
               load = avtProfile.outlierThreshold.load
               profileCmds.addCommand( 'path-selection outlier elimination threshold'
                                       ' load %f percent' %
                                       ( float( load ) / LOAD_SCALE ) )
