# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliMode.Bfd import ( RoutingBfdMode, RoutingBfdPeerMode,
                          RoutingBfdVrfMode, RoutingBfdVrfPeerMode,
                          RoutingBfdSbfdMode )
import Tac
from IpLibConsts import DEFAULT_VRF
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.IraCliSave import ipIntfCmdSeq
from CliSavePlugin.IraVrfCliSave import VrfDefinitionCliSaveMode
from RoutingIntfUtils import allIpIntfNames

from BfdLib import ( initiatorConfigDefault, reflectorMinRxDefault )

from Toggles.BfdToggleLib import (
   toggleBfdEchoL3SrcAddrEnabled,
   toggleBfdDscpConfigEnabled )
from TypeFuture import TacLazyType

SessionType = Tac.Type( "Bfd::SessionType" )
authType = Tac.Type( 'Bfd::BfdAuthType' )
BfdRole = Tac.Type( 'Bfd::BfdRole' )
BfdConstants = TacLazyType( 'Bfd::Constants' )

IntfConfigMode.addCommandSequence( 'Bfd.intf', after=[ ipIntfCmdSeq ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Bfd', before=[ IntfConfigMode ],
                                             after=[ VrfDefinitionCliSaveMode ] )

class RouterBfdConfigMode( RoutingBfdMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingBfdMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterBfdConfigMode,
                                       after=[ IntfConfigMode ] )
RouterBfdConfigMode.addCommandSequence( 'Bfd.config' )

class RouterBfdPeerConfigMode( RoutingBfdPeerMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingBfdPeerMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterBfdConfigMode.addChildMode( RouterBfdPeerConfigMode )
RouterBfdPeerConfigMode.addCommandSequence( 'Bfd.peer.config' )

class RouterBfdVrfConfigMode( RoutingBfdVrfMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingBfdVrfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterBfdConfigMode.addChildMode( RouterBfdVrfConfigMode )
RouterBfdVrfConfigMode.addCommandSequence( 'Bfd.vrf.config' )

class RouterBfdVrfPeerConfigMode( RoutingBfdVrfPeerMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingBfdVrfPeerMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterBfdVrfConfigMode.addChildMode( RouterBfdVrfPeerConfigMode )
RouterBfdVrfPeerConfigMode.addCommandSequence( 'Bfd.vrf.peer.config' )

class RouterBfdSbfdMode( RoutingBfdSbfdMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingBfdSbfdMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterBfdConfigMode.addChildMode( RouterBfdSbfdMode )
RouterBfdSbfdMode.addCommandSequence( 'Bfd.sbfd.config' )

def IntfId( intfName=None ):
   if intfName is not None:
      return Tac.ValueConst( 'Arnet::IntfId', intfName )
   return None

def bfdIntervalDefault():
   return Tac.Type( 'Bfd::BfdInterval' ).defval

def bfdMultiplierDefault():
   return Tac.Type( 'Bfd::BfdMultiplier' ).defval

def bfdSessionStatsIntervalDefault():
   return Tac.Type( 'Bfd::SessionStatsInterval' ).defval

def bfdSessionStatsIntervalDangerThreshold():
   return Tac.Type( 'Bfd::SessionStatsInterval' ).dangerThreshold

def getRootAndSaveCommand( cmdStr, root, legacyConfig=False ):
   if legacyConfig:
      saveRoot = root[ 'Bfd' ]
   else:
      mode = root[ RouterBfdConfigMode ].getSingletonInstance()
      saveRoot = mode[ 'Bfd.config' ]

   saveRoot.addCommand( cmdStr )

@CliSave.saver( 'Bfd::VxlanTunnelAppConfig', 'bfd/config/vxlanTunnelApp/test/',
                requireMounts=( 'bfd/config/global', ) )
def saveBfdVxlanTestConfig( bfdAppConfig, root, requireMounts, options ):
   legacyConfig = requireMounts[ 'bfd/config/global' ].legacyConfig
   for peer in bfdAppConfig.peerConfigVer:
      if bfdAppConfig.peerConfigVer[ peer ].version == 1:
         if peer in bfdAppConfig.vxlanTunnelConfig:
            vxlanTunnelConfig = bfdAppConfig.vxlanTunnelConfig[ peer ]
            getRootAndSaveCommand(
                  '%svxlan session %s %s %s %s %s %s %s %d %d %d' %
                     ( ( 'bfd ' if legacyConfig else 'test ' ),
                      peer.ip,
                      vxlanTunnelConfig.vxlanTunnel.outerSrcIp,
                      vxlanTunnelConfig.vxlanTunnel.outerDstPort,
                      vxlanTunnelConfig.vxlanTunnel.outerSrcPort,
                      vxlanTunnelConfig.vxlanTunnel.vni.vni,
                      vxlanTunnelConfig.vxlanTunnel.innerDstMac,
                      vxlanTunnelConfig.vxlanTunnel.innerSrcMac,
                      vxlanTunnelConfig.intervalParams.minTx,
                      vxlanTunnelConfig.intervalParams.minRx,
                      vxlanTunnelConfig.intervalParams.mult
                       ),
                  root, legacyConfig=legacyConfig )

@CliSave.saver( 'Bfd::AppConfig', 'bfd/config/app/test',
                requireMounts=( 'bfd/config/global', ) )
def saveBfdTestConfig( bfdAppConfig, root, requireMounts, options ):
   legacyConfig = requireMounts[ 'bfd/config/global' ].legacyConfig
   for peer in bfdAppConfig.peerConfigVer:
      if bfdAppConfig.peerConfigVer[ peer ].version == 1:
         if peer.type == SessionType.multihop:
            getRootAndSaveCommand( '%ssession multihop %s %s %s %s' %
                                      ( 'bfd ' if legacyConfig else 'test ',
                                      peer.ip,
                                      peer.srcIp,
                                      peer.vrf,
                                      peer.intf ),
                                   root, legacyConfig=legacyConfig )
         elif peer.type == SessionType.sbfdInitiator and not legacyConfig:
            cfgStr = ''
            cfg = bfdAppConfig.peerToInitiatorConfigMap.get( peer, None )
            if cfg:
               if cfg.timer:
                  cfgStr += ' interval %u multiplier %u' % ( cfg.timer.minTx,
                                                             cfg.timer.mult )
               if cfg.sbfdEchoRttEnabled:
                  cfgStr += ' rtt-echo-enabled'
            mode = root[ RouterBfdConfigMode ].getSingletonInstance()
            mode = mode[ RouterBfdSbfdMode ].getSingletonInstance()
            cmds = mode[ 'Bfd.sbfd.config' ]
            cmds.addCommand( 'test session %s %lu%s' %
                             ( peer.ip, peer.tunnelId, cfgStr ) )
         else:
            cmd = '{}session {} {} {}'.format(
               'bfd ' if legacyConfig else 'test ', peer.ip, peer.intf, peer.vrf )
            maybeCfg = bfdAppConfig.peerConfigVer[ peer ].maybeIntervalConfig
            if maybeCfg:
               cmd = '{} interval {} min-rx {} multiplier {}'.format(
                     cmd, maybeCfg.minTx, maybeCfg.minRx, maybeCfg.mult )
            getRootAndSaveCommand( cmd, root, legacyConfig=legacyConfig )

def _saveBfdPeerConfig( peerCliConfig, root, options ):
   for peer, peerConfig in peerCliConfig.items():
      if peer.vrf != DEFAULT_VRF:
         vrfMode = root[ RouterBfdVrfConfigMode ].getOrCreateModeInstance( peer.vrf )
         vpcm = vrfMode[ RouterBfdVrfPeerConfigMode ]
         # A peer ip of 0.0.0.0 signifies a placeholder entry, for the vrf parent
         # mode.  If this is the case, then we want to skip that peer.
         if peer.ip.stringValue != '0.0.0.0':
            peerMode = vpcm.getOrCreateModeInstance( ( peer.vrf, peer.ip ) )
            cmds = peerMode[ 'Bfd.vrf.peer.config' ]
         else:
            continue
      else:
         mode = root[ RouterBfdPeerConfigMode ].getOrCreateModeInstance( peer.ip )
         cmds = mode[ 'Bfd.peer.config' ]

      authMode, meticulous = getBfdAuthModeFromConfig( peerConfig.authType,
                                                       intfOrPeerConfig=True )
      if authMode:
         secretProfile = ''
         if authMode != 'disabled':
            secretProfile = 'shared-secret profile %s' % peerConfig.secretProfileName
         cmds.addCommand( 'authentication mode %s %s %s' %
                          ( authMode,
                            secretProfile,
                            ( 'meticulous' if meticulous else '' ) ) )
      elif options.saveAll:
         cmds.addCommand( 'no authentication' )

def _saveBfdSbfdConfig( bfdConfig, root, options ):
   mode = root[ RouterBfdSbfdMode ].getSingletonInstance()
   cmds = mode[ 'Bfd.sbfd.config' ]

   if bfdConfig.sbfdLocalIntf == bfdConfig.sbfdLocalIntfIp6:
      if bfdConfig.sbfdLocalIntf != '':
         cmds.addCommand( 'local-interface %s ipv4 ipv6' %
                          bfdConfig.sbfdLocalIntf )
      elif options.saveAll:
         cmds.addCommand( 'no local-interface' )
   else:
      if bfdConfig.sbfdLocalIntf != '':
         cmds.addCommand( 'local-interface %s ipv4' % bfdConfig.sbfdLocalIntf )
      if bfdConfig.sbfdLocalIntfIp6 != '':
         cmds.addCommand( 'local-interface %s ipv6' % bfdConfig.sbfdLocalIntfIp6 )

   if ( options.saveAll or
        bfdConfig.sbfdInitiatorConfig != initiatorConfigDefault() ):
      cmds.addCommand( 'initiator interval %d multiplier %d' %
                       ( bfdConfig.sbfdInitiatorConfig.minTx,
                         bfdConfig.sbfdInitiatorConfig.mult ) )

   if bfdConfig.sbfdRttEnabled:
      cmds.addCommand( 'initiator measurement delay round-trip' )
   elif options.saveAll:
      cmds.addCommand( 'no initiator measurement delay round-trip' )

   if ( options.saveAll or
        bfdConfig.sbfdReflectorMinRx != reflectorMinRxDefault() ):
      cmds.addCommand( 'reflector min-rx %d' % bfdConfig.sbfdReflectorMinRx )

   proxy = ' proxy' if bfdConfig.sbfdProxyEnabled else ''
   if ( bfdConfig.sbfdReflectorLocalDisc != 0 and
        bfdConfig.sbfdDiscIsU32 ):
      cmds.addCommand( 'reflector local-discriminator %u%s' % (
                       bfdConfig.sbfdReflectorLocalDisc, proxy ) )
   elif ( bfdConfig.sbfdReflectorLocalDisc != 0 and
          not bfdConfig.sbfdDiscIsU32 ):
      Disc = Tac.Value( 'Arnet::IpAddr',
                        bfdConfig.sbfdReflectorLocalDisc ).stringValue
      cmds.addCommand( f'reflector local-discriminator {Disc}{proxy}' )
   elif options.saveAll:
      cmds.addCommand( 'no reflector local-discriminator' )

def getBfdAuthModeFromConfig( bfdAuthType, intfOrPeerConfig=False ):
   mode = None
   # pylint: disable-next=consider-using-in
   meticulous = ( bfdAuthType == authType.authMeticulousMd5 or
                  bfdAuthType == authType.authMeticulousSha1 )

   if bfdAuthType == authType.authPassword:
      mode = 'simple'
   elif ( bfdAuthType == authType.authKeyedMd5 or # pylint: disable=consider-using-in
          bfdAuthType == authType.authMeticulousMd5 ):
      mode = 'md5'
   # pylint: disable-next=consider-using-in
   elif ( bfdAuthType == authType.authKeyedSha1 or
          bfdAuthType == authType.authMeticulousSha1 ):
      mode = 'sha1'
   # In intf or peer mode, authNone corresponds with the 'disabled' keyword
   elif intfOrPeerConfig and bfdAuthType == authType.authNone:
      mode = 'disabled'

   # Modes authInherit and authDebuSeqNum do not have corresponding keywords

   return mode, meticulous

@CliSave.saver( 'Bfd::ConfigGlobal', 'bfd/config/global',
                requireMounts=( 'bfd/config/peerCli', ) )
def saveBfdGlobalConfig( bfdConfig, root, requireMounts, options ):
   # Global config
   globalCfg = bfdConfig.globalCfg
   legacyConfig = bfdConfig.legacyConfig
   if ( options.saveAll or globalCfg ):
      getRootAndSaveCommand( "%sinterval %d %s %d multiplier %d default" %
                                ( 'bfd ' if legacyConfig else '',
                                  globalCfg.minTx,
                                  'min_rx' if legacyConfig else 'min-rx',
                                  globalCfg.minRx,
                                  globalCfg.mult ),
                             root, legacyConfig=legacyConfig )

   multihopGlobalCfg = bfdConfig.multihopGlobalCfg
   if multihopGlobalCfg is not None:
      getRootAndSaveCommand( "%smultihop interval %d %s %d multiplier %d" %
                                ( 'bfd ' if legacyConfig else '',
                                  multihopGlobalCfg.minTx,
                                  'min_rx' if legacyConfig else 'min-rx',
                                  multihopGlobalCfg.minRx,
                                  multihopGlobalCfg.mult ),
                             root, legacyConfig=legacyConfig )
   elif options.saveAll:
      getRootAndSaveCommand( "no %smultihop interval" %
                             ( 'bfd ' if legacyConfig else '' ),
                             root, legacyConfig=legacyConfig )

   # Bfd passive
   if bfdConfig.role == BfdRole.passive:
      getRootAndSaveCommand(
         'role passive', root, legacyConfig=bfdConfig.legacyConfig )
   elif options.saveAll and not bfdConfig.legacyConfig:
      getRootAndSaveCommand( 'no role passive', root, legacyConfig=False )

   # bfd qos dscp config
   if toggleBfdDscpConfigEnabled():
      if bfdConfig.dscpValue != BfdConstants.defaultIpDscp:
         # The value saved in "show config" is the 6 most significant
         # bits of the Sysdb DSCP value
         dscp = Tac.enumValue( 'Arnet::IpDscp', bfdConfig.dscpValue ) >> 2
         getRootAndSaveCommand( 'qos dscp %s' % dscp,
                                 root, legacyConfig=False )
      elif options.saveAll and not bfdConfig.legacyConfig:
         getRootAndSaveCommand( 'no qos dscp', root, legacyConfig=False )

   # bfd echo source L3 address config
   if toggleBfdEchoL3SrcAddrEnabled():
      if bfdConfig.echoSrcIpIntf != '':
         getRootAndSaveCommand( 'session echo local-interface %s' %
                                 bfdConfig.echoSrcIpIntf, root, legacyConfig=False )
      elif options.saveAll and not bfdConfig.legacyConfig:
         getRootAndSaveCommand( 'no session echo local-interface',
                                 root, legacyConfig=False )

   # Bfd slow timer
   if ( options.saveAll or
        ( bfdConfig.bfdSlowTimer != Tac.Type( 'Bfd::BfdSlowTimer' ).defval ) ):
      getRootAndSaveCommand( '%sslow-timer %d' %
                                ( 'bfd ' if bfdConfig.legacyConfig else '',
                             bfdConfig.bfdSlowTimer ),
                             root, legacyConfig=bfdConfig.legacyConfig )

   if bfdConfig.bfdAdminDown:
      getRootAndSaveCommand( '%sshutdown' %
                                ( 'bfd ' if bfdConfig.legacyConfig else '' ),
                             root, legacyConfig=bfdConfig.legacyConfig )
   elif options.saveAll:
      getRootAndSaveCommand( 'no %sshutdown' %
                                ( 'bfd ' if bfdConfig.legacyConfig else '' ),
                             root, legacyConfig=bfdConfig.legacyConfig )

   authMode, meticulous = getBfdAuthModeFromConfig( bfdConfig.authType )
   if authMode is not None:
      getRootAndSaveCommand( 'authentication mode %s shared-secret profile %s %s' %
                             ( authMode,
                               bfdConfig.secretProfileName,
                               ( 'meticulous' if meticulous else '' ) ),
                             root,
                             legacyConfig=False )
   # Only print 'no authentication' if legacyConfig is not True.  Otherwise, the
   # output of 'show running all' with legacyConfig enabled would feature the
   # 'router bfd' context, even though we are in legacy mode.
   elif options.saveAll and not bfdConfig.legacyConfig:
      getRootAndSaveCommand( 'no authentication', root, legacyConfig=False )

   # Lag local Ip address
   if not bfdConfig.localAddrV4.isAddrZero:
      getRootAndSaveCommand( "%slocal-address %s" %
                                ( 'bfd ' if bfdConfig.legacyConfig else '',
                                  bfdConfig.localAddrV4.stringValue ),
                             root, legacyConfig=bfdConfig.legacyConfig )
   if not bfdConfig.localAddrV6.isAddrZero:
      getRootAndSaveCommand( "%slocal-address %s" %
                                ( 'bfd ' if bfdConfig.legacyConfig else '',
                                  bfdConfig.localAddrV6.stringValue ),
                             root, legacyConfig=bfdConfig.legacyConfig )
   if bfdConfig.localAddrV4.isAddrZero and bfdConfig.localAddrV6.isAddrZero \
      and options.saveAll:
      getRootAndSaveCommand( "no %slocal-address" %
                                ( 'bfd ' if bfdConfig.legacyConfig else '' ),
                             root, legacyConfig=bfdConfig.legacyConfig )

   # HW Acceleration settings
   if bfdConfig.hwAccelerationConfig == 'disabledByCli':
      getRootAndSaveCommand( "hardware acceleration disabled", root,
                             legacyConfig=False )
   elif options.saveAll and not bfdConfig.legacyConfig:
      getRootAndSaveCommand( "no hardware acceleration disabled", root,
                             legacyConfig=False )

   # Session-stats interval setting
   if bfdConfig.sessionStatsInterval != bfdSessionStatsIntervalDefault():
      val = bfdConfig.sessionStatsInterval
      danger = bfdSessionStatsIntervalDangerThreshold()
      getRootAndSaveCommand(
            'session stats snapshot interval %s%d' %
                  ( ( 'dangerous ' if val < danger else '' ), val ),
            root, legacyConfig=False )
   elif options.saveAll and not bfdConfig.legacyConfig:
      getRootAndSaveCommand( 'no session stats snapshot interval', root,
                             legacyConfig=False )

   # Hidden debug commands
   if( bfdConfig.debugLevel != 1 or options.saveAll ):
      getRootAndSaveCommand( '%sdebug rbfd %d' %
                                ( 'bfd ' if bfdConfig.legacyConfig else '',
                                  bfdConfig.debugLevel ),
                             root, legacyConfig=bfdConfig.legacyConfig )

   if bfdConfig.legacyConfig:
      dbsCmd = 'bfd debug-seq-num'
   else:
      dbsCmd = 'debug seq-num'
   if bfdConfig.authType == authType.authDebugSeqNum:
      getRootAndSaveCommand( dbsCmd, root, legacyConfig=bfdConfig.legacyConfig )
   elif options.saveAll:
      getRootAndSaveCommand( 'no %s' % dbsCmd, root,
                             legacyConfig=bfdConfig.legacyConfig )

   # Save Bfd peer config
   peerCliConfig = requireMounts[ 'bfd/config/peerCli' ].peerCliConfig
   if peerCliConfig:
      _saveBfdPeerConfig( peerCliConfig,
                          root[ RouterBfdConfigMode ].getSingletonInstance(),
                          options )

   # Save Bfd Sbfd config
   if not bfdConfig.legacyConfig:
      # do not save sbfd config if we are in legacyConfig mode
      if ( options.saveAll or ( bfdConfig.sbfdLocalIntf != '' ) or
           ( bfdConfig.sbfdLocalIntfIp6 != '' ) or
           ( bfdConfig.sbfdInitiatorConfig != initiatorConfigDefault() ) or
           ( bfdConfig.sbfdReflectorMinRx != reflectorMinRxDefault() ) or
           ( bfdConfig.sbfdReflectorLocalDisc != 0 ) or
           ( bfdConfig.sbfdRttEnabled ) ):
         _saveBfdSbfdConfig( bfdConfig,
                             root[ RouterBfdConfigMode ].getSingletonInstance(),
                             options )

@CliSave.saver( 'Acl::Input::CpConfig', 'acl/cpconfig/cli',
                requireMounts=( 'bfd/config/global', ) )
def saveBfdAclConfig( aclCpConfig, root, requireMounts, options ):
   legacyConfig = requireMounts[ 'bfd/config/global' ].legacyConfig

   # save for bfd service acl
   for aclType in ( 'ip', 'ipv6' ):
      serviceAcl = aclCpConfig.cpConfig[ aclType ].serviceAcl
      for vrfName, serviceAclVrfConfig in serviceAcl.items():
         serviceAclConfig = serviceAclVrfConfig.service.get( 'bfd' )
         if serviceAclConfig:
            if vrfName == DEFAULT_VRF:
               getRootAndSaveCommand( '%s%s access-group %s' %
                                         ( 'bfd ' if legacyConfig else '',
                                           aclType, serviceAclConfig.aclName ),
                                      root, legacyConfig=legacyConfig )
            else:
               getRootAndSaveCommand( '%s%s access-group %s vrf %s' %
                                         ( 'bfd ' if legacyConfig else '',
                                           aclType, serviceAclConfig.aclName,
                                           vrfName ),
                                      root, legacyConfig=legacyConfig )

@CliSave.saver( 'Bfd::ConfigIntf', 'bfd/config/intf',
                requireMounts=( 'interface/config/all',
                                  'interface/status/all',
                                  'bfd/config/global',
                                  'bfd/config/static' ) )
def saveBfdConfig( entity, root, requireMounts, options ):
   cfgIntfNames = []
   staticSessionReqs = requireMounts[ 'bfd/config/static' ].peerIpsByIntf
   if options.saveAllDetail:
      cfgIntfNames = allIpIntfNames( requireMounts, includeEligible=True )
   elif options.saveAll:
      cfgIntfNames = set( allIpIntfNames( requireMounts ) )
      cfgIntfNames.update( entity.echoOn )
      cfgIntfNames.update( staticSessionReqs )
   else:
      cfgIntfNames = set()
      cfgIntfNames.update( entity.intfConfig,
                           entity.echoOn,
                           entity.authType,
                           staticSessionReqs )

   # Interface config
   for intfName in cfgIntfNames:
      intfConfig = entity.intfConfig.get( intfName )
      echoOn = entity.echoOn.get( intfName )
      authMode = entity.authType.get( intfName )
      authSecretName = entity.secretProfileName.get( intfName )
      reqs = staticSessionReqs.get( intfName )

      if ( ( intfConfig is None ) and ( not echoOn ) and
           ( authMode is None ) and ( authSecretName is None ) and
           ( not reqs ) and ( not options.saveAll ) ):
         continue

      _saveBfdIntfConfig( intfName, intfConfig, echoOn, authMode,
                          authSecretName, reqs, root,
                          requireMounts, options )

def _saveBfdIntfConfig( intfName, intfConfig, echoOn, authMode,
                        authSecretName, staticSessionRequests, root,
                        requireMounts, options ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
   cmds = mode[ 'Bfd.intf' ]
   globalCfg = requireMounts[ 'bfd/config/global' ].globalCfg
   legacyConfig = requireMounts[ 'bfd/config/global' ].legacyConfig
   minRxToken = 'min_rx' if legacyConfig else 'min-rx'
   if intfConfig:
      cmds.addCommand( "bfd interval %d %s %d multiplier %d" %
                       ( intfConfig.minTx, minRxToken, intfConfig.minRx,
                         intfConfig.mult ) )
   elif options.saveAll:
      cmds.addCommand( "bfd interval %d %s %d multiplier %d" %
                       ( globalCfg.minTx, minRxToken, globalCfg.minRx,
                         globalCfg.mult ) )

   if ( options.saveAll or echoOn ):
      echoOnStr = ''
      if not echoOn:
         echoOnStr = 'no '
      cmds.addCommand( "%sbfd echo" % echoOnStr )

   authMode, meticulous = getBfdAuthModeFromConfig( authMode, intfOrPeerConfig=True )
   if authMode is not None:
      if authMode == 'disabled':
         secret = ''
      else:
         secret = 'shared-secret profile %s' % authSecretName
      cmds.addCommand( 'bfd authentication mode %s %s %s' %
                             ( authMode,
                               secret,
                               'meticulous' if meticulous else '' ) )
   elif options.saveAll:
      cmds.addCommand( 'no bfd authentication mode' )

   # static bfd sessions
   if staticSessionRequests:
      for peerIp in sorted( staticSessionRequests.typesByPeerIp ):
         sessionTypes = staticSessionRequests.typesByPeerIp[ peerIp ].sessionTypes
         # Ordered sets are not yet supported in tacc, so we must sort here.
         for sessionType in sorted( sessionTypes ):
            multihop = "multihop " if sessionType == SessionType.multihop else ""
            cmds.addCommand( f'bfd static {multihop}neighbor {peerIp}' )
