#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliSave
import Tac
from CliSavePlugin.RoutingBgpCliSave import (
      RouterBgpBaseConfigMode,
      RouterBgpVrfAfConfigMode,
      RouterBgpVrfConfigMode,
      saveAfConfigCallbackDict,
      )
from BgpLib import (
      bgpConfigAttrAfList,
      vpnAfTypes,
      vpnAfTypeMap,
      vpnAfToAfiSafi,
      )
from RouteMapLib import isAsdotConfigured
from Tracing import Handle

__defaultTraceHandle__ = Handle( 'DefaultExportSavePlugin' )

RouterBgpVrfConfigMode.addCommandSequence(
      'Bgp.vrf.default-route-export' )
RouterBgpVrfAfConfigMode.addCommandSequence(
      'Bgp.vrf.afConfig.default-route-export' )

def vpnAfiSafiAllowedInAddressFamily( vpnAfiSafi, addrFamily ):
   # Default-route export not applicable to Flowspec VPN
   if ( vpnAfiSafi == Tac.Type( 'Routing::Bgp::AfiSafi' ).ipv4FlowspecVpn() or
        vpnAfiSafi == Tac.Type( 'Routing::Bgp::AfiSafi' ).ipv6FlowspecVpn() ):
      return False
   if addrFamily is None:
      return True
   elif addrFamily == 'ipv4':
      return vpnAfiSafi != vpnAfToAfiSafi(
                              Tac.Type( 'Routing::Bgp::VpnAf' ).vpnAfVpnV6 )
   elif addrFamily == 'ipv6':
      return vpnAfiSafi != vpnAfToAfiSafi(
                              Tac.Type( 'Routing::Bgp::VpnAf' ).vpnAfVpnV4 )
   return False

def saveDefaultExportConfig( bgpVrfConfig, addrFamily, saveAll=False ):
   cmds = []
   if addrFamily is None:
      defaultExport = bgpVrfConfig.defaultExport
   elif addrFamily == 'ipv4':
      defaultExport = bgpVrfConfig.defaultExportAfV4
   elif addrFamily == 'ipv6':
      defaultExport = bgpVrfConfig.defaultExportAfV6
   else:
      return cmds

   for vpnAfiSafi in vpnAfTypes:
      defaultExportCfg = defaultExport.get( vpnAfiSafi )

      if defaultExportCfg is None:
         if saveAll:
            if vpnAfiSafiAllowedInAddressFamily( vpnAfiSafi, addrFamily ):
               # pylint: disable-next=consider-using-f-string
               cmds.append( 'no default-route export %s' %
                                 vpnAfTypeMap[ vpnAfiSafi ] )
         continue

      # pylint: disable-next=consider-using-f-string
      cmd = 'default-route export %s' % vpnAfTypeMap[ vpnAfiSafi ]
      if defaultExportCfg.always:
         cmd += ' always'
      if defaultExportCfg.routeMap:
         # pylint: disable-next=consider-using-f-string
         cmd += ' route-map %s' % defaultExportCfg.routeMap
      if defaultExportCfg.rcf:
         # pylint: disable-next=consider-using-f-string
         cmd += ' rcf %s()' % defaultExportCfg.rcf
      cmds.append( cmd )

   return cmds

saveAfConfigCallbackDict[ 'default-route-export' ] = saveDefaultExportConfig

@CliSave.saver( 'Routing::Bgp::VrfConfigDir', 'routing/bgp/vrf/config',
                requireMounts=( 'routing/bgp/config',
                                  'routing/bgp/vrf/config',
                                  'routing/bgp/asn/config' ) )
def saveDefaultExportAfConfig( vrfConfigDir, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   saveAll = options.saveAll or options.saveAllDetail

   if bgpConfig.asNumber == 0:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )

   def saveCmds( bgpVrfConfig, vrf, cmds ):
      if cmds:
         bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode ].getOrCreateModeInstance(
               vrf )
         for cmd in cmds:
            bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( cmd )
      for af in bgpConfigAttrAfList():
         cmds = saveDefaultExportConfig( bgpVrfConfig, af, saveAll )
         if cmds:
            bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode ].getOrCreateModeInstance(
                  vrf )
            bgpVrfAfMode = bgpVrfMode[ RouterBgpVrfAfConfigMode
                  ].getOrCreateModeInstance( ( af, vrf ) )
            for cmd in cmds:
               # pylint: disable-next=consider-using-f-string
               bgpVrfAfMode[ 'Bgp.vrf.afConfig.%s' % af ].addCommand( cmd )

   # There is no need to check if cmds is empty.
   # Even if vrf level configuration is empty,
   # af level configuration may have been set separately.
   cmds = saveDefaultExportConfig( bgpConfig, None, saveAll )
   saveCmds( bgpConfig, 'default', cmds )

   for vrf in vrfConfigDir.vrfConfig:
      bgpVrfConfig = vrfConfigDir.vrfConfig[ vrf ]
      cmds = saveDefaultExportConfig( bgpVrfConfig, None, saveAll )
      saveCmds( bgpVrfConfig, vrf, cmds )
