#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliSavePlugin.RoutingBgpCliSave import ( RouterBgpBaseConfigMode,
                                              RouterBgpVrfConfigMode,
                                              RouterBgpBaseAfConfigMode,
                                              RouterBgpVrfAfConfigMode )
import CliSave
from Toggles import (
      BgpCommonToggleLib,
      RoutingLibToggleLib,
      FlowspecToggleLib )

from RouteMapLib import isAsdotConfigured
from BgpLib import bgpConfigAttrAfList

RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.flow-spec ipv4' )
RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.flow-spec ipv6' )
RouterBgpVrfAfConfigMode.addCommandSequence( 'Bgp.vrf.afConfig.flow-spec ipv4' )
RouterBgpVrfAfConfigMode.addCommandSequence( 'Bgp.vrf.afConfig.flow-spec ipv6' )

if BgpCommonToggleLib.toggleFlowspecVpnRxTxEnabled():
   # Flowspec VPN address families are only applicable to default VRF
   RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.flow-spec vpn-ipv4' )
   RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.flow-spec vpn-ipv6' )

def saveRedistributeConfig( bgpConfig, addrFamily, cmds ):
   if addrFamily == 'flow-spec ipv4':
      redistColl = bgpConfig.redistributeAfV4Flowspec
   else:
      assert addrFamily == 'flow-spec ipv6'
      redistColl = bgpConfig.redistributeAfV6Flowspec

   for policyName in sorted( redistColl ):
      redistConfig = redistColl[ policyName ]
      cmd = f'redistribute flow-spec policy {policyName}'
      if redistConfig.rcf:
         cmd += f' rcf {redistConfig.rcf}()'
      cmds.append( cmd )

def saveRtRedirectVrfLpmConfig( bgpConfig, addrFamily, cmds, options ):
   if addrFamily == 'flow-spec ipv4':
      rtRedirectVrfLpm = bgpConfig.rtRedirectVrfLpmV4Flowspec
   else:
      assert addrFamily == 'flow-spec ipv6'
      rtRedirectVrfLpm = bgpConfig.rtRedirectVrfLpmV6Flowspec
   if rtRedirectVrfLpm:
      cmds.append( 'route-target redirect vrf lpm' )
   elif options.saveAll:
      cmds.append( 'no route-target redirect vrf lpm' )

def saveFlowspecConfig( bgpConfig, defaultVrfBgpConfig, vrfName, addrFamily,
                        asdotConfigured, options ):
   cmds = []
   if not addrFamily.startswith( 'flow-spec' ):
      return cmds
   if 'vpn' in addrFamily:
      # Currently, no flowspec vpn specific configuration
      return cmds
   if RoutingLibToggleLib.toggleBgpFlowspecStaticPolicyEnabled():
      saveRedistributeConfig( bgpConfig, addrFamily, cmds )
   if FlowspecToggleLib.toggleFlowspecVrfSelectionEnabled():
      saveRtRedirectVrfLpmConfig( bgpConfig, addrFamily, cmds, options )
   return cmds

@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config',
                requireMounts=( 'routing/bgp/asn/config', ) )
def saveBaseFlowspecConfig( bgpConfig, root, requireMounts, options ):
   if bgpConfig.asNumber == 0:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, asdotConfigured ) )
   for af in bgpConfigAttrAfList():
      cmds = saveFlowspecConfig( bgpConfig, bgpConfig, "", af, asdotConfigured,
                                 options )
      if cmds:
         bgpAfMode = bgpMode[
            RouterBgpBaseAfConfigMode ].getOrCreateModeInstance( af )
         for cmd in cmds:
            bgpAfMode[ f'Bgp.afConfig.{af}' ].addCommand( cmd )

@CliSave.saver( 'Routing::Bgp::VrfConfigDir', 'routing/bgp/vrf/config',
                requireMounts=( 'routing/bgp/config',
                                'routing/bgp/asn/config', ) )
def saveVrfFlowspecConfig( vrfConfigDir, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   if bgpConfig.asNumber == 0:
      return

   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )

   for vrfName in vrfConfigDir.vrfConfig:
      vrfBgpConfig = vrfConfigDir.vrfConfig[ vrfName ]
      bgpVrfMode = bgpMode[
         RouterBgpVrfConfigMode ].getOrCreateModeInstance( vrfName )
      for af in bgpConfigAttrAfList( vrfName ):
         cmds = saveFlowspecConfig( vrfBgpConfig, bgpConfig, vrfName, af,
                                    asdotConfigured, options )
         if cmds:
            bgpVrfAfMode = bgpVrfMode[
               RouterBgpVrfAfConfigMode ].getOrCreateModeInstance( ( af, vrfName ) )
            for cmd in cmds:
               bgpVrfAfMode[ f'Bgp.vrf.afConfig.{af}' ].addCommand( cmd )
