#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetLib import formatRd
from BgpLib import routeTargetToPrint
from CliMode.BgpVpls import (
   BgpVplsInstanceMode,
   BgpVplsGroupMode,
   BgpVplsGroupSignalingMode,
)
from CliSave import (
   Mode,
   saver,
)
from CliSavePlugin.RoutingBgpCliSave import (
   RouterBgpBaseAfConfigMode,
   RouterBgpBaseConfigMode,
)
from RouteMapLib import isAsdotConfigured
from Toggles.BgpCommonToggleLib import toggleBgpVplsRcfImportExportEnabled
from Toggles.PseudowireToggleLib import toggleVplsBgpSignalingEnabled

RouterBgpBaseAfConfigMode.addCommandSequence( 'Bgp.afConfig.vpls' )

class BgpVplsInstanceConfigSaveMode( BgpVplsInstanceMode, Mode ):
   def __init__( self, vplsName ):
      BgpVplsInstanceMode.__init__( self, vplsName )
      Mode.__init__( self, vplsName )

class BgpVplsGroupConfigSaveMode( BgpVplsGroupMode, Mode ):
   def __init__( self, param ):
      vplsName, groupName = param
      BgpVplsGroupMode.__init__( self, vplsName, groupName )
      Mode.__init__( self, param )

# CliSave mode for "signaling" config mode used to configure the various BGP
# signaling parameters - see AID10962.
class BgpVplsGroupSignalingSaveMode( BgpVplsGroupSignalingMode, Mode ):
   def __init__( self, param ):
      vplsName, groupName = param
      BgpVplsGroupSignalingMode.__init__( self, vplsName, groupName )
      Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

RouterBgpBaseConfigMode.addChildMode( BgpVplsInstanceConfigSaveMode )
BgpVplsInstanceConfigSaveMode.addCommandSequence( "Bgp.vpls.instance.config" )
BgpVplsInstanceConfigSaveMode.addChildMode( BgpVplsGroupConfigSaveMode )
BgpVplsGroupConfigSaveMode.addCommandSequence( "Bgp.vpls.group.config" )
if toggleVplsBgpSignalingEnabled():
   BgpVplsGroupConfigSaveMode.addChildMode( BgpVplsGroupSignalingSaveMode )
   BgpVplsGroupSignalingSaveMode.addCommandSequence(
      "Bgp.vpls.group.signaling.config" )

@saver( "Routing::Bgp::BgpVplsPlugin::BgpVplsConfigDir", "vpls/bgp/config",
        requireMounts=( "routing/bgp/config", "routing/bgp/asn/config" ) )
def saveBgpVplsConfig( bgpVplsConfigDir, root, requireMounts, options ):
   bgpConfig = requireMounts[ "routing/bgp/config" ]
   asnConfig = requireMounts[ "routing/bgp/asn/config" ]
   if bgpConfig.asNumber == 0:
      return

   asdotConfigured = isAsdotConfigured( asnConfig )
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, asdotConfigured ) )
   for instCfg in bgpVplsConfigDir.instance.values():
      instMode = bgpMode[ BgpVplsInstanceConfigSaveMode ].getOrCreateModeInstance(
         instCfg.name )
      for groupCfg in instCfg.group.values():
         groupMode = instMode[ BgpVplsGroupConfigSaveMode ].getOrCreateModeInstance(
            ( instCfg.name, groupCfg.name ) )
         cmds = groupMode[ "Bgp.vpls.group.config" ]

         # Save RD
         if groupCfg.rd and groupCfg.rd != "INVALID":
            cmds.addCommand( "rd " + formatRd( groupCfg.rd, asdotConfigured ) )
         elif options.saveAll:
            cmds.addCommand( "no rd" )

         # Save RTs
         importExportRts = []
         for rt in sorted( groupCfg.importRt ):
            if rt in groupCfg.exportRt:
               importExportRts.append( rt )
               continue
            cmd = "route-target import vpls {}".format(
               routeTargetToPrint( rt, asdotConfigured=asdotConfigured ) )
            cmds.addCommand( cmd )

         for rt in importExportRts:
            cmd = "route-target import export vpls {}".format(
               routeTargetToPrint( rt, asdotConfigured=asdotConfigured ) )
            cmds.addCommand( cmd )

         for rt in sorted( groupCfg.exportRt ):
            if rt in groupCfg.importRt:
               continue
            cmd = "route-target export vpls {}".format(
               routeTargetToPrint( rt, asdotConfigured=asdotConfigured ) )
            cmds.addCommand( cmd )

         if options.saveAll and not groupCfg.importRt and not groupCfg.exportRt:
            cmds.addCommand( "no route-target all" )

         def saveBgpVplsRcfImportExportConfig( groupCfg, cmds ):
            rtImportRcf = "route-target import vpls rcf"
            rtImportExportRcf = "route-target import export vpls rcf"
            rtExportRcf = "route-target export vpls rcf"
            if groupCfg.importRcf:
               if groupCfg.importRcf != groupCfg.exportRcf:
                  cmd = f"{rtImportRcf} {groupCfg.importRcf}()"
               else:
                  cmd = f"{rtImportExportRcf} {groupCfg.importRcf}()"
               cmds.addCommand( cmd )
            elif options.saveAll:
               if groupCfg.exportRcf:
                  cmd = "no " + rtImportRcf
               else:
                  cmd = "no " + rtImportExportRcf
               cmds.addCommand( cmd )

            if groupCfg.exportRcf:
               if groupCfg.importRcf != groupCfg.exportRcf:
                  cmd = f"{rtExportRcf} {groupCfg.exportRcf}()"
                  cmds.addCommand( cmd )
            elif options.saveAll:
               if groupCfg.importRcf:
                  cmd = "no " + rtExportRcf
                  cmds.addCommand( cmd )

         if toggleBgpVplsRcfImportExportEnabled():
            saveBgpVplsRcfImportExportConfig( groupCfg, cmds )

         def saveBgpVplsSignalingConfig( groupCfg, groupMode ):
            signalingMode = groupMode[ BgpVplsGroupSignalingSaveMode ].\
               getOrCreateModeInstance( ( groupMode.vplsName, groupMode.groupName ) )
            cmds = signalingMode[ "Bgp.vpls.group.signaling.config" ]

            veId = groupCfg.bgpSignalingVeId
            if veId is not None:
               cmds.addCommand( f"ve-id {veId}" )
            elif options.saveAll:
               cmds.addCommand( "no ve-id" )

            profileName = groupCfg.bgpSignalingProfileName
            if profileName != groupCfg.bgpSignalingProfileNameDefault:
               cmds.addCommand( f"profile {profileName}" )
            elif options.saveAll:
               cmds.addCommand( "no profile" )

            vbs = groupCfg.bgpSignalingVbs
            if vbs != groupCfg.bgpSignalingVbsDefault:
               cmds.addCommand( f"ve-block-size {vbs}" )
            elif options.saveAll:
               cmds.addCommand( f"ve-block-size {groupCfg.bgpSignalingVbsDefault}" )

         if toggleVplsBgpSignalingEnabled():
            saveBgpVplsSignalingConfig( groupCfg, groupMode )
