#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.Cfm import CfmModeBase, MaintenanceDomainModeBase, CfmProfileModeBase, \
   MaintenanceAssociationModeBase, LocalMaintenanceEndPointModeBase, \
   RemoteMaintenanceEndPointModeBase
import CliSave
from CfmLib import (
   ccmTxIntervalToCliToken,
   defaultCcmTxIntervalWithUnit,
   getDefaultCcmDefectAlarm,
)
from CfmTypes import (
   defaultPrimaryVlanId,
   tacAisTxIntervalUnit,
   tacCos,
   tacCfmConst,
   tacMepDirection,
   tacDmMode,
   tacLmMode,
   tacPmTxInterval,
   tacSlmMode,
)
from MultiRangeRule import multiRangeToCanonicalString
import Tac
from Toggles.CfmToggleLib import (
   toggleCfmLocActionEnabled,
   toggleCfmPmMiEnabled,
)
import Ethernet
from TypeFuture import TacLazyType
EthAddr = TacLazyType( 'Arnet::EthAddr' )

CliSave.GlobalConfigMode.addCommandSequence( 'Cfm.config' )

#---------------------------------------------------------------------
# Modes
#---------------------------------------------------------------------
class CfmConfigMode( CfmModeBase, CliSave.Mode ):
   def __init__( self, param ):
      CfmModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class MaintenanceDomainConfigMode( MaintenanceDomainModeBase, CliSave.Mode ):
   def __init__( self, param ):
      MaintenanceDomainModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class CfmProfileConfigMode( CfmProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      CfmProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class MaintenanceAssociationConfigMode( MaintenanceAssociationModeBase,
                                        CliSave.Mode ):
   def __init__( self, param ):
      MaintenanceAssociationModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class LocalMaintenanceEndPointConfigMode( LocalMaintenanceEndPointModeBase,
                                          CliSave.Mode ):
   def __init__( self, param ):
      LocalMaintenanceEndPointModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class RemoteMaintenanceEndPointConfigMode( RemoteMaintenanceEndPointModeBase,
                                          CliSave.Mode ):
   def __init__( self, param ):
      RemoteMaintenanceEndPointModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

#---------------------------------------------------------------------
# SaveBlocks
#---------------------------------------------------------------------
CliSave.GlobalConfigMode.addChildMode( CfmConfigMode )
CfmConfigMode.addCommandSequence( 'Cfm.config' )

CfmConfigMode.addChildMode( MaintenanceDomainConfigMode )
MaintenanceDomainConfigMode.addCommandSequence( 'Cfm.mdConfig' )

CfmConfigMode.addChildMode( CfmProfileConfigMode )
CfmProfileConfigMode.addCommandSequence( 'Cfm.cfmProfileConfig' )

MaintenanceDomainConfigMode.addChildMode( MaintenanceAssociationConfigMode )
MaintenanceAssociationConfigMode.addCommandSequence( 'Cfm.maConfig' )

MaintenanceAssociationConfigMode.addChildMode( RemoteMaintenanceEndPointConfigMode )
RemoteMaintenanceEndPointConfigMode.addCommandSequence( 'Cfm.remoteMepConfig' )

MaintenanceAssociationConfigMode.addChildMode( LocalMaintenanceEndPointConfigMode,
                                      after=[ RemoteMaintenanceEndPointConfigMode ] )
LocalMaintenanceEndPointConfigMode.addCommandSequence( 'Cfm.localMepConfig' )

#---------------------------------------------------------------------
# Handlers
#---------------------------------------------------------------------
@CliSave.saver( 'CfmAgent::Config', 'cfm/config' )
def saveCfmConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   def getCfmConfigMode( root ):
      return root[ CfmConfigMode ].getSingletonInstance()

   def getMdConfigMode( mdConfig, root ):
      cfmConfigMode = getCfmConfigMode( root )
      return cfmConfigMode[ MaintenanceDomainConfigMode ].getOrCreateModeInstance(
         ( mdConfig.name, mdConfig.mdLevel ) )

   def getCfmProfileConfigMode( cfmProfileConfig, root ):
      cfmConfigMode = getCfmConfigMode( root )
      return cfmConfigMode[ CfmProfileConfigMode ].getOrCreateModeInstance(
         cfmProfileConfig.cfmProfileName )

   def getMaConfigMode( maConfig, root ):
      mdConfigMode = getMdConfigMode( maConfig.mdConfig, root )
      return mdConfigMode[
         MaintenanceAssociationConfigMode ].getOrCreateModeInstance(
            ( maConfig.mdConfig.name, maConfig.maNameId ) )

   def getLocalMepConfigMode( localMepConfig, root ):
      maConfigMode = getMaConfigMode( localMepConfig.maConfig, root )
      return maConfigMode[
         LocalMaintenanceEndPointConfigMode ].getOrCreateModeInstance(
            ( localMepConfig.maConfig.mdConfig.name,
              localMepConfig.maConfig.maNameId, localMepConfig.localMepId ) )

   def getRemoteMepConfigMode( remoteMepConfig, maConfig, root ):
      maConfigMode = getMaConfigMode( maConfig, root )
      return maConfigMode[
         RemoteMaintenanceEndPointConfigMode ].getOrCreateModeInstance(
            ( maConfig.mdConfig.name,
              maConfig.maNameId, remoteMepConfig.remoteMepId ) )

   def saveMdConfig( mdConfig, root, options ):
      mdConfigMode = getMdConfigMode( mdConfig, root )
      cmds = mdConfigMode[ 'Cfm.mdConfig' ]

      if mdConfig.defaultMip:
         cmds.addCommand( 'intermediate-point' )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no intermediate-point' )

   def saveMaConfig( maConfig, root, options ):
      maConfigMode = getMaConfigMode( maConfig, root )
      cmds = maConfigMode[ 'Cfm.maConfig' ]

      if maConfig.direction == tacMepDirection.MepDirectionUp:
         direction = 'up'
      elif maConfig.direction == tacMepDirection.MepDirectionDown:
         direction = 'down'
      else:
         direction = 'unknown'
      if direction in [ 'up', 'down' ]:
         cmds.addCommand( 'direction %s' % direction )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no direction' )

      if maConfig.cfmProfileName:
         cmds.addCommand( 'profile %s' % maConfig.cfmProfileName )
      if maConfig.primaryVlanId != defaultPrimaryVlanId:
         cmds.addCommand( 'vlan %d' % maConfig.primaryVlanId )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no vlan' )

   def saveLocalMepConfig( localMepConfig, root, options ):
      localMepConfigMode = getLocalMepConfigMode(
         localMepConfig, root )
      cmds = localMepConfigMode[ 'Cfm.localMepConfig' ]

      if localMepConfig.intfId:
         cmds.addCommand( 'interface %s' % localMepConfig.intfId )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no interface' )

      if localMepConfig.remoteMep:
         remoteMepStr = multiRangeToCanonicalString( localMepConfig.remoteMep )
         cmds.addCommand( 'remote end-point %s' % remoteMepStr )

   def saveRemoteMepConfig( remoteMepConfig, maConfig, root, options ):
      remoteMepConfigMode = getRemoteMepConfigMode(
         remoteMepConfig, maConfig, root )
      cmds = remoteMepConfigMode[ 'Cfm.remoteMepConfig' ]

      if remoteMepConfig.macAddr != EthAddr.ethAddrZero:
         macAddr = remoteMepConfig.macAddr
         macAddrStr = Ethernet.convertMacAddrCanonicalToDisplay( macAddr )
         cmds.addCommand( 'mac address %s' % macAddrStr )

   def saveCfmProfileConfig( cfmProfile, root, options ):
      cfmProfileConfigMode = getCfmProfileConfigMode( cfmProfile, root )
      cmds = cfmProfileConfigMode[ 'Cfm.cfmProfileConfig' ]

      if cfmProfile.ccmEnable:
         cmds.addCommand( 'continuity-check' )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no continuity-check' )

      ccmTxIntervalWithUnit = ccmTxIntervalToCliToken[
                                             cfmProfile.ccmConfig.ccmTxInterval ]
      if saveAllDetail or saveAll or \
         ccmTxIntervalWithUnit.ccmTxInterval != \
               defaultCcmTxIntervalWithUnit.ccmTxInterval or \
         ccmTxIntervalWithUnit.unit != defaultCcmTxIntervalWithUnit.unit:
         cmds.addCommand( 'continuity-check tx-interval %g %s'
                           % ( ccmTxIntervalWithUnit.ccmTxInterval,
                               ccmTxIntervalWithUnit.unit ) )

      if saveAllDetail or saveAll or \
         cfmProfile.ccmConfig.priority != tacCos.max:
         cmds.addCommand( 'continuity-check qos cos %d'
                           % cfmProfile.ccmConfig.priority )

      if saveAllDetail or saveAll or \
         cfmProfile.ccmConfig.ccmDefectAlarm != getDefaultCcmDefectAlarm():
         ccmDefectAlarm = cfmProfile.ccmConfig.ccmDefectAlarm
         alarmCmd = 'continuity-check alarm defect'
         if ccmDefectAlarm:
            ccmDefectAlarmEnum = TacLazyType( "Cfm::CcmDefectAlarm" )
            if ccmDefectAlarmEnum.ccmDefectAlarmRdiCcm in ccmDefectAlarm:
               alarmCmd += ' rdi-ccm'
            if ccmDefectAlarmEnum.ccmDefectAlarmMacStatus in ccmDefectAlarm:
               alarmCmd += ' mac-status'
            if ccmDefectAlarmEnum.ccmDefectAlarmRemoteCcm in ccmDefectAlarm:
               alarmCmd += ' loc-state'
            if ccmDefectAlarmEnum.ccmDefectAlarmErrorCcm in ccmDefectAlarm:
               alarmCmd += ' error-ccm'
            if ccmDefectAlarmEnum.ccmDefectAlarmXconCcm in ccmDefectAlarm:
               alarmCmd += ' cross-connection'
            cmds.addCommand( alarmCmd )
         else:
            cmds.addCommand( 'no ' + alarmCmd )

      if cfmProfile.aisEnable:
         cmds.addCommand( 'alarm indication' )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no alarm indication' )
      if cfmProfile.aisConfig.clientDomainState == 'configured':
         if saveAll or saveAllDetail or \
               cfmProfile.aisConfig.txIntervalUnit != tacAisTxIntervalUnit.seconds:
            cmds.addCommand( 'alarm indication client domain level %d '
                             'tx-interval %d %s' % (
                                cfmProfile.aisConfig.clientDomainLevel,
                                cfmProfile.aisConfig.txIntervalValue,
                                cfmProfile.aisConfig.txIntervalUnit ) )
         else:
            cmds.addCommand( 'alarm indication client domain level %d'
                             % cfmProfile.aisConfig.clientDomainLevel )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no alarm indication client' )

      def pmTxIntervalStr( txInterval ):
         # avoid the trailing ".0" for whole numbers
         txInterval = int( txInterval ) if txInterval.is_integer() else txInterval
         return str( txInterval )

      if cfmProfile.dmEnable:
         if cfmProfile.dmConfig.endMode == tacDmMode.dmModeSingle:
            cmds.addCommand( 'measurement delay single-ended' )
         else:
            cmds.addCommand( 'measurement delay dual-ended' )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no measurement delay' )

      if saveAllDetail or saveAll or \
            cfmProfile.dmConfig.txInterval != tacPmTxInterval.intervalDefault:
         cmds.addCommand( 'measurement delay tx-interval %s milliseconds'
                           % pmTxIntervalStr( cfmProfile.dmConfig.txInterval ) )

      if saveAllDetail or saveAll or \
            cfmProfile.dmConfig.priority != tacCos.max:
         cmds.addCommand( 'measurement delay qos cos %d'
                           % cfmProfile.dmConfig.priority )

      if cfmProfile.lmEnable:
         if cfmProfile.lmConfig.endMode == tacLmMode.lmModeSingle:
            cmds.addCommand( 'measurement loss single-ended' )
         else:
            cmds.addCommand( 'measurement loss dual-ended' )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no measurement loss' )

      if saveAllDetail or saveAll or \
            cfmProfile.lmConfig.txInterval != tacPmTxInterval.intervalDefault:
         cmds.addCommand( 'measurement loss tx-interval %s milliseconds'
                           % pmTxIntervalStr( cfmProfile.lmConfig.txInterval ) )

      if saveAllDetail or saveAll or \
            cfmProfile.lmConfig.priority != tacCos.max:
         cmds.addCommand( 'measurement loss qos cos %d'
                           % cfmProfile.lmConfig.priority )

      if cfmProfile.slmEnable:
         if cfmProfile.slmConfig.endMode == tacSlmMode.slmModeSingle:
            cmds.addCommand( 'measurement loss synthetic single-ended' )
         else:
            cmds.addCommand( 'measurement loss synthetic dual-ended' )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no measurement loss synthetic' )

      if saveAllDetail or saveAll or \
            cfmProfile.slmConfig.txInterval != tacPmTxInterval.intervalDefault or \
            cfmProfile.slmConfig.measurementTxFrames !=\
                                                   tacCfmConst.defaultSlmTxFrames:
         command = 'measurement loss synthetic tx-interval %s milliseconds'\
                               % pmTxIntervalStr( cfmProfile.slmConfig.txInterval )
         if saveAllDetail or saveAll or \
            cfmProfile.slmConfig.measurementTxFrames !=\
                                                   tacCfmConst.defaultSlmTxFrames:
            command += ' period %d frames'\
                                       % cfmProfile.slmConfig.measurementTxFrames
         cmds.addCommand( command )

      if saveAllDetail or saveAll or \
            cfmProfile.slmConfig.priority.keys() != [ tacCos.max ]:
         cmds.addCommand( 'measurement loss synthetic qos cos %s'
                  % multiRangeToCanonicalString( cfmProfile.slmConfig.priority ) )
      if toggleCfmPmMiEnabled():
         miConfig = cfmProfile.dmConfig.miConfig
         if miConfig != Tac.Value( 'Cfm::MeasurementIntervalConfig' ):
            command = 'measurement delay interval %d minutes' % \
                      ( miConfig.interval / 60 )
            cmds.addCommand( command )
         elif saveAll or saveAllDetail:
            cmds.addCommand( 'no measurement delay interval' )
         miConfig = cfmProfile.slmConfig.miConfig
         if miConfig != Tac.Value( 'Cfm::MeasurementIntervalConfig' ):
            command = 'measurement loss synthetic interval %d minutes' % \
                      ( miConfig.interval / 60 )
            cmds.addCommand( command )
         elif saveAll or saveAllDetail:
            cmds.addCommand( 'no measurement loss synthetic interval' )

   def saveCfmLmEnabledConfig( cfmLmEnabled, root ):
      cfmConfigMode = getCfmConfigMode( root )
      cmds = cfmConfigMode[ "Cfm.config" ]
      if cfmLmEnabled:
         cmds.addCommand( 'measurement loss inband' )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no measurement loss inband' )

   def saveCfmSlmEnabledConfig( cfmSlmEnabled, root ):
      cfmConfigMode = getCfmConfigMode( root )
      cmds = cfmConfigMode[ "Cfm.config" ]
      if cfmSlmEnabled:
         cmds.addCommand( 'measurement loss synthetic' )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no measurement loss synthetic' )

   def saveCfmLocActionConfig( locActionRoutingDisabled, root ):
      cfmConfigMode = getCfmConfigMode( root )
      cmds = cfmConfigMode[ "Cfm.config" ]
      locActionCmd = 'continuity-check loc-state action disable interface routing'
      if locActionRoutingDisabled:
         cmds.addCommand( locActionCmd )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no ' + locActionCmd )

   # Use of get() instead of items(), since this python code do not run
   # in context of the Activity Loop. Refer Bug 86221 for more info.
   for mdLevel in entity.mdConfig:
      mdConfig = entity.mdConfig.get( mdLevel )
      if mdConfig is None:
         continue
      saveMdConfig( mdConfig, root, options )

      for maNameId in mdConfig.maConfig:
         maConfig = mdConfig.maConfig.get( maNameId )
         if maConfig is None:
            continue
         saveMaConfig( maConfig, root, options )

         for remoteMepId in maConfig.remoteMepConfig:
            remoteMepConfig = maConfig.remoteMepConfig.get( remoteMepId )
            if remoteMepConfig is None:
               continue
            saveRemoteMepConfig( remoteMepConfig, maConfig, root, options )

         for localMepId in maConfig.localMepConfig:
            localMepConfig = maConfig.localMepConfig.get( localMepId )
            if localMepConfig is None:
               continue
            saveLocalMepConfig( localMepConfig, root, options )

   for cfmProfileName in entity.cfmProfile:
      cfmProfile = entity.cfmProfile.get( cfmProfileName )
      if cfmProfile is None:
         continue
      saveCfmProfileConfig( cfmProfile, root, options )

   saveCfmLmEnabledConfig( entity.cfmLmEnabled, root )

   saveCfmSlmEnabledConfig( entity.cfmSlmEnabled, root )

   if toggleCfmLocActionEnabled():
      saveCfmLocActionConfig( entity.locActionRoutingDisabled, root )
