# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliSaveBlock import SensitiveCommand
from CliMode.ContainerMgrMode import ContainerMgrMode
from CliMode.ContainerMode import ContainerMode
from CliMode.ContainerPersistStorageMode import ContainerPersistStorageMode
from CliMode.ContainerStartupMode import ContainerStartupMode
from CliMode.RegistryMode import RegistryMode
from CliMode.ContainerImagesMode import ImagesMode
from CliMode.ContainerProfileMode import ContainerProfileMode
from CliMode.ContainerProfilePersistStorageMode import \
     ContainerProfilePersistStorageMode
from CliMode.ContainerProfileStartupMode import ContainerProfileStartupMode
import DesCrypt
from Toggles.ContainerMgrToggleLib import toggleContainerStartupConditionEnabled
from TypeFuture import TacLazyType

CpuShares = TacLazyType( 'ContainerMgr::CpuShares' )
CpuCores = TacLazyType( 'ContainerMgr::CpuCores' )
MemoryHardLimit = TacLazyType( 'ContainerMgr::MemoryHardLimit' )
MemorySoftLimit = TacLazyType( 'ContainerMgr::MemorySoftLimit' )
SecurityMode = TacLazyType( 'ContainerMgr::SecurityMode' )
NetworkingMode = TacLazyType( 'ContainerMgr::NetworkingMode' )
LoggingDriver = TacLazyType( 'ContainerMgr::LoggingDriver' )
RestartPolicy = TacLazyType( 'ContainerMgr::RestartPolicy' )
StartupConditionPollInterval = \
      TacLazyType( 'ContainerMgr::StartupConditionPollInterval' )

class ContainerMgrConfigMode( ContainerMgrMode, CliSave.Mode ):
   def __init__( self, param ):
      ContainerMgrMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( ContainerMgrConfigMode )
ContainerMgrConfigMode.addCommandSequence( 'ContainerMgr.config' )

class RegistryConfigMode( RegistryMode, CliSave.Mode ):
   def __init__( self, param ):
      RegistryMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

ContainerMgrConfigMode.addChildMode( RegistryConfigMode )
RegistryConfigMode.addCommandSequence( 'ContainerMgr.registryConfigDir' )

class ImagesConfigMode( ImagesMode, CliSave.Mode ):
   def __init__( self, param ):
      ImagesMode.__init__( self )
      CliSave.Mode.__init__( self, None )

   def skipIfEmpty( self ):
      return True

ContainerMgrConfigMode.addChildMode( ImagesConfigMode )
ImagesConfigMode.addCommandSequence( 'ContainerMgr.imageLoadConfigDir' )

class ContainerConfigMode( ContainerMode, CliSave.Mode ):
   def __init__( self, param ):
      ContainerMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

ContainerMgrConfigMode.addChildMode( ContainerConfigMode )
ContainerConfigMode.addCommandSequence( 'ContainerMgr.containerConfig' )

class ContainerStartupConfigMode( ContainerStartupMode, CliSave.Mode ):
   def __init__( self, param ):
      ContainerStartupMode.__init__( self )
      CliSave.Mode.__init__( self, None )

   def skipIfEmpty( self ):
      return True

if toggleContainerStartupConditionEnabled():
   ContainerConfigMode.addChildMode( ContainerStartupConfigMode )
   ContainerStartupConfigMode.addCommandSequence(
         'ContainerMgr.containerConfig.startup' )

class ContainerPersistStorageConfigMode( ContainerPersistStorageMode, CliSave.Mode ):
   def __init__( self, param ):
      ContainerPersistStorageMode.__init__( self )
      CliSave.Mode.__init__( self, None )

   def skipIfEmpty( self ):
      return True

ContainerConfigMode.addChildMode( ContainerPersistStorageConfigMode )
ContainerPersistStorageConfigMode.addCommandSequence(
      'ContainerMgr.containerConfig.bindMount' )

class ContainerProfileConfigMode( ContainerProfileMode, CliSave.Mode ):
   def __init__( self, param ):
      ContainerProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

ContainerMgrConfigMode.addChildMode( ContainerProfileConfigMode )
ContainerProfileConfigMode.addCommandSequence(
      'ContainerMgr.containerProfileConfig' )

class ContainerProfileStartupConfigMode( ContainerProfileStartupMode,
                                                CliSave.Mode ):
   def __init__( self, param ):
      ContainerProfileStartupMode.__init__( self )
      CliSave.Mode.__init__( self, None )

   def skipIfEmpty( self ):
      return True

if toggleContainerStartupConditionEnabled():
   ContainerProfileConfigMode.addChildMode( ContainerProfileStartupConfigMode )
   ContainerProfileStartupConfigMode.addCommandSequence(
         'ContainerMgr.containerProfileConfig.startup' )

class ContainerProfilePersistStorageConfigMode( ContainerProfilePersistStorageMode,
                                                CliSave.Mode ):
   def __init__( self, param ):
      ContainerProfilePersistStorageMode.__init__( self )
      CliSave.Mode.__init__( self, None )

   def skipIfEmpty( self ):
      return True

ContainerProfileConfigMode.addChildMode(
      ContainerProfilePersistStorageConfigMode )
ContainerProfilePersistStorageConfigMode.addCommandSequence(
      'ContainerMgr.containerProfileConfig.bindMount' )

@CliSave.saver( 'ContainerMgr::ContainerMgrConfig', 'containerMgr/config' )
def saveContainerMgrConfig( entity, root, requireMounts, options ):
   if not entity.agentEnabled:
      return
   mode = root[ ContainerMgrConfigMode ].getSingletonInstance()
   cmds = mode[ 'ContainerMgr.config' ]
   saveAll = options.saveAll

   if entity.containerMgrArgs != entity.invalidString:
      cmds.addCommand( 'daemon args %s' % entity.containerMgrArgs )
   elif saveAll:
      cmds.addCommand( 'no daemon args' )

   if entity.persistentPath != entity.defaultPersistentPath:
      cmds.addCommand( 'persistent-path %s' % entity.persistentPath )
   elif saveAll:
      cmds.addCommand( 'no persistent-path' )

@CliSave.saver( 'ContainerMgr::RegistryConfigDir', 'containerMgr/registry/config',
                requireMounts=( 'containerMgr/config', ) )
def saveRegistryConfig( entity, root, requireMounts, options ):
   if not requireMounts[ 'containerMgr/config' ].agentEnabled:
      return

   saveAll = options.saveAll or options.saveAllDetail
   containerMgrMode = root[ ContainerMgrConfigMode ].getSingletonInstance()
   registryModeColl = containerMgrMode[ RegistryConfigMode ]

   for registry in entity.registry.values():
      registryMode = registryModeColl.getOrCreateModeInstance( registry.name )
      cmdsRegistry = registryMode[ 'ContainerMgr.registryConfigDir' ]

      if registry.params.insecure:
         cmdsRegistry.addCommand( 'insecure' )
      elif saveAll:
         cmdsRegistry.addCommand( 'no insecure' )

      if registry.params.userName:
         cmdsRegistry.addCommand( 'username %s' % registry.params.userName )
      elif saveAll:
         cmdsRegistry.addCommand( 'no username' )

      if registry.params.password:
         password = DesCrypt.encrypt( registry.name.encode(),
                                      registry.params.password.encode() ).decode()
         cmd = SensitiveCommand( 'password 7 {}', password )
         cmdsRegistry.addCommand( cmd )
      elif saveAll:
         cmdsRegistry.addCommand( 'no password' )

      if registry.params.serverName:
         cmdsRegistry.addCommand( 'server %s' % registry.params.serverName )
      elif saveAll:
         cmdsRegistry.addCommand( 'no server' )

def startupConditionCommands( bashCmd, pollInterval, saveAllOptions ):
   nonDefaultPollInterval = ( pollInterval !=
                              StartupConditionPollInterval.invalid )
   startupConditonBashCmdPrefix = 'trigger condition bash'
   startupCmds = []

   if bashCmd:
      if '\n' in bashCmd:
         # Multiline bash command: construct multi-line config
         # keywords go in first line, add an EOF as the last line
         # For all lines except last, indent it to be still inside the mode.
         lines = [ startupConditonBashCmdPrefix ]
         indentLen = 3
         indentation = ' ' * indentLen

         # The parser always adds a trailing newline
         assert bashCmd.endswith( '\n' )
         multilineCmd = bashCmd + "EOF"
         for line in multilineCmd.split( '\n' ):
            indentedLine = indentation + line
            lines.append( indentedLine )
         startupCmds.append( '\n'.join( lines ) )
      else:
         # Single line bashCmd:
         startupCmds.append( f"{startupConditonBashCmdPrefix} {bashCmd}" )
   elif saveAllOptions:
      startupCmds.append( f"no {startupConditonBashCmdPrefix}" )

   if nonDefaultPollInterval:
      startupCmds.append( f"trigger condition poll interval {pollInterval}" )
   elif saveAllOptions:
      startupCmds.append( "no trigger condition poll interval" )
   return startupCmds

@CliSave.saver( 'ContainerMgr::ContainerConfigDir', 'containerMgr/container/config',
                requireMounts=( 'containerMgr/config', ) )
def saveContainerConfig( entity, root, requireMounts, options ):
   if not requireMounts[ 'containerMgr/config' ].agentEnabled:
      return

   saveAllOptions = options.saveAll or options.saveAllDetail
   for container in entity.container.values():
      containerMgrMode = root[ ContainerMgrConfigMode ].getSingletonInstance()
      containerModeColl = containerMgrMode[ ContainerConfigMode ]
      containerMode = containerModeColl.getOrCreateModeInstance( container.name )
      containerCmds = containerMode[ 'ContainerMgr.containerConfig' ]

      params = container.params
      if params.imageName:
         containerCmds.addCommand( 'image %s' % params.imageName )
      elif saveAllOptions:
         containerCmds.addCommand( 'no image' )

      if container.enabled:
         if container.enabledState.usingDeprecated:
            containerCmds.addCommand( 'on-boot' )
         else:
            containerCmds.addCommand( 'no shutdown' )
      elif saveAllOptions:
         containerCmds.addCommand( 'shutdown' )

      if container.profileName != "":
         containerCmds.addCommand( 'profile %s' % container.profileName )
      elif saveAllOptions:
         containerCmds.addCommand( 'no profile' )

      if params.cpuShares != CpuShares.invalid:
         containerCmds.addCommand( 'cpu shares %s' % params.cpuShares )
      elif saveAllOptions:
         containerCmds.addCommand( 'no cpu shares' )

      if params.cpuCores != CpuCores.invalid:
         containerCmds.addCommand( 'cpu cores %s' % params.cpuCores )
      elif saveAllOptions:
         containerCmds.addCommand( 'no cpu cores' )

      if params.memoryHardLimit != MemoryHardLimit.invalid:
         containerCmds.addCommand( 'memory %s' % params.memoryHardLimit )
      elif saveAllOptions:
         containerCmds.addCommand( 'no memory' )

      if params.options:
         containerCmds.addCommand( 'options %s' % params.options )
      elif saveAllOptions:
         containerCmds.addCommand( 'no options' )

      if params.command:
         containerCmds.addCommand( 'command %s' % params.command )
      elif saveAllOptions:
         containerCmds.addCommand( 'no command' )

      if toggleContainerStartupConditionEnabled():
         bashCmd = params.startupConditionBash
         pollInterval = params.startupConditionPollInterval
         containerStartupModeColl = \
            containerMode[ ContainerStartupConfigMode ]
         containerStartupMode = \
            containerStartupModeColl.getSingletonInstance()
         startupCmdsKey = 'ContainerMgr.containerConfig.startup'
         startupCmds = containerStartupMode[ startupCmdsKey ]
         for startupCmd in startupConditionCommands( bashCmd, pollInterval,
                                                     saveAllOptions ):
            startupCmds.addCommand( startupCmd )

      containerPersistStorageModeColl = \
         containerMode[ ContainerPersistStorageConfigMode ]
      containerPersistStorageMode = \
         containerPersistStorageModeColl.getSingletonInstance()
      persistStorageCmds = \
         containerPersistStorageMode[ 'ContainerMgr.containerConfig.bindMount' ]

      for entry in sorted( container.bindMount.values() ):
         persistStorageCmds.addCommand( "mount src {} dst {}".format(
            entry.hostPath,
            entry.ctrPath ) )

ContainerConfigMode.addChildMode( ContainerPersistStorageConfigMode )
ContainerPersistStorageConfigMode.addCommandSequence(
      'ContainerMgr.containerConfig.bindMount' )

@CliSave.saver( 'ContainerMgr::ContainerProfileConfigDir',
                'containerMgr/container/profileConfig' )
def saveContainerProfileConfig( entity, root, requireMounts, options ):
   saveAllOptions = options.saveAll or options.saveAllDetail
   for profileConfig in entity.profile.values():
      containerMgrMode = root[ ContainerMgrConfigMode ].getSingletonInstance()
      containerProfileModeColl = containerMgrMode[ ContainerProfileConfigMode ]
      containerProfileMode = containerProfileModeColl.getOrCreateModeInstance(
            profileConfig.name )
      profileCmds = containerProfileMode[ 'ContainerMgr.containerProfileConfig' ]

      params = profileConfig.params
      if params.imageName:
         profileCmds.addCommand( 'image %s' % params.imageName )
      elif saveAllOptions:
         profileCmds.addCommand( 'no image' )

      if params.cpuShares != CpuShares.invalid:
         profileCmds.addCommand( 'cpu shares %s' % params.cpuShares )
      elif saveAllOptions:
         profileCmds.addCommand( 'no cpu shares' )

      if params.cpuCores != CpuCores.invalid:
         profileCmds.addCommand( 'cpu cores %s' % params.cpuCores )
      elif saveAllOptions:
         profileCmds.addCommand( 'no cpu cores' )

      if params.memoryHardLimit != MemoryHardLimit.invalid:
         profileCmds.addCommand( 'memory hard-limit %s' % params.memoryHardLimit )
      elif saveAllOptions:
         profileCmds.addCommand( 'no memory hard-limit' )

      if params.memorySoftLimit != MemorySoftLimit.invalid:
         profileCmds.addCommand( 'memory soft-limit %s' % params.memorySoftLimit )
      elif saveAllOptions:
         profileCmds.addCommand( 'no memory soft-limit' )

      if params.options:
         profileCmds.addCommand( 'options %s' % params.options )
      elif saveAllOptions:
         profileCmds.addCommand( 'no options' )

      if params.command:
         profileCmds.addCommand( 'command %s' % params.command )
      elif saveAllOptions:
         profileCmds.addCommand( 'no command' )

      if params.environment:
         profileCmds.addCommand( 'environment %s' % params.environment )
      elif saveAllOptions:
         profileCmds.addCommand( 'no environment' )

      if params.secMode == SecurityMode.secModePrivileged:
         profileCmds.addCommand( 'security mode privileged' )
      elif saveAllOptions:
         profileCmds.addCommand( 'no security mode' )

      if params.netMode == NetworkingMode.netModeHost:
         profileCmds.addCommand( 'networking mode host' )
      elif saveAllOptions:
         profileCmds.addCommand( 'no networking mode' )

      if params.loggingDriver == LoggingDriver.driverSyslog:
         profileCmds.addCommand( 'logging driver syslog' )
      elif saveAllOptions:
         profileCmds.addCommand( 'no logging driver' )

      if params.restartPolicy == RestartPolicy.restartPolicyAlways:
         profileCmds.addCommand( 'on-exit restart policy status all' )
      elif params.restartPolicy == RestartPolicy.restartPolicyNoRestart:
         profileCmds.addCommand( 'on-exit restart policy never' )
      elif saveAllOptions:
         profileCmds.addCommand( 'no on-exit restart policy' )

      if toggleContainerStartupConditionEnabled():
         bashCmd = params.startupConditionBash
         pollInterval = params.startupConditionPollInterval
         containerProfileStartupModeColl = \
            containerProfileMode[ ContainerProfileStartupConfigMode ]
         containerProfileStartupMode = \
            containerProfileStartupModeColl.getSingletonInstance()
         startupCmdsKey = 'ContainerMgr.containerProfileConfig.startup'
         startupCmds = containerProfileStartupMode[ startupCmdsKey ]
         for startupCmd in startupConditionCommands( bashCmd, pollInterval,
                                                     saveAllOptions ):
            startupCmds.addCommand( startupCmd )

      containerProfilePersistStorageModeColl = \
         containerProfileMode[ ContainerProfilePersistStorageConfigMode ]
      containerProfilePersistStorageMode = \
         containerProfilePersistStorageModeColl.getSingletonInstance()
      persistStorageCmds = containerProfilePersistStorageMode[
         'ContainerMgr.containerProfileConfig.bindMount' ]

      for entry in sorted( profileConfig.bindMount.values() ):
         persistStorageCmds.addCommand( "mount src {} dst {}".format(
            entry.hostPath,
            entry.ctrPath ) )

@CliSave.saver( 'ContainerMgr::ImageLoadConfigDir',
                'containerMgr/image/config',
                requireMounts=( 'containerMgr/config', ) )
def saveContainerImageLoadConfig( entity, root, requireMounts, options ):
   if not requireMounts[ 'containerMgr/config' ].agentEnabled:
      return
   containerMgrMode = root[ ContainerMgrConfigMode ].getSingletonInstance()
   imagesModeColl = containerMgrMode[ ImagesConfigMode ]
   imagesMode = imagesModeColl.getSingletonInstance()
   loadCmds = imagesMode[ 'ContainerMgr.imageLoadConfigDir' ]

   for pathUrl in sorted( entity.imagePathUrl ):
      loadCmds.addCommand( 'load %s' % pathUrl )
