# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import CliSave
from CliSavePlugin import Management
from IpLibConsts import DEFAULT_VRF

mgmtControllerConfigPath = "mgmt/controller/config"

class MgmtCvxConfigMode( Management.MgmtConfigMode ):
   def __init__( self, param ):
      Management.MgmtConfigMode.__init__( self, param )

   @staticmethod
   def modeName( clusterName ):
      return ( "cvx %s" % clusterName if clusterName and clusterName !=
               Tac.Value( "Controller::Constants" ).clusterDefaultName else "cvx" )

   def skipIfEmpty( self ):
      # We write to Sysdb upon entering the mode, so cannot skip
      return False

CliSave.GlobalConfigMode.addChildMode( MgmtCvxConfigMode )
MgmtCvxConfigMode.addCommandSequence( 'Mgmt.cvx' )

def saveMgmtCvxConfigHelper( cfg, root, options ):
   cmds = []

   if cfg.enabled or options.saveAll:
      cmds.append( "%sshutdown" % ( 'no ' if cfg.enabled else '' ) )

   for controllerConfig in cfg.controllerConfig.values():
      connectionConfig = controllerConfig.connectionConfig
      sysnameStr = ( " sysname %s" % connectionConfig.sysname
                     if connectionConfig.sysname else "" )
      host = ( connectionConfig.ip if connectionConfig.ip != '0.0.0.0'
               else connectionConfig.hostname )
      cmds.append( f"server host {host}{sysnameStr}" )
   
   if len( cfg.controllerConfig ) == 0 and options.saveAll:
      cmds.append( "no server host" )

   sourceIntfChanged = cfg.switchHost.sourceIntf != ''
   if sourceIntfChanged or options.saveAll:
      if cfg.switchHost.sourceIntf:
         cmds.append( "source-interface %s" %
                      cfg.switchHost.sourceIntf )
      else:
         cmds.append( "no source-interface" )
    
   timeoutType = Tac.Type( "Controller::Heartbeat::Timeout" )
   intervalType = Tac.Type( "Controller::Heartbeat::Interval" )

   intervalChanged = cfg.heartbeatConfig.punchInterval != intervalType.defaultVal
   if intervalChanged or options.saveAll:
      cmds.append( 'heartbeat-interval %s' % \
                      int( cfg.heartbeatConfig.punchInterval ) )

   timeoutChanged = cfg.heartbeatConfig.timeout != timeoutType.defaultVal
   if timeoutChanged or options.saveAll:
      cmds.append( 'heartbeat-timeout %s' % int( cfg.heartbeatConfig.timeout ) )
      
   sslProfileName = cfg.sslProfileConfig.sslProfileName
   if sslProfileName or options.saveAll:
      if sslProfileName:
         cmds.append( 'ssl profile %s' % sslProfileName )
      else:
         cmds.append( 'no ssl profile' )
   
   vrfNameChanged = cfg.switchHost.vrfName != DEFAULT_VRF
   if vrfNameChanged or options.saveAll:
      cmds.append( "vrf %s" %  cfg.switchHost.vrfName )

   return cmds

@CliSave.saver( 'ControllerClient::Config', 'mgmt/controller/config' )
def saveMgmtCvxConfig( cfg, root, requireMounts, options ):

   cmds = saveMgmtCvxConfigHelper( cfg, root, options )
   
   if cmds or cfg.clusterName:
      mode = root[ MgmtCvxConfigMode ].getOrCreateModeInstance(
         MgmtCvxConfigMode.modeName( cfg.clusterName ) )
      cmdSeq = mode[ "Mgmt.cvx" ]
      for cmd in cmds:
         cmdSeq.addCommand( cmd )

