# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import CliSave
from CliMode.DecapGroup import DecapGroupBaseMode
from ipaddress import ip_address, IPv4Address, IPv6Address
import sys

# pylint:disable=unicode-builtin # pylint: disable=bad-option-value
# pylint: disable-next=undefined-variable
unicodeType = str if sys.version_info.major > 2 else unicode

class DecapGroupConfigSaveMode( DecapGroupBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      self.saveOrder_ = 50
      DecapGroupBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addCommandSequence( 'Tunnel.DecapGroups' )
CliSave.GlobalConfigMode.addChildMode( DecapGroupConfigSaveMode,
                                       after=[ 'Tunnel.DecapGroups' ] )
DecapGroupConfigSaveMode.addCommandSequence( 'DecapGroups.cfg' )

#switch(config)# ip decap-group type udp destination port 6080 payload ip
#switch(config)# ip decap-group foo
#switch(config-dg-foo)# tunnel type gre
#switch(config-dg-foo)# tunnel decap-ip 1.1.0.1

@CliSave.saver( 'Tunnel::Decap::ConfigInput', "routing/tunneldecap/input/cli" )
def saveDecapGroups( entity, root, requireMounts, options ):
   cmds = root[ 'Tunnel.DecapGroups' ]
   for port, ptype in entity.globalDestPortToPayloadType.items():
      ptype = 'ip' if ptype == 'ipvx' else ptype
      # pylint: disable-next=consider-using-f-string
      cmd = 'ip decap-group type udp destination port {} payload {}'.format(
               port, ptype )
      cmds.addCommand( cmd )

   for dgName in entity.decapGroup:
      dg = entity.decapGroup[ dgName ]
      if dg.dynamic:
         continue
      mode = root[ DecapGroupConfigSaveMode ].getOrCreateModeInstance( dgName )
      cmdSeq = mode[ 'DecapGroups.cfg' ]

      if dg.tunnelType == 'gre':
         cmdSeq.addCommand( 'tunnel type gre' )

      elif dg.tunnelType == 'ipip':
         cmdSeq.addCommand( 'tunnel type ipip' )

      elif dg.tunnelType == 'udp':
         cmdSeq.addCommand( 'tunnel type udp' )
         if dg.destinationPort != 0:
            cmdSeq.addCommand( 'tunnel udp destination port ' + \
                               str( dg.destinationPort ) + \
                               ' payload ' + dg.payloadType )

      for each in dg.decapIp:
         cmdLine = ''
         addr, mask = each.stringValue.split('/')
         if ( isinstance( ip_address( unicodeType( addr ) ), IPv4Address )
            and ( mask == str( 32 )
            ) ) or ( isinstance( ip_address( unicodeType( addr ) ), IPv6Address )
            and ( mask == str( 128 ) ) ):
            cmdLine = 'tunnel decap-ip ' + addr
         else:
            cmdLine = 'tunnel decap-ip ' + each.stringValue
         cmdSeq.addCommand( str( cmdLine ) )
      
      for key in dg.decapIntf:
         # get exclude interface option, if any
         excludeIntfOption = ''
         if key in dg.decapExcludeIntfSet:
            # only permit exclude interface option on decap-interface all
            assert key.intfId == ""
            excludeIntf = dg.decapExcludeIntfSet[ key ].excludeIntf
            if excludeIntf:
               excludeIntfOption = ' exclude ' + \
                   ' '.join( [ str( intfId ) for intfId in excludeIntf ] )
         else:
            excludeIntfOption = ''

         if dg.decapIntf[ key ].oldConfig: 
            if key.intfId != "":
               cmdSeq.addCommand( 'tunnel decap-interface ' + 
                                  key.intfId )
            else:
               cmdSeq.addCommand( 'tunnel decap-interface all' + excludeIntfOption )
         else:
            if key.intfId != "":
               cmdSeq.addCommand( 
                  'tunnel decap-interface ' + key.intfId + \
                  ' address-family ' + key.addressFamily + \
                  ' address ' + dg.decapIntf[ key ].addressType ) 
            else:
               cmdSeq.addCommand( 
                  'tunnel decap-interface all' \
                  ' address-family ' + key.addressFamily + \
                  ' address ' + dg.decapIntf[ key ].addressType + excludeIntfOption )
      
      if dg.forwardingVrf:
         cmdSeq.addCommand( 'forwarding vrf ' + dg.forwardingVrf )
      

      if dg.qosTcFromMplsTc:
         cmdSeq.addCommand( 'tunnel overlay mpls qos map mpls-traffic-class to '
                            'traffic-class' )

      for greKey in dg.greKeyToForwardingVrfMapping:
         cmd = 'tunnel gre key ' + str( greKey ) + ' forwarding vrf ' + \
               str( dg.greKeyToForwardingVrfMapping[ greKey ] )
         cmdSeq.addCommand( cmd )

def Plugin( entityManager ):
   pass
