#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from itertools import chain

import CliSave
# Need to import this to get the DhcpRelay.dhcp config sequence
import CliSavePlugin.DhcpRelayHelperCliSave # pylint: disable=unused-import
from CliSavePlugin.IntfCliSave import IntfConfigMode
# Need to import this to get the Ira.ipIntf config sequence (which has the dhcp
# client command) BUG368914
import CliSavePlugin.IraCliSave # pylint: disable=unused-import
from RoutingIntfUtils import allRoutingProtocolIntfNames
from EosDhcpServerLib import convertLeaseSeconds
from EosDhcpServerLib import getSubOptionData
from EosDhcpServerLib import subOptionCmd
from EosDhcpServerLib import getOptionData
from EosDhcpServerLib import genOptionCmd
from EosDhcpServerLib import formatL2InfoCmd
from EosDhcpServerLib import formatInfoOptionHexOrStrCmd
from EosDhcpServerLib import formatRemoteIdHexOrStrCmd
from EosDhcpServerLib import formatVendorClassCmd
from EosDhcpServerLib import (
   featureFlexibleMatchingFuture,
   featureArbitraryOptionMatch,
   featureEchoClientId,
)
from CliMode.DhcpServer import DhcpServerBaseMode, DhcpServerSubnetBaseMode
from CliMode.DhcpServer import DhcpServerGlobalClientClassBaseMode
from CliMode.DhcpServer import DhcpServerClientClassAssignBaseMode
from CliMode.DhcpServer import DhcpServerSubnetClientClassAssignBaseMode
from CliMode.DhcpServer import DhcpServerRangeClientClassAssignBaseMode
from CliMode.DhcpServer import DhcpServerClientClassMatchBaseMode
from CliMode.DhcpServer import DhcpServerSubnetClientClassMatchBaseMode
from CliMode.DhcpServer import DhcpServerRangeClientClassMatchBaseMode
from CliMode.DhcpServer import DhcpServerClientClassNestedMatchBaseMode
from CliMode.DhcpServer import DhcpServerSubnetClientClassNestedMatchBaseMode
from CliMode.DhcpServer import DhcpServerRangeClientClassNestedMatchBaseMode
from CliMode.DhcpServer import DhcpServerVendorOptionBaseMode
from CliMode.DhcpServer import DhcpServerReservationsBaseMode
from CliMode.DhcpServer import DhcpServerReservationsMacAddressBaseMode
from CliMode.DhcpServer import DhcpServerRangeBaseMode
from CliMode.DhcpServer import DhcpServerSubnetClientClassBaseMode
from CliMode.DhcpServer import DhcpServerReservationsAristaSwitchBaseMode
from CliMode.DhcpServer import DhcpServerRangeClientClassBaseMode
from CliMode.DhcpServer import DhcpServerReservationsInfoOptionBaseMode
import Tac
from Arnet import EthAddr

ipAddr = Tac.Value( 'Arnet::IpAddr' )

# dhcp relay mode
class DhcpServerCliSaveMode( DhcpServerBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

# client class mode
class DhcpServerClientClassCliSaveMode( DhcpServerGlobalClientClassBaseMode,
                                        CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerGlobalClientClassBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def commentKey( self ):
      return "dhcp-client-class-{}-{}{}".format(
            self.af, self.clientClassName, self.vrfStr )

class DhcpServerClientClassV4CliSaveMode( DhcpServerClientClassCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerClientClassCliSaveMode.__init__( self, param )

class DhcpServerClientClassV6CliSaveMode( DhcpServerClientClassCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerClientClassCliSaveMode.__init__( self, param )

# client class assign mode
class DhcpServerClientClassAssignCliSaveMode( DhcpServerClientClassAssignBaseMode,
                                              CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerClientClassAssignBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class DhcpServerClientClassAssignV4CliSaveMode(
                      DhcpServerClientClassAssignCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerClientClassAssignCliSaveMode.__init__( self, param )

class DhcpServerClientClassAssignV6CliSaveMode(
                      DhcpServerClientClassAssignCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerClientClassAssignCliSaveMode.__init__( self, param )

# subnet client class assign mode
class DhcpServerSubnetClientClassAssignCliSaveMode(
      DhcpServerSubnetClientClassAssignBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerSubnetClientClassAssignBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class DhcpServerSubnetClientClassV4AssignCliSaveMode(
      DhcpServerSubnetClientClassAssignCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerSubnetClientClassAssignCliSaveMode.__init__( self, param )

class DhcpServerSubnetClientClassV6AssignCliSaveMode(
      DhcpServerSubnetClientClassAssignCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerSubnetClientClassAssignCliSaveMode.__init__( self, param )

# range client class assign mode
class DhcpServerRangeClientClassAssignCliSaveMode(
      DhcpServerRangeClientClassAssignBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerRangeClientClassAssignBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class DhcpServerRangeClientClassV4AssignCliSaveMode(
      DhcpServerRangeClientClassAssignCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerRangeClientClassAssignCliSaveMode.__init__( self, param )

class DhcpServerRangeClientClassV6AssignCliSaveMode(
      DhcpServerRangeClientClassAssignCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerRangeClientClassAssignCliSaveMode.__init__( self, param )

# client class match mode
class DhcpServerClientClassMatchCliSaveMode( DhcpServerClientClassMatchBaseMode,
                                             CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerClientClassMatchBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class DhcpServerClientClassV4MatchCliSaveMode(
      DhcpServerClientClassMatchCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerClientClassMatchCliSaveMode.__init__( self, param )

class DhcpServerClientClassV6MatchCliSaveMode(
      DhcpServerClientClassMatchCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerClientClassMatchCliSaveMode.__init__( self, param )

# subnet client class match mode
class DhcpServerSubnetClientClassMatchCliSaveMode(
      DhcpServerSubnetClientClassMatchBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerSubnetClientClassMatchBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class DhcpServerSubnetClientClassV4MatchCliSaveMode(
      DhcpServerSubnetClientClassMatchCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerSubnetClientClassMatchCliSaveMode.__init__( self, param )

class DhcpServerSubnetClientClassV6MatchCliSaveMode(
      DhcpServerSubnetClientClassMatchCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerSubnetClientClassMatchCliSaveMode.__init__( self, param )

# range client class match mode
class DhcpServerRangeClientClassMatchCliSaveMode(
      DhcpServerRangeClientClassMatchBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerRangeClientClassMatchBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class DhcpServerRangeClientClassV4MatchCliSaveMode(
      DhcpServerRangeClientClassMatchCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerRangeClientClassMatchCliSaveMode.__init__( self, param )

class DhcpServerRangeClientClassV6MatchCliSaveMode(
      DhcpServerRangeClientClassMatchCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerRangeClientClassMatchCliSaveMode.__init__( self, param )

# client class nested match mode
class DhcpServerClientClassNestedMatchCliSaveMode(
      DhcpServerClientClassNestedMatchBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerClientClassNestedMatchBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class DhcpServerClientClassV4NestedMatchCliSaveMode(
      DhcpServerClientClassNestedMatchCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerClientClassNestedMatchCliSaveMode.__init__( self, param )

class DhcpServerClientClassV6NestedMatchCliSaveMode(
      DhcpServerClientClassNestedMatchCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerClientClassNestedMatchCliSaveMode.__init__( self, param )

# subnet client class nested match mode
class DhcpServerSubnetClientClassNestedMatchCliSaveMode(
      DhcpServerSubnetClientClassNestedMatchBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerSubnetClientClassNestedMatchBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class DhcpServerSubnetClientClassV4NestedMatchCliSaveMode(
      DhcpServerSubnetClientClassNestedMatchCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerSubnetClientClassNestedMatchCliSaveMode.__init__( self, param )

class DhcpServerSubnetClientClassV6NestedMatchCliSaveMode(
      DhcpServerSubnetClientClassNestedMatchCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerSubnetClientClassNestedMatchCliSaveMode.__init__( self, param )

# range client class nested match mode
class DhcpServerRangeClientClassNestedMatchCliSaveMode(
      DhcpServerRangeClientClassNestedMatchBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerRangeClientClassNestedMatchBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class DhcpServerRangeClientClassV4NestedMatchCliSaveMode(
      DhcpServerRangeClientClassNestedMatchCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerRangeClientClassNestedMatchCliSaveMode.__init__( self, param )

class DhcpServerRangeClientClassV6NestedMatchCliSaveMode(
      DhcpServerRangeClientClassNestedMatchCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerRangeClientClassNestedMatchCliSaveMode.__init__( self, param )

# subnet mode
class DhcpServerSubnetCliSaveMode( DhcpServerSubnetBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerSubnetBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def commentKey( self ):
      return f"dhcp-subnet-{self.subnet}{self.vrfStr}"

class DhcpServerSubnetCliSaveV4Mode( DhcpServerSubnetCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerSubnetCliSaveMode.__init__( self, param )

class DhcpServerSubnetCliSaveV6Mode( DhcpServerSubnetCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerSubnetCliSaveMode.__init__( self, param )

# vendor-option mode
class DhcpServerVendorOptionCliSaveMode( DhcpServerVendorOptionBaseMode,
                                         CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerVendorOptionBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def commentKey( self ):
      return f"dhcp-vendor-{self.af}-{self.vendorId}{self.vrfStr}"

class DhcpServerVendorOptionCliSaveV4Mode( DhcpServerVendorOptionCliSaveMode ):

   def __init__( self, param ):
      # modeCmd is 'ipv4 vendorId' therefore we need to flip it
      param = tuple( reversed( param ) )
      DhcpServerVendorOptionCliSaveMode.__init__( self, param )

# reservations mode
class DhcpServerReservationsCliSaveMode( DhcpServerReservationsBaseMode,
                                         CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerReservationsBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class DhcpServerReservationsCliSaveV4Mode( DhcpServerReservationsCliSaveMode ):

   # param will be subnetConfig
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerReservationsCliSaveMode.__init__( self, param )

class DhcpServerReservationsCliSaveV6Mode( DhcpServerReservationsCliSaveMode ):

   # param will be subnetConfig
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerReservationsCliSaveMode.__init__( self, param )

# reservations mac-address mode
class DhcpServerReservationsMacAddressCliSaveMode(
                            DhcpServerReservationsMacAddressBaseMode,
                            CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerReservationsMacAddressBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class DhcpServerReservationsMacAddressCliSaveV4Mode(
                            DhcpServerReservationsMacAddressCliSaveMode ):
   # param will be set to ( subnetConfig, macAddr )
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerReservationsMacAddressCliSaveMode.__init__( self, param )

class DhcpServerReservationsMacAddressCliSaveV6Mode(
                            DhcpServerReservationsMacAddressCliSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerReservationsMacAddressCliSaveMode.__init__( self, param )

# arista info-option mode
class DhcpServerReservationsAristaInfoOptionCliSaveMode(
                            DhcpServerReservationsAristaSwitchBaseMode,
                            CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerReservationsAristaSwitchBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   @property
   def af( self ):
      return "ipv4"

class DhcpServerReservationsAristaRemoteIdCliSaveMode(
                            DhcpServerReservationsAristaSwitchBaseMode,
                            CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerReservationsAristaSwitchBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   @property
   def af( self ):
      return "ipv6"

# info-option mode
class DhcpServerReservationsInfoOptionCliSaveMode(
                            DhcpServerReservationsInfoOptionBaseMode,
                            CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerReservationsInfoOptionBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

# range mode
class DhcpServerSubnetRangeCliSaveMode( DhcpServerRangeBaseMode, CliSave.Mode ):
   # param will be set to ( subnetConfig, range, af )
   def __init__( self, param ):
      DhcpServerRangeBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class DhcpServerSubnetRangeCliSaveV4Mode( DhcpServerSubnetRangeCliSaveMode ):
   # param will be set to ( subnetConfig, range )
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerSubnetRangeCliSaveMode.__init__( self, param )

class DhcpServerSubnetRangeCliSaveV6Mode( DhcpServerSubnetRangeCliSaveMode ):
   # param will be set to ( subnetConfig, range )
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerSubnetRangeCliSaveMode.__init__( self, param )

# subnet's mode client class mode
class DhcpServerSubnetClientClassSaveMode(
                            DhcpServerSubnetClientClassBaseMode,
                            CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerSubnetClientClassBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def commentKey( self ):
      subnetPrefix = self.subnetConfig.subnetId
      return "dhcp-subnet-{}-client-class-{}{}".format(
            subnetPrefix, self.clientClassName, self.vrfStr )

class DhcpServerSubnetClientClassSaveV4Mode(
                            DhcpServerSubnetClientClassSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerSubnetClientClassSaveMode.__init__( self, param )

class DhcpServerSubnetClientClassSaveV6Mode(
                            DhcpServerSubnetClientClassSaveMode ):
   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerSubnetClientClassSaveMode.__init__( self, param )

# range's client class mode
class DhcpServerRangeClientClassSaveMode(
                           DhcpServerRangeClientClassBaseMode,
                           CliSave.Mode ):

   def __init__( self, param ):
      DhcpServerRangeClientClassBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def commentKey( self ):
      commentKeyString = "dhcp-subnet-{}-range-{}-{}-client-class-{}{}"
      return commentKeyString.format( self.subnetConfig.subnetId,
                                      self.range.start,
                                      self.range.end,
                                      self.clientClassName,
                                      self.vrfStr )

class DhcpServerRangeClientClassSaveV4Mode(
                           DhcpServerRangeClientClassSaveMode ):

   def __init__( self, param ):
      param = param + ( 'ipv4', )
      DhcpServerRangeClientClassSaveMode.__init__( self, param )

class DhcpServerRangeClientClassSaveV6Mode(
                           DhcpServerRangeClientClassSaveMode ):

   def __init__( self, param ):
      param = param + ( 'ipv6', )
      DhcpServerRangeClientClassSaveMode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( DhcpServerCliSaveMode,
                                       before=[ IntfConfigMode ],
                                       after=[ 'DhcpRelay.dhcp' ] )

DhcpServerCliSaveMode.addCommandSequence( 'DhcpServer.config' )

DhcpServerCliSaveMode.addChildMode( DhcpServerSubnetCliSaveV4Mode )
DhcpServerCliSaveMode.addChildMode( DhcpServerSubnetCliSaveV6Mode )
DhcpServerSubnetCliSaveV4Mode.addCommandSequence( 'DhcpServer.subnetV4' )
DhcpServerSubnetCliSaveV6Mode.addCommandSequence( 'DhcpServer.subnetV6' )

DhcpServerCliSaveMode.addChildMode( DhcpServerClientClassV4CliSaveMode )
DhcpServerCliSaveMode.addChildMode( DhcpServerClientClassV6CliSaveMode )
DhcpServerClientClassV4CliSaveMode.addCommandSequence(
      'DhcpServer.clientClassConfigIpv4' )
DhcpServerClientClassV6CliSaveMode.addCommandSequence(
      'DhcpServer.clientClassConfigIpv6' )
DhcpServerClientClassV4CliSaveMode.addChildMode(
      DhcpServerClientClassAssignV4CliSaveMode,
      before=[ 'DhcpServer.clientClassConfigIpv4' ] )
DhcpServerClientClassV6CliSaveMode.addChildMode(
      DhcpServerClientClassAssignV6CliSaveMode,
      before=[ 'DhcpServer.clientClassConfigIpv6' ] )
DhcpServerClientClassAssignV4CliSaveMode.addCommandSequence(
      'DhcpServer.assignmentsV4' )
DhcpServerClientClassAssignV6CliSaveMode.addCommandSequence(
      'DhcpServer.assignmentsV6' )
# Match Mode
DhcpServerClientClassV4CliSaveMode.addChildMode(
      DhcpServerClientClassV4MatchCliSaveMode,
      before=[ 'DhcpServer.clientClassConfigIpv4' ] )
DhcpServerClientClassV6CliSaveMode.addChildMode(
      DhcpServerClientClassV6MatchCliSaveMode,
      before=[ 'DhcpServer.clientClassConfigIpv6' ] )
DhcpServerClientClassV4MatchCliSaveMode.addCommandSequence(
      'DhcpServer.matchV4' )
DhcpServerClientClassV6MatchCliSaveMode.addCommandSequence(
      'DhcpServer.matchV6' )
# Nested Match Mode
# Make sure the nested match mode commands come after the match mode commands so
# that when a config replace is ran, the outer match mode commands will not be
# associated with the nested match mode.
DhcpServerClientClassV4MatchCliSaveMode.addChildMode(
      DhcpServerClientClassV4NestedMatchCliSaveMode,
      after=[ 'DhcpServer.matchV4' ] )
DhcpServerClientClassV6MatchCliSaveMode.addChildMode(
      DhcpServerClientClassV6NestedMatchCliSaveMode,
      after=[ 'DhcpServer.matchV6' ] )
DhcpServerClientClassV4NestedMatchCliSaveMode.addCommandSequence(
      'DhcpServer.nestedMatchV4' )
DhcpServerClientClassV6NestedMatchCliSaveMode.addCommandSequence(
      'DhcpServer.nestedMatchV6' )

DhcpServerCliSaveMode.addChildMode( DhcpServerVendorOptionCliSaveV4Mode )
DhcpServerVendorOptionCliSaveV4Mode.addCommandSequence(
                                    'DhcpServer.vendorOptionIpv4' )

DhcpServerSubnetCliSaveV4Mode.addChildMode( DhcpServerReservationsCliSaveV4Mode,
                                            before=[ 'DhcpServer.subnetV4' ] )
DhcpServerReservationsCliSaveV4Mode.\
      addCommandSequence( 'DhcpServer.reservationsV4' )
DhcpServerReservationsCliSaveV4Mode.\
      addChildMode( DhcpServerReservationsMacAddressCliSaveV4Mode )
DhcpServerReservationsMacAddressCliSaveV4Mode.\
      addCommandSequence( 'DhcpServer.reservationsMacAddrV4' )

DhcpServerSubnetCliSaveV4Mode.addChildMode( DhcpServerSubnetRangeCliSaveV4Mode,
                                            before=[ 'DhcpServer.subnetV4' ] )
DhcpServerSubnetRangeCliSaveV4Mode.addCommandSequence( 'DhcpServer.rangeV4' )
DhcpServerSubnetCliSaveV6Mode.addChildMode( DhcpServerSubnetRangeCliSaveV6Mode,
                                            before=[ 'DhcpServer.subnetV6' ] )
DhcpServerSubnetRangeCliSaveV6Mode.addCommandSequence( 'DhcpServer.rangeV6' )

DhcpServerSubnetCliSaveV6Mode.addChildMode(
                                          DhcpServerReservationsCliSaveV6Mode,
                                          before=[ 'DhcpServer.subnetV6' ] )
DhcpServerReservationsCliSaveV6Mode.\
      addCommandSequence( 'DhcpServer.reservationsV6' )
DhcpServerReservationsCliSaveV6Mode.\
      addChildMode( DhcpServerReservationsMacAddressCliSaveV6Mode )
DhcpServerReservationsMacAddressCliSaveV6Mode.\
      addCommandSequence( 'DhcpServer.reservationsMacAddrV6' )
DhcpServerReservationsCliSaveV4Mode.\
      addChildMode( DhcpServerReservationsAristaInfoOptionCliSaveMode )
DhcpServerReservationsAristaInfoOptionCliSaveMode.\
      addCommandSequence( 'DhcpServer.reservationsAristaInfoOptionV4' )
DhcpServerReservationsCliSaveV4Mode.\
      addChildMode( DhcpServerReservationsInfoOptionCliSaveMode )
DhcpServerReservationsInfoOptionCliSaveMode.\
      addCommandSequence( 'DhcpServer.reservationsInfoOptionV4' )
DhcpServerReservationsCliSaveV6Mode.\
      addChildMode( DhcpServerReservationsAristaRemoteIdCliSaveMode )
DhcpServerReservationsAristaRemoteIdCliSaveMode.\
      addCommandSequence( 'DhcpServer.reservationsAristaRemoteIdV6' )

DhcpServerSubnetCliSaveV4Mode.addChildMode(
                              DhcpServerSubnetClientClassSaveV4Mode )
DhcpServerSubnetCliSaveV6Mode.addChildMode(
                              DhcpServerSubnetClientClassSaveV6Mode )
DhcpServerSubnetClientClassSaveV4Mode. \
      addCommandSequence( 'DhcpServer.subnetClientClassV4' )
DhcpServerSubnetClientClassSaveV6Mode. \
      addCommandSequence( 'DhcpServer.subnetClientClassV6' )

# subnet assign mode
DhcpServerSubnetClientClassSaveV4Mode.addChildMode(
      DhcpServerSubnetClientClassV4AssignCliSaveMode,
      before=[ 'DhcpServer.subnetClientClassV4' ] )
DhcpServerSubnetClientClassSaveV6Mode.addChildMode(
      DhcpServerSubnetClientClassV6AssignCliSaveMode,
      before=[ 'DhcpServer.subnetClientClassV6' ] )
DhcpServerSubnetClientClassV4AssignCliSaveMode.addCommandSequence(
      'DhcpServer.assignmentsV4' )
DhcpServerSubnetClientClassV6AssignCliSaveMode.addCommandSequence(
      'DhcpServer.assignmentsV6' )

# subnet match mode
DhcpServerSubnetClientClassSaveV4Mode.addChildMode(
      DhcpServerSubnetClientClassV4MatchCliSaveMode,
      before=[ 'DhcpServer.subnetClientClassV4' ] )
DhcpServerSubnetClientClassSaveV6Mode.addChildMode(
      DhcpServerSubnetClientClassV6MatchCliSaveMode,
      before=[ 'DhcpServer.subnetClientClassV6' ] )
DhcpServerSubnetClientClassV4MatchCliSaveMode.addCommandSequence(
      'DhcpServer.matchV4' )
DhcpServerSubnetClientClassV6MatchCliSaveMode.addCommandSequence(
      'DhcpServer.matchV6' )
# subnet nested match mode
# Make sure the nested match mode commands come after the match mode commands so
# that when a config replace is ran, the outer match mode commands will not be
# associated with the nested match mode.
DhcpServerSubnetClientClassV4MatchCliSaveMode.addChildMode(
      DhcpServerSubnetClientClassV4NestedMatchCliSaveMode,
      after=[ 'DhcpServer.matchV4' ] )
DhcpServerSubnetClientClassV6MatchCliSaveMode.addChildMode(
      DhcpServerSubnetClientClassV6NestedMatchCliSaveMode,
      after=[ 'DhcpServer.matchV6' ] )
DhcpServerSubnetClientClassV4NestedMatchCliSaveMode.addCommandSequence(
      'DhcpServer.nestedMatchV4' )
DhcpServerSubnetClientClassV6NestedMatchCliSaveMode.addCommandSequence(
      'DhcpServer.nestedMatchV6' )

DhcpServerSubnetRangeCliSaveV4Mode.addChildMode(
                                   DhcpServerRangeClientClassSaveV4Mode,
                                   before=[ 'DhcpServer.rangeV4' ] )
DhcpServerSubnetRangeCliSaveV6Mode.addChildMode(
                                   DhcpServerRangeClientClassSaveV6Mode,
                                   before=[ 'DhcpServer.rangeV6' ] )
DhcpServerRangeClientClassSaveV4Mode.addCommandSequence(
                                    'DhcpServer.rangeClientClassV4' )
DhcpServerRangeClientClassSaveV6Mode.addCommandSequence(
                                    'DhcpServer.rangeClientClassV6' )
# range assign mode
DhcpServerRangeClientClassSaveV4Mode.addChildMode(
      DhcpServerRangeClientClassV4AssignCliSaveMode,
      before=[ 'DhcpServer.rangeClientClassV4' ] )
DhcpServerRangeClientClassSaveV6Mode.addChildMode(
      DhcpServerRangeClientClassV6AssignCliSaveMode,
      before=[ 'DhcpServer.rangeClientClassV6' ] )
DhcpServerRangeClientClassV4AssignCliSaveMode.addCommandSequence(
      'DhcpServer.assignmentsV4' )
DhcpServerRangeClientClassV6AssignCliSaveMode.addCommandSequence(
      'DhcpServer.assignmentsV6' )

# range match mode
DhcpServerRangeClientClassSaveV4Mode.addChildMode(
      DhcpServerRangeClientClassV4MatchCliSaveMode,
      before=[ 'DhcpServer.rangeClientClassV4' ] )
DhcpServerRangeClientClassSaveV6Mode.addChildMode(
      DhcpServerRangeClientClassV6MatchCliSaveMode,
      before=[ 'DhcpServer.rangeClientClassV6' ] )
DhcpServerRangeClientClassV4MatchCliSaveMode.addCommandSequence(
      'DhcpServer.matchV4' )
DhcpServerRangeClientClassV6MatchCliSaveMode.addCommandSequence(
      'DhcpServer.matchV6' )

# range nested match mode
# Make sure the nested match mode commands come after the match mode commands so
# that when a config replace is ran, the outer match mode commands will not be
# associated with the nested match mode.
DhcpServerRangeClientClassV4MatchCliSaveMode.addChildMode(
      DhcpServerRangeClientClassV4NestedMatchCliSaveMode,
      after=[ 'DhcpServer.matchV4' ] )
DhcpServerRangeClientClassV6MatchCliSaveMode.addChildMode(
      DhcpServerRangeClientClassV6NestedMatchCliSaveMode,
      after=[ 'DhcpServer.matchV6' ] )
DhcpServerRangeClientClassV4NestedMatchCliSaveMode.addCommandSequence(
      'DhcpServer.nestedMatchV4' )
DhcpServerRangeClientClassV6NestedMatchCliSaveMode.addCommandSequence(
      'DhcpServer.nestedMatchV6' )


IntfConfigMode.addCommandSequence( 'DhcpServer.intf', after=[ 'Ira.ipIntf' ] )

def saveIpv4ReservationsMacAddr( entity, root, saveMode, macAddr, cmdKey, saveAll,
                                 subnetEntity, vrf ):
   param = ( subnetEntity, macAddr.displayString, vrf )
   mode = root[ saveMode ].getOrCreateModeInstance( param )
   cmds = mode[ cmdKey ]

   if entity.ipAddr != ipAddr.ipAddrZero:
      cmds.addCommand( f"ipv4-address {entity.ipAddr}" )
   elif saveAll:
      cmds.addCommand( "no ipv4-address" )

   if entity.hostname != '':
      cmds.addCommand( f"hostname {entity.hostname}" )
   elif saveAll:
      cmds.addCommand( "no hostname" )

def saveIpv6ReservationsMacAddr( entity, root, saveMode, macAddr, cmdKey, saveAll,
                                 subnetEntity, vrf ):
   param = ( subnetEntity, macAddr.displayString, vrf )
   mode = root[ saveMode ].getOrCreateModeInstance( param )
   cmds = mode[ cmdKey ]

   if not entity.ipAddr.isZero:
      cmds.addCommand( f"ipv6-address {entity.ipAddr}" )
   elif saveAll:
      cmds.addCommand( "no ipv6-address" )

   if entity.hostname != '':
      cmds.addCommand( f"hostname {entity.hostname}" )
   elif saveAll:
      cmds.addCommand( "no hostname" )

def savePrivateOptions( entity, cmds, af='' ):
   for privateOption in chain( entity.stringPrivateOption.values(),
                               entity.ipAddrPrivateOption.values() ):
      optionString = 'private-option'
      optionCode = privateOption.key.code
      optionData = getOptionData( privateOption, raw=True )
      optionType = privateOption.type
      optionAlwaysSend = privateOption.alwaysSend
      cmd = genOptionCmd( optionString, optionCode, optionType, optionData,
                          optionAlwaysSend, af=af )
      cmds.addCommand( cmd )

def saveArbitraryOptions( entity, cmds, af='' ):
   for arbitraryOption in chain( entity.stringArbitraryOption.values(),
                                 entity.fqdnArbitraryOption.values(),
                                 entity.ipAddrArbitraryOption.values(),
                                 entity.hexArbitraryOption.values() ):
      optionString = 'option'
      optionCode = arbitraryOption.key.code
      optionData = getOptionData( arbitraryOption, raw=True )
      optionType = arbitraryOption.type
      optionAlwaysSend = arbitraryOption.alwaysSend
      cmd = genOptionCmd( optionString, optionCode, optionType, optionData,
                          optionAlwaysSend, af=af )
      if cmd:
         cmds.addCommand( cmd )

def saveAssignmentOptionsBase( entity, cmds, saveAll, af ):
   if af == "ipv4":
      if entity.defaultGateway != ipAddr.ipAddrZero:
         cmds.addCommand( f"default-gateway {entity.defaultGateway}" )
      elif saveAll:
         cmds.addCommand( "no default-gateway" )
   if featureFlexibleMatchingFuture():
      # subnet client class and range client class also have lease time under this
      # toggle, so move all client class lease commands here
      if entity.leaseTime:
         days, hours, minutes = convertLeaseSeconds( entity.leaseTime )
         leaseTimeCmd = "lease time {} days {} hours {} minutes".format( days, hours,
                                                                         minutes )
         cmds.addCommand( leaseTimeCmd )
      elif saveAll:
         cmds.addCommand( "no lease time" )

def saveAssignmentOptions( entity, root, saveMode, clientClassName, cmdKey,
                           saveAll, af, vrf ):
   # go to assignments save mode
   param = ( entity, clientClassName, vrf )
   mode = root[ saveMode[ 0 ] ].getOrCreateModeInstance( param )
   cmds = mode[ cmdKey ]

   # Save private-options
   if entity.privateOptionConfig:
      savePrivateOptions( entity.privateOptionConfig, cmds )

   if entity.arbitraryOptionConfig:
      saveArbitraryOptions( entity.arbitraryOptionConfig, cmds )

   if entity.dnsServers:
      servers = list( entity.dnsServers.values() )
      cmds.addCommand( "dns server {}".format( " ".join( map( str, servers ) ) ) )
   elif saveAll:
      cmds.addCommand( "no dns server" )

   if entity.tftpBootFileName != entity.tftpBootFileNameDefault:
      cmds.addCommand( "tftp server file {}".format(
         entity.tftpBootFileName ) )
   elif saveAll:
      cmds.addCommand( "no tftp server file" )

   if entity.domainName:
      cmds.addCommand( f"dns domain name {entity.domainName}" )
   elif saveAll:
      cmds.addCommand( "no dns domain name" )

   if af == 'ipv4':
      if entity.tftpServerOption66 != entity.tftpServerDefault:
         cmds.addCommand( 'tftp server option 66 {}'.format(
            entity.tftpServerOption66 ) )
      elif saveAll:
         cmds.addCommand( 'no tftp server option 66' )

      if entity.tftpServerOption150:
         cmds.addCommand( 'tftp server option 150 {}'.format(
            ' '.join( entity.tftpServerOption150.values() ) ) )
      elif saveAll:
         cmds.addCommand( 'no tftp server option 150' )
   # if not flexibleMatchingFuture, want this here so only global client class adds
   # lease time. Otherwise base will add it there
   if not featureFlexibleMatchingFuture():
      if entity.leaseTime:
         days, hours, minutes = convertLeaseSeconds( entity.leaseTime )
         leaseTimeCmd = "lease time {} days {} hours {} minutes".format( days, hours,
                                                                         minutes )
         cmds.addCommand( leaseTimeCmd )
      elif saveAll:
         cmds.addCommand( "no lease time" )
   saveAssignmentOptionsBase( entity, cmds, saveAll, af )

def saveSubnetAssignmentOptions( entity, root, parentParam, clientClassName,
                                 cmdKey, saveAll, af, vrf ):
   subnet, saveMode = parentParam
   param = ( subnet, entity, clientClassName, vrf )
   mode = root[ saveMode ].getOrCreateModeInstance( param )
   cmds = mode[ cmdKey ]

   if entity.arbitraryOptionConfig:
      saveArbitraryOptions( entity.arbitraryOptionConfig, cmds )

   saveAssignmentOptionsBase( entity, cmds, saveAll, af )

def saveRangeAssignmentOptions( entity, root, parentParam, clientClassName,
                                cmdKey, saveAll, af, vrf ):
   subnet, subnetRange, saveMode = parentParam
   # go to range assignments save mode
   param = ( subnet, subnetRange, entity, clientClassName, vrf )
   mode = root[ saveMode ].getOrCreateModeInstance( param )
   cmds = mode[ cmdKey ]

   v4 = af == 'ipv4'
   ipCmd = 'ipv4-address' if v4 else 'ipv6-address'
   ipCmdFmt = ipCmd + ' {}'
   if entity.assignedIp != entity.ipAddrDefault:
      cmds.addCommand( ipCmdFmt.format( str( entity.assignedIp ) ) )
   elif saveAll:
      cmds.addCommand( f"no {ipCmd}" )

   if entity.arbitraryOptionConfig:
      saveArbitraryOptions( entity.arbitraryOptionConfig, cmds )

   saveAssignmentOptionsBase( entity, cmds, saveAll, af )

def saveMatchCriteriaBaseCmds( matchCriteriaBase, cmds, af ):
   for vendorId in matchCriteriaBase.vendorId:
      cmds.addCommand( f'vendor-id {vendorId}' )

   for vendorClass in matchCriteriaBase.vendorClass:
      cmds.addCommand( formatVendorClassCmd( vendorClass ) )

   for mac in matchCriteriaBase.hostMacAddress:
      cmds.addCommand( f'mac-address {EthAddr( mac ).displayString}' )

   for duid in matchCriteriaBase.duid:
      cmds.addCommand( f'duid {duid}' )

   for l2Info in matchCriteriaBase.l2Info:
      cmds.addCommand( formatL2InfoCmd( l2Info, af ) )

   for circIdRemIdHexOrStr in matchCriteriaBase.circuitIdRemoteIdHexOrStr:
      cmds.addCommand( formatInfoOptionHexOrStrCmd( circIdRemIdHexOrStr ) )

   for remoteIdHexOrStr in matchCriteriaBase.remoteIdHexOrStr:
      cmds.addCommand( formatRemoteIdHexOrStrCmd( remoteIdHexOrStr ) )

   if featureArbitraryOptionMatch() and matchCriteriaBase.arbitraryOptionConfig:
      saveArbitraryOptions( matchCriteriaBase.arbitraryOptionConfig, cmds )

def saveNestedMatchConfig( subMatchCriteriaConfig, matchMode, param, saveMode,
                           cmdKey, saveAll, af, vrf ):
   matchName = subMatchCriteriaConfig.matchCriteriaName
   matchAny = subMatchCriteriaConfig.matchAny
   nestedMatchParam = param + ( matchAny, matchName, vrf )
   mode = matchMode[ saveMode ].getOrCreateModeInstance( nestedMatchParam )
   cmds = mode[ cmdKey ]

   saveMatchCriteriaBaseCmds( subMatchCriteriaConfig, cmds, af )

def saveMatchConfig( clientClassEntity, clientClassMode, param, saveMode, cmdKey,
                     nestedMatchMode, saveAll, af, vrf ):
   matchCriteriaConfig = clientClassEntity.matchCriteriaModeConfig
   matchAny = matchCriteriaConfig.matchAny
   matchParam = param + ( matchAny, vrf )
   matchMode = clientClassMode[ saveMode ].getOrCreateModeInstance( matchParam )
   cmds = matchMode[ cmdKey ]

   saveMatchCriteriaBaseCmds( matchCriteriaConfig, cmds, af )

   if af == 'ipv4':
      subCmdKey = 'DhcpServer.nestedMatchV4'
   else:
      subCmdKey = 'DhcpServer.nestedMatchV6'
   for subMatchConfig in matchCriteriaConfig.subMatchCriteria.values():
      saveNestedMatchConfig( subMatchConfig, matchMode, param, nestedMatchMode,
                             subCmdKey, saveAll, af, vrf )

def saveClientClassBase( clientClassEntity, root, saveMode, clientClassName,
                         cmdKey, saveAll, af, param, assignmentParam, matchParam,
                         saveAssignmentOptionsFunc, clientClassMode,
                         v4AssignMode, v6AssignMode, v4MatchMode, v6MatchMode,
                         v4NestedMatchMode, v6NestedMatchMode, vrf ):
   if af == 'ipv4':
      assignMode = v4AssignMode
      assignCmdKey = 'DhcpServer.assignmentsV4'
      matchMode = v4MatchMode
      matchCmdKey = 'DhcpServer.matchV4'
      nestedMatchMode = v4NestedMatchMode
   else:
      assignMode = v6AssignMode
      assignCmdKey = 'DhcpServer.assignmentsV6'
      matchMode = v6MatchMode
      matchCmdKey = 'DhcpServer.matchV6'
      nestedMatchMode = v6NestedMatchMode

   assignmentParam = assignmentParam + ( assignMode, )
   saveAssignmentOptionsFunc( clientClassEntity, clientClassMode, assignmentParam,
                              clientClassName, assignCmdKey, saveAll, af, vrf )

   if clientClassEntity.matchCriteriaModeConfig:
      saveMatchConfig( clientClassEntity, clientClassMode, matchParam, matchMode,
                       matchCmdKey, nestedMatchMode, saveAll, af, vrf )

def saveClientClass( clientClassEntity, root, saveMode, clientClassName, cmdKey,
                     saveAll, af, vrf ):
   param = ( clientClassName, )
   assignmentParam = ()
   clientClassMode = root[ saveMode ].getOrCreateModeInstance(
         ( clientClassName, vrf ) )
   matchParam = param
   saveClientClassBase(
         clientClassEntity, root, saveMode, clientClassName, cmdKey, saveAll,
         af, param, assignmentParam, matchParam, saveAssignmentOptions,
         clientClassMode, DhcpServerClientClassAssignV4CliSaveMode,
         DhcpServerClientClassAssignV6CliSaveMode,
         DhcpServerClientClassV4MatchCliSaveMode,
         DhcpServerClientClassV6MatchCliSaveMode,
         DhcpServerClientClassV4NestedMatchCliSaveMode,
         DhcpServerClientClassV6NestedMatchCliSaveMode, vrf )

def saveSubnetClientClass( clientClassEntity, root, saveMode, clientClassName,
                           cmdKey, saveAll, subnetConfig, af, vrf ):
   param = ( subnetConfig, clientClassName, vrf )
   assignmentParam = ( subnetConfig.subnetId, )
   matchParam = ( subnetConfig.subnetId, clientClassName )
   clientClassMode = root[ saveMode ].getOrCreateModeInstance( param )
   saveClientClassBase(
         clientClassEntity, root, saveMode, clientClassName, cmdKey, saveAll,
         af, param, assignmentParam, matchParam, saveSubnetAssignmentOptions,
         clientClassMode, DhcpServerSubnetClientClassV4AssignCliSaveMode,
         DhcpServerSubnetClientClassV6AssignCliSaveMode,
         DhcpServerSubnetClientClassV4MatchCliSaveMode,
         DhcpServerSubnetClientClassV6MatchCliSaveMode,
         DhcpServerSubnetClientClassV4NestedMatchCliSaveMode,
         DhcpServerSubnetClientClassV6NestedMatchCliSaveMode, vrf )

def saveRangeClientClass( clientClassEntity, root, saveMode,
                          clientClassName, cmdKey, saveAll,
                          rangeConfig, subnetConfig, af, vrf ):
   param = ( subnetConfig, rangeConfig, clientClassName, vrf )
   assignmentParam = ( subnetConfig.subnetId, rangeConfig.range )
   matchParam = param[ : -1 ]
   clientClassMode = root[ saveMode ].getOrCreateModeInstance( param )
   saveClientClassBase(
         clientClassEntity, root, saveMode, clientClassName, cmdKey, saveAll,
         af, param, assignmentParam, matchParam, saveRangeAssignmentOptions,
         clientClassMode, DhcpServerRangeClientClassV4AssignCliSaveMode,
         DhcpServerRangeClientClassV6AssignCliSaveMode,
         DhcpServerRangeClientClassV4MatchCliSaveMode,
         DhcpServerRangeClientClassV6MatchCliSaveMode,
         DhcpServerRangeClientClassV4NestedMatchCliSaveMode,
         DhcpServerRangeClientClassV6NestedMatchCliSaveMode, vrf )

def saveIpv4ReservationsAristaInfoOption( entity, root, saveMode, l2Info,
                                          cmdKey, saveAll, subnetEntity, vrf ):
   macAddr = l2Info.getRawAttribute( "mac" )
   param = ( subnetEntity, l2Info.intf, l2Info.vlanId, macAddr.displayString, vrf )
   mode = root[ saveMode ].getOrCreateModeInstance( param )
   cmds = mode[ cmdKey ]

   if entity.ipAddr != ipAddr.ipAddrZero:
      cmds.addCommand( f"ipv4-address {entity.ipAddr}" )
   elif saveAll:
      cmds.addCommand( "no ipv4-address" )

def saveIpv6ReservationsAristaRemoteId( entity, root, saveMode, l2Info,
                                          cmdKey, saveAll, subnetEntity, vrf ):
   macAddr = l2Info.getRawAttribute( "mac" )
   param = ( subnetEntity, l2Info.intf, l2Info.vlanId, macAddr.displayString, vrf )
   mode = root[ saveMode ].getOrCreateModeInstance( param )
   cmds = mode[ cmdKey ]

   if not entity.ipAddr.isZero:
      cmds.addCommand( f"ipv6-address {entity.ipAddr}" )
   elif saveAll:
      cmds.addCommand( "no ipv6-address" )

def saveIpv4ReservationsInfoOption( entity, root, saveMode, circuitRemoteHexOrStr,
                                    cmdKey, saveAll, subnetEntity, vrf ):
   param = ( subnetEntity, circuitRemoteHexOrStr, vrf )
   mode = root[ saveMode ].getOrCreateModeInstance( param )
   cmds = mode[ cmdKey ]

   if entity.ipAddr != ipAddr.ipAddrZero:
      cmds.addCommand( f"ipv4-address {entity.ipAddr}" )
   elif saveAll:
      cmds.addCommand( "no ipv4-address" )

def saveVendorOption( entity, root, saveMode, vendorId, cmdKey, saveAll, af, vrf ):
   # vendor-option <modeCmd>, this is to enter the mode
   modeCmd = ( af, vrf, vendorId )
   mode = root[ saveMode ].getOrCreateModeInstance( modeCmd )
   cmds = mode[ cmdKey ]

   # Similar to subnetRange (entity.ranges)
   for subOption in entity.subOptionConfig.values():
      optionData = getSubOptionData( subOption, raw=True )
      cmd = subOptionCmd( subOption.code, subOption.type, optionData,
                          addArrayToken=subOption.isArray )
      cmds.addCommand( cmd )

def saveClientClassAssignmentCmd( cmds, entity ):
   if entity.assignedClientClass:
      clientClassAssignmentCmd = 'client class assignment {}'
      clientClassStr = ' '.join( entity.assignedClientClass )
      cmds.addCommand( clientClassAssignmentCmd.format( clientClassStr ) )

def saveSubnetRangeConfig( entity, root, saveMode, subnetConfig, subnetRange, cmdKey,
                           saveAll, af, vrf ):
   param = ( subnetConfig, subnetRange, vrf )
   mode = root[ saveMode ].getOrCreateModeInstance( param )
   cmds = mode[ cmdKey ]

   saveClientClassAssignmentCmd( cmds, entity )
   clientClassCmdKey = 'DhcpServer.rangeClientClass'
   clientClassSaveMode = ''
   if af == 'ipv4':
      clientClassCmdKey += 'V4'
      clientClassSaveMode = DhcpServerRangeClientClassSaveV4Mode
   else:
      clientClassCmdKey += 'V6'
      clientClassSaveMode = DhcpServerRangeClientClassSaveV6Mode

   for clientClassName in entity.clientClassConfig:
      clientClassEntity = entity.clientClassConfig[ clientClassName ]
      saveRangeClientClass( clientClassEntity, mode, clientClassSaveMode,
                               clientClassName, clientClassCmdKey,
                               saveAll, entity, subnetConfig, af, vrf )

def saveIpv4SubnetConfig( entity, cmds, saveAll, root, vrf ):
   if entity.defaultGateway != ipAddr.ipAddrZero:
      cmds.addCommand( f"default-gateway {entity.defaultGateway}" )
   elif saveAll:
      cmds.addCommand( "no default-gateway" )

   if entity.tftpServerOption66 != entity.tftpServerDefault:
      cmds.addCommand( 'tftp server option 66 {}'.format(
         entity.tftpServerOption66 ) )
   elif saveAll:
      cmds.addCommand( 'no tftp server option 66' )

   if entity.tftpServerOption150:
      cmds.addCommand( 'tftp server option 150 {}'.format(
         ' '.join( entity.tftpServerOption150.values() ) ) )
   elif saveAll:
      cmds.addCommand( 'no tftp server option 150' )

   if entity.tftpBootFileName != entity.tftpBootFileNameDefault:
      cmds.addCommand( "tftp server file {}".format(
         entity.tftpBootFileName ) )
   elif saveAll:
      cmds.addCommand( "no tftp server file" )

   # reservations mac-address
   reservationMode = root[ DhcpServerReservationsCliSaveV4Mode ].\
                                      getOrCreateModeInstance( ( entity, vrf ) )
   for macAddr in entity.reservationsMacAddr:
      reservationEntity = entity.reservationsMacAddr[ macAddr ]
      macAddr = reservationEntity.getRawAttribute( "hostId" )
      saveIpv4ReservationsMacAddr( reservationEntity,
                                   reservationMode,
                                   DhcpServerReservationsMacAddressCliSaveV4Mode,
                                   macAddr, 'DhcpServer.reservationsMacAddrV4',
                                   saveAll, entity, vrf )

   reservationMode = root[ DhcpServerReservationsCliSaveV4Mode ].\
                                     getOrCreateModeInstance( ( entity, vrf ) )
   for reservationEntity in entity.reservationLayer2Intf.values():
      l2Info = reservationEntity.getRawAttribute( "layer2Info" )
      saveIpv4ReservationsAristaInfoOption(
            reservationEntity, reservationMode,
            DhcpServerReservationsAristaInfoOptionCliSaveMode,
            l2Info, 'DhcpServer.reservationsAristaInfoOptionV4',
            saveAll, entity, vrf )

   for reservationEntity in entity.reservationCircuitIdRemoteId.values():
      cIdRId = reservationEntity.getRawAttribute( "circuitIdRemoteId" )
      saveIpv4ReservationsInfoOption(
            reservationEntity, reservationMode,
            DhcpServerReservationsInfoOptionCliSaveMode,
            cIdRId, 'DhcpServer.reservationsInfoOptionV4',
            saveAll, entity, vrf )

   # subnet's client class mode
   for clientClassName in entity.clientClassConfig:
      clientClassEntity = entity.clientClassConfig[ clientClassName ]
      saveSubnetClientClass( clientClassEntity, root,
                             DhcpServerSubnetClientClassSaveV4Mode,
                             clientClassName, 'DhcpServer.subnetClientClassV4',
                             saveAll, entity, 'ipv4', vrf )

def saveIpv6SubnetConfig( entity, cmds, saveAll, root, vrf ):
   if entity.tftpBootFileName != entity.tftpBootFileNameDefault:
      cmds.addCommand( "tftp server file {}".format(
         entity.tftpBootFileName ) )
   elif saveAll:
      cmds.addCommand( "no tftp server file" )

   # subnet's client class mode
   for clientClassName in entity.clientClassConfig:
      clientClassEntity = entity.clientClassConfig[ clientClassName ]
      saveSubnetClientClass( clientClassEntity, root,
                             DhcpServerSubnetClientClassSaveV6Mode,
                             clientClassName, 'DhcpServer.subnetClientClassV6',
                             saveAll, entity, 'ipv6', vrf )

   # go to reservation mode
   mode = root[ DhcpServerReservationsCliSaveV6Mode ].\
                                      getOrCreateModeInstance( ( entity, vrf ) )
   for macAddr in entity.reservationsMacAddr:
      reservationEntity = entity.reservationsMacAddr[ macAddr ]
      macAddr = reservationEntity.getRawAttribute( "hostId" )
      saveIpv6ReservationsMacAddr( reservationEntity,
                                   mode,
                                   DhcpServerReservationsMacAddressCliSaveV6Mode,
                                   macAddr, 'DhcpServer.reservationsMacAddrV6',
                                   saveAll, entity, vrf )

   for reservationEntity in entity.reservationLayer2Intf.values():
      l2Info = reservationEntity.getRawAttribute( "layer2Info" )
      saveIpv6ReservationsAristaRemoteId(
            reservationEntity, mode,
            DhcpServerReservationsAristaRemoteIdCliSaveMode,
            l2Info, 'DhcpServer.reservationsAristaRemoteIdV6',
            saveAll, entity, vrf )

def saveSubnetConfig( entity, root, saveMode, subnet, cmdKey, saveAll, af, vrf ):
   param = ( entity, subnet, vrf )
   mode = root[ saveMode ].getOrCreateModeInstance( param )
   cmds = mode[ cmdKey ]

   if entity.subnetName:
      cmds.addCommand( f"name {entity.subnetName}" )
   elif saveAll:
      cmds.addCommand( "no name" )

   if entity.dnsServers:
      servers = list( entity.dnsServers.values() )
      cmds.addCommand( "dns server {}".format( " ".join( map( str, servers ) ) ) )
   elif saveAll:
      cmds.addCommand( "no dns server" )

   if entity.leaseTime:
      days, hours, minutes = convertLeaseSeconds( entity.leaseTime )
      leaseTimeCmd = "lease time {} days {} hours {} minutes".format(
            days, hours, minutes )
      cmds.addCommand( leaseTimeCmd )
   elif saveAll:
      cmds.addCommand( "no lease time" )

   if af == 'ipv4':
      saveIpv4SubnetConfig( entity, cmds, saveAll, mode, vrf )
   else:
      saveIpv6SubnetConfig( entity, cmds, saveAll, mode, vrf )

   # range
   saveClientClassAssignmentCmd( cmds, entity )

   if af == 'ipv4':
      dhcpServerSubnetRangeCliSaveModeType = DhcpServerSubnetRangeCliSaveV4Mode
      rangeCmdKey = 'DhcpServer.rangeV4'
   else:
      dhcpServerSubnetRangeCliSaveModeType = DhcpServerSubnetRangeCliSaveV6Mode
      rangeCmdKey = 'DhcpServer.rangeV6'
   # Since the range is ordered, we don't need to wrap this in sorted
   for subnetRange, rangeEntity in entity.rangeConfig.items():
      saveSubnetRangeConfig( rangeEntity, mode,
                             dhcpServerSubnetRangeCliSaveModeType,
                             entity, subnetRange, rangeCmdKey, saveAll, af, vrf )

def saveInterfaceConfig( entity, root, requireMounts, options ):
   if options.saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames( requireMounts,
                                                  includeEligible=True )
   elif options.saveAll:
      cfgIntfNames = set( allRoutingProtocolIntfNames( requireMounts ) )
      cfgIntfNames.update( entity.interfacesIpv4, entity.interfacesIpv6 )
   else:
      cfgIntfNames = set( entity.interfacesIpv4 )
      cfgIntfNames.update( entity.interfacesIpv6 )

   for intfId in cfgIntfNames:
      intfMode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
      cmds = intfMode[ 'DhcpServer.intf' ]
      if intfId in entity.interfacesIpv4:
         cmds.addCommand( "dhcp server ipv4" )
      elif options.saveAll:
         cmds.addCommand( "no dhcp server ipv4" )
      if intfId in entity.interfacesIpv6:
         cmds.addCommand( "dhcp server ipv6" )
      elif options.saveAll:
         cmds.addCommand( "no dhcp server ipv6" )

def saveAllEmptyInterfaceConfig( root, requireMounts, options ):
   if options.saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames( requireMounts,
                                                  includeEligible=True )
   elif options.saveAll:
      cfgIntfNames = set( allRoutingProtocolIntfNames( requireMounts ) )
   else:
      return

   for intfId in cfgIntfNames:
      intfMode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
      cmds = intfMode[ 'DhcpServer.intf' ]
      cmds.addCommand( "no dhcp server ipv4" )
      cmds.addCommand( "no dhcp server ipv6" )

def saveDhcpServerConfigVrf( entity, root, requireMounts, options, vrf ):
   if entity.dhcpServerMode == entity.dhcpServerModeDefault:
      return

   mode = root[ DhcpServerCliSaveMode ].getOrCreateModeInstance( vrf )
   cmds = mode[ 'DhcpServer.config' ]

   if entity.disabled != entity.disabledDefault:
      cmds.addCommand( "disabled" )
   elif options.saveAll:
      cmds.addCommand( "no disabled" )

   days, hours, minutes = convertLeaseSeconds( entity.leaseTimeIpv4 )
   leaseTimeCmd = "lease time ipv4 {} days {} hours {} minutes".format(
      days, hours, minutes )
   if ( entity.leaseTimeIpv4 != entity.leaseTimeIpv4Default ) or options.saveAll:
      cmds.addCommand( leaseTimeCmd )

   if entity.domainNameIpv4 != "":
      cmds.addCommand( f"dns domain name ipv4 {entity.domainNameIpv4}" )
   elif options.saveAll:
      cmds.addCommand( "no dns domain name ipv4" )

   if entity.dnsServersIpv4:
      servers = list( entity.dnsServersIpv4.values() )
      cmds.addCommand( "dns server ipv4 {}".format( " ".join( servers ) ) )
   elif options.saveAll:
      cmds.addCommand( "no dns server ipv4" )

   days, hours, minutes = convertLeaseSeconds( entity.leaseTimeIpv6 )
   leaseTimeCmd = "lease time ipv6 {} days {} hours {} minutes".format(
      days, hours, minutes )
   if ( entity.leaseTimeIpv6 != entity.leaseTimeIpv6Default ) or options.saveAll:
      cmds.addCommand( leaseTimeCmd )

   if entity.domainNameIpv6 != "":
      cmds.addCommand( f"dns domain name ipv6 {entity.domainNameIpv6}" )
   elif options.saveAll:
      cmds.addCommand( "no dns domain name ipv6" )

   if entity.dnsServersIpv6:
      servers = list( entity.dnsServersIpv6.values() )
      cmds.addCommand( "dns server ipv6 {}".format(
         " ".join( map( str, servers ) ) ) )
   elif options.saveAll:
      cmds.addCommand( "no dns server ipv6" )

   if entity.tftpServerOption66Ipv4 != entity.tftpServerDefault:
      cmds.addCommand( "tftp server option 66 ipv4 {}".format(
         entity.tftpServerOption66Ipv4 ) )
   elif options.saveAll:
      cmds.addCommand( "no tftp server option 66 ipv4" )

   if entity.tftpServerOption150Ipv4:
      cmds.addCommand( "tftp server option 150 ipv4 {}".format(
         ' '.join( entity.tftpServerOption150Ipv4.values() ) ) )
   elif options.saveAll:
      cmds.addCommand( "no tftp server option 150 ipv4" )

   if entity.tftpBootFileNameIpv4 != entity.tftpBootFileNameDefault:
      bootFile = entity.tftpBootFileNameIpv4
      cmds.addCommand( f"tftp server file ipv4 {bootFile}" )
   elif options.saveAll:
      cmds.addCommand( "no tftp server file ipv4" )

   # Since the subnetConfigIpv4 is ordered, we don't need to wrap this in sorted
   for subnet in entity.subnetConfigIpv4:
      subnetEntity = entity.subnetConfigIpv4[ subnet ]
      saveSubnetConfig( subnetEntity, mode, DhcpServerSubnetCliSaveV4Mode,
                        subnet, 'DhcpServer.subnetV4', options.saveAll, 'ipv4',
                        vrf )

   for subnet in entity.subnetConfigIpv6:
      subnetEntity = entity.subnetConfigIpv6[ subnet ]
      saveSubnetConfig( subnetEntity, mode, DhcpServerSubnetCliSaveV6Mode,
                        subnet, 'DhcpServer.subnetV6', options.saveAll, 'ipv6',
                        vrf )

   # client-class
   for clientClassName in entity.clientClassConfigIpv4:
      clientClassEntity = entity.clientClassConfigIpv4[ clientClassName ]
      saveClientClass( clientClassEntity, mode,
                       DhcpServerClientClassV4CliSaveMode,
                       clientClassName, 'DhcpServer.clientClassConfigIpv4',
                       options.saveAll, 'ipv4', vrf )

   for clientClassName in entity.clientClassConfigIpv6:
      clientClassEntity = entity.clientClassConfigIpv6[ clientClassName ]
      saveClientClass( clientClassEntity, mode,
                       DhcpServerClientClassV6CliSaveMode,
                       clientClassName, 'DhcpServer.clientClassConfigIpv6',
                       options.saveAll, 'ipv6', vrf )

   # global private-options
   if entity.globalPrivateOptionIpv4:
      savePrivateOptions( entity.globalPrivateOptionIpv4, cmds, 'ipv4' )

   if entity.globalPrivateOptionIpv6:
      savePrivateOptions( entity.globalPrivateOptionIpv6, cmds, 'ipv6' )

   # other options
   if entity.tftpBootFileNameIpv6 != entity.tftpBootFileNameDefault:
      bootFile = entity.tftpBootFileNameIpv6
      cmds.addCommand( f"tftp server file ipv6 {bootFile}" )
   elif options.saveAll:
      cmds.addCommand( "no tftp server file ipv6" )

   # global arbitrary options
   if entity.globalArbitraryOptionIpv4:
      saveArbitraryOptions( entity.globalArbitraryOptionIpv4, cmds, 'ipv4' )
   if entity.globalArbitraryOptionIpv6:
      saveArbitraryOptions( entity.globalArbitraryOptionIpv6, cmds, 'ipv6' )

   # vendor-option
   for vendorId in entity.vendorOptionIpv4:
      vendorEntity = entity.vendorOptionIpv4[ vendorId ]
      saveVendorOption( vendorEntity, mode, DhcpServerVendorOptionCliSaveV4Mode,
                        vendorId, 'DhcpServer.vendorOptionIpv4', options.saveAll,
                        'ipv4', vrf )

   if entity.debugLogPath != entity.debugLogPathDefault:
      cmds.addCommand( f"debug log {entity.debugLogUrl}" )
   elif options.saveAll:
      cmds.addCommand( "no debug log" )

   if featureEchoClientId():
      if entity.echoClientIdIpv4 != entity.echoClientIdIpv4Default:
         cmds.addCommand( "option ipv4 client-id disable" )
      elif options.saveAll:
         cmds.addCommand( "no option ipv4 client-id disable" )

@CliSave.saver( 'DhcpServer::VrfConfigDir', 'dhcpServer/vrf/config',
                requireMounts=( 'interface/config/all', 'interface/status/all',
                                'l3/intf/config' ) )
def saveDhcpServerVrfConfigDir( entity, root, requireMounts, options ):
   for vrf in entity.vrfConfig:
      saveInterfaceConfig( entity.vrfConfig[ vrf ], root, requireMounts, options )
      saveDhcpServerConfigVrf( entity.vrfConfig[ vrf ], root, requireMounts,
                               options, vrf )

   if not entity.vrfConfig:
      # This needs to be done in the saveAll case, so that "no dhcp server" is added
      # to the interface config
      saveAllEmptyInterfaceConfig( root, requireMounts, options )
