#!/usr/bin/env python3
# Copyright (c) 2008-2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave, Tracing, Tac
from Toggles.DhcpLibToggleLib import toggleDhcpOptionsProfilingEnabled
import Toggles.Dot1xToggleLib as Dot1xToggle
from MultiRangeRule import multiRangeToCanonicalString
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.Security import SecurityConfigMode
from Dot1xLib import portControlEnum2Str, hostModeEnum2Str, Dot1xConsts
from CliMode.Dot1x import Dot1xBaseMode, Dot1xCacheResultsBaseMode
from TypeFuture import TacLazyType
__defaultTraceHandle__ = Tracing.Handle( 'Dot1xCliSave' )

CliSave.GlobalConfigMode.addCommandSequence( 'dot1x.global' )
IntfConfigMode.addCommandSequence( 'dot1x.intf' )

SessionReplaceDetection = TacLazyType( "Dot1x::SessionReplaceDetection" )
CachedResultsTimeoutUnit = TacLazyType( "Dot1x::CachedResultsTimeoutUnit" )
CachedResultsTimeout = TacLazyType( "Dot1x::CachedResultsTimeout" )
Dot1xDeviceProfilingAttrState = TacLazyType( "Dot1x::Dot1xDeviceProfilingAttrState" )
Dot1xProfilingDhcpOptionType = TacLazyType(
      "Dot1x::Dot1xProfilingDhcpOptionType" )

class Dot1xConfigMode( Dot1xBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      Dot1xBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class Dot1xCacheResultsConfigMode( Dot1xCacheResultsBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      Dot1xCacheResultsBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( Dot1xConfigMode,
                                       before=[ 'dot1x.global' ],
                                       after=[ SecurityConfigMode ] )
Dot1xConfigMode.addCommandSequence( 'dot1x.config' )

Dot1xConfigMode.addChildMode( Dot1xCacheResultsConfigMode )
Dot1xCacheResultsConfigMode.addCommandSequence( 'dot1x.resultcaching.config' )

def authCacheTimeoutConvert( cachedResultsTimeout ):
   timeout = ''
   if cachedResultsTimeout.unit == CachedResultsTimeoutUnit.timeoutUnitDays:
      # pylint: disable-next=consider-using-f-string
      timeout = ' timeout %d days' % ( cachedResultsTimeout.timeout // 86400 )
   elif cachedResultsTimeout.unit == CachedResultsTimeoutUnit.timeoutUnitHours:
      # pylint: disable-next=consider-using-f-string
      timeout = ' timeout %d hours' % ( cachedResultsTimeout.timeout // 3600 )
   elif cachedResultsTimeout.unit == CachedResultsTimeoutUnit.timeoutUnitMinutes:
      # pylint: disable-next=consider-using-f-string
      timeout = ' timeout %d minutes' % ( cachedResultsTimeout.timeout // 60 )
   elif cachedResultsTimeout.unit == CachedResultsTimeoutUnit.timeoutUnitSeconds:
      # pylint: disable-next=consider-using-f-string
      timeout = ' timeout %d seconds' % cachedResultsTimeout.timeout
   return timeout

def saveDot1xIntfAaaUnresponsiveTraffic( cmds, isSet, saveAll,
                                         cache=None, allow=None, device='',
                                         vlan='', acl='',
                                         timeout='' ):
   if isSet:
      if cache and allow:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'dot1x aaa unresponsive%s action '
                          'apply cached-results%s else traffic allow%s' %
                          ( device, timeout, vlan ) )
      elif cache:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'dot1x aaa unresponsive%s action '
                          'apply cached-results%s' % ( device, timeout ) )
      elif allow:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'dot1x aaa unresponsive%s action '
                          'traffic allow%s%s' % ( device, vlan, acl ) )
      else:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'dot1x aaa unresponsive%s action '
                          'disabled' % device )
   elif saveAll:
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'default dot1x aaa unresponsive%s action' % device )

def genDot1xPortServerGroupList( methodList ):
   groupList = None
   for _, grp in methodList.items():
      if groupList:
         groupList = groupList + " group " + grp
      else:
         groupList = "group " + grp
   return groupList

def saveDot1xIntfPortServerGroupList( cmds, saveAll, entity, root, intfName ):
   portGroupEntity = entity.portServerGroupList
   portIntfList = portGroupEntity.get( intfName )
   if portIntfList:
      ml = portIntfList.method
      if len( ml ):
         groupList = genDot1xPortServerGroupList( ml )
         cmdList = f'dot1x aaa server {groupList}'
         cmds.addCommand( cmdList )
      elif saveAll:
         cmds.addCommand( 'no dot1x aaa server' )

@CliSave.saver( 'Dot1x::Config', 'dot1x/config' )
def saveDot1xConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll

   cmds = root[ 'dot1x.global' ]
   if entity.dot1xEnabled:
      cmds.addCommand( 'dot1x system-auth-control' )
   elif saveAll:
      cmds.addCommand( 'no dot1x system-auth-control' )
   if entity.lldpBypass:
      cmds.addCommand( 'dot1x protocol lldp bypass' )
   elif saveAll:
      cmds.addCommand( 'no dot1x protocol lldp bypass' )
   if entity.bpduBypass:
      cmds.addCommand( 'dot1x protocol bpdu bypass' )
   elif saveAll:
      cmds.addCommand( 'no dot1x protocol bpdu bypass' )
   if Dot1xToggle.toggleDot1xLacpBypassEnabled():
      if entity.lacpBypass:
         cmds.addCommand( 'dot1x protocol lacp bypass' )
      elif saveAll:
         cmds.addCommand( 'no dot1x protocol lacp bypass' )
   if entity.dot1xDynAuthEnabled:
      cmds.addCommand( 'dot1x dynamic-authorization' )
   elif saveAll:
      cmds.addCommand( 'no dot1x dynamic-authorization' )

   cacheResults = False
   cacheResultsForPhone = False
   if entity.aaaUnresponsivePhoneApplyCachedResults == 'aaaUnresponsiveApplyCached':
      cacheResultsForPhone = True
   if entity.aaaUnresponsiveApplyCachedResults:
      cacheResults = True
   if cacheResultsForPhone or entity.aaaUnresponsivePhoneAllow or saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      timeout = authCacheTimeoutConvert(
            entity.aaaUnresponsivePhoneApplyCachedResultsTimeout )
      if entity.aaaUnresponsivePhoneAllow and cacheResultsForPhone:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'aaa unresponsive phone action apply cached-results%s '
                          'else traffic allow' % timeout )
      elif entity.aaaUnresponsivePhoneAllow:
         cmds.addCommand( 'aaa unresponsive phone action traffic allow' )
      elif cacheResultsForPhone:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'aaa unresponsive phone action apply cached-results%s' %
                          timeout )
      else:
         cmds.addCommand( 'no aaa unresponsive phone action' )

   if cacheResults or entity.aaaUnresponsiveTrafficAllow.enabled() or saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      timeout = authCacheTimeoutConvert(
            entity.aaaUnresponsiveApplyCachedResultsTimeout )
      if entity.aaaUnresponsiveTrafficAllow.enabled():
         if cacheResults:
            # pylint: disable-next=consider-using-f-string
            cmdStr = ( 'aaa unresponsive action apply cached-results%s else '
                       'traffic allow' % timeout )
         else:
            cmdStr = 'aaa unresponsive action traffic allow'
         if entity.aaaUnresponsiveTrafficAllow.vlanId:
            # pylint: disable-next=consider-using-f-string
            cmdStr += ' vlan %d' % entity.aaaUnresponsiveTrafficAllow.vlanId
         cmds.addCommand( cmdStr )
      elif cacheResults:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'aaa unresponsive action apply cached-results%s' %
                          timeout )
      else:
         cmds.addCommand( 'no aaa unresponsive action' )

   if entity.eapolLogoffIgnore:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cacheMode = mode[ Dot1xCacheResultsConfigMode ].getOrCreateModeInstance( None )
      cmds = cacheMode[ 'dot1x.resultcaching.config' ]
      cmds.addCommand( 'eapol logoff ignore' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cacheMode = mode[ Dot1xCacheResultsConfigMode ].getOrCreateModeInstance( None )
      cmds = cacheMode[ 'dot1x.resultcaching.config' ]
      cmds.addCommand( 'no eapol logoff ignore' )

   if ( saveAll or entity.aaaUnresponsiveEapResponse !=
        entity.aaaUnresponsiveEapResponseDefault ):
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      if ( entity.aaaUnresponsiveEapResponse !=
           entity.aaaUnresponsiveEapResponseDefault ):
         cmds.addCommand( 'aaa unresponsive eap response disabled' )
      elif saveAll:
         cmds.addCommand( 'aaa unresponsive eap response success' )

   if entity.accountingUpdateInterval != entity.defaultAccountingUpdateInterval:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'aaa accounting update interval %d seconds' % \
                       entity.accountingUpdateInterval )
   elif saveAll: 
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no aaa accounting update interval' )

   if entity.mbaDelay != entity.defaultMbaDelay:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'mac based authentication delay %d seconds' %
                       entity.mbaDelay )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no mac based authentication delay' )

   if entity.mbaHoldPeriod != entity.defaultMbaHoldPeriod:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'mac based authentication hold period %d seconds' %
                       entity.mbaHoldPeriod )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no mac based authentication hold period' )

   if entity.defaultServiceType != entity.serviceType:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'radius av-pair service-type' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no radius av-pair service-type' )

   if entity.defaultMultipleFilterId != entity.multipleFilterId:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'radius av-pair filter-id multiple' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no radius av-pair filter-id multiple' )

   if entity.defaultFilterIdDelim != entity.filterIdDelim:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'radius av-pair filter-id delimiter period' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no radius av-pair filter-id delimiter period' )

   if entity.applyIpv4v6Acl:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'radius av-pair filter-id ipv4 ipv6 required' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no radius av-pair filter-id ipv4 ipv6 required' )

   if entity.defaultFramedMtu != entity.framedMtu:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'radius av-pair framed-mtu %d' % entity.framedMtu )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no radius av-pair framed-mtu' )

   if entity.mbaUserName != Tac.Value( "Dot1x::MbaUserName" ):
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      msg = 'mac-based-auth radius av-pair user-name delimiter'
      delim = 'none'
      if entity.mbaUserName.userNameDelim == '-':
         delim = 'hyphen'
      elif entity.mbaUserName.userNameDelim == ':':
         delim = 'colon'
      elif entity.mbaUserName.userNameDelim == '.':
         delim = 'period'
      case = 'uppercase' if entity.mbaUserName.userNameIsUpperCase else 'lowercase'
      cmds.addCommand( f'{msg} {delim} {case}' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no mac-based-auth radius av-pair user-name' )

   if entity.disableVlanChangeLogoff:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'eapol vlan change logoff disabled' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no eapol vlan change logoff disabled' )

   if entity.aaaUnresponsiveRecoveryReauth:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'aaa unresponsive recovery action reauthenticate' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no aaa unresponsive recovery action reauthenticate' )

   if entity.sendIdentityReqNewMac == 'macNewSendIdReq':
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'event mac new action send request-identity unicast' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no event mac new action send request-identity unicast' )

   cmd = 'event mac logged-off action send request-identity unicast'
   if entity.sendIdentityReqDisconnectMac == 'macDisconnectSendIdReq':
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( cmd )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no ' + cmd )

   if ( saveAll or entity.guestVlan ):
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      guestVlanBaseCmd = 'eapol unresponsive action traffic allow'
      if entity.guestVlan:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( '%s vlan %d' % ( guestVlanBaseCmd, entity.guestVlan ) )
      elif saveAll:
         cmds.addCommand( 'no ' + guestVlanBaseCmd )

   if entity.disconnectTimeout != entity.disconnectTimeoutDefault:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'supplicant disconnect cached-results timeout %d seconds' %
                       entity.disconnectTimeout )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no supplicant disconnect cached-results timeout' )

   if entity.captivePortal != Tac.Value( "Dot1x::CaptivePortal" ):
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmd = 'captive-portal'
      if entity.captivePortal.url:
         # pylint: disable-next=consider-using-f-string
         cmd += ' url %s' % entity.captivePortal.url
      if entity.captivePortalSslProfileName:
         # pylint: disable-next=consider-using-f-string
         cmd += ' ssl profile %s' % entity.captivePortalSslProfileName
      cmds.addCommand( cmd )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no captive-portal' )

   if Dot1xToggle.toggleDot1xWebFqdnAllowlistEnabled():
      if entity.captivePortalAllowlist:
         mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
         cmds = mode[ 'dot1x.config' ]
         for fqdn in sorted( entity.captivePortalAllowlist ):
            cmds.addCommand( f'captive-portal bypass {fqdn}' )
      elif saveAll:
         mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
         cmds = mode[ 'dot1x.config' ]
         cmds.addCommand( 'no captive-portal bypass' )

   if entity.captivePortalIpv4Acl != "":
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      # pylint: disable-next=consider-using-f-string
      cmd = 'captive-portal access-list ipv4 %s' % entity.captivePortalIpv4Acl
      cmds.addCommand( cmd )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no captive-portal access-list ipv4' )

   if entity.captivePortalStartLimitInfinite:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'captive-portal start limit infinite' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no captive-portal start limit infinite' )

   if entity.dot1xAclInputPriority != entity.dot1xAclInputPriorityDefault:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'access-list dot1x priority low' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no access-list dot1x priority low' )

   if entity.vlanGroup:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      for groupName in entity.vlanGroup:
         vlans = multiRangeToCanonicalString(
                  list( entity.vlanGroup[ groupName ].vlanId ) )
         cmd = 'vlan assignment group ' + groupName + ' members ' + vlans
         cmds.addCommand( cmd )

   if Dot1xToggle.toggleDot1xIpTrackingEnabled():
      if entity.ipTrackingEnabled != False: # pylint: disable=singleton-comparison
         mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
         cmds = mode[ 'dot1x.config' ]
         cmd = 'address tracking ipv4'
         cmds.addCommand( cmd )
      elif saveAll:
         mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
         cmds = mode[ 'dot1x.config' ]
         cmds.addCommand( 'no address tracking ipv4' )

   if entity.dropCounterEnabled:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'statistics packets dropped' )
   elif saveAll:
      cmds.addCommand( 'no statistics packets dropped' )

   if entity.sessionMoveEapol:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'session move allow eapol' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no session move allow eapol' )

   if entity.sessionReplaceDetection != SessionReplaceDetection.detectionDisabled:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmd = 'single-host session replace detection'
      if entity.sessionReplaceDetection == SessionReplaceDetection.errdisable:
         cmd += ' errdisable'
      cmds.addCommand( cmd )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no single-host session replace detection' )

   def saveProfilingState( profilingState, cmd, serverGroupName=None ):
      if ( profilingState !=
            Dot1xDeviceProfilingAttrState.disableProfiling ):
         mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
         cmds = mode[ 'dot1x.config' ]
         if serverGroupName:
            cmd += f' group {serverGroupName}'
         if ( profilingState ==
               Dot1xDeviceProfilingAttrState.authOnlyProfiling ):
            cmd += ' auth-only'
         cmds.addCommand( cmd )
      elif saveAll:
         mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
         cmds = mode[ 'dot1x.config' ]
         cmds.addCommand( 'no ' + cmd )

   saveProfilingState( entity.lldpSysNameProfilingState,
                       cmd='radius av-pair lldp system-name' )
   saveProfilingState( entity.lldpSysDescProfilingState,
                       cmd='radius av-pair lldp system-description' )

   if toggleDhcpOptionsProfilingEnabled():
      def getProfilingState( option ):
         if option in entity.dot1xProfilingDhcpOption:
            return entity.dot1xProfilingDhcpOption[ option ].state
         else:
            return Dot1xDeviceProfilingAttrState.disableProfiling

      saveProfilingState(
            getProfilingState( Dot1xProfilingDhcpOptionType.hostname ),
            cmd='radius av-pair dhcp hostname' )
      saveProfilingState(
            getProfilingState( Dot1xProfilingDhcpOptionType.paramReqList ),
            cmd='radius av-pair dhcp parameter-request-list' )
      saveProfilingState(
            getProfilingState( Dot1xProfilingDhcpOptionType.vendorClassId ),
            cmd='radius av-pair dhcp vendor-class-id' )

   if Dot1xToggle.toggleDot1xFramedIpSourceL3Enabled():
      saveProfilingState( entity.framedIpSourceL3ProfilingState,
                          cmd='radius av-pair framed-ip source l3-neighbor',
                          serverGroupName=entity.framedIpSourceL3ServerGroup )

   for intfName, dot1xIntfConfig in entity.dot1xIntfConfig.items():
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      cmds = mode[ 'dot1x.intf' ]

      if dot1xIntfConfig.dot1xEnabled:
         cmds.addCommand( 'dot1x pae authenticator' )
      elif saveAll:
         cmds.addCommand( 'no dot1x pae authenticator' )
      # port server group list
      if Dot1xToggle.toggleDot1xPortRadiusServerGroupEnabled():
         saveDot1xIntfPortServerGroupList( cmds, saveAll, entity, root, intfName )
      if dot1xIntfConfig.authFailVlan or dot1xIntfConfig.authFailAcl:
         vlan = dot1xIntfConfig.authFailVlan
         acl = dot1xIntfConfig.authFailAcl
         # pylint: disable-next=consider-using-f-string
         vlanCmd = ' vlan %d' % vlan if vlan else ''
         # pylint: disable-next=consider-using-f-string
         aclCmd = ' access-list %s' % acl if acl else ''
         cmds.addCommand( 'dot1x authentication failure action traffic allow' +
            vlanCmd + aclCmd )
      elif dot1xIntfConfig.afDropConfigured:
         cmds.addCommand( 'dot1x authentication failure action traffic drop' )
      elif saveAll:
         cmds.addCommand(
            'no dot1x authentication failure action traffic allow' )

      isSetPhone = ( dot1xIntfConfig.aaaUnresponsivePhoneApplyCachedResults !=
                     'aaaUnresponsiveApplyCachedUnset' or
                     dot1xIntfConfig.aaaUnresponsivePhoneAllow.isSet )
      cachePhone = ( dot1xIntfConfig.aaaUnresponsivePhoneApplyCachedResults ==
                     'aaaUnresponsiveApplyCached' )
      allowPhone = dot1xIntfConfig.aaaUnresponsivePhoneAllow.value
      timeout = authCacheTimeoutConvert(
            dot1xIntfConfig.aaaUnresponsivePhoneApplyCachedResultsTimeout )
      saveDot1xIntfAaaUnresponsiveTraffic( cmds, isSetPhone, saveAll,
                                           cachePhone, allowPhone,
                                           timeout=timeout,
                                           device=' phone' )
      isSet = ( dot1xIntfConfig.aaaUnresponsiveApplyCachedResults.isSet or
                dot1xIntfConfig.aaaUnresponsiveTrafficAllow.isSet or
                dot1xIntfConfig.aaaUnresponsiveAcl != "" )
      cache = dot1xIntfConfig.aaaUnresponsiveApplyCachedResults.value
      allow = dot1xIntfConfig.aaaUnresponsiveTrafficAllow.value
      # pylint: disable-next=consider-using-f-string
      vlan = ( ' vlan %d' % dot1xIntfConfig.aaaUnresponsiveTrafficAllow.vlanId
               if dot1xIntfConfig.aaaUnresponsiveTrafficAllow.vlanId else '' )
      aclName = dot1xIntfConfig.aaaUnresponsiveAcl
      # pylint: disable-next=consider-using-f-string
      acl = ( ' access-list %s' % aclName if aclName else'' )
      timeout = authCacheTimeoutConvert(
            dot1xIntfConfig.aaaUnresponsiveApplyCachedResultsTimeout )
      saveDot1xIntfAaaUnresponsiveTraffic( cmds, isSet, saveAll, cache, allow,
                                           timeout=timeout,
                                           vlan=vlan, acl=acl )
      if ( dot1xIntfConfig.aaaUnresponsiveEapResponse !=
           'aaaUnresponsiveEapResponseUnset' ):
         if ( dot1xIntfConfig.aaaUnresponsiveEapResponse ==
              'aaaUnresponsiveEapResponseDisabled' ):
            cmds.addCommand( 'dot1x aaa unresponsive eap response disabled' )
         else:
            cmds.addCommand( 'dot1x aaa unresponsive eap response success' )
      elif saveAll:
         cmds.addCommand( 'default dot1x aaa unresponsive eap response' )

      if dot1xIntfConfig.reauth:
         cmds.addCommand( 'dot1x reauthentication' )
      elif options.saveAll:
         cmds.addCommand( 'no dot1x reauthentication' )

      if dot1xIntfConfig.portCtrlSetting != 'forceAuth' or saveAll:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'dot1x port-control %s' 
               % portControlEnum2Str[ dot1xIntfConfig.portCtrlSetting ] )
      if dot1xIntfConfig.forceAuthPhone:
         cmds.addCommand( 'dot1x port-control force-authorized phone' )
      elif saveAll:
         cmds.addCommand( 'no dot1x port-control force-authorized phone' )

      if dot1xIntfConfig.hostMode != 'singleHost' and (
            dot1xIntfConfig.hostMode != 'multiHost' or saveAll ):
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'dot1x host-mode %s'
               % hostModeEnum2Str[ dot1xIntfConfig.hostMode ] )

      if dot1xIntfConfig.sessionReplaceEnabled:
         cmds.addCommand( 'dot1x host-mode single-host session replaceable' )
      elif dot1xIntfConfig.hostMode == 'singleHost':
         cmds.addCommand( 'dot1x host-mode single-host' )

      if dot1xIntfConfig.eapolDisabled:
         cmds.addCommand( 'dot1x eapol disabled' )
      elif saveAll:
         cmds.addCommand( 'no dot1x eapol disabled' )
      if dot1xIntfConfig.aclMode == 'modePerMac':
         cmds.addCommand( 'dot1x mac based access-list' )
      elif saveAll:
         cmds.addCommand( 'no dot1x mac based access-list' )

      if dot1xIntfConfig.mbaHostMode or dot1xIntfConfig.mbaAuthAlways:
         if dot1xIntfConfig.mbaHostMode:
            cmds.addCommand( 'dot1x mac based authentication host-mode common' )
         elif saveAll:
            cmds.addCommand( 'no dot1x mac based authentication host-mode common' )
         if dot1xIntfConfig.mbaAuthAlways:
            cmds.addCommand( 'dot1x mac based authentication always' )
         elif saveAll:
            cmds.addCommand( 'no dot1x mac based authentication always' )
      elif dot1xIntfConfig.mbaEnabled:
         cmds.addCommand( 'dot1x mac based authentication' )
         if saveAll:
            cmds.addCommand( 'no dot1x mac based authentication host-mode common' )
            cmds.addCommand( 'no dot1x mac based authentication always' )
      elif saveAll:
         cmds.addCommand( 'no dot1x mac based authentication' )

      if dot1xIntfConfig.quietPeriod != \
            dot1xIntfConfig.defaultQuietPeriod or saveAll:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'dot1x timeout quiet-period %d' 
                                 % dot1xIntfConfig.quietPeriod )
     
      # pylint: disable-next=singleton-comparison
      if dot1xIntfConfig.reauthTimeoutIgnore != False:
         cmds.addCommand( 'dot1x timeout reauth-timeout-ignore always' )

      if dot1xIntfConfig.txPeriod != dot1xIntfConfig.defaultTxPeriod or saveAll:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'dot1x timeout tx-period %d' 
                                 % dot1xIntfConfig.txPeriod )
      if dot1xIntfConfig.reauthOpts == 'useSetTimeout':
         if dot1xIntfConfig.reauthPeriod != dot1xIntfConfig.defaultReauthPeriod \
            or saveAll:
            # pylint: disable-next=consider-using-f-string
            cmds.addCommand( 'dot1x timeout reauth-period %d' 
                                 % dot1xIntfConfig.reauthPeriod )
      else:
         cmds.addCommand( 'dot1x timeout reauth-period server' )
      if dot1xIntfConfig.idleTimeout != dot1xIntfConfig.defaultIdleTimeout:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'dot1x timeout idle-host %d seconds'
                           % dot1xIntfConfig.idleTimeout )
      elif saveAll:
         cmds.addCommand( 'no dot1x timeout idle-host' )

      if dot1xIntfConfig.maxReauthReq != dot1xIntfConfig.defaultMaxReauthReq:
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'dot1x reauthorization request limit %d'
                          % dot1xIntfConfig.maxReauthReq )
      elif saveAll:
         # the default case should always have the new syntax
         # pylint: disable-next=consider-using-f-string
         cmds.addCommand( 'dot1x reauthorization request limit %d'
                          % dot1xIntfConfig.maxReauthReq )

      if dot1xIntfConfig.unauthorizedAccessVlanEgress:
         cmds.addCommand( 'dot1x unauthorized access vlan membership egress' )
      elif saveAll:
         cmds.addCommand( 'no dot1x unauthorized access vlan membership egress' )

      if dot1xIntfConfig.unauthorizedNativeVlanEgress:
         cmds.addCommand( 'dot1x unauthorized native vlan membership egress' )
      elif saveAll:
         cmds.addCommand( 'no dot1x unauthorized native vlan membership egress' )

      if dot1xIntfConfig.mbaFallback:
         if dot1xIntfConfig.mbaTimeout != Dot1xConsts.mbaTimeoutDefault:
            # pylint: disable-next=consider-using-f-string
            cmds.addCommand( 'dot1x eapol authentication failure fallback mba'\
                  ' timeout %d' % dot1xIntfConfig.mbaTimeout )
         else:
            cmds.addCommand( 'dot1x eapol authentication failure fallback mba' )
      elif saveAll:
         cmds.addCommand( 'no dot1x eapol authentication failure fallback mba' )

      baseCmd = 'dot1x eapol unresponsive action traffic'
      if vlan := dot1xIntfConfig.guestVlan.vlan:
         cmds.addCommand( f'{baseCmd} allow vlan {vlan}' )
      elif dot1xIntfConfig.guestVlan.disabled:
         cmds.addCommand( f'{baseCmd} disabled' )
      elif saveAll:
         cmds.addCommand( f'no {baseCmd}' )

      # pylint: disable-msg=W0105
      '''
      if dot1xIntfConfig.authFailVlan != 0:
         cmds.addCommand( 'dot1x auth-fail vlan %d' % dot1xIntfConfig.authFailVlan )
      elif saveAll:
         cmds.addCommand( 'no dot1x auth-fail vlan' )

      if dot1xIntfConfig.authFailMaxAttempt != \
            dot1xIntfConfig.defaultAuthFailMaxAttempt or saveAll:
         cmds.addCommand( 'dot1x auth-fail max-attempts %d' 
                              % dot1xIntfConfig.authFailMaxAttempt )
      '''
