# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac, Tracing
from CliMode.Dot1x import Dot1xSuppProfileBaseMode
from CliSavePlugin.Dot1xCliSave import Dot1xConfigMode
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.Security import mgmtSecurityConfigPath
import ReversibleSecretCli

__defaultTraceHandle__ = Tracing.Handle( 'Dot1xSupplicantCliSave' )

CliSave.GlobalConfigMode.addCommandSequence( 'dot1x.global' )
IntfConfigMode.addCommandSequence( 'dot1x.intf' )

class Dot1xProfileConfigMode( Dot1xSuppProfileBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      Dot1xSuppProfileBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

Dot1xConfigMode.addChildMode( Dot1xProfileConfigMode )
Dot1xProfileConfigMode.addCommandSequence( 'dot1x.supp.profile' )

def saveProfileConfig( profile, root, config, options, requireMounts ):
   suppMode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
   mode = suppMode[ Dot1xProfileConfigMode ].getOrCreateModeInstance( profile.name )
   cmds = mode[ 'dot1x.supp.profile' ]

   if profile.identity != "":
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'identity %s' % profile.identity )
   elif options.saveAll:
      cmds.addCommand( 'no identity' )

   if profile.eapMethod != profile.eapMethodDefault:
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'eap-method %s' % profile.eapMethod )
   elif options.saveAll:
      cmds.addCommand( 'no eap-method' )

   securityConfig = requireMounts[ mgmtSecurityConfigPath ]
   if profile.encryptedPassword:
      cmd = ReversibleSecretCli.getCliSaveCommand( 'passphrase {}',
                                                   securityConfig,
                                                   profile.encryptedPassword )
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no passphrase' )

   if profile.sslProfileName != '':
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'ssl profile %s' % profile.sslProfileName )
   elif options.saveAll:
      cmds.addCommand( 'no ssl profile' )

def saveIntfConfig( intfConfig, root, config, saveAll ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfConfig.intfId )
   cmds = mode[ 'dot1x.intf' ]

   # pylint: disable-next=consider-using-f-string
   cmds.addCommand( 'dot1x pae supplicant %s' % intfConfig.profileName )

@CliSave.saver( 'Dot1x::Config', 'dot1x/config',
                requireMounts=( mgmtSecurityConfigPath, ) )
def saveDot1xSuppConfig( config, root, requireMounts, options ):
   saveAll = options.saveAll

   if config.supplicantLoggingEnabled or saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      if config.supplicantLoggingEnabled:
         cmds.addCommand( 'supplicant logging' )
      elif saveAll:
         cmds.addCommand( 'no supplicant logging' )

   # Save supplicant profile config.
   for profileName in sorted( config.supplicantProfile ):
      saveProfileConfig( config.supplicantProfile[ profileName ], root, config,
                         options, requireMounts )
   
   # Save supplicant interface config.
   for intfId in config.supplicantIntfConfig:
      saveIntfConfig( config.supplicantIntfConfig[ intfId ], root, config,
                      saveAll )
