#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
from CliMode.EtbaMode import PlatformTfaBaseMode
from CliSavePlugin import IntfCliSave
from TypeFuture import TacLazyType

EthIntfId = TacLazyType( 'Arnet::EthIntfId' )

class PlatformTfaCliSaveMode( PlatformTfaBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      PlatformTfaBaseMode.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( PlatformTfaCliSaveMode )
CliSave.GlobalConfigMode.addCommandSequence( 'Etba.cli' )
PlatformTfaCliSaveMode.addCommandSequence( 'Etba.cli' )
IntfCliSave.IntfConfigMode.addCommandSequence( 'Etba.cli' )

@CliSave.saver( 'Bridging::Etba::CliConfig', 'bridging/etba/cli/config',
                requireMounts=( "bridging/etba/status", "bridging/etba/config",
                                "bridging/etba/cli/tracingConfig" ) )
def saveEtbaCliConfig( etbaCliConfig, root, requireMounts, options ):
   etbaMode = root[ PlatformTfaCliSaveMode ].getSingletonInstance()
   etbaCmds = etbaMode[ "Etba.cli" ]

   for fName, qtraceFilter in sorted( etbaCliConfig.qtraceFilter.items() ):
      intfStr = ''
      if qtraceFilter.intf:
         intfStr = f'interface {qtraceFilter.intf}'
      if qtraceFilter.bpfFilter:
         etbaCmds.addCommand( f'qtrace packet filter {fName} '
                              f'{qtraceFilter.bpfFilter} {intfStr}'.strip() )
      else:
         etbaCmds.addCommand( f'qtrace packet filter {fName} {intfStr}' )

   ArfaModeType = Tac.Type( "Arfa::ArfaMode" )
   if etbaCliConfig.arfaMode != ArfaModeType.unknownMode:
      arfaModeDict = { ArfaModeType.arfaOnlyMode: "arfa",
                       ArfaModeType.hybridMode: "python" }
      # The arfaMode used by Etba is dependent on three possible configs, so we need
      # to look for the current mode in the status, not the config from the CLI.
      statusMode = requireMounts[ "bridging/etba/status" ].arfaMode
      # If Etba hasn't started yet, we fall back to the configured mode
      if statusMode == ArfaModeType.unknownMode:
         saveMode = etbaCliConfig.arfaMode
      else:
         saveMode = statusMode
      arfaMode = arfaModeDict[ saveMode ]
      # pylint: disable-next=consider-using-f-string
      etbaCmds.addCommand( "personality %s" % arfaMode )

   tracingConfig = requireMounts[ 'bridging/etba/cli/tracingConfig' ]
   if tracingConfig.verboseTracingBitmap:
      # pylint: disable-next=consider-using-f-string
      etbaCmds.addCommand( "tracing verbose 0x%x" %
                           tracingConfig.verboseTracingBitmap )
   
   # Populate Phy Ethernet interfaces with info about
   # 'phy control-frame' disabled config
   extTapDevices = requireMounts[ "bridging/etba/config" ].extTapDevice
   for intfName in extTapDevices:
      if not EthIntfId.isEthIntfId( intfName ):
         continue
      mode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance( intfName )
      intfCmds = mode[ 'Etba.cli' ]
      syntax = "platform tfa phy control-frame disabled"

      if intfName in etbaCliConfig.phyControlDisabled:
         intfCmds.addCommand( syntax )
      elif options.saveAll or options.saveAllDetail:
         intfCmds.addCommand( f"no {syntax}" )
